/*
LinphoneWrapper.swift
Copyright (c) 2010-2022 Belledonne Communications SARL.

This file is part of Liblinphone
(see https://gitlab.linphone.org/BC/public/liblinphone).

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU Affero General Public License as
published by the Free Software Foundation, either version 3 of the
License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Affero General Public License for more details.

You should have received a copy of the GNU Affero General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.
*/

import Foundation
import linphone
#if canImport(SwiftUI)
import SwiftUI
#endif


///Enum describing RTP AVPF activation modes. 
public enum AVPFMode:Int
{
	
	/// Use default value defined at upper level. 
	case Default = -1
	
	/// AVPF is disabled. 
	case Disabled = 0
	
	/// AVPF is enabled. 
	case Enabled = 1
}

///Enum describing the authentication methods. 
public enum AuthMethod:Int
{
	
	/// Digest authentication requested. 
	case HttpDigest = 0
	
	/// Client certificate requested. 
	case Tls = 1
	
	/// Bearer authentication. 
	case Bearer = 2
	
	/// Basic authentication. 
	case Basic = 3
}

///``BaudotMode`` enum represents the Baudot mode to use for the call. 
public enum BaudotMode:Int
{
	
	/// Send and receive audio. 
	case Voice = 0
	
	/// Send and receive Baudot tones. 
	case Tty = 1
	
	/// Send Baudot tones, but receive audio. 
	case HearingCarryOver = 2
	
	/// Send audio, but receive Baudot tones. 
	case VoiceCarryOver = 3
}

///``BaudotStandard`` enum represents the Baudot standard to use to send Baudot///tones in the call. 
public enum BaudotStandard:Int
{
	
	/// Send 45.45 baud US Baudot tones. 
	case Us = 0
	
	/// Send 50 baud European Baudot tones. 
	case Europe = 1
}

///Codec priority policies. 
public enum CodecPriorityPolicy:Int
{
	
	/// In this mode, codecs have initial default ordering, that can be changed by the
	/// application The answerer of a call accepts codecs with the order given in the
	/// offer. 
	case Basic = 0
	
	/// In this mode, the codec list is managed by the ``Core`` according to hardware
	/// capabilities in the goal of optimizing video quality and user experience. 
	case Auto = 1
}

///Describes the state of the remote configuring process of the ``Core`` object,///'Skipped' when the feature is disabled. 
public enum ConfiguringState:Int
{
	
	case Successful = 0
	
	case Failed = 1
	
	case Skipped = 2
}

///Consolidated presence information: 'online' means the user is open for///communication, 'busy' means the user is open for communication but involved in///an other activity, 'do not disturb' means the user is not open for///communication, and 'offline' means that no presence information is available. 
public enum ConsolidatedPresence:Int
{
	
	case Online = 0
	
	case Busy = 1
	
	case DoNotDisturb = 2
	
	case Offline = 3
}

///Enum describing the result of the echo canceller calibration process. 
public enum EcCalibratorStatus:Int
{
	
	/// The echo canceller calibration process is on going. 
	case InProgress = 0
	
	/// The echo canceller calibration has been performed and produced an echo delay
	/// measure. 
	case Done = 1
	
	/// The echo canceller calibration process has failed. 
	case Failed = 2
	
	/// The echo canceller calibration has been performed and no echo has been
	/// detected. 
	case DoneNoEcho = 3
}

///Ephemeral chat message policies. 
public enum EphemeralChatMessagePolicy:Int
{
	
	/// In this mode, an ephemeral chat message expires at the same time for all
	/// participants, when every participant has read the message and its lifetime
	/// expires. 
	case Default = 0
	
	/// In this mode, an ephemeral chat message expires independently for each
	/// participant, after the duration of the lifetime once this participant has read
	/// the message. 
	case Individual = 1
}

///Describes the global state of the ``Core`` object. 
public enum GlobalState:Int
{
	
	/// State in which we're in after ``Core/stop()``. 
	case Off = 0
	
	/// Transient state for when we call ``Core/start()`` 
	case Startup = 1
	
	/// Indicates ``Core`` has been started and is up and running. 
	case On = 2
	
	/// Transient state for when we call ``Core/stop()`` 
	case Shutdown = 3
	
	/// Transient state between Startup and On if there is a remote provisionning URI
	/// configured. 
	case Configuring = 4
	
	/// ``Core`` state after being created by linphone_factory_create_core, generally
	/// followed by a call to ``Core/start()`` 
	case Ready = 5
}

///Enum describing ICE states. 
public enum IceState:Int
{
	
	/// ICE has not been activated for this call or stream. 
	case NotActivated = 0
	
	/// ICE processing has failed. 
	case Failed = 1
	
	/// ICE process is in progress. 
	case InProgress = 2
	
	/// ICE has established a direct connection to the remote host. 
	case HostConnection = 3
	
	/// ICE has established a connection to the remote host through one or several
	/// NATs. 
	case ReflexiveConnection = 4
	
	/// ICE has established a connection through a relay. 
	case RelayConnection = 5
}


public enum LimeState:Int
{
	
	/// Lime is not used at all. 
	case Disabled = 0
	
	/// Lime is always used. 
	case Mandatory = 1
	
	/// Lime is used only if we already shared a secret with remote. 
	case Preferred = 2
}

///Whether or not to keep a file with the logs. 
public enum LogCollectionState:Int
{
	
	case Disabled = 0
	
	case Enabled = 1
	
	case EnabledWithoutPreviousLogHandler = 2
}

///Verbosity levels of log messages. 
public struct LogLevel:OptionSet
{
	public let rawValue: Int

	public init(rawValue: Int) {
		self.rawValue = rawValue
	}

	
	/// Level for debug messages. 
	public static let Debug = LogLevel(rawValue: 1<<0)
	
	/// Level for traces. 
	public static let Trace = LogLevel(rawValue: 1<<1)
	
	/// Level for information messages. 
	public static let Message = LogLevel(rawValue: 1<<2)
	
	/// Level for warning messages. 
	public static let Warning = LogLevel(rawValue: 1<<3)
	
	/// Level for error messages. 
	public static let Error = LogLevel(rawValue: 1<<4)
	
	/// Level for fatal error messages. 
	public static let Fatal = LogLevel(rawValue: 1<<5)
}

///Indicates for a given media the stream direction. 
public enum MediaDirection:Int
{
	
	/// Default value, shouldn't be used. 
	case Invalid = -1
	
	/// No active media not supported yet. 
	case Inactive = 0
	
	/// Media is only being sent, it won't be received. 
	case SendOnly = 1
	
	/// Media will only be received, nothing will be sent. 
	case RecvOnly = 2
	
	/// Media will be sent and received. 
	case SendRecv = 3
}

///Enum describing type of media encryption types. 
public enum MediaEncryption:Int
{
	
	/// No media encryption is used. 
	case None = 0
	
	/// Use SRTP media encryption. 
	case SRTP = 1
	
	/// Use ZRTP media encryption. 
	case ZRTP = 2
	
	/// Use DTLS media encryption. 
	case DTLS = 3
}

///Enum representing the file format of a recording. 
public enum MediaFileFormat:Int
{
	
	case Unknown = 0
	
	case Wav = 1
	
	/// < WAVE file format, .wav file extension. 
	case Mkv = 2
	
	/// < Standard Matroska file format, supports video, .mkv or .mka file extension. 
	case Smff = 3
}

///Media resource usage. 
public enum MediaResourceMode:Int
{
	
	/// Media resources are not shared. 
	case Exclusive = 0
	
	/// Media resources are shared. 
	case Shared = 1
}

///Basic status as defined in section 4.1.4 of RFC 3863. 
public enum PresenceBasicStatus:Int
{
	
	/// This value means that the associated contact element, if any, is ready to
	/// accept communication. 
	case Open = 0
	
	/// This value means that the associated contact element, if any, is unable to
	/// accept communication. 
	case Closed = 1
}

///Defines privacy policy to apply as described by rfc3323. 
public enum Privacy:Int
{
	
	/// Privacy services must not perform any privacy function. 
	case None = 0
	
	/// Request that privacy services provide a user-level privacy function. 
	case User = 1
	
	/// Request that privacy services modify headers that cannot be set arbitrarily by
	/// the user (Contact/Via). 
	case Header = 2
	
	/// Request that privacy services provide privacy for session media. 
	case Session = 4
	
	/// rfc3325 The presence of this privacy type in a Privacy header field indicates
	/// that the user would like the Network Asserted Identity to be kept private with
	/// respect to SIP entities outside the Trust Domain with which the user
	/// authenticated. 
	case Id = 8
	
	/// Privacy service must perform the specified services or fail the request. 
	case Critical = 16
	
	/// Special keyword to use privacy as defined either globally or by proxy using
	/// ``ProxyConfig/setPrivacy(privacy:)`` 
	case Default = 32768
}

///Enum for publish states. 
public enum PublishState:Int
{
	
	/// Initial state, do not use. 
	case None = 0
	
	/// An incoming publish is received. 
	case IncomingReceived = 1
	
	/// Publish is accepted. 
	case Ok = 2
	
	/// Publish encoutered an error, ``Event/getReason()`` gives reason code. 
	case Error = 3
	
	/// Publish is about to expire, only sent if [sip]->refresh_generic_publish
	/// property is set to 0. 
	case Expiring = 4
	
	/// Event has been un published. 
	case Cleared = 5
	
	/// Publish is about to terminate. 
	case Terminating = 6
	
	/// An outgoing publish was created and submitted. 
	case OutgoingProgress = 7
	
	/// Publish is about to refresh. 
	case Refreshing = 8
}

///Enum describing various failure reasons or contextual information for some///events. 
public enum Reason:Int
{
	
	/// No reason has been set by the core. 
	case None = 0
	
	/// No response received from remote. 
	case NoResponse = 1
	
	/// Authentication failed due to bad credentials or resource forbidden. 
	case Forbidden = 2
	
	/// The call has been declined. 
	case Declined = 3
	
	/// Destination of the call was not found. 
	case NotFound = 4
	
	/// The call was not answered in time (request timeout) 
	case NotAnswered = 5
	
	/// Phone line was busy. 
	case Busy = 6
	
	/// Unsupported content. 
	case UnsupportedContent = 7
	
	/// Bad event. 
	case BadEvent = 8
	
	/// Transport error: connection failures, disconnections etc... 
	case IOError = 9
	
	/// Do not disturb reason. 
	case DoNotDisturb = 10
	
	/// Operation is unauthorized because missing credential. 
	case Unauthorized = 11
	
	/// Operation is rejected due to incompatible or unsupported media parameters. 
	case NotAcceptable = 12
	
	/// Operation could not be executed by server or remote client because it didn't
	/// have any context for it. 
	case NoMatch = 13
	
	/// Resource moved permanently. 
	case MovedPermanently = 14
	
	/// Resource no longer exists. 
	case Gone = 15
	
	/// Temporarily unavailable. 
	case TemporarilyUnavailable = 16
	
	/// Address incomplete. 
	case AddressIncomplete = 17
	
	/// Not implemented. 
	case NotImplemented = 18
	
	/// Bad gateway. 
	case BadGateway = 19
	
	/// The received request contains a Session-Expires header field with a duration
	/// below the minimum timer. 
	case SessionIntervalTooSmall = 20
	
	/// Server timeout. 
	case ServerTimeout = 21
	
	/// Unknown reason. 
	case Unknown = 22
	
	/// The call has been transferred. 
	case Transferred = 23
	
	/// Conditional Request Failed. 
	case ConditionalRequestFailed = 24
	
	case SasCheckRequired = 25
}

///Describes proxy registration states. 
public enum RegistrationState:Int
{
	
	/// Initial state for registrations. 
	case None = 0
	
	/// Registration is in progress. 
	case Progress = 1
	
	/// Registration is successful. 
	case Ok = 2
	
	/// Unregistration succeeded. 
	case Cleared = 3
	
	/// Registration failed. 
	case Failed = 4
	
	/// Registration refreshing. 
	case Refreshing = 5
}

///Enum for search direction. 
public enum SearchDirection:Int
{
	
	case Up = 0
	
	case Down = 1
}

///Security level determined by type of encryption (point-to-point, end-to-end,///etc...) and whether or not a SAS validation was made with the remote(s) end(s). 
public enum SecurityLevel:Int
{
	
	/// Security failure. 
	case Unsafe = 0
	
	/// No encryption. 
	case None = 1
	
	/// End-to-end encrypted. 
	case EndToEndEncrypted = 2
	
	/// End-to-end encrypted and verified. 
	case EndToEndEncryptedAndVerified = 3
	
	/// Point-to-point encrypted. 
	case PointToPointEncrypted = 4
}

///Session Timers refresher. 
public enum SessionExpiresRefresher:Int
{
	
	case Unspecified = 0
	
	case UAS = 1
	
	case UAC = 2
}

///All signal units that a device can use. 
public enum SignalStrengthUnit:Int
{
	
	case Rssi = 0
	
	case Dbm = 1
}

///All signal types that a device can use. 
public enum SignalType:Int
{
	
	case Wifi = 0
	
	case Mobile = 1
	
	case Other = 2
}

///Enum describing type of SRTP encryption suite. 
public enum SrtpSuite:Int
{
	
	case AESCM128HMACSHA180 = 0
	
	case AESCM128HMACSHA132 = 1
	
	case AES192CMHMACSHA180 = 2
	
	case AES192CMHMACSHA132 = 3
	
	case AES256CMHMACSHA180 = 4
	
	case AES256CMHMACSHA132 = 5
	
	case AEADAES128GCM = 6
	
	case AEADAES256GCM = 7
	
	case Invalid = 255
}

///Enum describing the stream types. 
public enum StreamType:Int
{
	
	case Audio = 0
	
	case Video = 1
	
	case Text = 2
	
	case Unknown = 3
}

///Enum controlling behavior for incoming subscription request. 
public enum SubscribePolicy:Int
{
	
	/// Does not automatically accept an incoming subscription request. 
	case SPWait = 0
	
	/// Rejects incoming subscription request. 
	case SPDeny = 1
	
	/// Automatically accepts a subscription request. 
	case SPAccept = 2
}

///Enum for subscription direction (incoming or outgoing). 
public enum SubscriptionDir:Int
{
	
	/// Incoming subscription. 
	case Incoming = 0
	
	/// Outgoing subscription. 
	case Outgoing = 1
	
	/// Invalid subscription direction. 
	case InvalidDir = 2
}

///Enum for subscription states. 
public enum SubscriptionState:Int
{
	
	/// Initial state, should not be used. 
	case None = 0
	
	/// An outgoing subcription was sent. 
	case OutgoingProgress = 1
	
	/// An incoming subcription is received. 
	case IncomingReceived = 2
	
	/// Subscription is pending, waiting for user approval. 
	case Pending = 3
	
	/// Subscription is accepted. 
	case Active = 4
	
	/// Subscription is terminated normally. 
	case Terminated = 5
	
	/// Subscription was terminated by an error, indicated by ``Event/getReason()`` 
	case Error = 6
	
	/// Subscription is about to expire, only sent if [sip]->refresh_generic_subscribe
	/// property is set to 0. 
	case Expiring = 7
}


public enum SupportLevel:Int
{
	
	/// No support for the feature. 
	case NoSupport = 0
	
	/// Optional support for the feature. 
	case Optional = 1
	
	/// Mandatory support for the feature. 
	case Mandatory = 2
}

///Enum listing frequent telephony tones. 
public enum ToneID:Int
{
	
	/// Not a tone. 
	case Undefined = 0
	
	/// Busy tone. 
	case Busy = 1
	
	/// Call waiting tone. 
	case CallWaiting = 2
	
	/// Call on hold tone. 
	case CallOnHold = 3
	
	/// Tone played when call is abruptly disconnected (media lost) 
	case CallLost = 4
	
	/// When the call end for any reason but lost. 
	case CallEnd = 5
	
	/// When the call is not answered. 
	case CallNotAnswered = 6
	
	/// When the SAS check is required. 
	case SasCheckRequired = 7
}

///Enum describing transport type for ``Address``. 
public enum TransportType:Int
{
	
	case Udp = 0
	
	case Tcp = 1
	
	case Tls = 2
	
	case Dtls = 3
}

///Enum describing uPnP states. 
public enum UpnpState:Int
{
	
	/// uPnP is not activate 
	case Idle = 0
	
	/// uPnP process is in progress 
	case Pending = 1
	
	/// Internal use: Only used by port binding. 
	case Adding = 2
	
	/// Internal use: Only used by port binding. 
	case Removing = 3
	
	/// uPnP is not available 
	case NotAvailable = 4
	
	/// uPnP is enabled 
	case Ok = 5
	
	/// uPnP processing has failed 
	case Ko = 6
	
	/// IGD router is blacklisted. 
	case Blacklisted = 7
}

///Enum describing the result of a version update check. 
public enum VersionUpdateCheckResult:Int
{
	
	case UpToDate = 0
	
	case NewVersionAvailable = 1
	
	case Error = 2
}

///Enum representing the sub type of the screen sharing. 
public enum VideoSourceScreenSharingType:Int
{
	
	/// The screen sharing is done from a display. 
	case Display = 0
	
	/// The screen sharing is done from a window. 
	case Window = 1
	
	/// The screen sharing is done from an area. 
	case Area = 2
}

///Enum representing the type of a video source. 
public enum VideoSourceType:Int
{
	
	case Unknown = 0
	
	/// The video source is another call. 
	case Call = 1
	
	/// The video source is a camera. 
	case Camera = 2
	
	/// The video source is an image. 
	case Image = 3
	
	/// The video source is a screen sharing. 
	case ScreenSharing = 4
}

///Enum describing the types of argument for ``XmlRpcRequest``. 
public enum XmlRpcArgType:Int
{
	
	case None = 0
	
	case Int = 1
	
	case String = 2
	
	case StringStruct = 3
}

///Enum describing the status of a ``XmlRpcRequest``. 
public enum XmlRpcStatus:Int
{
	
	case Pending = 0
	
	case Ok = 1
	
	case Failed = 2
}

///Enum describing the ZRTP key exchange algorithns. 
public enum ZrtpKeyAgreement:Int
{
	
	case Invalid = 0
	
	case Dh2K = 1
	
	case Dh3K = 2
	
	case Ec25 = 3
	
	case Ec38 = 4
	
	case Ec52 = 5
	
	case X255 = 6
	
	case X448 = 7
	
	case K255 = 8
	
	case K448 = 9
	
	case Kyb1 = 10
	
	case Kyb2 = 11
	
	case Kyb3 = 12
	
	case Hqc1 = 13
	
	case Hqc2 = 14
	
	case Hqc3 = 15
	
	case K255Kyb512 = 16
	
	case K255Hqc128 = 17
	
	case K448Kyb1024 = 18
	
	case K448Hqc256 = 19
	
	case K255Kyb512Hqc128 = 20
	
	case K448Kyb1024Hqc256 = 21
	
	case Mlk1 = 22
	
	case Mlk2 = 23
	
	case Mlk3 = 24
	
	case K255Mlk512 = 25
	
	case K448Mlk1024 = 26
}

///Enum describing the ZRTP SAS validation status of a peer URI. 
public enum ZrtpPeerStatus:Int
{
	
	/// Peer URI unkown or never validated/invalidated the SAS. 
	case Unknown = 0
	
	/// Peer URI SAS rejected in database. 
	case Invalid = 1
	
	/// Peer URI SAS validated in database. 
	case Valid = 2
}


func charArrayToString(charPointer: UnsafePointer<CChar>?) -> String {
	return charPointer != nil ? String(cString: charPointer!) : ""
}

// This is a specific work around: these functions can return "defaultString" in some cases,
// However the Swift will automatically create a invisible cDefaultString of type char* to pass as argument
// to the C function. The cDefaultString will then be "returned" by linphone_config_get_string, and
// instantly destroyed by the Swift garbage collector, causing an empty string as a result.

public extension Config {
	/// Retrieves a configuration item as a string, given its section, key, and default
	/// value.
	/// The default value string is returned if the config item isn't found.
	/// - Parameter section: The section from which to retrieve a configuration item
	/// - Parameter key: The name of the configuration item to retrieve
	/// - Parameter defaultString: The default value to return if not found.
	/// - Returns: the found value or the default one if not found.
	public func getString(section:String, key:String, defaultString:String?) -> String
	{
		var result = ""
		(defaultString ?? "").withCString { cStr in
			result = charArrayToString(charPointer: linphone_config_get_string(cPtr, section, key, cStr))
		}
		return result
	}

	/// Retrieves a default configuration item as a string, given its section, key, and
	/// default value.
	/// The default value string is returned if the config item isn't found.
	/// - Parameter section: The section from which to retrieve the default value
	/// - Parameter key: The name of the configuration item to retrieve
	/// - Parameter defaultValue: The default value to return if not found
	/// - Returns: the found default value or default_value if not found.
	public func getDefaultString(section:String, key:String, defaultValue:String) -> String
	{
		var result = ""
		defaultValue.withCString { cStr in
			result = charArrayToString(charPointer: linphone_config_get_section_param_string(cPtr, section, key, cStr))
		}
		return result
	}

	/// Retrieves a section parameter item as a string, given its section and key.
	/// The default value string is returned if the config item isn't found.
	/// - Parameter section: The section from which to retrieve the default value
	/// - Parameter key: The name of the configuration item to retrieve
	/// - Parameter defaultValue: The default value to return if not found.
	/// - Returns: the found default value or default_value if not found.
	public func getSectionParamString(section:String, key:String, defaultValue:String?) -> String
	{
		var result = ""
		(defaultValue ?? "").withCString { cStr in
			result = charArrayToString(charPointer: linphone_config_get_section_param_string(cPtr, section, key, cStr))
		}
		return result
	}
}

/// Class basic linphone class
public class LinphoneObject {
	var cPtr:OpaquePointer?

	/* Initializes a new LinphoneObject with the provided cPointer.
     - Parameters:
     - cPointer: The OpaquePointer of c lib
     - Returns: new LinphoneObject
  */
	init(cPointer:OpaquePointer) {
		cPtr = cPointer
		belle_sip_object_ref(UnsafeMutableRawPointer(cPtr))
	}

	deinit {
		belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  nil, nil)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPtr))
	}
}

func StringArrayToBctbxList(list: [String]?) -> UnsafeMutablePointer<bctbx_list_t>? {
	var cList: UnsafeMutablePointer<bctbx_list_t>? = nil
	for data in list ?? [] {
		let sData:NSString = data as NSString
		cList = bctbx_list_append(cList, unsafeBitCast(sData.utf8String, to: UnsafeMutablePointer<CChar>.self))
	}
	return cList
}

func BctbxListToStringArray(list: UnsafeMutablePointer<bctbx_list_t>) -> [String]? {
	var sList = [String]()
	var cList = list
	while (cList.pointee.data != nil) {
		sList.append(String(cString: unsafeBitCast(cList.pointee.data, to: UnsafePointer<CChar>.self)))
		cList = UnsafeMutablePointer<bctbx_list_t>(cList.pointee.next)
	}
	return sList
}

func ObjectArrayToBctbxList<T:LinphoneObject>(list: [T]?) -> UnsafeMutablePointer<bctbx_list_t>? {
	var cList: UnsafeMutablePointer<bctbx_list_t>? = nil
	for data in list ?? []{
		cList = bctbx_list_append(cList, UnsafeMutableRawPointer(data.cPtr))
	}
	return cList
}

#if canImport(SwiftUI) && (os(iOS) || os(tvOS)) && (arch(arm64) || arch(x86_64)) ///does not compile on armv7 nor macOS
@available(iOS 13.0, tvOS 13.0, *)
@available(macOS, unavailable)
@available(watchOS, unavailable)
/// Add this object in your SwiftUI while giving him the callback necessary to link its view to the Core (using Core.nativeVideoWindowId for example).
public struct LinphoneVideoViewHolder: UIViewRepresentable {
	private let _view = UIView()
	private let _setVideoWindowFn : (UIView) -> Void

	public init(setVideoWindowFn : @escaping (UIView) -> Void)
	{
		_setVideoWindowFn = setVideoWindowFn
	}

	public func makeUIView(context: Context) -> UIView {
		_setVideoWindowFn(_view)
		return _view
	}

	public func updateUIView(_ uiView: UIView, context: Context) {}
}
#endif

protocol LinphoneObjectDelegate {
	var cPtr: OpaquePointer {get set}
}

enum LinphoneError: Error {
	case exception(result: String)
}

public var coreQueue : DispatchQueue = DispatchQueue(label:"core.queue")

public class LinphoneAsyncHelper {

	static func postOnCoreQueue(lambda : @escaping ()->()) {
		coreQueue.async {
			lambda()
		}
	}

	static func postOnMainQueue(lambda : @escaping()->()) {
		DispatchQueue.main.async {
			lambda()
		}
	}

}


public protocol AccountDelegate : AnyObject {
	
	
	/// Get the registration state changed callback. 
	/// - Returns: The current registration state changed callback. 
	func onRegistrationStateChanged(account: Account, state: RegistrationState, message: String)
	
	/// Get the message waiting indication changed callback. 
	/// - Returns: The current message waiting indication changed callback. 
	func onMessageWaitingIndicationChanged(account: Account, mwi: MessageWaitingIndication)
	
	/// Get the conference information updated callback. 
	/// - Returns: The current conference information updated callback. 
	func onConferenceInformationUpdated(account: Account, infos: [ConferenceInfo])
}

public extension AccountDelegate {
	
	func onRegistrationStateChanged(account: Account, state: RegistrationState, message: String) {}
	
	func onMessageWaitingIndicationChanged(account: Account, mwi: MessageWaitingIndication) {}
	
	func onConferenceInformationUpdated(account: Account, infos: [ConferenceInfo]) {}
}

public final class AccountDelegateStub : AccountDelegate
{
	var _onRegistrationStateChanged: ((Account, RegistrationState, String) -> Void)?
	var _onMessageWaitingIndicationChanged: ((Account, MessageWaitingIndication) -> Void)?
	var _onConferenceInformationUpdated: ((Account, [ConferenceInfo]) -> Void)?

	
	public func onRegistrationStateChanged(account: Account, state: RegistrationState, message: String){_onRegistrationStateChanged.map{$0(account, state, message)}}
	
	public func onMessageWaitingIndicationChanged(account: Account, mwi: MessageWaitingIndication){_onMessageWaitingIndicationChanged.map{$0(account, mwi)}}
	
	public func onConferenceInformationUpdated(account: Account, infos: [ConferenceInfo]){_onConferenceInformationUpdated.map{$0(account, infos)}}

	public init (
		onRegistrationStateChanged: ((Account, RegistrationState, String) -> Void)? = nil,
		onMessageWaitingIndicationChanged: ((Account, MessageWaitingIndication) -> Void)? = nil,
		onConferenceInformationUpdated: ((Account, [ConferenceInfo]) -> Void)? = nil
	) {
		self._onRegistrationStateChanged = onRegistrationStateChanged
		self._onMessageWaitingIndicationChanged = onMessageWaitingIndicationChanged
		self._onConferenceInformationUpdated = onConferenceInformationUpdated
	}
}

class AccountDelegateManager
{
	var cPtr: OpaquePointer
	public var delegate: AccountDelegate?

	public init() {
		cPtr = linphone_factory_create_account_cbs(linphone_factory_get())
		belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passRetained(self).toOpaque()), {
			data in
			if (data != nil) {
				Unmanaged<AccountDelegateManager>.fromOpaque(data!).release()
			}
		})

		linphone_account_cbs_set_registration_state_changed(cPtr, { (account, state, message) -> Void in
			if (account != nil) {
				let sObject = Account.getSwiftObject(cObject: account!)
				let delegate = sObject.currentDelegate
				delegate?.onRegistrationStateChanged(account: sObject, state: RegistrationState(rawValue: Int(state.rawValue))!, message: charArrayToString(charPointer: message))
			}
		})

		linphone_account_cbs_set_message_waiting_indication_changed(cPtr, { (account, mwi) -> Void in
			if (account != nil) {
				let sObject = Account.getSwiftObject(cObject: account!)
				let delegate = sObject.currentDelegate
				delegate?.onMessageWaitingIndicationChanged(account: sObject, mwi: MessageWaitingIndication.getSwiftObject(cObject: mwi!))
			}
		})

		linphone_account_cbs_set_conference_information_updated(cPtr, { (account, infos) -> Void in
			if (account != nil) {
				let sObject = Account.getSwiftObject(cObject: account!)
				let delegate = sObject.currentDelegate
				var infossList = [ConferenceInfo]()
				let infoscList = infos
				var listTemp = infoscList
				while (listTemp != nil) {
					let data = unsafeBitCast(listTemp!.pointee.data, to: OpaquePointer.self)
					infossList.append(ConferenceInfo.getSwiftObject(cObject: data))
					listTemp = UnsafePointer<bctbx_list_t>(listTemp!.pointee.next)
				}
				delegate?.onConferenceInformationUpdated(account: sObject, infos: infossList)
			}
		})
	}


}

public protocol AccountCreatorDelegate : AnyObject {
	
	
	/// Get the create account request. 
	/// - Returns: The current create account request. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServicesRequestDelegate`` instead 
	func onCreateAccount(creator: AccountCreator, status: AccountCreator.Status, response: String)
	
	/// Get the is account exist request. 
	/// - Returns: The current is account exist request. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServicesRequestDelegate`` instead 
	func onIsAccountExist(creator: AccountCreator, status: AccountCreator.Status, response: String)
	
	/// Get the activate account request. 
	/// - Returns: The current activate account request. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServicesRequestDelegate`` instead 
	func onActivateAccount(creator: AccountCreator, status: AccountCreator.Status, response: String)
	
	/// Get the activate account request. 
	/// - Returns: The current activate account request. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServicesRequestDelegate`` instead 
	func onSendToken(creator: AccountCreator, status: AccountCreator.Status, response: String)
	
	/// Get the callback on account creation request token. 
	/// In response:
	/// -"token" is the request token to used with
	/// ``AccountCreator/requestAccountCreationTokenUsingRequestToken()``.
	/// -"validation_url" is a URL to redirect the user into a browser for validation.
	/// In status:
	/// -LinphoneAccountCreatorStatusRequestOk: the request token should be in response
	/// with the validation url.
	/// - Returns: The current request token request. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServicesRequestDelegate`` instead 
	func onAccountCreationRequestToken(creator: AccountCreator, status: AccountCreator.Status, response: String)
	
	/// Get the callback on account creation token. 
	/// In response, "token" is the token to pass to
	/// ``AccountCreator/setToken(token:)``. It is used for
	/// ``AccountCreator/createAccount()``
	/// In status:
	/// -LinphoneAccountCreatorStatusRequestOk: token can be retrieved from the "token"
	/// field in response.
	/// -LinphoneAccountCreatorStatusRequestFailed: request token has not been
	/// validated. Recall
	/// ``AccountCreator/requestAccountCreationTokenUsingRequestToken()`` after some
	/// time.
	/// -LinphoneAccountCreatorStatusMissingArguments: request_token has not been set
	/// from ``AccountCreator/setToken(token:)``.
	/// -LinphoneAccountCreatorStatusServerError: URL is not reachable.
	/// - Returns: The current request token request. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServicesRequestDelegate`` instead 
	func onAccountCreationTokenUsingRequestToken(creator: AccountCreator, status: AccountCreator.Status, response: String)
	
	/// Get the is account activated request. 
	/// - Returns: The current is account activated request. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServicesRequestDelegate`` instead 
	func onIsAccountActivated(creator: AccountCreator, status: AccountCreator.Status, response: String)
	
	/// Get the link account request. 
	/// - Returns: The current link account request. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServicesRequestDelegate`` instead 
	func onLinkAccount(creator: AccountCreator, status: AccountCreator.Status, response: String)
	
	/// Get the activate alias request. 
	/// - Returns: The current link account request. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServicesRequestDelegate`` instead 
	func onActivateAlias(creator: AccountCreator, status: AccountCreator.Status, response: String)
	
	/// Get the is alias used request. 
	/// - Returns: The current is alias used request. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServicesRequestDelegate`` instead 
	func onIsAliasUsed(creator: AccountCreator, status: AccountCreator.Status, response: String)
	
	/// Get the is account linked request. 
	/// - Returns: The current is account linked request. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServicesRequestDelegate`` instead 
	func onIsAccountLinked(creator: AccountCreator, status: AccountCreator.Status, response: String)
	
	/// Get the recover account request. 
	/// - Returns: The current recover account request. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServicesRequestDelegate`` instead 
	func onRecoverAccount(creator: AccountCreator, status: AccountCreator.Status, response: String)
	
	/// Get the update account request. 
	/// - Returns: The current update account request. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServicesRequestDelegate`` instead 
	func onUpdateAccount(creator: AccountCreator, status: AccountCreator.Status, response: String)
	
	/// Get the login linphone account request. 
	/// - Returns: The current login linphone account request. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServicesRequestDelegate`` instead 
	func onLoginLinphoneAccount(creator: AccountCreator, status: AccountCreator.Status, response: String)
}

public extension AccountCreatorDelegate {
	
	func onCreateAccount(creator: AccountCreator, status: AccountCreator.Status, response: String) {}
	
	func onIsAccountExist(creator: AccountCreator, status: AccountCreator.Status, response: String) {}
	
	func onActivateAccount(creator: AccountCreator, status: AccountCreator.Status, response: String) {}
	
	func onSendToken(creator: AccountCreator, status: AccountCreator.Status, response: String) {}
	
	func onAccountCreationRequestToken(creator: AccountCreator, status: AccountCreator.Status, response: String) {}
	
	func onAccountCreationTokenUsingRequestToken(creator: AccountCreator, status: AccountCreator.Status, response: String) {}
	
	func onIsAccountActivated(creator: AccountCreator, status: AccountCreator.Status, response: String) {}
	
	func onLinkAccount(creator: AccountCreator, status: AccountCreator.Status, response: String) {}
	
	func onActivateAlias(creator: AccountCreator, status: AccountCreator.Status, response: String) {}
	
	func onIsAliasUsed(creator: AccountCreator, status: AccountCreator.Status, response: String) {}
	
	func onIsAccountLinked(creator: AccountCreator, status: AccountCreator.Status, response: String) {}
	
	func onRecoverAccount(creator: AccountCreator, status: AccountCreator.Status, response: String) {}
	
	func onUpdateAccount(creator: AccountCreator, status: AccountCreator.Status, response: String) {}
	
	func onLoginLinphoneAccount(creator: AccountCreator, status: AccountCreator.Status, response: String) {}
}

public final class AccountCreatorDelegateStub : AccountCreatorDelegate
{
	var _onCreateAccount: ((AccountCreator, AccountCreator.Status, String) -> Void)?
	var _onIsAccountExist: ((AccountCreator, AccountCreator.Status, String) -> Void)?
	var _onActivateAccount: ((AccountCreator, AccountCreator.Status, String) -> Void)?
	var _onSendToken: ((AccountCreator, AccountCreator.Status, String) -> Void)?
	var _onAccountCreationRequestToken: ((AccountCreator, AccountCreator.Status, String) -> Void)?
	var _onAccountCreationTokenUsingRequestToken: ((AccountCreator, AccountCreator.Status, String) -> Void)?
	var _onIsAccountActivated: ((AccountCreator, AccountCreator.Status, String) -> Void)?
	var _onLinkAccount: ((AccountCreator, AccountCreator.Status, String) -> Void)?
	var _onActivateAlias: ((AccountCreator, AccountCreator.Status, String) -> Void)?
	var _onIsAliasUsed: ((AccountCreator, AccountCreator.Status, String) -> Void)?
	var _onIsAccountLinked: ((AccountCreator, AccountCreator.Status, String) -> Void)?
	var _onRecoverAccount: ((AccountCreator, AccountCreator.Status, String) -> Void)?
	var _onUpdateAccount: ((AccountCreator, AccountCreator.Status, String) -> Void)?
	var _onLoginLinphoneAccount: ((AccountCreator, AccountCreator.Status, String) -> Void)?

	
	public func onCreateAccount(creator: AccountCreator, status: AccountCreator.Status, response: String){_onCreateAccount.map{$0(creator, status, response)}}
	
	public func onIsAccountExist(creator: AccountCreator, status: AccountCreator.Status, response: String){_onIsAccountExist.map{$0(creator, status, response)}}
	
	public func onActivateAccount(creator: AccountCreator, status: AccountCreator.Status, response: String){_onActivateAccount.map{$0(creator, status, response)}}
	
	public func onSendToken(creator: AccountCreator, status: AccountCreator.Status, response: String){_onSendToken.map{$0(creator, status, response)}}
	
	public func onAccountCreationRequestToken(creator: AccountCreator, status: AccountCreator.Status, response: String){_onAccountCreationRequestToken.map{$0(creator, status, response)}}
	
	public func onAccountCreationTokenUsingRequestToken(creator: AccountCreator, status: AccountCreator.Status, response: String){_onAccountCreationTokenUsingRequestToken.map{$0(creator, status, response)}}
	
	public func onIsAccountActivated(creator: AccountCreator, status: AccountCreator.Status, response: String){_onIsAccountActivated.map{$0(creator, status, response)}}
	
	public func onLinkAccount(creator: AccountCreator, status: AccountCreator.Status, response: String){_onLinkAccount.map{$0(creator, status, response)}}
	
	public func onActivateAlias(creator: AccountCreator, status: AccountCreator.Status, response: String){_onActivateAlias.map{$0(creator, status, response)}}
	
	public func onIsAliasUsed(creator: AccountCreator, status: AccountCreator.Status, response: String){_onIsAliasUsed.map{$0(creator, status, response)}}
	
	public func onIsAccountLinked(creator: AccountCreator, status: AccountCreator.Status, response: String){_onIsAccountLinked.map{$0(creator, status, response)}}
	
	public func onRecoverAccount(creator: AccountCreator, status: AccountCreator.Status, response: String){_onRecoverAccount.map{$0(creator, status, response)}}
	
	public func onUpdateAccount(creator: AccountCreator, status: AccountCreator.Status, response: String){_onUpdateAccount.map{$0(creator, status, response)}}
	
	public func onLoginLinphoneAccount(creator: AccountCreator, status: AccountCreator.Status, response: String){_onLoginLinphoneAccount.map{$0(creator, status, response)}}

	public init (
		onCreateAccount: ((AccountCreator, AccountCreator.Status, String) -> Void)? = nil,
		onIsAccountExist: ((AccountCreator, AccountCreator.Status, String) -> Void)? = nil,
		onActivateAccount: ((AccountCreator, AccountCreator.Status, String) -> Void)? = nil,
		onSendToken: ((AccountCreator, AccountCreator.Status, String) -> Void)? = nil,
		onAccountCreationRequestToken: ((AccountCreator, AccountCreator.Status, String) -> Void)? = nil,
		onAccountCreationTokenUsingRequestToken: ((AccountCreator, AccountCreator.Status, String) -> Void)? = nil,
		onIsAccountActivated: ((AccountCreator, AccountCreator.Status, String) -> Void)? = nil,
		onLinkAccount: ((AccountCreator, AccountCreator.Status, String) -> Void)? = nil,
		onActivateAlias: ((AccountCreator, AccountCreator.Status, String) -> Void)? = nil,
		onIsAliasUsed: ((AccountCreator, AccountCreator.Status, String) -> Void)? = nil,
		onIsAccountLinked: ((AccountCreator, AccountCreator.Status, String) -> Void)? = nil,
		onRecoverAccount: ((AccountCreator, AccountCreator.Status, String) -> Void)? = nil,
		onUpdateAccount: ((AccountCreator, AccountCreator.Status, String) -> Void)? = nil,
		onLoginLinphoneAccount: ((AccountCreator, AccountCreator.Status, String) -> Void)? = nil
	) {
		self._onCreateAccount = onCreateAccount
		self._onIsAccountExist = onIsAccountExist
		self._onActivateAccount = onActivateAccount
		self._onSendToken = onSendToken
		self._onAccountCreationRequestToken = onAccountCreationRequestToken
		self._onAccountCreationTokenUsingRequestToken = onAccountCreationTokenUsingRequestToken
		self._onIsAccountActivated = onIsAccountActivated
		self._onLinkAccount = onLinkAccount
		self._onActivateAlias = onActivateAlias
		self._onIsAliasUsed = onIsAliasUsed
		self._onIsAccountLinked = onIsAccountLinked
		self._onRecoverAccount = onRecoverAccount
		self._onUpdateAccount = onUpdateAccount
		self._onLoginLinphoneAccount = onLoginLinphoneAccount
	}
}

class AccountCreatorDelegateManager
{
	var cPtr: OpaquePointer
	public var delegate: AccountCreatorDelegate?

	public init() {
		cPtr = linphone_factory_create_account_creator_cbs(linphone_factory_get())
		belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passRetained(self).toOpaque()), {
			data in
			if (data != nil) {
				Unmanaged<AccountCreatorDelegateManager>.fromOpaque(data!).release()
			}
		})

		linphone_account_creator_cbs_set_create_account(cPtr, { (creator, status, response) -> Void in
			if (creator != nil) {
				let sObject = AccountCreator.getSwiftObject(cObject: creator!)
				let delegate = sObject.currentDelegate
				delegate?.onCreateAccount(creator: sObject, status: AccountCreator.Status(rawValue: Int(status.rawValue))!, response: charArrayToString(charPointer: response))
			}
		})

		linphone_account_creator_cbs_set_is_account_exist(cPtr, { (creator, status, response) -> Void in
			if (creator != nil) {
				let sObject = AccountCreator.getSwiftObject(cObject: creator!)
				let delegate = sObject.currentDelegate
				delegate?.onIsAccountExist(creator: sObject, status: AccountCreator.Status(rawValue: Int(status.rawValue))!, response: charArrayToString(charPointer: response))
			}
		})

		linphone_account_creator_cbs_set_activate_account(cPtr, { (creator, status, response) -> Void in
			if (creator != nil) {
				let sObject = AccountCreator.getSwiftObject(cObject: creator!)
				let delegate = sObject.currentDelegate
				delegate?.onActivateAccount(creator: sObject, status: AccountCreator.Status(rawValue: Int(status.rawValue))!, response: charArrayToString(charPointer: response))
			}
		})

		linphone_account_creator_cbs_set_send_token(cPtr, { (creator, status, response) -> Void in
			if (creator != nil) {
				let sObject = AccountCreator.getSwiftObject(cObject: creator!)
				let delegate = sObject.currentDelegate
				delegate?.onSendToken(creator: sObject, status: AccountCreator.Status(rawValue: Int(status.rawValue))!, response: charArrayToString(charPointer: response))
			}
		})

		linphone_account_creator_cbs_set_account_creation_request_token(cPtr, { (creator, status, response) -> Void in
			if (creator != nil) {
				let sObject = AccountCreator.getSwiftObject(cObject: creator!)
				let delegate = sObject.currentDelegate
				delegate?.onAccountCreationRequestToken(creator: sObject, status: AccountCreator.Status(rawValue: Int(status.rawValue))!, response: charArrayToString(charPointer: response))
			}
		})

		linphone_account_creator_cbs_set_account_creation_token_using_request_token(cPtr, { (creator, status, response) -> Void in
			if (creator != nil) {
				let sObject = AccountCreator.getSwiftObject(cObject: creator!)
				let delegate = sObject.currentDelegate
				delegate?.onAccountCreationTokenUsingRequestToken(creator: sObject, status: AccountCreator.Status(rawValue: Int(status.rawValue))!, response: charArrayToString(charPointer: response))
			}
		})

		linphone_account_creator_cbs_set_is_account_activated(cPtr, { (creator, status, response) -> Void in
			if (creator != nil) {
				let sObject = AccountCreator.getSwiftObject(cObject: creator!)
				let delegate = sObject.currentDelegate
				delegate?.onIsAccountActivated(creator: sObject, status: AccountCreator.Status(rawValue: Int(status.rawValue))!, response: charArrayToString(charPointer: response))
			}
		})

		linphone_account_creator_cbs_set_link_account(cPtr, { (creator, status, response) -> Void in
			if (creator != nil) {
				let sObject = AccountCreator.getSwiftObject(cObject: creator!)
				let delegate = sObject.currentDelegate
				delegate?.onLinkAccount(creator: sObject, status: AccountCreator.Status(rawValue: Int(status.rawValue))!, response: charArrayToString(charPointer: response))
			}
		})

		linphone_account_creator_cbs_set_activate_alias(cPtr, { (creator, status, response) -> Void in
			if (creator != nil) {
				let sObject = AccountCreator.getSwiftObject(cObject: creator!)
				let delegate = sObject.currentDelegate
				delegate?.onActivateAlias(creator: sObject, status: AccountCreator.Status(rawValue: Int(status.rawValue))!, response: charArrayToString(charPointer: response))
			}
		})

		linphone_account_creator_cbs_set_is_alias_used(cPtr, { (creator, status, response) -> Void in
			if (creator != nil) {
				let sObject = AccountCreator.getSwiftObject(cObject: creator!)
				let delegate = sObject.currentDelegate
				delegate?.onIsAliasUsed(creator: sObject, status: AccountCreator.Status(rawValue: Int(status.rawValue))!, response: charArrayToString(charPointer: response))
			}
		})

		linphone_account_creator_cbs_set_is_account_linked(cPtr, { (creator, status, response) -> Void in
			if (creator != nil) {
				let sObject = AccountCreator.getSwiftObject(cObject: creator!)
				let delegate = sObject.currentDelegate
				delegate?.onIsAccountLinked(creator: sObject, status: AccountCreator.Status(rawValue: Int(status.rawValue))!, response: charArrayToString(charPointer: response))
			}
		})

		linphone_account_creator_cbs_set_recover_account(cPtr, { (creator, status, response) -> Void in
			if (creator != nil) {
				let sObject = AccountCreator.getSwiftObject(cObject: creator!)
				let delegate = sObject.currentDelegate
				delegate?.onRecoverAccount(creator: sObject, status: AccountCreator.Status(rawValue: Int(status.rawValue))!, response: charArrayToString(charPointer: response))
			}
		})

		linphone_account_creator_cbs_set_update_account(cPtr, { (creator, status, response) -> Void in
			if (creator != nil) {
				let sObject = AccountCreator.getSwiftObject(cObject: creator!)
				let delegate = sObject.currentDelegate
				delegate?.onUpdateAccount(creator: sObject, status: AccountCreator.Status(rawValue: Int(status.rawValue))!, response: charArrayToString(charPointer: response))
			}
		})

		linphone_account_creator_cbs_set_login_linphone_account(cPtr, { (creator, status, response) -> Void in
			if (creator != nil) {
				let sObject = AccountCreator.getSwiftObject(cObject: creator!)
				let delegate = sObject.currentDelegate
				delegate?.onLoginLinphoneAccount(creator: sObject, status: AccountCreator.Status(rawValue: Int(status.rawValue))!, response: charArrayToString(charPointer: response))
			}
		})
	}


}

public protocol AccountManagerServicesRequestDelegate : AnyObject {
	
	
	/// Gets the request successful callback. 
	/// - Returns: The current request successful callback. 
	func onRequestSuccessful(request: AccountManagerServicesRequest, data: String)
	
	/// Gets the request error callback. 
	/// - Returns: The current request error callback. 
	func onRequestError(request: AccountManagerServicesRequest, statusCode: Int, errorMessage: String, parameterErrors: Dictionary?)
	
	/// Gets the devices list fetched callback. 
	/// - Returns: The current devices list fetched callback. 
	func onDevicesListFetched(request: AccountManagerServicesRequest, devicesList: [AccountDevice])
}

public extension AccountManagerServicesRequestDelegate {
	
	func onRequestSuccessful(request: AccountManagerServicesRequest, data: String) {}
	
	func onRequestError(request: AccountManagerServicesRequest, statusCode: Int, errorMessage: String, parameterErrors: Dictionary?) {}
	
	func onDevicesListFetched(request: AccountManagerServicesRequest, devicesList: [AccountDevice]) {}
}

public final class AccountManagerServicesRequestDelegateStub : AccountManagerServicesRequestDelegate
{
	var _onRequestSuccessful: ((AccountManagerServicesRequest, String) -> Void)?
	var _onRequestError: ((AccountManagerServicesRequest, Int, String, Dictionary?) -> Void)?
	var _onDevicesListFetched: ((AccountManagerServicesRequest, [AccountDevice]) -> Void)?

	
	public func onRequestSuccessful(request: AccountManagerServicesRequest, data: String){_onRequestSuccessful.map{$0(request, data)}}
	
	public func onRequestError(request: AccountManagerServicesRequest, statusCode: Int, errorMessage: String, parameterErrors: Dictionary?){_onRequestError.map{$0(request, statusCode, errorMessage, parameterErrors)}}
	
	public func onDevicesListFetched(request: AccountManagerServicesRequest, devicesList: [AccountDevice]){_onDevicesListFetched.map{$0(request, devicesList)}}

	public init (
		onRequestSuccessful: ((AccountManagerServicesRequest, String) -> Void)? = nil,
		onRequestError: ((AccountManagerServicesRequest, Int, String, Dictionary?) -> Void)? = nil,
		onDevicesListFetched: ((AccountManagerServicesRequest, [AccountDevice]) -> Void)? = nil
	) {
		self._onRequestSuccessful = onRequestSuccessful
		self._onRequestError = onRequestError
		self._onDevicesListFetched = onDevicesListFetched
	}
}

class AccountManagerServicesRequestDelegateManager
{
	var cPtr: OpaquePointer
	public var delegate: AccountManagerServicesRequestDelegate?

	public init() {
		cPtr = linphone_factory_create_account_manager_services_request_cbs(linphone_factory_get())
		belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passRetained(self).toOpaque()), {
			data in
			if (data != nil) {
				Unmanaged<AccountManagerServicesRequestDelegateManager>.fromOpaque(data!).release()
			}
		})

		linphone_account_manager_services_request_cbs_set_request_successful(cPtr, { (request, data) -> Void in
			if (request != nil) {
				let sObject = AccountManagerServicesRequest.getSwiftObject(cObject: request!)
				let delegate = sObject.currentDelegate
				delegate?.onRequestSuccessful(request: sObject, data: charArrayToString(charPointer: data))
			}
		})

		linphone_account_manager_services_request_cbs_set_request_error(cPtr, { (request, statusCode, errorMessage, parameterErrors) -> Void in
			if (request != nil) {
				let sObject = AccountManagerServicesRequest.getSwiftObject(cObject: request!)
				let delegate = sObject.currentDelegate
				delegate?.onRequestError(request: sObject, statusCode: Int(statusCode), errorMessage: charArrayToString(charPointer: errorMessage), parameterErrors: (parameterErrors != nil) ? Dictionary.getSwiftObject(cObject: parameterErrors!) : nil)
			}
		})

		linphone_account_manager_services_request_cbs_set_devices_list_fetched(cPtr, { (request, devicesList) -> Void in
			if (request != nil) {
				let sObject = AccountManagerServicesRequest.getSwiftObject(cObject: request!)
				let delegate = sObject.currentDelegate
				var devicesListsList = [AccountDevice]()
				let devicesListcList = devicesList
				var listTemp = devicesListcList
				while (listTemp != nil) {
					let data = unsafeBitCast(listTemp!.pointee.data, to: OpaquePointer.self)
					devicesListsList.append(AccountDevice.getSwiftObject(cObject: data))
					listTemp = UnsafePointer<bctbx_list_t>(listTemp!.pointee.next)
				}
				delegate?.onDevicesListFetched(request: sObject, devicesList: devicesListsList)
			}
		})
	}


}

public protocol AlertDelegate : AnyObject {
	
	
	/// Get the callback for when the alert is terminated. 
	/// - Returns: The LinphoneAlertCbsTerminatedCb callback to execute.    
	func onTerminated(alert: Alert)
}

public extension AlertDelegate {
	
	func onTerminated(alert: Alert) {}
}

public final class AlertDelegateStub : AlertDelegate
{
	var _onTerminated: ((Alert) -> Void)?

	
	public func onTerminated(alert: Alert){_onTerminated.map{$0(alert)}}

	public init (
		onTerminated: ((Alert) -> Void)? = nil
	) {
		self._onTerminated = onTerminated
	}
}

class AlertDelegateManager
{
	var cPtr: OpaquePointer
	public var delegate: AlertDelegate?

	public init() {
		cPtr = linphone_factory_create_alert_cbs(linphone_factory_get())
		belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passRetained(self).toOpaque()), {
			data in
			if (data != nil) {
				Unmanaged<AlertDelegateManager>.fromOpaque(data!).release()
			}
		})

		linphone_alert_cbs_set_terminated(cPtr, { (alert) -> Void in
			if (alert != nil) {
				let sObject = Alert.getSwiftObject(cObject: alert!)
				let delegate = sObject.currentDelegate
				delegate?.onTerminated(alert: sObject)
			}
		})
	}


}

public protocol CallDelegate : AnyObject {
	
	
	/// Get the dtmf received callback. 
	/// - Returns: The current dtmf received callback. 
	func onDtmfReceived(call: Call, dtmf: Int)
	
	/// Get the GoClear Ack sent callback. 
	/// - Returns: The GoClear Ack sent callback. 
	func onGoclearAckSent(call: Call)
	
	/// Get the security level downgraded callback. 
	/// - Returns: The current security level downgraded callback. 
	func onSecurityLevelDowngraded(call: Call)
	
	/// Get the encryption changed callback. 
	/// - Returns: The current encryption changed callback. 
	func onEncryptionChanged(call: Call, on: Bool, authenticationToken: String)
	
	/// Get the authentication token verified callback. 
	/// - Returns: The current authentication token verified callback. 
	func onAuthenticationTokenVerified(call: Call, verified: Bool)
	
	/// Get the send master key changed callback. 
	/// - Returns: The current send master key changed callback. 
	func onSendMasterKeyChanged(call: Call, sendMasterKey: String)
	
	/// Get the receive master key changed callback. 
	/// - Returns: The current receive master key changed callback. 
	func onReceiveMasterKeyChanged(call: Call, receiveMasterKey: String)
	
	/// Get the info message received callback. 
	/// - Returns: The current info message received callback. 
	func onInfoMessageReceived(call: Call, message: InfoMessage)
	
	/// Get the state changed callback. 
	/// - Returns: The current state changed callback. 
	func onStateChanged(call: Call, state: Call.State, message: String)
	
	/// Get the stats updated callback. 
	/// - Returns: The current stats updated callback. 
	func onStatsUpdated(call: Call, stats: CallStats)
	
	/// Get the transfer state changed callback. 
	/// - Returns: The current transfer state changed callback. 
	func onTransferStateChanged(call: Call, state: Call.State)
	
	/// Get the refer requested callback. 
	/// - Returns: The refer requested callback. 
	func onReferRequested(call: Call, referTo: Address)
	
	/// Get the ACK processing callback. 
	/// - Returns: The current ack processing callback. 
	func onAckProcessing(call: Call, ack: Headers, isReceived: Bool)
	
	/// Get the TMMBR received callback. 
	/// - Returns: The current TMMBR received callback. 
	func onTmmbrReceived(call: Call, streamIndex: Int, tmmbr: Int)
	
	/// Get the snapshot taken callback. 
	/// - Returns: The current snapshot taken callback. 
	func onSnapshotTaken(call: Call, filePath: String)
	
	/// Get the next video frame decoded callback. 
	/// - Returns: The current next video frame decoded callback. 
	func onNextVideoFrameDecoded(call: Call)
	
	/// Get the camera not working callback. 
	/// - Returns: The camera not working callback. 
	func onCameraNotWorking(call: Call, cameraName: String)
	
	/// Get the callback that will be used to notify that there are errors from the
	/// video rendering. 
	/// Check LinphoneCallCbsVideoDisplayErrorOccurredCb for more details.
	/// - Returns: The failing rendering callback. 
	func onVideoDisplayErrorOccurred(call: Call, errorCode: Int)
	
	/// Get the audio device changed callback. 
	/// - Returns: The audio device changed callback. 
	func onAudioDeviceChanged(call: Call, audioDevice: AudioDevice)
	
	/// Get the call remote recording callback. 
	/// - Returns: The call remote recording callback. 
	func onRemoteRecording(call: Call, recording: Bool)
	
	/// Get the Baudot detected callback. 
	/// - Returns: The Baudot detected callback. 
	func onBaudotDetected(call: Call, standard: BaudotStandard)
}

public extension CallDelegate {
	
	func onDtmfReceived(call: Call, dtmf: Int) {}
	
	func onGoclearAckSent(call: Call) {}
	
	func onSecurityLevelDowngraded(call: Call) {}
	
	func onEncryptionChanged(call: Call, on: Bool, authenticationToken: String) {}
	
	func onAuthenticationTokenVerified(call: Call, verified: Bool) {}
	
	func onSendMasterKeyChanged(call: Call, sendMasterKey: String) {}
	
	func onReceiveMasterKeyChanged(call: Call, receiveMasterKey: String) {}
	
	func onInfoMessageReceived(call: Call, message: InfoMessage) {}
	
	func onStateChanged(call: Call, state: Call.State, message: String) {}
	
	func onStatsUpdated(call: Call, stats: CallStats) {}
	
	func onTransferStateChanged(call: Call, state: Call.State) {}
	
	func onReferRequested(call: Call, referTo: Address) {}
	
	func onAckProcessing(call: Call, ack: Headers, isReceived: Bool) {}
	
	func onTmmbrReceived(call: Call, streamIndex: Int, tmmbr: Int) {}
	
	func onSnapshotTaken(call: Call, filePath: String) {}
	
	func onNextVideoFrameDecoded(call: Call) {}
	
	func onCameraNotWorking(call: Call, cameraName: String) {}
	
	func onVideoDisplayErrorOccurred(call: Call, errorCode: Int) {}
	
	func onAudioDeviceChanged(call: Call, audioDevice: AudioDevice) {}
	
	func onRemoteRecording(call: Call, recording: Bool) {}
	
	func onBaudotDetected(call: Call, standard: BaudotStandard) {}
}

public final class CallDelegateStub : CallDelegate
{
	var _onDtmfReceived: ((Call, Int) -> Void)?
	var _onGoclearAckSent: ((Call) -> Void)?
	var _onSecurityLevelDowngraded: ((Call) -> Void)?
	var _onEncryptionChanged: ((Call, Bool, String) -> Void)?
	var _onAuthenticationTokenVerified: ((Call, Bool) -> Void)?
	var _onSendMasterKeyChanged: ((Call, String) -> Void)?
	var _onReceiveMasterKeyChanged: ((Call, String) -> Void)?
	var _onInfoMessageReceived: ((Call, InfoMessage) -> Void)?
	var _onStateChanged: ((Call, Call.State, String) -> Void)?
	var _onStatsUpdated: ((Call, CallStats) -> Void)?
	var _onTransferStateChanged: ((Call, Call.State) -> Void)?
	var _onReferRequested: ((Call, Address) -> Void)?
	var _onAckProcessing: ((Call, Headers, Bool) -> Void)?
	var _onTmmbrReceived: ((Call, Int, Int) -> Void)?
	var _onSnapshotTaken: ((Call, String) -> Void)?
	var _onNextVideoFrameDecoded: ((Call) -> Void)?
	var _onCameraNotWorking: ((Call, String) -> Void)?
	var _onVideoDisplayErrorOccurred: ((Call, Int) -> Void)?
	var _onAudioDeviceChanged: ((Call, AudioDevice) -> Void)?
	var _onRemoteRecording: ((Call, Bool) -> Void)?
	var _onBaudotDetected: ((Call, BaudotStandard) -> Void)?

	
	public func onDtmfReceived(call: Call, dtmf: Int){_onDtmfReceived.map{$0(call, dtmf)}}
	
	public func onGoclearAckSent(call: Call){_onGoclearAckSent.map{$0(call)}}
	
	public func onSecurityLevelDowngraded(call: Call){_onSecurityLevelDowngraded.map{$0(call)}}
	
	public func onEncryptionChanged(call: Call, on: Bool, authenticationToken: String){_onEncryptionChanged.map{$0(call, on, authenticationToken)}}
	
	public func onAuthenticationTokenVerified(call: Call, verified: Bool){_onAuthenticationTokenVerified.map{$0(call, verified)}}
	
	public func onSendMasterKeyChanged(call: Call, sendMasterKey: String){_onSendMasterKeyChanged.map{$0(call, sendMasterKey)}}
	
	public func onReceiveMasterKeyChanged(call: Call, receiveMasterKey: String){_onReceiveMasterKeyChanged.map{$0(call, receiveMasterKey)}}
	
	public func onInfoMessageReceived(call: Call, message: InfoMessage){_onInfoMessageReceived.map{$0(call, message)}}
	
	public func onStateChanged(call: Call, state: Call.State, message: String){_onStateChanged.map{$0(call, state, message)}}
	
	public func onStatsUpdated(call: Call, stats: CallStats){_onStatsUpdated.map{$0(call, stats)}}
	
	public func onTransferStateChanged(call: Call, state: Call.State){_onTransferStateChanged.map{$0(call, state)}}
	
	public func onReferRequested(call: Call, referTo: Address){_onReferRequested.map{$0(call, referTo)}}
	
	public func onAckProcessing(call: Call, ack: Headers, isReceived: Bool){_onAckProcessing.map{$0(call, ack, isReceived)}}
	
	public func onTmmbrReceived(call: Call, streamIndex: Int, tmmbr: Int){_onTmmbrReceived.map{$0(call, streamIndex, tmmbr)}}
	
	public func onSnapshotTaken(call: Call, filePath: String){_onSnapshotTaken.map{$0(call, filePath)}}
	
	public func onNextVideoFrameDecoded(call: Call){_onNextVideoFrameDecoded.map{$0(call)}}
	
	public func onCameraNotWorking(call: Call, cameraName: String){_onCameraNotWorking.map{$0(call, cameraName)}}
	
	public func onVideoDisplayErrorOccurred(call: Call, errorCode: Int){_onVideoDisplayErrorOccurred.map{$0(call, errorCode)}}
	
	public func onAudioDeviceChanged(call: Call, audioDevice: AudioDevice){_onAudioDeviceChanged.map{$0(call, audioDevice)}}
	
	public func onRemoteRecording(call: Call, recording: Bool){_onRemoteRecording.map{$0(call, recording)}}
	
	public func onBaudotDetected(call: Call, standard: BaudotStandard){_onBaudotDetected.map{$0(call, standard)}}

	public init (
		onDtmfReceived: ((Call, Int) -> Void)? = nil,
		onGoclearAckSent: ((Call) -> Void)? = nil,
		onSecurityLevelDowngraded: ((Call) -> Void)? = nil,
		onEncryptionChanged: ((Call, Bool, String) -> Void)? = nil,
		onAuthenticationTokenVerified: ((Call, Bool) -> Void)? = nil,
		onSendMasterKeyChanged: ((Call, String) -> Void)? = nil,
		onReceiveMasterKeyChanged: ((Call, String) -> Void)? = nil,
		onInfoMessageReceived: ((Call, InfoMessage) -> Void)? = nil,
		onStateChanged: ((Call, Call.State, String) -> Void)? = nil,
		onStatsUpdated: ((Call, CallStats) -> Void)? = nil,
		onTransferStateChanged: ((Call, Call.State) -> Void)? = nil,
		onReferRequested: ((Call, Address) -> Void)? = nil,
		onAckProcessing: ((Call, Headers, Bool) -> Void)? = nil,
		onTmmbrReceived: ((Call, Int, Int) -> Void)? = nil,
		onSnapshotTaken: ((Call, String) -> Void)? = nil,
		onNextVideoFrameDecoded: ((Call) -> Void)? = nil,
		onCameraNotWorking: ((Call, String) -> Void)? = nil,
		onVideoDisplayErrorOccurred: ((Call, Int) -> Void)? = nil,
		onAudioDeviceChanged: ((Call, AudioDevice) -> Void)? = nil,
		onRemoteRecording: ((Call, Bool) -> Void)? = nil,
		onBaudotDetected: ((Call, BaudotStandard) -> Void)? = nil
	) {
		self._onDtmfReceived = onDtmfReceived
		self._onGoclearAckSent = onGoclearAckSent
		self._onSecurityLevelDowngraded = onSecurityLevelDowngraded
		self._onEncryptionChanged = onEncryptionChanged
		self._onAuthenticationTokenVerified = onAuthenticationTokenVerified
		self._onSendMasterKeyChanged = onSendMasterKeyChanged
		self._onReceiveMasterKeyChanged = onReceiveMasterKeyChanged
		self._onInfoMessageReceived = onInfoMessageReceived
		self._onStateChanged = onStateChanged
		self._onStatsUpdated = onStatsUpdated
		self._onTransferStateChanged = onTransferStateChanged
		self._onReferRequested = onReferRequested
		self._onAckProcessing = onAckProcessing
		self._onTmmbrReceived = onTmmbrReceived
		self._onSnapshotTaken = onSnapshotTaken
		self._onNextVideoFrameDecoded = onNextVideoFrameDecoded
		self._onCameraNotWorking = onCameraNotWorking
		self._onVideoDisplayErrorOccurred = onVideoDisplayErrorOccurred
		self._onAudioDeviceChanged = onAudioDeviceChanged
		self._onRemoteRecording = onRemoteRecording
		self._onBaudotDetected = onBaudotDetected
	}
}

class CallDelegateManager
{
	var cPtr: OpaquePointer
	public var delegate: CallDelegate?

	public init() {
		cPtr = linphone_factory_create_call_cbs(linphone_factory_get())
		belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passRetained(self).toOpaque()), {
			data in
			if (data != nil) {
				Unmanaged<CallDelegateManager>.fromOpaque(data!).release()
			}
		})

		linphone_call_cbs_set_dtmf_received(cPtr, { (call, dtmf) -> Void in
			if (call != nil) {
				let sObject = Call.getSwiftObject(cObject: call!)
				let delegate = sObject.currentDelegate
				delegate?.onDtmfReceived(call: sObject, dtmf: Int(dtmf))
			}
		})

		linphone_call_cbs_set_goclear_ack_sent(cPtr, { (call) -> Void in
			if (call != nil) {
				let sObject = Call.getSwiftObject(cObject: call!)
				let delegate = sObject.currentDelegate
				delegate?.onGoclearAckSent(call: sObject)
			}
		})

		linphone_call_cbs_set_security_level_downgraded(cPtr, { (call) -> Void in
			if (call != nil) {
				let sObject = Call.getSwiftObject(cObject: call!)
				let delegate = sObject.currentDelegate
				delegate?.onSecurityLevelDowngraded(call: sObject)
			}
		})

		linphone_call_cbs_set_encryption_changed(cPtr, { (call, on, authenticationToken) -> Void in
			if (call != nil) {
				let sObject = Call.getSwiftObject(cObject: call!)
				let delegate = sObject.currentDelegate
				delegate?.onEncryptionChanged(call: sObject, on: on != 0, authenticationToken: charArrayToString(charPointer: authenticationToken))
			}
		})

		linphone_call_cbs_set_authentication_token_verified(cPtr, { (call, verified) -> Void in
			if (call != nil) {
				let sObject = Call.getSwiftObject(cObject: call!)
				let delegate = sObject.currentDelegate
				delegate?.onAuthenticationTokenVerified(call: sObject, verified: verified != 0)
			}
		})

		linphone_call_cbs_set_send_master_key_changed(cPtr, { (call, sendMasterKey) -> Void in
			if (call != nil) {
				let sObject = Call.getSwiftObject(cObject: call!)
				let delegate = sObject.currentDelegate
				delegate?.onSendMasterKeyChanged(call: sObject, sendMasterKey: charArrayToString(charPointer: sendMasterKey))
			}
		})

		linphone_call_cbs_set_receive_master_key_changed(cPtr, { (call, receiveMasterKey) -> Void in
			if (call != nil) {
				let sObject = Call.getSwiftObject(cObject: call!)
				let delegate = sObject.currentDelegate
				delegate?.onReceiveMasterKeyChanged(call: sObject, receiveMasterKey: charArrayToString(charPointer: receiveMasterKey))
			}
		})

		linphone_call_cbs_set_info_message_received(cPtr, { (call, message) -> Void in
			if (call != nil) {
				let sObject = Call.getSwiftObject(cObject: call!)
				let delegate = sObject.currentDelegate
				delegate?.onInfoMessageReceived(call: sObject, message: InfoMessage.getSwiftObject(cObject: message!))
			}
		})

		linphone_call_cbs_set_state_changed(cPtr, { (call, state, message) -> Void in
			if (call != nil) {
				let sObject = Call.getSwiftObject(cObject: call!)
				let delegate = sObject.currentDelegate
				delegate?.onStateChanged(call: sObject, state: Call.State(rawValue: Int(state.rawValue))!, message: charArrayToString(charPointer: message))
			}
		})

		linphone_call_cbs_set_stats_updated(cPtr, { (call, stats) -> Void in
			if (call != nil) {
				let sObject = Call.getSwiftObject(cObject: call!)
				let delegate = sObject.currentDelegate
				delegate?.onStatsUpdated(call: sObject, stats: CallStats.getSwiftObject(cObject: stats!))
			}
		})

		linphone_call_cbs_set_transfer_state_changed(cPtr, { (call, state) -> Void in
			if (call != nil) {
				let sObject = Call.getSwiftObject(cObject: call!)
				let delegate = sObject.currentDelegate
				delegate?.onTransferStateChanged(call: sObject, state: Call.State(rawValue: Int(state.rawValue))!)
			}
		})

		linphone_call_cbs_set_refer_requested(cPtr, { (call, referTo) -> Void in
			if (call != nil) {
				let sObject = Call.getSwiftObject(cObject: call!)
				let delegate = sObject.currentDelegate
				delegate?.onReferRequested(call: sObject, referTo: Address.getSwiftObject(cObject: referTo!))
			}
		})

		linphone_call_cbs_set_ack_processing(cPtr, { (call, ack, isReceived) -> Void in
			if (call != nil) {
				let sObject = Call.getSwiftObject(cObject: call!)
				let delegate = sObject.currentDelegate
				delegate?.onAckProcessing(call: sObject, ack: Headers.getSwiftObject(cObject: ack!), isReceived: isReceived != 0)
			}
		})

		linphone_call_cbs_set_tmmbr_received(cPtr, { (call, streamIndex, tmmbr) -> Void in
			if (call != nil) {
				let sObject = Call.getSwiftObject(cObject: call!)
				let delegate = sObject.currentDelegate
				delegate?.onTmmbrReceived(call: sObject, streamIndex: Int(streamIndex), tmmbr: Int(tmmbr))
			}
		})

		linphone_call_cbs_set_snapshot_taken(cPtr, { (call, filePath) -> Void in
			if (call != nil) {
				let sObject = Call.getSwiftObject(cObject: call!)
				let delegate = sObject.currentDelegate
				delegate?.onSnapshotTaken(call: sObject, filePath: charArrayToString(charPointer: filePath))
			}
		})

		linphone_call_cbs_set_next_video_frame_decoded(cPtr, { (call) -> Void in
			if (call != nil) {
				let sObject = Call.getSwiftObject(cObject: call!)
				let delegate = sObject.currentDelegate
				delegate?.onNextVideoFrameDecoded(call: sObject)
			}
		})

		linphone_call_cbs_set_camera_not_working(cPtr, { (call, cameraName) -> Void in
			if (call != nil) {
				let sObject = Call.getSwiftObject(cObject: call!)
				let delegate = sObject.currentDelegate
				delegate?.onCameraNotWorking(call: sObject, cameraName: charArrayToString(charPointer: cameraName))
			}
		})

		linphone_call_cbs_set_video_display_error_occurred(cPtr, { (call, errorCode) -> Void in
			if (call != nil) {
				let sObject = Call.getSwiftObject(cObject: call!)
				let delegate = sObject.currentDelegate
				delegate?.onVideoDisplayErrorOccurred(call: sObject, errorCode: Int(errorCode))
			}
		})

		linphone_call_cbs_set_audio_device_changed(cPtr, { (call, audioDevice) -> Void in
			if (call != nil) {
				let sObject = Call.getSwiftObject(cObject: call!)
				let delegate = sObject.currentDelegate
				delegate?.onAudioDeviceChanged(call: sObject, audioDevice: AudioDevice.getSwiftObject(cObject: audioDevice!))
			}
		})

		linphone_call_cbs_set_remote_recording(cPtr, { (call, recording) -> Void in
			if (call != nil) {
				let sObject = Call.getSwiftObject(cObject: call!)
				let delegate = sObject.currentDelegate
				delegate?.onRemoteRecording(call: sObject, recording: recording != 0)
			}
		})

		linphone_call_cbs_set_baudot_detected(cPtr, { (call, standard) -> Void in
			if (call != nil) {
				let sObject = Call.getSwiftObject(cObject: call!)
				let delegate = sObject.currentDelegate
				delegate?.onBaudotDetected(call: sObject, standard: BaudotStandard(rawValue: Int(standard.rawValue))!)
			}
		})
	}


}

public protocol ChatMessageDelegate : AnyObject {
	
	
	/// Get the message state changed callback. 
	/// - Returns: The current message state changed callback. 
	func onMsgStateChanged(message: ChatMessage, state: ChatMessage.State)
	
	/// Get the new reaction callback. 
	/// - Returns: The current new reaction callback. 
	func onNewMessageReaction(message: ChatMessage, reaction: ChatMessageReaction)
	
	/// Gets the removed reaction callback. 
	/// - Returns: The current new reaction callback. 
	func onReactionRemoved(message: ChatMessage, address: Address)
	
	/// Gets the content edited callback. 
	/// - Returns: The current new content edited callback. 
	func onContentEdited(message: ChatMessage)
	
	/// Gets the retracted callback. 
	/// - Returns: The current new retracted callback. 
	func onRetracted(message: ChatMessage)
	
	/// Get the download file transfer terminated callback. 
	/// - Warning: this callback is called everytime a message attachment is
	/// downloaded. For a message with more than one attachment, this callback will be
	/// called as many times as attachments in the message. 
	/// - Returns: The current file transfer terminated callback. 
	func onFileTransferTerminated(message: ChatMessage, content: Content)
	
	/// Get the file transfer receive callback. 
	/// - Returns: The current file transfer receive callback. 
	func onFileTransferRecv(message: ChatMessage, content: Content, buffer: Buffer)
	
	/// Get the file transfer send callback. 
	/// - Returns: The current file transfer send callback. 
	/// - Deprecated: 17/08/2020 Use
	/// linphone_chat_message_cbs_get_file_transfer_send_chunk instead. 
	func onFileTransferSend(message: ChatMessage, content: Content, offset: Int, size: Int) -> Buffer?
	
	/// Get the file transfer send callback. 
	/// - Returns: The current file transfer send callback. 
	func onFileTransferSendChunk(message: ChatMessage, content: Content, offset: Int, size: Int, buffer: Buffer)
	
	/// Get the file transfer progress indication callback. 
	/// - Returns: The current file transfer progress indication callback. 
	func onFileTransferProgressIndication(message: ChatMessage, content: Content, offset: Int, total: Int)
	
	/// Get the participant IMDN state changed callback. 
	/// - Returns: The current participant IMDN state changed callback. 
	func onParticipantImdnStateChanged(message: ChatMessage, state: ParticipantImdnState)
	
	/// Get the current "ephemeral message timer started" callback. 
	/// This callback is called when the message deletion timer starts (the message has
	/// been viewed). 
	/// - Returns: The current ephemeral message timer started callback. 
	func onEphemeralMessageTimerStarted(message: ChatMessage)
	
	/// Get the ephemeral message deleted callback. 
	/// This callback is used when a message deletion timer runs out (message is
	/// deleted). 
	/// - Returns: The current ephemeral message deleted callback. 
	func onEphemeralMessageDeleted(message: ChatMessage)
}

public extension ChatMessageDelegate {
	
	func onMsgStateChanged(message: ChatMessage, state: ChatMessage.State) {}
	
	func onNewMessageReaction(message: ChatMessage, reaction: ChatMessageReaction) {}
	
	func onReactionRemoved(message: ChatMessage, address: Address) {}
	
	func onContentEdited(message: ChatMessage) {}
	
	func onRetracted(message: ChatMessage) {}
	
	func onFileTransferTerminated(message: ChatMessage, content: Content) {}
	
	func onFileTransferRecv(message: ChatMessage, content: Content, buffer: Buffer) {}
	
	func onFileTransferSend(message: ChatMessage, content: Content, offset: Int, size: Int) -> Buffer? {return nil}
	
	func onFileTransferSendChunk(message: ChatMessage, content: Content, offset: Int, size: Int, buffer: Buffer) {}
	
	func onFileTransferProgressIndication(message: ChatMessage, content: Content, offset: Int, total: Int) {}
	
	func onParticipantImdnStateChanged(message: ChatMessage, state: ParticipantImdnState) {}
	
	func onEphemeralMessageTimerStarted(message: ChatMessage) {}
	
	func onEphemeralMessageDeleted(message: ChatMessage) {}
}

public final class ChatMessageDelegateStub : ChatMessageDelegate
{
	var _onMsgStateChanged: ((ChatMessage, ChatMessage.State) -> Void)?
	var _onNewMessageReaction: ((ChatMessage, ChatMessageReaction) -> Void)?
	var _onReactionRemoved: ((ChatMessage, Address) -> Void)?
	var _onContentEdited: ((ChatMessage) -> Void)?
	var _onRetracted: ((ChatMessage) -> Void)?
	var _onFileTransferTerminated: ((ChatMessage, Content) -> Void)?
	var _onFileTransferRecv: ((ChatMessage, Content, Buffer) -> Void)?
	var _onFileTransferSend: ((ChatMessage, Content, Int, Int) -> Void)?
	var _onFileTransferSendChunk: ((ChatMessage, Content, Int, Int, Buffer) -> Void)?
	var _onFileTransferProgressIndication: ((ChatMessage, Content, Int, Int) -> Void)?
	var _onParticipantImdnStateChanged: ((ChatMessage, ParticipantImdnState) -> Void)?
	var _onEphemeralMessageTimerStarted: ((ChatMessage) -> Void)?
	var _onEphemeralMessageDeleted: ((ChatMessage) -> Void)?

	
	public func onMsgStateChanged(message: ChatMessage, state: ChatMessage.State){_onMsgStateChanged.map{$0(message, state)}}
	
	public func onNewMessageReaction(message: ChatMessage, reaction: ChatMessageReaction){_onNewMessageReaction.map{$0(message, reaction)}}
	
	public func onReactionRemoved(message: ChatMessage, address: Address){_onReactionRemoved.map{$0(message, address)}}
	
	public func onContentEdited(message: ChatMessage){_onContentEdited.map{$0(message)}}
	
	public func onRetracted(message: ChatMessage){_onRetracted.map{$0(message)}}
	
	public func onFileTransferTerminated(message: ChatMessage, content: Content){_onFileTransferTerminated.map{$0(message, content)}}
	
	public func onFileTransferRecv(message: ChatMessage, content: Content, buffer: Buffer){_onFileTransferRecv.map{$0(message, content, buffer)}}
	
	public func onFileTransferSend(message: ChatMessage, content: Content, offset: Int, size: Int){_onFileTransferSend.map{$0(message, content, offset, size)}}
	
	public func onFileTransferSendChunk(message: ChatMessage, content: Content, offset: Int, size: Int, buffer: Buffer){_onFileTransferSendChunk.map{$0(message, content, offset, size, buffer)}}
	
	public func onFileTransferProgressIndication(message: ChatMessage, content: Content, offset: Int, total: Int){_onFileTransferProgressIndication.map{$0(message, content, offset, total)}}
	
	public func onParticipantImdnStateChanged(message: ChatMessage, state: ParticipantImdnState){_onParticipantImdnStateChanged.map{$0(message, state)}}
	
	public func onEphemeralMessageTimerStarted(message: ChatMessage){_onEphemeralMessageTimerStarted.map{$0(message)}}
	
	public func onEphemeralMessageDeleted(message: ChatMessage){_onEphemeralMessageDeleted.map{$0(message)}}

	public init (
		onMsgStateChanged: ((ChatMessage, ChatMessage.State) -> Void)? = nil,
		onNewMessageReaction: ((ChatMessage, ChatMessageReaction) -> Void)? = nil,
		onReactionRemoved: ((ChatMessage, Address) -> Void)? = nil,
		onContentEdited: ((ChatMessage) -> Void)? = nil,
		onRetracted: ((ChatMessage) -> Void)? = nil,
		onFileTransferTerminated: ((ChatMessage, Content) -> Void)? = nil,
		onFileTransferRecv: ((ChatMessage, Content, Buffer) -> Void)? = nil,
		onFileTransferSend: ((ChatMessage, Content, Int, Int) -> Void)? = nil,
		onFileTransferSendChunk: ((ChatMessage, Content, Int, Int, Buffer) -> Void)? = nil,
		onFileTransferProgressIndication: ((ChatMessage, Content, Int, Int) -> Void)? = nil,
		onParticipantImdnStateChanged: ((ChatMessage, ParticipantImdnState) -> Void)? = nil,
		onEphemeralMessageTimerStarted: ((ChatMessage) -> Void)? = nil,
		onEphemeralMessageDeleted: ((ChatMessage) -> Void)? = nil
	) {
		self._onMsgStateChanged = onMsgStateChanged
		self._onNewMessageReaction = onNewMessageReaction
		self._onReactionRemoved = onReactionRemoved
		self._onContentEdited = onContentEdited
		self._onRetracted = onRetracted
		self._onFileTransferTerminated = onFileTransferTerminated
		self._onFileTransferRecv = onFileTransferRecv
		self._onFileTransferSend = onFileTransferSend
		self._onFileTransferSendChunk = onFileTransferSendChunk
		self._onFileTransferProgressIndication = onFileTransferProgressIndication
		self._onParticipantImdnStateChanged = onParticipantImdnStateChanged
		self._onEphemeralMessageTimerStarted = onEphemeralMessageTimerStarted
		self._onEphemeralMessageDeleted = onEphemeralMessageDeleted
	}
}

class ChatMessageDelegateManager
{
	var cPtr: OpaquePointer
	public var delegate: ChatMessageDelegate?

	public init() {
		cPtr = linphone_factory_create_chat_message_cbs(linphone_factory_get())
		belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passRetained(self).toOpaque()), {
			data in
			if (data != nil) {
				Unmanaged<ChatMessageDelegateManager>.fromOpaque(data!).release()
			}
		})

		linphone_chat_message_cbs_set_msg_state_changed(cPtr, { (message, state) -> Void in
			if (message != nil) {
				let sObject = ChatMessage.getSwiftObject(cObject: message!)
				let delegate = sObject.currentDelegate
				delegate?.onMsgStateChanged(message: sObject, state: ChatMessage.State(rawValue: Int(state.rawValue))!)
			}
		})

		linphone_chat_message_cbs_set_new_message_reaction(cPtr, { (message, reaction) -> Void in
			if (message != nil) {
				let sObject = ChatMessage.getSwiftObject(cObject: message!)
				let delegate = sObject.currentDelegate
				delegate?.onNewMessageReaction(message: sObject, reaction: ChatMessageReaction.getSwiftObject(cObject: reaction!))
			}
		})

		linphone_chat_message_cbs_set_reaction_removed(cPtr, { (message, address) -> Void in
			if (message != nil) {
				let sObject = ChatMessage.getSwiftObject(cObject: message!)
				let delegate = sObject.currentDelegate
				delegate?.onReactionRemoved(message: sObject, address: Address.getSwiftObject(cObject: address!))
			}
		})

		linphone_chat_message_cbs_set_content_edited(cPtr, { (message) -> Void in
			if (message != nil) {
				let sObject = ChatMessage.getSwiftObject(cObject: message!)
				let delegate = sObject.currentDelegate
				delegate?.onContentEdited(message: sObject)
			}
		})

		linphone_chat_message_cbs_set_retracted(cPtr, { (message) -> Void in
			if (message != nil) {
				let sObject = ChatMessage.getSwiftObject(cObject: message!)
				let delegate = sObject.currentDelegate
				delegate?.onRetracted(message: sObject)
			}
		})

		linphone_chat_message_cbs_set_file_transfer_terminated(cPtr, { (message, content) -> Void in
			if (message != nil) {
				let sObject = ChatMessage.getSwiftObject(cObject: message!)
				let delegate = sObject.currentDelegate
				delegate?.onFileTransferTerminated(message: sObject, content: Content.getSwiftObject(cObject: content!))
			}
		})

		linphone_chat_message_cbs_set_file_transfer_recv(cPtr, { (message, content, buffer) -> Void in
			if (message != nil) {
				let sObject = ChatMessage.getSwiftObject(cObject: message!)
				let delegate = sObject.currentDelegate
				delegate?.onFileTransferRecv(message: sObject, content: Content.getSwiftObject(cObject: content!), buffer: Buffer.getSwiftObject(cObject: buffer!))
			}
		})

		linphone_chat_message_cbs_set_file_transfer_send(cPtr, { (message, content, offset, size) -> OpaquePointer? in
			if (message != nil) {
				let sObject = ChatMessage.getSwiftObject(cObject: message!)
				let delegate = sObject.currentDelegate
				return delegate?.onFileTransferSend(message: sObject, content: Content.getSwiftObject(cObject: content!), offset: Int(offset), size: Int(size))?.cPtr
			} else {
				return nil
			}
		})

		linphone_chat_message_cbs_set_file_transfer_send_chunk(cPtr, { (message, content, offset, size, buffer) -> Void in
			if (message != nil) {
				let sObject = ChatMessage.getSwiftObject(cObject: message!)
				let delegate = sObject.currentDelegate
				delegate?.onFileTransferSendChunk(message: sObject, content: Content.getSwiftObject(cObject: content!), offset: Int(offset), size: Int(size), buffer: Buffer.getSwiftObject(cObject: buffer!))
			}
		})

		linphone_chat_message_cbs_set_file_transfer_progress_indication(cPtr, { (message, content, offset, total) -> Void in
			if (message != nil) {
				let sObject = ChatMessage.getSwiftObject(cObject: message!)
				let delegate = sObject.currentDelegate
				delegate?.onFileTransferProgressIndication(message: sObject, content: Content.getSwiftObject(cObject: content!), offset: Int(offset), total: Int(total))
			}
		})

		linphone_chat_message_cbs_set_participant_imdn_state_changed(cPtr, { (message, state) -> Void in
			if (message != nil) {
				let sObject = ChatMessage.getSwiftObject(cObject: message!)
				let delegate = sObject.currentDelegate
				delegate?.onParticipantImdnStateChanged(message: sObject, state: ParticipantImdnState.getSwiftObject(cObject: state!))
			}
		})

		linphone_chat_message_cbs_set_ephemeral_message_timer_started(cPtr, { (message) -> Void in
			if (message != nil) {
				let sObject = ChatMessage.getSwiftObject(cObject: message!)
				let delegate = sObject.currentDelegate
				delegate?.onEphemeralMessageTimerStarted(message: sObject)
			}
		})

		linphone_chat_message_cbs_set_ephemeral_message_deleted(cPtr, { (message) -> Void in
			if (message != nil) {
				let sObject = ChatMessage.getSwiftObject(cObject: message!)
				let delegate = sObject.currentDelegate
				delegate?.onEphemeralMessageDeleted(message: sObject)
			}
		})
	}


}

public protocol ChatRoomDelegate : AnyObject {
	
	
	/// Get the is-composing received callback. 
	/// - Returns: The current is-composing received callback. 
	func onIsComposingReceived(chatRoom: ChatRoom, remoteAddress: Address, isComposing: Bool)
	
	/// Get the message received callback. 
	/// - Returns: The current message received callback. 
	func onMessageReceived(chatRoom: ChatRoom, message: ChatMessage)
	
	/// Get the chat messages received callback. 
	/// Only called when aggregation is enabled (aka [sip] chat_messages_aggregation ==
	/// 1 or using linphone_core_set_chat_messages_aggregation_enabled), it replaces
	/// the single message received callback. 
	/// - Returns: The current chat messages received callback. 
	func onMessagesReceived(chatRoom: ChatRoom, chatMessages: [ChatMessage])
	
	/// Get the new event log callback. 
	/// This callback will be called before every other ``EventLog`` related callback. 
	/// - Returns: The current event log created callback. 
	func onNewEvent(chatRoom: ChatRoom, eventLog: EventLog)
	
	/// Get the new event logs callback. 
	/// This callback will be called before every other ``EventLog`` related callback. 
	/// - Returns: The current event logs created callback. 
	func onNewEvents(chatRoom: ChatRoom, eventLogs: [EventLog])
	
	/// Get the chat message received callback. 
	/// - Returns: The current chat message received callback. 
	func onChatMessageReceived(chatRoom: ChatRoom, eventLog: EventLog)
	
	/// Get the chat messages received callback. 
	/// Only called when aggregation is enabled (aka [sip] chat_messages_aggregation ==
	/// 1 or using ``Core/setChatMessagesAggregationEnabled(enabled:)``), it replaces
	/// the single chat message received callback. 
	/// - Returns: The current chat message received callback. 
	func onChatMessagesReceived(chatRoom: ChatRoom, eventLogs: [EventLog])
	
	/// Get the chat message sending callback. 
	/// - Returns: The current chat message being sent callback. 
	func onChatMessageSending(chatRoom: ChatRoom, eventLog: EventLog)
	
	/// Get the chat message sent callback. 
	/// - Returns: The current chat message sent callback. 
	func onChatMessageSent(chatRoom: ChatRoom, eventLog: EventLog)
	
	/// Get the participant added callback. 
	/// - Returns: The current participant added callback. 
	func onParticipantAdded(chatRoom: ChatRoom, eventLog: EventLog)
	
	/// Get the participant removed callback. 
	/// - Returns: The current participant removed callback. 
	func onParticipantRemoved(chatRoom: ChatRoom, eventLog: EventLog)
	
	/// Get the participant admin status changed callback. 
	/// - Returns: The current participant admin status changed callback. 
	func onParticipantAdminStatusChanged(chatRoom: ChatRoom, eventLog: EventLog)
	
	/// Get the leave failed callback. 
	/// - Returns: The current leave failed callback. 
	func onOperationFailed(chatRoom: ChatRoom)
	
	/// Get the state changed callback. 
	/// - Returns: The current state changed callback. 
	func onStateChanged(chatRoom: ChatRoom, newState: ChatRoom.State)
	
	/// Get the security event callback. 
	/// - Returns: The security event callback to be used. 
	func onSecurityEvent(chatRoom: ChatRoom, eventLog: EventLog)
	
	/// Get the subject changed callback. 
	/// - Returns: The current subject changed callback. 
	func onSubjectChanged(chatRoom: ChatRoom, eventLog: EventLog)
	
	/// Get the message early failure callback. 
	/// - Returns: The current message early failure callback. 
	func onMessageEarlyFailure(chatRoom: ChatRoom, eventLog: EventLog)
	
	/// Get the undecryptable message received callback. 
	/// - Returns: The current undecryptable message received callback. 
	func onUndecryptableMessageReceived(chatRoom: ChatRoom, message: ChatMessage)
	
	/// Get the participant device added callback. 
	/// - Returns: The current participant device added callback. 
	func onParticipantDeviceAdded(chatRoom: ChatRoom, eventLog: EventLog)
	
	/// Get the participant device removed callback. 
	/// - Returns: The current participant device removed callback. 
	func onParticipantDeviceRemoved(chatRoom: ChatRoom, eventLog: EventLog)
	
	/// Get the participant device state callback. 
	/// - Returns: The current participant device state callback. 
	func onParticipantDeviceStateChanged(chatRoom: ChatRoom, eventLog: EventLog, state: ParticipantDevice.State)
	
	/// Get the participant device media availability changed callback. 
	/// - Returns: The current participant device media availability changed callback. 
	func onParticipantDeviceMediaAvailabilityChanged(chatRoom: ChatRoom, eventLog: EventLog)
	
	/// Get the conference joined callback. 
	/// - Returns: The current conference joined callback. 
	func onConferenceJoined(chatRoom: ChatRoom, eventLog: EventLog)
	
	/// Get the conference left callback. 
	/// - Returns: The current conference left callback. 
	func onConferenceLeft(chatRoom: ChatRoom, eventLog: EventLog)
	
	/// Get the ephemeral event callback. 
	/// - Returns: The ephemeral event callback to be used. 
	func onEphemeralEvent(chatRoom: ChatRoom, eventLog: EventLog)
	
	/// Get the current "ephemeral message timer started" callback. 
	/// This callback is called when a message deletion timer starts (the message has
	/// been viewed). 
	/// - Returns: The current ephemeral message "timer started" callback. 
	func onEphemeralMessageTimerStarted(chatRoom: ChatRoom, eventLog: EventLog)
	
	/// Get the ephemeral message deleted callback. 
	/// This callback is used when a message deletion timer runs out (message is
	/// deleted). 
	/// - Returns: The current ephemeral message deleted callback. 
	func onEphemeralMessageDeleted(chatRoom: ChatRoom, eventLog: EventLog)
	
	/// Get the conference address generation callback. 
	/// - Returns: The current conference address generation callback 
	func onConferenceAddressGeneration(chatRoom: ChatRoom)
	
	/// Get the participant registration subscription callback. 
	/// - Returns: The participant registration subscription callback 
	func onParticipantRegistrationSubscriptionRequested(chatRoom: ChatRoom, participantAddress: Address)
	
	/// Get the participant registration unsubscription callback. 
	/// - Returns: The participant registration unsubscription callback 
	func onParticipantRegistrationUnsubscriptionRequested(chatRoom: ChatRoom, participantAddress: Address)
	
	/// Get the message should be stored callback. 
	/// - Returns: The message should be stored callback 
	func onChatMessageShouldBeStored(chatRoom: ChatRoom, message: ChatMessage)
	
	/// Get the message's participant state changed callback. 
	/// - Returns: The message's participant state changed callback callback 
	func onChatMessageParticipantImdnStateChanged(chatRoom: ChatRoom, message: ChatMessage, state: ParticipantImdnState)
	
	/// Get the "marked as read" callback. 
	/// - Returns: The marked as read callback. 
	func onChatRoomRead(chatRoom: ChatRoom)
	
	/// Get the new reaction callback. 
	/// - Returns: The current new reaction callback. 
	func onNewMessageReaction(chatRoom: ChatRoom, message: ChatMessage, reaction: ChatMessageReaction)
	
	/// Get the message content edited callback. 
	/// - Returns: The current message content edited callback. 
	func onMessageContentEdited(chatRoom: ChatRoom, message: ChatMessage)
	
	/// Get the message retracted callback. 
	/// - Returns: The current message retracted callback. 
	func onMessageRetracted(chatRoom: ChatRoom, message: ChatMessage)
}

public extension ChatRoomDelegate {
	
	func onIsComposingReceived(chatRoom: ChatRoom, remoteAddress: Address, isComposing: Bool) {}
	
	func onMessageReceived(chatRoom: ChatRoom, message: ChatMessage) {}
	
	func onMessagesReceived(chatRoom: ChatRoom, chatMessages: [ChatMessage]) {}
	
	func onNewEvent(chatRoom: ChatRoom, eventLog: EventLog) {}
	
	func onNewEvents(chatRoom: ChatRoom, eventLogs: [EventLog]) {}
	
	func onChatMessageReceived(chatRoom: ChatRoom, eventLog: EventLog) {}
	
	func onChatMessagesReceived(chatRoom: ChatRoom, eventLogs: [EventLog]) {}
	
	func onChatMessageSending(chatRoom: ChatRoom, eventLog: EventLog) {}
	
	func onChatMessageSent(chatRoom: ChatRoom, eventLog: EventLog) {}
	
	func onParticipantAdded(chatRoom: ChatRoom, eventLog: EventLog) {}
	
	func onParticipantRemoved(chatRoom: ChatRoom, eventLog: EventLog) {}
	
	func onParticipantAdminStatusChanged(chatRoom: ChatRoom, eventLog: EventLog) {}
	
	func onOperationFailed(chatRoom: ChatRoom) {}
	
	func onStateChanged(chatRoom: ChatRoom, newState: ChatRoom.State) {}
	
	func onSecurityEvent(chatRoom: ChatRoom, eventLog: EventLog) {}
	
	func onSubjectChanged(chatRoom: ChatRoom, eventLog: EventLog) {}
	
	func onMessageEarlyFailure(chatRoom: ChatRoom, eventLog: EventLog) {}
	
	func onUndecryptableMessageReceived(chatRoom: ChatRoom, message: ChatMessage) {}
	
	func onParticipantDeviceAdded(chatRoom: ChatRoom, eventLog: EventLog) {}
	
	func onParticipantDeviceRemoved(chatRoom: ChatRoom, eventLog: EventLog) {}
	
	func onParticipantDeviceStateChanged(chatRoom: ChatRoom, eventLog: EventLog, state: ParticipantDevice.State) {}
	
	func onParticipantDeviceMediaAvailabilityChanged(chatRoom: ChatRoom, eventLog: EventLog) {}
	
	func onConferenceJoined(chatRoom: ChatRoom, eventLog: EventLog) {}
	
	func onConferenceLeft(chatRoom: ChatRoom, eventLog: EventLog) {}
	
	func onEphemeralEvent(chatRoom: ChatRoom, eventLog: EventLog) {}
	
	func onEphemeralMessageTimerStarted(chatRoom: ChatRoom, eventLog: EventLog) {}
	
	func onEphemeralMessageDeleted(chatRoom: ChatRoom, eventLog: EventLog) {}
	
	func onConferenceAddressGeneration(chatRoom: ChatRoom) {}
	
	func onParticipantRegistrationSubscriptionRequested(chatRoom: ChatRoom, participantAddress: Address) {}
	
	func onParticipantRegistrationUnsubscriptionRequested(chatRoom: ChatRoom, participantAddress: Address) {}
	
	func onChatMessageShouldBeStored(chatRoom: ChatRoom, message: ChatMessage) {}
	
	func onChatMessageParticipantImdnStateChanged(chatRoom: ChatRoom, message: ChatMessage, state: ParticipantImdnState) {}
	
	func onChatRoomRead(chatRoom: ChatRoom) {}
	
	func onNewMessageReaction(chatRoom: ChatRoom, message: ChatMessage, reaction: ChatMessageReaction) {}
	
	func onMessageContentEdited(chatRoom: ChatRoom, message: ChatMessage) {}
	
	func onMessageRetracted(chatRoom: ChatRoom, message: ChatMessage) {}
}

public final class ChatRoomDelegateStub : ChatRoomDelegate
{
	var _onIsComposingReceived: ((ChatRoom, Address, Bool) -> Void)?
	var _onMessageReceived: ((ChatRoom, ChatMessage) -> Void)?
	var _onMessagesReceived: ((ChatRoom, [ChatMessage]) -> Void)?
	var _onNewEvent: ((ChatRoom, EventLog) -> Void)?
	var _onNewEvents: ((ChatRoom, [EventLog]) -> Void)?
	var _onChatMessageReceived: ((ChatRoom, EventLog) -> Void)?
	var _onChatMessagesReceived: ((ChatRoom, [EventLog]) -> Void)?
	var _onChatMessageSending: ((ChatRoom, EventLog) -> Void)?
	var _onChatMessageSent: ((ChatRoom, EventLog) -> Void)?
	var _onParticipantAdded: ((ChatRoom, EventLog) -> Void)?
	var _onParticipantRemoved: ((ChatRoom, EventLog) -> Void)?
	var _onParticipantAdminStatusChanged: ((ChatRoom, EventLog) -> Void)?
	var _onOperationFailed: ((ChatRoom) -> Void)?
	var _onStateChanged: ((ChatRoom, ChatRoom.State) -> Void)?
	var _onSecurityEvent: ((ChatRoom, EventLog) -> Void)?
	var _onSubjectChanged: ((ChatRoom, EventLog) -> Void)?
	var _onMessageEarlyFailure: ((ChatRoom, EventLog) -> Void)?
	var _onUndecryptableMessageReceived: ((ChatRoom, ChatMessage) -> Void)?
	var _onParticipantDeviceAdded: ((ChatRoom, EventLog) -> Void)?
	var _onParticipantDeviceRemoved: ((ChatRoom, EventLog) -> Void)?
	var _onParticipantDeviceStateChanged: ((ChatRoom, EventLog, ParticipantDevice.State) -> Void)?
	var _onParticipantDeviceMediaAvailabilityChanged: ((ChatRoom, EventLog) -> Void)?
	var _onConferenceJoined: ((ChatRoom, EventLog) -> Void)?
	var _onConferenceLeft: ((ChatRoom, EventLog) -> Void)?
	var _onEphemeralEvent: ((ChatRoom, EventLog) -> Void)?
	var _onEphemeralMessageTimerStarted: ((ChatRoom, EventLog) -> Void)?
	var _onEphemeralMessageDeleted: ((ChatRoom, EventLog) -> Void)?
	var _onConferenceAddressGeneration: ((ChatRoom) -> Void)?
	var _onParticipantRegistrationSubscriptionRequested: ((ChatRoom, Address) -> Void)?
	var _onParticipantRegistrationUnsubscriptionRequested: ((ChatRoom, Address) -> Void)?
	var _onChatMessageShouldBeStored: ((ChatRoom, ChatMessage) -> Void)?
	var _onChatMessageParticipantImdnStateChanged: ((ChatRoom, ChatMessage, ParticipantImdnState) -> Void)?
	var _onChatRoomRead: ((ChatRoom) -> Void)?
	var _onNewMessageReaction: ((ChatRoom, ChatMessage, ChatMessageReaction) -> Void)?
	var _onMessageContentEdited: ((ChatRoom, ChatMessage) -> Void)?
	var _onMessageRetracted: ((ChatRoom, ChatMessage) -> Void)?

	
	public func onIsComposingReceived(chatRoom: ChatRoom, remoteAddress: Address, isComposing: Bool){_onIsComposingReceived.map{$0(chatRoom, remoteAddress, isComposing)}}
	
	public func onMessageReceived(chatRoom: ChatRoom, message: ChatMessage){_onMessageReceived.map{$0(chatRoom, message)}}
	
	public func onMessagesReceived(chatRoom: ChatRoom, chatMessages: [ChatMessage]){_onMessagesReceived.map{$0(chatRoom, chatMessages)}}
	
	public func onNewEvent(chatRoom: ChatRoom, eventLog: EventLog){_onNewEvent.map{$0(chatRoom, eventLog)}}
	
	public func onNewEvents(chatRoom: ChatRoom, eventLogs: [EventLog]){_onNewEvents.map{$0(chatRoom, eventLogs)}}
	
	public func onChatMessageReceived(chatRoom: ChatRoom, eventLog: EventLog){_onChatMessageReceived.map{$0(chatRoom, eventLog)}}
	
	public func onChatMessagesReceived(chatRoom: ChatRoom, eventLogs: [EventLog]){_onChatMessagesReceived.map{$0(chatRoom, eventLogs)}}
	
	public func onChatMessageSending(chatRoom: ChatRoom, eventLog: EventLog){_onChatMessageSending.map{$0(chatRoom, eventLog)}}
	
	public func onChatMessageSent(chatRoom: ChatRoom, eventLog: EventLog){_onChatMessageSent.map{$0(chatRoom, eventLog)}}
	
	public func onParticipantAdded(chatRoom: ChatRoom, eventLog: EventLog){_onParticipantAdded.map{$0(chatRoom, eventLog)}}
	
	public func onParticipantRemoved(chatRoom: ChatRoom, eventLog: EventLog){_onParticipantRemoved.map{$0(chatRoom, eventLog)}}
	
	public func onParticipantAdminStatusChanged(chatRoom: ChatRoom, eventLog: EventLog){_onParticipantAdminStatusChanged.map{$0(chatRoom, eventLog)}}
	
	public func onOperationFailed(chatRoom: ChatRoom){_onOperationFailed.map{$0(chatRoom)}}
	
	public func onStateChanged(chatRoom: ChatRoom, newState: ChatRoom.State){_onStateChanged.map{$0(chatRoom, newState)}}
	
	public func onSecurityEvent(chatRoom: ChatRoom, eventLog: EventLog){_onSecurityEvent.map{$0(chatRoom, eventLog)}}
	
	public func onSubjectChanged(chatRoom: ChatRoom, eventLog: EventLog){_onSubjectChanged.map{$0(chatRoom, eventLog)}}
	
	public func onMessageEarlyFailure(chatRoom: ChatRoom, eventLog: EventLog){_onMessageEarlyFailure.map{$0(chatRoom, eventLog)}}
	
	public func onUndecryptableMessageReceived(chatRoom: ChatRoom, message: ChatMessage){_onUndecryptableMessageReceived.map{$0(chatRoom, message)}}
	
	public func onParticipantDeviceAdded(chatRoom: ChatRoom, eventLog: EventLog){_onParticipantDeviceAdded.map{$0(chatRoom, eventLog)}}
	
	public func onParticipantDeviceRemoved(chatRoom: ChatRoom, eventLog: EventLog){_onParticipantDeviceRemoved.map{$0(chatRoom, eventLog)}}
	
	public func onParticipantDeviceStateChanged(chatRoom: ChatRoom, eventLog: EventLog, state: ParticipantDevice.State){_onParticipantDeviceStateChanged.map{$0(chatRoom, eventLog, state)}}
	
	public func onParticipantDeviceMediaAvailabilityChanged(chatRoom: ChatRoom, eventLog: EventLog){_onParticipantDeviceMediaAvailabilityChanged.map{$0(chatRoom, eventLog)}}
	
	public func onConferenceJoined(chatRoom: ChatRoom, eventLog: EventLog){_onConferenceJoined.map{$0(chatRoom, eventLog)}}
	
	public func onConferenceLeft(chatRoom: ChatRoom, eventLog: EventLog){_onConferenceLeft.map{$0(chatRoom, eventLog)}}
	
	public func onEphemeralEvent(chatRoom: ChatRoom, eventLog: EventLog){_onEphemeralEvent.map{$0(chatRoom, eventLog)}}
	
	public func onEphemeralMessageTimerStarted(chatRoom: ChatRoom, eventLog: EventLog){_onEphemeralMessageTimerStarted.map{$0(chatRoom, eventLog)}}
	
	public func onEphemeralMessageDeleted(chatRoom: ChatRoom, eventLog: EventLog){_onEphemeralMessageDeleted.map{$0(chatRoom, eventLog)}}
	
	public func onConferenceAddressGeneration(chatRoom: ChatRoom){_onConferenceAddressGeneration.map{$0(chatRoom)}}
	
	public func onParticipantRegistrationSubscriptionRequested(chatRoom: ChatRoom, participantAddress: Address){_onParticipantRegistrationSubscriptionRequested.map{$0(chatRoom, participantAddress)}}
	
	public func onParticipantRegistrationUnsubscriptionRequested(chatRoom: ChatRoom, participantAddress: Address){_onParticipantRegistrationUnsubscriptionRequested.map{$0(chatRoom, participantAddress)}}
	
	public func onChatMessageShouldBeStored(chatRoom: ChatRoom, message: ChatMessage){_onChatMessageShouldBeStored.map{$0(chatRoom, message)}}
	
	public func onChatMessageParticipantImdnStateChanged(chatRoom: ChatRoom, message: ChatMessage, state: ParticipantImdnState){_onChatMessageParticipantImdnStateChanged.map{$0(chatRoom, message, state)}}
	
	public func onChatRoomRead(chatRoom: ChatRoom){_onChatRoomRead.map{$0(chatRoom)}}
	
	public func onNewMessageReaction(chatRoom: ChatRoom, message: ChatMessage, reaction: ChatMessageReaction){_onNewMessageReaction.map{$0(chatRoom, message, reaction)}}
	
	public func onMessageContentEdited(chatRoom: ChatRoom, message: ChatMessage){_onMessageContentEdited.map{$0(chatRoom, message)}}
	
	public func onMessageRetracted(chatRoom: ChatRoom, message: ChatMessage){_onMessageRetracted.map{$0(chatRoom, message)}}

	public init (
		onIsComposingReceived: ((ChatRoom, Address, Bool) -> Void)? = nil,
		onMessageReceived: ((ChatRoom, ChatMessage) -> Void)? = nil,
		onMessagesReceived: ((ChatRoom, [ChatMessage]) -> Void)? = nil,
		onNewEvent: ((ChatRoom, EventLog) -> Void)? = nil,
		onNewEvents: ((ChatRoom, [EventLog]) -> Void)? = nil,
		onChatMessageReceived: ((ChatRoom, EventLog) -> Void)? = nil,
		onChatMessagesReceived: ((ChatRoom, [EventLog]) -> Void)? = nil,
		onChatMessageSending: ((ChatRoom, EventLog) -> Void)? = nil,
		onChatMessageSent: ((ChatRoom, EventLog) -> Void)? = nil,
		onParticipantAdded: ((ChatRoom, EventLog) -> Void)? = nil,
		onParticipantRemoved: ((ChatRoom, EventLog) -> Void)? = nil,
		onParticipantAdminStatusChanged: ((ChatRoom, EventLog) -> Void)? = nil,
		onOperationFailed: ((ChatRoom) -> Void)? = nil,
		onStateChanged: ((ChatRoom, ChatRoom.State) -> Void)? = nil,
		onSecurityEvent: ((ChatRoom, EventLog) -> Void)? = nil,
		onSubjectChanged: ((ChatRoom, EventLog) -> Void)? = nil,
		onMessageEarlyFailure: ((ChatRoom, EventLog) -> Void)? = nil,
		onUndecryptableMessageReceived: ((ChatRoom, ChatMessage) -> Void)? = nil,
		onParticipantDeviceAdded: ((ChatRoom, EventLog) -> Void)? = nil,
		onParticipantDeviceRemoved: ((ChatRoom, EventLog) -> Void)? = nil,
		onParticipantDeviceStateChanged: ((ChatRoom, EventLog, ParticipantDevice.State) -> Void)? = nil,
		onParticipantDeviceMediaAvailabilityChanged: ((ChatRoom, EventLog) -> Void)? = nil,
		onConferenceJoined: ((ChatRoom, EventLog) -> Void)? = nil,
		onConferenceLeft: ((ChatRoom, EventLog) -> Void)? = nil,
		onEphemeralEvent: ((ChatRoom, EventLog) -> Void)? = nil,
		onEphemeralMessageTimerStarted: ((ChatRoom, EventLog) -> Void)? = nil,
		onEphemeralMessageDeleted: ((ChatRoom, EventLog) -> Void)? = nil,
		onConferenceAddressGeneration: ((ChatRoom) -> Void)? = nil,
		onParticipantRegistrationSubscriptionRequested: ((ChatRoom, Address) -> Void)? = nil,
		onParticipantRegistrationUnsubscriptionRequested: ((ChatRoom, Address) -> Void)? = nil,
		onChatMessageShouldBeStored: ((ChatRoom, ChatMessage) -> Void)? = nil,
		onChatMessageParticipantImdnStateChanged: ((ChatRoom, ChatMessage, ParticipantImdnState) -> Void)? = nil,
		onChatRoomRead: ((ChatRoom) -> Void)? = nil,
		onNewMessageReaction: ((ChatRoom, ChatMessage, ChatMessageReaction) -> Void)? = nil,
		onMessageContentEdited: ((ChatRoom, ChatMessage) -> Void)? = nil,
		onMessageRetracted: ((ChatRoom, ChatMessage) -> Void)? = nil
	) {
		self._onIsComposingReceived = onIsComposingReceived
		self._onMessageReceived = onMessageReceived
		self._onMessagesReceived = onMessagesReceived
		self._onNewEvent = onNewEvent
		self._onNewEvents = onNewEvents
		self._onChatMessageReceived = onChatMessageReceived
		self._onChatMessagesReceived = onChatMessagesReceived
		self._onChatMessageSending = onChatMessageSending
		self._onChatMessageSent = onChatMessageSent
		self._onParticipantAdded = onParticipantAdded
		self._onParticipantRemoved = onParticipantRemoved
		self._onParticipantAdminStatusChanged = onParticipantAdminStatusChanged
		self._onOperationFailed = onOperationFailed
		self._onStateChanged = onStateChanged
		self._onSecurityEvent = onSecurityEvent
		self._onSubjectChanged = onSubjectChanged
		self._onMessageEarlyFailure = onMessageEarlyFailure
		self._onUndecryptableMessageReceived = onUndecryptableMessageReceived
		self._onParticipantDeviceAdded = onParticipantDeviceAdded
		self._onParticipantDeviceRemoved = onParticipantDeviceRemoved
		self._onParticipantDeviceStateChanged = onParticipantDeviceStateChanged
		self._onParticipantDeviceMediaAvailabilityChanged = onParticipantDeviceMediaAvailabilityChanged
		self._onConferenceJoined = onConferenceJoined
		self._onConferenceLeft = onConferenceLeft
		self._onEphemeralEvent = onEphemeralEvent
		self._onEphemeralMessageTimerStarted = onEphemeralMessageTimerStarted
		self._onEphemeralMessageDeleted = onEphemeralMessageDeleted
		self._onConferenceAddressGeneration = onConferenceAddressGeneration
		self._onParticipantRegistrationSubscriptionRequested = onParticipantRegistrationSubscriptionRequested
		self._onParticipantRegistrationUnsubscriptionRequested = onParticipantRegistrationUnsubscriptionRequested
		self._onChatMessageShouldBeStored = onChatMessageShouldBeStored
		self._onChatMessageParticipantImdnStateChanged = onChatMessageParticipantImdnStateChanged
		self._onChatRoomRead = onChatRoomRead
		self._onNewMessageReaction = onNewMessageReaction
		self._onMessageContentEdited = onMessageContentEdited
		self._onMessageRetracted = onMessageRetracted
	}
}

class ChatRoomDelegateManager
{
	var cPtr: OpaquePointer
	public var delegate: ChatRoomDelegate?

	public init() {
		cPtr = linphone_factory_create_chat_room_cbs(linphone_factory_get())
		belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passRetained(self).toOpaque()), {
			data in
			if (data != nil) {
				Unmanaged<ChatRoomDelegateManager>.fromOpaque(data!).release()
			}
		})

		linphone_chat_room_cbs_set_is_composing_received(cPtr, { (chatRoom, remoteAddress, isComposing) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onIsComposingReceived(chatRoom: sObject, remoteAddress: Address.getSwiftObject(cObject: remoteAddress!), isComposing: isComposing != 0)
			}
		})

		linphone_chat_room_cbs_set_message_received(cPtr, { (chatRoom, message) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onMessageReceived(chatRoom: sObject, message: ChatMessage.getSwiftObject(cObject: message!))
			}
		})

		linphone_chat_room_cbs_set_messages_received(cPtr, { (chatRoom, chatMessages) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				var chatMessagessList = [ChatMessage]()
				let chatMessagescList = chatMessages
				var listTemp = chatMessagescList
				while (listTemp != nil) {
					let data = unsafeBitCast(listTemp!.pointee.data, to: OpaquePointer.self)
					chatMessagessList.append(ChatMessage.getSwiftObject(cObject: data))
					listTemp = UnsafePointer<bctbx_list_t>(listTemp!.pointee.next)
				}
				delegate?.onMessagesReceived(chatRoom: sObject, chatMessages: chatMessagessList)
			}
		})

		linphone_chat_room_cbs_set_new_event(cPtr, { (chatRoom, eventLog) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onNewEvent(chatRoom: sObject, eventLog: EventLog.getSwiftObject(cObject: eventLog!))
			}
		})

		linphone_chat_room_cbs_set_new_events(cPtr, { (chatRoom, eventLogs) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				var eventLogssList = [EventLog]()
				let eventLogscList = eventLogs
				var listTemp = eventLogscList
				while (listTemp != nil) {
					let data = unsafeBitCast(listTemp!.pointee.data, to: OpaquePointer.self)
					eventLogssList.append(EventLog.getSwiftObject(cObject: data))
					listTemp = UnsafePointer<bctbx_list_t>(listTemp!.pointee.next)
				}
				delegate?.onNewEvents(chatRoom: sObject, eventLogs: eventLogssList)
			}
		})

		linphone_chat_room_cbs_set_chat_message_received(cPtr, { (chatRoom, eventLog) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onChatMessageReceived(chatRoom: sObject, eventLog: EventLog.getSwiftObject(cObject: eventLog!))
			}
		})

		linphone_chat_room_cbs_set_chat_messages_received(cPtr, { (chatRoom, eventLogs) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				var eventLogssList = [EventLog]()
				let eventLogscList = eventLogs
				var listTemp = eventLogscList
				while (listTemp != nil) {
					let data = unsafeBitCast(listTemp!.pointee.data, to: OpaquePointer.self)
					eventLogssList.append(EventLog.getSwiftObject(cObject: data))
					listTemp = UnsafePointer<bctbx_list_t>(listTemp!.pointee.next)
				}
				delegate?.onChatMessagesReceived(chatRoom: sObject, eventLogs: eventLogssList)
			}
		})

		linphone_chat_room_cbs_set_chat_message_sending(cPtr, { (chatRoom, eventLog) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onChatMessageSending(chatRoom: sObject, eventLog: EventLog.getSwiftObject(cObject: eventLog!))
			}
		})

		linphone_chat_room_cbs_set_chat_message_sent(cPtr, { (chatRoom, eventLog) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onChatMessageSent(chatRoom: sObject, eventLog: EventLog.getSwiftObject(cObject: eventLog!))
			}
		})

		linphone_chat_room_cbs_set_participant_added(cPtr, { (chatRoom, eventLog) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onParticipantAdded(chatRoom: sObject, eventLog: EventLog.getSwiftObject(cObject: eventLog!))
			}
		})

		linphone_chat_room_cbs_set_participant_removed(cPtr, { (chatRoom, eventLog) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onParticipantRemoved(chatRoom: sObject, eventLog: EventLog.getSwiftObject(cObject: eventLog!))
			}
		})

		linphone_chat_room_cbs_set_participant_admin_status_changed(cPtr, { (chatRoom, eventLog) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onParticipantAdminStatusChanged(chatRoom: sObject, eventLog: EventLog.getSwiftObject(cObject: eventLog!))
			}
		})

		linphone_chat_room_cbs_set_operation_failed(cPtr, { (chatRoom) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onOperationFailed(chatRoom: sObject)
			}
		})

		linphone_chat_room_cbs_set_state_changed(cPtr, { (chatRoom, newState) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onStateChanged(chatRoom: sObject, newState: ChatRoom.State(rawValue: Int(newState.rawValue))!)
			}
		})

		linphone_chat_room_cbs_set_security_event(cPtr, { (chatRoom, eventLog) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onSecurityEvent(chatRoom: sObject, eventLog: EventLog.getSwiftObject(cObject: eventLog!))
			}
		})

		linphone_chat_room_cbs_set_subject_changed(cPtr, { (chatRoom, eventLog) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onSubjectChanged(chatRoom: sObject, eventLog: EventLog.getSwiftObject(cObject: eventLog!))
			}
		})

		linphone_chat_room_cbs_set_message_early_failure(cPtr, { (chatRoom, eventLog) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onMessageEarlyFailure(chatRoom: sObject, eventLog: EventLog.getSwiftObject(cObject: eventLog!))
			}
		})

		linphone_chat_room_cbs_set_undecryptable_message_received(cPtr, { (chatRoom, message) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onUndecryptableMessageReceived(chatRoom: sObject, message: ChatMessage.getSwiftObject(cObject: message!))
			}
		})

		linphone_chat_room_cbs_set_participant_device_added(cPtr, { (chatRoom, eventLog) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onParticipantDeviceAdded(chatRoom: sObject, eventLog: EventLog.getSwiftObject(cObject: eventLog!))
			}
		})

		linphone_chat_room_cbs_set_participant_device_removed(cPtr, { (chatRoom, eventLog) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onParticipantDeviceRemoved(chatRoom: sObject, eventLog: EventLog.getSwiftObject(cObject: eventLog!))
			}
		})

		linphone_chat_room_cbs_set_participant_device_state_changed(cPtr, { (chatRoom, eventLog, state) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onParticipantDeviceStateChanged(chatRoom: sObject, eventLog: EventLog.getSwiftObject(cObject: eventLog!), state: ParticipantDevice.State(rawValue: Int(state.rawValue))!)
			}
		})

		linphone_chat_room_cbs_set_participant_device_media_availability_changed(cPtr, { (chatRoom, eventLog) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onParticipantDeviceMediaAvailabilityChanged(chatRoom: sObject, eventLog: EventLog.getSwiftObject(cObject: eventLog!))
			}
		})

		linphone_chat_room_cbs_set_conference_joined(cPtr, { (chatRoom, eventLog) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onConferenceJoined(chatRoom: sObject, eventLog: EventLog.getSwiftObject(cObject: eventLog!))
			}
		})

		linphone_chat_room_cbs_set_conference_left(cPtr, { (chatRoom, eventLog) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onConferenceLeft(chatRoom: sObject, eventLog: EventLog.getSwiftObject(cObject: eventLog!))
			}
		})

		linphone_chat_room_cbs_set_ephemeral_event(cPtr, { (chatRoom, eventLog) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onEphemeralEvent(chatRoom: sObject, eventLog: EventLog.getSwiftObject(cObject: eventLog!))
			}
		})

		linphone_chat_room_cbs_set_ephemeral_message_timer_started(cPtr, { (chatRoom, eventLog) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onEphemeralMessageTimerStarted(chatRoom: sObject, eventLog: EventLog.getSwiftObject(cObject: eventLog!))
			}
		})

		linphone_chat_room_cbs_set_ephemeral_message_deleted(cPtr, { (chatRoom, eventLog) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onEphemeralMessageDeleted(chatRoom: sObject, eventLog: EventLog.getSwiftObject(cObject: eventLog!))
			}
		})

		linphone_chat_room_cbs_set_conference_address_generation(cPtr, { (chatRoom) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onConferenceAddressGeneration(chatRoom: sObject)
			}
		})

		linphone_chat_room_cbs_set_participant_registration_subscription_requested(cPtr, { (chatRoom, participantAddress) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onParticipantRegistrationSubscriptionRequested(chatRoom: sObject, participantAddress: Address.getSwiftObject(cObject: participantAddress!))
			}
		})

		linphone_chat_room_cbs_set_participant_registration_unsubscription_requested(cPtr, { (chatRoom, participantAddress) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onParticipantRegistrationUnsubscriptionRequested(chatRoom: sObject, participantAddress: Address.getSwiftObject(cObject: participantAddress!))
			}
		})

		linphone_chat_room_cbs_set_chat_message_should_be_stored(cPtr, { (chatRoom, message) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onChatMessageShouldBeStored(chatRoom: sObject, message: ChatMessage.getSwiftObject(cObject: message!))
			}
		})

		linphone_chat_room_cbs_set_chat_message_participant_imdn_state_changed(cPtr, { (chatRoom, message, state) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onChatMessageParticipantImdnStateChanged(chatRoom: sObject, message: ChatMessage.getSwiftObject(cObject: message!), state: ParticipantImdnState.getSwiftObject(cObject: state!))
			}
		})

		linphone_chat_room_cbs_set_chat_room_read(cPtr, { (chatRoom) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onChatRoomRead(chatRoom: sObject)
			}
		})

		linphone_chat_room_cbs_set_new_message_reaction(cPtr, { (chatRoom, message, reaction) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onNewMessageReaction(chatRoom: sObject, message: ChatMessage.getSwiftObject(cObject: message!), reaction: ChatMessageReaction.getSwiftObject(cObject: reaction!))
			}
		})

		linphone_chat_room_cbs_set_message_content_edited(cPtr, { (chatRoom, message) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onMessageContentEdited(chatRoom: sObject, message: ChatMessage.getSwiftObject(cObject: message!))
			}
		})

		linphone_chat_room_cbs_set_message_retracted(cPtr, { (chatRoom, message) -> Void in
			if (chatRoom != nil) {
				let sObject = ChatRoom.getSwiftObject(cObject: chatRoom!)
				let delegate = sObject.currentDelegate
				delegate?.onMessageRetracted(chatRoom: sObject, message: ChatMessage.getSwiftObject(cObject: message!))
			}
		})
	}


}

public protocol ConferenceDelegate : AnyObject {
	
	
	/// Get the allowed participant list changed callback. 
	/// - Returns: The current allowed participant list changed callback. 
	func onAllowedParticipantListChanged(conference: Conference)
	
	/// Get the participant added callback. 
	/// - Returns: The current participant added callback. 
	func onParticipantAdded(conference: Conference, participant: Participant)
	
	/// Get the participant removed callback. 
	/// - Returns: The current participant removed callback. 
	func onParticipantRemoved(conference: Conference, participant: Participant)
	
	/// Get the participant device added callback. 
	/// - Returns: The current participant device added callback. 
	func onParticipantDeviceAdded(conference: Conference, participantDevice: ParticipantDevice)
	
	/// Get the participant device removed callback. 
	/// - Returns: The current participant device removed callback. 
	func onParticipantDeviceRemoved(conference: Conference, participantDevice: ParticipantDevice)
	
	/// Get the participant device joining request callback. 
	/// - Returns: The current participant device joining request callback. 
	func onParticipantDeviceJoiningRequest(conference: Conference, participantDevice: ParticipantDevice)
	
	/// Get the participant role changed callback. 
	/// - Returns: The current participant role changed callback. 
	func onParticipantRoleChanged(conference: Conference, participant: Participant)
	
	/// Get the participant admin status changed callback. 
	/// - Returns: The current participant admin status changed callback. 
	func onParticipantAdminStatusChanged(conference: Conference, participant: Participant)
	
	/// Get the participant device state changed callback. 
	/// - Returns: The current participant device state changed callback. 
	func onParticipantDeviceStateChanged(conference: Conference, device: ParticipantDevice, state: ParticipantDevice.State)
	
	/// Get the participant device is screen sharing changed callback. 
	/// - Returns: The current participant device is screen sharing changed callback. 
	func onParticipantDeviceScreenSharingChanged(conference: Conference, device: ParticipantDevice, enabled: Bool)
	
	/// Get the participant device media availability changed callback. 
	/// - Returns: The current participant device media availability changed callback. 
	func onParticipantDeviceMediaAvailabilityChanged(conference: Conference, device: ParticipantDevice)
	
	/// Get the participant device media capabilities changed callback. 
	/// - Returns: The current participant device media capabilities changed callback. 
	func onParticipantDeviceMediaCapabilityChanged(conference: Conference, device: ParticipantDevice)
	
	/// Get the leave failed callback. 
	/// - Returns: The current leave failed callback. 
	func onOperationFailed(conference: Conference)
	
	/// Get the state changed callback. 
	/// - Returns: The current state changed callback. 
	func onStateChanged(conference: Conference, newState: Conference.State)
	
	/// Get the available media changed callback. 
	/// - Returns: The current available media changed callback. 
	func onAvailableMediaChanged(conference: Conference)
	
	/// Get the subject changed callback. 
	/// - Returns: The current subject changed callback. 
	func onSubjectChanged(conference: Conference, subject: String)
	
	/// Get the participant device is speaking changed callback. 
	/// - Returns: The current participant device is speaking changed callback. 
	func onParticipantDeviceIsSpeakingChanged(conference: Conference, participantDevice: ParticipantDevice, isSpeaking: Bool)
	
	/// Get the participant device is muted callback. 
	/// - Returns: The current participant device is muted callback. 
	func onParticipantDeviceIsMuted(conference: Conference, participantDevice: ParticipantDevice, isMuted: Bool)
	
	/// Get the audio device changed callback. 
	/// - Returns: The current audio device changed callback. 
	func onAudioDeviceChanged(conference: Conference, audioDevice: AudioDevice)
	
	/// Get the actively speaking participant device callback. 
	/// - Returns: The current active speaker participant device callback. 
	func onActiveSpeakerParticipantDevice(conference: Conference, participantDevice: ParticipantDevice?)
	
	/// Get the full state received callback. 
	/// - Returns: The current full state received callback. 
	func onFullStateReceived(conference: Conference)
}

public extension ConferenceDelegate {
	
	func onAllowedParticipantListChanged(conference: Conference) {}
	
	func onParticipantAdded(conference: Conference, participant: Participant) {}
	
	func onParticipantRemoved(conference: Conference, participant: Participant) {}
	
	func onParticipantDeviceAdded(conference: Conference, participantDevice: ParticipantDevice) {}
	
	func onParticipantDeviceRemoved(conference: Conference, participantDevice: ParticipantDevice) {}
	
	func onParticipantDeviceJoiningRequest(conference: Conference, participantDevice: ParticipantDevice) {}
	
	func onParticipantRoleChanged(conference: Conference, participant: Participant) {}
	
	func onParticipantAdminStatusChanged(conference: Conference, participant: Participant) {}
	
	func onParticipantDeviceStateChanged(conference: Conference, device: ParticipantDevice, state: ParticipantDevice.State) {}
	
	func onParticipantDeviceScreenSharingChanged(conference: Conference, device: ParticipantDevice, enabled: Bool) {}
	
	func onParticipantDeviceMediaAvailabilityChanged(conference: Conference, device: ParticipantDevice) {}
	
	func onParticipantDeviceMediaCapabilityChanged(conference: Conference, device: ParticipantDevice) {}
	
	func onOperationFailed(conference: Conference) {}
	
	func onStateChanged(conference: Conference, newState: Conference.State) {}
	
	func onAvailableMediaChanged(conference: Conference) {}
	
	func onSubjectChanged(conference: Conference, subject: String) {}
	
	func onParticipantDeviceIsSpeakingChanged(conference: Conference, participantDevice: ParticipantDevice, isSpeaking: Bool) {}
	
	func onParticipantDeviceIsMuted(conference: Conference, participantDevice: ParticipantDevice, isMuted: Bool) {}
	
	func onAudioDeviceChanged(conference: Conference, audioDevice: AudioDevice) {}
	
	func onActiveSpeakerParticipantDevice(conference: Conference, participantDevice: ParticipantDevice?) {}
	
	func onFullStateReceived(conference: Conference) {}
}

public final class ConferenceDelegateStub : ConferenceDelegate
{
	var _onAllowedParticipantListChanged: ((Conference) -> Void)?
	var _onParticipantAdded: ((Conference, Participant) -> Void)?
	var _onParticipantRemoved: ((Conference, Participant) -> Void)?
	var _onParticipantDeviceAdded: ((Conference, ParticipantDevice) -> Void)?
	var _onParticipantDeviceRemoved: ((Conference, ParticipantDevice) -> Void)?
	var _onParticipantDeviceJoiningRequest: ((Conference, ParticipantDevice) -> Void)?
	var _onParticipantRoleChanged: ((Conference, Participant) -> Void)?
	var _onParticipantAdminStatusChanged: ((Conference, Participant) -> Void)?
	var _onParticipantDeviceStateChanged: ((Conference, ParticipantDevice, ParticipantDevice.State) -> Void)?
	var _onParticipantDeviceScreenSharingChanged: ((Conference, ParticipantDevice, Bool) -> Void)?
	var _onParticipantDeviceMediaAvailabilityChanged: ((Conference, ParticipantDevice) -> Void)?
	var _onParticipantDeviceMediaCapabilityChanged: ((Conference, ParticipantDevice) -> Void)?
	var _onOperationFailed: ((Conference) -> Void)?
	var _onStateChanged: ((Conference, Conference.State) -> Void)?
	var _onAvailableMediaChanged: ((Conference) -> Void)?
	var _onSubjectChanged: ((Conference, String) -> Void)?
	var _onParticipantDeviceIsSpeakingChanged: ((Conference, ParticipantDevice, Bool) -> Void)?
	var _onParticipantDeviceIsMuted: ((Conference, ParticipantDevice, Bool) -> Void)?
	var _onAudioDeviceChanged: ((Conference, AudioDevice) -> Void)?
	var _onActiveSpeakerParticipantDevice: ((Conference, ParticipantDevice?) -> Void)?
	var _onFullStateReceived: ((Conference) -> Void)?

	
	public func onAllowedParticipantListChanged(conference: Conference){_onAllowedParticipantListChanged.map{$0(conference)}}
	
	public func onParticipantAdded(conference: Conference, participant: Participant){_onParticipantAdded.map{$0(conference, participant)}}
	
	public func onParticipantRemoved(conference: Conference, participant: Participant){_onParticipantRemoved.map{$0(conference, participant)}}
	
	public func onParticipantDeviceAdded(conference: Conference, participantDevice: ParticipantDevice){_onParticipantDeviceAdded.map{$0(conference, participantDevice)}}
	
	public func onParticipantDeviceRemoved(conference: Conference, participantDevice: ParticipantDevice){_onParticipantDeviceRemoved.map{$0(conference, participantDevice)}}
	
	public func onParticipantDeviceJoiningRequest(conference: Conference, participantDevice: ParticipantDevice){_onParticipantDeviceJoiningRequest.map{$0(conference, participantDevice)}}
	
	public func onParticipantRoleChanged(conference: Conference, participant: Participant){_onParticipantRoleChanged.map{$0(conference, participant)}}
	
	public func onParticipantAdminStatusChanged(conference: Conference, participant: Participant){_onParticipantAdminStatusChanged.map{$0(conference, participant)}}
	
	public func onParticipantDeviceStateChanged(conference: Conference, device: ParticipantDevice, state: ParticipantDevice.State){_onParticipantDeviceStateChanged.map{$0(conference, device, state)}}
	
	public func onParticipantDeviceScreenSharingChanged(conference: Conference, device: ParticipantDevice, enabled: Bool){_onParticipantDeviceScreenSharingChanged.map{$0(conference, device, enabled)}}
	
	public func onParticipantDeviceMediaAvailabilityChanged(conference: Conference, device: ParticipantDevice){_onParticipantDeviceMediaAvailabilityChanged.map{$0(conference, device)}}
	
	public func onParticipantDeviceMediaCapabilityChanged(conference: Conference, device: ParticipantDevice){_onParticipantDeviceMediaCapabilityChanged.map{$0(conference, device)}}
	
	public func onOperationFailed(conference: Conference){_onOperationFailed.map{$0(conference)}}
	
	public func onStateChanged(conference: Conference, newState: Conference.State){_onStateChanged.map{$0(conference, newState)}}
	
	public func onAvailableMediaChanged(conference: Conference){_onAvailableMediaChanged.map{$0(conference)}}
	
	public func onSubjectChanged(conference: Conference, subject: String){_onSubjectChanged.map{$0(conference, subject)}}
	
	public func onParticipantDeviceIsSpeakingChanged(conference: Conference, participantDevice: ParticipantDevice, isSpeaking: Bool){_onParticipantDeviceIsSpeakingChanged.map{$0(conference, participantDevice, isSpeaking)}}
	
	public func onParticipantDeviceIsMuted(conference: Conference, participantDevice: ParticipantDevice, isMuted: Bool){_onParticipantDeviceIsMuted.map{$0(conference, participantDevice, isMuted)}}
	
	public func onAudioDeviceChanged(conference: Conference, audioDevice: AudioDevice){_onAudioDeviceChanged.map{$0(conference, audioDevice)}}
	
	public func onActiveSpeakerParticipantDevice(conference: Conference, participantDevice: ParticipantDevice?){_onActiveSpeakerParticipantDevice.map{$0(conference, participantDevice)}}
	
	public func onFullStateReceived(conference: Conference){_onFullStateReceived.map{$0(conference)}}

	public init (
		onAllowedParticipantListChanged: ((Conference) -> Void)? = nil,
		onParticipantAdded: ((Conference, Participant) -> Void)? = nil,
		onParticipantRemoved: ((Conference, Participant) -> Void)? = nil,
		onParticipantDeviceAdded: ((Conference, ParticipantDevice) -> Void)? = nil,
		onParticipantDeviceRemoved: ((Conference, ParticipantDevice) -> Void)? = nil,
		onParticipantDeviceJoiningRequest: ((Conference, ParticipantDevice) -> Void)? = nil,
		onParticipantRoleChanged: ((Conference, Participant) -> Void)? = nil,
		onParticipantAdminStatusChanged: ((Conference, Participant) -> Void)? = nil,
		onParticipantDeviceStateChanged: ((Conference, ParticipantDevice, ParticipantDevice.State) -> Void)? = nil,
		onParticipantDeviceScreenSharingChanged: ((Conference, ParticipantDevice, Bool) -> Void)? = nil,
		onParticipantDeviceMediaAvailabilityChanged: ((Conference, ParticipantDevice) -> Void)? = nil,
		onParticipantDeviceMediaCapabilityChanged: ((Conference, ParticipantDevice) -> Void)? = nil,
		onOperationFailed: ((Conference) -> Void)? = nil,
		onStateChanged: ((Conference, Conference.State) -> Void)? = nil,
		onAvailableMediaChanged: ((Conference) -> Void)? = nil,
		onSubjectChanged: ((Conference, String) -> Void)? = nil,
		onParticipantDeviceIsSpeakingChanged: ((Conference, ParticipantDevice, Bool) -> Void)? = nil,
		onParticipantDeviceIsMuted: ((Conference, ParticipantDevice, Bool) -> Void)? = nil,
		onAudioDeviceChanged: ((Conference, AudioDevice) -> Void)? = nil,
		onActiveSpeakerParticipantDevice: ((Conference, ParticipantDevice?) -> Void)? = nil,
		onFullStateReceived: ((Conference) -> Void)? = nil
	) {
		self._onAllowedParticipantListChanged = onAllowedParticipantListChanged
		self._onParticipantAdded = onParticipantAdded
		self._onParticipantRemoved = onParticipantRemoved
		self._onParticipantDeviceAdded = onParticipantDeviceAdded
		self._onParticipantDeviceRemoved = onParticipantDeviceRemoved
		self._onParticipantDeviceJoiningRequest = onParticipantDeviceJoiningRequest
		self._onParticipantRoleChanged = onParticipantRoleChanged
		self._onParticipantAdminStatusChanged = onParticipantAdminStatusChanged
		self._onParticipantDeviceStateChanged = onParticipantDeviceStateChanged
		self._onParticipantDeviceScreenSharingChanged = onParticipantDeviceScreenSharingChanged
		self._onParticipantDeviceMediaAvailabilityChanged = onParticipantDeviceMediaAvailabilityChanged
		self._onParticipantDeviceMediaCapabilityChanged = onParticipantDeviceMediaCapabilityChanged
		self._onOperationFailed = onOperationFailed
		self._onStateChanged = onStateChanged
		self._onAvailableMediaChanged = onAvailableMediaChanged
		self._onSubjectChanged = onSubjectChanged
		self._onParticipantDeviceIsSpeakingChanged = onParticipantDeviceIsSpeakingChanged
		self._onParticipantDeviceIsMuted = onParticipantDeviceIsMuted
		self._onAudioDeviceChanged = onAudioDeviceChanged
		self._onActiveSpeakerParticipantDevice = onActiveSpeakerParticipantDevice
		self._onFullStateReceived = onFullStateReceived
	}
}

class ConferenceDelegateManager
{
	var cPtr: OpaquePointer
	public var delegate: ConferenceDelegate?

	public init() {
		cPtr = linphone_factory_create_conference_cbs(linphone_factory_get())
		belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passRetained(self).toOpaque()), {
			data in
			if (data != nil) {
				Unmanaged<ConferenceDelegateManager>.fromOpaque(data!).release()
			}
		})

		linphone_conference_cbs_set_allowed_participant_list_changed(cPtr, { (conference) -> Void in
			if (conference != nil) {
				let sObject = Conference.getSwiftObject(cObject: conference!)
				let delegate = sObject.currentDelegate
				delegate?.onAllowedParticipantListChanged(conference: sObject)
			}
		})

		linphone_conference_cbs_set_participant_added(cPtr, { (conference, participant) -> Void in
			if (conference != nil) {
				let sObject = Conference.getSwiftObject(cObject: conference!)
				let delegate = sObject.currentDelegate
				delegate?.onParticipantAdded(conference: sObject, participant: Participant.getSwiftObject(cObject: participant!))
			}
		})

		linphone_conference_cbs_set_participant_removed(cPtr, { (conference, participant) -> Void in
			if (conference != nil) {
				let sObject = Conference.getSwiftObject(cObject: conference!)
				let delegate = sObject.currentDelegate
				delegate?.onParticipantRemoved(conference: sObject, participant: Participant.getSwiftObject(cObject: participant!))
			}
		})

		linphone_conference_cbs_set_participant_device_added(cPtr, { (conference, participantDevice) -> Void in
			if (conference != nil) {
				let sObject = Conference.getSwiftObject(cObject: conference!)
				let delegate = sObject.currentDelegate
				delegate?.onParticipantDeviceAdded(conference: sObject, participantDevice: ParticipantDevice.getSwiftObject(cObject: participantDevice!))
			}
		})

		linphone_conference_cbs_set_participant_device_removed(cPtr, { (conference, participantDevice) -> Void in
			if (conference != nil) {
				let sObject = Conference.getSwiftObject(cObject: conference!)
				let delegate = sObject.currentDelegate
				delegate?.onParticipantDeviceRemoved(conference: sObject, participantDevice: ParticipantDevice.getSwiftObject(cObject: participantDevice!))
			}
		})

		linphone_conference_cbs_set_participant_device_joining_request(cPtr, { (conference, participantDevice) -> Void in
			if (conference != nil) {
				let sObject = Conference.getSwiftObject(cObject: conference!)
				let delegate = sObject.currentDelegate
				delegate?.onParticipantDeviceJoiningRequest(conference: sObject, participantDevice: ParticipantDevice.getSwiftObject(cObject: participantDevice!))
			}
		})

		linphone_conference_cbs_set_participant_role_changed(cPtr, { (conference, participant) -> Void in
			if (conference != nil) {
				let sObject = Conference.getSwiftObject(cObject: conference!)
				let delegate = sObject.currentDelegate
				delegate?.onParticipantRoleChanged(conference: sObject, participant: Participant.getSwiftObject(cObject: participant!))
			}
		})

		linphone_conference_cbs_set_participant_admin_status_changed(cPtr, { (conference, participant) -> Void in
			if (conference != nil) {
				let sObject = Conference.getSwiftObject(cObject: conference!)
				let delegate = sObject.currentDelegate
				delegate?.onParticipantAdminStatusChanged(conference: sObject, participant: Participant.getSwiftObject(cObject: participant!))
			}
		})

		linphone_conference_cbs_set_participant_device_state_changed(cPtr, { (conference, device, state) -> Void in
			if (conference != nil) {
				let sObject = Conference.getSwiftObject(cObject: conference!)
				let delegate = sObject.currentDelegate
				delegate?.onParticipantDeviceStateChanged(conference: sObject, device: ParticipantDevice.getSwiftObject(cObject: device!), state: ParticipantDevice.State(rawValue: Int(state.rawValue))!)
			}
		})

		linphone_conference_cbs_set_participant_device_screen_sharing_changed(cPtr, { (conference, device, enabled) -> Void in
			if (conference != nil) {
				let sObject = Conference.getSwiftObject(cObject: conference!)
				let delegate = sObject.currentDelegate
				delegate?.onParticipantDeviceScreenSharingChanged(conference: sObject, device: ParticipantDevice.getSwiftObject(cObject: device!), enabled: enabled != 0)
			}
		})

		linphone_conference_cbs_set_participant_device_media_availability_changed(cPtr, { (conference, device) -> Void in
			if (conference != nil) {
				let sObject = Conference.getSwiftObject(cObject: conference!)
				let delegate = sObject.currentDelegate
				delegate?.onParticipantDeviceMediaAvailabilityChanged(conference: sObject, device: ParticipantDevice.getSwiftObject(cObject: device!))
			}
		})

		linphone_conference_cbs_set_participant_device_media_capability_changed(cPtr, { (conference, device) -> Void in
			if (conference != nil) {
				let sObject = Conference.getSwiftObject(cObject: conference!)
				let delegate = sObject.currentDelegate
				delegate?.onParticipantDeviceMediaCapabilityChanged(conference: sObject, device: ParticipantDevice.getSwiftObject(cObject: device!))
			}
		})

		linphone_conference_cbs_set_operation_failed(cPtr, { (conference) -> Void in
			if (conference != nil) {
				let sObject = Conference.getSwiftObject(cObject: conference!)
				let delegate = sObject.currentDelegate
				delegate?.onOperationFailed(conference: sObject)
			}
		})

		linphone_conference_cbs_set_state_changed(cPtr, { (conference, newState) -> Void in
			if (conference != nil) {
				let sObject = Conference.getSwiftObject(cObject: conference!)
				let delegate = sObject.currentDelegate
				delegate?.onStateChanged(conference: sObject, newState: Conference.State(rawValue: Int(newState.rawValue))!)
			}
		})

		linphone_conference_cbs_set_available_media_changed(cPtr, { (conference) -> Void in
			if (conference != nil) {
				let sObject = Conference.getSwiftObject(cObject: conference!)
				let delegate = sObject.currentDelegate
				delegate?.onAvailableMediaChanged(conference: sObject)
			}
		})

		linphone_conference_cbs_set_subject_changed(cPtr, { (conference, subject) -> Void in
			if (conference != nil) {
				let sObject = Conference.getSwiftObject(cObject: conference!)
				let delegate = sObject.currentDelegate
				delegate?.onSubjectChanged(conference: sObject, subject: charArrayToString(charPointer: subject))
			}
		})

		linphone_conference_cbs_set_participant_device_is_speaking_changed(cPtr, { (conference, participantDevice, isSpeaking) -> Void in
			if (conference != nil) {
				let sObject = Conference.getSwiftObject(cObject: conference!)
				let delegate = sObject.currentDelegate
				delegate?.onParticipantDeviceIsSpeakingChanged(conference: sObject, participantDevice: ParticipantDevice.getSwiftObject(cObject: participantDevice!), isSpeaking: isSpeaking != 0)
			}
		})

		linphone_conference_cbs_set_participant_device_is_muted(cPtr, { (conference, participantDevice, isMuted) -> Void in
			if (conference != nil) {
				let sObject = Conference.getSwiftObject(cObject: conference!)
				let delegate = sObject.currentDelegate
				delegate?.onParticipantDeviceIsMuted(conference: sObject, participantDevice: ParticipantDevice.getSwiftObject(cObject: participantDevice!), isMuted: isMuted != 0)
			}
		})

		linphone_conference_cbs_set_audio_device_changed(cPtr, { (conference, audioDevice) -> Void in
			if (conference != nil) {
				let sObject = Conference.getSwiftObject(cObject: conference!)
				let delegate = sObject.currentDelegate
				delegate?.onAudioDeviceChanged(conference: sObject, audioDevice: AudioDevice.getSwiftObject(cObject: audioDevice!))
			}
		})

		linphone_conference_cbs_set_active_speaker_participant_device(cPtr, { (conference, participantDevice) -> Void in
			if (conference != nil) {
				let sObject = Conference.getSwiftObject(cObject: conference!)
				let delegate = sObject.currentDelegate
				delegate?.onActiveSpeakerParticipantDevice(conference: sObject, participantDevice: (participantDevice != nil) ? ParticipantDevice.getSwiftObject(cObject: participantDevice!) : nil)
			}
		})

		linphone_conference_cbs_set_full_state_received(cPtr, { (conference) -> Void in
			if (conference != nil) {
				let sObject = Conference.getSwiftObject(cObject: conference!)
				let delegate = sObject.currentDelegate
				delegate?.onFullStateReceived(conference: sObject)
			}
		})
	}


}

public protocol ConferenceSchedulerDelegate : AnyObject {
	
	
	/// Get the state changed callback. 
	/// - Returns: The current state changed callback. 
	func onStateChanged(conferenceScheduler: ConferenceScheduler, state: ConferenceScheduler.State)
	
	/// Get the invitations sent callback. 
	/// - Returns: The current invitations sent callback. 
	func onInvitationsSent(conferenceScheduler: ConferenceScheduler, failedInvitations: [Address])
}

public extension ConferenceSchedulerDelegate {
	
	func onStateChanged(conferenceScheduler: ConferenceScheduler, state: ConferenceScheduler.State) {}
	
	func onInvitationsSent(conferenceScheduler: ConferenceScheduler, failedInvitations: [Address]) {}
}

public final class ConferenceSchedulerDelegateStub : ConferenceSchedulerDelegate
{
	var _onStateChanged: ((ConferenceScheduler, ConferenceScheduler.State) -> Void)?
	var _onInvitationsSent: ((ConferenceScheduler, [Address]) -> Void)?

	
	public func onStateChanged(conferenceScheduler: ConferenceScheduler, state: ConferenceScheduler.State){_onStateChanged.map{$0(conferenceScheduler, state)}}
	
	public func onInvitationsSent(conferenceScheduler: ConferenceScheduler, failedInvitations: [Address]){_onInvitationsSent.map{$0(conferenceScheduler, failedInvitations)}}

	public init (
		onStateChanged: ((ConferenceScheduler, ConferenceScheduler.State) -> Void)? = nil,
		onInvitationsSent: ((ConferenceScheduler, [Address]) -> Void)? = nil
	) {
		self._onStateChanged = onStateChanged
		self._onInvitationsSent = onInvitationsSent
	}
}

class ConferenceSchedulerDelegateManager
{
	var cPtr: OpaquePointer
	public var delegate: ConferenceSchedulerDelegate?

	public init() {
		cPtr = linphone_factory_create_conference_scheduler_cbs(linphone_factory_get())
		belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passRetained(self).toOpaque()), {
			data in
			if (data != nil) {
				Unmanaged<ConferenceSchedulerDelegateManager>.fromOpaque(data!).release()
			}
		})

		linphone_conference_scheduler_cbs_set_state_changed(cPtr, { (conferenceScheduler, state) -> Void in
			if (conferenceScheduler != nil) {
				let sObject = ConferenceScheduler.getSwiftObject(cObject: conferenceScheduler!)
				let delegate = sObject.currentDelegate
				delegate?.onStateChanged(conferenceScheduler: sObject, state: ConferenceScheduler.State(rawValue: Int(state.rawValue))!)
			}
		})

		linphone_conference_scheduler_cbs_set_invitations_sent(cPtr, { (conferenceScheduler, failedInvitations) -> Void in
			if (conferenceScheduler != nil) {
				let sObject = ConferenceScheduler.getSwiftObject(cObject: conferenceScheduler!)
				let delegate = sObject.currentDelegate
				var failedInvitationssList = [Address]()
				let failedInvitationscList = failedInvitations
				var listTemp = failedInvitationscList
				while (listTemp != nil) {
					let data = unsafeBitCast(listTemp!.pointee.data, to: OpaquePointer.self)
					failedInvitationssList.append(Address.getSwiftObject(cObject: data))
					listTemp = UnsafePointer<bctbx_list_t>(listTemp!.pointee.next)
				}
				delegate?.onInvitationsSent(conferenceScheduler: sObject, failedInvitations: failedInvitationssList)
			}
		})
	}


}

public protocol CoreDelegate : AnyObject {
	
	
	/// Gets the on alert callback. 
	/// - Returns: The LinphoneCoreCbsNewAlertTriggeredCb callback called. 
	func onNewAlertTriggered(core: Core, alert: Alert)
	
	/// Get the global state changed callback. 
	/// - Returns: The callback. 
	func onGlobalStateChanged(core: Core, state: GlobalState, message: String)
	
	/// Get the LinphoneCoreCbsRegistrationStateChangedCb callback. 
	/// - Returns: The callback. 
	/// - Deprecated: 30/09/2020. see
	/// linphone_account_cbs_get_registration_state_changed 
	func onRegistrationStateChanged(core: Core, proxyConfig: ProxyConfig, state: RegistrationState, message: String)
	
	/// Get the conference info received callback. 
	/// - Returns: The current conference info received callback. 
	func onConferenceInfoReceived(core: Core, conferenceInfo: ConferenceInfo)
	
	/// Get the push notification received callback. 
	/// - Returns: The current push notification received callback. 
	func onPushNotificationReceived(core: Core, payload: String)
	
	/// Get the preview display error callback for preview. 
	/// - Returns: The callback to use 
	func onPreviewDisplayErrorOccurred(core: Core, errorCode: Int)
	
	/// Get the LinphoneCoreCbsCallStateChangedCb callback. 
	/// - Returns: The callback. 
	func onCallStateChanged(core: Core, call: Call, state: Call.State, message: String)
	
	/// Get the LinphoneCoreCbsNotifyPresenceReceivedCb callback. 
	/// - Returns: The callback. 
	func onNotifyPresenceReceived(core: Core, linphoneFriend: Friend)
	
	/// Get the LinphoneCoreCbsNotifyPresenceReceivedForUriOrTelCb callback. 
	/// - Returns: The callback. 
	func onNotifyPresenceReceivedForUriOrTel(core: Core, linphoneFriend: Friend, uriOrTel: String, presenceModel: PresenceModel)
	
	/// Get the LinphoneCoreCbsNewSubscriptionRequestedCb callback. 
	/// - Returns: The callback. 
	func onNewSubscriptionRequested(core: Core, linphoneFriend: Friend, url: String)
	
	/// Get the LinphoneCoreCbsAuthenticationRequestedCb callback. 
	/// - Returns: The callback. 
	func onAuthenticationRequested(core: Core, authInfo: AuthInfo, method: AuthMethod)
	
	/// Get the LinphoneCoreCbsCallLogUpdatedCb callback. 
	/// - Returns: The callback. 
	func onCallLogUpdated(core: Core, callLog: CallLog)
	
	/// Get the LinphoneCoreCbsCallIdUpdatedCb callback. 
	/// - Returns: The callback. 
	func onCallIdUpdated(core: Core, previousCallId: String, currentCallId: String)
	
	/// Get the LinphoneCoreCbsMessageReceivedCb callback. 
	/// - Returns: The callback. 
	func onMessageReceived(core: Core, chatRoom: ChatRoom, message: ChatMessage)
	
	/// Get the LinphoneCoreCbsNewMessageReactionCb callback. 
	/// - Returns: The callback. 
	func onNewMessageReaction(core: Core, chatRoom: ChatRoom, message: ChatMessage, reaction: ChatMessageReaction)
	
	/// Get the LinphoneCoreCbsReactionRemovedCb callback. 
	/// - Returns: The callback. 
	func onReactionRemoved(core: Core, chatRoom: ChatRoom, message: ChatMessage, address: Address)
	
	/// Get the LinphoneCoreCbsMessagesReceivedCb callback. 
	/// - Returns: The callback. 
	func onMessagesReceived(core: Core, chatRoom: ChatRoom, messages: [ChatMessage])
	
	/// Get the LinphoneCoreCbsMessageSentCb callback. 
	/// - Returns: The callback. 
	func onMessageSent(core: Core, chatRoom: ChatRoom, message: ChatMessage)
	
	/// Get the LinphoneCoreCbsChatRoomSessionStateChangedCb callback. 
	/// - Returns: The callback. 
	func onChatRoomSessionStateChanged(core: Core, chatRoom: ChatRoom, state: Call.State, message: String)
	
	/// Get the LinphoneCoreCbsChatRoomReadCb callback. 
	/// - Returns: The callback. 
	func onChatRoomRead(core: Core, chatRoom: ChatRoom)
	
	/// Get the LinphoneCoreCbsMessageReceivedUnableDecryptCb callback. 
	/// - Returns: The callback. 
	func onMessageReceivedUnableDecrypt(core: Core, chatRoom: ChatRoom, message: ChatMessage)
	
	/// Get the LinphoneCoreCbsIsComposingReceivedCb callback. 
	/// - Returns: The callback. 
	func onIsComposingReceived(core: Core, chatRoom: ChatRoom)
	
	/// Get the LinphoneCoreCbsDtmfReceivedCb callback. 
	/// - Returns: The callback. 
	func onDtmfReceived(core: Core, call: Call, dtmf: Int)
	
	/// Get the LinphoneCoreCbsReferReceivedCb callback. 
	/// - Returns: The callback. 
	func onReferReceived(core: Core, referToAddr: Address, customHeaders: Headers, content: Content?)
	
	/// Gets the LinphoneCoreCbsCallGoClearAckSentCb callback. 
	/// - Returns: The callback. 
	func onCallGoclearAckSent(core: Core, call: Call)
	
	/// Gets the LinphoneCoreCbsCallEncryptionChangedCb callback. 
	/// - Returns: The callback. 
	func onCallEncryptionChanged(core: Core, call: Call, mediaEncryptionEnabled: Bool, authenticationToken: String)
	
	/// Gets the LinphoneCoreCbsCallSendMasterKeyChangedCb callback. 
	/// - Returns: The callback. 
	func onCallSendMasterKeyChanged(core: Core, call: Call, masterKey: String)
	
	/// Gets the LinphoneCoreCbsCallReceiveMasterKeyChangedCb callback. 
	/// - Returns: The callback. 
	func onCallReceiveMasterKeyChanged(core: Core, call: Call, masterKey: String)
	
	/// Gets the LinphoneCoreCbsTransferStateChangedCb callback. 
	/// - Returns: The callback. 
	func onTransferStateChanged(core: Core, transferred: Call, callState: Call.State)
	
	/// Gets the LinphoneCoreCbsBuddyInfoUpdatedCb callback. 
	/// - Returns: The callback. 
	func onBuddyInfoUpdated(core: Core, linphoneFriend: Friend)
	
	/// Gets the LinphoneCoreCbsCallStatsUpdatedCb callback. 
	/// - Returns: The callback. 
	func onCallStatsUpdated(core: Core, call: Call, callStats: CallStats)
	
	/// Gets the LinphoneCoreCbsInfoReceivedCb callback. 
	/// - Returns: The callback. 
	func onInfoReceived(core: Core, call: Call, message: InfoMessage)
	
	/// Gets the LinphoneCoreCbsSubscriptionStateChangedCb callback. 
	/// - Returns: The callback. 
	func onSubscriptionStateChanged(core: Core, linphoneEvent: Event, state: SubscriptionState)
	
	/// Gets the LinphoneCoreCbsNotifySentCb callback. 
	/// - Returns: The callback. 
	func onNotifySent(core: Core, linphoneEvent: Event, body: Content?)
	
	/// Gets the LinphoneCoreCbsNotifyReceivedCb callback. 
	/// - Returns: The callback. 
	func onNotifyReceived(core: Core, linphoneEvent: Event, notifiedEvent: String, body: Content?)
	
	/// Gets the LinphoneCoreCbsSubscribeReceivedCb callback. 
	/// - Returns: The callback. 
	func onSubscribeReceived(core: Core, linphoneEvent: Event, subscribeEvent: String, body: Content?)
	
	/// Gets the LinphoneCoreCbsPublishStateChangedCb callback. 
	/// - Returns: The callback. 
	func onPublishStateChanged(core: Core, linphoneEvent: Event, state: PublishState)
	
	/// Gets the LinphoneCoreCbsPublishReceivedCb callback. 
	/// - Returns: The callback. 
	func onPublishReceived(core: Core, linphoneEvent: Event, publishEvent: String, body: Content?)
	
	/// Gets the LinphoneCoreCbsConfiguringStatusCb callback. 
	/// - Returns: The callback. 
	func onConfiguringStatus(core: Core, status: ConfiguringState, message: String)
	
	/// Gets the LinphoneCoreCbsNetworkReachableCb callback. 
	/// - Returns: The callback. 
	func onNetworkReachable(core: Core, reachable: Bool)
	
	/// Gets the LinphoneCoreCbsLogCollectionUploadStateChangedCb callback. 
	/// - Returns: The callback. 
	func onLogCollectionUploadStateChanged(core: Core, state: Core.LogCollectionUploadState, info: String)
	
	/// Gets the LinphoneCoreCbsLogCollectionUploadProgressIndicationCb callback. 
	/// - Returns: The callback. 
	func onLogCollectionUploadProgressIndication(core: Core, offset: Int, total: Int)
	
	/// Gets the LinphoneCoreCbsFriendListCreatedCb callback. 
	/// - Returns: The callback. 
	func onFriendListCreated(core: Core, friendList: FriendList)
	
	/// Gets the LinphoneCoreCbsFriendListRemovedCb callback. 
	/// - Returns: The callback. 
	func onFriendListRemoved(core: Core, friendList: FriendList)
	
	/// Gets the call created callback. 
	/// - Returns: The current call created callback. 
	func onCallCreated(core: Core, call: Call)
	
	/// Gets the version update check result callback. 
	/// - Returns: The current callback 
	func onVersionUpdateCheckResultReceived(core: Core, result: VersionUpdateCheckResult, version: String, url: String)
	
	/// Gets the conference state changed callback. 
	/// - Returns: The current callback 
	func onConferenceStateChanged(core: Core, conference: Conference, state: Conference.State)
	
	/// Gets the chat room state changed callback. 
	/// - Returns: The current callback 
	func onChatRoomStateChanged(core: Core, chatRoom: ChatRoom, state: ChatRoom.State)
	
	/// Gets the chat room subject changed callback. 
	/// - Returns: The current callback 
	func onChatRoomSubjectChanged(core: Core, chatRoom: ChatRoom)
	
	/// Gets the chat room ephemeral message deleted callback. 
	/// - Returns: The current callback 
	func onChatRoomEphemeralMessageDeleted(core: Core, chatRoom: ChatRoom)
	
	/// Gets the IMEE user registration callback. 
	/// - Returns: The current callback 
	func onImeeUserRegistration(core: Core, status: Bool, userId: String, info: String)
	
	/// Gets the qrcode found callback. 
	/// - Returns: The current callback 
	func onQrcodeFound(core: Core, result: String)
	
	/// Gets the first call started callback. 
	/// - Returns: The current callback 
	func onFirstCallStarted(core: Core)
	
	/// Gets the last call ended callback. 
	/// - Returns: The current callback 
	func onLastCallEnded(core: Core)
	
	/// Gets the audio device changed callback. 
	/// - Returns: The current callback 
	func onAudioDeviceChanged(core: Core, audioDevice: AudioDevice)
	
	/// Gets the audio devices list updated callback. 
	/// - Returns: The current callback 
	func onAudioDevicesListUpdated(core: Core)
	
	/// Sets a callback to call each time the echo-canceler calibration is completed. 
	func onEcCalibrationResult(core: Core, status: EcCalibratorStatus, delayMs: Int)
	
	/// Sets a callback to call when the echo-canceler calibrator has completed its
	/// audio graph. 
	func onEcCalibrationAudioInit(core: Core)
	
	/// Sets a callback to call when the echo-canceler calibrator destroys its audio
	/// graph. 
	func onEcCalibrationAudioUninit(core: Core)
	
	/// Gets the account registration state changed callback. 
	/// - Returns: The current account registration state changed callback. 
	func onAccountRegistrationStateChanged(core: Core, account: Account, state: RegistrationState, message: String)
	
	/// Gets the default account changed callback. 
	/// - Returns: The default account changed callback that will be triggered. 
	func onDefaultAccountChanged(core: Core, account: Account?)
	
	/// Gets the account added callback. 
	/// - Returns: The account added callback that will be triggered. 
	func onAccountAdded(core: Core, account: Account)
	
	/// Gets the new account removed callback. 
	/// - Returns: The new account removed callback that will be triggered. 
	func onAccountRemoved(core: Core, account: Account)
	
	/// Gets the message waiting indication changed callback. 
	/// - Returns: The current message waiting indication changed callback. 
	func onMessageWaitingIndicationChanged(core: Core, lev: Event, mwi: MessageWaitingIndication)
	
	/// Gets the snapshot taken callback. 
	/// - Returns: The current snapshot taken callback. 
	func onSnapshotTaken(core: Core, filePath: String)
	
	/// Gets the message content edited callback. 
	/// - Returns: The LinphoneCoreCbsMessageContentEditedCb callback called. 
	func onMessageContentEdited(core: Core, chatRoom: ChatRoom, message: ChatMessage)
	
	/// Gets the message retracted callback. 
	/// - Returns: The LinphoneCoreCbsMessageRetractedCb callback called. 
	func onMessageRetracted(core: Core, chatRoom: ChatRoom, message: ChatMessage)
}

public extension CoreDelegate {
	
	func onNewAlertTriggered(core: Core, alert: Alert) {}
	
	func onGlobalStateChanged(core: Core, state: GlobalState, message: String) {}
	
	func onRegistrationStateChanged(core: Core, proxyConfig: ProxyConfig, state: RegistrationState, message: String) {}
	
	func onConferenceInfoReceived(core: Core, conferenceInfo: ConferenceInfo) {}
	
	func onPushNotificationReceived(core: Core, payload: String) {}
	
	func onPreviewDisplayErrorOccurred(core: Core, errorCode: Int) {}
	
	func onCallStateChanged(core: Core, call: Call, state: Call.State, message: String) {}
	
	func onNotifyPresenceReceived(core: Core, linphoneFriend: Friend) {}
	
	func onNotifyPresenceReceivedForUriOrTel(core: Core, linphoneFriend: Friend, uriOrTel: String, presenceModel: PresenceModel) {}
	
	func onNewSubscriptionRequested(core: Core, linphoneFriend: Friend, url: String) {}
	
	func onAuthenticationRequested(core: Core, authInfo: AuthInfo, method: AuthMethod) {}
	
	func onCallLogUpdated(core: Core, callLog: CallLog) {}
	
	func onCallIdUpdated(core: Core, previousCallId: String, currentCallId: String) {}
		
	func onMessageReceived(core: Core, chatRoom: ChatRoom, message: ChatMessage) {}
	
	func onNewMessageReaction(core: Core, chatRoom: ChatRoom, message: ChatMessage, reaction: ChatMessageReaction) {}
	
	func onReactionRemoved(core: Core, chatRoom: ChatRoom, message: ChatMessage, address: Address) {}
	
	func onMessagesReceived(core: Core, chatRoom: ChatRoom, messages: [ChatMessage]) {}
	
	func onMessageSent(core: Core, chatRoom: ChatRoom, message: ChatMessage) {}
	
	func onChatRoomSessionStateChanged(core: Core, chatRoom: ChatRoom, state: Call.State, message: String) {}
	
	func onChatRoomRead(core: Core, chatRoom: ChatRoom) {}
	
	func onMessageReceivedUnableDecrypt(core: Core, chatRoom: ChatRoom, message: ChatMessage) {}
	
	func onIsComposingReceived(core: Core, chatRoom: ChatRoom) {}
	
	func onDtmfReceived(core: Core, call: Call, dtmf: Int) {}
	
	func onReferReceived(core: Core, referToAddr: Address, customHeaders: Headers, content: Content?) {}
	
	func onCallGoclearAckSent(core: Core, call: Call) {}
	
	func onCallEncryptionChanged(core: Core, call: Call, mediaEncryptionEnabled: Bool, authenticationToken: String) {}
	
	func onCallSendMasterKeyChanged(core: Core, call: Call, masterKey: String) {}
	
	func onCallReceiveMasterKeyChanged(core: Core, call: Call, masterKey: String) {}
	
	func onTransferStateChanged(core: Core, transferred: Call, callState: Call.State) {}
	
	func onBuddyInfoUpdated(core: Core, linphoneFriend: Friend) {}
	
	func onCallStatsUpdated(core: Core, call: Call, callStats: CallStats) {}
	
	func onInfoReceived(core: Core, call: Call, message: InfoMessage) {}
	
	func onSubscriptionStateChanged(core: Core, linphoneEvent: Event, state: SubscriptionState) {}
	
	func onNotifySent(core: Core, linphoneEvent: Event, body: Content?) {}
	
	func onNotifyReceived(core: Core, linphoneEvent: Event, notifiedEvent: String, body: Content?) {}
	
	func onSubscribeReceived(core: Core, linphoneEvent: Event, subscribeEvent: String, body: Content?) {}
	
	func onPublishStateChanged(core: Core, linphoneEvent: Event, state: PublishState) {}
	
	func onPublishReceived(core: Core, linphoneEvent: Event, publishEvent: String, body: Content?) {}
	
	func onConfiguringStatus(core: Core, status: ConfiguringState, message: String) {}
	
	func onNetworkReachable(core: Core, reachable: Bool) {}
	
	func onLogCollectionUploadStateChanged(core: Core, state: Core.LogCollectionUploadState, info: String) {}
	
	func onLogCollectionUploadProgressIndication(core: Core, offset: Int, total: Int) {}
	
	func onFriendListCreated(core: Core, friendList: FriendList) {}
	
	func onFriendListRemoved(core: Core, friendList: FriendList) {}
	
	func onCallCreated(core: Core, call: Call) {}
	
	func onVersionUpdateCheckResultReceived(core: Core, result: VersionUpdateCheckResult, version: String, url: String) {}
	
	func onConferenceStateChanged(core: Core, conference: Conference, state: Conference.State) {}
	
	func onChatRoomStateChanged(core: Core, chatRoom: ChatRoom, state: ChatRoom.State) {}
	
	func onChatRoomSubjectChanged(core: Core, chatRoom: ChatRoom) {}
	
	func onChatRoomEphemeralMessageDeleted(core: Core, chatRoom: ChatRoom) {}
	
	func onImeeUserRegistration(core: Core, status: Bool, userId: String, info: String) {}
	
	func onQrcodeFound(core: Core, result: String) {}
	
	func onFirstCallStarted(core: Core) {}
	
	func onLastCallEnded(core: Core) {}
	
	func onAudioDeviceChanged(core: Core, audioDevice: AudioDevice) {}
	
	func onAudioDevicesListUpdated(core: Core) {}
	
	func onEcCalibrationResult(core: Core, status: EcCalibratorStatus, delayMs: Int) {}
	
	func onEcCalibrationAudioInit(core: Core) {}
	
	func onEcCalibrationAudioUninit(core: Core) {}
	
	func onAccountRegistrationStateChanged(core: Core, account: Account, state: RegistrationState, message: String) {}
	
	func onDefaultAccountChanged(core: Core, account: Account?) {}
	
	func onAccountAdded(core: Core, account: Account) {}
	
	func onAccountRemoved(core: Core, account: Account) {}
	
	func onMessageWaitingIndicationChanged(core: Core, lev: Event, mwi: MessageWaitingIndication) {}
	
	func onSnapshotTaken(core: Core, filePath: String) {}
	
	func onMessageContentEdited(core: Core, chatRoom: ChatRoom, message: ChatMessage) {}
	
	func onMessageRetracted(core: Core, chatRoom: ChatRoom, message: ChatMessage) {}
}

public final class CoreDelegateStub : CoreDelegate
{
	var _onNewAlertTriggered: ((Core, Alert) -> Void)?
	var _onGlobalStateChanged: ((Core, GlobalState, String) -> Void)?
	var _onRegistrationStateChanged: ((Core, ProxyConfig, RegistrationState, String) -> Void)?
	var _onConferenceInfoReceived: ((Core, ConferenceInfo) -> Void)?
	var _onPushNotificationReceived: ((Core, String) -> Void)?
	var _onPreviewDisplayErrorOccurred: ((Core, Int) -> Void)?
	var _onCallStateChanged: ((Core, Call, Call.State, String) -> Void)?
	var _onNotifyPresenceReceived: ((Core, Friend) -> Void)?
	var _onNotifyPresenceReceivedForUriOrTel: ((Core, Friend, String, PresenceModel) -> Void)?
	var _onNewSubscriptionRequested: ((Core, Friend, String) -> Void)?
	var _onAuthenticationRequested: ((Core, AuthInfo, AuthMethod) -> Void)?
	var _onCallLogUpdated: ((Core, CallLog) -> Void)?
	var _onCallIdUpdated: ((Core, String, String) -> Void)?
	var _onMessageReceived: ((Core, ChatRoom, ChatMessage) -> Void)?
	var _onNewMessageReaction: ((Core, ChatRoom, ChatMessage, ChatMessageReaction) -> Void)?
	var _onReactionRemoved: ((Core, ChatRoom, ChatMessage, Address) -> Void)?
	var _onMessagesReceived: ((Core, ChatRoom, [ChatMessage]) -> Void)?
	var _onMessageSent: ((Core, ChatRoom, ChatMessage) -> Void)?
	var _onChatRoomSessionStateChanged: ((Core, ChatRoom, Call.State, String) -> Void)?
	var _onChatRoomRead: ((Core, ChatRoom) -> Void)?
	var _onMessageReceivedUnableDecrypt: ((Core, ChatRoom, ChatMessage) -> Void)?
	var _onIsComposingReceived: ((Core, ChatRoom) -> Void)?
	var _onDtmfReceived: ((Core, Call, Int) -> Void)?
	var _onReferReceived: ((Core, Address, Headers, Content?) -> Void)?
	var _onCallGoclearAckSent: ((Core, Call) -> Void)?
	var _onCallEncryptionChanged: ((Core, Call, Bool, String) -> Void)?
	var _onCallSendMasterKeyChanged: ((Core, Call, String) -> Void)?
	var _onCallReceiveMasterKeyChanged: ((Core, Call, String) -> Void)?
	var _onTransferStateChanged: ((Core, Call, Call.State) -> Void)?
	var _onBuddyInfoUpdated: ((Core, Friend) -> Void)?
	var _onCallStatsUpdated: ((Core, Call, CallStats) -> Void)?
	var _onInfoReceived: ((Core, Call, InfoMessage) -> Void)?
	var _onSubscriptionStateChanged: ((Core, Event, SubscriptionState) -> Void)?
	var _onNotifySent: ((Core, Event, Content?) -> Void)?
	var _onNotifyReceived: ((Core, Event, String, Content?) -> Void)?
	var _onSubscribeReceived: ((Core, Event, String, Content?) -> Void)?
	var _onPublishStateChanged: ((Core, Event, PublishState) -> Void)?
	var _onPublishReceived: ((Core, Event, String, Content?) -> Void)?
	var _onConfiguringStatus: ((Core, ConfiguringState, String) -> Void)?
	var _onNetworkReachable: ((Core, Bool) -> Void)?
	var _onLogCollectionUploadStateChanged: ((Core, Core.LogCollectionUploadState, String) -> Void)?
	var _onLogCollectionUploadProgressIndication: ((Core, Int, Int) -> Void)?
	var _onFriendListCreated: ((Core, FriendList) -> Void)?
	var _onFriendListRemoved: ((Core, FriendList) -> Void)?
	var _onCallCreated: ((Core, Call) -> Void)?
	var _onVersionUpdateCheckResultReceived: ((Core, VersionUpdateCheckResult, String, String) -> Void)?
	var _onConferenceStateChanged: ((Core, Conference, Conference.State) -> Void)?
	var _onChatRoomStateChanged: ((Core, ChatRoom, ChatRoom.State) -> Void)?
	var _onChatRoomSubjectChanged: ((Core, ChatRoom) -> Void)?
	var _onChatRoomEphemeralMessageDeleted: ((Core, ChatRoom) -> Void)?
	var _onImeeUserRegistration: ((Core, Bool, String, String) -> Void)?
	var _onQrcodeFound: ((Core, String) -> Void)?
	var _onFirstCallStarted: ((Core) -> Void)?
	var _onLastCallEnded: ((Core) -> Void)?
	var _onAudioDeviceChanged: ((Core, AudioDevice) -> Void)?
	var _onAudioDevicesListUpdated: ((Core) -> Void)?
	var _onEcCalibrationResult: ((Core, EcCalibratorStatus, Int) -> Void)?
	var _onEcCalibrationAudioInit: ((Core) -> Void)?
	var _onEcCalibrationAudioUninit: ((Core) -> Void)?
	var _onAccountRegistrationStateChanged: ((Core, Account, RegistrationState, String) -> Void)?
	var _onDefaultAccountChanged: ((Core, Account?) -> Void)?
	var _onAccountAdded: ((Core, Account) -> Void)?
	var _onAccountRemoved: ((Core, Account) -> Void)?
	var _onMessageWaitingIndicationChanged: ((Core, Event, MessageWaitingIndication) -> Void)?
	var _onSnapshotTaken: ((Core, String) -> Void)?
	var _onMessageContentEdited: ((Core, ChatRoom, ChatMessage) -> Void)?
	var _onMessageRetracted: ((Core, ChatRoom, ChatMessage) -> Void)?

	
	public func onNewAlertTriggered(core: Core, alert: Alert){_onNewAlertTriggered.map{$0(core, alert)}}
	
	public func onGlobalStateChanged(core: Core, state: GlobalState, message: String){_onGlobalStateChanged.map{$0(core, state, message)}}
	
	public func onRegistrationStateChanged(core: Core, proxyConfig: ProxyConfig, state: RegistrationState, message: String){_onRegistrationStateChanged.map{$0(core, proxyConfig, state, message)}}
	
	public func onConferenceInfoReceived(core: Core, conferenceInfo: ConferenceInfo){_onConferenceInfoReceived.map{$0(core, conferenceInfo)}}
	
	public func onPushNotificationReceived(core: Core, payload: String){_onPushNotificationReceived.map{$0(core, payload)}}
	
	public func onPreviewDisplayErrorOccurred(core: Core, errorCode: Int){_onPreviewDisplayErrorOccurred.map{$0(core, errorCode)}}
	
	public func onCallStateChanged(core: Core, call: Call, state: Call.State, message: String){_onCallStateChanged.map{$0(core, call, state, message)}}
	
	public func onNotifyPresenceReceived(core: Core, linphoneFriend: Friend){_onNotifyPresenceReceived.map{$0(core, linphoneFriend)}}
	
	public func onNotifyPresenceReceivedForUriOrTel(core: Core, linphoneFriend: Friend, uriOrTel: String, presenceModel: PresenceModel){_onNotifyPresenceReceivedForUriOrTel.map{$0(core, linphoneFriend, uriOrTel, presenceModel)}}
	
	public func onNewSubscriptionRequested(core: Core, linphoneFriend: Friend, url: String){_onNewSubscriptionRequested.map{$0(core, linphoneFriend, url)}}
	
	public func onAuthenticationRequested(core: Core, authInfo: AuthInfo, method: AuthMethod){_onAuthenticationRequested.map{$0(core, authInfo, method)}}
	
	public func onCallLogUpdated(core: Core, callLog: CallLog){_onCallLogUpdated.map{$0(core, callLog)}}
	
	public func onCallIdUpdated(core: Core, previousCallId: String, currentCallId: String){_onCallIdUpdated.map{$0(core, previousCallId, currentCallId)}}
		
	public func onMessageReceived(core: Core, chatRoom: ChatRoom, message: ChatMessage){_onMessageReceived.map{$0(core, chatRoom, message)}}
	
	public func onNewMessageReaction(core: Core, chatRoom: ChatRoom, message: ChatMessage, reaction: ChatMessageReaction){_onNewMessageReaction.map{$0(core, chatRoom, message, reaction)}}
	
	public func onReactionRemoved(core: Core, chatRoom: ChatRoom, message: ChatMessage, address: Address){_onReactionRemoved.map{$0(core, chatRoom, message, address)}}
	
	public func onMessagesReceived(core: Core, chatRoom: ChatRoom, messages: [ChatMessage]){_onMessagesReceived.map{$0(core, chatRoom, messages)}}
	
	public func onMessageSent(core: Core, chatRoom: ChatRoom, message: ChatMessage){_onMessageSent.map{$0(core, chatRoom, message)}}
	
	public func onChatRoomSessionStateChanged(core: Core, chatRoom: ChatRoom, state: Call.State, message: String){_onChatRoomSessionStateChanged.map{$0(core, chatRoom, state, message)}}
	
	public func onChatRoomRead(core: Core, chatRoom: ChatRoom){_onChatRoomRead.map{$0(core, chatRoom)}}
	
	public func onMessageReceivedUnableDecrypt(core: Core, chatRoom: ChatRoom, message: ChatMessage){_onMessageReceivedUnableDecrypt.map{$0(core, chatRoom, message)}}
	
	public func onIsComposingReceived(core: Core, chatRoom: ChatRoom){_onIsComposingReceived.map{$0(core, chatRoom)}}
	
	public func onDtmfReceived(core: Core, call: Call, dtmf: Int){_onDtmfReceived.map{$0(core, call, dtmf)}}
	
	public func onReferReceived(core: Core, referToAddr: Address, customHeaders: Headers, content: Content?){_onReferReceived.map{$0(core, referToAddr, customHeaders, content)}}
	
	public func onCallGoclearAckSent(core: Core, call: Call){_onCallGoclearAckSent.map{$0(core, call)}}
	
	public func onCallEncryptionChanged(core: Core, call: Call, mediaEncryptionEnabled: Bool, authenticationToken: String){_onCallEncryptionChanged.map{$0(core, call, mediaEncryptionEnabled, authenticationToken)}}
	
	public func onCallSendMasterKeyChanged(core: Core, call: Call, masterKey: String){_onCallSendMasterKeyChanged.map{$0(core, call, masterKey)}}
	
	public func onCallReceiveMasterKeyChanged(core: Core, call: Call, masterKey: String){_onCallReceiveMasterKeyChanged.map{$0(core, call, masterKey)}}
	
	public func onTransferStateChanged(core: Core, transferred: Call, callState: Call.State){_onTransferStateChanged.map{$0(core, transferred, callState)}}
	
	public func onBuddyInfoUpdated(core: Core, linphoneFriend: Friend){_onBuddyInfoUpdated.map{$0(core, linphoneFriend)}}
	
	public func onCallStatsUpdated(core: Core, call: Call, callStats: CallStats){_onCallStatsUpdated.map{$0(core, call, callStats)}}
	
	public func onInfoReceived(core: Core, call: Call, message: InfoMessage){_onInfoReceived.map{$0(core, call, message)}}
	
	public func onSubscriptionStateChanged(core: Core, linphoneEvent: Event, state: SubscriptionState){_onSubscriptionStateChanged.map{$0(core, linphoneEvent, state)}}
	
	public func onNotifySent(core: Core, linphoneEvent: Event, body: Content?){_onNotifySent.map{$0(core, linphoneEvent, body)}}
	
	public func onNotifyReceived(core: Core, linphoneEvent: Event, notifiedEvent: String, body: Content?){_onNotifyReceived.map{$0(core, linphoneEvent, notifiedEvent, body)}}
	
	public func onSubscribeReceived(core: Core, linphoneEvent: Event, subscribeEvent: String, body: Content?){_onSubscribeReceived.map{$0(core, linphoneEvent, subscribeEvent, body)}}
	
	public func onPublishStateChanged(core: Core, linphoneEvent: Event, state: PublishState){_onPublishStateChanged.map{$0(core, linphoneEvent, state)}}
	
	public func onPublishReceived(core: Core, linphoneEvent: Event, publishEvent: String, body: Content?){_onPublishReceived.map{$0(core, linphoneEvent, publishEvent, body)}}
	
	public func onConfiguringStatus(core: Core, status: ConfiguringState, message: String){_onConfiguringStatus.map{$0(core, status, message)}}
	
	public func onNetworkReachable(core: Core, reachable: Bool){_onNetworkReachable.map{$0(core, reachable)}}
	
	public func onLogCollectionUploadStateChanged(core: Core, state: Core.LogCollectionUploadState, info: String){_onLogCollectionUploadStateChanged.map{$0(core, state, info)}}
	
	public func onLogCollectionUploadProgressIndication(core: Core, offset: Int, total: Int){_onLogCollectionUploadProgressIndication.map{$0(core, offset, total)}}
	
	public func onFriendListCreated(core: Core, friendList: FriendList){_onFriendListCreated.map{$0(core, friendList)}}
	
	public func onFriendListRemoved(core: Core, friendList: FriendList){_onFriendListRemoved.map{$0(core, friendList)}}
	
	public func onCallCreated(core: Core, call: Call){_onCallCreated.map{$0(core, call)}}
	
	public func onVersionUpdateCheckResultReceived(core: Core, result: VersionUpdateCheckResult, version: String, url: String){_onVersionUpdateCheckResultReceived.map{$0(core, result, version, url)}}
	
	public func onConferenceStateChanged(core: Core, conference: Conference, state: Conference.State){_onConferenceStateChanged.map{$0(core, conference, state)}}
	
	public func onChatRoomStateChanged(core: Core, chatRoom: ChatRoom, state: ChatRoom.State){_onChatRoomStateChanged.map{$0(core, chatRoom, state)}}
	
	public func onChatRoomSubjectChanged(core: Core, chatRoom: ChatRoom){_onChatRoomSubjectChanged.map{$0(core, chatRoom)}}
	
	public func onChatRoomEphemeralMessageDeleted(core: Core, chatRoom: ChatRoom){_onChatRoomEphemeralMessageDeleted.map{$0(core, chatRoom)}}
	
	public func onImeeUserRegistration(core: Core, status: Bool, userId: String, info: String){_onImeeUserRegistration.map{$0(core, status, userId, info)}}
	
	public func onQrcodeFound(core: Core, result: String){_onQrcodeFound.map{$0(core, result)}}
	
	public func onFirstCallStarted(core: Core){_onFirstCallStarted.map{$0(core)}}
	
	public func onLastCallEnded(core: Core){_onLastCallEnded.map{$0(core)}}
	
	public func onAudioDeviceChanged(core: Core, audioDevice: AudioDevice){_onAudioDeviceChanged.map{$0(core, audioDevice)}}
	
	public func onAudioDevicesListUpdated(core: Core){_onAudioDevicesListUpdated.map{$0(core)}}
	
	public func onEcCalibrationResult(core: Core, status: EcCalibratorStatus, delayMs: Int){_onEcCalibrationResult.map{$0(core, status, delayMs)}}
	
	public func onEcCalibrationAudioInit(core: Core){_onEcCalibrationAudioInit.map{$0(core)}}
	
	public func onEcCalibrationAudioUninit(core: Core){_onEcCalibrationAudioUninit.map{$0(core)}}
	
	public func onAccountRegistrationStateChanged(core: Core, account: Account, state: RegistrationState, message: String){_onAccountRegistrationStateChanged.map{$0(core, account, state, message)}}
	
	public func onDefaultAccountChanged(core: Core, account: Account?){_onDefaultAccountChanged.map{$0(core, account)}}
	
	public func onAccountAdded(core: Core, account: Account){_onAccountAdded.map{$0(core, account)}}
	
	public func onAccountRemoved(core: Core, account: Account){_onAccountRemoved.map{$0(core, account)}}
	
	public func onMessageWaitingIndicationChanged(core: Core, lev: Event, mwi: MessageWaitingIndication){_onMessageWaitingIndicationChanged.map{$0(core, lev, mwi)}}
	
	public func onSnapshotTaken(core: Core, filePath: String){_onSnapshotTaken.map{$0(core, filePath)}}
	
	public func onMessageContentEdited(core: Core, chatRoom: ChatRoom, message: ChatMessage){_onMessageContentEdited.map{$0(core, chatRoom, message)}}
	
	public func onMessageRetracted(core: Core, chatRoom: ChatRoom, message: ChatMessage){_onMessageRetracted.map{$0(core, chatRoom, message)}}

	public init (
		onNewAlertTriggered: ((Core, Alert) -> Void)? = nil,
		onGlobalStateChanged: ((Core, GlobalState, String) -> Void)? = nil,
		onRegistrationStateChanged: ((Core, ProxyConfig, RegistrationState, String) -> Void)? = nil,
		onConferenceInfoReceived: ((Core, ConferenceInfo) -> Void)? = nil,
		onPushNotificationReceived: ((Core, String) -> Void)? = nil,
		onPreviewDisplayErrorOccurred: ((Core, Int) -> Void)? = nil,
		onCallStateChanged: ((Core, Call, Call.State, String) -> Void)? = nil,
		onNotifyPresenceReceived: ((Core, Friend) -> Void)? = nil,
		onNotifyPresenceReceivedForUriOrTel: ((Core, Friend, String, PresenceModel) -> Void)? = nil,
		onNewSubscriptionRequested: ((Core, Friend, String) -> Void)? = nil,
		onAuthenticationRequested: ((Core, AuthInfo, AuthMethod) -> Void)? = nil,
		onCallLogUpdated: ((Core, CallLog) -> Void)? = nil,
		onCallIdUpdated: ((Core, String, String) -> Void)? = nil,
		onMessageReceived: ((Core, ChatRoom, ChatMessage) -> Void)? = nil,
		onNewMessageReaction: ((Core, ChatRoom, ChatMessage, ChatMessageReaction) -> Void)? = nil,
		onReactionRemoved: ((Core, ChatRoom, ChatMessage, Address) -> Void)? = nil,
		onMessagesReceived: ((Core, ChatRoom, [ChatMessage]) -> Void)? = nil,
		onMessageSent: ((Core, ChatRoom, ChatMessage) -> Void)? = nil,
		onChatRoomSessionStateChanged: ((Core, ChatRoom, Call.State, String) -> Void)? = nil,
		onChatRoomRead: ((Core, ChatRoom) -> Void)? = nil,
		onMessageReceivedUnableDecrypt: ((Core, ChatRoom, ChatMessage) -> Void)? = nil,
		onIsComposingReceived: ((Core, ChatRoom) -> Void)? = nil,
		onDtmfReceived: ((Core, Call, Int) -> Void)? = nil,
		onReferReceived: ((Core, Address, Headers, Content?) -> Void)? = nil,
		onCallGoclearAckSent: ((Core, Call) -> Void)? = nil,
		onCallEncryptionChanged: ((Core, Call, Bool, String) -> Void)? = nil,
		onCallSendMasterKeyChanged: ((Core, Call, String) -> Void)? = nil,
		onCallReceiveMasterKeyChanged: ((Core, Call, String) -> Void)? = nil,
		onTransferStateChanged: ((Core, Call, Call.State) -> Void)? = nil,
		onBuddyInfoUpdated: ((Core, Friend) -> Void)? = nil,
		onCallStatsUpdated: ((Core, Call, CallStats) -> Void)? = nil,
		onInfoReceived: ((Core, Call, InfoMessage) -> Void)? = nil,
		onSubscriptionStateChanged: ((Core, Event, SubscriptionState) -> Void)? = nil,
		onNotifySent: ((Core, Event, Content?) -> Void)? = nil,
		onNotifyReceived: ((Core, Event, String, Content?) -> Void)? = nil,
		onSubscribeReceived: ((Core, Event, String, Content?) -> Void)? = nil,
		onPublishStateChanged: ((Core, Event, PublishState) -> Void)? = nil,
		onPublishReceived: ((Core, Event, String, Content?) -> Void)? = nil,
		onConfiguringStatus: ((Core, ConfiguringState, String) -> Void)? = nil,
		onNetworkReachable: ((Core, Bool) -> Void)? = nil,
		onLogCollectionUploadStateChanged: ((Core, Core.LogCollectionUploadState, String) -> Void)? = nil,
		onLogCollectionUploadProgressIndication: ((Core, Int, Int) -> Void)? = nil,
		onFriendListCreated: ((Core, FriendList) -> Void)? = nil,
		onFriendListRemoved: ((Core, FriendList) -> Void)? = nil,
		onCallCreated: ((Core, Call) -> Void)? = nil,
		onVersionUpdateCheckResultReceived: ((Core, VersionUpdateCheckResult, String, String) -> Void)? = nil,
		onConferenceStateChanged: ((Core, Conference, Conference.State) -> Void)? = nil,
		onChatRoomStateChanged: ((Core, ChatRoom, ChatRoom.State) -> Void)? = nil,
		onChatRoomSubjectChanged: ((Core, ChatRoom) -> Void)? = nil,
		onChatRoomEphemeralMessageDeleted: ((Core, ChatRoom) -> Void)? = nil,
		onImeeUserRegistration: ((Core, Bool, String, String) -> Void)? = nil,
		onQrcodeFound: ((Core, String) -> Void)? = nil,
		onFirstCallStarted: ((Core) -> Void)? = nil,
		onLastCallEnded: ((Core) -> Void)? = nil,
		onAudioDeviceChanged: ((Core, AudioDevice) -> Void)? = nil,
		onAudioDevicesListUpdated: ((Core) -> Void)? = nil,
		onEcCalibrationResult: ((Core, EcCalibratorStatus, Int) -> Void)? = nil,
		onEcCalibrationAudioInit: ((Core) -> Void)? = nil,
		onEcCalibrationAudioUninit: ((Core) -> Void)? = nil,
		onAccountRegistrationStateChanged: ((Core, Account, RegistrationState, String) -> Void)? = nil,
		onDefaultAccountChanged: ((Core, Account?) -> Void)? = nil,
		onAccountAdded: ((Core, Account) -> Void)? = nil,
		onAccountRemoved: ((Core, Account) -> Void)? = nil,
		onMessageWaitingIndicationChanged: ((Core, Event, MessageWaitingIndication) -> Void)? = nil,
		onSnapshotTaken: ((Core, String) -> Void)? = nil,
		onMessageContentEdited: ((Core, ChatRoom, ChatMessage) -> Void)? = nil,
		onMessageRetracted: ((Core, ChatRoom, ChatMessage) -> Void)? = nil
	) {
		self._onNewAlertTriggered = onNewAlertTriggered
		self._onGlobalStateChanged = onGlobalStateChanged
		self._onRegistrationStateChanged = onRegistrationStateChanged
		self._onConferenceInfoReceived = onConferenceInfoReceived
		self._onPushNotificationReceived = onPushNotificationReceived
		self._onPreviewDisplayErrorOccurred = onPreviewDisplayErrorOccurred
		self._onCallStateChanged = onCallStateChanged
		self._onNotifyPresenceReceived = onNotifyPresenceReceived
		self._onNotifyPresenceReceivedForUriOrTel = onNotifyPresenceReceivedForUriOrTel
		self._onNewSubscriptionRequested = onNewSubscriptionRequested
		self._onAuthenticationRequested = onAuthenticationRequested
		self._onCallLogUpdated = onCallLogUpdated
		self._onCallIdUpdated = onCallIdUpdated
		self._onMessageReceived = onMessageReceived
		self._onNewMessageReaction = onNewMessageReaction
		self._onReactionRemoved = onReactionRemoved
		self._onMessagesReceived = onMessagesReceived
		self._onMessageSent = onMessageSent
		self._onChatRoomSessionStateChanged = onChatRoomSessionStateChanged
		self._onChatRoomRead = onChatRoomRead
		self._onMessageReceivedUnableDecrypt = onMessageReceivedUnableDecrypt
		self._onIsComposingReceived = onIsComposingReceived
		self._onDtmfReceived = onDtmfReceived
		self._onReferReceived = onReferReceived
		self._onCallGoclearAckSent = onCallGoclearAckSent
		self._onCallEncryptionChanged = onCallEncryptionChanged
		self._onCallSendMasterKeyChanged = onCallSendMasterKeyChanged
		self._onCallReceiveMasterKeyChanged = onCallReceiveMasterKeyChanged
		self._onTransferStateChanged = onTransferStateChanged
		self._onBuddyInfoUpdated = onBuddyInfoUpdated
		self._onCallStatsUpdated = onCallStatsUpdated
		self._onInfoReceived = onInfoReceived
		self._onSubscriptionStateChanged = onSubscriptionStateChanged
		self._onNotifySent = onNotifySent
		self._onNotifyReceived = onNotifyReceived
		self._onSubscribeReceived = onSubscribeReceived
		self._onPublishStateChanged = onPublishStateChanged
		self._onPublishReceived = onPublishReceived
		self._onConfiguringStatus = onConfiguringStatus
		self._onNetworkReachable = onNetworkReachable
		self._onLogCollectionUploadStateChanged = onLogCollectionUploadStateChanged
		self._onLogCollectionUploadProgressIndication = onLogCollectionUploadProgressIndication
		self._onFriendListCreated = onFriendListCreated
		self._onFriendListRemoved = onFriendListRemoved
		self._onCallCreated = onCallCreated
		self._onVersionUpdateCheckResultReceived = onVersionUpdateCheckResultReceived
		self._onConferenceStateChanged = onConferenceStateChanged
		self._onChatRoomStateChanged = onChatRoomStateChanged
		self._onChatRoomSubjectChanged = onChatRoomSubjectChanged
		self._onChatRoomEphemeralMessageDeleted = onChatRoomEphemeralMessageDeleted
		self._onImeeUserRegistration = onImeeUserRegistration
		self._onQrcodeFound = onQrcodeFound
		self._onFirstCallStarted = onFirstCallStarted
		self._onLastCallEnded = onLastCallEnded
		self._onAudioDeviceChanged = onAudioDeviceChanged
		self._onAudioDevicesListUpdated = onAudioDevicesListUpdated
		self._onEcCalibrationResult = onEcCalibrationResult
		self._onEcCalibrationAudioInit = onEcCalibrationAudioInit
		self._onEcCalibrationAudioUninit = onEcCalibrationAudioUninit
		self._onAccountRegistrationStateChanged = onAccountRegistrationStateChanged
		self._onDefaultAccountChanged = onDefaultAccountChanged
		self._onAccountAdded = onAccountAdded
		self._onAccountRemoved = onAccountRemoved
		self._onMessageWaitingIndicationChanged = onMessageWaitingIndicationChanged
		self._onSnapshotTaken = onSnapshotTaken
		self._onMessageContentEdited = onMessageContentEdited
		self._onMessageRetracted = onMessageRetracted
	}
}

class CoreDelegateManager
{
	var cPtr: OpaquePointer
	public var delegate: CoreDelegate?

	public init() {
		cPtr = linphone_factory_create_core_cbs(linphone_factory_get())
		belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passRetained(self).toOpaque()), {
			data in
			if (data != nil) {
				Unmanaged<CoreDelegateManager>.fromOpaque(data!).release()
			}
		})

		linphone_core_cbs_set_new_alert_triggered(cPtr, { (core, alert) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onNewAlertTriggered(core: sObject, alert: Alert.getSwiftObject(cObject: alert!))
			}
		})

		linphone_core_cbs_set_global_state_changed(cPtr, { (core, state, message) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onGlobalStateChanged(core: sObject, state: GlobalState(rawValue: Int(state.rawValue))!, message: charArrayToString(charPointer: message))
			}
		})

		linphone_core_cbs_set_registration_state_changed(cPtr, { (core, proxyConfig, state, message) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onRegistrationStateChanged(core: sObject, proxyConfig: ProxyConfig.getSwiftObject(cObject: proxyConfig!), state: RegistrationState(rawValue: Int(state.rawValue))!, message: charArrayToString(charPointer: message))
			}
		})

		linphone_core_cbs_set_conference_info_received(cPtr, { (core, conferenceInfo) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onConferenceInfoReceived(core: sObject, conferenceInfo: ConferenceInfo.getSwiftObject(cObject: conferenceInfo!))
			}
		})

		linphone_core_cbs_set_push_notification_received(cPtr, { (core, payload) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onPushNotificationReceived(core: sObject, payload: charArrayToString(charPointer: payload))
			}
		})

		linphone_core_cbs_set_preview_display_error_occurred(cPtr, { (core, errorCode) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onPreviewDisplayErrorOccurred(core: sObject, errorCode: Int(errorCode))
			}
		})

		linphone_core_cbs_set_call_state_changed(cPtr, { (core, call, state, message) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onCallStateChanged(core: sObject, call: Call.getSwiftObject(cObject: call!), state: Call.State(rawValue: Int(state.rawValue))!, message: charArrayToString(charPointer: message))
			}
		})

		linphone_core_cbs_set_notify_presence_received(cPtr, { (core, linphoneFriend) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onNotifyPresenceReceived(core: sObject, linphoneFriend: Friend.getSwiftObject(cObject: linphoneFriend!))
			}
		})

		linphone_core_cbs_set_notify_presence_received_for_uri_or_tel(cPtr, { (core, linphoneFriend, uriOrTel, presenceModel) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onNotifyPresenceReceivedForUriOrTel(core: sObject, linphoneFriend: Friend.getSwiftObject(cObject: linphoneFriend!), uriOrTel: charArrayToString(charPointer: uriOrTel), presenceModel: PresenceModel.getSwiftObject(cObject: presenceModel!))
			}
		})

		linphone_core_cbs_set_new_subscription_requested(cPtr, { (core, linphoneFriend, url) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onNewSubscriptionRequested(core: sObject, linphoneFriend: Friend.getSwiftObject(cObject: linphoneFriend!), url: charArrayToString(charPointer: url))
			}
		})

		linphone_core_cbs_set_authentication_requested(cPtr, { (core, authInfo, method) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onAuthenticationRequested(core: sObject, authInfo: AuthInfo.getSwiftObject(cObject: authInfo!), method: AuthMethod(rawValue: Int(method.rawValue))!)
			}
		})

		linphone_core_cbs_set_call_log_updated(cPtr, { (core, callLog) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onCallLogUpdated(core: sObject, callLog: CallLog.getSwiftObject(cObject: callLog!))
			}
		})

		linphone_core_cbs_set_call_id_updated(cPtr, { (core, previousCallId, currentCallId) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onCallIdUpdated(core: sObject, previousCallId: charArrayToString(charPointer: previousCallId), currentCallId: charArrayToString(charPointer: currentCallId))
			}
		})

		linphone_core_cbs_set_message_received(cPtr, { (core, chatRoom, message) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onMessageReceived(core: sObject, chatRoom: ChatRoom.getSwiftObject(cObject: chatRoom!), message: ChatMessage.getSwiftObject(cObject: message!))
			}
		})

		linphone_core_cbs_set_new_message_reaction(cPtr, { (core, chatRoom, message, reaction) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onNewMessageReaction(core: sObject, chatRoom: ChatRoom.getSwiftObject(cObject: chatRoom!), message: ChatMessage.getSwiftObject(cObject: message!), reaction: ChatMessageReaction.getSwiftObject(cObject: reaction!))
			}
		})

		linphone_core_cbs_set_reaction_removed(cPtr, { (core, chatRoom, message, address) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onReactionRemoved(core: sObject, chatRoom: ChatRoom.getSwiftObject(cObject: chatRoom!), message: ChatMessage.getSwiftObject(cObject: message!), address: Address.getSwiftObject(cObject: address!))
			}
		})

		linphone_core_cbs_set_messages_received(cPtr, { (core, chatRoom, messages) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				var messagessList = [ChatMessage]()
				let messagescList = messages
				var listTemp = messagescList
				while (listTemp != nil) {
					let data = unsafeBitCast(listTemp!.pointee.data, to: OpaquePointer.self)
					messagessList.append(ChatMessage.getSwiftObject(cObject: data))
					listTemp = UnsafePointer<bctbx_list_t>(listTemp!.pointee.next)
				}
				delegate?.onMessagesReceived(core: sObject, chatRoom: ChatRoom.getSwiftObject(cObject: chatRoom!), messages: messagessList)
			}
		})

		linphone_core_cbs_set_message_sent(cPtr, { (core, chatRoom, message) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onMessageSent(core: sObject, chatRoom: ChatRoom.getSwiftObject(cObject: chatRoom!), message: ChatMessage.getSwiftObject(cObject: message!))
			}
		})

		linphone_core_cbs_set_chat_room_session_state_changed(cPtr, { (core, chatRoom, state, message) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onChatRoomSessionStateChanged(core: sObject, chatRoom: ChatRoom.getSwiftObject(cObject: chatRoom!), state: Call.State(rawValue: Int(state.rawValue))!, message: charArrayToString(charPointer: message))
			}
		})

		linphone_core_cbs_set_chat_room_read(cPtr, { (core, chatRoom) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onChatRoomRead(core: sObject, chatRoom: ChatRoom.getSwiftObject(cObject: chatRoom!))
			}
		})

		linphone_core_cbs_set_message_received_unable_decrypt(cPtr, { (core, chatRoom, message) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onMessageReceivedUnableDecrypt(core: sObject, chatRoom: ChatRoom.getSwiftObject(cObject: chatRoom!), message: ChatMessage.getSwiftObject(cObject: message!))
			}
		})

		linphone_core_cbs_set_is_composing_received(cPtr, { (core, chatRoom) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onIsComposingReceived(core: sObject, chatRoom: ChatRoom.getSwiftObject(cObject: chatRoom!))
			}
		})

		linphone_core_cbs_set_dtmf_received(cPtr, { (core, call, dtmf) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onDtmfReceived(core: sObject, call: Call.getSwiftObject(cObject: call!), dtmf: Int(dtmf))
			}
		})

		linphone_core_cbs_set_refer_received(cPtr, { (core, referToAddr, customHeaders, content) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onReferReceived(core: sObject, referToAddr: Address.getSwiftObject(cObject: referToAddr!), customHeaders: Headers.getSwiftObject(cObject: customHeaders!), content: (content != nil) ? Content.getSwiftObject(cObject: content!) : nil)
			}
		})

		linphone_core_cbs_set_call_goclear_ack_sent(cPtr, { (core, call) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onCallGoclearAckSent(core: sObject, call: Call.getSwiftObject(cObject: call!))
			}
		})

		linphone_core_cbs_set_call_encryption_changed(cPtr, { (core, call, mediaEncryptionEnabled, authenticationToken) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onCallEncryptionChanged(core: sObject, call: Call.getSwiftObject(cObject: call!), mediaEncryptionEnabled: mediaEncryptionEnabled != 0, authenticationToken: charArrayToString(charPointer: authenticationToken))
			}
		})

		linphone_core_cbs_set_call_send_master_key_changed(cPtr, { (core, call, masterKey) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onCallSendMasterKeyChanged(core: sObject, call: Call.getSwiftObject(cObject: call!), masterKey: charArrayToString(charPointer: masterKey))
			}
		})

		linphone_core_cbs_set_call_receive_master_key_changed(cPtr, { (core, call, masterKey) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onCallReceiveMasterKeyChanged(core: sObject, call: Call.getSwiftObject(cObject: call!), masterKey: charArrayToString(charPointer: masterKey))
			}
		})

		linphone_core_cbs_set_transfer_state_changed(cPtr, { (core, transferred, callState) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onTransferStateChanged(core: sObject, transferred: Call.getSwiftObject(cObject: transferred!), callState: Call.State(rawValue: Int(callState.rawValue))!)
			}
		})

		linphone_core_cbs_set_buddy_info_updated(cPtr, { (core, linphoneFriend) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onBuddyInfoUpdated(core: sObject, linphoneFriend: Friend.getSwiftObject(cObject: linphoneFriend!))
			}
		})

		linphone_core_cbs_set_call_stats_updated(cPtr, { (core, call, callStats) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onCallStatsUpdated(core: sObject, call: Call.getSwiftObject(cObject: call!), callStats: CallStats.getSwiftObject(cObject: callStats!))
			}
		})

		linphone_core_cbs_set_info_received(cPtr, { (core, call, message) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onInfoReceived(core: sObject, call: Call.getSwiftObject(cObject: call!), message: InfoMessage.getSwiftObject(cObject: message!))
			}
		})

		linphone_core_cbs_set_subscription_state_changed(cPtr, { (core, linphoneEvent, state) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onSubscriptionStateChanged(core: sObject, linphoneEvent: Event.getSwiftObject(cObject: linphoneEvent!), state: SubscriptionState(rawValue: Int(state.rawValue))!)
			}
		})

		linphone_core_cbs_set_notify_sent(cPtr, { (core, linphoneEvent, body) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onNotifySent(core: sObject, linphoneEvent: Event.getSwiftObject(cObject: linphoneEvent!), body: (body != nil) ? Content.getSwiftObject(cObject: body!) : nil)
			}
		})

		linphone_core_cbs_set_notify_received(cPtr, { (core, linphoneEvent, notifiedEvent, body) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onNotifyReceived(core: sObject, linphoneEvent: Event.getSwiftObject(cObject: linphoneEvent!), notifiedEvent: charArrayToString(charPointer: notifiedEvent), body: (body != nil) ? Content.getSwiftObject(cObject: body!) : nil)
			}
		})

		linphone_core_cbs_set_subscribe_received(cPtr, { (core, linphoneEvent, subscribeEvent, body) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onSubscribeReceived(core: sObject, linphoneEvent: Event.getSwiftObject(cObject: linphoneEvent!), subscribeEvent: charArrayToString(charPointer: subscribeEvent), body: (body != nil) ? Content.getSwiftObject(cObject: body!) : nil)
			}
		})

		linphone_core_cbs_set_publish_state_changed(cPtr, { (core, linphoneEvent, state) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onPublishStateChanged(core: sObject, linphoneEvent: Event.getSwiftObject(cObject: linphoneEvent!), state: PublishState(rawValue: Int(state.rawValue))!)
			}
		})

		linphone_core_cbs_set_publish_received(cPtr, { (core, linphoneEvent, publishEvent, body) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onPublishReceived(core: sObject, linphoneEvent: Event.getSwiftObject(cObject: linphoneEvent!), publishEvent: charArrayToString(charPointer: publishEvent), body: (body != nil) ? Content.getSwiftObject(cObject: body!) : nil)
			}
		})

		linphone_core_cbs_set_configuring_status(cPtr, { (core, status, message) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onConfiguringStatus(core: sObject, status: ConfiguringState(rawValue: Int(status.rawValue))!, message: charArrayToString(charPointer: message))
			}
		})

		linphone_core_cbs_set_network_reachable(cPtr, { (core, reachable) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onNetworkReachable(core: sObject, reachable: reachable != 0)
			}
		})

		linphone_core_cbs_set_log_collection_upload_state_changed(cPtr, { (core, state, info) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onLogCollectionUploadStateChanged(core: sObject, state: Core.LogCollectionUploadState(rawValue: Int(state.rawValue))!, info: charArrayToString(charPointer: info))
			}
		})

		linphone_core_cbs_set_log_collection_upload_progress_indication(cPtr, { (core, offset, total) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onLogCollectionUploadProgressIndication(core: sObject, offset: Int(offset), total: Int(total))
			}
		})

		linphone_core_cbs_set_friend_list_created(cPtr, { (core, friendList) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onFriendListCreated(core: sObject, friendList: FriendList.getSwiftObject(cObject: friendList!))
			}
		})

		linphone_core_cbs_set_friend_list_removed(cPtr, { (core, friendList) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onFriendListRemoved(core: sObject, friendList: FriendList.getSwiftObject(cObject: friendList!))
			}
		})

		linphone_core_cbs_set_call_created(cPtr, { (core, call) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onCallCreated(core: sObject, call: Call.getSwiftObject(cObject: call!))
			}
		})

		linphone_core_cbs_set_version_update_check_result_received(cPtr, { (core, result, version, url) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onVersionUpdateCheckResultReceived(core: sObject, result: VersionUpdateCheckResult(rawValue: Int(result.rawValue))!, version: charArrayToString(charPointer: version), url: charArrayToString(charPointer: url))
			}
		})

		linphone_core_cbs_set_conference_state_changed(cPtr, { (core, conference, state) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onConferenceStateChanged(core: sObject, conference: Conference.getSwiftObject(cObject: conference!), state: Conference.State(rawValue: Int(state.rawValue))!)
			}
		})

		linphone_core_cbs_set_chat_room_state_changed(cPtr, { (core, chatRoom, state) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onChatRoomStateChanged(core: sObject, chatRoom: ChatRoom.getSwiftObject(cObject: chatRoom!), state: ChatRoom.State(rawValue: Int(state.rawValue))!)
			}
		})

		linphone_core_cbs_set_chat_room_subject_changed(cPtr, { (core, chatRoom) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onChatRoomSubjectChanged(core: sObject, chatRoom: ChatRoom.getSwiftObject(cObject: chatRoom!))
			}
		})

		linphone_core_cbs_set_chat_room_ephemeral_message_deleted(cPtr, { (core, chatRoom) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onChatRoomEphemeralMessageDeleted(core: sObject, chatRoom: ChatRoom.getSwiftObject(cObject: chatRoom!))
			}
		})

		linphone_core_cbs_set_imee_user_registration(cPtr, { (core, status, userId, info) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onImeeUserRegistration(core: sObject, status: status != 0, userId: charArrayToString(charPointer: userId), info: charArrayToString(charPointer: info))
			}
		})

		linphone_core_cbs_set_qrcode_found(cPtr, { (core, result) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onQrcodeFound(core: sObject, result: charArrayToString(charPointer: result))
			}
		})

		linphone_core_cbs_set_first_call_started(cPtr, { (core) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onFirstCallStarted(core: sObject)
			}
		})

		linphone_core_cbs_set_last_call_ended(cPtr, { (core) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onLastCallEnded(core: sObject)
			}
		})

		linphone_core_cbs_set_audio_device_changed(cPtr, { (core, audioDevice) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onAudioDeviceChanged(core: sObject, audioDevice: AudioDevice.getSwiftObject(cObject: audioDevice!))
			}
		})

		linphone_core_cbs_set_audio_devices_list_updated(cPtr, { (core) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onAudioDevicesListUpdated(core: sObject)
			}
		})

		linphone_core_cbs_set_ec_calibration_result(cPtr, { (core, status, delayMs) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onEcCalibrationResult(core: sObject, status: EcCalibratorStatus(rawValue: Int(status.rawValue))!, delayMs: Int(delayMs))
			}
		})

		linphone_core_cbs_set_ec_calibration_audio_init(cPtr, { (core) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onEcCalibrationAudioInit(core: sObject)
			}
		})

		linphone_core_cbs_set_ec_calibration_audio_uninit(cPtr, { (core) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onEcCalibrationAudioUninit(core: sObject)
			}
		})

		linphone_core_cbs_set_account_registration_state_changed(cPtr, { (core, account, state, message) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onAccountRegistrationStateChanged(core: sObject, account: Account.getSwiftObject(cObject: account!), state: RegistrationState(rawValue: Int(state.rawValue))!, message: charArrayToString(charPointer: message))
			}
		})

		linphone_core_cbs_set_default_account_changed(cPtr, { (core, account) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onDefaultAccountChanged(core: sObject, account: (account != nil) ? Account.getSwiftObject(cObject: account!) : nil)
			}
		})

		linphone_core_cbs_set_account_added(cPtr, { (core, account) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onAccountAdded(core: sObject, account: Account.getSwiftObject(cObject: account!))
			}
		})

		linphone_core_cbs_set_account_removed(cPtr, { (core, account) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onAccountRemoved(core: sObject, account: Account.getSwiftObject(cObject: account!))
			}
		})

		linphone_core_cbs_set_message_waiting_indication_changed(cPtr, { (core, lev, mwi) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onMessageWaitingIndicationChanged(core: sObject, lev: Event.getSwiftObject(cObject: lev!), mwi: MessageWaitingIndication.getSwiftObject(cObject: mwi!))
			}
		})

		linphone_core_cbs_set_snapshot_taken(cPtr, { (core, filePath) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onSnapshotTaken(core: sObject, filePath: charArrayToString(charPointer: filePath))
			}
		})

		linphone_core_cbs_set_message_content_edited(cPtr, { (core, chatRoom, message) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onMessageContentEdited(core: sObject, chatRoom: ChatRoom.getSwiftObject(cObject: chatRoom!), message: ChatMessage.getSwiftObject(cObject: message!))
			}
		})

		linphone_core_cbs_set_message_retracted(cPtr, { (core, chatRoom, message) -> Void in
			if (core != nil) {
				let sObject = Core.getSwiftObject(cObject: core!)
				let delegate = sObject.currentDelegate
				delegate?.onMessageRetracted(core: sObject, chatRoom: ChatRoom.getSwiftObject(cObject: chatRoom!), message: ChatMessage.getSwiftObject(cObject: message!))
			}
		})
	}


}

public protocol EventDelegate : AnyObject {
	
	
	/// Get the notify response callback. 
	/// - Returns: The current notify response callback. 
	func onNotifyResponse(event: Event)
	
	/// Get the notify received callback. 
	/// - Returns: The current notify received callback. 
	func onNotifyReceived(event: Event, content: Content?)
	
	/// Get the subscribe received callback. 
	/// - Returns: The current subscribe received callback. 
	func onSubscribeReceived(event: Event)
	
	/// Get the subscribe state changed callback. 
	/// - Returns: The current subscribe state changed callback. 
	func onSubscribeStateChanged(event: Event, state: SubscriptionState)
	
	/// Get the publish received callback. 
	/// - Returns: The current publish received callback. 
	func onPublishReceived(event: Event, content: Content?)
	
	/// Get the publish state changed callback. 
	/// - Returns: The current publish state changed callback. 
	func onPublishStateChanged(event: Event, state: PublishState)
}

public extension EventDelegate {
	
	func onNotifyResponse(event: Event) {}
	
	func onNotifyReceived(event: Event, content: Content?) {}
	
	func onSubscribeReceived(event: Event) {}
	
	func onSubscribeStateChanged(event: Event, state: SubscriptionState) {}
	
	func onPublishReceived(event: Event, content: Content?) {}
	
	func onPublishStateChanged(event: Event, state: PublishState) {}
}

public final class EventDelegateStub : EventDelegate
{
	var _onNotifyResponse: ((Event) -> Void)?
	var _onNotifyReceived: ((Event, Content?) -> Void)?
	var _onSubscribeReceived: ((Event) -> Void)?
	var _onSubscribeStateChanged: ((Event, SubscriptionState) -> Void)?
	var _onPublishReceived: ((Event, Content?) -> Void)?
	var _onPublishStateChanged: ((Event, PublishState) -> Void)?

	
	public func onNotifyResponse(event: Event){_onNotifyResponse.map{$0(event)}}
	
	public func onNotifyReceived(event: Event, content: Content?){_onNotifyReceived.map{$0(event, content)}}
	
	public func onSubscribeReceived(event: Event){_onSubscribeReceived.map{$0(event)}}
	
	public func onSubscribeStateChanged(event: Event, state: SubscriptionState){_onSubscribeStateChanged.map{$0(event, state)}}
	
	public func onPublishReceived(event: Event, content: Content?){_onPublishReceived.map{$0(event, content)}}
	
	public func onPublishStateChanged(event: Event, state: PublishState){_onPublishStateChanged.map{$0(event, state)}}

	public init (
		onNotifyResponse: ((Event) -> Void)? = nil,
		onNotifyReceived: ((Event, Content?) -> Void)? = nil,
		onSubscribeReceived: ((Event) -> Void)? = nil,
		onSubscribeStateChanged: ((Event, SubscriptionState) -> Void)? = nil,
		onPublishReceived: ((Event, Content?) -> Void)? = nil,
		onPublishStateChanged: ((Event, PublishState) -> Void)? = nil
	) {
		self._onNotifyResponse = onNotifyResponse
		self._onNotifyReceived = onNotifyReceived
		self._onSubscribeReceived = onSubscribeReceived
		self._onSubscribeStateChanged = onSubscribeStateChanged
		self._onPublishReceived = onPublishReceived
		self._onPublishStateChanged = onPublishStateChanged
	}
}

class EventDelegateManager
{
	var cPtr: OpaquePointer
	public var delegate: EventDelegate?

	public init() {
		cPtr = linphone_factory_create_event_cbs(linphone_factory_get())
		belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passRetained(self).toOpaque()), {
			data in
			if (data != nil) {
				Unmanaged<EventDelegateManager>.fromOpaque(data!).release()
			}
		})

		linphone_event_cbs_set_notify_response(cPtr, { (event) -> Void in
			if (event != nil) {
				let sObject = Event.getSwiftObject(cObject: event!)
				let delegate = sObject.currentDelegate
				delegate?.onNotifyResponse(event: sObject)
			}
		})

		linphone_event_cbs_set_notify_received(cPtr, { (event, content) -> Void in
			if (event != nil) {
				let sObject = Event.getSwiftObject(cObject: event!)
				let delegate = sObject.currentDelegate
				delegate?.onNotifyReceived(event: sObject, content: (content != nil) ? Content.getSwiftObject(cObject: content!) : nil)
			}
		})

		linphone_event_cbs_set_subscribe_received(cPtr, { (event) -> Void in
			if (event != nil) {
				let sObject = Event.getSwiftObject(cObject: event!)
				let delegate = sObject.currentDelegate
				delegate?.onSubscribeReceived(event: sObject)
			}
		})

		linphone_event_cbs_set_subscribe_state_changed(cPtr, { (event, state) -> Void in
			if (event != nil) {
				let sObject = Event.getSwiftObject(cObject: event!)
				let delegate = sObject.currentDelegate
				delegate?.onSubscribeStateChanged(event: sObject, state: SubscriptionState(rawValue: Int(state.rawValue))!)
			}
		})

		linphone_event_cbs_set_publish_received(cPtr, { (event, content) -> Void in
			if (event != nil) {
				let sObject = Event.getSwiftObject(cObject: event!)
				let delegate = sObject.currentDelegate
				delegate?.onPublishReceived(event: sObject, content: (content != nil) ? Content.getSwiftObject(cObject: content!) : nil)
			}
		})

		linphone_event_cbs_set_publish_state_changed(cPtr, { (event, state) -> Void in
			if (event != nil) {
				let sObject = Event.getSwiftObject(cObject: event!)
				let delegate = sObject.currentDelegate
				delegate?.onPublishStateChanged(event: sObject, state: PublishState(rawValue: Int(state.rawValue))!)
			}
		})
	}


}

public protocol FriendDelegate : AnyObject {
	
	
	/// Get the presence received callback. 
	/// - Returns: The current presence received callback. 
	func onPresenceReceived(linphoneFriend: Friend)
}

public extension FriendDelegate {
	
	func onPresenceReceived(linphoneFriend: Friend) {}
}

public final class FriendDelegateStub : FriendDelegate
{
	var _onPresenceReceived: ((Friend) -> Void)?

	
	public func onPresenceReceived(linphoneFriend: Friend){_onPresenceReceived.map{$0(linphoneFriend)}}

	public init (
		onPresenceReceived: ((Friend) -> Void)? = nil
	) {
		self._onPresenceReceived = onPresenceReceived
	}
}

class FriendDelegateManager
{
	var cPtr: OpaquePointer
	public var delegate: FriendDelegate?

	public init() {
		cPtr = linphone_factory_create_friend_cbs(linphone_factory_get())
		belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passRetained(self).toOpaque()), {
			data in
			if (data != nil) {
				Unmanaged<FriendDelegateManager>.fromOpaque(data!).release()
			}
		})

		linphone_friend_cbs_set_presence_received(cPtr, { (linphoneFriend) -> Void in
			if (linphoneFriend != nil) {
				let sObject = Friend.getSwiftObject(cObject: linphoneFriend!)
				let delegate = sObject.currentDelegate
				delegate?.onPresenceReceived(linphoneFriend: sObject)
			}
		})
	}


}

public protocol FriendListDelegate : AnyObject {
	
	
	/// Get the contact created callback. 
	/// - Returns: The current contact created callback. 
	func onContactCreated(friendList: FriendList, linphoneFriend: Friend)
	
	/// Get the contact deleted callback. 
	/// - Returns: The current contact deleted callback. 
	func onContactDeleted(friendList: FriendList, linphoneFriend: Friend)
	
	/// Get the contact updated callback. 
	/// - Returns: The current contact updated callback. 
	func onContactUpdated(friendList: FriendList, newFriend: Friend, oldFriend: Friend)
	
	/// Get the sync status changed callback. 
	/// - Returns: The current sync status changedcallback. 
	func onSyncStatusChanged(friendList: FriendList, status: FriendList.SyncStatus, message: String)
	
	/// Get the presence received callback. 
	/// - Returns: The current presence received callback. 
	func onPresenceReceived(friendList: FriendList, friends: [Friend])
	
	/// Get the new SIP address discovered callback. 
	/// - Returns: The current new SIP address discovered callback. 
	func onNewSipAddressDiscovered(friendList: FriendList, linphoneFriend: Friend, sipUri: String)
}

public extension FriendListDelegate {
	
	func onContactCreated(friendList: FriendList, linphoneFriend: Friend) {}
	
	func onContactDeleted(friendList: FriendList, linphoneFriend: Friend) {}
	
	func onContactUpdated(friendList: FriendList, newFriend: Friend, oldFriend: Friend) {}
	
	func onSyncStatusChanged(friendList: FriendList, status: FriendList.SyncStatus, message: String) {}
	
	func onPresenceReceived(friendList: FriendList, friends: [Friend]) {}
	
	func onNewSipAddressDiscovered(friendList: FriendList, linphoneFriend: Friend, sipUri: String) {}
}

public final class FriendListDelegateStub : FriendListDelegate
{
	var _onContactCreated: ((FriendList, Friend) -> Void)?
	var _onContactDeleted: ((FriendList, Friend) -> Void)?
	var _onContactUpdated: ((FriendList, Friend, Friend) -> Void)?
	var _onSyncStatusChanged: ((FriendList, FriendList.SyncStatus, String) -> Void)?
	var _onPresenceReceived: ((FriendList, [Friend]) -> Void)?
	var _onNewSipAddressDiscovered: ((FriendList, Friend, String) -> Void)?

	
	public func onContactCreated(friendList: FriendList, linphoneFriend: Friend){_onContactCreated.map{$0(friendList, linphoneFriend)}}
	
	public func onContactDeleted(friendList: FriendList, linphoneFriend: Friend){_onContactDeleted.map{$0(friendList, linphoneFriend)}}
	
	public func onContactUpdated(friendList: FriendList, newFriend: Friend, oldFriend: Friend){_onContactUpdated.map{$0(friendList, newFriend, oldFriend)}}
	
	public func onSyncStatusChanged(friendList: FriendList, status: FriendList.SyncStatus, message: String){_onSyncStatusChanged.map{$0(friendList, status, message)}}
	
	public func onPresenceReceived(friendList: FriendList, friends: [Friend]){_onPresenceReceived.map{$0(friendList, friends)}}
	
	public func onNewSipAddressDiscovered(friendList: FriendList, linphoneFriend: Friend, sipUri: String){_onNewSipAddressDiscovered.map{$0(friendList, linphoneFriend, sipUri)}}

	public init (
		onContactCreated: ((FriendList, Friend) -> Void)? = nil,
		onContactDeleted: ((FriendList, Friend) -> Void)? = nil,
		onContactUpdated: ((FriendList, Friend, Friend) -> Void)? = nil,
		onSyncStatusChanged: ((FriendList, FriendList.SyncStatus, String) -> Void)? = nil,
		onPresenceReceived: ((FriendList, [Friend]) -> Void)? = nil,
		onNewSipAddressDiscovered: ((FriendList, Friend, String) -> Void)? = nil
	) {
		self._onContactCreated = onContactCreated
		self._onContactDeleted = onContactDeleted
		self._onContactUpdated = onContactUpdated
		self._onSyncStatusChanged = onSyncStatusChanged
		self._onPresenceReceived = onPresenceReceived
		self._onNewSipAddressDiscovered = onNewSipAddressDiscovered
	}
}

class FriendListDelegateManager
{
	var cPtr: OpaquePointer
	public var delegate: FriendListDelegate?

	public init() {
		cPtr = linphone_factory_create_friend_list_cbs(linphone_factory_get())
		belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passRetained(self).toOpaque()), {
			data in
			if (data != nil) {
				Unmanaged<FriendListDelegateManager>.fromOpaque(data!).release()
			}
		})

		linphone_friend_list_cbs_set_contact_created(cPtr, { (friendList, linphoneFriend) -> Void in
			if (friendList != nil) {
				let sObject = FriendList.getSwiftObject(cObject: friendList!)
				let delegate = sObject.currentDelegate
				delegate?.onContactCreated(friendList: sObject, linphoneFriend: Friend.getSwiftObject(cObject: linphoneFriend!))
			}
		})

		linphone_friend_list_cbs_set_contact_deleted(cPtr, { (friendList, linphoneFriend) -> Void in
			if (friendList != nil) {
				let sObject = FriendList.getSwiftObject(cObject: friendList!)
				let delegate = sObject.currentDelegate
				delegate?.onContactDeleted(friendList: sObject, linphoneFriend: Friend.getSwiftObject(cObject: linphoneFriend!))
			}
		})

		linphone_friend_list_cbs_set_contact_updated(cPtr, { (friendList, newFriend, oldFriend) -> Void in
			if (friendList != nil) {
				let sObject = FriendList.getSwiftObject(cObject: friendList!)
				let delegate = sObject.currentDelegate
				delegate?.onContactUpdated(friendList: sObject, newFriend: Friend.getSwiftObject(cObject: newFriend!), oldFriend: Friend.getSwiftObject(cObject: oldFriend!))
			}
		})

		linphone_friend_list_cbs_set_sync_status_changed(cPtr, { (friendList, status, message) -> Void in
			if (friendList != nil) {
				let sObject = FriendList.getSwiftObject(cObject: friendList!)
				let delegate = sObject.currentDelegate
				delegate?.onSyncStatusChanged(friendList: sObject, status: FriendList.SyncStatus(rawValue: Int(status.rawValue))!, message: charArrayToString(charPointer: message))
			}
		})

		linphone_friend_list_cbs_set_presence_received(cPtr, { (friendList, friends) -> Void in
			if (friendList != nil) {
				let sObject = FriendList.getSwiftObject(cObject: friendList!)
				let delegate = sObject.currentDelegate
				var friendssList = [Friend]()
				let friendscList = friends
				var listTemp = friendscList
				while (listTemp != nil) {
					let data = unsafeBitCast(listTemp!.pointee.data, to: OpaquePointer.self)
					friendssList.append(Friend.getSwiftObject(cObject: data))
					listTemp = UnsafePointer<bctbx_list_t>(listTemp!.pointee.next)
				}
				delegate?.onPresenceReceived(friendList: sObject, friends: friendssList)
			}
		})

		linphone_friend_list_cbs_set_new_sip_address_discovered(cPtr, { (friendList, linphoneFriend, sipUri) -> Void in
			if (friendList != nil) {
				let sObject = FriendList.getSwiftObject(cObject: friendList!)
				let delegate = sObject.currentDelegate
				delegate?.onNewSipAddressDiscovered(friendList: sObject, linphoneFriend: Friend.getSwiftObject(cObject: linphoneFriend!), sipUri: charArrayToString(charPointer: sipUri))
			}
		})
	}


}

public protocol LoggingServiceDelegate : AnyObject {
	
	
	/// Gets the value of the message event callback. 
	/// - Returns: the current LinphoneLoggingServiceCbsLogMessageWrittenCb 
	func onLogMessageWritten(logService: LoggingService, domain: String, level: LogLevel, message: String)
}

public extension LoggingServiceDelegate {
	
	func onLogMessageWritten(logService: LoggingService, domain: String, level: LogLevel, message: String) {}
}

public final class LoggingServiceDelegateStub : LoggingServiceDelegate
{
	var _onLogMessageWritten: ((LoggingService, String, LogLevel, String) -> Void)?

	
	public func onLogMessageWritten(logService: LoggingService, domain: String, level: LogLevel, message: String){_onLogMessageWritten.map{$0(logService, domain, level, message)}}

	public init (
		onLogMessageWritten: ((LoggingService, String, LogLevel, String) -> Void)? = nil
	) {
		self._onLogMessageWritten = onLogMessageWritten
	}
}

class LoggingServiceDelegateManager
{
	var cPtr: OpaquePointer
	public var delegate: LoggingServiceDelegate?

	public init() {
		cPtr = linphone_factory_create_logging_service_cbs(linphone_factory_get())
		belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passRetained(self).toOpaque()), {
			data in
			if (data != nil) {
				Unmanaged<LoggingServiceDelegateManager>.fromOpaque(data!).release()
			}
		})

		linphone_logging_service_cbs_set_log_message_written(cPtr, { (logService, domain, level, message) -> Void in
			if (logService != nil) {
				let sObject = LoggingService.getSwiftObject(cObject: logService!)
				let delegate = sObject.currentDelegate
				delegate?.onLogMessageWritten(logService: sObject, domain: charArrayToString(charPointer: domain), level: LogLevel(rawValue: Int(level.rawValue)), message: charArrayToString(charPointer: message))
			}
		})
	}


}

public protocol MagicSearchDelegate : AnyObject {
	
	
	/// Get the received results callback. 
	/// - Returns: The current result received callback. 
	func onSearchResultsReceived(magicSearch: MagicSearch)
	
	/// Get the ldap callback on having more results. 
	/// - Returns: The ldap callback on having more results. 
	/// - Deprecated: 18/11/2024 use
	/// linphone_magic_search_cbs_get_more_results_available instead. 
	func onLdapHaveMoreResults(magicSearch: MagicSearch, ldap: Ldap)
	
	/// Gets the callback notifying more results for a source flag are available. 
	/// - Returns: The more results available callback. 
	/// - Deprecated: 07/07/2025 use
	/// linphone_magic_search_cbs_get_results_limit_reached instead. 
	func onMoreResultsAvailable(magicSearch: MagicSearch, source: MagicSearch.Source)
	
	/// Gets the callback notifying the search results limit has been reached, which
	/// means more results are available for currently set query parameters and thus
	/// user should refine it's request. 
	/// - Returns: The results limit reached callback. 
	func onResultsLimitReached(magicSearch: MagicSearch, sourcesFlag: Int)
}

public extension MagicSearchDelegate {
	
	func onSearchResultsReceived(magicSearch: MagicSearch) {}
	
	func onLdapHaveMoreResults(magicSearch: MagicSearch, ldap: Ldap) {}
	
	func onMoreResultsAvailable(magicSearch: MagicSearch, source: MagicSearch.Source) {}
	
	func onResultsLimitReached(magicSearch: MagicSearch, sourcesFlag: Int) {}
}

public final class MagicSearchDelegateStub : MagicSearchDelegate
{
	var _onSearchResultsReceived: ((MagicSearch) -> Void)?
	var _onLdapHaveMoreResults: ((MagicSearch, Ldap) -> Void)?
	var _onMoreResultsAvailable: ((MagicSearch, MagicSearch.Source) -> Void)?
	var _onResultsLimitReached: ((MagicSearch, Int) -> Void)?

	
	public func onSearchResultsReceived(magicSearch: MagicSearch){_onSearchResultsReceived.map{$0(magicSearch)}}
	
	public func onLdapHaveMoreResults(magicSearch: MagicSearch, ldap: Ldap){_onLdapHaveMoreResults.map{$0(magicSearch, ldap)}}
	
	public func onMoreResultsAvailable(magicSearch: MagicSearch, source: MagicSearch.Source){_onMoreResultsAvailable.map{$0(magicSearch, source)}}
	
	public func onResultsLimitReached(magicSearch: MagicSearch, sourcesFlag: Int){_onResultsLimitReached.map{$0(magicSearch, sourcesFlag)}}

	public init (
		onSearchResultsReceived: ((MagicSearch) -> Void)? = nil,
		onLdapHaveMoreResults: ((MagicSearch, Ldap) -> Void)? = nil,
		onMoreResultsAvailable: ((MagicSearch, MagicSearch.Source) -> Void)? = nil,
		onResultsLimitReached: ((MagicSearch, Int) -> Void)? = nil
	) {
		self._onSearchResultsReceived = onSearchResultsReceived
		self._onLdapHaveMoreResults = onLdapHaveMoreResults
		self._onMoreResultsAvailable = onMoreResultsAvailable
		self._onResultsLimitReached = onResultsLimitReached
	}
}

class MagicSearchDelegateManager
{
	var cPtr: OpaquePointer
	public var delegate: MagicSearchDelegate?

	public init() {
		cPtr = linphone_factory_create_magic_search_cbs(linphone_factory_get())
		belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passRetained(self).toOpaque()), {
			data in
			if (data != nil) {
				Unmanaged<MagicSearchDelegateManager>.fromOpaque(data!).release()
			}
		})

		linphone_magic_search_cbs_set_search_results_received(cPtr, { (magicSearch) -> Void in
			if (magicSearch != nil) {
				let sObject = MagicSearch.getSwiftObject(cObject: magicSearch!)
				let delegate = sObject.currentDelegate
				delegate?.onSearchResultsReceived(magicSearch: sObject)
			}
		})

		linphone_magic_search_cbs_set_ldap_have_more_results(cPtr, { (magicSearch, ldap) -> Void in
			if (magicSearch != nil) {
				let sObject = MagicSearch.getSwiftObject(cObject: magicSearch!)
				let delegate = sObject.currentDelegate
				delegate?.onLdapHaveMoreResults(magicSearch: sObject, ldap: Ldap.getSwiftObject(cObject: ldap!))
			}
		})

		linphone_magic_search_cbs_set_more_results_available(cPtr, { (magicSearch, source) -> Void in
			if (magicSearch != nil) {
				let sObject = MagicSearch.getSwiftObject(cObject: magicSearch!)
				let delegate = sObject.currentDelegate
				delegate?.onMoreResultsAvailable(magicSearch: sObject, source: MagicSearch.Source(rawValue: Int(source.rawValue)))
			}
		})

		linphone_magic_search_cbs_set_results_limit_reached(cPtr, { (magicSearch, sourcesFlag) -> Void in
			if (magicSearch != nil) {
				let sObject = MagicSearch.getSwiftObject(cObject: magicSearch!)
				let delegate = sObject.currentDelegate
				delegate?.onResultsLimitReached(magicSearch: sObject, sourcesFlag: Int(sourcesFlag))
			}
		})
	}


}

public protocol ParticipantDeviceDelegate : AnyObject {
	
	
	/// Get the is this participant device speaking changed callback. 
	/// - Returns: The current is this participant device speaking changed callback. 
	func onIsSpeakingChanged(participantDevice: ParticipantDevice, isSpeaking: Bool)
	
	/// Get the is this participant device muted callback. 
	/// - Returns: The current is this participant device muted callback. 
	func onIsMuted(participantDevice: ParticipantDevice, isMuted: Bool)
	
	/// Get the is this participant device screen sharing changed callback. 
	/// - Returns: The current is this participant device screen sharing callback. 
	func onScreenSharingChanged(participantDevice: ParticipantDevice, isScreenSharing: Bool)
	
	/// Get the participant device conference state changed callback. 
	/// - Returns: The current participant device conference state changed callback. 
	func onStateChanged(participantDevice: ParticipantDevice, state: ParticipantDevice.State)
	
	/// Get the participant device stream capability changed callback. 
	/// - Returns: The current participant device stream capability changed callback. 
	func onStreamCapabilityChanged(participantDevice: ParticipantDevice, direction: MediaDirection, streamType: StreamType)
	
	/// Get the participant device thumbnail stream capability changed callback. 
	/// - Returns: The current participant device thumbnail stream capability changed
	/// callback. 
	func onThumbnailStreamCapabilityChanged(participantDevice: ParticipantDevice, direction: MediaDirection)
	
	/// Get the participant device stream availability changed callback. 
	/// - Returns: The current participant device stream availability changed callback. 
	func onStreamAvailabilityChanged(participantDevice: ParticipantDevice, available: Bool, streamType: StreamType)
	
	/// Get the participant device thumbnail stream availability changed callback. 
	/// - Returns: The current participant device thumbnail stream availability changed
	/// callback. 
	func onThumbnailStreamAvailabilityChanged(participantDevice: ParticipantDevice, available: Bool)
	
	/// Get the callback that will be used to notify that there are errors from the
	/// video rendering of the participant device. 
	/// Check LinphoneCallCbsVideoDisplayErrorOccurredCb for more details.
	/// - Returns: The failing video rendering callback. 
	func onVideoDisplayErrorOccurred(participantDevice: ParticipantDevice, errorCode: Int)
}

public extension ParticipantDeviceDelegate {
	
	func onIsSpeakingChanged(participantDevice: ParticipantDevice, isSpeaking: Bool) {}
	
	func onIsMuted(participantDevice: ParticipantDevice, isMuted: Bool) {}
	
	func onScreenSharingChanged(participantDevice: ParticipantDevice, isScreenSharing: Bool) {}
	
	func onStateChanged(participantDevice: ParticipantDevice, state: ParticipantDevice.State) {}
	
	func onStreamCapabilityChanged(participantDevice: ParticipantDevice, direction: MediaDirection, streamType: StreamType) {}
	
	func onThumbnailStreamCapabilityChanged(participantDevice: ParticipantDevice, direction: MediaDirection) {}
	
	func onStreamAvailabilityChanged(participantDevice: ParticipantDevice, available: Bool, streamType: StreamType) {}
	
	func onThumbnailStreamAvailabilityChanged(participantDevice: ParticipantDevice, available: Bool) {}
	
	func onVideoDisplayErrorOccurred(participantDevice: ParticipantDevice, errorCode: Int) {}
}

public final class ParticipantDeviceDelegateStub : ParticipantDeviceDelegate
{
	var _onIsSpeakingChanged: ((ParticipantDevice, Bool) -> Void)?
	var _onIsMuted: ((ParticipantDevice, Bool) -> Void)?
	var _onScreenSharingChanged: ((ParticipantDevice, Bool) -> Void)?
	var _onStateChanged: ((ParticipantDevice, ParticipantDevice.State) -> Void)?
	var _onStreamCapabilityChanged: ((ParticipantDevice, MediaDirection, StreamType) -> Void)?
	var _onThumbnailStreamCapabilityChanged: ((ParticipantDevice, MediaDirection) -> Void)?
	var _onStreamAvailabilityChanged: ((ParticipantDevice, Bool, StreamType) -> Void)?
	var _onThumbnailStreamAvailabilityChanged: ((ParticipantDevice, Bool) -> Void)?
	var _onVideoDisplayErrorOccurred: ((ParticipantDevice, Int) -> Void)?

	
	public func onIsSpeakingChanged(participantDevice: ParticipantDevice, isSpeaking: Bool){_onIsSpeakingChanged.map{$0(participantDevice, isSpeaking)}}
	
	public func onIsMuted(participantDevice: ParticipantDevice, isMuted: Bool){_onIsMuted.map{$0(participantDevice, isMuted)}}
	
	public func onScreenSharingChanged(participantDevice: ParticipantDevice, isScreenSharing: Bool){_onScreenSharingChanged.map{$0(participantDevice, isScreenSharing)}}
	
	public func onStateChanged(participantDevice: ParticipantDevice, state: ParticipantDevice.State){_onStateChanged.map{$0(participantDevice, state)}}
	
	public func onStreamCapabilityChanged(participantDevice: ParticipantDevice, direction: MediaDirection, streamType: StreamType){_onStreamCapabilityChanged.map{$0(participantDevice, direction, streamType)}}
	
	public func onThumbnailStreamCapabilityChanged(participantDevice: ParticipantDevice, direction: MediaDirection){_onThumbnailStreamCapabilityChanged.map{$0(participantDevice, direction)}}
	
	public func onStreamAvailabilityChanged(participantDevice: ParticipantDevice, available: Bool, streamType: StreamType){_onStreamAvailabilityChanged.map{$0(participantDevice, available, streamType)}}
	
	public func onThumbnailStreamAvailabilityChanged(participantDevice: ParticipantDevice, available: Bool){_onThumbnailStreamAvailabilityChanged.map{$0(participantDevice, available)}}
	
	public func onVideoDisplayErrorOccurred(participantDevice: ParticipantDevice, errorCode: Int){_onVideoDisplayErrorOccurred.map{$0(participantDevice, errorCode)}}

	public init (
		onIsSpeakingChanged: ((ParticipantDevice, Bool) -> Void)? = nil,
		onIsMuted: ((ParticipantDevice, Bool) -> Void)? = nil,
		onScreenSharingChanged: ((ParticipantDevice, Bool) -> Void)? = nil,
		onStateChanged: ((ParticipantDevice, ParticipantDevice.State) -> Void)? = nil,
		onStreamCapabilityChanged: ((ParticipantDevice, MediaDirection, StreamType) -> Void)? = nil,
		onThumbnailStreamCapabilityChanged: ((ParticipantDevice, MediaDirection) -> Void)? = nil,
		onStreamAvailabilityChanged: ((ParticipantDevice, Bool, StreamType) -> Void)? = nil,
		onThumbnailStreamAvailabilityChanged: ((ParticipantDevice, Bool) -> Void)? = nil,
		onVideoDisplayErrorOccurred: ((ParticipantDevice, Int) -> Void)? = nil
	) {
		self._onIsSpeakingChanged = onIsSpeakingChanged
		self._onIsMuted = onIsMuted
		self._onScreenSharingChanged = onScreenSharingChanged
		self._onStateChanged = onStateChanged
		self._onStreamCapabilityChanged = onStreamCapabilityChanged
		self._onThumbnailStreamCapabilityChanged = onThumbnailStreamCapabilityChanged
		self._onStreamAvailabilityChanged = onStreamAvailabilityChanged
		self._onThumbnailStreamAvailabilityChanged = onThumbnailStreamAvailabilityChanged
		self._onVideoDisplayErrorOccurred = onVideoDisplayErrorOccurred
	}
}

class ParticipantDeviceDelegateManager
{
	var cPtr: OpaquePointer
	public var delegate: ParticipantDeviceDelegate?

	public init() {
		cPtr = linphone_factory_create_participant_device_cbs(linphone_factory_get())
		belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passRetained(self).toOpaque()), {
			data in
			if (data != nil) {
				Unmanaged<ParticipantDeviceDelegateManager>.fromOpaque(data!).release()
			}
		})

		linphone_participant_device_cbs_set_is_speaking_changed(cPtr, { (participantDevice, isSpeaking) -> Void in
			if (participantDevice != nil) {
				let sObject = ParticipantDevice.getSwiftObject(cObject: participantDevice!)
				let delegate = sObject.currentDelegate
				delegate?.onIsSpeakingChanged(participantDevice: sObject, isSpeaking: isSpeaking != 0)
			}
		})

		linphone_participant_device_cbs_set_is_muted(cPtr, { (participantDevice, isMuted) -> Void in
			if (participantDevice != nil) {
				let sObject = ParticipantDevice.getSwiftObject(cObject: participantDevice!)
				let delegate = sObject.currentDelegate
				delegate?.onIsMuted(participantDevice: sObject, isMuted: isMuted != 0)
			}
		})

		linphone_participant_device_cbs_set_screen_sharing_changed(cPtr, { (participantDevice, isScreenSharing) -> Void in
			if (participantDevice != nil) {
				let sObject = ParticipantDevice.getSwiftObject(cObject: participantDevice!)
				let delegate = sObject.currentDelegate
				delegate?.onScreenSharingChanged(participantDevice: sObject, isScreenSharing: isScreenSharing != 0)
			}
		})

		linphone_participant_device_cbs_set_state_changed(cPtr, { (participantDevice, state) -> Void in
			if (participantDevice != nil) {
				let sObject = ParticipantDevice.getSwiftObject(cObject: participantDevice!)
				let delegate = sObject.currentDelegate
				delegate?.onStateChanged(participantDevice: sObject, state: ParticipantDevice.State(rawValue: Int(state.rawValue))!)
			}
		})

		linphone_participant_device_cbs_set_stream_capability_changed(cPtr, { (participantDevice, direction, streamType) -> Void in
			if (participantDevice != nil) {
				let sObject = ParticipantDevice.getSwiftObject(cObject: participantDevice!)
				let delegate = sObject.currentDelegate
				delegate?.onStreamCapabilityChanged(participantDevice: sObject, direction: MediaDirection(rawValue: Int(direction.rawValue))!, streamType: StreamType(rawValue: Int(streamType.rawValue))!)
			}
		})

		linphone_participant_device_cbs_set_thumbnail_stream_capability_changed(cPtr, { (participantDevice, direction) -> Void in
			if (participantDevice != nil) {
				let sObject = ParticipantDevice.getSwiftObject(cObject: participantDevice!)
				let delegate = sObject.currentDelegate
				delegate?.onThumbnailStreamCapabilityChanged(participantDevice: sObject, direction: MediaDirection(rawValue: Int(direction.rawValue))!)
			}
		})

		linphone_participant_device_cbs_set_stream_availability_changed(cPtr, { (participantDevice, available, streamType) -> Void in
			if (participantDevice != nil) {
				let sObject = ParticipantDevice.getSwiftObject(cObject: participantDevice!)
				let delegate = sObject.currentDelegate
				delegate?.onStreamAvailabilityChanged(participantDevice: sObject, available: available != 0, streamType: StreamType(rawValue: Int(streamType.rawValue))!)
			}
		})

		linphone_participant_device_cbs_set_thumbnail_stream_availability_changed(cPtr, { (participantDevice, available) -> Void in
			if (participantDevice != nil) {
				let sObject = ParticipantDevice.getSwiftObject(cObject: participantDevice!)
				let delegate = sObject.currentDelegate
				delegate?.onThumbnailStreamAvailabilityChanged(participantDevice: sObject, available: available != 0)
			}
		})

		linphone_participant_device_cbs_set_video_display_error_occurred(cPtr, { (participantDevice, errorCode) -> Void in
			if (participantDevice != nil) {
				let sObject = ParticipantDevice.getSwiftObject(cObject: participantDevice!)
				let delegate = sObject.currentDelegate
				delegate?.onVideoDisplayErrorOccurred(participantDevice: sObject, errorCode: Int(errorCode))
			}
		})
	}


}

public protocol PlayerDelegate : AnyObject {
	
	
	/// Get the end-of-file reached callback. 
	/// - Returns: The current end-of-file reached callback. 
	func onEofReached(player: Player)
}

public extension PlayerDelegate {
	
	func onEofReached(player: Player) {}
}

public final class PlayerDelegateStub : PlayerDelegate
{
	var _onEofReached: ((Player) -> Void)?

	
	public func onEofReached(player: Player){_onEofReached.map{$0(player)}}

	public init (
		onEofReached: ((Player) -> Void)? = nil
	) {
		self._onEofReached = onEofReached
	}
}

class PlayerDelegateManager
{
	var cPtr: OpaquePointer
	public var delegate: PlayerDelegate?

	public init() {
		cPtr = linphone_factory_create_player_cbs(linphone_factory_get())
		belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passRetained(self).toOpaque()), {
			data in
			if (data != nil) {
				Unmanaged<PlayerDelegateManager>.fromOpaque(data!).release()
			}
		})

		linphone_player_cbs_set_eof_reached(cPtr, { (player) -> Void in
			if (player != nil) {
				let sObject = Player.getSwiftObject(cObject: player!)
				let delegate = sObject.currentDelegate
				delegate?.onEofReached(player: sObject)
			}
		})
	}


}

public protocol XmlRpcRequestDelegate : AnyObject {
	
	
	/// Get the response callback. 
	/// - Returns: The current response callback. 
	func onResponse(request: XmlRpcRequest)
}

public extension XmlRpcRequestDelegate {
	
	func onResponse(request: XmlRpcRequest) {}
}

public final class XmlRpcRequestDelegateStub : XmlRpcRequestDelegate
{
	var _onResponse: ((XmlRpcRequest) -> Void)?

	
	public func onResponse(request: XmlRpcRequest){_onResponse.map{$0(request)}}

	public init (
		onResponse: ((XmlRpcRequest) -> Void)? = nil
	) {
		self._onResponse = onResponse
	}
}

class XmlRpcRequestDelegateManager
{
	var cPtr: OpaquePointer
	public var delegate: XmlRpcRequestDelegate?

	public init() {
		cPtr = linphone_factory_create_xml_rpc_request_cbs(linphone_factory_get())
		belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passRetained(self).toOpaque()), {
			data in
			if (data != nil) {
				Unmanaged<XmlRpcRequestDelegateManager>.fromOpaque(data!).release()
			}
		})

		linphone_xml_rpc_request_cbs_set_response(cPtr, { (request) -> Void in
			if (request != nil) {
				let sObject = XmlRpcRequest.getSwiftObject(cObject: request!)
				let delegate = sObject.currentDelegate
				delegate?.onResponse(request: sObject)
			}
		})
	}


}



/// Object that represents a Linphone Account. 
/// This object replaces the deprecated ``ProxyConfig``. Use a ``AccountParams``
/// object to configure it. 
public class Account : LinphoneObject
{
	var delegateManagers : [AccountDelegateManager] = []

	static public func getSwiftObject(cObject:OpaquePointer) -> Account {
		let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
		if (result != nil) {
			return Unmanaged<Account>.fromOpaque(result!).takeUnretainedValue()
		}
		let sObject = Account(cPointer: cObject)
		belle_sip_object_data_set(UnsafeMutablePointer(cObject), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(sObject).toOpaque()), nil)
		return sObject
	}

	public var getCobject: OpaquePointer? {
		return cPtr
	}
	
	
	
	public func addDelegate(delegate: AccountDelegate)
	{
		let manager = AccountDelegateManager()
		manager.delegate = delegate
		delegateManagers.append(manager)
		linphone_account_add_callbacks(cPtr, manager.cPtr)
		belle_sip_object_unref(UnsafeMutableRawPointer(manager.cPtr))
		belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(self).toOpaque()), nil)
	}	
	
	
	public func removeDelegate(delegate: AccountDelegate)
	{
		if let index = delegateManagers.firstIndex(where: { $0.delegate === delegate as AnyObject}) {
			linphone_account_remove_callbacks(cPtr, delegateManagers[index].cPtr)
			delegateManagers.remove(at: index)
			belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(self).toOpaque()), nil)
		}
	}	
	
	
	/// Create a new ``Account`` with a Proxy config backpointer. 
	/// This is only intended to be used while keeping a backward compatibility with
	/// proxy config. 
	/// - Parameter lc: The ``Core`` object.    
	/// - Parameter params: The ``AccountParams`` object.    
	/// - Parameter config: The ``ProxyConfig`` object.    
	/// - Returns: The newly created ``Account`` object.    
	static public func newWithConfig(lc:Core?, params:AccountParams, config:ProxyConfig?) -> Account?
	{
		let cPointer = linphone_account_new_with_config(lc?.cPtr, params.cPtr, config?.cPtr)
		if (cPointer == nil) {
			return nil
		}
		let result = Account.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	/// Indicates whether AVPF/SAVPF is being used for calls using this account. 
	/// - Returns: true if AVPF/SAVPF is enabled, false otherwise. 
	public var avpfEnabled: Bool
	{
	
						return linphone_account_avpf_enabled(cPtr) != 0

	}
		
	
	/// Returns the list of call logs for a given account. 
	/// This list must be freed after use. 
	/// - Returns: The list of call logs  .       
	public var callLogs: [CallLog]
	{
	
						var swiftList = [CallLog]()
			let cList = linphone_account_get_call_logs(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(CallLog.getSwiftObject(cObject: data))
				listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
				bctbx_list_free_with_data(cList, belle_sip_object_unref)
			return swiftList

	}
		
	
	/// Returns the list of chat rooms for a given account. 
	/// - Returns: The list of chat rooms  .    
	public var chatRooms: [ChatRoom]
	{
	
						var swiftList = [ChatRoom]()
			let cList = linphone_account_get_chat_rooms(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(ChatRoom.getSwiftObject(cObject: data))
				listTemp = UnsafePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
			return swiftList

	}
		
	
	/// Returns the list of conference information stored locally for a given account. 
	/// This list must be freed after use. 
	/// - Returns: The list of call logs  .    
	/// - Warning: This method also start the synchronization with the CCMP server,
	/// should it be defined in the ``AccountParams``. The application may want to wait
	/// for the callback conference_information_updated to get an up-to-date list of
	/// conferences 
	public var conferenceInformationList: [ConferenceInfo]
	{
	
						var swiftList = [ConferenceInfo]()
			let cList = linphone_account_get_conference_information_list(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(ConferenceInfo.getSwiftObject(cObject: data))
				listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
				bctbx_list_free(cList)
			return swiftList

	}
		
	/// Set the contact address for the account. 
	/// A client application should not use this function, as the Contact address is
	/// provided by the registrar in the 200 Ok. This function is mainly intended for
	/// server applications. 
	/// - Parameter addr: a ``Address`` to use as contact.    
	
	/// Return the contact address of the account. 
	/// - Returns: a ``Address`` correspong to the contact address of the account.    
	public var contactAddress: Address?
	{
	
		get
		{ 
						let cPointer = linphone_account_get_contact_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_account_set_contact_address(cPtr, newValue?.cPtr)
		}
	}
		
	
	/// Get the ``Core`` object to which is associated the ``Account``. 
	/// - Returns: The ``Core`` object to which is associated the ``Account``.    
	public var core: Core?
	{
	
						let cPointer = linphone_account_get_core(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Core.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Gets the current ``AccountDelegate``. 
	/// This is meant only to be called from a callback to be able to get the user_data
	/// associated with the ``AccountDelegate`` that is calling the callback. 
	/// - Returns: The ``AccountDelegate`` that has called the last callback.    
	public var currentDelegate: AccountDelegate?
	{
	
			let cObject = linphone_account_get_current_callbacks(cPtr)
			let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
			if (result != nil) {
				return Unmanaged<AccountDelegateManager>.fromOpaque(result!).takeUnretainedValue().delegate
				}
			return nil
			
	}
		
	/// Mark this account as being dependent on the given one. 
	/// The dependency must refer to an account previously added to the core and which
	/// idkey property is defined.
	/// - See also: ``AccountParams/setIdkey(idkey:)``
	/// The account marked as dependent will wait for successful registration on its
	/// dependency before triggering its own.
	/// Once registered, both accounts will share the same contact address (the
	/// 'dependency' one).
	/// This mecanism must be enabled before the account is added to the core
	/// - Parameter dependsOn: The ``Account`` this one shall be depending on.    
	
	/// Get the dependency of a ``Account``. 
	/// - Returns: The account this one is dependent upon, or nil if not marked
	/// dependent.    
	public var dependency: Account?
	{
	
		get
		{ 
						let cPointer = linphone_account_get_dependency(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Account.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_account_set_dependency(cPtr, newValue?.cPtr)
		}
	}
		
	
	/// Get the reason why registration failed when the account state is
	/// LinphoneRegistrationFailed. 
	/// - Returns: The ``Reason`` why registration failed for this account. 
	public var error: Reason
	{
	
						return Reason(rawValue: Int(linphone_account_get_error(cPtr).rawValue))!

	}
		
	
	/// Get detailed information why registration failed when the account state is
	/// LinphoneRegistrationFailed. 
	/// - Returns: The ``ErrorInfo`` explaining why registration failed for this
	/// account.    
	public var errorInfo: ErrorInfo?
	{
	
						let cPointer = linphone_account_get_error_info(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = ErrorInfo.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Indicates whether AVPF/SAVPF is being used for calls using this account. 
	/// - Returns: true if AVPF/SAVPF is enabled, false otherwise. 
	/// - Deprecated: 16/12/2021 Use ``avpfEnabled()`` instead. 
	@available(*, deprecated)
	public var isAvpfEnabled: Bool
	{
	
						return linphone_account_is_avpf_enabled(cPtr) != 0

	}
		
	
	/// Returns the missed calls count for a given account. 
	/// - Returns: The missed calls count. 
	public var missedCallsCount: Int
	{
	
						return Int(linphone_account_get_missed_calls_count(cPtr))

	}
		
	/// Set the ``AccountParams`` used by this ``Account``. 
	/// - Parameter params: The ``AccountParams`` object.    
	
	/// Get the ``AccountParams`` as read-only object. 
	/// To make changes, clone the returned object using ``AccountParams/clone()``
	/// method, make your changes on it and apply them using with
	/// ``setParams(params:)``. 
	/// - Returns: The ``AccountParams`` attached to this account.    
	public var params: AccountParams?
	{
	
		get
		{ 
						let cPointer = linphone_account_get_params(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = AccountParams.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_account_set_params(cPtr, newValue?.cPtr)
		}
	}
		
	
	/// Get the registration state of the given account. 
	/// - Returns: The ``RegistrationState`` of the account. 
	public var state: RegistrationState
	{
	
						return RegistrationState(rawValue: Int(linphone_account_get_state(cPtr).rawValue))!

	}
		
	
	/// Get the transport from either service route, route or addr. 
	/// - Returns: The transport as a string (I.E udp, tcp, tls, dtls). 
	/// - Deprecated: 01/03/2021 Use Linphone_account_params_get_transport() instead. 
	@available(*, deprecated)
	public var transport: TransportType
	{
	
						return TransportType(rawValue: Int(linphone_account_get_transport(cPtr).rawValue))!

	}
		
	
	/// Returns the unread chat message count for a given account. 
	/// - Returns: The unread chat message count. 
	public var unreadChatMessageCount: Int
	{
	
						return Int(linphone_account_get_unread_chat_message_count(cPtr))

	}
		
	/// Assign a user pointer to the account. 
	/// - Parameter userData: The user data.    
	
	/// Retrieve the user pointer associated with the account. 
	/// - Returns: The user data.    
	public var userData: UnsafeMutableRawPointer?
	{
	
		get
		{ 
						return linphone_account_get_user_data(cPtr)
		}
		set
		{
			linphone_account_set_user_data(cPtr, newValue)
		}
	}
		
	
	
	/// Set one custom parameter to this ``Account``. 
	/// - Parameter key: key of the searched parameter.    
	/// - Parameter value: value of the searched parameter.    
	public func addCustomParam(key:String, value:String) 
	{
		linphone_account_add_custom_param(cPtr, key, value)
	}
	
	
	
	/// Deletes all the call logs related to this account from the database. 
	public func clearCallLogs() 
	{
		linphone_account_clear_call_logs(cPtr)
	}
	
	
	
	/// Instantiate a new account with values from source. 
	/// - Returns: The newly created ``Account`` object.    
	public func clone() -> Account?
	{
		let cPointer = linphone_account_clone(cPtr)
		if (cPointer == nil) {
			return nil
		}
		let result = Account.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Returns a filtered list of chat rooms for a given account. 
	/// - Parameter filter: the criteria a chat room must meet somehow (subject,
	/// participant address, friend's name).    
	/// - Returns: The list of chat rooms  .       
	public func filterChatRooms(filter:String) -> [ChatRoom]
	{
		var swiftList = [ChatRoom]()
		let cList = linphone_account_filter_chat_rooms(cPtr, filter)
		var listTemp = cList
		while (listTemp != nil) {
			let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
			swiftList.append(ChatRoom.getSwiftObject(cObject: data))
			listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
		}
			bctbx_list_free_with_data(cList, belle_sip_object_unref)
		return swiftList
	}
	
	
	
	/// Find authentication info matching account, if any, similarly to
	/// linphone_core_find_auth_info. 
	/// - Returns: a ``AuthInfo`` matching account criteria if possible, nil if nothing
	/// can be found.    
	public func findAuthInfo() -> AuthInfo?
	{
		let cPointer = linphone_account_find_auth_info(cPtr)
		if (cPointer == nil) {
			return nil
		}
		let result = AuthInfo.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Returns the list of call logs for a given account. 
	/// This list must be freed after use. 
	/// - Parameter remoteAddress: the ``Address`` object to filter call logs.    
	/// - Returns: The list of filtered call logs  .       
	public func getCallLogsForAddress(remoteAddress:Address) -> [CallLog]
	{
		var swiftList = [CallLog]()
		let cList = linphone_account_get_call_logs_for_address(cPtr, remoteAddress.cPtr)
		var listTemp = cList
		while (listTemp != nil) {
			let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
			swiftList.append(CallLog.getSwiftObject(cObject: data))
			listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
		}
			bctbx_list_free_with_data(cList, belle_sip_object_unref)
		return swiftList
	}
	
	
	
	/// Obtain the value of a header sent by the server in last answer to REGISTER. 
	/// - Parameter headerName: The header name for which to fetch corresponding value.
	///    
	/// - Returns: The value of the queried header.    
	public func getCustomHeader(headerName:String) -> String
	{
		let cstr = linphone_account_get_custom_header(cPtr, headerName)
		let result = charArrayToString(charPointer: cstr)
		return result
	}
	
	
	
	/// Get the custom parameter with key to this ``Account``. 
	/// - Parameter key: key of the searched parameter.    
	/// - Returns: The value of the parameter with key if found or an empty string
	/// otherwise.    
	public func getCustomParam(key:String) -> String
	{
		let cstr = linphone_account_get_custom_param(cPtr, key)
		let result = charArrayToString(charPointer: cstr)
		return result
	}
	
	
	
	/// Detect if the given input is a phone number or not. 
	/// - Parameter username: The string to parse.    
	/// - Returns: true if input is a phone number, false otherwise. 
	public func isPhoneNumber(username:String) -> Bool
	{
		return linphone_account_is_phone_number(cPtr, username) != 0
	}
	
	
	
	/// Normalize a human readable phone number into a basic string. 
	/// 888-444-222 becomes 888444222 or +33888444222 depending on the ``Account``
	/// object. This function will always generate a normalized username if input is a
	/// phone number. 
	/// - Parameter username: The string to parse.    
	/// - Returns: nil if input is an invalid phone number, normalized phone number
	/// from username input otherwise.       
	public func normalizePhoneNumber(username:String) -> String
	{
		let cstr = linphone_account_normalize_phone_number(cPtr, username)
		let result = charArrayToString(charPointer: cstr)
		if (cstr != nil) {
			bctbx_free(cstr)
		}
		return result
	}
	
	
	
	/// Normalize a human readable sip uri into a fully qualified ``Address``. 
	/// A sip address should look like DisplayName <sip:username@domain:port> .
	/// Basically this function performs the following tasks
	/// -if a phone number is entered, prepend country prefix and eventually escape the
	/// '+' by 00 of the proxy config.
	/// -if no domain part is supplied, append the domain name of the proxy config.
	/// Returns nil if no proxy is provided at this point.
	/// -if no sip: is present, prepend it.
	/// The result is a syntactically correct SIP address. 
	/// - Parameter username: The string to parse.    
	/// - Returns: nil if invalid input, normalized sip address otherwise.       
	public func normalizeSipUri(username:String) -> Address?
	{
		let cPointer = linphone_account_normalize_sip_uri(cPtr, username)
		if (cPointer == nil) {
			return nil
		}
		let result = Address.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Prevent an account from refreshing its registration. 
	/// This is useful to let registrations to expire naturally (or) when the
	/// application wants to keep control on when refreshes are sent. However,
	/// linphone_core_set_network_reachable(lc,true) will always request the accounts
	/// to refresh their registrations. The refreshing operations can be resumed with
	/// ``refreshRegister()``. 
	public func pauseRegister() 
	{
		linphone_account_pause_register(cPtr)
	}
	
	
	
	/// Refresh a proxy registration. 
	/// This is useful if for example you resuming from suspend, thus IP address may
	/// have changed. 
	public func refreshRegister() 
	{
		linphone_account_refresh_register(cPtr)
	}
	
	
	
	/// Re-sets the number of missed calls for this account to 0. 
	public func resetMissedCallsCount() 
	{
		linphone_account_reset_missed_calls_count(cPtr)
	}
	
	
	
	/// Set the value of a custom header sent to the server in REGISTERs request. 
	/// - Parameter headerName: The header name.    
	/// - Parameter headerValue: The header value.    
	public func setCustomHeader(headerName:String, headerValue:String?) 
	{
		linphone_account_set_custom_header(cPtr, headerName, headerValue)
	}
}


/// The object used to configure an account on a server via XML-RPC See: Override
/// account creator request 
/// The object used to configure an account on a server via XML-RPC See: Override
/// account creator request 
public class AccountCreator : LinphoneObject
{
	var delegateManagers : [AccountCreatorDelegateManager] = []

	static public func getSwiftObject(cObject:OpaquePointer) -> AccountCreator {
		let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
		if (result != nil) {
			return Unmanaged<AccountCreator>.fromOpaque(result!).takeUnretainedValue()
		}
		let sObject = AccountCreator(cPointer: cObject)
		belle_sip_object_data_set(UnsafeMutablePointer(cObject), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(sObject).toOpaque()), nil)
		return sObject
	}

	public var getCobject: OpaquePointer? {
		return cPtr
	}

	
	///Enum describing the status of server request, used by the ``AccountCreator``. 
	public enum Status:Int
	{
		
		/// Request status. 
		case RequestOk = 0
		/// Request failed. 
		case RequestFailed = 1
		/// Request failed due to missing argument(s) 
		case MissingArguments = 2
		/// Request failed due to missing callback(s) 
		case MissingCallbacks = 3
		/// Account status. 
		case AccountCreated = 4
		/// Account not created. 
		case AccountNotCreated = 5
		/// Account exist. 
		case AccountExist = 6
		/// Account exist with alias. 
		case AccountExistWithAlias = 7
		/// Account not exist. 
		case AccountNotExist = 8
		/// Account was created with Alias. 
		case AliasIsAccount = 9
		/// Alias exist. 
		case AliasExist = 10
		/// Alias not exist. 
		case AliasNotExist = 11
		/// Account activated. 
		case AccountActivated = 12
		/// Account already activated. 
		case AccountAlreadyActivated = 13
		/// Account not activated. 
		case AccountNotActivated = 14
		/// Account linked. 
		case AccountLinked = 15
		/// Account not linked. 
		case AccountNotLinked = 16
		/// Server. 
		case ServerError = 17
		/// Error cannot send SMS. 
		case PhoneNumberInvalid = 18
		/// Error key doesn't match. 
		case WrongActivationCode = 19
		/// Error too many SMS sent. 
		case PhoneNumberOverused = 20
		/// Error algo isn't MD5 or SHA-256. 
		case AlgoNotSupported = 21
		/// Generic error. 
		case UnexpectedError = 22
		/// This API isn't implemented in the current backend. 
		case NotImplementedError = 23
		/// Request has been denied, probably due to invalid auth token. 
		case RequestNotAuthorized = 24
		/// Request has been denied, due to too many requests sent in given period. 
		case RequestTooManyRequests = 25
	}

	
	///Enum describing transport checking, used by the ``AccountCreator``. 
	public enum TransportStatus:Int
	{
		
		/// Transport ok. 
		case Ok = 0
		/// Transport invalid. 
		case Unsupported = 1
	}

	
	///Enum describing domain checking, used by the ``AccountCreator``. 
	public enum DomainStatus:Int
	{
		
		/// Domain ok. 
		case Ok = 0
		/// Domain invalid. 
		case Invalid = 1
	}

	
	///Enum describing activation code checking, used by the ``AccountCreator``. 
	public enum ActivationCodeStatus:Int
	{
		
		/// Activation code ok. 
		case Ok = 0
		/// Activation code too short. 
		case TooShort = 1
		/// Activation code too long. 
		case TooLong = 2
		/// Contain invalid characters. 
		case InvalidCharacters = 3
	}

	
	///Enum describing algorithm checking, used by the ``AccountCreator``. 
	public enum AlgoStatus:Int
	{
		
		/// Algorithm ok. 
		case Ok = 0
		/// Algorithm not supported. 
		case NotSupported = 1
	}

	
	///Enum describing language checking, used by the ``AccountCreator``. 
	public enum LanguageStatus:Int
	{
		
		/// Language ok. 
		case Ok = 0
	}

	
	///Enum describing password checking, used by the ``AccountCreator``. 
	public enum PasswordStatus:Int
	{
		
		/// Password ok. 
		case Ok = 0
		/// Password too short. 
		case TooShort = 1
		/// Password too long. 
		case TooLong = 2
		/// Contain invalid characters. 
		case InvalidCharacters = 3
		/// Missing specific characters. 
		case MissingCharacters = 4
	}

	
	///Enum describing email checking, used by the ``AccountCreator``. 
	public enum EmailStatus:Int
	{
		
		/// Email ok. 
		case Ok = 0
		/// Email malformed. 
		case Malformed = 1
		/// Contain invalid characters. 
		case InvalidCharacters = 2
	}

	
	///Enum describing username checking, used by the ``AccountCreator``. 
	public enum UsernameStatus:Int
	{
		
		/// Username ok. 
		case Ok = 0
		/// Username too short. 
		case TooShort = 1
		/// Username too long. 
		case TooLong = 2
		/// Contain invalid characters. 
		case InvalidCharacters = 3
		/// Invalid username. 
		case Invalid = 4
	}

	
	///Enum describing backend used in the ``AccountCreator``. 
	public enum Backend:Int
	{
		
		/// XMLRPC Backend. 
		case XMLRPC = 0
		/// FlexiAPI Backend. 
		case FlexiAPI = 1
	}

	
	///Enum describing phone number checking, used by the ``AccountCreator``. 
	public enum PhoneNumberStatus:Int
	{
		
		/// Phone number ok. 
		case Ok = 1
		/// Phone number too short. 
		case TooShort = 2
		/// Phone number too long. 
		case TooLong = 4
		/// Country code invalid. 
		case InvalidCountryCode = 8
		/// Phone number invalid. 
		case Invalid = 16
	}
	
	
	
	public func addDelegate(delegate: AccountCreatorDelegate)
	{
		let manager = AccountCreatorDelegateManager()
		manager.delegate = delegate
		delegateManagers.append(manager)
		linphone_account_creator_add_callbacks(cPtr, manager.cPtr)
		belle_sip_object_unref(UnsafeMutableRawPointer(manager.cPtr))
		belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(self).toOpaque()), nil)
	}	
	
	
	public func removeDelegate(delegate: AccountCreatorDelegate)
	{
		if let index = delegateManagers.firstIndex(where: { $0.delegate === delegate as AnyObject}) {
			linphone_account_creator_remove_callbacks(cPtr, delegateManagers[index].cPtr)
			delegateManagers.remove(at: index)
			belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(self).toOpaque()), nil)
		}
	}	
	
	
	/// Create a ``AccountCreator`` and set Linphone Request callbacks. 
	/// - Parameter core: The ``Core`` used for the XML-RPC communication    
	/// - Returns: The new ``AccountCreator`` object.    
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	@available(*, deprecated)
	static public func create(core:Core) throws -> AccountCreator
	{
		let cPointer = linphone_account_creator_create(core.cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null AccountCreator value")
		}
		let result = AccountCreator.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	/// Assign a proxy config pointer to the ``AccountCreator``. 
	/// - Parameter account: The ``Account`` to associate with the ``AccountCreator``. 
	///   
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	
	public var account: Account?
	{
	
	@available(*, deprecated)
		willSet
		{
			linphone_account_creator_set_account(cPtr, newValue?.cPtr)
		}
	}
		
	/// Set the account creation request token received to be used to check user
	/// validation. 
	/// - Parameter token: The token to set    
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	
	/// Get the account creation request token received to be used to check user
	/// validation. 
	/// - Returns: The token set, if any    
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	public var accountCreationRequestToken: String?
	{
	@available(*, deprecated)
		get
		{ 
			
			let cPointer = linphone_account_creator_get_account_creation_request_token(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
	@available(*, deprecated)
		set
		{
			linphone_account_creator_set_account_creation_request_token(cPtr, newValue)
		}
	}
		
	/// Set the activation code. 
	/// - Parameter activationCode: The activation code to set    
	/// - Returns: ``Ok`` if everything is OK, or a specific error otherwise. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	
	/// Get the activation code. 
	/// - Returns: The activation code of the ``AccountCreator``.    
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	public var activationCode: String?
	{
	@available(*, deprecated)
		get
		{ 
			
			let cPointer = linphone_account_creator_get_activation_code(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
	@available(*, deprecated)
		set
		{
			linphone_account_creator_set_activation_code(cPtr, newValue)
		}
	}
		
	/// Set the supported algorithm. 
	/// - Parameter algorithm: The algorithm to use    
	/// - Returns: LinphoneAccountCreatorAlgoStatusOk if everything is OK, or a
	/// specific error otherwise. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	
	/// Get the algorithm configured in the account creator. 
	/// - Returns: The algorithm of the ``AccountCreator``.    
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	public var algorithm: String?
	{
	@available(*, deprecated)
		get
		{ 
			
			let cPointer = linphone_account_creator_get_algorithm(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
	@available(*, deprecated)
		set
		{
			linphone_account_creator_set_algorithm(cPtr, newValue)
		}
	}
		
	/// Set the set_as_default property. 
	/// - Parameter setAsDefault: true for the created proxy config to be set as
	/// default in ``Core``, false otherwise 
	/// - Returns: ``RequestOk`` if everything is OK, or a specific error otherwise. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	
	public var asDefault: Bool?
	{
	
	@available(*, deprecated)
		willSet
		{
			linphone_account_creator_set_as_default(cPtr, newValue==true ? 1:0)
		}
	}
		
	
	/// Get the current ``AccountCreatorDelegate`` object associated with a
	/// ``AccountCreator``. 
	/// - Returns: The current ``AccountCreatorDelegate`` object associated with the
	/// ``AccountCreator``.    
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	@available(*, deprecated)
	public var currentDelegate: AccountCreatorDelegate?
	{
	
			let cObject = linphone_account_creator_get_current_callbacks(cPtr)
			let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
			if (result != nil) {
				return Unmanaged<AccountCreatorDelegateManager>.fromOpaque(result!).takeUnretainedValue().delegate
				}
			return nil
			
	}
		
	/// Set the display name. 
	/// - Parameter displayName: The display name to set    
	/// - Returns: ``Ok`` if everything is OK, or a specific error otherwise. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	
	/// Get the display name. 
	/// - Returns: The display name of the ``AccountCreator``.    
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	public var displayName: String?
	{
	@available(*, deprecated)
		get
		{ 
			
			let cPointer = linphone_account_creator_get_display_name(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
	@available(*, deprecated)
		set
		{
			linphone_account_creator_set_display_name(cPtr, newValue)
		}
	}
		
	/// Set the domain. 
	/// - Parameter domain: The domain to set    
	/// - Returns: ``Ok`` if everything is OK, or a specific error otherwise. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	
	/// Get the domain. 
	/// - Returns: The domain of the ``AccountCreator``.    
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	public var domain: String?
	{
	@available(*, deprecated)
		get
		{ 
			
			let cPointer = linphone_account_creator_get_domain(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
	@available(*, deprecated)
		set
		{
			linphone_account_creator_set_domain(cPtr, newValue)
		}
	}
		
	/// Set the email. 
	/// - Parameter email: The email to set    
	/// - Returns: ``Ok`` if everything is OK, or a specific error otherwise. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	
	/// Get the email. 
	/// - Returns: The email of the ``AccountCreator``.    
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	public var email: String?
	{
	@available(*, deprecated)
		get
		{ 
			
			let cPointer = linphone_account_creator_get_email(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
	@available(*, deprecated)
		set
		{
			linphone_account_creator_set_email(cPtr, newValue)
		}
	}
		
	/// Set the ha1. 
	/// - Parameter ha1: The ha1 to set    
	/// - Returns: ``Ok`` if everything is OK, or a specific error otherwise. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	
	/// Get the ha1. 
	/// - Returns: The ha1 of the ``AccountCreator``.    
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	public var ha1: String?
	{
	@available(*, deprecated)
		get
		{ 
			
			let cPointer = linphone_account_creator_get_ha1(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
	@available(*, deprecated)
		set
		{
			linphone_account_creator_set_ha1(cPtr, newValue)
		}
	}
		
	/// Set the language to use in email or SMS if supported. 
	/// - Parameter lang: The language to use    
	/// - Returns: ``Ok`` if everything is OK, or a specific error otherwise. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	
	/// Get the language use in email of SMS. 
	/// - Returns: The language of the ``AccountCreator``.    
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	public var language: String?
	{
	@available(*, deprecated)
		get
		{ 
			
			let cPointer = linphone_account_creator_get_language(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
	@available(*, deprecated)
		set
		{
			linphone_account_creator_set_language(cPtr, newValue)
		}
	}
		
	/// Set the password. 
	/// - Parameter password: The password to set    
	/// - Returns: ``Ok`` if everything is OK, or specific(s) error(s) otherwise. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	
	/// Get the password. 
	/// - Returns: The password of the ``AccountCreator``.    
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	public var password: String?
	{
	@available(*, deprecated)
		get
		{ 
			
			let cPointer = linphone_account_creator_get_password(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
	@available(*, deprecated)
		set
		{
			linphone_account_creator_set_password(cPtr, newValue)
		}
	}
		
	
	/// Get the international prefix. 
	/// - Returns: The international prefix (or phone country code) of the
	/// ``AccountCreator``.    
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	@available(*, deprecated)
	public var phoneCountryCode: String?
	{
	
			
			let cPointer = linphone_account_creator_get_phone_country_code(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Get the RFC 3966 normalized phone number. 
	/// - Returns: The phone number of the ``AccountCreator``.    
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	@available(*, deprecated)
	public var phoneNumber: String?
	{
	
			
			let cPointer = linphone_account_creator_get_phone_number(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	/// Set the param to be used by the backend to send the push notification to the
	/// device asking for an auth token. 
	/// - Parameter pnParam: The pn_param to set    
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	
	/// Get the param to be used by the backend to send the push notification to the
	/// device asking for an auth token. 
	/// - Returns: The pn_param set, if any    
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	public var pnParam: String?
	{
	@available(*, deprecated)
		get
		{ 
			
			let cPointer = linphone_account_creator_get_pn_param(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
	@available(*, deprecated)
		set
		{
			linphone_account_creator_set_pn_param(cPtr, newValue)
		}
	}
		
	/// Set the prid to be used by the backend to send the push notification to the
	/// device asking for an auth token. 
	/// - Parameter pnPrid: The pn_prid to set    
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	
	/// Get the prid to be used by the backend to send the push notification to the
	/// device asking for an auth token. 
	/// - Returns: The pn_prid set, if any    
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	public var pnPrid: String?
	{
	@available(*, deprecated)
		get
		{ 
			
			let cPointer = linphone_account_creator_get_pn_prid(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
	@available(*, deprecated)
		set
		{
			linphone_account_creator_set_pn_prid(cPtr, newValue)
		}
	}
		
	/// Set the provider to be used by the backend to send the push notification to the
	/// device asking for an auth token. 
	/// - Parameter pnProvider: The pn_provider to set    
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	
	/// Get the provider to be used by the backend to send the push notification to the
	/// device asking for an auth token. 
	/// - Returns: The pn_provider set, if any    
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	public var pnProvider: String?
	{
	@available(*, deprecated)
		get
		{ 
			
			let cPointer = linphone_account_creator_get_pn_provider(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
	@available(*, deprecated)
		set
		{
			linphone_account_creator_set_pn_provider(cPtr, newValue)
		}
	}
		
	/// Assign a proxy config pointer to the ``AccountCreator``. 
	/// - Parameter cfg: The ``ProxyConfig`` to associate with the ``AccountCreator``. 
	///   
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	
	public var proxyConfig: ProxyConfig?
	{
	
	@available(*, deprecated)
		willSet
		{
			linphone_account_creator_set_proxy_config(cPtr, newValue?.cPtr)
		}
	}
		
	
	/// Get the set_as_default property. 
	/// - Returns: true if account will be set as default, false otherwise. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	@available(*, deprecated)
	public var setAsDefault: Bool
	{
	
						return linphone_account_creator_get_set_as_default(cPtr) != 0

	}
		
	/// Set the authentication token received by push notification to be used to
	/// authenticate next queries, if required. 
	/// - Parameter token: The token to set    
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	
	/// Get the authentication token set (if any) to be used to authenticate next
	/// queries, if required. 
	/// - Returns: The token set, if any    
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	public var token: String?
	{
	@available(*, deprecated)
		get
		{ 
			
			let cPointer = linphone_account_creator_get_token(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
	@available(*, deprecated)
		set
		{
			linphone_account_creator_set_token(cPtr, newValue)
		}
	}
		
	/// Set Transport. 
	/// - Parameter transport: The ``TransportType`` to set 
	/// - Returns: ``Ok`` if everything is OK, or a specific error otherwise. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	
	/// Get Transport. 
	/// - Returns: The ``TransportType`` of the creator. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	public var transport: TransportType
	{
	@available(*, deprecated)
		get
		{ 
						return TransportType(rawValue: Int(linphone_account_creator_get_transport(cPtr).rawValue))!
		}
	@available(*, deprecated)
		set
		{
			linphone_account_creator_set_transport(cPtr, LinphoneTransportType(rawValue: CUnsignedInt(newValue.rawValue)))
		}
	}
		
	/// Assign a user pointer to the ``AccountCreator``. 
	/// - Parameter userData: The user pointer to associate with the
	/// ``AccountCreator``.    
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	
	/// Retrieve the user pointer associated with the ``AccountCreator``. 
	/// - Returns: The user pointer associated with the ``AccountCreator``.    
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	public var userData: UnsafeMutableRawPointer?
	{
	@available(*, deprecated)
		get
		{ 
						return linphone_account_creator_get_user_data(cPtr)
		}
	@available(*, deprecated)
		set
		{
			linphone_account_creator_set_user_data(cPtr, newValue)
		}
	}
		
	/// Set the username. 
	/// - Parameter username: The username to set    
	/// - Returns: ``Ok`` if everything is OK, or a specific error otherwise. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	
	/// Get the username. 
	/// - Returns: The username of the ``AccountCreator``.    
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	public var username: String?
	{
	@available(*, deprecated)
		get
		{ 
			
			let cPointer = linphone_account_creator_get_username(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
	@available(*, deprecated)
		set
		{
			linphone_account_creator_set_username(cPtr, newValue)
		}
	}
		
	
	
	/// Send a request to activate an account on server. 
	/// - Returns: ``RequestOk`` if the request has been sent, ``RequestFailed``
	/// otherwise 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	@available(*, deprecated)
	public func activateAccount() -> AccountCreator.Status
	{
		return AccountCreator.Status(rawValue: Int(linphone_account_creator_activate_account(cPtr).rawValue))!
	}
	
	
	
	/// Send a request to activate an alias. 
	/// - Returns: ``RequestOk`` if the request has been sent, ``RequestFailed``
	/// otherwise 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	@available(*, deprecated)
	public func activateAlias() -> AccountCreator.Status
	{
		return AccountCreator.Status(rawValue: Int(linphone_account_creator_activate_alias(cPtr).rawValue))!
	}
	
	
	
	/// Send a request to create an account on server. 
	/// - Returns: ``RequestOk`` if the request has been sent, ``RequestFailed``
	/// otherwise 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	@available(*, deprecated)
	public func createAccount() throws -> AccountCreator.Status
	{
		return AccountCreator.Status(rawValue: Int(linphone_account_creator_create_account(cPtr).rawValue))!
	}
	
	
	
	/// Create and configure a ``Account`` and a ``AuthInfo`` from informations set in
	/// the ``AccountCreator``. 
	/// - Returns: A ``Account`` object if successful, nil otherwise.       
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	@available(*, deprecated)
	public func createAccountInCore() throws -> Account
	{
		let cPointer = linphone_account_creator_create_account_in_core(cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null Account value")
		}
		let result = Account.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Create and configure a proxy config and a authentication info for an account
	/// creator. 
	/// - Returns: A ``ProxyConfig`` object if successful, nil otherwise.    
	/// - Deprecated: 05/05/2023 Use ``createAccountInCore()`` instead. 
	@available(*, deprecated)
	public func createProxyConfig() throws -> ProxyConfig
	{
		let cPointer = linphone_account_creator_create_proxy_config(cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null ProxyConfig value")
		}
		let result = ProxyConfig.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Send a request to create a push account on server. 
	/// Push accounts are used in account dependent situation when account cannot send
	/// push notifications. A username and password are automatically generated, an
	/// account is automatically activated. 
	/// - Returns: ``RequestOk`` if the request has been sent, ``RequestFailed``
	/// otherwise 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	@available(*, deprecated)
	public func createPushAccount() throws -> AccountCreator.Status
	{
		return AccountCreator.Status(rawValue: Int(linphone_account_creator_create_push_account(cPtr).rawValue))!
	}
	
	
	
	/// Send a request to know if an account is activated on server. 
	/// - Returns: ``RequestOk`` if the request has been sent, ``RequestFailed``
	/// otherwise 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	@available(*, deprecated)
	public func isAccountActivated() -> AccountCreator.Status
	{
		return AccountCreator.Status(rawValue: Int(linphone_account_creator_is_account_activated(cPtr).rawValue))!
	}
	
	
	
	/// Send a request to know the existence of account on server. 
	/// - Returns: ``RequestOk`` if the request has been sent, ``RequestFailed``
	/// otherwise 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	@available(*, deprecated)
	public func isAccountExist() -> AccountCreator.Status
	{
		return AccountCreator.Status(rawValue: Int(linphone_account_creator_is_account_exist(cPtr).rawValue))!
	}
	
	
	
	/// Send a request to know if an account is linked. 
	/// - Returns: ``RequestOk`` if the request has been sent, ``RequestFailed``
	/// otherwise 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	@available(*, deprecated)
	public func isAccountLinked() -> AccountCreator.Status
	{
		return AccountCreator.Status(rawValue: Int(linphone_account_creator_is_account_linked(cPtr).rawValue))!
	}
	
	
	
	/// Send a request to know if an alias is used. 
	/// - Returns: ``RequestOk`` if the request has been sent, ``RequestFailed``
	/// otherwise 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	@available(*, deprecated)
	public func isAliasUsed() -> AccountCreator.Status
	{
		return AccountCreator.Status(rawValue: Int(linphone_account_creator_is_alias_used(cPtr).rawValue))!
	}
	
	
	
	/// Send a request to link an account to an alias. 
	/// - Returns: ``RequestOk`` if the request has been sent, ``RequestFailed``
	/// otherwise 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	@available(*, deprecated)
	public func linkAccount() -> AccountCreator.Status
	{
		return AccountCreator.Status(rawValue: Int(linphone_account_creator_link_account(cPtr).rawValue))!
	}
	
	
	
	/// Send a request to get the password & algorithm of an account using the
	/// confirmation key. 
	/// - Returns: ``RequestOk`` if everything is OK, or a specific error otherwise. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	@available(*, deprecated)
	public func loginLinphoneAccount() -> AccountCreator.Status
	{
		return AccountCreator.Status(rawValue: Int(linphone_account_creator_login_linphone_account(cPtr).rawValue))!
	}
	
	
	
	/// Send a request to recover an account. 
	/// - Returns: ``RequestOk`` if the request has been sent, ``RequestFailed``
	/// otherwise 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	@available(*, deprecated)
	public func recoverAccount() -> AccountCreator.Status
	{
		return AccountCreator.Status(rawValue: Int(linphone_account_creator_recover_account(cPtr).rawValue))!
	}
	
	
	
	/// Request an account creation "request_token" to be used on account creations. 
	/// The request_token is retrieved from the callback
	/// linphone_account_creator_cbs_get_account_creation_request_token 
	/// - Returns: ``RequestOk`` if everything is OK, or a specific error otherwise. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	@available(*, deprecated)
	public func requestAccountCreationRequestToken() -> AccountCreator.Status
	{
		return AccountCreator.Status(rawValue: Int(linphone_account_creator_request_account_creation_request_token(cPtr).rawValue))!
	}
	
	
	
	/// Send a request to get a token to be used for account creation from a
	/// request_token. 
	/// The token is retrieved from the callback
	/// linphone_account_creator_cbs_get_account_creation_token_using_request_token 
	/// - Returns: ``RequestOk`` if the request has been sent, ``RequestFailed``
	/// otherwise 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	@available(*, deprecated)
	public func requestAccountCreationTokenUsingRequestToken() -> AccountCreator.Status
	{
		return AccountCreator.Status(rawValue: Int(linphone_account_creator_request_account_creation_token_using_request_token(cPtr).rawValue))!
	}
	
	
	
	/// Request an auth token to be send by the backend by push notification. 
	/// - Returns: ``RequestOk`` if everything is OK, or a specific error otherwise. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	@available(*, deprecated)
	public func requestAuthToken() -> AccountCreator.Status
	{
		return AccountCreator.Status(rawValue: Int(linphone_account_creator_request_auth_token(cPtr).rawValue))!
	}
	
	
	
	/// Reset the account creator entries like username, password, phone number... 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	@available(*, deprecated)
	public func reset() 
	{
		linphone_account_creator_reset(cPtr)
	}
	
	
	
	/// Set the phone number normalized. 
	/// - Parameter phoneNumber: The phone number to set    
	/// - Parameter countryCode: Country code to associate phone number with    
	/// - Returns: ``Ok`` if everything is OK, or specific(s) error(s) otherwise. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	@available(*, deprecated)
	public func setPhoneNumber(phoneNumber:String?, countryCode:String?) -> UInt
	{
		return UInt(linphone_account_creator_set_phone_number(cPtr, phoneNumber, countryCode))
	}
	
	
	
	/// Send a request to update an account. 
	/// - Returns: ``RequestOk`` if the request has been sent, ``RequestFailed``
	/// otherwise 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	@available(*, deprecated)
	public func updateAccount() -> AccountCreator.Status
	{
		return AccountCreator.Status(rawValue: Int(linphone_account_creator_update_account(cPtr).rawValue))!
	}
	
	
	
	/// Require the account creator to use special "test admin account". 
	/// - Warning: The "test admin account" is a special feature required for automated
	/// test, and requires the APP_EVERYONE_IS_ADMIN property to be enabled on the
	/// remote Flexisip Account Manager (FlexiAPI). This feature must never be turned
	/// on for a production-stage app. 
	/// - Deprecated: 11/06/2024 use ``AccountManagerServices`` instead 
	@available(*, deprecated)
	public func useTestAdminAccount() 
	{
		linphone_account_creator_use_test_admin_account(cPtr)
	}
}


/// Object that represents a device that at least once was connected to a given
/// account. 
public class AccountDevice : LinphoneObject
{

	static public func getSwiftObject(cObject:OpaquePointer) -> AccountDevice {
		let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
		if (result != nil) {
			return Unmanaged<AccountDevice>.fromOpaque(result!).takeUnretainedValue()
		}
		let sObject = AccountDevice(cPointer: cObject)
		belle_sip_object_data_set(UnsafeMutablePointer(cObject), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(sObject).toOpaque()), nil)
		return sObject
	}

	public var getCobject: OpaquePointer? {
		return cPtr
	}
	
	
	/// Gets the timestamp at which this devices was updated for the last time. 
	/// - Returns: the time under ISO 8601 format at which the device was updated.    
	public var lastUpdateTime: String?
	{
	
			
			let cPointer = linphone_account_device_get_last_update_time(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Gets the timestamp at which this devices was updated for the last time. 
	/// - Returns: the timestamp (time_t) at which the device was updated.    
	public var lastUpdateTimestamp: time_t?
	{
	
						return linphone_account_device_get_last_update_timestamp(cPtr)

	}
		
	
	/// Gets the name of the device. 
	/// - Returns: the name of the device.    
	public var name: String?
	{
	
			
			let cPointer = linphone_account_device_get_name(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Gets the user-agent of the device. 
	/// - Returns: the user-agent of the device.    
	public var userAgent: String
	{
	
			
			let cPointer = linphone_account_device_get_user_agent(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Gets the UUID of the device. 
	/// - Returns: the UUID of the device.    
	public var uuid: String
	{
	
			
			let cPointer = linphone_account_device_get_uuid(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
	}


/// Creates and manage SIP user accounts remotely, using the REST API of the
/// Flexisip Account Manager. 
/// - See also: https://gitlab.linphone.org/BC/public/flexisip-account-manager 
public class AccountManagerServices : LinphoneObject
{

	static public func getSwiftObject(cObject:OpaquePointer) -> AccountManagerServices {
		let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
		if (result != nil) {
			return Unmanaged<AccountManagerServices>.fromOpaque(result!).takeUnretainedValue()
		}
		let sObject = AccountManagerServices(cPointer: cObject)
		belle_sip_object_data_set(UnsafeMutablePointer(cObject), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(sObject).toOpaque()), nil)
		return sObject
	}

	public var getCobject: OpaquePointer? {
		return cPtr
	}
	
	/// Sets the preferred language to get error messages from account manager (if
	/// supported, otherwise will be english). 
	/// - Parameter language: The language (en, fr, etc...) you'd like to have error
	/// messages in (if possible).    
	
	/// Gets the configurer preferred language, if set. 
	/// - Returns: the language previously set, if any (otherwise nil).    
	public var language: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_account_manager_services_get_language(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_account_manager_services_set_language(cPtr, newValue)
		}
	}
		
	
	
	/// Requests to delete a device from the list of currently known devices. 
	/// - Parameter sipIdentity: the SIP identity URI that identifies your account for
	/// which you want the devices list.    
	/// - Parameter device: the ``AccountDevice`` to delete.    
	/// - Returns: the ``AccountManagerServicesRequest`` request object.    
	public func createDeleteDeviceRequest(sipIdentity:Address, device:AccountDevice) throws -> AccountManagerServicesRequest
	{
		let cPointer = linphone_account_manager_services_create_delete_device_request(cPtr, sipIdentity.cPtr, device.cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null AccountManagerServicesRequest value")
		}
		let result = AccountManagerServicesRequest.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Requests a an account creation request token that once validated using the URL
	/// returned by this method upon success, will allow you to call
	/// ``createGetAccountCreationTokenFromRequestTokenRequest(requestToken:)`` to
	/// obtain a valid account creation token. 
	/// Once the account creation token is obtained, you can call
	/// ``createNewAccountUsingTokenRequest(username:password:algorithm:token:)``. 
	/// - Returns: the ``AccountManagerServicesRequest`` request object.    
	public func createGetAccountCreationRequestTokenRequest() throws -> AccountManagerServicesRequest
	{
		let cPointer = linphone_account_manager_services_create_get_account_creation_request_token_request(cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null AccountManagerServicesRequest value")
		}
		let result = AccountManagerServicesRequest.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Converts an account creation request token obtained by
	/// linphone_account_manager_services_request_account_creation_request_token to an
	/// account creation token. 
	/// The obtained token can be used to call
	/// ``createNewAccountUsingTokenRequest(username:password:algorithm:token:)``. 
	/// - Parameter requestToken: the token obtained previously and validated using
	/// your web browser.    
	/// - Returns: the ``AccountManagerServicesRequest`` request object.    
	public func createGetAccountCreationTokenFromRequestTokenRequest(requestToken:String) throws -> AccountManagerServicesRequest
	{
		let cPointer = linphone_account_manager_services_create_get_account_creation_token_from_request_token_request(cPtr, requestToken)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null AccountManagerServicesRequest value")
		}
		let result = AccountManagerServicesRequest.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Requests the list of devices currently known. 
	/// - Parameter sipIdentity: the SIP identity URI that identifies your account for
	/// which you want the devices list.    
	/// - Returns: the ``AccountManagerServicesRequest`` request object.    
	public func createGetDevicesListRequest(sipIdentity:Address) throws -> AccountManagerServicesRequest
	{
		let cPointer = linphone_account_manager_services_create_get_devices_list_request(cPtr, sipIdentity.cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null AccountManagerServicesRequest value")
		}
		let result = AccountManagerServicesRequest.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Validates the link between an email address and an account using a code
	/// received by email after calling
	/// ``createSendEmailLinkingCodeByEmailRequest(sipIdentity:emailAddress:)`` 
	/// - Parameter sipIdentity: the SIP identity URI that identifies the account to
	/// which you want to link the email address to.    
	/// - Parameter code: the code received by email.    
	/// - Returns: the ``AccountManagerServicesRequest`` request object.    
	public func createLinkEmailToAccountUsingCodeRequest(sipIdentity:Address, code:String) throws -> AccountManagerServicesRequest
	{
		let cPointer = linphone_account_manager_services_create_link_email_to_account_using_code_request(cPtr, sipIdentity.cPtr, code)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null AccountManagerServicesRequest value")
		}
		let result = AccountManagerServicesRequest.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Validates the link between a phone number and an account using a code received
	/// by SMS after calling
	/// ``createSendPhoneNumberLinkingCodeBySmsRequest(sipIdentity:phoneNumber:)`` 
	/// - Parameter sipIdentity: the SIP identity URI that identifies the account to
	/// which you want to link the phone number to.    
	/// - Parameter code: the code received by SMS.    
	/// - Returns: the ``AccountManagerServicesRequest`` request object.    
	public func createLinkPhoneNumberToAccountUsingCodeRequest(sipIdentity:Address, code:String) throws -> AccountManagerServicesRequest
	{
		let cPointer = linphone_account_manager_services_create_link_phone_number_to_account_using_code_request(cPtr, sipIdentity.cPtr, code)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null AccountManagerServicesRequest value")
		}
		let result = AccountManagerServicesRequest.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates an account using an account creation token, for example obtained using
	/// ``createSendAccountCreationTokenByPushRequest(pnProvider:pnParam:pnPrid:)``. 
	/// - Parameter username: the username of the newly created account.    
	/// - Parameter password: the password to use for the newly created account.    
	/// - Parameter algorithm: the algorithm to use to hash the password (must be
	/// either MD5 or SHA-256).    
	/// - Parameter token: the account creation token obtained previously.    
	/// - Returns: the ``AccountManagerServicesRequest`` request object.    
	public func createNewAccountUsingTokenRequest(username:String, password:String, algorithm:String, token:String) throws -> AccountManagerServicesRequest
	{
		let cPointer = linphone_account_manager_services_create_new_account_using_token_request(cPtr, username, password, algorithm, token)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null AccountManagerServicesRequest value")
		}
		let result = AccountManagerServicesRequest.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Requests a push notification to be sent to device, containing a valid account
	/// creation token. 
	/// Provider, param & prid can be obtained from
	/// ``Core/getPushNotificationConfig()``, but on iOS may need some modifications
	/// (depending on debug mode for example). Once the token is obtained, you can call
	/// ``createNewAccountUsingTokenRequest(username:password:algorithm:token:)``. 
	/// - Parameter pnProvider: The provider, for example "apns.dev".    
	/// - Parameter pnParam: The parameters, for example
	/// "ABCD1234.org.linphone.phone.remote".    
	/// - Parameter pnPrid: The prid, also known as push token.    
	/// - Returns: the ``AccountManagerServicesRequest`` request object.    
	public func createSendAccountCreationTokenByPushRequest(pnProvider:String, pnParam:String, pnPrid:String) throws -> AccountManagerServicesRequest
	{
		let cPointer = linphone_account_manager_services_create_send_account_creation_token_by_push_request(cPtr, pnProvider, pnParam, pnPrid)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null AccountManagerServicesRequest value")
		}
		let result = AccountManagerServicesRequest.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Requests a push notification to be sent to device, containing a valid account
	/// recovery token. 
	/// Provider, param & prid can be obtained from
	/// ``Core/getPushNotificationConfig()``, but on iOS may need some modifications
	/// (depending on debug mode for example). Once the token is obtained, you can use
	/// it to open the recovery webpage on the flexisip account manager at
	/// https://account_manager.domain.tld/recovery/phone/<recovery token>?phone=<phone
	/// number> 
	/// Provider, param & prid can be obtained from
	/// ``Core/getPushNotificationConfig()``, but on iOS may need some modifications
	/// (depending on debug mode for example). Once the token is obtained, you can use
	/// it to open the recovery webpage on the flexisip account manager at
	/// https://account_manager.domain.tld/recovery/phone/<recovery token>?phone=<phone
	/// number> 
	/// - Parameter pnProvider: The provider, for example "apns.dev".    
	/// - Parameter pnParam: The parameters, for example
	/// "ABCD1234.org.linphone.phone.remote".    
	/// - Parameter pnPrid: The prid, also known as push token.    
	/// - Returns: the ``AccountManagerServicesRequest`` request object.    
	public func createSendAccountRecoveryTokenByPushRequest(pnProvider:String, pnParam:String, pnPrid:String) throws -> AccountManagerServicesRequest
	{
		let cPointer = linphone_account_manager_services_create_send_account_recovery_token_by_push_request(cPtr, pnProvider, pnParam, pnPrid)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null AccountManagerServicesRequest value")
		}
		let result = AccountManagerServicesRequest.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Requests a code to be sent to a given email address, that can be used later to
	/// associate said email to an account using
	/// ``createLinkEmailToAccountUsingCodeRequest(sipIdentity:code:)``. 
	/// - Parameter sipIdentity: the SIP identity URI that identifies the account to
	/// which you want to link the email address to.    
	/// - Parameter emailAddress: the email address to which to send the code to.    
	/// - Returns: the ``AccountManagerServicesRequest`` request object.    
	public func createSendEmailLinkingCodeByEmailRequest(sipIdentity:Address, emailAddress:String) throws -> AccountManagerServicesRequest
	{
		let cPointer = linphone_account_manager_services_create_send_email_linking_code_by_email_request(cPtr, sipIdentity.cPtr, emailAddress)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null AccountManagerServicesRequest value")
		}
		let result = AccountManagerServicesRequest.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Requests a code to be sent to a given phone number, that can be used later to
	/// associate said phone number to an account using
	/// ``createLinkPhoneNumberToAccountUsingCodeRequest(sipIdentity:code:)``. 
	/// - Parameter sipIdentity: the SIP identity URI that identifies the account to
	/// which you want to link the phone number to.    
	/// - Parameter phoneNumber: the phone number to which to send the code by SMS.    
	/// - Returns: the ``AccountManagerServicesRequest`` request object.    
	public func createSendPhoneNumberLinkingCodeBySmsRequest(sipIdentity:Address, phoneNumber:String) throws -> AccountManagerServicesRequest
	{
		let cPointer = linphone_account_manager_services_create_send_phone_number_linking_code_by_sms_request(cPtr, sipIdentity.cPtr, phoneNumber)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null AccountManagerServicesRequest value")
		}
		let result = AccountManagerServicesRequest.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
}


/// Request object created by ``AccountManagerServices``. 
public class AccountManagerServicesRequest : LinphoneObject
{
	var delegateManagers : [AccountManagerServicesRequestDelegateManager] = []

	static public func getSwiftObject(cObject:OpaquePointer) -> AccountManagerServicesRequest {
		let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
		if (result != nil) {
			return Unmanaged<AccountManagerServicesRequest>.fromOpaque(result!).takeUnretainedValue()
		}
		let sObject = AccountManagerServicesRequest(cPointer: cObject)
		belle_sip_object_data_set(UnsafeMutablePointer(cObject), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(sObject).toOpaque()), nil)
		return sObject
	}

	public var getCobject: OpaquePointer? {
		return cPtr
	}

	
	///List of all supported ``AccountManagerServicesRequest`` requests, allowing to
	///know which one triggered either a callback from the
	///``AccountManagerServicesRequestDelegate``. 
	public enum Kind:Int
	{
		
		/// Asks the account manager to send us an account creation token by push
		/// notification. 
		case SendAccountCreationTokenByPush = 0
		/// Asks the account manager to create a request token to be validated in a web
		/// browser (captcha) and that can later be used to create an account creation
		/// token. 
		case AccountCreationRequestToken = 1
		/// Asks the account manager to consume the account creation request token to
		/// create an account creation token. 
		case AccountCreationTokenFromAccountCreationRequestToken = 2
		/// Uses an account creation token to create the account. 
		case CreateAccountUsingToken = 3
		/// Asks for a code to be sent by SMS to a given phone number to link that phone
		/// number with an account. 
		case SendPhoneNumberLinkingCodeBySms = 4
		/// Uses the code received by SMS to confirm the link between an account and a
		/// phone number. 
		case LinkPhoneNumberUsingCode = 5
		/// Asks for a code to be sent by email to a given email address to link that
		/// address with an account. 
		case SendEmailLinkingCodeByEmail = 6
		/// Uses the code received by email to confirm the link between an account and an
		/// email address. 
		case LinkEmailUsingCode = 7
		/// Gets the list of devices for account. 
		case GetDevicesList = 8
		/// Removes an account device. 
		case DeleteDevice = 9
		/// Asks the account manager to send us an account recovery token by push
		/// notification. 
		case SendAccountRecoveryTokenByPush = 10
		case GetCreationTokenAsAdmin = 100
		case GetAccountInfoAsAdmin = 101
		case DeleteAccountAsAdmin = 102
	}
	
	
	
	public func addDelegate(delegate: AccountManagerServicesRequestDelegate)
	{
		let manager = AccountManagerServicesRequestDelegateManager()
		manager.delegate = delegate
		delegateManagers.append(manager)
		linphone_account_manager_services_request_add_callbacks(cPtr, manager.cPtr)
		belle_sip_object_unref(UnsafeMutableRawPointer(manager.cPtr))
		belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(self).toOpaque()), nil)
	}	
	
	
	public func removeDelegate(delegate: AccountManagerServicesRequestDelegate)
	{
		if let index = delegateManagers.firstIndex(where: { $0.delegate === delegate as AnyObject}) {
			linphone_account_manager_services_request_remove_callbacks(cPtr, delegateManagers[index].cPtr)
			delegateManagers.remove(at: index)
			belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(self).toOpaque()), nil)
		}
	}	
	
	/// Gets the current ``AccountManagerServicesRequest``. 
	/// This is meant only to be called from a callback to be able to get the user_data
	/// associated with the ``AccountManagerServicesRequestDelegate`` that is calling
	/// the callback. 
	/// - Returns: The ``AccountManagerServicesRequestDelegate`` that has called the
	/// last callback.    
	public var currentDelegate: AccountManagerServicesRequestDelegate?
	{
	
			let cObject = linphone_account_manager_services_request_get_current_callbacks(cPtr)
			let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
			if (result != nil) {
				return Unmanaged<AccountManagerServicesRequestDelegateManager>.fromOpaque(result!).takeUnretainedValue().delegate
				}
			return nil
			
	}
		
	
	/// Gets the ``Kind`` representing this request. 
	/// - Returns: the ``Kind``. 
	public var type: AccountManagerServicesRequest.Kind
	{
	
						return AccountManagerServicesRequest.Kind(rawValue: Int(linphone_account_manager_services_request_get_type(cPtr).rawValue))!

	}
		
	
	
	/// Executes the request represented by ``AccountManagerServicesRequest``. 
	public func submit() 
	{
		linphone_account_manager_services_request_submit(cPtr)
	}
}


/// Object that is used to set the different parameters of a ``Account``. 
/// Note that authenticated accounts should have a corresponding ``AuthInfo`` added
/// to the ``Core`` to register properly. 
public class AccountParams : LinphoneObject
{

	static public func getSwiftObject(cObject:OpaquePointer) -> AccountParams {
		let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
		if (result != nil) {
			return Unmanaged<AccountParams>.fromOpaque(result!).takeUnretainedValue()
		}
		let sObject = AccountParams(cPointer: cObject)
		belle_sip_object_data_set(UnsafeMutablePointer(cObject), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(sObject).toOpaque()), nil)
		return sObject
	}

	public var getCobject: OpaquePointer? {
		return cPtr
	}
	
	
	
	/// Create a new ``AccountParams`` object from a configuration. 
	/// - Parameter lc: The ``Core`` object.    
	/// - Parameter index: The index of the configuration. 
	/// - Returns: The newly created ``AccountParams`` object.    
	static public func newWithConfig(lc:Core, index:Int) -> AccountParams?
	{
		let cPointer = linphone_account_params_new_with_config(lc.cPtr, CInt(index))
		if (cPointer == nil) {
			return nil
		}
		let result = AccountParams.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	/// Set the audio video conference factory uri. 
	/// - Parameter address: The ``Address`` to set.    
	
	/// Get the audio video conference factory uri. 
	/// - Returns: The ``Address`` of the audio video conference factory.    
	public var audioVideoConferenceFactoryAddress: Address?
	{
	
		get
		{ 
						let cPointer = linphone_account_params_get_audio_video_conference_factory_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_account_params_set_audio_video_conference_factory_address(cPtr, newValue?.cPtr)
		}
	}
		
	/// Enable the use of RTCP feedback (also known as AVPF profile). 
	/// - Parameter mode: the enablement mode, which can be ``Default`` (use ``Core``'s
	/// mode), ``Enabled`` (avpf is enabled), or ``Disabled`` (disabled). 
	
	/// Get enablement status of RTCP feedback (also known as AVPF profile). 
	/// - Returns: the enablement mode, which can be ``Default`` (use ``Core``'s mode),
	/// ``Enabled`` (avpf is enabled), or ``Disabled`` (disabled). 
	public var avpfMode: AVPFMode
	{
	
		get
		{ 
						return AVPFMode(rawValue: Int(linphone_account_params_get_avpf_mode(cPtr).rawValue))!
		}
		set
		{
			linphone_account_params_set_avpf_mode(cPtr, LinphoneAVPFMode(rawValue: CInt(newValue.rawValue)))
		}
	}
		
	/// Set the interval between regular RTCP reports when using AVPF/SAVPF. 
	/// - Parameter interval: The interval in seconds (between 0 and 5 seconds). 
	
	/// Get the interval between regular RTCP reports when using AVPF/SAVPF. 
	/// - Returns: The interval in seconds. 
	public var avpfRrInterval: UInt8
	{
	
		get
		{ 
						return linphone_account_params_get_avpf_rr_interval(cPtr)
		}
		set
		{
			linphone_account_params_set_avpf_rr_interval(cPtr, newValue)
		}
	}
		
	/// Set the CCMP server address. 
	/// - Parameter url: The URL of the CCMP server.    
	
	/// Get the CCMP server address. 
	/// - Returns: The URL of the CCMP server.    
	public var ccmpServerUrl: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_account_params_get_ccmp_server_url(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_account_params_set_ccmp_server_url(cPtr, newValue)
		}
	}
		
	
	/// Get the CCMP user ID. 
	/// - Returns: The ID of the CCMP user.    
	public var ccmpUserId: String?
	{
	
			
			let cPointer = linphone_account_params_get_ccmp_user_id(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	/// Set the conference factory uri. 
	/// - Parameter address: The ``Address`` to set.    
	
	/// Get the conference factory uri. 
	/// - Returns: The ``Address`` of the conference factory.    
	public var conferenceFactoryAddress: Address?
	{
	
		get
		{ 
						let cPointer = linphone_account_params_get_conference_factory_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_account_params_set_conference_factory_address(cPtr, newValue?.cPtr)
		}
	}
		
	/// Set the conference factory uri. 
	/// - Parameter uri: The uri of the conference factory.    
	/// - Deprecated: 16/08/2023 Use ``setConferenceFactoryAddress(address:)`` instead. 
	
	/// Get the conference factory uri. 
	/// - Returns: The uri of the conference factory.    
	public var conferenceFactoryUri: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_account_params_get_conference_factory_uri(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
	@available(*, deprecated)
		set
		{
			linphone_account_params_set_conference_factory_uri(cPtr, newValue)
		}
	}
		
	/// Set optional contact parameters that will be added to the contact information
	/// sent in the registration. 
	/// - Parameter contactParams: A string contaning the additional parameters in text
	/// form, like "myparam=something;myparam2=something_else"   
	/// The main use case for this function is provide the proxy additional information
	/// regarding the user agent, like for example unique identifier or apple push id.
	/// As an example, the contact address in the SIP register sent will look like
	/// <sip:joe@15.128.128.93:50421>;apple-push-id=43143-DFE23F-2323-FA2232. 
	/// The main use case for this function is provide the proxy additional information
	/// regarding the user agent, like for example unique identifier or apple push id.
	/// As an example, the contact address in the SIP register sent will look like
	/// <sip:joe@15.128.128.93:50421>;apple-push-id=43143-DFE23F-2323-FA2232. 
	
	/// Returns the contact parameters. 
	/// - Returns: The previously set contact parameters.    
	public var contactParameters: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_account_params_get_contact_parameters(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_account_params_set_contact_parameters(cPtr, newValue)
		}
	}
		
	/// Set optional contact parameters that will be added to the contact information
	/// sent in the registration, inside the URI. 
	/// - Parameter contactUriParams: A string containing the additional parameters in
	/// text form, like "myparam=something;myparam2=something_else"   
	/// The main use case for this function is provide the proxy additional information
	/// regarding the user agent, like for example unique identifier or apple push id.
	/// As an example, the contact address in the SIP register sent will look like
	/// <sip:joe@15.128.128.93:50421;apple-push-id=43143-DFE23F-2323-FA2232>. 
	/// The main use case for this function is provide the proxy additional information
	/// regarding the user agent, like for example unique identifier or apple push id.
	/// As an example, the contact address in the SIP register sent will look like
	/// <sip:joe@15.128.128.93:50421;apple-push-id=43143-DFE23F-2323-FA2232>. 
	
	/// Return the contact URI parameters. 
	/// - Returns: The previously set contact URI parameters.    
	public var contactUriParameters: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_account_params_get_contact_uri_parameters(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_account_params_set_contact_uri_parameters(cPtr, newValue)
		}
	}
		
	/// Indicates whether chat messages sent by this account in a ``Basic`` chat room
	/// will be using CPIM format or not. 
	/// - Parameter enable: true to send messages in CPIM format, false to keep using
	/// the SIP SIMPLE format. 
	
	/// Indicates whether chat messages sent by this account in a ``Basic`` chat room
	/// will be using CPIM format or not. 
	/// By default SIP SIMPLE format is used for "basic" chat rooms, CPIM is only used
	/// for ``FlexisipChat`` chat rooms. - See also:
	/// https://wiki.linphone.org/xwiki/wiki/public/view/Lib/Features/Instant%20Messaging/Reply%20to%20a%20specific%20message/ 
	/// - Returns: true if chat messages will be sent out in CPIM format, false if chat
	/// messages will be sent out as SIP SIMPLE. 
	public var cpimInBasicChatRoomEnabled: Bool
	{
	
		get
		{ 
						return linphone_account_params_cpim_in_basic_chat_room_enabled(cPtr) != 0
		}
		set
		{
			linphone_account_params_enable_cpim_in_basic_chat_room(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Set a an optional custom contact address to be used when registering to the SIP
	/// server. 
	/// This is typically used to supply an alternative SIP address to call in case the
	/// main one is not reachable. 
	/// - Parameter contact: a ``Address`` the optional alternative contact address.    
	
	/// Get the custom contact address previously used when registering to the SIP
	/// server. 
	/// - Returns: a ``Address``    
	public var customContact: Address?
	{
	
		get
		{ 
						let cPointer = linphone_account_params_get_custom_contact(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_account_params_set_custom_contact(cPtr, newValue?.cPtr)
		}
	}
		
	/// Set whether liblinphone should replace "+" by international calling prefix
	/// (ICP) in dialed numbers (passed to linphone_core_invite). 
	/// - Parameter enable: true to replace + by the international prefix, false
	/// otherwise. 
	
	/// Return whether or not the + should be replaced by the International Call
	/// Prefix. 
	/// - Returns: Whether liblinphone should replace "+" by the International Call
	/// Prefix. in dialed numbers (passed to ``Core/invite(url:)``). 
	public var dialEscapePlusEnabled: Bool
	{
	
		get
		{ 
						return linphone_account_params_dial_escape_plus_enabled(cPtr) != 0
		}
		set
		{
			linphone_account_params_enable_dial_escape_plus(cPtr, newValue==true ? 1:0)
		}
	}
		
	
	/// Get the domain name of the given account params. 
	/// - Returns: The domain name of the account params.    
	public var domain: String?
	{
	
			
			let cPointer = linphone_account_params_get_domain(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	/// Sets the registration expiration time in seconds. 
	/// - Parameter expires: The expiration time to set. 
	
	/// Get the account params expires. 
	/// - Returns: The duration of registration. 
	public var expires: Int
	{
	
		get
		{ 
						return Int(linphone_account_params_get_expires(cPtr))
		}
		set
		{
			linphone_account_params_set_expires(cPtr, CInt(newValue))
		}
	}
		
	
	/// Get the identity of the account params. 
	/// - Returns: The SIP identity that belongs to this account params.    
	/// - Deprecated: 01/03/2021 Use ``getIdentityAddress()`` instead. 
	@available(*, deprecated)
	public var identity: String?
	{
	
			
			let cPointer = linphone_account_params_get_identity(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	/// Sets the user identity as a SIP address. 
	/// This identity is normally formed with display name, username and domain, such
	/// as: Alice <sip:alice@example.net> The REGISTER messages will have from and to
	/// set to this identity. 
	/// This identity is normally formed with display name, username and domain, such
	/// as: Alice <sip:alice@example.net> The REGISTER messages will have from and to
	/// set to this identity. 
	/// - Parameter identity: The ``Address`` of the identity to set.    
	/// - Returns: 0 if successful, -1 otherwise. 
	
	/// Get the identity address of the account params. 
	/// - Returns: The SIP identity that belongs to this account params.    
	public var identityAddress: Address?
	{
	
						let cPointer = linphone_account_params_get_identity_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result

	}
	public func setIdentityaddress(newValue: Address?) throws
	{
		let exception_result = linphone_account_params_set_identity_address(cPtr, newValue?.cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "username setter returned value \(exception_result)")
		}
	}
		
	/// Set the idkey property on the given account params. 
	/// This property can the be referenced by another account params 'depends_on' to
	/// create a dependency relation between them.
	/// - Parameter idkey: The idkey string to associate to the given
	/// ``AccountParams``.    
	
	/// Get the idkey property of a ``AccountParams``. 
	/// - Returns: The idkey string, or nil.    
	public var idkey: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_account_params_get_idkey(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_account_params_set_idkey(cPtr, newValue)
		}
	}
		
	/// Defines whether the encryption of instant messages is mandatory. 
	/// If it is, clear text messages will be denied. 
	/// - Parameter mandatory: true to set it mandatory; false otherwise. 
	
	/// Checks if encryption is mandatory for instant messages or not. 
	/// - Returns: true if encryption is mandatory; false otherwise. 
	public var instantMessagingEncryptionMandatory: Bool
	{
	
		get
		{ 
						return linphone_account_params_get_instant_messaging_encryption_mandatory(cPtr) != 0
		}
		set
		{
			linphone_account_params_set_instant_messaging_encryption_mandatory(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets an international prefix (country code) to be automatically prepended when
	/// inviting a number with ``Core/invite(url:)`` or when using
	/// ``Account/normalizePhoneNumber(username:)``. 
	/// This international prefix shall usually be the country code of the country
	/// where the user is living, without "+". - Warning: It is also referred as 'ccc'
	/// (Calling Country Code) and must not be confused with the ICP (International
	/// Call Prefix). The ICP is a fixed property of the country dial plan, and cannot
	/// be set in the ``AccountParams`` . 
	/// - Parameter prefix: The prefix to set (withouth the +).    
	
	/// Gets the prefix set for this account params. 
	/// - Returns: The international prefix if set, null otherwise.    
	public var internationalPrefix: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_account_params_get_international_prefix(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_account_params_set_international_prefix(cPtr, newValue)
		}
	}
		
	/// Sets the ISO country code matching the international prefix. 
	/// - Parameter prefixIsoCountryCode: The ISO country code to set.    
	
	/// Gets the ISO country code set for the international prefix in this account
	/// params. 
	/// - Returns: The international prefix ISO country code if set, null otherwise.    
	public var internationalPrefixIsoCountryCode: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_account_params_get_international_prefix_iso_country_code(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_account_params_set_international_prefix_iso_country_code(cPtr, newValue)
		}
	}
		
	
	/// Gets whether push notifications are available or not (Android & iOS only). 
	/// - Returns: true if push notifications are available, false otherwise 
	public var isPushNotificationAvailable: Bool
	{
	
						return linphone_account_params_is_push_notification_available(cPtr) != 0

	}
		
	/// Set the base(s) x3dh algorithm. 
	/// accept an ordered comma separated list (without space) of lime base algorithms
	/// accepted values are a combination of: c25519, c448 and c25519mlk512 nil is also
	/// valid, it will unset the value 
	/// - Parameter algo: The x3dh base algorithm.    
	
	/// Get the base x3dh algorithm. 
	/// - Returns: The x3dh base algorithm.    
	public var limeAlgo: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_account_params_get_lime_algo(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_account_params_set_lime_algo(cPtr, newValue)
		}
	}
		
	/// Set the x3dh server url. 
	/// If empty, this function will disable LIME X3DH from core. Otherwise, or if
	/// different from the existing value, this will (re-)initialize the LIME X3DH
	/// engine. 
	/// - Parameter url: The x3dh server url.    
	
	/// Get the x3dh server url. 
	/// - Returns: The x3dh server url.    
	public var limeServerUrl: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_account_params_get_lime_server_url(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_account_params_set_lime_server_url(cPtr, newValue)
		}
	}
		
	/// Sets the Message Waiting Indication server address. 
	/// - Parameter address: The Message Waiting Indication server address.    
	
	/// Gets the Message Waiting Indication server address. 
	/// - Returns: The Message Waiting Indication server address.    
	public var mwiServerAddress: Address?
	{
	
		get
		{ 
						let cPointer = linphone_account_params_get_mwi_server_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_account_params_set_mwi_server_address(cPtr, newValue?.cPtr)
		}
	}
		
	/// Set the policy to use to pass through NATs/firewalls when using this account
	/// params. 
	/// If it is set to nil, the default NAT policy from the core will be used instead. 
	/// - Parameter policy: The ``NatPolicy`` object.    
	/// - See also: ``Core/setNatPolicy(policy:)`` 
	
	/// Get The policy that is used to pass through NATs/firewalls when using this
	/// account params. 
	/// If it is set to nil, the default NAT policy from the core will be used instead. 
	/// - Returns: The ``NatPolicy`` object in use.    
	/// - See also: ``Core/getNatPolicy()`` 
	public var natPolicy: NatPolicy?
	{
	
		get
		{ 
						let cPointer = linphone_account_params_get_nat_policy(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = NatPolicy.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_account_params_set_nat_policy(cPtr, newValue?.cPtr)
		}
	}
		
	/// If enabled, the proxy will be used as the only route. 
	/// - Warning: This function will replace or remove any routes previously set with
	/// ``setRoutesAddresses(routes:)`` if any. For that reason
	/// ``enableOutboundProxy(enable:)`` shall not be used with
	/// ``setRoutesAddresses(routes:)``. Application shall consider either using the
	/// restrictive notion of outbound proxy (a single route identical to the registrar
	/// URI), either the notion of predefined route-set where routes can be multiple
	/// and different from registrar URI. 
	/// - Parameter enable: true to enable, false otherwise. 
	
	/// Tell if the proxy is used as the only route. 
	/// - Returns: enable true if enabled, false otherwise. 
	public var outboundProxyEnabled: Bool
	{
	
		get
		{ 
						return linphone_account_params_outbound_proxy_enabled(cPtr) != 0
		}
		set
		{
			linphone_account_params_enable_outbound_proxy(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets an URI for the account picture. 
	/// - Parameter uri: The account picture URI.    
	
	/// Gets the account picture URI if set, nil otherwise. 
	/// - Returns: The account picture URI.    
	public var pictureUri: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_account_params_get_picture_uri(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_account_params_set_picture_uri(cPtr, newValue)
		}
	}
		
	/// Set default privacy policy for all calls routed through this proxy. 
	/// - Parameter privacy: LinphonePrivacyMask to configure privacy 
	
	/// Get default privacy policy for all calls routed through this proxy. 
	/// - Returns: Privacy mode as LinphonePrivacyMask 
	public var privacy: UInt
	{
	
		get
		{ 
						return UInt(linphone_account_params_get_privacy(cPtr))
		}
		set
		{
			linphone_account_params_set_privacy(cPtr, CUnsignedInt(newValue))
		}
	}
		
	/// Indicates either or not, PUBLISH must be issued for this ``AccountParams``. 
	/// - Parameter enable: If true, publish will be engaged. 
	
	/// Tell if the PUBLISH is enabled. 
	/// - Returns: true if PUBLISH request is enabled for this proxy. 
	public var publishEnabled: Bool
	{
	
		get
		{ 
						return linphone_account_params_publish_enabled(cPtr) != 0
		}
		set
		{
			linphone_account_params_enable_publish(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Set the publish expiration time in second. 
	/// - Parameter expires: The expire time in seconds. 
	
	/// Get the publish expiration time in second. 
	/// Default value is the registration expiration value. 
	/// - Returns: The expire time in seconds. 
	public var publishExpires: Int
	{
	
		get
		{ 
						return Int(linphone_account_params_get_publish_expires(cPtr))
		}
		set
		{
			linphone_account_params_set_publish_expires(cPtr, CInt(newValue))
		}
	}
		
	/// Indicates whether to add to the contact parameters the push notification
	/// information. 
	/// For IOS, it indicates for VOIP push notification. 
	/// - Parameter allow: true to allow push notification information, false
	/// otherwise. 
	
	/// Indicates whether to add to the contact parameters the push notification
	/// information. 
	/// For IOS, it indicates for VOIP push notification. 
	/// - Returns: true if push notification informations should be added, false
	/// otherwise. 
	public var pushNotificationAllowed: Bool
	{
	
		get
		{ 
						return linphone_account_params_get_push_notification_allowed(cPtr) != 0
		}
		set
		{
			linphone_account_params_set_push_notification_allowed(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets the push notification configuration. 
	/// - Parameter config: The ``PushNotificationConfig`` object.    
	
	/// Returns the push notification configuration. 
	/// - Returns: The ``PushNotificationConfig`` object.    
	public var pushNotificationConfig: PushNotificationConfig?
	{
	
		get
		{ 
						let cPointer = linphone_account_params_get_push_notification_config(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = PushNotificationConfig.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_account_params_set_push_notification_config(cPtr, newValue?.cPtr)
		}
	}
		
	/// Set the route of the collector end-point when using quality reporting. 
	/// This SIP address should be used on server-side to process packets directly
	/// before discarding packets. Collector address should be a non existing account
	/// and will not receive any messages. If nil, reports will be send to the proxy
	/// domain. 
	/// - Parameter collector: route of the collector end-point, if nil PUBLISH will be
	/// sent to the proxy domain.    
	
	/// Get the route of the collector end-point when using quality reporting. 
	/// This SIP address should be used on server-side to process packets directly
	/// before discarding packets. Collector address should be a non existing account
	/// and will not receive any messages. If nil, reports will be send to the proxy
	/// domain. 
	/// - Returns: The SIP address of the collector end-point.    
	public var qualityReportingCollector: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_account_params_get_quality_reporting_collector(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_account_params_set_quality_reporting_collector(cPtr, newValue)
		}
	}
		
	/// Indicates whether quality statistics during call should be stored and sent to a
	/// collector according to RFC 6035. 
	/// - Parameter enable: true to store quality statistics and send them to the
	/// collector, false to disable it. 
	
	/// Indicates whether quality statistics during call should be stored and sent to a
	/// collector according to RFC 6035. 
	/// - Returns: true if quality repotring is enabled, false otherwise. 
	public var qualityReportingEnabled: Bool
	{
	
		get
		{ 
						return linphone_account_params_quality_reporting_enabled(cPtr) != 0
		}
		set
		{
			linphone_account_params_enable_quality_reporting(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Set the interval between 2 interval reports sending when using quality
	/// reporting. 
	/// If call exceed interval size, an interval report will be sent to the collector.
	/// On call termination, a session report will be sent for the remaining period.
	/// Value must be 0 (disabled) or positive. 
	/// - Parameter interval: The interval in seconds, 0 means interval reports are
	/// disabled. 
	
	/// Get the interval between interval reports when using quality reporting. 
	/// - Returns: The interval in seconds, 0 means interval reports are disabled. 
	public var qualityReportingInterval: Int
	{
	
		get
		{ 
						return Int(linphone_account_params_get_quality_reporting_interval(cPtr))
		}
		set
		{
			linphone_account_params_set_quality_reporting_interval(cPtr, CInt(newValue))
		}
	}
		
	/// Set the realm of the given account params. 
	/// This is optional, but recommended as it allows digest authentication context to
	/// be re-used accross subsequent SIP requests, which reduces by almost half the
	/// number of SIP rmessages exchanged between a client and a server. The server is
	/// required to support the qop=auth digest authentication mode to benefit from
	/// this feature. - See also: rfc7616 https://datatracker.ietf.org/doc/html/rfc7616 
	/// - Parameter realm: New realm value.    
	
	/// Get the realm of the given account params. 
	/// This is optional, but recommended as it allows digest authentication context to
	/// be re-used accross subsequent SIP requests, which reduces by almost half the
	/// number of SIP rmessages exchanged between a client and a server. The server is
	/// required to support the qop=auth digest authentication mode to benefit from
	/// this feature. - See also: rfc7616 https://datatracker.ietf.org/doc/html/rfc7616 
	/// - Returns: The realm of the account params.    
	public var realm: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_account_params_get_realm(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_account_params_set_realm(cPtr, newValue)
		}
	}
		
	/// Associate a persistent reference key to the account params. 
	/// The reference key can be for example an id to an external database. It is
	/// stored in the config file, thus can survive to process exits/restarts.
	/// - Parameter refkey: The reference key string to associate to the account
	/// params.    
	
	/// Get the persistent reference key associated to the account params. 
	/// The reference key can be for example an id to an external database. It is
	/// stored in the config file, thus can survive to process exits/restarts.
	/// - Returns: The reference key string that has been associated to the account
	/// params, or nil if none has been associated.    
	public var refKey: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_account_params_get_ref_key(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_account_params_set_ref_key(cPtr, newValue)
		}
	}
		
	/// Indicates either or not, REGISTRATION must be issued for this
	/// ``AccountParams``. 
	/// - Parameter enable: If true, registration will be engaged. 
	
	/// Returns whether the account params is enabled or not. 
	/// - Returns: true if registration to the proxy is enabled. 
	public var registerEnabled: Bool
	{
	
		get
		{ 
						return linphone_account_params_register_enabled(cPtr) != 0
		}
		set
		{
			linphone_account_params_enable_register(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Indicates whether to add to the contact parameters the push notification
	/// information. 
	/// - Parameter allow: true to allow remote push notification information, false
	/// otherwise. 
	
	/// Indicates whether to add to the contact parameters the push notification
	/// information. 
	/// - Returns: true if remote push notification informations should be added, false
	/// otherwise. 
	public var remotePushNotificationAllowed: Bool
	{
	
		get
		{ 
						return linphone_account_params_get_remote_push_notification_allowed(cPtr) != 0
		}
		set
		{
			linphone_account_params_set_remote_push_notification_allowed(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets a list of SIP route. 
	/// When a route is set, all outgoing calls will go to the route's destination if
	/// this account is the default one (see ``Core/setDefaultAccount(account:)``).
	/// - Warning: This function shall not be used in conjunction with
	/// ``enableOutboundProxy(enable:)``. 
	/// - Parameter routes: A list of routes.      
	/// - Returns: -1 if routes are invalid, 0 otherwise. 
	
	/// Gets the list of the routes set for this account params. 
	/// - Returns: The list of routes.      
	public var routesAddresses: [Address]
	{
	
						var swiftList = [Address]()
			let cList = linphone_account_params_get_routes_addresses(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(Address.getSwiftObject(cObject: data))
				listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
				bctbx_list_free(cList)
			return swiftList

	}
	public func setRoutesaddresses(newValue: [Address]) throws
	{
		var exception_result: LinphoneStatus = 0
		withExtendedLifetime(newValue){
			var cList: UnsafeMutablePointer<bctbx_list_t>? = nil
			for data in newValue {
				cList = bctbx_list_append(cList, UnsafeMutableRawPointer(data.cPtr))
			}
			exception_result = linphone_account_params_set_routes_addresses(cPtr, cList)
			bctbx_list_free(cList)
		}
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "username setter returned value \(exception_result)")
		}
	}
		
	/// Indicates whether support of rtp bundle is assumed. 
	/// See ``enableRtpBundle(value:)`` for background information about rtp bundle.
	/// Assumption that RTP bundling support allows interesting optimizations, such as
	/// not gathering RTCP candidates, and not gathering candidates for video stream
	/// when making an outgoing call. This setting is meaningful only if rtp bundle is
	/// enabled. See https://datatracker.ietf.org/doc/html/rfc8843 for more information
	/// about the feature. 
	/// See ``enableRtpBundle(value:)`` for background information about rtp bundle.
	/// Assumption that RTP bundling support allows interesting optimizations, such as
	/// not gathering RTCP candidates, and not gathering candidates for video stream
	/// when making an outgoing call. This setting is meaningful only if rtp bundle is
	/// enabled. See https://datatracker.ietf.org/doc/html/rfc8843 for more information
	/// about the feature. 
	/// - Parameter value: a boolean to indicate whether RTP bundle support can be
	/// assumed. 
	
	/// Returns whether RTP bundle mode is assumed. 
	/// See https://datatracker.ietf.org/doc/html/rfc8843 for more information. 
	/// See https://datatracker.ietf.org/doc/html/rfc8843 for more information. 
	/// - Returns: a boolean indicating when rtp bundle support is assumed. 
	public var rtpBundleAssumptionEnabled: Bool
	{
	
		get
		{ 
						return linphone_account_params_rtp_bundle_assumption_enabled(cPtr) != 0
		}
		set
		{
			linphone_account_params_enable_rtp_bundle_assumption(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Enables or disables RTP bundle mode (Media Multiplexing). 
	/// See https://datatracker.ietf.org/doc/html/rfc8843 for more information about
	/// the feature. When enabled, liblinphone will try to negociate the use of a
	/// single port for all streams when doing an outgoing call. It automatically
	/// enables rtcp-mux. 
	/// See https://datatracker.ietf.org/doc/html/rfc8843 for more information about
	/// the feature. When enabled, liblinphone will try to negociate the use of a
	/// single port for all streams when doing an outgoing call. It automatically
	/// enables rtcp-mux. 
	/// - Parameter value: a boolean to indicate whether the feature is to be enabled. 
	
	/// Returns whether RTP bundle mode (also known as Media Multiplexing) is enabled. 
	/// See https://datatracker.ietf.org/doc/html/rfc8843 for more information. 
	/// See https://datatracker.ietf.org/doc/html/rfc8843 for more information. 
	/// - Returns: a boolean indicating the enablement of rtp bundle mode. 
	public var rtpBundleEnabled: Bool
	{
	
		get
		{ 
						return linphone_account_params_rtp_bundle_enabled(cPtr) != 0
		}
		set
		{
			linphone_account_params_enable_rtp_bundle(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets the proxy address. 
	/// Examples of valid sip proxy address are:
	/// -IP address: sip:87.98.157.38
	/// -IP address with port: sip:87.98.157.38:5062
	/// -hostnames : sip:sip.example.net
	/// - Parameter serverAddress: The proxy address to set.    
	/// - Returns: 0 if successful, -1 otherwise. 
	/// - Deprecated: 01/03/2021 Use ``setServerAddress(serverAddress:)`` instead. 
	
	/// Get the account params proxy address. 
	/// - Returns: The proxy's SIP address.    
	/// - Deprecated: 01/03/2021 Use ``getServerAddress()`` instead. 
	@available(*, deprecated)
	public var serverAddr: String?
	{
	
			
			let cPointer = linphone_account_params_get_server_addr(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
	@available(*, deprecated)
	public func setServeraddr(newValue: String?) throws
	{
		let exception_result = linphone_account_params_set_server_addr(cPtr, newValue)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "username setter returned value \(exception_result)")
		}
	}
		
	/// Sets the SIP proxy or registrar address. 
	/// Examples of valid sip proxy address are:
	/// -IP address: sip:87.98.157.38
	/// -IP address with port: sip:87.98.157.38:5062
	/// -hostnames : sip:sip.example.net
	/// - Parameter serverAddress: The proxy as ``Address`` to set.    
	/// - Returns: 0 if successful, -1 otherwise. 
	
	/// Get the account params SIP proxy or registrar address. 
	/// - Returns: The proxy's SIP ``Address``.    
	public var serverAddress: Address?
	{
	
						let cPointer = linphone_account_params_get_server_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result

	}
	public func setServeraddress(newValue: Address?) throws
	{
		let exception_result = linphone_account_params_set_server_address(cPtr, newValue?.cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "username setter returned value \(exception_result)")
		}
	}
		
	/// Sets the list of supported tags. 
	/// - Parameter supportedTags: The list of supported tags  .    
	
	/// Gets the list of supported tags. 
	/// - Returns: The list of supported tags  .    
	public var supportedTagsList: [String]
	{
	
		get
		{ 
						var swiftList = [String]()
			let cList = linphone_account_params_get_supported_tags_list(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				swiftList.append(String(cString: unsafeBitCast(listTemp!.pointee.data, to: UnsafePointer<CChar>.self)))
				listTemp = UnsafePointer<bctbx_list_t>(listTemp!.pointee.next)
			}
			return swiftList
		}
		set
		{
			var cList: UnsafeMutablePointer<bctbx_list_t>? = nil
			for data in newValue {
				let sData:NSString = data as NSString
				cList = bctbx_list_append(cList, unsafeBitCast(sData.utf8String, to: UnsafeMutablePointer<CChar>.self))
			}
			linphone_account_params_set_supported_tags_list(cPtr, cList)
		}
	}
		
	/// Sets the transport type of the server address. 
	/// - Parameter transport: The ``TransportType`` to set. 
	
	/// Returns the transport type of the server address. 
	/// - Returns: The ``TransportType`` of the server address. 
	public var transport: TransportType
	{
	
		get
		{ 
						return TransportType(rawValue: Int(linphone_account_params_get_transport(cPtr).rawValue))!
		}
		set
		{
			linphone_account_params_set_transport(cPtr, LinphoneTransportType(rawValue: CUnsignedInt(newValue.rawValue)))
		}
	}
		
	/// Sets whether the account will unREGISTER when the core stops but only if the
	/// push notifications are not allowed for the account. 
	/// - Parameter unregister: true to unregister the account, false otherwise. 
	
	/// Gets whether the account will unREGISTER when the core stops but only if the
	/// push notifications are not allowed for the account. 
	/// - Returns: true if the account will unREGISTER at stop, false otherwise. 
	public var unregisterAtStopEnabled: Bool
	{
	
		get
		{ 
						return linphone_account_params_unregister_at_stop_enabled(cPtr) != 0
		}
		set
		{
			linphone_account_params_enable_unregister_at_stop(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// If enabled, the international prefix will automaticaly be used for calls and
	/// chats. 
	/// - Parameter enable: true to use the international prefix for calls, false
	/// otherwise. 
	
	/// Return whether or not the international prefix will automaticaly be used for
	/// calls and chats. 
	/// - Returns: Whether we should use international prefix automatically for calls. 
	public var useInternationalPrefixForCallsAndChats: Bool
	{
	
		get
		{ 
						return linphone_account_params_get_use_international_prefix_for_calls_and_chats(cPtr) != 0
		}
		set
		{
			linphone_account_params_set_use_international_prefix_for_calls_and_chats(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Assign a user pointer to the account params. 
	/// - Parameter userData: The user data.    
	
	/// Retrieve the user pointer associated with the account params. 
	/// - Returns: The user data.    
	public var userData: UnsafeMutableRawPointer?
	{
	
		get
		{ 
						return linphone_account_params_get_user_data(cPtr)
		}
		set
		{
			linphone_account_params_set_user_data(cPtr, newValue)
		}
	}
		
	/// Sets the Voicemail address. 
	/// - Parameter address: The Voicemail address.    
	
	/// Gets the Voicemail address. 
	/// - Returns: The Voicemail address.    
	public var voicemailAddress: Address?
	{
	
		get
		{ 
						let cPointer = linphone_account_params_get_voicemail_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_account_params_set_voicemail_address(cPtr, newValue?.cPtr)
		}
	}
		
	
	
	/// Set one custom parameter to this ``AccountParams``. 
	/// - Parameter key: key of the searched parameter.    
	/// - Parameter value: value of the searched parameter.    
	public func addCustomParam(key:String, value:String) 
	{
		linphone_account_params_add_custom_param(cPtr, key, value)
	}
	
	
	
	/// Instantiate a new account params with values from source. 
	/// - Returns: The newly created ``AccountParams`` object.    
	public func clone() -> AccountParams?
	{
		let cPointer = linphone_account_params_clone(cPtr)
		if (cPointer == nil) {
			return nil
		}
		let result = AccountParams.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Get the custom parameter with key to this ``AccountParams``. 
	/// - Parameter key: key of the searched parameter.    
	/// - Returns: The value of the parameter with key if found or an empty string
	/// otherwise.    
	public func getCustomParam(key:String) -> String
	{
		let cstr = linphone_account_params_get_custom_param(cPtr, key)
		let result = charArrayToString(charPointer: cstr)
		return result
	}
}


/// Object that represents a parsed SIP address. 
/// A SIP address is made of display name, username, domain name, port, and various
/// uri headers (such as tags). It looks like 'Alice <sip:alice@example.net>'.
/// A SIP address is made of display name, username, domain name, port, and various
/// uri headers (such as tags). It looks like 'Alice <sip:alice@example.net>'.
/// You can create an address using ``Factory/createAddress(addr:)`` or
/// ``Core/interpretUrl(url:applyInternationalPrefix:)`` and both will return a nil
/// object if it doesn't match the grammar defined by the standard.
/// This object is used in almost every other major objects to identity people
/// (including yourself) & servers.
/// The ``Address`` has methods to extract and manipulate all parts of the address. 
public class Address : LinphoneObject
{

	static public func getSwiftObject(cObject:OpaquePointer) -> Address {
		let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
		if (result != nil) {
			return Unmanaged<Address>.fromOpaque(result!).takeUnretainedValue()
		}
		let sObject = Address(cPointer: cObject)
		belle_sip_object_data_set(UnsafeMutablePointer(cObject), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(sObject).toOpaque()), nil)
		return sObject
	}

	public var getCobject: OpaquePointer? {
		return cPtr
	}

	
	///Enum describing Ip family. 
	public enum Family:Int
	{
		
		/// IpV4. 
		case Inet = 0
		/// IpV6. 
		case Inet6 = 1
		/// Unknown. 
		case Unspec = 2
	}
	
	/// Sets the display name. 
	/// - Parameter displayName: the display name to set.    
	
	/// Returns the display name. 
	/// - Returns: the display name if any, nil otherwise.    
	public var displayName: String?
	{
	
			
			let cPointer = linphone_address_get_display_name(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
	public func setDisplayname(newValue: String?) throws
	{
		let exception_result = linphone_address_set_display_name(cPtr, newValue)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "username setter returned value \(exception_result)")
		}
	}
		
	/// Sets the domain. 
	/// - Parameter domain: the domain to set.    
	
	/// Returns the domain name. 
	/// - Returns: the domain name if any, nil otherwise.    
	public var domain: String?
	{
	
			
			let cPointer = linphone_address_get_domain(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
	public func setDomain(newValue: String?) throws
	{
		let exception_result = linphone_address_set_domain(cPtr, newValue)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "username setter returned value \(exception_result)")
		}
	}
		
	
	/// returns whether the address is a routable SIP address or not 
	/// - Returns: true if it is a routable SIP address, false otherwise 
	public var isSip: Bool
	{
	
						return linphone_address_is_sip(cPtr) != 0

	}
		
	
	/// Returns if address is valid. 
	/// - Returns: true is the address is valid, false otherwise 
	public var isValid: Bool
	{
	
						return linphone_address_is_valid(cPtr) != 0

	}
		
	/// Set the value of the method parameter. 
	/// - Parameter methodParam: the value to set to the method parameter.    
	
	/// Get the value of the method parameter. 
	/// - Returns: the value of the parameter or nil.    
	public var methodParam: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_address_get_method_param(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_address_set_method_param(cPtr, newValue)
		}
	}
		
	/// Set the password encoded in the address. 
	/// It is used for basic authentication (not recommended). 
	/// - Parameter password: the password to set.    
	
	/// Get the password encoded in the address. 
	/// It is used for basic authentication (not recommended). 
	/// - Returns: the password if any, nil otherwise.    
	public var password: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_address_get_password(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_address_set_password(cPtr, newValue)
		}
	}
		
	/// Sets the port number. 
	/// - Parameter port: the port to set in the address 
	
	/// Get port number as an integer value, 0 if not present. 
	/// - Returns: the port set in the address or 0 if not present. 
	public var port: Int
	{
	
						return Int(linphone_address_get_port(cPtr))

	}
	public func setPort(newValue: Int) throws
	{
		let exception_result = linphone_address_set_port(cPtr, CInt(newValue))
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "username setter returned value \(exception_result)")
		}
	}
		
	
	/// Returns the address scheme, normally "sip". 
	/// - Returns: the scheme if any, nil otherwise.    
	public var scheme: String?
	{
	
			
			let cPointer = linphone_address_get_scheme(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	/// Make the address refer to a secure location (sips scheme) 
	/// - Parameter enabled: true if address is requested to be secure. 
	
	/// Returns whether the address refers to a secure location (sips) or not. 
	/// - Returns: true if address refers to a secure location, false otherwise 
	public var secure: Bool
	{
	
		get
		{ 
						return linphone_address_get_secure(cPtr) != 0
		}
		set
		{
			linphone_address_set_secure(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Set a transport. 
	/// - Parameter transport: a ``TransportType`` 
	
	/// Get the transport. 
	/// - Returns: a ``TransportType``, default value if not set is UDP. 
	public var transport: TransportType
	{
	
						return TransportType(rawValue: Int(linphone_address_get_transport(cPtr).rawValue))!

	}
	public func setTransport(newValue: TransportType) throws
	{
		let exception_result = linphone_address_set_transport(cPtr, LinphoneTransportType(rawValue: CUnsignedInt(newValue.rawValue)))
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "username setter returned value \(exception_result)")
		}
	}
		
	/// Set the value of the parameters of the URI of the address. 
	/// - Parameter params: The parameters string 
	
	public var uriParams: String = ""
	{
	
		willSet
		{
			linphone_address_set_uri_params(cPtr, newValue)
		}
	}
		
	/// Sets the username. 
	/// - Parameter username: the username to set.    
	
	/// Returns the username. 
	/// - Returns: the username name if any, nil otherwise.    
	public var username: String?
	{
	
			
			let cPointer = linphone_address_get_username(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
	public func setUsername(newValue: String?) throws
	{
		let exception_result = linphone_address_set_username(cPtr, newValue)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "username setter returned value \(exception_result)")
		}
	}
		
	
	
	/// Returns the address as a string. 
	/// The returned char * must be freed by the application. Use ms_free(). 
	/// - Returns: a string representation of the address.       
	public func asString() -> String
	{
		let cstr = linphone_address_as_string(cPtr)
		let result = charArrayToString(charPointer: cstr)
		if (cstr != nil) {
			bctbx_free(cstr)
		}
		return result
	}
	
	
	
	/// Returns the SIP uri only as a string, that is display name is removed. 
	/// The returned char * must be freed by the application. Use ms_free(). 
	/// - Returns: a string representation of the address.       
	public func asStringUriOnly() -> String
	{
		let cstr = linphone_address_as_string_uri_only(cPtr)
		let result = charArrayToString(charPointer: cstr)
		if (cstr != nil) {
			bctbx_free(cstr)
		}
		return result
	}
	
	
	
	/// Returns the SIP uri only as a string, that is display name is removed. 
	/// This function always returns a string whose URI parameters has a constant order
	/// allowing to easily compute hashes for example. The returned char * must be
	/// freed by the application. Use ms_free(). 
	/// - Returns: a string representation of the address.       
	public func asStringUriOnlyOrdered() -> String
	{
		let cstr = linphone_address_as_string_uri_only_ordered(cPtr)
		let result = charArrayToString(charPointer: cstr)
		if (cstr != nil) {
			bctbx_free(cstr)
		}
		return result
	}
	
	
	
	/// Removes address's tags and uri headers so that it is displayable to the user. 
	public func clean() 
	{
		linphone_address_clean(cPtr)
	}
	
	
	
	/// Clones a ``Address`` object. 
	/// - Returns: a new ``Address`` object.    
	public func clone() -> Address?
	{
		let cPointer = linphone_address_clone(cPtr)
		if (cPointer == nil) {
			return nil
		}
		let result = Address.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Compare two ``Address`` taking the tags and headers into account. 
	/// - Parameter address2: ``Address`` object.    
	/// - Returns: Boolean value telling if the ``Address`` objects are equal. 
	/// - See also: ``weakEqual(address2:)`` 
	public func equal(address2:Address) -> Bool
	{
		return linphone_address_equal(cPtr, address2.cPtr) != 0
	}
	
	
	
	/// Get the header encoded in the address. 
	/// - Parameter headerName: the header name.    
	/// - Returns: the header value or nil if it doesn't exists.    
	public func getHeader(headerName:String) -> String
	{
		let cstr = linphone_address_get_header(cPtr, headerName)
		let result = charArrayToString(charPointer: cstr)
		return result
	}
	
	
	
	/// Get the value of a parameter of the address. 
	/// - Parameter paramName: The name of the parameter.    
	/// - Returns: The value of the parameter or nil if it doesn't exists.    
	public func getParam(paramName:String) -> String
	{
		let cstr = linphone_address_get_param(cPtr, paramName)
		let result = charArrayToString(charPointer: cstr)
		return result
	}
	
	
	
	/// Get the value of a parameter of the URI of the address. 
	/// - Parameter uriParamName: The name of the parameter.    
	/// - Returns: The value of the parameter or nil if it doesn't exists.    
	public func getUriParam(uriParamName:String) -> String
	{
		let cstr = linphone_address_get_uri_param(cPtr, uriParamName)
		let result = charArrayToString(charPointer: cstr)
		return result
	}
	
	
	
	/// Tell whether a parameter is present in the address. 
	/// - Parameter paramName: The name of the parameter.    
	/// - Returns: A boolean value telling whether the parameter is present in the
	/// address 
	public func hasParam(paramName:String) -> Bool
	{
		return linphone_address_has_param(cPtr, paramName) != 0
	}
	
	
	
	/// Tell whether a parameter is present in the URI of the address. 
	/// - Parameter uriParamName: The name of the parameter.    
	/// - Returns: A boolean value telling whether the parameter is present in the URI
	/// of the address 
	public func hasUriParam(uriParamName:String) -> Bool
	{
		return linphone_address_has_uri_param(cPtr, uriParamName) != 0
	}
	
	
	
	/// Compare two addresses. 
	/// - Parameter other: an other ``Address`` object.    
	public func lesser(other:Address) -> Bool
	{
		return linphone_address_lesser(cPtr, other.cPtr) != 0
	}
	
	
	
	/// Removes the value of a parameter of the URI of the address. 
	/// - Parameter uriParamName: The name of the parameter.    
	public func removeUriParam(uriParamName:String) 
	{
		linphone_address_remove_uri_param(cPtr, uriParamName)
	}
	
	
	
	/// Set a header into the address. 
	/// Headers appear in the URI with '?', such as
	/// <sip:test@linphone.org?SomeHeader=SomeValue>. 
	/// Headers appear in the URI with '?', such as
	/// <sip:test@linphone.org?SomeHeader=SomeValue>. 
	/// - Parameter headerName: the header name.    
	/// - Parameter headerValue: the header value.    
	public func setHeader(headerName:String, headerValue:String?) 
	{
		linphone_address_set_header(cPtr, headerName, headerValue)
	}
	
	
	
	/// Set the value of a parameter of the address. 
	/// - Parameter paramName: The name of the parameter.    
	/// - Parameter paramValue: The new value of the parameter.    
	public func setParam(paramName:String, paramValue:String?) 
	{
		linphone_address_set_param(cPtr, paramName, paramValue)
	}
	
	
	
	/// Set the value of a parameter of the URI of the address. 
	/// - Parameter uriParamName: The name of the parameter.    
	/// - Parameter uriParamValue: The new value of the parameter.    
	public func setUriParam(uriParamName:String, uriParamValue:String?) 
	{
		linphone_address_set_uri_param(cPtr, uriParamName, uriParamValue)
	}
	
	
	
	/// Compare two ``Address`` ignoring tags and headers, basically just domain,
	/// username, and port. 
	/// - Parameter address2: ``Address`` object.    
	/// - Returns: Boolean value telling if the ``Address`` objects are equal. 
	/// - See also: ``equal(address2:)`` 
	public func weakEqual(address2:Address) -> Bool
	{
		return linphone_address_weak_equal(cPtr, address2.cPtr) != 0
	}
}


/// Object that represents an alert. 
/// Alerts are raised at run-time when particular conditions are met, for example
/// bad network quality. The full list of available alert types is described by the
/// ``Kind`` enum. An application is notified of new alerts through the
/// ``CoreDelegate`` interface. Once raised, the application may use the
/// ``AlertDelegate`` interface to get notified when the alert stops. For each kind
/// of alert, a ``Dictionary`` is filled with relevant informations, returned by
/// ``getInformations()``. The keys available are documented per-type in ``Kind``
/// enum. 
public class Alert : LinphoneObject
{
	var delegateManagers : [AlertDelegateManager] = []

	static public func getSwiftObject(cObject:OpaquePointer) -> Alert {
		let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
		if (result != nil) {
			return Unmanaged<Alert>.fromOpaque(result!).takeUnretainedValue()
		}
		let sObject = Alert(cPointer: cObject)
		belle_sip_object_data_set(UnsafeMutablePointer(cObject), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(sObject).toOpaque()), nil)
		return sObject
	}

	public var getCobject: OpaquePointer? {
		return cPtr
	}

	
	///All kinds of alerts. 
	public enum Kind:Int
	{
		
		/// Camera is not working. 
		case QoSCameraMisfunction = 0
		/// Camera is capturing low framerate. 
		case QoSCameraLowFramerate = 1
		/// Video decoding has stopped for a given period (10 s by default). 
		case QoSVideoStalled = 2
		/// A received media stream suffers from high loss or late rate. 
		case QoSHighLossLateRate = 3
		/// A report of high loss rate is received from remote party. 
		case QoSHighRemoteLossRate = 4
		/// Packet Burst phenomenon. 
		case QoSBurstOccured = 5
		/// Loss rate is significant but retransmissions fail to arrive on time. 
		case QoSRetransmissionFailures = 6
		/// Low bandwidth detected. 
		case QoSLowDownloadBandwidthEstimation = 7
		/// Low quality (bitrate) video received. 
		case QoSLowQualityReceivedVideo = 8
		/// Low quality video is being sent. 
		case QoSLowQualitySentVideo = 9
		/// The operating system reports a low radio signal (wifi or mobile) 
		case QoSLowSignal = 10
		/// The operating system reports a loss of radio signal (wifi or mobile). 
		case QoSLostSignal = 11
	}
	
	
	
	public func addDelegate(delegate: AlertDelegate)
	{
		let manager = AlertDelegateManager()
		manager.delegate = delegate
		delegateManagers.append(manager)
		linphone_alert_add_callbacks(cPtr, manager.cPtr)
		belle_sip_object_unref(UnsafeMutableRawPointer(manager.cPtr))
		belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(self).toOpaque()), nil)
	}	
	
	
	public func removeDelegate(delegate: AlertDelegate)
	{
		if let index = delegateManagers.firstIndex(where: { $0.delegate === delegate as AnyObject}) {
			linphone_alert_remove_callbacks(cPtr, delegateManagers[index].cPtr)
			delegateManagers.remove(at: index)
			belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(self).toOpaque()), nil)
		}
	}	
	
	
	/// Provide a string describing the alert type. 
	/// - Parameter type: the ``Kind`` 
	/// - Returns: a string 
	static public func typeToString(type:Alert.Kind) -> String
	{
		let cstr = linphone_alert_type_to_string(LinphoneAlertType(rawValue: CUnsignedInt(type.rawValue)))
		let result = charArrayToString(charPointer: cstr)
		return result
	}
	
	
	/// Return the call from the alert. 
	/// - Returns: A ``Call`` from the alert.    
	public var call: Call?
	{
	
						let cPointer = linphone_alert_get_call(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Call.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Gets the current ``AlertDelegate``. 
	/// This is meant only to be called from a callback to be able to get the user_data
	/// associated with the ``AlertDelegate`` that is calling the callback. 
	/// - Returns: The ``AlertDelegate`` that has called the last callback.    
	public var currentDelegate: AlertDelegate?
	{
	
			let cObject = linphone_alert_get_current_callbacks(cPtr)
			let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
			if (result != nil) {
				return Unmanaged<AlertDelegateManager>.fromOpaque(result!).takeUnretainedValue().delegate
				}
			return nil
			
	}
		
	
	/// Return the end time of the alert. 
	/// - Returns: the end time of the alert. 
	public var endTime: time_t
	{
	
						return linphone_alert_get_end_time(cPtr)

	}
		
	
	/// Return more informations about the alerts. 
	/// - Returns: A ``Dictionary`` containing informations about the current alert.    
	public var informations: Dictionary?
	{
	
						let cPointer = linphone_alert_get_informations(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Dictionary.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Return the start time of the alert. 
	/// - Returns: the start time of the alert. 
	public var startTime: time_t
	{
	
						return linphone_alert_get_start_time(cPtr)

	}
		
	
	/// Return the state of the alert. 
	/// - Returns: true if and only if the alert is active. 
	public var state: Bool
	{
	
						return linphone_alert_get_state(cPtr) != 0

	}
		
	
	/// Return the type of the alert. 
	/// - Returns: A ``Kind`` corresponding to the current alert. 
	public var type: Alert.Kind
	{
	
						return Alert.Kind(rawValue: Int(linphone_alert_get_type(cPtr).rawValue))!

	}
		
	
	
	/// Clone the given alert. 
	/// - Returns: A new alert with exactly same informations that param.    
	public func clone() -> Alert?
	{
		let cPointer = linphone_alert_clone(cPtr)
		if (cPointer == nil) {
			return nil
		}
		let result = Alert.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Notify the alert if it is terminated. 
	public func notifyOnTerminated() 
	{
		linphone_alert_notify_on_terminated(cPtr)
	}
}


/// Object holding audio device information. 
/// It contains the name of the device, it's type if available (Earpiece, Speaker,
/// Bluetooth, etc..) and capabilities (input, output or both) the name of the
/// driver that created it (filter in mediastreamer).
/// You can use the ``AudioDevice`` objects to configure default input/output
/// devices or do it dynamically during a call.
/// To get the list of available devices, use ``Core/getAudioDevices()``. This list
/// will be limited to one device of each type. Use
/// ``Core/getExtendedAudioDevices()`` for a complete list. 
public class AudioDevice : LinphoneObject
{

	static public func getSwiftObject(cObject:OpaquePointer) -> AudioDevice {
		let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
		if (result != nil) {
			return Unmanaged<AudioDevice>.fromOpaque(result!).takeUnretainedValue()
		}
		let sObject = AudioDevice(cPointer: cObject)
		belle_sip_object_data_set(UnsafeMutablePointer(cObject), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(sObject).toOpaque()), nil)
		return sObject
	}

	public var getCobject: OpaquePointer? {
		return cPtr
	}

	
	///``Capabilities`` enum represents whether a device can record audio, play audio
	///or both 
	public struct Capabilities:OptionSet
	{
		public let rawValue: Int

		public init(rawValue: Int) {
			self.rawValue = rawValue
		}

		
		/// Can record audio. 
		public static let CapabilityRecord = Capabilities(rawValue: 1<<0)
		/// Can play audio. 
		public static let CapabilityPlay = Capabilities(rawValue: 1<<1)
		/// Can play and record audio. 
		public static let CapabilityAll = Capabilities(rawValue: 3)
	}

	
	///``Kind`` enum represents the different types of an audio device. 
	public enum Kind:Int
	{
		
		/// Unknown. 
		case Unknown = 0
		/// Microphone. 
		case Microphone = 1
		/// Earpiece. 
		case Earpiece = 2
		/// Speaker. 
		case Speaker = 3
		/// Bluetooth. 
		case Bluetooth = 4
		/// Bluetooth A2DP. 
		case BluetoothA2DP = 5
		/// Telephony. 
		case Telephony = 6
		/// AuxLine. 
		case AuxLine = 7
		/// GenericUsb. 
		case GenericUsb = 8
		/// Headset. 
		case Headset = 9
		/// Headphones. 
		case Headphones = 10
		/// Hearing Aid. 
		case HearingAid = 11
		/// HDMI. 
		case Hdmi = 12
	}
	
	
	/// Returns the capabilities of the device. 
	/// - Returns: the ``Capabilities`` of the audio device (RECORD, PLAY or both) as a
	/// bit mask 
	public var capabilities: AudioDevice.Capabilities
	{
	
						return AudioDevice.Capabilities(rawValue: Int(linphone_audio_device_get_capabilities(cPtr).rawValue))

	}
		
	
	/// Returns the name of the audio device. 
	/// - Returns: the name of the audio device.    
	public var deviceName: String
	{
	
			
			let cPointer = linphone_audio_device_get_device_name(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Returns the driver name used by the device. 
	/// - Returns: the name of the driver used by this audio device.    
	public var driverName: String
	{
	
			
			let cPointer = linphone_audio_device_get_driver_name(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Returns whether the audio device automatically follows the system's audio
	/// routing policy. 
	/// This capability is available on some system (typically iOS) and might be
	/// convenient to simply specify liblinphone to let the system decide about which
	/// audio route is being used to handle a call. The actual ``Kind`` may be unknown
	/// at some point, typically when no calls are running, otherwise it is reflected
	/// to be the actual system's audio route. 
	/// - Returns: true if the audio device automatically follows the system audio
	/// routing policy. 
	public var followsSystemRoutingPolicy: Bool
	{
	
						return linphone_audio_device_get_follows_system_routing_policy(cPtr) != 0

	}
		
	
	/// Returns the id of the audio device. 
	/// - Returns: the id of the audio device.    
	public var id: String
	{
	
			
			let cPointer = linphone_audio_device_get_id(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Returns the type of the device. 
	/// - Returns: the ``Kind`` of the audio device (microphone, speaker, earpiece,
	/// bluetooth, etc...) 
	public var type: AudioDevice.Kind
	{
	
						return AudioDevice.Kind(rawValue: Int(linphone_audio_device_get_type(cPtr).rawValue))!

	}
		
	/// Tells the audio device to be used to play the ringtone or not. 
	/// - Parameter useForRinging: Whether the audio device must be used to play the
	/// ringtone or not. 
	
	/// Returns whether the audio device is used to play the ringtone or not. 
	/// - Returns: Whether the audio device is used to play the ringtone or not. 
	public var useForRinging: Bool
	{
	
		get
		{ 
						return linphone_audio_device_get_use_for_ringing(cPtr) != 0
		}
		set
		{
			linphone_audio_device_set_use_for_ringing(cPtr, newValue==true ? 1:0)
		}
	}
		
	
	
	/// Returns whether or not the audio device has the given capability. 
	/// - Parameter capability: the ``Capabilities`` to check 
	/// - Returns: true if the audio device has the capability, false otherwise 
	public func hasCapability(capability:AudioDevice.Capabilities) -> Bool
	{
		return linphone_audio_device_has_capability(cPtr, LinphoneAudioDeviceCapabilities(rawValue: CUnsignedInt(capability.rawValue))) != 0
	}
}


/// Object holding authentication information. 
/// In most case, authentication information consists of a username and password.
/// If realm isn't set, it will be deduced automatically from the first
/// authentication challenge as for the hash algorithm. Sometimes, a userid is
/// required by the proxy and then domain can be useful to discriminate different
/// credentials. You can also use this object if you need to use a client
/// certificate.
/// Once created and filled, a ``AuthInfo`` must be added to the ``Core`` in order
/// to become known and used automatically when needed. Use
/// ``Core/addAuthInfo(info:)`` for that purpose.
/// The ``Core`` object can take the initiative to request authentication
/// information when needed to the application through the
/// authentication_requested() callback of it's ``CoreDelegate``.
/// The application can respond to this information request later using
/// ``Core/addAuthInfo(info:)``. This will unblock all pending authentication
/// transactions and retry them with authentication headers. 
public class AuthInfo : LinphoneObject
{

	static public func getSwiftObject(cObject:OpaquePointer) -> AuthInfo {
		let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
		if (result != nil) {
			return Unmanaged<AuthInfo>.fromOpaque(result!).takeUnretainedValue()
		}
		let sObject = AuthInfo(cPointer: cObject)
		belle_sip_object_data_set(UnsafeMutablePointer(cObject), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(sObject).toOpaque()), nil)
		return sObject
	}

	public var getCobject: OpaquePointer? {
		return cPtr
	}
	
	/// Set an access token to authenticate, to be used when the server supports the
	/// "bearer" authentication method. 
	/// The ``AuthInfo`` takes a reference to the supplied token, it is not making any
	/// copy. 
	/// - Parameter token: The ``BearerToken`` object representing the access token.    
	
	/// Return a previously set access token. 
	/// - Returns: the access token, as a ``BearerToken`` object    
	public var accessToken: BearerToken?
	{
	
		get
		{ 
						let cPointer = linphone_auth_info_get_access_token(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = BearerToken.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_auth_info_set_access_token(cPtr, newValue?.cPtr)
		}
	}
		
	/// Sets the algorithm to use. 
	/// - Parameter algorithm: The algorithm.    
	
	/// Gets the algorithm. 
	/// - Returns: The algorithm.    
	public var algorithm: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_auth_info_get_algorithm(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_auth_info_set_algorithm(cPtr, newValue)
		}
	}
		
	/// Set the authorization server uri. 
	/// - Parameter uri: the authorization server uri.    
	
	/// Get the previously set authorization server uri. 
	/// - Returns: the authorization server uri.    
	public var authorizationServer: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_auth_info_get_authorization_server(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_auth_info_set_authorization_server(cPtr, newValue)
		}
	}
		
	/// Sets the available algorithms list without testing unicity. 
	/// - Parameter algorithms: The available algorithms list.      
	
	/// Gets all available algorithms. 
	/// - Returns: A list of available algorithms.      
	public var availableAlgorithms: [String]
	{
	
		get
		{ 
						var swiftList = [String]()
			let cList = linphone_auth_info_get_available_algorithms(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				swiftList.append(String(cString: unsafeBitCast(listTemp!.pointee.data, to: UnsafePointer<CChar>.self)))
				listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp!.pointee.next)
			}
				bctbx_list_free(cList)
			return swiftList
		}
		set
		{
			var cList: UnsafeMutablePointer<bctbx_list_t>? = nil
			for data in newValue {
				let sData:NSString = data as NSString
				cList = bctbx_list_append(cList, unsafeBitCast(sData.utf8String, to: UnsafeMutablePointer<CChar>.self))
			}
			linphone_auth_info_set_available_algorithms(cPtr, cList)
		}
	}
		
	/// Set the OAUTH2 client_id. 
	/// The client_id may be used to renew access token from refresh token. If a
	/// client_secret is required, it has to be set through
	/// ``setClientSecret(clientSecret:)``. - See also: ``setRefreshToken(token:)`` 
	/// - Parameter clientId: the client_id.    
	
	/// Get the previously set OAUTH2 client_id. 
	/// - Returns: the client_id.    
	public var clientId: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_auth_info_get_client_id(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_auth_info_set_client_id(cPtr, newValue)
		}
	}
		
	/// Set the OAUTH2 client_secret. 
	/// The client_secret may be used to renew access token from refresh token.
	/// - See also: ``setRefreshToken(token:)`` 
	/// - Parameter clientSecret: the client_secret.    
	
	/// Get the previously set OAUTH2 client_secret. 
	/// - Returns: the client_secret.    
	public var clientSecret: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_auth_info_get_client_secret(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_auth_info_set_client_secret(cPtr, newValue)
		}
	}
		
	/// Sets the domain for which this authentication is valid. 
	/// - Parameter domain: The domain.    This should not be necessary because realm
	/// is supposed to be unique and sufficient. However, many SIP servers don't set
	/// realm correctly, then domain has to be used to distinguish between several SIP
	/// account bearing the same username. 
	
	/// Gets the domain. 
	/// - Returns: The domain.    
	public var domain: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_auth_info_get_domain(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_auth_info_set_domain(cPtr, newValue)
		}
	}
		
	/// Set the expiration time for the current authentication information. 
	/// - Parameter expires: The new expiration time in seconds. Use 0 to indicate no
	/// expiration. 
	
	/// Get the expiration time for the current authentication information. 
	/// - Returns: The expiration time as a number of seconds since the Epoch,
	/// 1970-01-01 00:00:00 +0000 (UTC) 
	public var expires: time_t
	{
	
		get
		{ 
						return linphone_auth_info_get_expires(cPtr)
		}
		set
		{
			linphone_auth_info_set_expires(cPtr, newValue)
		}
	}
		
	/// Sets the ha1. 
	/// - Parameter ha1: The ha1.    
	
	/// Gets the ha1. 
	/// - Returns: The ha1.    
	public var ha1: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_auth_info_get_ha1(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_auth_info_set_ha1(cPtr, newValue)
		}
	}
		
	/// Sets the password. 
	/// - Parameter password: The password.    
	
	/// Gets the password. 
	/// - Returns: The password.    
	public var password: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_auth_info_get_password(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_auth_info_set_password(cPtr, newValue)
		}
	}
		
	/// Sets the realm. 
	/// - Parameter realm: The realm.    
	
	/// Gets the realm. 
	/// - Returns: The realm.    
	public var realm: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_auth_info_get_realm(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_auth_info_set_realm(cPtr, newValue)
		}
	}
		
	/// Set a refresh token to use when the access token set with
	/// ``setAccessToken(token:)`` expires. 
	/// The ``AuthInfo`` takes a reference to the supplied token, it is not making any
	/// copy. Provided that the token endpoint uri was assigned with
	/// ``setTokenEndpointUri(uri:)``, once the ``AuthInfo`` is pushed in the ``Core``,
	/// ``AuthInfo`` object automatically obtains new access token when needed, from
	/// the supplied refresh token. 
	/// - Parameter token: The ``BearerToken`` object representing the refresh token.  
	///  
	
	/// Return a previously set refresh token. 
	/// - Returns: the refresh token, as a ``BearerToken`` object.    
	public var refreshToken: BearerToken?
	{
	
		get
		{ 
						let cPointer = linphone_auth_info_get_refresh_token(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = BearerToken.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_auth_info_set_refresh_token(cPtr, newValue?.cPtr)
		}
	}
		
	/// Sets the TLS certificate. 
	/// - Parameter tlsCert: The TLS certificate.    
	
	/// Gets the TLS certificate. 
	/// - Returns: The TLS certificate.    
	public var tlsCert: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_auth_info_get_tls_cert(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_auth_info_set_tls_cert(cPtr, newValue)
		}
	}
		
	/// Sets the TLS certificate path. 
	/// - Parameter tlsCertPath: The TLS certificate path.    
	
	/// Gets the TLS certificate path. 
	/// - Returns: The TLS certificate path.    
	public var tlsCertPath: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_auth_info_get_tls_cert_path(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_auth_info_set_tls_cert_path(cPtr, newValue)
		}
	}
		
	/// Sets the TLS key. 
	/// - Parameter tlsKey: The TLS key.    
	
	/// Gets the TLS key. 
	/// - Returns: The TLS key.    
	public var tlsKey: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_auth_info_get_tls_key(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_auth_info_set_tls_key(cPtr, newValue)
		}
	}
		
	/// Sets the TLS key path. 
	/// - Parameter tlsKeyPath: The TLS key path.    
	
	/// Gets the TLS key path. 
	/// - Returns: The TLS key path.    
	public var tlsKeyPath: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_auth_info_get_tls_key_path(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_auth_info_set_tls_key_path(cPtr, newValue)
		}
	}
		
	/// Set the token endpoint https uri (OAUTH2). 
	/// The token endpoint uri is used to renew access token from refresh token. - See
	/// also: ``setRefreshToken(token:)`` 
	/// - Parameter uri: the token endpoint uri.    
	
	/// Get the previously set token endpoint https uri (OAUTH2). 
	/// - Returns: the token endpoint uri.    
	public var tokenEndpointUri: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_auth_info_get_token_endpoint_uri(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_auth_info_set_token_endpoint_uri(cPtr, newValue)
		}
	}
		
	/// Sets the user ID. 
	/// - Parameter userId: The userid.   
	
	/// Gets the user id. 
	/// - Returns: The user id.    
	public var userid: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_auth_info_get_userid(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_auth_info_set_userid(cPtr, newValue)
		}
	}
		
	/// Sets the username. 
	/// - Parameter username: The username.    
	
	/// Gets the username. 
	/// - Returns: The username.    
	public var username: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_auth_info_get_username(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_auth_info_set_username(cPtr, newValue)
		}
	}
		
	
	
	/// Add an unique algorithm in the the available algorithms list : Algorithms that
	/// already exist will not be added. 
	/// - Parameter algorithm: The algorithm to add.    
	public func addAvailableAlgorithm(algorithm:String?) 
	{
		linphone_auth_info_add_available_algorithm(cPtr, algorithm)
	}
	
	
	
	/// Remove all algorithms from the available algorithms list. 
	public func clearAvailableAlgorithms() 
	{
		linphone_auth_info_clear_available_algorithms(cPtr)
	}
	
	
	
	/// Instantiates a new auth info with values from source. 
	/// - Returns: The newly created ``AuthInfo`` object.    
	public func clone() -> AuthInfo?
	{
		let cPointer = linphone_auth_info_clone(cPtr)
		if (cPointer == nil) {
			return nil
		}
		let result = AuthInfo.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Check if Authinfos are the same without taking account algorithms. 
	/// - Parameter authInfo2: The second ``AuthInfo`` object.    
	/// - Returns: true if all fields (Username, UserId, Realm, Domain) are the same. 
	public func isEqualButAlgorithms(authInfo2:AuthInfo?) -> Bool
	{
		return linphone_auth_info_is_equal_but_algorithms(cPtr, authInfo2?.cPtr) != 0
	}
}


/// Object that represents a bearer token (eg OAUTH). 
/// SIP servers may support "bearer" kind of authentication, in which case an
/// authentication token needs to be supplied in order to authenticate to the SIP
/// service. Applications are responsible to obtain the token from an
/// authentication server. In order to pass it to liblinphone for usage, the token
/// needs to be encapsulated into a ``BearerToken``, together with its expiration
/// time and target server name for which it is intended to use, then passed into a
/// ``AuthInfo`` object. Both access and refresh tokens may be represented. If both
/// are provided to the ``AuthInfo``, then liblinphone automatically uses the
/// refresh token to obtain a new access token when the latter is expired. 
public class BearerToken : LinphoneObject
{

	static public func getSwiftObject(cObject:OpaquePointer) -> BearerToken {
		let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
		if (result != nil) {
			return Unmanaged<BearerToken>.fromOpaque(result!).takeUnretainedValue()
		}
		let sObject = BearerToken(cPointer: cObject)
		belle_sip_object_data_set(UnsafeMutablePointer(cObject), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(sObject).toOpaque()), nil)
		return sObject
	}

	public var getCobject: OpaquePointer? {
		return cPtr
	}
	
	
	/// Get the token exiration time, as a number of seconds since EPOCH. 
	/// - Returns: the expiration time 
	public var expirationTime: time_t
	{
	
						return linphone_bearer_token_get_expiration_time(cPtr)

	}
		
	
	/// Get the token as a string. 
	/// - Returns: the token.    
	public var token: String
	{
	
			
			let cPointer = linphone_bearer_token_get_token(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
	}


/// The object representing a data buffer. 
public class Buffer : LinphoneObject
{

	static public func getSwiftObject(cObject:OpaquePointer) -> Buffer {
		let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
		if (result != nil) {
			return Unmanaged<Buffer>.fromOpaque(result!).takeUnretainedValue()
		}
		let sObject = Buffer(cPointer: cObject)
		belle_sip_object_data_set(UnsafeMutablePointer(cObject), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(sObject).toOpaque()), nil)
		return sObject
	}

	public var getCobject: OpaquePointer? {
		return cPtr
	}
	
	
	
	/// Create a new ``Buffer`` object from existing data. 
	/// - Parameter data: The initial data to store in the ``Buffer``.    
	/// - Parameter size: The size of the initial data to store in the ``Buffer``. 
	/// - Returns: A new ``Buffer`` object.    
	static public func newFromData(data:UnsafePointer<UInt8>, size:Int) -> Buffer?
	{
		let cPointer = linphone_buffer_new_from_data(data, size)
		if (cPointer == nil) {
			return nil
		}
		let result = Buffer.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Create a new ``Buffer`` object from a string. 
	/// - Parameter data: The initial string content of the ``Buffer``.    
	/// - Returns: A new ``Buffer`` object.    
	static public func newFromString(data:String) -> Buffer?
	{
		let cPointer = linphone_buffer_new_from_string(data)
		if (cPointer == nil) {
			return nil
		}
		let result = Buffer.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	/// Get the content of the data buffer. 
	/// - Returns: The content of the data buffer.    
	public var content: UnsafePointer<UInt8>
	{
	
						return linphone_buffer_get_content(cPtr)

	}
		
	
	/// Tell whether the ``Buffer`` is empty. 
	/// - Returns: A boolean value telling whether the ``Buffer`` is empty or not. 
	public var isEmpty: Bool
	{
	
						return linphone_buffer_is_empty(cPtr) != 0

	}
		
	/// Set the size of the content of the data buffer. 
	/// - Parameter size: The size of the content of the data buffer. 
	
	/// Get the size of the content of the data buffer. 
	/// - Returns: The size of the content of the data buffer. 
	public var size: Int
	{
	
		get
		{ 
						return Int(linphone_buffer_get_size(cPtr))
		}
		set
		{
			linphone_buffer_set_size(cPtr, (newValue))
		}
	}
		
	/// Set the string content of the data buffer. 
	/// - Parameter content: The string content of the data buffer.    
	
	/// Get the string content of the data buffer. 
	/// - Returns: The string content of the data buffer.    
	public var stringContent: String
	{
	
		get
		{ 
			
			let cPointer = linphone_buffer_get_string_content(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_buffer_set_string_content(cPtr, newValue)
		}
	}
		
	/// Assign a user pointer to the buffer. 
	/// - Parameter userData: The user pointer to associate with the buffer.    
	
	/// Retrieve the user pointer associated with the buffer. 
	/// - Returns: The user pointer associated with the buffer.    
	public var userData: UnsafeMutableRawPointer?
	{
	
		get
		{ 
						return linphone_buffer_get_user_data(cPtr)
		}
		set
		{
			linphone_buffer_set_user_data(cPtr, newValue)
		}
	}
		
	
	
	/// Set the content of the data buffer. 
	/// - Parameter content: The content of the data buffer.    
	/// - Parameter size: The size of the content of the data buffer. 
	public func setContent(content:UnsafePointer<UInt8>, size:Int) 
	{
		linphone_buffer_set_content(cPtr, content, size)
	}
}


/// This object represents a call issued or received by the ``Core``. 
/// Linphone only allows at most one active call at any given time and it will be
/// in ``StreamsRunning``. However, if the core is locally hosting a
/// ``Conference``, you may have some or all the calls in the conference in
/// ``StreamsRunning`` as well as an additional active call outside of the
/// conference in ``StreamsRunning`` if the local participant of the ``Conference``
/// is not part of it.
/// You can get the ``State`` of the call using ``getState()``, it's current
/// ``CallParams`` with ``getCurrentParams()`` and the latest statistics by calling
/// ``getAudioStats()`` or ``getVideoStats()``.
/// The application can receive the various kind of events occuring in a call
/// through the ``CallDelegate`` interface, see also ``addDelegate(cbs:)``. 
public class Call : LinphoneObject
{
	var delegateManagers : [CallDelegateManager] = []

	static public func getSwiftObject(cObject:OpaquePointer) -> Call {
		let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
		if (result != nil) {
			return Unmanaged<Call>.fromOpaque(result!).takeUnretainedValue()
		}
		let sObject = Call(cPointer: cObject)
		belle_sip_object_data_set(UnsafeMutablePointer(cObject), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(sObject).toOpaque()), nil)
		return sObject
	}

	public var getCobject: OpaquePointer? {
		return cPtr
	}

	
	///Enum representing the status of a call. 
	public enum Status:Int
	{
		
		/// The call was sucessful. 
		case Success = 0
		/// The call was aborted (caller hanged up) 
		case Aborted = 1
		/// The call was missed (incoming call timed out without being answered or hanged
		/// up) 
		case Missed = 2
		/// The call was declined, either locally or by remote end. 
		case Declined = 3
		/// The call was aborted before being advertised to the application - for protocol
		/// reasons. 
		case EarlyAborted = 4
		/// The call was answered on another device. 
		case AcceptedElsewhere = 5
		/// The call was declined on another device. 
		case DeclinedElsewhere = 6
	}

	
	///Enum representing the direction of a call. 
	public enum Dir:Int
	{
		
		/// outgoing calls 
		case Outgoing = 0
		/// incoming calls 
		case Incoming = 1
	}

	
	///``State`` enum represents the different states a call can reach into. 
	public enum State:Int
	{
		
		/// Initial state. 
		case Idle = 0
		/// Incoming call received. 
		case IncomingReceived = 1
		/// PushIncoming call received. 
		case PushIncomingReceived = 2
		/// Outgoing call initialized. 
		case OutgoingInit = 3
		/// Outgoing call in progress. 
		case OutgoingProgress = 4
		/// Outgoing call ringing. 
		case OutgoingRinging = 5
		/// Outgoing call early media. 
		case OutgoingEarlyMedia = 6
		/// Connected. 
		case Connected = 7
		/// Streams running. 
		case StreamsRunning = 8
		/// Pausing. 
		case Pausing = 9
		/// Paused. 
		case Paused = 10
		/// Resuming. 
		case Resuming = 11
		/// Referred. 
		case Referred = 12
		/// Error. 
		case Error = 13
		/// Call end. 
		case End = 14
		/// Paused by remote. 
		case PausedByRemote = 15
		/// The call's parameters are updated for example when video is asked by remote. 
		case UpdatedByRemote = 16
		/// We are proposing early media to an incoming call. 
		case IncomingEarlyMedia = 17
		/// We have initiated a call update. 
		case Updating = 18
		/// The call object is now released. 
		case Released = 19
		/// The call is updated by remote while not yet answered (SIP UPDATE in early
		/// dialog received) 
		case EarlyUpdatedByRemote = 20
		/// We are updating the call while not yet answered (SIP UPDATE in early dialog
		/// sent) 
		case EarlyUpdating = 21
	}
	
	
	
	public func addDelegate(delegate: CallDelegate)
	{
		let manager = CallDelegateManager()
		manager.delegate = delegate
		delegateManagers.append(manager)
		linphone_call_add_callbacks(cPtr, manager.cPtr)
		belle_sip_object_unref(UnsafeMutableRawPointer(manager.cPtr))
		belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(self).toOpaque()), nil)
	}	
	
	
	public func removeDelegate(delegate: CallDelegate)
	{
		if let index = delegateManagers.firstIndex(where: { $0.delegate === delegate as AnyObject}) {
			linphone_call_remove_callbacks(cPtr, delegateManagers[index].cPtr)
			delegateManagers.remove(at: index)
			belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(self).toOpaque()), nil)
		}
	}	
	
	/// Returns a copy of the call statistics for the audio stream. 
	/// - Returns: a ``CallStats`` object for the audio stream or nil if it isn't
	/// available.       
	public var audioStats: CallStats?
	{
	
						let cPointer = linphone_call_get_audio_stats(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = CallStats.getSwiftObject(cObject:cPointer!)
			belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
			return result

	}
		
	
	/// Returns the ZRTP authentication token to verify. 
	/// - Returns: the authentication token to verify or nil if ZRTP isn't enabled.    
	public var authenticationToken: String?
	{
	
			
			let cPointer = linphone_call_get_authentication_token(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	/// Sets the result of ZRTP short code verification by user. 
	/// If remote party also does the same, it will update the ZRTP cache so that
	/// user's verification will not be required for the two users. 
	/// - Parameter verified: whether the ZRTP SAS is verified. 
	
	/// Returns whether ZRTP authentication token is verified. 
	/// If not, it must be verified by users as described in ZRTP procedure. Once done,
	/// the application must inform of the results with
	/// ``setAuthenticationTokenVerified(verified:)``. 
	/// - Returns: true if authentication token is verifed, false otherwise. 
	public var authenticationTokenVerified: Bool
	{
	
		get
		{ 
						return linphone_call_get_authentication_token_verified(cPtr) != 0
		}
		set
		{
			linphone_call_set_authentication_token_verified(cPtr, newValue==true ? 1:0)
		}
	}
		
	
	/// Returns call quality averaged over all the duration of the call. 
	/// See ``getCurrentQuality()`` for more details about quality measurement. 
	/// - Returns: the call average quality since tbe beginning of the call. 
	public var averageQuality: Float
	{
	
						return linphone_call_get_average_quality(cPtr)

	}
		
	/// Indicates whether received Baudot tones should be detected. 
	/// The Baudot functionality is to be enabled first by calling
	/// ``Core/enableBaudot(enabled:)``. 
	/// - Parameter enabled: wether or not to detect received Baudot tones. 
	
	public var baudotDetectionEnabled: Bool?
	{
	
		willSet
		{
			linphone_call_enable_baudot_detection(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Defines the Baudot mode for the call. 
	/// The Baudot functionality is to be enabled first by calling
	/// ``Core/enableBaudot(enabled:)``. 
	/// - Parameter mode: The Baudot mode to use for the call. 
	
	public var baudotMode: BaudotMode?
	{
	
		willSet
		{
			linphone_call_set_baudot_mode(cPtr, LinphoneBaudotMode(rawValue: CUnsignedInt(newValue!.rawValue)))
		}
	}
		
	/// Set the Baudot significant pause timeout after which a LETTERS tone is
	/// retransmitted before resuming transmission (in seconds). 
	/// Default is 5s. The Baudot functionality is to be enabled first by calling
	/// ``Core/enableBaudot(enabled:)``. 
	/// - Parameter seconds: The significant pause timeout in seconds. 
	
	public var baudotPauseTimeout: UInt8 = 0
	{
	
		willSet
		{
			linphone_call_set_baudot_pause_timeout(cPtr, newValue)
		}
	}
		
	/// Defines the Baudot standard to use for sending Baudot tones in the call. 
	/// The Baudot functionality is to be enabled first by calling
	/// ``Core/enableBaudot(enabled:)``. 
	/// - Parameter standard: The Baudot standard to use for sending Baudot tones. 
	
	public var baudotSendingStandard: BaudotStandard?
	{
	
		willSet
		{
			linphone_call_set_baudot_sending_standard(cPtr, LinphoneBaudotStandard(rawValue: CUnsignedInt(newValue!.rawValue)))
		}
	}
		
	
	/// Gets the call log associated to this call. 
	/// - Returns: The ``CallLog`` associated with the specified ``Call``.    
	public var callLog: CallLog?
	{
	
						let cPointer = linphone_call_get_call_log(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = CallLog.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	/// Indicates whether camera input should be sent to remote end. 
	/// - Parameter enabled: wether or not to send local video stream. 
	
	/// Returns if camera pictures are allowed to be sent to the remote party. 
	/// - Returns: true if local video stream is being sent, false otherwise. 
	public var cameraEnabled: Bool
	{
	
		get
		{ 
						return linphone_call_camera_enabled(cPtr) != 0
		}
		set
		{
			linphone_call_enable_camera(cPtr, newValue==true ? 1:0)
		}
	}
		
	
	/// Obtain a chat room for real time messaging from a call if not already existing,
	/// else return existing one. 
	/// No reference is given to the caller: the chat room will be deleted when the
	/// call is ended. The call must have been accepted with a real time text stream
	/// (see ``CallParams/enableRealtimeText(yesno:)``). 
	/// - Returns: ``ChatRoom`` where real time messaging can take place or nil if chat
	/// room couldn't be created.    
	public var chatRoom: ChatRoom?
	{
	
						let cPointer = linphone_call_get_chat_room(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = ChatRoom.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Returns the associated conference object if any. 
	/// - Returns: A pointer on ``Conference`` or nil if the call is not part of any
	/// conference.    
	public var conference: Conference?
	{
	
						let cPointer = linphone_call_get_conference(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Conference.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Gets the core that has created the specified call. 
	/// - Returns: The ``Core`` object that has created the specified call.    
	public var core: Core?
	{
	
						let cPointer = linphone_call_get_core(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Core.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Gets the currently invoked ``CallDelegate`` if any. 
	/// This is meant only to be called from a callback to be able to get the user_data
	/// associated with the ``CallDelegate`` that is calling the callback. 
	/// - Returns: The ``CallDelegate`` that has called the last callback    
	public var currentDelegate: CallDelegate?
	{
	
			let cObject = linphone_call_get_current_callbacks(cPtr)
			let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
			if (result != nil) {
				return Unmanaged<CallDelegateManager>.fromOpaque(result!).takeUnretainedValue().delegate
				}
			return nil
			
	}
		
	
	/// Returns current parameters associated to the call. 
	/// - Returns: the current ``CallParams`` of this call.    
	public var currentParams: CallParams?
	{
	
						let cPointer = linphone_call_get_current_params(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = CallParams.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Obtains real-time quality rating of the call. 
	/// Based on local RTP statistics and RTCP feedback, a quality rating is computed
	/// and updated during all the duration of the call. This function returns its
	/// value at the time of the function call. It is expected that the rating is
	/// updated at least every 5 seconds or so. The rating is a floating point number
	/// comprised between 0 and 5.
	/// 4-5 = good quality  3-4 = average quality  2-3 = poor quality  1-2 = very poor
	/// quality  0-1 = can't be worse, mostly unusable 
	/// - Returns: The function returns -1 if no quality measurement is available, for
	/// example if no active audio stream exist. Otherwise it returns the quality
	/// rating. 
	public var currentQuality: Float
	{
	
						return linphone_call_get_current_quality(cPtr)

	}
		
	
	/// Returns direction of the call (incoming or outgoing). 
	/// - Returns: the ``Dir`` 
	public var dir: Call.Dir
	{
	
						return Call.Dir(rawValue: Int(linphone_call_get_dir(cPtr).rawValue))!

	}
		
	
	/// Returns the diversion address associated to this call. 
	/// - Returns: the diversion address as ``Address`` or nil.    
	public var diversionAddress: Address?
	{
	
						let cPointer = linphone_call_get_diversion_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Returns call's duration in seconds. 
	/// - Returns: the call's duration in seconds. 
	public var duration: Int
	{
	
						return Int(linphone_call_get_duration(cPtr))

	}
		
	/// Enables or disables echo cancellation for this call. 
	/// - See also: ``Core/enableEchoCancellation(enable:)``. 
	/// - Parameter enable: wether to enable echo cancellation or not. 
	
	/// Returns if echo cancellation is enabled. 
	/// - See also: ``Core/enableEchoCancellation(enable:)``. 
	/// - Returns: true if echo cancellation is enabled, false otherwise. 
	public var echoCancellationEnabled: Bool
	{
	
		get
		{ 
						return linphone_call_echo_cancellation_enabled(cPtr) != 0
		}
		set
		{
			linphone_call_enable_echo_cancellation(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Enables or disables echo limiter for this call. 
	/// - See also: ``Core/enableEchoLimiter(enable:)``. 
	/// - Parameter enable: wether to enable echo limiter or not. 
	
	/// Returns if echo limiter is enabled. 
	/// - See also: ``Core/enableEchoLimiter(enable:)``. 
	/// - Returns: true if echo limiter is enabled, false otherwise. 
	public var echoLimiterEnabled: Bool
	{
	
		get
		{ 
						return linphone_call_echo_limiter_enabled(cPtr) != 0
		}
		set
		{
			linphone_call_enable_echo_limiter(cPtr, newValue==true ? 1:0)
		}
	}
		
	
	/// Returns full details about call errors or termination reasons. 
	/// - Returns: ``ErrorInfo`` object holding the reason error.    
	public var errorInfo: ErrorInfo?
	{
	
						let cPointer = linphone_call_get_error_info(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = ErrorInfo.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	/// Sets the given ``AudioDevice`` as input for this call only. 
	/// - Parameter audioDevice: The ``AudioDevice``. nil does nothing.    
	
	/// Gets the current input device for this call. 
	/// - Returns: the ``AudioDevice`` used by this call as input or nil if there is
	/// currently no soundcard configured (depending on the state of the call)    
	public var inputAudioDevice: AudioDevice?
	{
	
		get
		{ 
						let cPointer = linphone_call_get_input_audio_device(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = AudioDevice.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_call_set_input_audio_device(cPtr, newValue?.cPtr)
		}
	}
		
	
	/// Returns whether or not the call is currently being recorded. 
	/// - Returns: true if recording is in progress, false otherwise 
	/// - Deprecated: 15/09/2021 Use ``CallParams/isRecording()`` instead. 
	@available(*, deprecated)
	public var isRecording: Bool
	{
	
						return linphone_call_is_recording(cPtr) != 0

	}
		
	
	/// Returns the local ZRTP authentication token to verify by the remote. 
	/// - Returns: the local authentication token to verify or nil if ZRTP isn't
	/// enabled.    
	public var localAuthenticationToken: String?
	{
	
			
			let cPointer = linphone_call_get_local_authentication_token(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Returns the local tag of the ``Call``. 
	/// - Returns: the local tag.    
	public var localTag: String
	{
	
			
			let cPointer = linphone_call_get_local_tag(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	/// Sets microphone muted state. 
	/// The boolean value given is applied logical-and with the value given to
	/// ``Core/enableMic(enable:)``. 
	/// - Parameter muted: The microphone muted state. 
	/// - Warning: This method only mutes the call passed as argument. If this call is
	/// part of a conference, it is strongly recommended to call
	/// ``Conference/setMicrophoneMuted(muted:)`` to ensure that the setting is
	/// correctly apply across all participants and the conference callbacks are
	/// called. 
	
	/// Gets microphone muted state. 
	/// Note that the microphone may be disabled globally if false was given to
	/// ``Core/enableMic(enable:)``. 
	/// - Returns: The microphone muted state. 
	/// - Warning: This method returns state of the mute capability of the call passed
	/// as argument. If this call is part of a conference, it is strongly recommended
	/// to call ``Conference/getMicrophoneMuted()`` to know whether this device is
	/// muted or not. 
	public var microphoneMuted: Bool
	{
	
		get
		{ 
						return linphone_call_get_microphone_muted(cPtr) != 0
		}
		set
		{
			linphone_call_set_microphone_muted(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets microphone volume gain. 
	/// If the sound backend supports it, the new gain will synchronized with the
	/// system mixer. 
	/// - Parameter volume: Percentage of the max supported gain. Valid values are in [
	/// 0.0 : 1.0 ]. 
	
	/// Gets microphone volume gain. 
	/// If the sound backend supports it, the returned gain is equal to the gain set
	/// with the system mixer. 
	/// - Returns: double Percentage of the max supported volume gain. Valid values are
	/// in [ 0.0 : 1.0 ]. In case of failure, a negative value is returned 
	public var microphoneVolumeGain: Float
	{
	
		get
		{ 
						return linphone_call_get_microphone_volume_gain(cPtr)
		}
		set
		{
			linphone_call_set_microphone_volume_gain(cPtr, newValue)
		}
	}
		
	/// Set the native video window id where the video is to be displayed. 
	/// For MacOS, Linux, Windows: if not set or 0 a window will be automatically
	/// created, unless the special id -1 is given. - See also:
	/// ``Core/setNativeVideoWindowId(windowId:)`` for a general discussion about
	/// window IDs. 
	/// - Parameter windowId: the native video window id.    
	
	/// Get the native window handle of the video window, casted as an unsigned long. 
	/// - See also: ``Core/setNativeVideoWindowId(windowId:)`` for a general discussion
	/// about window IDs. 
	/// - Returns: the native video window id (type may vary depending on platform).    
	public var nativeVideoWindowId: UnsafeMutableRawPointer?
	{
	
		get
		{ 
						return linphone_call_get_native_video_window_id(cPtr)
		}
		set
		{
			linphone_call_set_native_video_window_id(cPtr, newValue)
		}
	}
		
	/// Sets the given ``AudioDevice`` as output for this call only. 
	/// - Parameter audioDevice: The ``AudioDevice``. nil does nothing.    
	
	/// Gets the current output device for this call. 
	/// - Returns: the ``AudioDevice`` used by this call as output or nil if there is
	/// currently no soundcard configured (depending on the state of the call)    
	public var outputAudioDevice: AudioDevice?
	{
	
		get
		{ 
						let cPointer = linphone_call_get_output_audio_device(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = AudioDevice.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_call_set_output_audio_device(cPtr, newValue?.cPtr)
		}
	}
		
	/// Sets call parameters - advanced and not recommended feature - use with caution. 
	/// Local call parameters applicable to an outgoing or incoming shall usually be
	/// passed to ``Core/inviteAddressWithParams(addr:params:)`` or
	/// ``acceptWithParams(params:)``. However, in some cases it might be desirable
	/// from a software design standpoint to modify local parameters outside of the
	/// application layer, typically in the purpose of implementing a custom logic
	/// including special headers in INVITE or 200Ok requests, driven by a
	/// call_state_changed listener method. This function accepts to assign a new
	/// ``CallParams`` only in ``OutgoingInit`` and ``IncomingReceived`` states.
	/// - Parameter params: the ``CallParams`` object    
	
	/// Returns read-only local parameters associated with the call. 
	/// This is typically the parameters passed at call initiation to
	/// ``Core/inviteAddressWithParams(addr:params:)`` or
	/// ``acceptWithParams(params:)``, or some default parameters if no ``CallParams``
	/// was explicitely passed during call initiation. 
	/// - Returns: the call's local parameters.    
	public var params: CallParams?
	{
	
		get
		{ 
						let cPointer = linphone_call_get_params(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = CallParams.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_call_set_params(cPtr, newValue?.cPtr)
		}
	}
		
	
	/// Gets the mesured playback volume level (received from remote) in dbm0. 
	/// - Returns: float Volume level in dbm0. 
	public var playVolume: Float
	{
	
						return linphone_call_get_play_volume(cPtr)

	}
		
	
	/// Gets a player associated with the call to play a local file and stream it to
	/// the remote peer. 
	/// - Returns: A ``Player`` object.    
	public var player: Player?
	{
	
						let cPointer = linphone_call_get_player(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Player.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Returns the reason for a call termination (either error or normal termination) 
	/// - Returns: the ``Reason`` of the call termination. 
	public var reason: Reason
	{
	
						return Reason(rawValue: Int(linphone_call_get_reason(cPtr).rawValue))!

	}
		
	
	/// Gets the mesured record volume level (sent to remote) in dbm0. 
	/// - Returns: float Volume level in dbm0. 
	public var recordVolume: Float
	{
	
						return linphone_call_get_record_volume(cPtr)

	}
		
	
	/// Gets the refer-to uri (if the call was transferred). 
	/// - Returns: The refer-to uri of the call (if it was transferred).    
	public var referTo: String?
	{
	
			
			let cPointer = linphone_call_get_refer_to(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Gets the refer-to uri (if the call was transferred). 
	/// - Returns: The refer-to uri of the call (if it was transferred).    
	public var referToAddress: Address?
	{
	
						let cPointer = linphone_call_get_refer_to_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Gets the referred-by address, which is set when an incoming call is received as
	/// a consequence of a call transfer operation by a third party. 
	/// The referred-by address is the sip identity of the one who initiated the
	/// transfer. 
	/// - Returns: The referred-by address    
	public var referredByAddress: Address?
	{
	
						let cPointer = linphone_call_get_referred_by_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Returns the remote address associated to this call. 
	/// - Returns: The ``Address`` of the remote end of the call.    
	public var remoteAddress: Address?
	{
	
						let cPointer = linphone_call_get_remote_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Returns the remote address associated to this call as a string. 
	/// The result string must be freed by user using ms_free(). 
	/// - Returns: the remote address as a string.       
	/// - Deprecated: 06/07/2020 use ``getRemoteAddress()`` instead. 
	@available(*, deprecated)
	public var remoteAddressAsString: String?
	{
	
			
			let cPointer = linphone_call_get_remote_address_as_string(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			if (cPointer != nil) {
				bctbx_free(cPointer)
			}
			return result

	}
		
	
	/// Returns a list of 4 remote ZRTP authentication tokens. 
	/// The user needs to select one. 
	/// - Returns: the authentication tokens to verify or nil if ZRTP isn't enabled.   
	///   
	public var remoteAuthenticationTokens: [String]
	{
	
						var swiftList = [String]()
			let cList = linphone_call_get_remote_authentication_tokens(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				swiftList.append(String(cString: unsafeBitCast(listTemp!.pointee.data, to: UnsafePointer<CChar>.self)))
				listTemp = UnsafePointer<bctbx_list_t>(listTemp!.pointee.next)
			}
			return swiftList

	}
		
	
	/// Returns the far end's sip contact as a string, if available. 
	/// - Returns: the remote contact or nil.    
	public var remoteContact: String?
	{
	
			
			let cPointer = linphone_call_get_remote_contact(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Returns the far end's sip contact as an address, if available. 
	/// - Returns: the remote contact as a ``Address`` or nil.    
	public var remoteContactAddress: Address?
	{
	
						let cPointer = linphone_call_get_remote_contact_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Returns call parameters proposed by remote. 
	/// This is useful when receiving an incoming call, to know whether the remote
	/// party supports video, encryption or whatever.
	/// - Returns: the ``CallParams`` suggested by the remote or nil.    
	public var remoteParams: CallParams?
	{
	
						let cPointer = linphone_call_get_remote_params(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = CallParams.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Returns the remote tag of the ``Call``. 
	/// - Returns: the remote tag.    
	public var remoteTag: String
	{
	
			
			let cPointer = linphone_call_get_remote_tag(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Returns the far end's user agent description string, if available. 
	/// - Returns: the remote user agent or nil.    
	public var remoteUserAgent: String?
	{
	
			
			let cPointer = linphone_call_get_remote_user_agent(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Returns the call object this call is replacing, if any. 
	/// Call replacement can occur during call transfers. By default, the core
	/// automatically terminates the replaced call and accept the new one. This
	/// function allows the application to know whether a new incoming call is a one
	/// that replaces another one. 
	/// - Returns: the ``Call`` object this call is replacing or nil.    
	public var replacedCall: Call?
	{
	
						let cPointer = linphone_call_get_replaced_call(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Call.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// The address to which the call has been sent, taken directly from the SIP
	/// request-URI of the INVITE. 
	/// Usually equal to the To field, except when e.g. using a fallback contact
	/// address. You should probably use getToAddress() instead, unless you know what
	/// you're doing. 
	/// Usually equal to the To field, except when e.g. using a fallback contact
	/// address. You should probably use getToAddress() instead, unless you know what
	/// you're doing. 
	/// - Returns: the ``Address`` matching the URI of the INVITE request.    
	public var requestAddress: Address?
	{
	
						let cPointer = linphone_call_get_request_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	/// Sets speaker muted state. 
	/// - Parameter muted: The speaker muted state. 
	
	/// Gets speaker muted state. 
	/// - Returns: The speaker muted state. 
	public var speakerMuted: Bool
	{
	
		get
		{ 
						return linphone_call_get_speaker_muted(cPtr) != 0
		}
		set
		{
			linphone_call_set_speaker_muted(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets speaker volume gain. 
	/// If the sound backend supports it, the new gain will synchronized with the
	/// system mixer. 
	/// - Parameter volume: Percentage of the max supported gain. Valid values are in [
	/// 0.0 : 1.0 ]. 
	
	/// Gets speaker volume gain. 
	/// If the sound backend supports it, the returned gain is equal to the gain set
	/// with the system mixer. 
	/// - Returns: Percentage of the max supported volume gain. Valid values are in [
	/// 0.0 : 1.0 ]. In case of failure, a negative value is returned 
	public var speakerVolumeGain: Float
	{
	
		get
		{ 
						return linphone_call_get_speaker_volume_gain(cPtr)
		}
		set
		{
			linphone_call_set_speaker_volume_gain(cPtr, newValue)
		}
	}
		
	
	/// Retrieves the call's current state. 
	/// - Returns: the current ``State`` of this call. 
	public var state: Call.State
	{
	
						return Call.State(rawValue: Int(linphone_call_get_state(cPtr).rawValue))!

	}
		
	
	/// Returns the number of stream for the given call. 
	/// - Returns: the amount of streams for this call. 
	public var streamCount: Int
	{
	
						return Int(linphone_call_get_stream_count(cPtr))

	}
		
	
	/// Returns a copy of the call statistics for the text stream. 
	/// - Returns: a ``CallStats`` object for the text stream or nil if it isn't
	/// available.       
	public var textStats: CallStats?
	{
	
						let cPointer = linphone_call_get_text_stats(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = CallStats.getSwiftObject(cObject:cPointer!)
			belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
			return result

	}
		
	
	/// Returns the to address with its headers associated to this call. 
	/// - Returns: the ``Address`` matching the TO of the call.    
	public var toAddress: Address?
	{
	
						let cPointer = linphone_call_get_to_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Returns the current transfer state, if a transfer has been initiated from this
	/// call. 
	/// - See also: linphone_core_transfer_call ,
	/// linphone_core_transfer_call_to_another 
	/// - Returns: the ``State``. 
	public var transferState: Call.State
	{
	
						return Call.State(rawValue: Int(linphone_call_get_transfer_state(cPtr).rawValue))!

	}
		
	
	/// When this call has received a transfer request, returns the new call that was
	/// automatically created as a result of the transfer. 
	/// - Returns: the transfer ``Call`` created.    
	public var transferTargetCall: Call?
	{
	
						let cPointer = linphone_call_get_transfer_target_call(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Call.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Gets the transferer if this call was started automatically as a result of an
	/// incoming transfer request. 
	/// The call in which the transfer request was received is returned in this case. 
	/// - Returns: The transferer ``Call`` if the specified call was started
	/// automatically as a result of an incoming transfer request, nil otherwise.    
	public var transfererCall: Call?
	{
	
						let cPointer = linphone_call_get_transferer_call(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Call.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	/// Assigns a user pointer to the call. 
	/// - Parameter userData: The user pointer to associate with the call.    
	
	/// Retrieves the user pointer associated with the call. 
	/// - Returns: The user pointer associated with the call.    
	public var userData: UnsafeMutableRawPointer?
	{
	
		get
		{ 
						return linphone_call_get_user_data(cPtr)
		}
		set
		{
			linphone_call_set_user_data(cPtr, newValue)
		}
	}
		
	/// Sets the video source of a call. 
	/// - Parameter descriptor: The ``VideoSourceDescriptor`` describing the video
	/// source to set    
	
	/// Gets the video source of a call. 
	/// - Returns: The ``VideoSourceDescriptor`` describing the video source that is
	/// set    
	public var videoSource: VideoSourceDescriptor?
	{
	
		get
		{ 
						let cPointer = linphone_call_get_video_source(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = VideoSourceDescriptor.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_call_set_video_source(cPtr, newValue?.cPtr)
		}
	}
		
	
	/// Returns a copy of the call statistics for the video stream. 
	/// - Returns: a ``CallStats`` object for the video stream or nil if it isn't
	/// available.       
	public var videoStats: CallStats?
	{
	
						let cPointer = linphone_call_get_video_stats(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = CallStats.getSwiftObject(cObject:cPointer!)
			belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
			return result

	}
		
	
	/// Returns whether ZRTP cache mismatch. 
	/// If mismatch, the ZRTP authentication token must be verified by users as
	/// described in ZRTP procedure. 
	/// - Returns: true if ZRTP cache mismatch, false otherwise. 
	public var zrtpCacheMismatchFlag: Bool
	{
	
						return linphone_call_get_zrtp_cache_mismatch_flag(cPtr) != 0

	}
		
	
	
	/// Accepts an incoming call. 
	/// Basically, the application is notified of incoming calls within the
	/// call_state_changed callback of the LinphoneCoreVTable structure, where it will
	/// receive a ``IncomingReceived`` event with the associated ``Call`` object. The
	/// application can later accept the call using this method. 
	/// - Returns: 0 on success, -1 on failure 
	public func accept() throws 
	{
		let exception_result = linphone_call_accept(cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "accept returned value \(exception_result)")
		}
	}
	
	
	
	/// Accepts an early media session for an incoming call. 
	/// This is identical as calling ``acceptEarlyMediaWithParams(params:)`` with nil
	/// parameters. 
	/// - Returns: 0 if successful, -1 otherwise 
	/// - See also: ``acceptEarlyMediaWithParams(params:)`` 
	public func acceptEarlyMedia() throws 
	{
		let exception_result = linphone_call_accept_early_media(cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "acceptEarlyMedia returned value \(exception_result)")
		}
	}
	
	
	
	/// When receiving an incoming, accepts to start a media session as early-media. 
	/// This means the call is not accepted but audio & video streams can be
	/// established if the remote party supports early media. However, unlike after
	/// call acceptance, mic and camera input are not sent during early-media, though
	/// received audio & video are played normally. The call can then later be fully
	/// accepted using ``accept()`` or ``acceptWithParams(params:)``. 
	/// - Parameter params: The call parameters to use (can be nil).    
	/// - Returns: 0 if successful, -1 otherwise 
	public func acceptEarlyMediaWithParams(params:CallParams?) throws 
	{
		let exception_result = linphone_call_accept_early_media_with_params(cPtr, params?.cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "acceptEarlyMediaWithParams returned value \(exception_result)")
		}
	}
	
	
	
	/// Accepts and execute a transfer (ie an incoming REFER request), to use when
	/// [sip] 'auto_accept_refer' property is false. 
	/// Default behaviour is to accept and execute the transfer automatically. 
	public func acceptTransfer() 
	{
		linphone_call_accept_transfer(cPtr)
	}
	
	
	
	/// Accepts call modifications initiated by other end through an incoming reINVITE
	/// or UPDATE request. 
	/// ``acceptUpdate(params:)`` is typically used in response to a
	/// ``UpdatedByRemote`` state notification. When such notification arrives, the
	/// application has several choices:
	/// -use ``deferUpdate()`` so that it can have the time to prompt the user, and
	/// then later use ``acceptUpdate(params:)`` to answer
	/// -doing nothing, in which case ``acceptUpdate(params:)`` is internally called
	/// automatically to generate a default response.
	/// -immediately ``acceptUpdate(params:)`` to generate a response.
	/// An application may use ``getRemoteParams()`` to get information about the call
	/// parameters proposed by the other party (for example when he wants to add a
	/// video stream), in order to decide what to do, like for example requesting the
	/// end-user approval.
	/// The params argument must be constructed with ``Core/createCallParams(call:)``.
	/// ``Core/createCallParams(call:)`` will initialize the returned ``CallParams``
	/// according to the offered parameters, previously used local parameters and local
	/// policies. For example, if the remote user-agent wants to add a video stream but
	/// the ``VideoActivationPolicy`` of the ``Core`` is to not automatically accept
	/// video, then the ``CallParams`` will get its video stream disabled. The
	/// application is free to modify the ``CallParams`` before passing it to
	/// ``acceptUpdate(params:)``. Using null as params argument is allowed and
	/// equivalent to passing a ``CallParams`` created by
	/// ``Core/createCallParams(call:)`` and left unmodified.
	/// The answer generated by ``acceptUpdate(params:)`` at first follows RFC3264 SDP
	/// offer answer model rules. The supplied ``CallParams`` allows to give
	/// application instructions about the response to generate, such as accepting or
	/// not the enablement of new stream. ``acceptUpdate(params:)`` does not have the
	/// pretention to control every aspect of the generation of the answer of an SDP
	/// offer/answer procedure.
	/// - Parameter params: A ``CallParams`` object describing the call parameters to
	/// accept.    
	/// - Returns: 0 if successful, -1 otherwise (actually when this function call is
	/// performed outside ot ``UpdatedByRemote`` state) 
	public func acceptUpdate(params:CallParams?) throws 
	{
		let exception_result = linphone_call_accept_update(cPtr, params?.cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "acceptUpdate returned value \(exception_result)")
		}
	}
	
	
	
	/// Accepts an incoming call, with parameters. 
	/// Basically, the application is notified of incoming calls within the
	/// call_state_changed callback of the LinphoneCoreVTable structure, where it will
	/// receive a ``IncomingReceived`` event with the associated ``Call`` object. The
	/// application can later accept the call using this method. 
	/// - Parameter params: The specific parameters for this call, for example, whether
	/// video is accepted or not. Use nil to use default parameters.    
	/// - Returns: 0 on success, -1 on failure 
	public func acceptWithParams(params:CallParams?) throws 
	{
		let exception_result = linphone_call_accept_with_params(cPtr, params?.cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "acceptWithParams returned value \(exception_result)")
		}
	}
	
	
	
	/// Tells whether a call has been asked to autoanswer. 
	/// - Returns: A boolean value telling whether the call has been asked to
	/// autoanswer 
	public func askedToAutoanswer() -> Bool
	{
		return linphone_call_asked_to_autoanswer(cPtr) != 0
	}
	
	
	
	/// Stops current DTMF sequence sending. 
	/// Please note that some DTMF could be already sent, depending on when this
	/// function call is delayed from ``sendDtmfs(dtmfs:)``. This function will be
	/// automatically called if call state change to anything but
	/// LinphoneCallStreamsRunning. 
	public func cancelDtmfs() 
	{
		linphone_call_cancel_dtmfs(cPtr)
	}
	
	
	
	/// Verifies that the half ZRTP short authentication string (SAS) selected by the
	/// user is correct, and set the verification result accordingly. 
	/// If the remote party also verifies his/her half code correctly, the ZRTP cache
	/// will be updated to prevent future verification requirements for these two
	/// users. 
	/// - Parameter selectedValue: The ZRTP SAS selected by the user, or an empty
	/// string if the user cannot find the SAS 
	public func checkAuthenticationTokenSelected(selectedValue:String) 
	{
		linphone_call_check_authentication_token_selected(cPtr, selectedValue)
	}
	
	
	
	/// Method to be called after the user confirm that he/she is notifed of the on
	/// going Go Clear procedure. 
	/// - Warning: this operation must be imperatevely initiate by a user action on
	/// sending of the GoClear ACK 
	public func confirmGoClear() 
	{
		linphone_call_confirm_go_clear(cPtr)
	}
	
	
	
	/// Create a native video window id where the video is to be displayed. 
	/// - See also: ``Core/setNativeVideoWindowId(windowId:)`` for a general discussion
	/// about window IDs.
	/// A context can be used to prevent Linphone from allocating the container
	/// (MSOglContextInfo for MSOGL). nil if not used.
	/// A context can be used to prevent Linphone from allocating the container
	/// (MSOglContextInfo for MSOGL). nil if not used.
	/// - Parameter context: preallocated Window ID (Used only for MSOGL)    
	/// - Returns: the native video window id (type may vary depending on platform).    
	public func createNativeVideoWindowId(context:UnsafeMutableRawPointer?) throws -> UnsafeMutableRawPointer
	{
		return linphone_call_create_native_video_window_id_2(cPtr, context)
	}
	
	
	
	/// Create a native video window id where the video is to be displayed. 
	/// - See also: ``Core/setNativeVideoWindowId(windowId:)`` for a general discussion
	/// about window IDs.
	/// - Returns: the native video window id (type may vary depending on platform).    
	public func createNativeVideoWindowId() throws -> UnsafeMutableRawPointer
	{
		return linphone_call_create_native_video_window_id(cPtr)
	}
	
	
	
	/// Create a ``Event`` in order to send NOTIFY requests through the SIP dialog
	/// created by the call. 
	/// The call state must have passed through ``Connected``. 
	/// - Parameter event: The event type to be notified. 
	/// - Returns: a new ``Event``    
	public func createNotify(event:String) throws -> Event
	{
		let cPointer = linphone_call_create_notify(cPtr, event)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null Event value")
		}
		let result = Event.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Declines a pending incoming call, with a reason. 
	/// - Parameter reason: The reason for rejecting the call: ``Declined`` or ``Busy`` 
	/// - Returns: 0 on success, -1 on failure 
	public func decline(reason:Reason) throws 
	{
		let exception_result = linphone_call_decline(cPtr, LinphoneReason(rawValue: CUnsignedInt(reason.rawValue)))
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "decline returned value \(exception_result)")
		}
	}
	
	
	
	/// Declines a pending incoming call, with a ``ErrorInfo`` object. 
	/// - Parameter ei: ``ErrorInfo`` containing more information on the call
	/// rejection.    
	/// - Returns: 0 on success, -1 on failure 
	public func declineWithErrorInfo(ei:ErrorInfo?) -> Int
	{
		return Int(linphone_call_decline_with_error_info(cPtr, ei?.cPtr))
	}
	
	
	
	/// When receiving a ``UpdatedByRemote`` state notification, prevent ``Core`` from
	/// performing an automatic answer. 
	/// When receiving a ``UpdatedByRemote`` state notification (ie an incoming
	/// reINVITE), the default behaviour of ``Core`` is defined by the
	/// "defer_update_default" option of the "sip" section of the config. If this
	/// option is 0 (the default) then the ``Core`` automatically answers the reINIVTE
	/// with call parameters unchanged. However when for example when the remote party
	/// updated the call to propose a video stream, it can be useful to prompt the user
	/// before answering. This can be achieved by calling
	/// linphone_core_defer_call_update during the call state notification, to
	/// deactivate the automatic answer that would just confirm the audio but reject
	/// the video. Then, when the user responds to dialog prompt, it becomes possible
	/// to call ``acceptUpdate(params:)`` to answer the reINVITE, with video possibly
	/// enabled in the ``CallParams`` argument.
	/// The ``UpdatedByRemote`` notification can also happen when receiving an INVITE
	/// without SDP. In such case, an unchanged offer is made in the 200Ok, and when
	/// the ACK containing the SDP answer is received, ``UpdatedByRemote`` is triggered
	/// to notify the application of possible changes in the media session. However in
	/// such case defering the update has no meaning since we just generated an offer.
	/// - Returns: 0 if successful, -1 if the ``deferUpdate()`` was done outside a
	/// valid ``UpdatedByRemote`` notification 
	public func deferUpdate() throws 
	{
		let exception_result = linphone_call_defer_update(cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "deferUpdate returned value \(exception_result)")
		}
	}
	
	
	
	/// Returns a copy of the call statistics for a particular stream type. 
	/// - Parameter type: the ``StreamType`` 
	/// - Returns: a ``CallStats`` object for the given stream or nil if stream isn't
	/// available.       
	public func getStats(type:StreamType) -> CallStats?
	{
		let cPointer = linphone_call_get_stats(cPtr, LinphoneStreamType(rawValue: CUnsignedInt(type.rawValue)))
		if (cPointer == nil) {
			return nil
		}
		let result = CallStats.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Returns the value of the header name. 
	/// - Parameter headerName: the name of the header to check.    
	/// - Returns: the value of the header if exists.    
	/// - Deprecated: 27/10/2020. Use ``CallParams/getCustomHeader(headerName:)`` on
	/// ``getRemoteParams()`` instead. 
	@available(*, deprecated)
	public func getToHeader(headerName:String) -> String
	{
		let cstr = linphone_call_get_to_header(cPtr, headerName)
		let result = charArrayToString(charPointer: cstr)
		return result
	}
	
	
	
	/// Returns if this calls has received a transfer that has not been executed yet. 
	/// Pending transfers are executed when this call is being paused or closed,
	/// locally or by remote endpoint. If the call is already paused while receiving
	/// the transfer request, the transfer immediately occurs. 
	/// - Returns: true if transfer is pending, false otherwise. 
	public func hasTransferPending() -> Bool
	{
		return linphone_call_has_transfer_pending(cPtr) != 0
	}
	
	
	
	/// Indicates whether an operation is in progress at the media side. 
	/// It can be a bad idea to initiate signaling operations (adding video, pausing
	/// the call, removing video, changing video parameters) while the media is busy in
	/// establishing the connection (typically ICE connectivity checks). It can result
	/// in failures generating loss of time in future operations in the call.
	/// Applications are invited to check this function after each call state change to
	/// decide whether certain operations are permitted or not. 
	/// - Returns: true if media is busy in establishing the connection, false
	/// otherwise. 
	public func mediaInProgress() -> Bool
	{
		return linphone_call_media_in_progress(cPtr) != 0
	}
	
	
	
	/// Starts the process of replying 180 Ringing. 
	/// This function is used in conjonction with
	/// ``Core/enableAutoSendRinging(enable:)``. If the automatic sending of the 180
	/// Ringing is disabled, this function needs to be called manually before the call
	/// timeouts.
	public func notifyRinging() 
	{
		linphone_call_notify_ringing(cPtr)
	}
	
	
	
	/// Calls generic OpenGL render for a given call. 
	public func oglRender() 
	{
		linphone_call_ogl_render(cPtr)
	}
	
	
	
	/// Pauses the call. 
	/// If a music file has been setup using ``Core/setPlayFile(file:)``, this file
	/// will be played to the remote user. The only way to resume a paused call is to
	/// call ``resume()``. 
	/// - Returns: 0 on success, -1 on failure 
	/// - See also: ``resume()`` 
	public func pause() throws 
	{
		let exception_result = linphone_call_pause(cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "pause returned value \(exception_result)")
		}
	}
	
	
	
	/// Redirect the specified call to the given redirect URI. 
	/// - Parameter redirectUri: The URI to redirect the call to    
	/// - Returns: 0 if successful, -1 on error. 
	/// - Deprecated: 27/10/2020. Use ``redirectTo(redirectAddress:)`` instead. 
	@available(*, deprecated)
	public func redirect(redirectUri:String) throws 
	{
		let exception_result = linphone_call_redirect(cPtr, redirectUri)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "redirect returned value \(exception_result)")
		}
	}
	
	
	
	/// Redirects the specified call to the given redirect Address. 
	/// - Parameter redirectAddress: The ``Address`` to redirect the call to    
	/// - Returns: 0 if successful, -1 on error. 
	public func redirectTo(redirectAddress:Address) throws 
	{
		let exception_result = linphone_call_redirect_to(cPtr, redirectAddress.cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "redirectTo returned value \(exception_result)")
		}
	}
	
	
	
	/// Requests the callback passed to linphone_call_cbs_set_next_video_frame_decoded
	/// to be called the next time the video decoder properly decodes a video frame. 
	public func requestNotifyNextVideoFrameDecoded() 
	{
		linphone_call_request_notify_next_video_frame_decoded(cPtr)
	}
	
	
	
	/// Resumes a call. 
	/// The call needs to have been paused previously with ``pause()``. 
	/// - Returns: 0 on success, -1 on failure 
	/// - See also: ``pause()`` 
	public func resume() throws 
	{
		let exception_result = linphone_call_resume(cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "resume returned value \(exception_result)")
		}
	}
	
	
	
	/// Sends the specified dtmf. 
	/// The dtmf is automatically played to the user. - See also:
	/// ``Core/setUseRfc2833ForDtmf(useRfc2833:)`` 
	/// - Parameter dtmf: The dtmf name specified as a char, such as '0', '#' etc... 
	/// - Returns: 0 if successful, -1 on error. 
	public func sendDtmf(dtmf:CChar) throws 
	{
		let exception_result = linphone_call_send_dtmf(cPtr, dtmf)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "sendDtmf returned value \(exception_result)")
		}
	}
	
	
	
	/// Sends a list of dtmf. 
	/// The dtmfs are automatically sent to remote, separated by some needed
	/// customizable delay. Sending is canceled if the call state changes to something
	/// not LinphoneCallStreamsRunning. - See also:
	/// ``Core/setUseRfc2833ForDtmf(useRfc2833:)`` 
	/// - Parameter dtmfs: A dtmf sequence such as '123#123123'    
	/// - Returns: -2 if there is already a DTMF sequence, -1 if call is not ready, 0
	/// otherwise. 
	public func sendDtmfs(dtmfs:String) throws 
	{
		let exception_result = linphone_call_send_dtmfs(cPtr, dtmfs)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "sendDtmfs returned value \(exception_result)")
		}
	}
	
	
	
	/// Sends an info message through an established call. 
	/// - Parameter info: the ``InfoMessage`` to send    
	public func sendInfoMessage(info:InfoMessage) throws 
	{
		let exception_result = linphone_call_send_info_message(cPtr, info.cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "sendInfoMessage returned value \(exception_result)")
		}
	}
	
	
	
	/// Requests remote side to send us a Video Fast Update. 
	public func sendVfuRequest() 
	{
		linphone_call_send_vfu_request(cPtr)
	}
	
	
	
	/// If the user skips the ZRTP authentication check, stop the security alert. 
	public func skipZrtpAuthentication() 
	{
		linphone_call_skip_zrtp_authentication(cPtr)
	}
	
	
	
	/// Starts call recording. 
	/// Video record is only available if this function is called in state
	/// StreamRunning. The output file where audio is recorded must be previously
	/// specified with ``CallParams/setRecordFile(path:)``. 
	public func startRecording() 
	{
		linphone_call_start_recording(cPtr)
	}
	
	
	
	/// Stops call recording. 
	public func stopRecording() 
	{
		linphone_call_stop_recording(cPtr)
	}
	
	
	
	/// Takes a photo of currently captured video and write it into a jpeg file. 
	/// Note that the snapshot is asynchronous, an application shall not assume that
	/// the file is created when the function returns. 
	/// - Parameter filePath: a path where to write the jpeg content.    
	/// - Returns: 0 if successful, -1 otherwise (typically if jpeg format is not
	/// supported). 
	public func takePreviewSnapshot(filePath:String) throws 
	{
		let exception_result = linphone_call_take_preview_snapshot(cPtr, filePath)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "takePreviewSnapshot returned value \(exception_result)")
		}
	}
	
	
	
	/// Takes a photo of currently received video and write it into a jpeg file. 
	/// Note that the snapshot is asynchronous, an application shall not assume that
	/// the file is created when the function returns. 
	/// - Parameter filePath: a path where to write the jpeg content.    
	/// - Returns: 0 if successful, -1 otherwise (typically if jpeg format is not
	/// supported). 
	public func takeVideoSnapshot(filePath:String) throws 
	{
		let exception_result = linphone_call_take_video_snapshot(cPtr, filePath)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "takeVideoSnapshot returned value \(exception_result)")
		}
	}
	
	
	
	/// Terminates (ends) a call. 
	/// - Returns: 0 on success, -1 on failure 
	public func terminate() throws 
	{
		let exception_result = linphone_call_terminate(cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "terminate returned value \(exception_result)")
		}
	}
	
	
	
	/// Terminates a call with additional information, serialized as a Reason header. 
	/// - Parameter ei: ``ErrorInfo``    
	/// - Returns: 0 on success, -1 on failure 
	public func terminateWithErrorInfo(ei:ErrorInfo?) throws 
	{
		let exception_result = linphone_call_terminate_with_error_info(cPtr, ei?.cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "terminateWithErrorInfo returned value \(exception_result)")
		}
	}
	
	
	
	/// Performs a simple call transfer to the specified destination. 
	/// The remote endpoint is expected to issue a new call to the specified
	/// destination. The current call remains active and thus can be later paused or
	/// terminated. It is possible to follow the progress of the transfer provided that
	/// transferee sends notification about it. In this case, the
	/// transfer_state_changed callback of the LinphoneCoreVTable is invoked to notify
	/// of the state of the new call at the other party. The notified states are
	/// ``OutgoingInit``, ``OutgoingProgress``, ``OutgoingRinging`` and ``Connected``. 
	/// - Parameter referTo: The destination the call is to be referred to.    
	/// - Returns: 0 on success, -1 on failure 
	/// - Deprecated: 27/10/2020. Use ``transferTo(referTo:)`` instead. 
	@available(*, deprecated)
	public func transfer(referTo:String) throws 
	{
		let exception_result = linphone_call_transfer(cPtr, referTo)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "transfer returned value \(exception_result)")
		}
	}
	
	
	
	/// Performs a simple call transfer to the specified destination. 
	/// The remote endpoint is expected to issue a new call to the specified
	/// destination. The current call remains active and thus can be later paused or
	/// terminated. It is possible to follow the progress of the transfer provided that
	/// transferee sends notification about it. In this case, the 'transfer state
	/// changed' callback of the ``CoreDelegate`` is invoked to notify of the state of
	/// the new call to the other party. The notified states are ``OutgoingInit`` ,
	/// ``OutgoingProgress``, ``OutgoingRinging`` and ``Connected``. 
	/// - Parameter referTo: The ``Address`` the call is to be referred to.    
	/// - Returns: 0 on success, -1 on failure 
	public func transferTo(referTo:Address) throws 
	{
		let exception_result = linphone_call_transfer_to(cPtr, referTo.cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "transferTo returned value \(exception_result)")
		}
	}
	
	
	
	/// Transfers a call to destination of another running call. 
	/// This is used for "attended transfer" scenarios. The transferred call is
	/// supposed to be in paused state, so that it is able to accept the transfer
	/// immediately. The destination call is a call previously established to introduce
	/// the transferred person. This method will send a transfer request to the
	/// transferred person. The phone of the transferred is then expected to
	/// automatically call to the destination of the transfer. The receiver of the
	/// transfer will then automatically close the call with us (the 'dest' call). It
	/// is possible to follow the progress of the transfer provided that transferee
	/// sends notification about it. In this case, the "transfer state changed"
	/// callback of the ``CoreDelegate`` is invoked to notify of the state of the new
	/// call to the other party. The notified states are ``OutgoingInit`` ,
	/// ``OutgoingProgress``, ``OutgoingRinging`` and ``Connected``. 
	/// - Parameter dest: A running call whose remote person will receive the transfer 
	///   
	/// - Returns: 0 on success, -1 on failure 
	public func transferToAnother(dest:Call) throws 
	{
		let exception_result = linphone_call_transfer_to_another(cPtr, dest.cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "transferToAnother returned value \(exception_result)")
		}
	}
	
	
	
	/// Updates a running call according to supplied call parameters or parameters
	/// changed in the ``Core``. 
	/// It triggers a SIP reINVITE in order to perform a new offer/answer of media
	/// capabilities. Changing the size of the transmitted video after calling
	/// linphone_core_set_preferred_video_size can be used by passing nil as params
	/// argument. In case no changes are requested through the ``CallParams`` argument,
	/// then this argument can be omitted and set to nil. WARNING: Updating a call in
	/// the ``Paused`` state will still result in a paused call even if the media
	/// directions set in the params are sendrecv. To resume a paused call, you need to
	/// call ``resume()``.
	/// - Parameter params: The new call parameters to use (may be nil).    
	/// - Returns: 0 if successful, -1 otherwise. 
	public func update(params:CallParams?) throws 
	{
		let exception_result = linphone_call_update(cPtr, params?.cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "update returned value \(exception_result)")
		}
	}
	
	
	
	/// Performs a zoom of the video displayed during a call. 
	/// The zoom ensures that all the screen is fullfilled with the video. 
	/// - Parameter zoomFactor: a floating point number describing the zoom factor. A
	/// value 1.0 corresponds to no zoom applied. 
	/// - Parameter cx: a floating point number pointing the horizontal center of the
	/// zoom to be applied. This value should be between 0.0 and 1.0. 
	/// - Parameter cy: a floating point number pointing the vertical center of the
	/// zoom to be applied. This value should be between 0.0 and 1.0. 
	public func zoom(zoomFactor:Float, cx:Float, cy:Float) 
	{
		linphone_call_zoom(cPtr, zoomFactor, cx, cy)
	}
}


/// Object used to keep track of all calls initiated, received or missed. 
/// It contains the call ID, date & time at which the call took place and it's
/// duration (0 if it wasn't answered). You can also know if video was enabled or
/// not or if it was a conference, as well as it's average quality.
/// If needed, you can also create a fake ``CallLog`` using
/// ``Core/createCallLog(from:to:dir:duration:startTime:connectedTime:status:videoEnabled:quality:)``, otherwise use ``Core/getCallLogs()`` or even ``Call/getCallLog()`` to get the log of an ongoing call. 
public class CallLog : LinphoneObject
{

	static public func getSwiftObject(cObject:OpaquePointer) -> CallLog {
		let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
		if (result != nil) {
			return Unmanaged<CallLog>.fromOpaque(result!).takeUnretainedValue()
		}
		let sObject = CallLog(cPointer: cObject)
		belle_sip_object_data_set(UnsafeMutablePointer(cObject), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(sObject).toOpaque()), nil)
		return sObject
	}

	public var getCobject: OpaquePointer? {
		return cPtr
	}
	
	
	/// Gets the call ID used by the call. 
	/// - Returns: The call ID used by the call as a string.    
	public var callId: String?
	{
	
			
			let cPointer = linphone_call_log_get_call_id(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Returns the chat room associated with this call-log, if any. 
	/// This method is typically useful in order to retrieve an IM conversation
	/// associated with a past conference call. 
	/// - Returns: The ``ChatRoom`` associated.    
	public var chatRoom: ChatRoom?
	{
	
						let cPointer = linphone_call_log_get_chat_room(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = ChatRoom.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Retrieves the conference info associated to this call log in DB. 
	/// - Returns: The ``ConferenceInfo`` associated.    
	public var conferenceInfo: ConferenceInfo?
	{
	
						let cPointer = linphone_call_log_get_conference_info(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = ConferenceInfo.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Gets the direction of the call. 
	/// - Returns: The ``Call.Dir`` of the call. 
	public var dir: Call.Dir
	{
	
						return Call.Dir(rawValue: Int(linphone_call_log_get_dir(cPtr).rawValue))!

	}
		
	
	/// Gets the duration of the call since it was connected. 
	/// - Returns: The duration of the call in seconds. 
	public var duration: Int
	{
	
						return Int(linphone_call_log_get_duration(cPtr))

	}
		
	
	/// When the call was failed, return an object describing the failure. 
	/// - Returns: ``ErrorInfo`` about the error encountered by the call associated
	/// with this call log or nil.    
	public var errorInfo: ErrorInfo?
	{
	
						let cPointer = linphone_call_log_get_error_info(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = ErrorInfo.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Gets the origin address (ie from) of the call. 
	/// - Returns: The origin ``Address`` (ie from) of the call.    
	public var fromAddress: Address?
	{
	
						let cPointer = linphone_call_log_get_from_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Gets the local address (that is from or to depending on call direction) 
	/// - Returns: The local ``Address`` of the call    
	public var localAddress: Address?
	{
	
						let cPointer = linphone_call_log_get_local_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Gets the overall quality indication of the call. 
	/// - See also: ``Call/getCurrentQuality()`` 
	/// - Returns: The overall quality indication of the call. 
	public var quality: Float
	{
	
						return linphone_call_log_get_quality(cPtr)

	}
		
	/// Associates a persistent reference key to the call log. 
	/// The reference key can be for example an id to an external database. It is
	/// stored in the config file, thus can survive to process exits/restarts.
	/// - Parameter refkey: The reference key string to associate to the call log.    
	
	/// Gets the persistent reference key associated to the call log. 
	/// The reference key can be for example an id to an external database. It is
	/// stored in the config file, thus can survive to process exits/restarts.
	/// - Returns: The reference key string that has been associated to the call log,
	/// or nil if none has been associated.    
	public var refKey: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_call_log_get_ref_key(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_call_log_set_ref_key(cPtr, newValue)
		}
	}
		
	/// Sets the remote address (that is 'from' or 'to' depending on call direction). 
	/// It allows to fill more information that the SDK doesn't have. A use case can be
	/// to fill the display name (coming from an external address book) into a call log
	/// on incoming call. When the call end, the database will take account of the new
	/// information and can be used later 
	/// - Parameter address: ``Address`` object    
	
	/// Gets the remote address (that is from or to depending on call direction). 
	/// - Returns: The remote ``Address`` of the call.    
	public var remoteAddress: Address?
	{
	
		get
		{ 
						let cPointer = linphone_call_log_get_remote_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_call_log_set_remote_address(cPtr, newValue?.cPtr)
		}
	}
		
	
	/// Gets the start date of the call. 
	/// - Returns: The date of the beginning of the call. 
	public var startDate: time_t
	{
	
						return linphone_call_log_get_start_date(cPtr)

	}
		
	
	/// Gets the status of the call. 
	/// - Returns: The ``Call.Status`` of the call. 
	public var status: Call.Status
	{
	
						return Call.Status(rawValue: Int(linphone_call_log_get_status(cPtr).rawValue))!

	}
		
	
	/// Gets the destination address (ie to) of the call. 
	/// - Returns: The destination ``Address`` (ie to) of the call.    
	public var toAddress: Address?
	{
	
						let cPointer = linphone_call_log_get_to_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	/// Assigns a user data to the call log. 
	/// - Parameter userData: The user data to associate with the call log.    
	
	/// Gets the user data associated with the call log. 
	/// - Returns: The user data associated with the call log.    
	public var userData: UnsafeMutableRawPointer?
	{
	
		get
		{ 
						return linphone_call_log_get_user_data(cPtr)
		}
		set
		{
			linphone_call_log_set_user_data(cPtr, newValue)
		}
	}
		
	
	/// Tells whether video was enabled at the end of the call or not. 
	/// - Returns: A boolean value telling whether video was enabled at the end of the
	/// call. 
	public var videoEnabled: Bool
	{
	
						return linphone_call_log_video_enabled(cPtr) != 0

	}
		
	
	
	/// Gets a human readable string describing the call. 
	/// - Note: : the returned string must be freed by the application (use ms_free()). 
	/// - Returns: A human readable string describing the call.       
	public func toStr() -> String
	{
		let cstr = linphone_call_log_to_str(cPtr)
		let result = charArrayToString(charPointer: cstr)
		if (cstr != nil) {
			bctbx_free(cstr)
		}
		return result
	}
	
	
	
	/// Tells whether that call was part of a conference. 
	/// - Returns: true if the call was part of a conference, false otherwise. 
	public func wasConference() -> Bool
	{
		return linphone_call_log_was_conference(cPtr) != 0
	}
}


/// An object containing various parameters of a ``Call``. 
/// You can specify your params while answering an incoming call using
/// ``Call/acceptWithParams(params:)`` or while initiating an outgoing call with
/// ``Core/inviteAddressWithParams(addr:params:)``.
/// This object can be created using ``Core/createCallParams(call:)``, using nil
/// for the call pointer if you plan to use it for an outgoing call.
/// For each call, three ``CallParams`` are available: yours, your correspondent's
/// and the one that describe the current state of the call that is the result of
/// the negociation between the previous two. For example, you might enable a
/// certain feature in your call param but this feature can be denied in the
/// remote's configuration, hence the difference.
/// - See also: ``Call/getCurrentParams()``, ``Call/getRemoteParams()`` and
/// ``Call/getParams()``. 
public class CallParams : LinphoneObject
{

	static public func getSwiftObject(cObject:OpaquePointer) -> CallParams {
		let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
		if (result != nil) {
			return Unmanaged<CallParams>.fromOpaque(result!).takeUnretainedValue()
		}
		let sObject = CallParams(cPointer: cObject)
		belle_sip_object_data_set(UnsafeMutablePointer(cObject), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(sObject).toOpaque()), nil)
		return sObject
	}

	public var getCobject: OpaquePointer? {
		return cPtr
	}
	
	/// Set the ``Account`` to use for the call. 
	/// - Parameter account: The ``Account`` to use, or nil if none has been selected.
	/// The ``CallParams`` keeps a reference to it and removes the previous one, if
	/// any.    
	
	/// Get the ``Account`` that is used for the call. 
	/// - Returns: The selected ``Account`` for the call, or nil if none has been
	/// selected.    
	public var account: Account?
	{
	
		get
		{ 
						let cPointer = linphone_call_params_get_account(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Account.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_call_params_set_account(cPtr, newValue?.cPtr)
		}
	}
		
	/// Refine bandwidth settings for this call by setting a bandwidth limit for audio
	/// streams. 
	/// As a consequence, codecs whose bitrates are not compatible with this limit
	/// won't be used. 
	/// - Parameter bandwidth: The audio bandwidth limit to set in kbit/s. 
	
	public var audioBandwidthLimit: Int = 0
	{
	
		willSet
		{
			linphone_call_params_set_audio_bandwidth_limit(cPtr, CInt(newValue))
		}
	}
		
	/// Set the audio stream direction. 
	/// - Parameter direction: The audio stream ``MediaDirection`` associated with this
	/// call params. 
	
	/// Get the audio stream direction. 
	/// - Returns: The audio stream ``MediaDirection`` associated with the call params. 
	public var audioDirection: MediaDirection
	{
	
		get
		{ 
						return MediaDirection(rawValue: Int(linphone_call_params_get_audio_direction(cPtr).rawValue))!
		}
		set
		{
			linphone_call_params_set_audio_direction(cPtr, LinphoneMediaDirection(rawValue: CInt(newValue.rawValue)))
		}
	}
		
	/// Enable audio stream. 
	/// - Parameter enabled: A boolean value telling whether to enable audio or not. 
	
	/// Tell whether audio is enabled or not. 
	/// - Returns: A boolean value telling whether audio is enabled or not. 
	public var audioEnabled: Bool
	{
	
		get
		{ 
						return linphone_call_params_audio_enabled(cPtr) != 0
		}
		set
		{
			linphone_call_params_enable_audio(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Use to enable multicast rtp for audio stream. 
	/// -If enabled, outgoing calls put a multicast address from
	/// ``Core/getVideoMulticastAddr()`` into audio cline. In case of outgoing call
	/// audio stream is sent to this multicast address.  For incoming calls behavior is
	/// unchanged. 
	
	/// Use to get multicast state of audio stream. 
	/// - Returns: true if subsequent calls will propose multicast ip set by
	/// ``Core/setAudioMulticastAddr(ip:)`` 
	public var audioMulticastEnabled: Bool
	{
	
		get
		{ 
						return linphone_call_params_audio_multicast_enabled(cPtr) != 0
		}
		set
		{
			linphone_call_params_enable_audio_multicast(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Toggle feedback extension for AVP. 
	/// - Parameter enable: wether or not AVPF should be enabled 
	
	/// Whether or not the feedback extension will be used for AVP. 
	/// - Returns: true if AVPF is enabled, false otherwise 
	public var avpfEnabled: Bool
	{
	
		get
		{ 
						return linphone_call_params_avpf_enabled(cPtr) != 0
		}
		set
		{
			linphone_call_params_enable_avpf(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Enable camera stream. 
	/// - Parameter enabled: A boolean value telling whether to enable camera or not. 
	
	/// Tell whether camera is enabled or not. 
	/// The value returned by this function has a different meaning whether it is from
	/// local or remote parameters. The former states the will of the user to use the
	/// camera of his/her device. On the other hand, the latter is just a guess to know
	/// whether the remote party enabled its camera or not. For example, while the call
	/// is part of a conference a core will understand that the remote party disabled
	/// its camera if the thumbnail stream's direction is inactive. 
	/// - Returns: A boolean value telling whether camera is enabled or not. 
	public var cameraEnabled: Bool
	{
	
		get
		{ 
						return linphone_call_params_camera_enabled(cPtr) != 0
		}
		set
		{
			linphone_call_params_enable_camera(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Define whether capability negotiation (RFC5939) reINVITE is enabled. 
	/// - Parameter enable: true to enable capability negotiation reINVITE; false
	/// otherwise. 
	
	/// Check if the capability negotiation (RFC5939) reINVITE is enabled or not. 
	/// - Returns: true if capability negotiation reINVITE is enabled; false otherwise. 
	public var capabilityNegotiationReinviteEnabled: Bool
	{
	
		get
		{ 
						return linphone_call_params_capability_negotiation_reinvite_enabled(cPtr) != 0
		}
		set
		{
			linphone_call_params_enable_capability_negotiation_reinvite(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Enable capability negotiations (RFC5939). 
	/// - Parameter enabled: A boolean value telling whether to enable capability
	/// negotiations or not. 
	
	/// Indicates whether capability negotiations (RFC5939) is enabled. 
	/// - Returns: a boolean indicating the enablement of capability negotiations. 
	public var capabilityNegotiationsEnabled: Bool
	{
	
		get
		{ 
						return linphone_call_params_capability_negotiations_enabled(cPtr) != 0
		}
		set
		{
			linphone_call_params_enable_capability_negotiations(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Enable merging of cfg lines with consecutive indexes if capability negotiations
	/// (RFC5939) is enabled. 
	/// - Parameter enabled: A boolean value telling whether to merge pcfg and acfg
	/// lines 
	
	public var cfgLinesMergingEnabled: Bool?
	{
	
		willSet
		{
			linphone_call_params_enable_cfg_lines_merging(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Set video layout for conference. 
	/// - Parameter layout: ``Conference.Layout`` to use as default when creating a
	/// conference 
	
	/// Set video layout for conference. 
	/// - Returns: Preferred ``Conference.Layout`` to use at the start of a conference 
	public var conferenceVideoLayout: Conference.Layout
	{
	
		get
		{ 
						return Conference.Layout(rawValue: Int(linphone_call_params_get_conference_video_layout(cPtr).rawValue))!
		}
		set
		{
			linphone_call_params_set_conference_video_layout(cPtr, LinphoneConferenceLayout(rawValue: CUnsignedInt(newValue.rawValue)))
		}
	}
		
	
	/// Gets a list of ``Content`` set if exists. 
	/// - Returns: A list of ``Content`` set if exists, nil otherwise.         
	public var customContents: [Content]
	{
	
						var swiftList = [Content]()
			let cList = linphone_call_params_get_custom_contents(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(Content.getSwiftObject(cObject: data))
				listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
				bctbx_list_free_with_data(cList, belle_sip_object_unref)
			return swiftList

	}
		
	/// Enable sending of real early media (during outgoing calls). 
	/// - Parameter enabled: A boolean value telling whether to enable early media
	/// sending or not. 
	
	/// Indicate whether sending of early media was enabled. 
	/// - Returns: A boolean value telling whether sending of early media was enabled. 
	public var earlyMediaSendingEnabled: Bool
	{
	
		get
		{ 
						return linphone_call_params_early_media_sending_enabled(cPtr) != 0
		}
		set
		{
			linphone_call_params_enable_early_media_sending(cPtr, newValue==true ? 1:0)
		}
	}
		
	
	/// Tell whether FEC is enabled or not. 
	/// The FEC is enbaled whether a FEC stream is found. 
	/// - Returns: A boolean value telling whether FEC is enabled or not. 
	public var fecEnabled: Bool
	{
	
						return linphone_call_params_fec_enabled(cPtr) != 0

	}
		
	/// Force the from header of a call when instanciating it (if set, it precludes the
	/// search in proxy and primary contact) 
	/// - Parameter fromValue: The value of the forced from, null to delete it.    
	
	/// Get the from header in the CallParams. 
	/// - Returns: The content of the from header, may be null.    
	public var fromHeader: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_call_params_get_from_header(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_call_params_set_from_header(cPtr, newValue)
		}
	}
		
	/// Sets the given ``AudioDevice`` as default input for a call to be created later. 
	/// - Warning: This method won't have any effect once the call has been created!
	/// Instead use ``Call/setInputAudioDevice(audioDevice:)`` when call has been
	/// created. 
	/// - Parameter audioDevice: The ``AudioDevice``. nil does nothing.    
	
	/// Gets the default input audio device for a call that will be created using this
	/// call params. 
	/// - Warning: This method only concerns the call creation, it doesn't reflect the
	/// currently used input audio device of the call. Instead use
	/// ``Call/getInputAudioDevice()`` when call has been created. 
	/// - Returns: the ``AudioDevice`` that will be used by default as input when the
	/// call will be created    
	public var inputAudioDevice: AudioDevice?
	{
	
		get
		{ 
						let cPointer = linphone_call_params_get_input_audio_device(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = AudioDevice.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_call_params_set_input_audio_device(cPtr, newValue?.cPtr)
		}
	}
		
	
	/// Check if the capability negotiation (RFC5939) reINVITE is enabled or not. 
	/// - Returns: true if capability negotiation reINVITE is enabled; false otherwise. 
	/// - Deprecated: 16/12/2021 Use ``capabilityNegotiationReinviteEnabled()`` instead.
	@available(*, deprecated)
	public var isCapabilityNegotiationReinviteEnabled: Bool
	{
	
						return linphone_call_params_is_capability_negotiation_reinvite_enabled(cPtr) != 0

	}
		
	
	/// Indicates whether the call is being recorded. 
	/// - Returns: true if the call is being recorded, false otherwise. 
	public var isRecording: Bool
	{
	
						return linphone_call_params_is_recording(cPtr) != 0

	}
		
	
	/// Check if call parameters are valid. 
	/// - Returns: true if the parameters are valid; false otherwise. 
	public var isValid: Bool
	{
	
						return linphone_call_params_is_valid(cPtr) != 0

	}
		
	
	/// Tell whether the call is part of the locally managed conference. 
	/// - Warning: If a conference server is used to manage conferences, that function
	/// does not return true even if the conference is running. If you want to test
	/// whether the conference is running, you should test whether
	/// linphone_core_get_conference() return a non-null pointer. 
	/// - Returns: A boolean value telling whether the call is part of the locally
	/// managed conference. 
	public var localConferenceMode: Bool
	{
	
						return linphone_call_params_get_local_conference_mode(cPtr) != 0

	}
		
	/// Indicate low bandwith mode. 
	/// Configuring a call to low bandwidth mode will result in the core to activate
	/// several settings for the call in order to ensure that bitrate usage is lowered
	/// to the minimum possible. Typically, ptime (packetization time) will be
	/// increased, audio codec's output bitrate will be targetted to 20kbit/s provided
	/// that it is achievable by the codec selected after SDP handshake. Video is
	/// automatically disabled. 
	/// - Parameter enabled: A boolean value telling whether to activate the low
	/// bandwidth mode or not. 
	
	/// Tell whether the call has been configured in low bandwidth mode or not. 
	/// This mode can be automatically discovered thanks to a stun server when
	/// activate_edge_workarounds=1 in section [net] of configuration file. An
	/// application that would have reliable way to know network capacity may not use
	/// activate_edge_workarounds=1 but instead manually configure low bandwidth mode
	/// with ``enableLowBandwidth(enabled:)``. When enabled, this param may transform a
	/// call request with video in audio only mode. 
	/// - Returns: A boolean value telling whether the low bandwidth mode has been
	/// configured/detected. 
	public var lowBandwidthEnabled: Bool
	{
	
		get
		{ 
						return linphone_call_params_low_bandwidth_enabled(cPtr) != 0
		}
		set
		{
			linphone_call_params_enable_low_bandwidth(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Set requested media encryption for a call. 
	/// - Parameter encryption: The ``MediaEncryption`` to use for the call. 
	
	/// Get the kind of media encryption selected for the call. 
	/// - Returns: The kind of ``MediaEncryption`` selected for the call. 
	public var mediaEncryption: MediaEncryption
	{
	
		get
		{ 
						return MediaEncryption(rawValue: Int(linphone_call_params_get_media_encryption(cPtr).rawValue))!
		}
		set
		{
			linphone_call_params_set_media_encryption(cPtr, LinphoneMediaEncryption(rawValue: CUnsignedInt(newValue.rawValue)))
		}
	}
		
	/// Enable or disable the microphone at the call creation. 
	/// - Warning: This method won't have any effect once the call has been created!
	/// Instead use ``Call/setMicrophoneMuted(muted:)`` when call has been created. 
	/// - Parameter enable: true to enable the microphone, false to disable it. 
	
	/// Tells whether the microphone will be enabled when the call will be created. 
	/// - Warning: This method only concerns the call creation, it doesn't reflect the
	/// actual microphone status during a call. Instead use
	/// ``Call/getMicrophoneMuted()`` when call has been created. 
	/// - Returns: true if the microphone will be enabled, false if disabled. 
	public var micEnabled: Bool
	{
	
		get
		{ 
						return linphone_call_params_mic_enabled(cPtr) != 0
		}
		set
		{
			linphone_call_params_enable_mic(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets the given ``AudioDevice`` as default output for a call to be created
	/// later. 
	/// - Warning: This method won't have any effect once the call has been created!
	/// Instead use ``Call/setOutputAudioDevice(audioDevice:)`` when call has been
	/// created. 
	/// - Parameter audioDevice: The ``AudioDevice``. nil does nothing.    
	
	/// Gets the default output audio device for a call that will be created using this
	/// call params. 
	/// - Warning: This method only concerns the call creation, it doesn't reflect the
	/// currently used output audio device of the call. Instead use
	/// ``Call/getOutputAudioDevice()`` when call has been created. 
	/// - Returns: the ``AudioDevice`` that will be used by default as output when the
	/// call will be created    
	public var outputAudioDevice: AudioDevice?
	{
	
		get
		{ 
						let cPointer = linphone_call_params_get_output_audio_device(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = AudioDevice.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_call_params_set_output_audio_device(cPtr, newValue?.cPtr)
		}
	}
		
	/// Set requested level of privacy for the call. 
	/// - Parameter privacy: The LinphonePrivacyMask to used for the call. 
	
	/// Get requested level of privacy for the call. 
	/// - Returns: The LinphonePrivacyMask used for the call. 
	public var privacy: UInt
	{
	
		get
		{ 
						return UInt(linphone_call_params_get_privacy(cPtr))
		}
		set
		{
			linphone_call_params_set_privacy(cPtr, CUnsignedInt(newValue))
		}
	}
		
	/// Set the ``ProxyConfig`` to use for the call. 
	/// - Parameter proxyConfig: The ``ProxyConfig`` to use, or nil if none has been
	/// selected. The ``CallParams`` keep a reference on it and remove the older if it
	/// exists.    
	/// - Deprecated: 28/02/2021 Use ``setAccount(account:)`` instead. 
	
	/// Get the ``ProxyConfig`` that is used for the call. 
	/// - Returns: The selected ``ProxyConfig`` for the call, or nil if none has been
	/// selected.    
	/// - Deprecated: 28/02/2021 Use ``getAccount()`` instead. 
	public var proxyConfig: ProxyConfig?
	{
	@available(*, deprecated)
		get
		{ 
						let cPointer = linphone_call_params_get_proxy_config(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = ProxyConfig.getSwiftObject(cObject:cPointer!)
			return result
		}
	@available(*, deprecated)
		set
		{
			linphone_call_params_set_proxy_config(cPtr, newValue?.cPtr)
		}
	}
		
	/// Use to enable real time text following rfc4103. 
	/// If enabled, outgoing calls put a m=text line in SDP offer . 
	/// - Parameter yesno: if yes, subsequent outgoing calls will propose rtt 
	/// - Returns: 0 
	
	/// Use to get real time text following rfc4103. 
	/// - Returns: returns true if call rtt is activated. 
	public var realtimeTextEnabled: Bool
	{
	
						return linphone_call_params_realtime_text_enabled(cPtr) != 0

	}
	public func setRealtimetextenabled(newValue: Bool) throws
	{
		let exception_result = linphone_call_params_enable_realtime_text(cPtr, newValue==true ? 1:0)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "username setter returned value \(exception_result)")
		}
	}
		
	/// Use to set keep alive interval for real time text following rfc4103. 
	/// - Parameter interval: The keep alive interval for real time text, 25000 by
	/// default. 
	
	/// Use to get keep alive interval of real time text following rfc4103. 
	/// - Returns: returns keep alive interval of real time text. 
	public var realtimeTextKeepaliveInterval: UInt
	{
	
		get
		{ 
						return UInt(linphone_call_params_get_realtime_text_keepalive_interval(cPtr))
		}
		set
		{
			linphone_call_params_set_realtime_text_keepalive_interval(cPtr, CUnsignedInt(newValue))
		}
	}
		
	
	/// Get the framerate of the video that is received. 
	/// - Returns: The actual received framerate in frames per seconds, 0 if not
	/// available. 
	public var receivedFramerate: Float
	{
	
						return linphone_call_params_get_received_framerate(cPtr)

	}
		
	
	/// Get the definition of the received video. 
	/// - Returns: The received ``VideoDefinition`` or nil.    
	public var receivedVideoDefinition: VideoDefinition?
	{
	
						let cPointer = linphone_call_params_get_received_video_definition(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = VideoDefinition.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	/// Enable recording of the call. 
	/// This function must be used before the call parameters are assigned to the call.
	/// The call recording can be started and paused after the call is established with
	/// ``Call/startRecording()`` and linphone_call_pause_recording(). 
	/// - Parameter path: A string containing the path and filename of the file where
	/// audio/video streams are to be written. The filename must have an extension that
	/// maps to any of the supported file formats listed in ``MediaFileFormat`` enum.  
	///  
	
	/// Get the path for the audio recording of the call. 
	/// - Returns: The path to the audio recording of the call or nil.    
	public var recordFile: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_call_params_get_record_file(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_call_params_set_record_file(cPtr, newValue)
		}
	}
		
	/// Enables or disables RTP bundle mode (Media Multiplexing). 
	/// See https://datatracker.ietf.org/doc/html/rfc8843 for more information about
	/// the feature. When enabled, liblinphone will try to negociate the use of a
	/// single port for all streams. It automatically enables rtcp-mux. 
	/// See https://datatracker.ietf.org/doc/html/rfc8843 for more information about
	/// the feature. When enabled, liblinphone will try to negociate the use of a
	/// single port for all streams. It automatically enables rtcp-mux. 
	/// - Parameter value: a boolean to indicate whether the feature is to be enabled. 
	/// - Deprecated: This property can no longer be controlled via ``CallParams``. Use
	/// ``AccountParams/enableRtpBundle(value:)``.
	
	/// Indicates whether RTP bundle mode (also known as Media Multiplexing) is
	/// enabled. 
	/// See https://datatracker.ietf.org/doc/html/rfc8843 for more information. 
	/// See https://datatracker.ietf.org/doc/html/rfc8843 for more information. 
	/// - Returns: a boolean indicating the enablement of rtp bundle mode. 
	public var rtpBundleEnabled: Bool
	{
	
		get
		{ 
						return linphone_call_params_rtp_bundle_enabled(cPtr) != 0
		}
	@available(*, deprecated)
		set
		{
			linphone_call_params_enable_rtp_bundle(cPtr, newValue==true ? 1:0)
		}
	}
		
	
	/// Get the RTP profile being used. 
	/// - Returns: The RTP profile.    
	public var rtpProfile: String
	{
	
			
			let cPointer = linphone_call_params_get_rtp_profile(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	/// Enable screen sharing stream. 
	/// - Parameter enabled: A boolean value telling whether to enable screen sharing
	/// or not. 
	
	/// Tell whether screen sharing is enabled or not. 
	/// - Returns: A boolean value telling whether screen sharing is enabled or not. 
	public var screenSharingEnabled: Bool
	{
	
		get
		{ 
						return linphone_call_params_screen_sharing_enabled(cPtr) != 0
		}
		set
		{
			linphone_call_params_enable_screen_sharing(cPtr, newValue==true ? 1:0)
		}
	}
		
	
	/// Get the framerate of the video that is sent. 
	/// - Returns: The actual sent framerate in frames per seconds, 0 if not available. 
	public var sentFramerate: Float
	{
	
						return linphone_call_params_get_sent_framerate(cPtr)

	}
		
	
	/// Get the definition of the sent video. 
	/// - Returns: The sent ``VideoDefinition`` or nil.    
	public var sentVideoDefinition: VideoDefinition?
	{
	
						let cPointer = linphone_call_params_get_sent_video_definition(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = VideoDefinition.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	/// Set the session name of the media session (ie in SDP). 
	/// Subject from the SIP message (which is different) can be set using
	/// linphone_call_params_set_custom_header(). 
	/// - Parameter name: The session name to be used.    
	
	/// Get the session name of the media session (ie in SDP). 
	/// Subject from the SIP message can be retrieved using
	/// ``getCustomHeader(headerName:)`` and is different. 
	/// - Returns: The session name of the media session or nil.    
	public var sessionName: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_call_params_get_session_name(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_call_params_set_session_name(cPtr, newValue)
		}
	}
		
	/// Enable merging of tcap lines with consecutive indexes if capability
	/// negotiations (RFC5939) is enabled. 
	/// - Parameter enabled: A boolean value telling whether to merge tcap lines 
	
	public var tcapLineMergingEnabled: Bool?
	{
	
		willSet
		{
			linphone_call_params_enable_tcap_line_merging(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Define whether tone indications are enabled. 
	/// - Parameter enable: true to enable tone indications; false otherwise. 
	
	/// Check if tone indications are enabled. 
	/// - Returns: true if tone indications are enabled; false otherwise. 
	public var toneIndicationsEnabled: Bool
	{
	
		get
		{ 
						return linphone_call_params_tone_indications_enabled(cPtr) != 0
		}
		set
		{
			linphone_call_params_enable_tone_indications(cPtr, newValue==true ? 1:0)
		}
	}
		
	
	/// Get the audio payload type that has been selected by a call. 
	/// - Returns: The selected ``PayloadType``. nil is returned if no audio payload
	/// type has been selected by the call.    
	public var usedAudioPayloadType: PayloadType?
	{
	
						let cPointer = linphone_call_params_get_used_audio_payload_type(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = PayloadType.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Get the text payload type that has been selected by a call. 
	/// - Returns: The selected ``PayloadType``. nil is returned if no text payload
	/// type has been selected by the call.    
	public var usedTextPayloadType: PayloadType?
	{
	
						let cPointer = linphone_call_params_get_used_text_payload_type(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = PayloadType.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Get the video payload type that has been selected by a call. 
	/// - Returns: The selected ``PayloadType``. nil is returned if no video payload
	/// type has been selected by the call.    
	public var usedVideoPayloadType: PayloadType?
	{
	
						let cPointer = linphone_call_params_get_used_video_payload_type(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = PayloadType.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	/// Assign a user data to the call params. 
	/// - Parameter userData: The user data to associate with the call params.    
	
	/// Get the user data associated with the call params. 
	/// - Returns: The user data associated with the call params.    
	public var userData: UnsafeMutableRawPointer?
	{
	
		get
		{ 
						return linphone_call_params_get_user_data(cPtr)
		}
		set
		{
			linphone_call_params_set_user_data(cPtr, newValue)
		}
	}
		
	/// Set the video stream direction. 
	/// - Parameter direction: The video stream ``MediaDirection`` associated with this
	/// call params. 
	
	/// Get the video stream direction. 
	/// - Returns: The video stream ``MediaDirection`` associated with the call params. 
	public var videoDirection: MediaDirection
	{
	
		get
		{ 
						return MediaDirection(rawValue: Int(linphone_call_params_get_video_direction(cPtr).rawValue))!
		}
		set
		{
			linphone_call_params_set_video_direction(cPtr, LinphoneMediaDirection(rawValue: CInt(newValue.rawValue)))
		}
	}
		
	/// Enable video stream. 
	/// - Parameter enabled: A boolean value telling whether to enable video or not. 
	
	/// Tell whether video is enabled or not. 
	/// - Returns: A boolean value telling whether video is enabled or not. 
	public var videoEnabled: Bool
	{
	
		get
		{ 
						return linphone_call_params_video_enabled(cPtr) != 0
		}
		set
		{
			linphone_call_params_enable_video(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Use to enable multicast rtp for video stream. 
	/// If enabled, outgoing calls put a multicast address from
	/// ``Core/getVideoMulticastAddr()`` into video cline. In case of outgoing call
	/// video stream is sent to this multicast address.  For incoming calls behavior is
	/// unchanged. 
	/// - Parameter yesno: if yes, subsequent outgoing calls will propose multicast ip
	/// set by ``Core/setVideoMulticastAddr(ip:)`` 
	
	/// Use to get multicast state of video stream. 
	/// - Returns: true if subsequent calls will propose multicast ip set by
	/// ``Core/setVideoMulticastAddr(ip:)`` 
	public var videoMulticastEnabled: Bool
	{
	
		get
		{ 
						return linphone_call_params_video_multicast_enabled(cPtr) != 0
		}
		set
		{
			linphone_call_params_enable_video_multicast(cPtr, newValue==true ? 1:0)
		}
	}
		
	
	
	/// Adds a ``Content`` to be added to the INVITE SDP. 
	/// - Parameter content: The ``Content`` to be added.    
	public func addCustomContent(content:Content) 
	{
		linphone_call_params_add_custom_content(cPtr, content.cPtr)
	}
	
	
	
	/// Add a custom SIP header in the INVITE for a call. 
	/// - Parameter headerName: The name of the header to add.    
	/// - Parameter headerValue: The content of the header to add.    
	public func addCustomHeader(headerName:String, headerValue:String?) 
	{
		linphone_call_params_add_custom_header(cPtr, headerName, headerValue)
	}
	
	
	
	/// Add a custom attribute related to all the streams in the SDP exchanged within
	/// SIP messages during a call. 
	/// - Parameter attributeName: The name of the attribute to add.    
	/// - Parameter attributeValue: The content value of the attribute to add.    
	public func addCustomSdpAttribute(attributeName:String, attributeValue:String?) 
	{
		linphone_call_params_add_custom_sdp_attribute(cPtr, attributeName, attributeValue)
	}
	
	
	
	/// Add a custom attribute related to a specific stream in the SDP exchanged within
	/// SIP messages during a call. 
	/// - Parameter type: The type of the stream to add a custom SDP attribute to. 
	/// - Parameter attributeName: The name of the attribute to add.    
	/// - Parameter attributeValue: The content value of the attribute to add.    
	public func addCustomSdpMediaAttribute(type:StreamType, attributeName:String, attributeValue:String?) 
	{
		linphone_call_params_add_custom_sdp_media_attribute(cPtr, LinphoneStreamType(rawValue: CUnsignedInt(type.rawValue)), attributeName, attributeValue)
	}
	
	
	
	/// Indicates whether cfg lines with consecutive indexes are going to be merged or
	/// not if capability negotiations (RFC5939) is enabled. 
	/// - Returns: a boolean indicating the enablement of pcfg and acfg line merging 
	public func cfgLinesMerged() -> Bool
	{
		return linphone_call_params_cfg_lines_merged(cPtr) != 0
	}
	
	
	
	/// Clear the custom SDP attributes related to all the streams in the SDP exchanged
	/// within SIP messages during a call. 
	public func clearCustomSdpAttributes() 
	{
		linphone_call_params_clear_custom_sdp_attributes(cPtr)
	}
	
	
	
	/// Clear the custom SDP attributes related to a specific stream in the SDP
	/// exchanged within SIP messages during a call. 
	/// - Parameter type: The type of the stream to clear the custom SDP attributes
	/// from. 
	public func clearCustomSdpMediaAttributes(type:StreamType) 
	{
		linphone_call_params_clear_custom_sdp_media_attributes(cPtr, LinphoneStreamType(rawValue: CUnsignedInt(type.rawValue)))
	}
	
	
	
	/// Copy an existing ``CallParams`` object to a new ``CallParams`` object. 
	/// ``copy()`` is error-prone, leading to inconsistent parameters being passed to
	/// ``Core/inviteAddressWithParams(addr:params:)`` or
	/// ``Call/acceptWithParams(params:)``. - Deprecated: use exclusively
	/// ``Core/createCallParams(call:)`` to create ``CallParams`` object. 
	/// - Returns: A copy of the ``CallParams`` object.       
	@available(*, deprecated)
	public func copy() -> CallParams?
	{
		let cPointer = linphone_call_params_copy(cPtr)
		if (cPointer == nil) {
			return nil
		}
		let result = CallParams.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Define whether ringing is disabled. 
	/// - Parameter disable: true to disable ringing; false otherwise. 
	public func disableRinging(disable:Bool) 
	{
		linphone_call_params_disable_ringing(cPtr, disable==true ? 1:0)
	}
	
	
	
	/// Get a custom SIP header. 
	/// - Parameter headerName: The name of the header to get.    
	/// - Returns: The content of the header or nil if not found.    
	public func getCustomHeader(headerName:String) -> String
	{
		let cstr = linphone_call_params_get_custom_header(cPtr, headerName)
		let result = charArrayToString(charPointer: cstr)
		return result
	}
	
	
	
	/// Get a custom SDP attribute that is related to all the streams. 
	/// - Parameter attributeName: The name of the attribute to get.    
	/// - Returns: The content value of the attribute or nil if not found.    
	public func getCustomSdpAttribute(attributeName:String) -> String
	{
		let cstr = linphone_call_params_get_custom_sdp_attribute(cPtr, attributeName)
		let result = charArrayToString(charPointer: cstr)
		return result
	}
	
	
	
	/// Get a custom SDP attribute that is related to a specific stream. 
	/// - Parameter type: The type of the stream to add a custom SDP attribute to. 
	/// - Parameter attributeName: The name of the attribute to get.    
	/// - Returns: The content value of the attribute or nil if not found.    
	public func getCustomSdpMediaAttribute(type:StreamType, attributeName:String) -> String
	{
		let cstr = linphone_call_params_get_custom_sdp_media_attribute(cPtr, LinphoneStreamType(rawValue: CUnsignedInt(type.rawValue)), attributeName)
		let result = charArrayToString(charPointer: cstr)
		return result
	}
	
	
	
	/// Returns true if a custom SDP attribute that is related to all the streams is
	/// present. 
	/// - Parameter attributeName: The name of the attribute to get.    
	/// - Returns: Whether the attribute is present. 
	public func hasCustomSdpAttribute(attributeName:String) -> Bool
	{
		return linphone_call_params_has_custom_sdp_attribute(cPtr, attributeName) != 0
	}
	
	
	
	/// Indicates whether a custom SDP attribute that is related to a specific stream
	/// is present or not. 
	/// - Parameter type: The type of the stream to add a custom SDP attribute to. 
	/// - Parameter attributeName: The name of the attribute to get.    
	/// - Returns: Whether the attribute is present. 
	public func hasCustomSdpMediaAttribute(type:StreamType, attributeName:String) -> Bool
	{
		return linphone_call_params_has_custom_sdp_media_attribute(cPtr, LinphoneStreamType(rawValue: CUnsignedInt(type.rawValue)), attributeName) != 0
	}
	
	
	
	/// Returns the encryption is supported. 
	/// - Parameter encryption: The ``MediaEncryption`` to check whether is supported 
	/// - Returns: a boolean indicating whether the encryption is supported 
	public func isMediaEncryptionSupported(encryption:MediaEncryption) -> Bool
	{
		return linphone_call_params_is_media_encryption_supported(cPtr, LinphoneMediaEncryption(rawValue: CUnsignedInt(encryption.rawValue))) != 0
	}
	
	
	
	/// Check if ringing is disabled. 
	/// - Returns: true if ringing is disabled; false otherwise. 
	public func ringingDisabled() -> Bool
	{
		return linphone_call_params_ringing_disabled(cPtr) != 0
	}
	
	
	
	/// Indicates whether tcap lines with consecutive indexes are going to be merged or
	/// not if capability negotiations (RFC5939) is enabled. 
	/// - Returns: a boolean indicating the enablement of tcap line merging 
	public func tcapLinesMerged() -> Bool
	{
		return linphone_call_params_tcap_lines_merged(cPtr) != 0
	}
}


/// This object carry various statistic informations regarding the quality of an
/// audio or video stream for a given ``Call``. 
/// To receive these informations periodically and as soon as they are computed,
/// implement the call_stats_updated() callback inside a ``CoreDelegate``.
/// At any time, the application can access latest computed statistics using
/// ``Call/getAudioStats()`` and ``Call/getVideoStats()``. 
public class CallStats : LinphoneObject
{

	static public func getSwiftObject(cObject:OpaquePointer) -> CallStats {
		let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
		if (result != nil) {
			return Unmanaged<CallStats>.fromOpaque(result!).takeUnretainedValue()
		}
		let sObject = CallStats(cPointer: cObject)
		belle_sip_object_data_set(UnsafeMutablePointer(cObject), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(sObject).toOpaque()), nil)
		return sObject
	}

	public var getCobject: OpaquePointer? {
		return cPtr
	}
	
	
	/// Get the bandwidth measurement of the received stream, expressed in kbit/s,
	/// including IP/UDP/RTP headers. 
	/// - Returns: The bandwidth measurement of the received stream in kbit/s. 
	public var downloadBandwidth: Float
	{
	
						return linphone_call_stats_get_download_bandwidth(cPtr)

	}
		
	
	/// Get the estimated bandwidth measurement of the received stream, expressed in
	/// kbit/s, including IP/UDP/RTP headers. 
	/// - Returns: The estimated bandwidth measurement of the received stream in
	/// kbit/s. 
	public var estimatedDownloadBandwidth: Float
	{
	
						return linphone_call_stats_get_estimated_download_bandwidth(cPtr)

	}
		
	
	/// If the FEC is enabled, gets the cumulative number of lost source packets of the
	/// RTP session that have not been repaired by the current FEC stream. 
	/// - Returns: The cumulative number of lost packets 
	public var fecCumulativeLostPacketsNumber: UInt64
	{
	
						return linphone_call_stats_get_fec_cumulative_lost_packets_number(cPtr)

	}
		
	
	/// Get the bandwidth measurement of the part of the received stream dedicated to
	/// FEC, expressed in kbit/s, including IP/UDP/RTP headers. 
	/// - Returns: The bandwidth measurement of the received FEC stream in kbit/s. 
	public var fecDownloadBandwidth: Float
	{
	
						return linphone_call_stats_get_fec_download_bandwidth(cPtr)

	}
		
	
	/// If the FEC is enabled, gets the cumulative number of source packets of the RTP
	/// session that have been repaired by the current FEC stream. 
	/// - Returns: The cumulative number of repaired packets 
	public var fecRepairedPacketsNumber: UInt64
	{
	
						return linphone_call_stats_get_fec_repaired_packets_number(cPtr)

	}
		
	
	/// Get the bandwidth measurement of the part of the sent stream dedicated to FEC,
	/// expressed in kbit/s, including IP/UDP/RTP headers. 
	/// - Returns: The bandwidth measurement of the sent stream in kbit/s. 
	public var fecUploadBandwidth: Float
	{
	
						return linphone_call_stats_get_fec_upload_bandwidth(cPtr)

	}
		
	
	/// Get the state of ICE processing. 
	/// - Returns: The ``IceState`` of ICE processing 
	public var iceState: IceState
	{
	
						return IceState(rawValue: Int(linphone_call_stats_get_ice_state(cPtr).rawValue))!

	}
		
	
	/// Get the IP address family of the remote peer. 
	/// - Returns: The IP address family ``Address.Family`` of the remote peer. 
	public var ipFamilyOfRemote: Address.Family
	{
	
						return Address.Family(rawValue: Int(linphone_call_stats_get_ip_family_of_remote(cPtr).rawValue))!

	}
		
	
	/// Did ZRTP used a Post Quantum algorithm to perform a key exchange. 
	/// - Returns: true if the ZRTP key exchange was performed using a PQ algo false
	/// otherwise: ZRTP exchange not completed or not using a PQ algo 
	public var isZrtpKeyAgreementAlgoPostQuantum: Bool
	{
	
						return linphone_call_stats_is_zrtp_key_agreement_algo_post_quantum(cPtr) != 0

	}
		
	
	/// Get the jitter buffer size in ms. 
	/// - Returns: The jitter buffer size in ms. 
	public var jitterBufferSizeMs: Float
	{
	
						return linphone_call_stats_get_jitter_buffer_size_ms(cPtr)

	}
		
	
	/// Gets the cumulative number of late packets. 
	/// - Returns: The cumulative number of late packets 
	public var latePacketsCumulativeNumber: UInt64
	{
	
						return linphone_call_stats_get_late_packets_cumulative_number(cPtr)

	}
		
	
	/// Gets the local late rate since last report, expressed as a percentage. 
	/// - Returns: The local late rate 
	public var localLateRate: Float
	{
	
						return linphone_call_stats_get_local_late_rate(cPtr)

	}
		
	
	/// Get the local loss rate since last report, expressed as a percentage. 
	/// - Returns: The local loss rate 
	public var localLossRate: Float
	{
	
						return linphone_call_stats_get_local_loss_rate(cPtr)

	}
		
	
	/// Gets the remote reported interarrival jitter, expressed in seconds. 
	/// - Returns: The interarrival jitter at last received receiver report 
	public var receiverInterarrivalJitter: Float
	{
	
						return linphone_call_stats_get_receiver_interarrival_jitter(cPtr)

	}
		
	
	/// Gets the remote reported loss rate since last report, expressed as a
	/// percentage. 
	/// - Returns: The receiver loss rate 
	public var receiverLossRate: Float
	{
	
						return linphone_call_stats_get_receiver_loss_rate(cPtr)

	}
		
	
	/// Get the round trip delay in s. 
	/// - Returns: The round trip delay in s. 
	public var roundTripDelay: Float
	{
	
						return linphone_call_stats_get_round_trip_delay(cPtr)

	}
		
	
	/// Get the bandwidth measurement of the received RTCP, expressed in kbit/s,
	/// including IP/UDP/RTP headers. 
	/// - Returns: The bandwidth measurement of the received RTCP in kbit/s. 
	public var rtcpDownloadBandwidth: Float
	{
	
						return linphone_call_stats_get_rtcp_download_bandwidth(cPtr)

	}
		
	
	/// Get the bandwidth measurement of the sent RTCP, expressed in kbit/s, including
	/// IP/UDP/RTP headers. 
	/// - Returns: The bandwidth measurement of the sent RTCP in kbit/s. 
	public var rtcpUploadBandwidth: Float
	{
	
						return linphone_call_stats_get_rtcp_upload_bandwidth(cPtr)

	}
		
	
	/// Get the RTP cumulative number of incoming packet lost. 
	/// - Returns: The number of RTP cumulative number of incoming packet lost 
	public var rtpCumPacketLoss: Int64
	{
	
						return linphone_call_stats_get_rtp_cum_packet_loss(cPtr)

	}
		
	
	/// Get the RTP incoming packets discarded because the queue exceeds its max size. 
	/// - Returns: The RTP incoming packets discarded because the queue exceeds its max
	/// size 
	public var rtpDiscarded: UInt64
	{
	
						return linphone_call_stats_get_rtp_discarded(cPtr)

	}
		
	
	/// Get the number of received bytes excluding IPv4/IPv6/UDP headers and including
	/// late and duplicate packets. 
	/// - Returns: the number of received bytes excluding IPv4/IPv6/UDP headers and
	/// including late and duplicate packets 
	public var rtpHwRecv: UInt64
	{
	
						return linphone_call_stats_get_rtp_hw_recv(cPtr)

	}
		
	
	/// Get the number of RTP received packets. 
	/// - Returns: The number of RTP received packets 
	public var rtpPacketRecv: UInt64
	{
	
						return linphone_call_stats_get_rtp_packet_recv(cPtr)

	}
		
	
	/// Get the number of RTP outgoing packets. 
	/// - Returns: The number of RTP outgoing packets 
	public var rtpPacketSent: UInt64
	{
	
						return linphone_call_stats_get_rtp_packet_sent(cPtr)

	}
		
	
	/// Get the RTP incoming recv_bytes of payload and delivered in time to the
	/// application. 
	/// - Returns: The number of recv_bytes of payload and delivered in time to the
	/// application 
	public var rtpRecv: UInt64
	{
	
						return linphone_call_stats_get_rtp_recv(cPtr)

	}
		
	
	/// Get the RTP outgoing sent_bytes (excluding IP header) 
	/// - Returns: The number of outgoing sent_bytes (excluding IP header) 
	public var rtpSent: UInt64
	{
	
						return linphone_call_stats_get_rtp_sent(cPtr)

	}
		
	
	/// Gets the local interarrival jitter, expressed in seconds. 
	/// - Returns: The interarrival jitter at last emitted sender report 
	public var senderInterarrivalJitter: Float
	{
	
						return linphone_call_stats_get_sender_interarrival_jitter(cPtr)

	}
		
	
	/// Get the local loss rate since last report, expressed as a percentage. 
	/// - Returns: The sender loss rate 
	public var senderLossRate: Float
	{
	
						return linphone_call_stats_get_sender_loss_rate(cPtr)

	}
		
	
	/// Get the method used for SRTP key exchange. 
	/// - Returns: The ``MediaEncryption`` method used to exchange the SRTP keys    
	public var srtpSource: MediaEncryption
	{
	
						return MediaEncryption(rawValue: Int(linphone_call_stats_get_srtp_source(cPtr).rawValue))!

	}
		
	
	/// Get the SRTP Cryto suite in use. 
	/// - Returns: The SRTP crypto suite currently in use ``SrtpSuite``    
	public var srtpSuite: SrtpSuite
	{
	
						return SrtpSuite(rawValue: Int(linphone_call_stats_get_srtp_suite(cPtr).rawValue))!

	}
		
	
	/// Get the type of the stream the stats refer to. 
	/// - Returns: The ``StreamType`` the stats refer to 
	public var type: StreamType
	{
	
						return StreamType(rawValue: Int(linphone_call_stats_get_type(cPtr).rawValue))!

	}
		
	
	/// Get the bandwidth measurement of the sent stream, expressed in kbit/s,
	/// including IP/UDP/RTP headers. 
	/// - Returns: The bandwidth measurement of the sent stream in kbit/s. 
	public var uploadBandwidth: Float
	{
	
						return linphone_call_stats_get_upload_bandwidth(cPtr)

	}
		
	
	/// Get the state of uPnP processing. 
	/// - Returns: The ``UpnpState`` of uPnP processing. 
	public var upnpState: UpnpState
	{
	
						return UpnpState(rawValue: Int(linphone_call_stats_get_upnp_state(cPtr).rawValue))!

	}
		
	/// Sets the user data in the ``CallStats`` object. 
	/// - Parameter data: the user data.    
	
	/// Gets the user data in the ``CallStats`` object. 
	/// - Returns: the user data.    
	public var userData: UnsafeMutableRawPointer?
	{
	
		get
		{ 
						return linphone_call_stats_get_user_data(cPtr)
		}
		set
		{
			linphone_call_stats_set_user_data(cPtr, newValue)
		}
	}
		
	
	/// Get the ZRTP algorithm statistics details (authentication method) 
	/// - Returns: The auth tag algo 
	public var zrtpAuthTagAlgo: String
	{
	
			
			let cPointer = linphone_call_stats_get_zrtp_auth_tag_algo(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Get the ZRTP algorithm statistics details (cipher) 
	/// - Returns: The cipher algo 
	public var zrtpCipherAlgo: String
	{
	
			
			let cPointer = linphone_call_stats_get_zrtp_cipher_algo(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Get the ZRTP algorithm statistics details (hash function) 
	/// - Returns: The hash algo 
	public var zrtpHashAlgo: String
	{
	
			
			let cPointer = linphone_call_stats_get_zrtp_hash_algo(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Get the ZRTP algorithm statistics details (key agreeement) 
	/// - Returns: The key agreement algo 
	public var zrtpKeyAgreementAlgo: String
	{
	
			
			let cPointer = linphone_call_stats_get_zrtp_key_agreement_algo(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Get the ZRTP algorithm statistics details (SAS display) 
	/// - Returns: The sas algo 
	public var zrtpSasAlgo: String
	{
	
			
			let cPointer = linphone_call_stats_get_zrtp_sas_algo(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
	}


/// The ``CardDavParams`` object represents a remote CardDAV server used by
/// ``MagicSearch`` as a plugin source. 
public class CardDavParams : LinphoneObject
{

	static public func getSwiftObject(cObject:OpaquePointer) -> CardDavParams {
		let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
		if (result != nil) {
			return Unmanaged<CardDavParams>.fromOpaque(result!).takeUnretainedValue()
		}
		let sObject = CardDavParams(cPointer: cObject)
		belle_sip_object_data_set(UnsafeMutablePointer(cObject), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(sObject).toOpaque()), nil)
		return sObject
	}

	public var getCobject: OpaquePointer? {
		return cPtr
	}
	
	/// Sets the list of vCard RFC fields to use to match the domain filter on. 
	/// For example, you can use "IMPP". 
	/// - Parameter list: the list of vCard RFC fields to use to match the domain
	/// filter on.      
	/// - Warning: DO NOT USE ANYMORE, it won't have any effect. 
	/// - Deprecated: 06/10/2025 domain filter will be applied locally instead of using
	/// a CardDAV prop filter XML tag. 
	
	/// Gets the list of vCard RFC fields to use to match the domain filter on. 
	/// For example, you can use "IMPP". 
	/// - Returns: The list of vCard fields to make the query on using domain filter.  
	///       
	/// - Warning: DO NOT USE ANYMORE, it won't have any effect. 
	/// - Deprecated: 06/10/2025 domain filter will be applied locally instead of using
	/// a CardDAV prop filter XML tag. 
	public var domainFields: [String]
	{
	@available(*, deprecated)
		get
		{ 
						var swiftList = [String]()
			let cList = linphone_card_dav_params_get_domain_fields(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				swiftList.append(String(cString: unsafeBitCast(listTemp!.pointee.data, to: UnsafePointer<CChar>.self)))
				listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp!.pointee.next)
			}
				bctbx_list_free_with_data(cList, bctbx_free)
			return swiftList
		}
	@available(*, deprecated)
		set
		{
			var cList: UnsafeMutablePointer<bctbx_list_t>? = nil
			for data in newValue {
				let sData:NSString = data as NSString
				cList = bctbx_list_append(cList, unsafeBitCast(sData.utf8String, to: UnsafeMutablePointer<CChar>.self))
			}
			linphone_card_dav_params_set_domain_fields(cPtr, cList)
		}
	}
		
	/// Sets the matching policy: true if the remote vCard value must match the filter
	/// exactly, false if it only needs to contain the filter. 
	/// - Parameter exactMatch: the policy to use: true if the remote vCard value must
	/// match the filter exactly, false if it only needs to contain the filter. 
	
	/// Gets the matching policy: true if the remote vCard value must match the filter
	/// exactly, false if it only needs to contain the filter. 
	/// - Returns: The matching policy: true if the remote vCard value must match the
	/// filter exactly, false if it only needs to contain the filter. 
	public var useExactMatchPolicy: Bool
	{
	
		get
		{ 
						return linphone_card_dav_params_get_use_exact_match_policy(cPtr) != 0
		}
		set
		{
			linphone_card_dav_params_set_use_exact_match_policy(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets the list of vCard RFC fields to use to match user input text on. 
	/// For example, you can use "FN", "N", "IMPP", "ORG", etc... 
	/// - Parameter list: the list of vCard RFC fields to use to match user input text
	/// on.      
	
	/// Gets the list of vCard RFC fields to use to match user input text on. 
	/// For example, you can use "FN", "N", "IMPP", "ORG", etc... 
	/// - Returns: The list of vCard fields to make the query on using user input.     
	///    
	public var userInputFields: [String]
	{
	
		get
		{ 
						var swiftList = [String]()
			let cList = linphone_card_dav_params_get_user_input_fields(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				swiftList.append(String(cString: unsafeBitCast(listTemp!.pointee.data, to: UnsafePointer<CChar>.self)))
				listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp!.pointee.next)
			}
				bctbx_list_free_with_data(cList, bctbx_free)
			return swiftList
		}
		set
		{
			var cList: UnsafeMutablePointer<bctbx_list_t>? = nil
			for data in newValue {
				let sData:NSString = data as NSString
				cList = bctbx_list_append(cList, unsafeBitCast(sData.utf8String, to: UnsafeMutablePointer<CChar>.self))
			}
			linphone_card_dav_params_set_user_input_fields(cPtr, cList)
		}
	}
		
	
	
	/// Instantiates a new ``CardDavParams`` with values from source. 
	/// - Returns: The newly created ``CardDavParams`` object.    
	public func clone() -> CardDavParams?
	{
		let cPointer = linphone_card_dav_params_clone(cPtr)
		if (cPointer == nil) {
			return nil
		}
		let result = CardDavParams.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
}


/// A ``ChatMessage`` represents an instant message that can be send or received
/// through a ``ChatRoom``. 
/// To create a ``ChatMessage``, use ``ChatRoom/createEmptyMessage()``, then either
/// add text using ``addUtf8TextContent(text:)`` or a ``Content`` with file
/// informations using ``addFileContent(content:)``. A valid ``Content`` for file
/// transfer must contain a type and subtype, the name of the file and it's size.
/// Finally call ``send()`` to send it.
/// To send files through a ``ChatMessage``, you need to have configured a file
/// transfer server URL with ``Core/setFileTransferServer(serverUrl:)``. On the
/// receiving side, either use ``downloadContent(content:)`` to download received
/// files or enable auto-download in the ``Core`` using
/// ``Core/setMaxSizeForAutoDownloadIncomingFiles(size:)``, -1 disabling the
/// feature and 0 always downloading files no matter it's size.
/// Keep in mind a ``ChatMessage`` created by a ``Basic`` ``ChatRoom`` can only
/// contain one ``Content``, either text or file. 
public class ChatMessage : LinphoneObject
{
	var delegateManagers : [ChatMessageDelegateManager] = []

	static public func getSwiftObject(cObject:OpaquePointer) -> ChatMessage {
		let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
		if (result != nil) {
			return Unmanaged<ChatMessage>.fromOpaque(result!).takeUnretainedValue()
		}
		let sObject = ChatMessage(cPointer: cObject)
		belle_sip_object_data_set(UnsafeMutablePointer(cObject), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(sObject).toOpaque()), nil)
		return sObject
	}

	public var getCobject: OpaquePointer? {
		return cPtr
	}

	
	///``Direction`` is used to indicate if a message is outgoing or incoming. 
	public enum Direction:Int
	{
		
		/// Incoming message. 
		case Incoming = 0
		/// Outgoing message. 
		case Outgoing = 1
	}

	
	///``State`` is used to notify if messages have been successfully delivered or
	///not. 
	public enum State:Int
	{
		
		/// Initial state. 
		case Idle = 0
		/// Delivery in progress. 
		case InProgress = 1
		/// Message successfully delivered and acknowledged by the server. 
		case Delivered = 2
		/// Message was not delivered. 
		case NotDelivered = 3
		/// Message was received and acknowledged but cannot get file from server. 
		case FileTransferError = 4
		/// File transfer has been completed successfully. 
		case FileTransferDone = 5
		/// Message successfully delivered an acknowledged by the remote user. 
		case DeliveredToUser = 6
		/// Message successfully displayed to the remote user. 
		case Displayed = 7
		/// File transfer is in progress. 
		case FileTransferInProgress = 8
		/// Message is pending delivery. 
		case PendingDelivery = 9
		/// The user cancelled the file transfer. 
		case FileTransferCancelling = 10
		/// Message cannot be sent right now and it is queued. 
		case Queued = 11
	}
	
	
	
	public func addDelegate(delegate: ChatMessageDelegate)
	{
		let manager = ChatMessageDelegateManager()
		manager.delegate = delegate
		delegateManagers.append(manager)
		linphone_chat_message_add_callbacks(cPtr, manager.cPtr)
		belle_sip_object_unref(UnsafeMutableRawPointer(manager.cPtr))
		belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(self).toOpaque()), nil)
	}	
	
	
	public func removeDelegate(delegate: ChatMessageDelegate)
	{
		if let index = delegateManagers.firstIndex(where: { $0.delegate === delegate as AnyObject}) {
			linphone_chat_message_remove_callbacks(cPtr, delegateManagers[index].cPtr)
			delegateManagers.remove(at: index)
			belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(self).toOpaque()), nil)
		}
	}	
	/// Linphone message has an app-specific field that can store a text. 
	/// The application might want to use it for keeping data over restarts, like
	/// thumbnail image path.
	/// Invoking this function will attempt to update the message storage to reflect
	/// the change if it is enabled.
	/// - Parameter data: the data to store into the message.    
	
	/// Linphone message has an app-specific field that can store a text. 
	/// The application might want to use it for keeping data over restarts, like
	/// thumbnail image path. 
	/// - Returns: the application-specific data or nil if none has been stored.    
	public var appdata: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_chat_message_get_appdata(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_chat_message_set_appdata(cPtr, newValue)
		}
	}
		
	
	/// Gets the SIP call-id accociated with the message. 
	/// - Returns: the call-id    
	public var callId: String
	{
	
			
			let cPointer = linphone_chat_message_get_call_id(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Returns the chatroom this message belongs to. 
	/// - Returns: the ``ChatRoom`` in which this message has been sent or received.    
	public var chatRoom: ChatRoom?
	{
	
						let cPointer = linphone_chat_message_get_chat_room(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = ChatRoom.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	/// Sets the content type of a chat message. 
	/// This content type must match a content that is text representable, such as
	/// text/plain, text/html or image/svg+xml. 
	/// - Parameter contentType: The new content type of the chat message    
	
	/// Gets the content type of a chat message. 
	/// - Returns: The content type of the chat message    
	public var contentType: String
	{
	
		get
		{ 
			
			let cPointer = linphone_chat_message_get_content_type(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_chat_message_set_content_type(cPtr, newValue)
		}
	}
		
	
	/// Returns the list of contents in the message. 
	/// - Returns: The list of ``Content``.      
	public var contents: [Content]
	{
	
						var swiftList = [Content]()
			let cList = linphone_chat_message_get_contents(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(Content.getSwiftObject(cObject: data))
				listTemp = UnsafePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
			return swiftList

	}
		
	
	/// Returns back pointer to ``Core`` object. 
	/// - Returns: the ``Core`` object associated with this message.    
	public var core: Core?
	{
	
						let cPointer = linphone_chat_message_get_core(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Core.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Gets the current ``ChatMessageDelegate`` being invoked, if any. 
	/// This is meant only to be called from a callback to be able to get the user_data
	/// associated with the ``ChatMessageDelegate`` that is calling the callback. 
	/// - Returns: The ``ChatMessageDelegate`` that has called the last callback.    
	public var currentDelegate: ChatMessageDelegate?
	{
	
			let cObject = linphone_chat_message_get_current_callbacks(cPtr)
			let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
			if (result != nil) {
				return Unmanaged<ChatMessageDelegateManager>.fromOpaque(result!).takeUnretainedValue().delegate
				}
			return nil
			
	}
		
	
	/// Returns the real time at which an ephemeral message expires and will be
	/// deleted. 
	/// - See also: ``isEphemeral()`` 
	/// - Returns: the time at which an ephemeral message expires. 0 means the message
	/// has not been read. 
	public var ephemeralExpireTime: time_t
	{
	
						return linphone_chat_message_get_ephemeral_expire_time(cPtr)

	}
		
	
	/// Returns lifetime of an ephemeral message. 
	/// The lifetime is the duration after which the ephemeral message will disappear
	/// once viewed. - See also: ``isEphemeral()`` 
	/// - Returns: the lifetime of an ephemeral message, by default 0 (disabled). 
	public var ephemeralLifetime: Int
	{
	
						return Int(linphone_chat_message_get_ephemeral_lifetime(cPtr))

	}
		
	
	/// Gets full details about delivery error of a chat message. 
	/// - Returns: a ``ErrorInfo`` describing the details.    
	public var errorInfo: ErrorInfo?
	{
	
						let cPointer = linphone_chat_message_get_error_info(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = ErrorInfo.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	/// Messages can carry external body as defined by rfc2017. 
	/// - Parameter externalBodyUrl: ex: access-type=URL; URL="http://www.foo.com/file"
	///    
	
	/// Messages can carry external body as defined by rfc2017. 
	/// - Returns: external body url or nil if not present.    
	public var externalBodyUrl: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_chat_message_get_external_body_url(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_chat_message_set_external_body_url(cPtr, newValue)
		}
	}
		
	
	/// Gets the file transfer information (used by callbacks to recover informations
	/// during a rcs file transfer) 
	/// - Returns: a pointer to the ``Content`` structure or nil if not present.    
	public var fileTransferInformation: Content?
	{
	
						let cPointer = linphone_chat_message_get_file_transfer_information(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Content.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Gets the forward info if available as a string. 
	/// - Returns: the original sender of the message if it has been forwarded, nil
	/// otherwise.    
	public var forwardInfo: String?
	{
	
			
			let cPointer = linphone_chat_message_get_forward_info(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Gets origin of the message. 
	/// - Returns: the ``Address`` of the sender.    
	public var fromAddress: Address?
	{
	
						let cPointer = linphone_chat_message_get_from_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Returns wether this message can be edited (maximum allowed time to edit it was
	/// reached or not). 
	/// - Returns: true if the message can be edited, false otherwise. 
	public var isEditable: Bool
	{
	
						return linphone_chat_message_is_editable(cPtr) != 0

	}
		
	
	/// Returns wether this message has been edited by it's sender after it was sent. 
	/// - Returns: true if the message has been edited after it was sent, false
	/// otherwise. 
	public var isEdited: Bool
	{
	
						return linphone_chat_message_is_edited(cPtr) != 0

	}
		
	
	/// Returns wether the chat message is an ephemeral message or not. 
	/// An ephemeral message will automatically disappear from the recipient's screen
	/// after the message has been viewed. 
	/// - Returns: true if it is an ephemeral message, false otherwise 
	public var isEphemeral: Bool
	{
	
						return linphone_chat_message_is_ephemeral(cPtr) != 0

	}
		
	
	/// Return whether or not a chat message is a file transfer. 
	/// - Returns: Whether or not the message is a file transfer 
	/// - Deprecated: 06/07/2020 check if ``getContents()`` contains a ``Content`` for
	/// which ``Content/isFileTransfer()`` returns true. 
	@available(*, deprecated)
	public var isFileTransfer: Bool
	{
	
						return linphone_chat_message_is_file_transfer(cPtr) != 0

	}
		
	
	/// Gets whether or not a file is currently being downloaded or uploaded. 
	/// - Returns: true if download or upload is in progress, false otherwise 
	public var isFileTransferInProgress: Bool
	{
	
						return linphone_chat_message_is_file_transfer_in_progress(cPtr) != 0

	}
		
	
	/// Returns wether the chat message is a forward message or not. 
	/// - Returns: true if it is a forward message, false otherwise 
	public var isForward: Bool
	{
	
						return linphone_chat_message_is_forward(cPtr) != 0

	}
		
	
	/// Returns wehther the message has been sent or received. 
	/// - Returns: true if message has been sent, false if it has been received. 
	public var isOutgoing: Bool
	{
	
						return linphone_chat_message_is_outgoing(cPtr) != 0

	}
		
	
	/// Returns wether the message has been read or not. 
	/// - Returns: true if message has been marked as read, false otherwise. 
	public var isRead: Bool
	{
	
						return linphone_chat_message_is_read(cPtr) != 0

	}
		
	
	/// Returns wether the chat message is a reply message or not. 
	/// - Returns: true if it is a reply message, false otherwise 
	public var isReply: Bool
	{
	
						return linphone_chat_message_is_reply(cPtr) != 0

	}
		
	
	/// Returns wether this message can be retracted (maximum allowed time to retract
	/// it was reached or not). 
	/// - Returns: true if the message can be retracted, false otherwise. 
	public var isRetractable: Bool
	{
	
						return linphone_chat_message_is_retractable(cPtr) != 0

	}
		
	
	/// Returns wether this message has been retracted by it's sender after it was
	/// sent. 
	/// - Returns: true if the message has been retracted after it was sent, false
	/// otherwise. 
	public var isRetracted: Bool
	{
	
						return linphone_chat_message_is_retracted(cPtr) != 0

	}
		
	
	/// Gets if the message was end-to-end encrypted when transferred. 
	/// - Returns: true if the message was end-to-end encrypted when transferred, false
	/// otherwise. 
	public var isSecured: Bool
	{
	
						return linphone_chat_message_is_secured(cPtr) != 0

	}
		
	
	/// Return whether or not a chat message is a text. 
	/// - Returns: Whether or not the message is a text 
	/// - Deprecated: 06/07/2020 check if ``getContents()`` contains a ``Content`` with
	/// a PlainText content type. 
	@available(*, deprecated)
	public var isText: Bool
	{
	
						return linphone_chat_message_is_text(cPtr) != 0

	}
		
	
	/// Returns the local address the message was sent or received with. 
	/// - Returns: the ``Address`` of the local address used to send/receive this
	/// message.    
	public var localAddress: Address?
	{
	
						let cPointer = linphone_chat_message_get_local_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Get the message identifier. 
	/// It is used to identify a message so that it can be notified as delivered and/or
	/// displayed. 
	/// - Returns: The message identifier.    
	public var messageId: String
	{
	
			
			let cPointer = linphone_chat_message_get_message_id(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Returns our own reaction for a given chat message, if any. 
	/// - Returns: Our own ``ChatMessageReaction`` for that message if any, nil
	/// otherwise.    
	public var ownReaction: ChatMessageReaction?
	{
	
						let cPointer = linphone_chat_message_get_own_reaction(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = ChatMessageReaction.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Returns the peer (remote) address of the message. 
	/// - Returns: the ``Address`` of the peer address used to send/receive this
	/// message.    
	public var peerAddress: Address?
	{
	
						let cPointer = linphone_chat_message_get_peer_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Gets the list of reactions received for this chat message. 
	/// Warning: list is ordered by content of reaction message, not by received
	/// timestamp! 
	/// - Returns: The sorted list of reaction if any.         
	public var reactions: [ChatMessageReaction]
	{
	
						var swiftList = [ChatMessageReaction]()
			let cList = linphone_chat_message_get_reactions(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(ChatMessageReaction.getSwiftObject(cObject: data))
				listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
				bctbx_list_free_with_data(cList, belle_sip_object_unref)
			return swiftList

	}
		
	
	/// Returns the ``ChatMessage`` this message is a reply to. 
	/// - Returns: the original message ``ChatMessage``.    
	public var replyMessage: ChatMessage?
	{
	
						let cPointer = linphone_chat_message_get_reply_message(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = ChatMessage.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Returns the ID of the message this is a reply to. 
	/// - Returns: the original message id.    
	public var replyMessageId: String?
	{
	
			
			let cPointer = linphone_chat_message_get_reply_message_id(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Returns the address of the sender of the message this is a reply to. 
	/// - Returns: the original message sender ``Address``.    
	public var replyMessageSenderAddress: Address?
	{
	
						let cPointer = linphone_chat_message_get_reply_message_sender_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Gets the state of the message. 
	/// - Returns: the current ``State`` of the message. 
	public var state: ChatMessage.State
	{
	
						return ChatMessage.State(rawValue: Int(linphone_chat_message_get_state(cPtr).rawValue))!

	}
		
	
	/// Gets the text content if available as a string. 
	/// - Returns: the ``Content`` buffer if available in System Locale, null
	/// otherwise.    
	/// - Deprecated: 01/07/2020. Use ``getUtf8Text()`` instead. 
	@available(*, deprecated)
	public var textContent: String?
	{
	
			
			let cPointer = linphone_chat_message_get_text_content(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Gets the time the message was sent. 
	/// - Returns: the timestamp of when the message was sent. 
	public var time: time_t
	{
	
						return linphone_chat_message_get_time(cPtr)

	}
		
	
	/// Gets destination of the message. 
	/// - Returns: the ``Address`` of the recipient.    
	public var toAddress: Address?
	{
	
						let cPointer = linphone_chat_message_get_to_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	/// Sets if a chat message is to be stored. 
	/// This content type must match a content that is text representable, such as
	/// text/plain, text/html or image/svg+xml. 
	/// - Parameter toBeStored: Whether or not the chat message is to be stored 
	
	/// Gets if a chat message is to be stored. 
	/// - Returns: Whether or not the message is to be stored 
	public var toBeStored: Bool
	{
	
		get
		{ 
						return linphone_chat_message_get_to_be_stored(cPtr) != 0
		}
		set
		{
			linphone_chat_message_set_to_be_stored(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Assigns a user pointer to the chat message. 
	/// - Parameter userData: The user pointer to associate with the chat message.    
	
	/// Retrieves the user pointer associated with the chat message. 
	/// - Returns: The user pointer associated with the chat message.    
	public var userData: UnsafeMutableRawPointer?
	{
	
		get
		{ 
						return linphone_chat_message_get_user_data(cPtr)
		}
		set
		{
			linphone_chat_message_set_user_data(cPtr, newValue)
		}
	}
		
	/// Sets a text to be sent, given as a string of UTF-8 characters. 
	/// - Parameter text: The text in UTF8 to set.    
	/// - Returns: 0 if succeed. 
	
	/// Gets text part of this message. 
	/// - Returns: The text as UTF8 characters or nil if no text.    
	public var utf8Text: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_chat_message_get_utf8_text(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_chat_message_set_utf8_text(cPtr, newValue)
		}
	}
		
	
	
	/// Adds a content to the ChatMessage. 
	/// - Parameter content: the ``Content`` object to add.    
	public func addContent(content:Content) 
	{
		linphone_chat_message_add_content(cPtr, content.cPtr)
	}
	
	
	
	/// Adds custom headers to the message. 
	/// - Parameter headerName: name of the header    
	/// - Parameter headerValue: header value    
	public func addCustomHeader(headerName:String, headerValue:String?) 
	{
		linphone_chat_message_add_custom_header(cPtr, headerName, headerValue)
	}
	
	
	
	/// Adds a file content to the ChatMessage. 
	/// - Parameter content: the ``Content`` object to add.    
	public func addFileContent(content:Content) 
	{
		linphone_chat_message_add_file_content(cPtr, content.cPtr)
	}
	
	
	
	/// Creates a ``Content`` of type PlainText with the given text as body. 
	/// - Parameter text: The text in System Locale to add to the message.    
	/// - Deprecated: 01/07/2020. Use ``addUtf8TextContent(text:)`` instead. 
	@available(*, deprecated)
	public func addTextContent(text:String) 
	{
		linphone_chat_message_add_text_content(cPtr, text)
	}
	
	
	
	/// Creates a ``Content`` of type text/plain with the provided string, and attach
	/// it to the message. 
	/// - Parameter text: The text to add to the message.    
	public func addUtf8TextContent(text:String) 
	{
		linphone_chat_message_add_utf8_text_content(cPtr, text)
	}
	
	
	
	/// Cancels an ongoing file transfer attached to this message (upload or download). 
	public func cancelFileTransfer() 
	{
		linphone_chat_message_cancel_file_transfer(cPtr)
	}
	
	
	
	/// Creates a emoji reaction for the given chat mesage. 
	/// To send it, use ``ChatMessageReaction/send()``. 
	/// - Parameter utf8Reaction: the emoji character(s) as UTF-8.    
	/// - Returns: a ``ChatMessageReaction`` object.    
	public func createReaction(utf8Reaction:String) throws -> ChatMessageReaction
	{
		let cPointer = linphone_chat_message_create_reaction(cPtr, utf8Reaction)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null ChatMessageReaction value")
		}
		let result = ChatMessageReaction.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Starts the download of the ``Content`` referenced in the ``ChatMessage`` from
	/// remote server. 
	/// - Parameter content: the ``Content`` object to download (must have the
	/// ``Content/isFileTransfer()`` method return true).    
	/// - Returns: false if there is an error, true otherwise. 
	public func downloadContent(content:Content) -> Bool
	{
		return linphone_chat_message_download_content(cPtr, content.cPtr) != 0
	}
	
	
	
	/// Starts the download of all the ``Content`` objects representing file transfers
	/// included in the message (``Content/isFileTransfer()`` method returns true). 
	/// - Returns: false if there is an error, true otherwise. 
	public func downloadContents() -> Bool
	{
		return linphone_chat_message_download_contents(cPtr) != 0
	}
	
	
	
	/// Retrieves a custom header value given its name. 
	/// - Parameter headerName: header name searched    
	/// - Returns: the custom header value or nil if not found.    
	public func getCustomHeader(headerName:String) -> String
	{
		let cstr = linphone_chat_message_get_custom_header(cPtr, headerName)
		let result = charArrayToString(charPointer: cstr)
		return result
	}
	
	
	
	/// Gets the list of participants for which the imdn state has reached the
	/// specified state and the time at which they did. 
	/// - Parameter state: The ``State`` the imdn have reached (only use
	/// LinphoneChatMessageStateDelivered, LinphoneChatMessageStateDeliveredToUser,
	/// LinphoneChatMessageStateDisplayed and LinphoneChatMessageStateNotDelivered) 
	/// - Returns: The list of participants.         
	public func getParticipantsByImdnState(state:ChatMessage.State) -> [ParticipantImdnState]
	{
		var swiftList = [ParticipantImdnState]()
		let cList = linphone_chat_message_get_participants_by_imdn_state(cPtr, LinphoneChatMessageState(rawValue: CUnsignedInt(state.rawValue)))
		var listTemp = cList
		while (listTemp != nil) {
			let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
			swiftList.append(ParticipantImdnState.getSwiftObject(cObject: data))
			listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
		}
			bctbx_list_free_with_data(cList, belle_sip_object_unref)
		return swiftList
	}
	
	
	
	/// Returns wether the chat message has a conference invitation content or not. 
	/// - Returns: true if it has one, false otherwise. 
	public func hasConferenceInvitationContent() -> Bool
	{
		return linphone_chat_message_has_conference_invitation_content(cPtr) != 0
	}
	
	
	
	/// Returns wether the chat message has a text content or not. 
	/// - Returns: true if it has one, false otherwise. 
	/// - Deprecated: 27/10/2020. Check if ``getContents()`` contains a ``Content`` for
	/// which it's content type is PlainText. 
	@available(*, deprecated)
	public func hasTextContent() -> Bool
	{
		return linphone_chat_message_has_text_content(cPtr) != 0
	}
	
	
	
	/// Marks the message as read. 
	/// Only triggers LinphoneChatRoomCbsChatRoomReadCb if it was the last unread
	/// message. 
	public func markAsRead() 
	{
		linphone_chat_message_mark_as_read(cPtr)
	}
	
	
	
	/// Fulfills a chat message character by character and send the character
	/// immediately as real-time text (RFC4103 / T.140) or as Baudot tones. 
	/// The method used to send the character depends on if real-time text is enabled
	/// or not. If it is, real-time text is of course used, otherwise Baudot will be
	/// used if it is enabled in the ``Core`` (see ``Core/enableBaudot(enabled:)``). If
	/// real-time text is used, the ``ChatRoom`` the message was created from must be a
	/// real-time text capable chat room: it must be obtained by placing or receiving a
	/// call with real-time text capabilities (see
	/// ``CallParams/enableRealtimeText(yesno:)`` ), and getting the ``ChatRoom``
	/// interface from the call with ``Call/getChatRoom()``. When the message is
	/// terminated (ie a new line needs to be started), use ``send()`` in order to
	/// trigger the sending of the new line character and have the full message
	/// (comprising all characters sent so far) stored in local database. 
	/// - Parameter character: The character to send (T.140 char for real-time text). 
	/// - Returns: 0 if succeed. 
	public func putChar(character:UInt32) throws 
	{
		let exception_result = linphone_chat_message_put_char(cPtr, character)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "putChar returned value \(exception_result)")
		}
	}
	
	
	
	/// Removes a content from the ChatMessage. 
	/// - Parameter content: the ``Content`` object to remove.    
	public func removeContent(content:Content) 
	{
		linphone_chat_message_remove_content(cPtr, content.cPtr)
	}
	
	
	
	/// Removes a custom header from the message. 
	/// - Parameter headerName: name of the header to remove    
	public func removeCustomHeader(headerName:String) 
	{
		linphone_chat_message_remove_custom_header(cPtr, headerName)
	}
	
	
	
	/// Sends a chat message. 
	public func send() 
	{
		linphone_chat_message_send(cPtr)
	}
}


/// A chat message reaction is an emoji sent by someone in the same chat room to
/// react to a specific ``ChatMessage``. 
/// To create a ``ChatMessageReaction``, use
/// ``ChatMessage/createReaction(utf8Reaction:)``. Once you are ready, send the
/// reaction using ``send()``.
/// Reactions are available using ``ChatMessage/getReactions()`` and will be
/// notified using dedicated callbacks either in ``CoreDelegate`` or
/// ``ChatMessageDelegate``. 
public class ChatMessageReaction : LinphoneObject
{

	static public func getSwiftObject(cObject:OpaquePointer) -> ChatMessageReaction {
		let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
		if (result != nil) {
			return Unmanaged<ChatMessageReaction>.fromOpaque(result!).takeUnretainedValue()
		}
		let sObject = ChatMessageReaction(cPointer: cObject)
		belle_sip_object_data_set(UnsafeMutablePointer(cObject), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(sObject).toOpaque()), nil)
		return sObject
	}

	public var getCobject: OpaquePointer? {
		return cPtr
	}
	
	
	/// Returns the emoji(s) used for the reaction. 
	/// - Returns: the emoji(s) used as UTF-8 characters.    
	public var body: String
	{
	
			
			let cPointer = linphone_chat_message_reaction_get_body(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Allows to get the Call ID associated with a ``ChatMessageReaction``. 
	/// - Returns: the Call ID associated with this reaction. 
	public var callId: String
	{
	
			
			let cPointer = linphone_chat_message_reaction_get_call_id(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Returns the ``Address`` of the participant that sent this reaction. 
	/// - Returns: the ``Address`` that sent this reaction.    
	public var fromAddress: Address?
	{
	
						let cPointer = linphone_chat_message_reaction_get_from_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	
	/// Sends a ``ChatMessageReaction``. 
	public func send() 
	{
		linphone_chat_message_reaction_send(cPtr)
	}
}


/// Object defining settings strictly associated with ``ChatRoom`` objects such as
/// ephemeral settings and backends. 
/// It is automatically created when you enable chat capabilities in the
/// ``ConferenceParams``.
/// If the ``ChatRoom`` backend is ``Basic``, then no other parameter is required,
/// but ``ChatMessage`` sent and received won't benefit from all features a
/// ``FlexisipChat`` can offer like conversation with multiple participants and a
/// subject, end-to-end encryption, ephemeral messages, etc... but this type is the
/// only one that can interoperate with other SIP clients or with non-flexisip SIP
/// proxies. 
public class ChatParams : LinphoneObject
{

	static public func getSwiftObject(cObject:OpaquePointer) -> ChatParams {
		let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
		if (result != nil) {
			return Unmanaged<ChatParams>.fromOpaque(result!).takeUnretainedValue()
		}
		let sObject = ChatParams(cPointer: cObject)
		belle_sip_object_data_set(UnsafeMutablePointer(cObject), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(sObject).toOpaque()), nil)
		return sObject
	}

	public var getCobject: OpaquePointer? {
		return cPtr
	}
	
	/// Set the backend implementation of these text capabilities of the chat
	/// parameters. 
	/// - Parameter backend: The ``ChatRoom.Backend`` enum value 
	
	/// Get the backend implementation of the text capabilities of the chat associated
	/// with the given parameters. 
	/// - Returns: the ``ChatRoom.Backend`` 
	public var backend: ChatRoom.Backend
	{
	
		get
		{ 
						return ChatRoom.Backend(rawValue: Int(linphone_chat_params_get_backend(cPtr).rawValue))
		}
		set
		{
			linphone_chat_params_set_backend(cPtr, LinphoneChatRoomBackend(rawValue: CUnsignedInt(newValue.rawValue)))
		}
	}
		
	/// Set the encryption backend implementation of these text capabilities of the
	/// chat parameters. 
	/// - Parameter backend: The ``ChatRoom.EncryptionBackend`` enum value 
	
	/// Get the encryption implementation of the text capabilities of the chat
	/// associated with the given parameters. 
	/// - Returns: the ``ChatRoom.EncryptionBackend`` 
	public var encryptionBackend: ChatRoom.EncryptionBackend
	{
	
		get
		{ 
						return ChatRoom.EncryptionBackend(rawValue: Int(linphone_chat_params_get_encryption_backend(cPtr).rawValue))
		}
		set
		{
			linphone_chat_params_set_encryption_backend(cPtr, LinphoneChatRoomEncryptionBackend(rawValue: CUnsignedInt(newValue.rawValue)))
		}
	}
		
	
	/// Get the encryption status of the text capabilities of the chat associated with
	/// the given parameters. 
	/// - Returns: true if encryption is enabled, false otherwise 
	public var encryptionEnabled: Bool
	{
	
						return linphone_chat_params_encryption_enabled(cPtr) != 0

	}
		
	/// Set lifetime (in seconds) for all new ephemral messages in the text
	/// capabilities of the chat. 
	/// After the message is read, it will be deleted after "time" seconds. - See also:
	/// linphone_chat_params_ephemeral_enabled() 
	/// - Parameter time: The ephemeral lifetime, default is disabled (0) 
	
	/// Get lifetime (in seconds) for all new ephemeral messages in the text
	/// capabilities of the chat. 
	/// After the message is read, it will be deleted after "time" seconds. - See also:
	/// linphone_chat_params_ephemeral_enabled() 
	/// - Returns: the ephemeral lifetime (in seconds) 
	public var ephemeralLifetime: Int
	{
	
		get
		{ 
						return Int(linphone_chat_params_get_ephemeral_lifetime(cPtr))
		}
		set
		{
			linphone_chat_params_set_ephemeral_lifetime(cPtr, (newValue))
		}
	}
		
	/// Enables or disables forcing of ephemeral messages for the text capabilities of
	/// the chat associated with the given parameters. 
	/// - Parameter mode: Ephemeral message mode ``ChatRoom.EphemeralMode``. 
	
	/// Get the ephemeral message mode of the text capabilities of the chat associated
	/// with the given parameters. 
	/// - Returns: the ephemeral message mode ``ChatRoom.EphemeralMode`` 
	public var ephemeralMode: ChatRoom.EphemeralMode
	{
	
		get
		{ 
						return ChatRoom.EphemeralMode(rawValue: Int(linphone_chat_params_get_ephemeral_mode(cPtr).rawValue))!
		}
		set
		{
			linphone_chat_params_set_ephemeral_mode(cPtr, LinphoneChatRoomEphemeralMode(rawValue: CUnsignedInt(newValue.rawValue)))
		}
	}
		
	/// Enables or disables real time text for the text capabilities of the chat
	/// associated with the given parameters. 
	/// - Parameter rtt: true to enable real time text, false to disable. 
	
	/// Get the real time text status of the text capabilities of the chat associated
	/// with the given parameters. 
	/// - Returns: true if real time text is enabled, false otherwise 
	public var rttEnabled: Bool
	{
	
		get
		{ 
						return linphone_chat_params_rtt_enabled(cPtr) != 0
		}
		set
		{
			linphone_chat_params_enable_rtt(cPtr, newValue==true ? 1:0)
		}
	}
		
	
	
	/// Clone a ``ChatParams``. 
	/// - Returns: An allocated ``ChatParams`` with the same parameters than params    
	public func clone() -> ChatParams?
	{
		let cPointer = linphone_chat_params_clone(cPtr)
		if (cPointer == nil) {
			return nil
		}
		let result = ChatParams.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
}


/// A chat room is the place where ``ChatMessage`` are exchanged. 
/// To create (or find) a ``ChatRoom``, you first need a ``ChatRoomParams`` object.
/// A chat room is uniquely identified by it's local and remote SIP addresses,
/// meaning you can only have one chat room between two accounts (unless the
/// backend is ``FlexisipChat``). Then you can call
/// ``Core/searchChatRoom(params:localAddr:remoteAddr:participants:)`` or
/// ``Core/createChatRoom(params:localAddr:participants:)``.
/// Be careful as a ``FlexisipChat`` backend ``ChatRoom`` will be created
/// asynchronously, so make sure you add a ``ChatRoomDelegate`` to the returned
/// object to be notified when it will be in state ``Created``.
/// All chat rooms are loaded from database when the ``Core`` starts, and you can
/// get them using ``Core/getChatRooms()``. This method doesn't return empty chat
/// rooms nor ones for which the local address doesn't match an existing
/// ``Account`` identity, unless you specify otherwise in the [misc] section of
/// your configuration file by setting hide_empty_chat_rooms=0 and/or
/// hide_chat_rooms_from_removed_proxies=0. 
public class ChatRoom : LinphoneObject
{
	var delegateManagers : [ChatRoomDelegateManager] = []

	static public func getSwiftObject(cObject:OpaquePointer) -> ChatRoom {
		let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
		if (result != nil) {
			return Unmanaged<ChatRoom>.fromOpaque(result!).takeUnretainedValue()
		}
		let sObject = ChatRoom(cPointer: cObject)
		belle_sip_object_data_set(UnsafeMutablePointer(cObject), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(sObject).toOpaque()), nil)
		return sObject
	}

	public var getCobject: OpaquePointer? {
		return cPtr
	}

	
	///``Capabilities`` is used to indicate the capabilities of a chat room. 
	public struct Capabilities:OptionSet
	{
		public let rawValue: Int

		public init(rawValue: Int) {
			self.rawValue = rawValue
		}

		
		/// No capabilities. 
		public static let None = Capabilities([])
		/// No server. 
		public static let Basic = Capabilities(rawValue: 1<<0)
		/// Supports RTT. 
		public static let RealTimeText = Capabilities(rawValue: 1<<1)
		/// Use server (supports group chat) 
		public static let Conference = Capabilities(rawValue: 1<<2)
		/// Special proxy chat room flag. 
		public static let Proxy = Capabilities(rawValue: 1<<3)
		/// Chat room migratable from Basic to Conference. 
		public static let Migratable = Capabilities(rawValue: 1<<4)
		/// A communication between two participants (can be Basic or Conference) 
		public static let OneToOne = Capabilities(rawValue: 1<<5)
		/// Chat room is encrypted. 
		public static let Encrypted = Capabilities(rawValue: 1<<6)
		/// Chat room can enable ephemeral messages. 
		public static let Ephemeral = Capabilities(rawValue: 1<<7)
	}

	
	///``Backend`` is used to indicate the backend implementation of a chat room. 
	public struct Backend:OptionSet
	{
		public let rawValue: Int

		public init(rawValue: Int) {
			self.rawValue = rawValue
		}

		
		/// Basic (client-to-client) chat room. 
		public static let Basic = Backend(rawValue: 1<<0)
		/// Server-based chat room. 
		public static let FlexisipChat = Backend(rawValue: 1<<1)
	}

	
	///TODO move to encryption engine object when available ``SecurityLevel`` is used
	///to indicate the encryption security level of a chat room. 
	public enum SecurityLevel:Int
	{
		
		/// Security failure. 
		case Unsafe = 0
		/// No encryption. 
		case ClearText = 1
		/// Encrypted. 
		case Encrypted = 2
		/// Encrypted and verified. 
		case Safe = 3
	}

	
	///``HistoryFilter`` is used to determine which filter to apply to history
	///functions. 
	public struct HistoryFilter:OptionSet
	{
		public let rawValue: Int

		public init(rawValue: Int) {
			self.rawValue = rawValue
		}

		
		/// No filter. 
		public static let None = HistoryFilter([])
		/// Retrieve calls. 
		public static let Call = HistoryFilter(rawValue: 1<<0)
		/// Retrieve chat messages. 
		public static let ChatMessage = HistoryFilter(rawValue: 1<<1)
		/// Retrieve chat messages and security events. 
		public static let ChatMessageSecurity = HistoryFilter(rawValue: 1<<2)
		/// Retrieve all chat room events. 
		public static let Info = HistoryFilter(rawValue: 1<<3)
		/// Retrieve all chat room events without device events. 
		public static let InfoNoDevice = HistoryFilter(rawValue: 1<<4)
	}

	
	///``EphemeralMode`` is used to the ephemeral message mode used by a chat room. 
	public enum EphemeralMode:Int
	{
		
		/// Each device manages its own ephemeral settings. 
		case DeviceManaged = 0
		/// Ephemeral settings are chatroom wide and only admins can change them. 
		case AdminManaged = 1
	}

	
	///``EncryptionBackend`` is used to indicate the encryption engine used by a chat
	///room. 
	public struct EncryptionBackend:OptionSet
	{
		public let rawValue: Int

		public init(rawValue: Int) {
			self.rawValue = rawValue
		}

		
		/// No encryption. 
		public static let None = EncryptionBackend([])
		/// Lime x3dh encryption. 
		public static let Lime = EncryptionBackend(rawValue: 1<<0)
	}

	
	///``State`` is used to indicate the current state of a chat room. 
	public enum State:Int
	{
		
		/// Initial state. 
		case None = 0
		/// Chat room is now instantiated on local. 
		case Instantiated = 1
		/// One creation request was sent to the server. 
		case CreationPending = 2
		/// Chat room was created on the server. 
		case Created = 3
		/// Chat room creation failed. 
		case CreationFailed = 4
		/// Wait for chat room termination. 
		case TerminationPending = 5
		/// Chat room exists on server but not in local. 
		case Terminated = 6
		/// The chat room termination failed. 
		case TerminationFailed = 7
		/// Chat room was deleted on the server. 
		case Deleted = 8
	}
	
	
	
	public func addDelegate(delegate: ChatRoomDelegate)
	{
		let manager = ChatRoomDelegateManager()
		manager.delegate = delegate
		delegateManagers.append(manager)
		linphone_chat_room_add_callbacks(cPtr, manager.cPtr)
		belle_sip_object_unref(UnsafeMutableRawPointer(manager.cPtr))
		belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(self).toOpaque()), nil)
	}	
	
	
	public func removeDelegate(delegate: ChatRoomDelegate)
	{
		if let index = delegateManagers.firstIndex(where: { $0.delegate === delegate as AnyObject}) {
			linphone_chat_room_remove_callbacks(cPtr, delegateManagers[index].cPtr)
			delegateManagers.remove(at: index)
			belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(self).toOpaque()), nil)
		}
	}	
	
	
	/// Converts a ``State`` enum to a string. 
	/// - Parameter state: a ``State`` to convert to string 
	/// - Returns: the string representation of the ``State``    
	static public func stateToString(state:ChatRoom.State) -> String
	{
		let cstr = linphone_chat_room_state_to_string(LinphoneChatRoomState(rawValue: CUnsignedInt(state.rawValue)))
		let result = charArrayToString(charPointer: cstr)
		return result
	}
	
	
	/// Returns the local account to which this chat room is related. 
	/// - Returns: the related ``Account`` object if any, nil otherwise.    
	public var account: Account?
	{
	
						let cPointer = linphone_chat_room_get_account(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Account.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Gets the current call associated to this chatroom if any To commit a message,
	/// use ``ChatMessage/send()`` 
	/// - Returns: ``Call`` or nil.    
	public var call: Call?
	{
	
						let cPointer = linphone_chat_room_get_call(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Call.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Gets the capabilities of a chat room. 
	/// - Returns: The capabilities of the chat room (as a
	/// LinphoneChatRoomCapabilitiesMask) 
	public var capabilities: UInt
	{
	
						return UInt(linphone_chat_room_get_capabilities(cPtr))

	}
		
	
	/// When realtime text is enabled in a ``Call`` (see
	/// ``CallParams/realtimeTextEnabled()``), LinphoneCoreIsComposingReceivedCb is
	/// called everytime a Real-Time Text character is received from peer. 
	/// ``getChar()`` pops a character previously received from the receive queue, and
	/// returns it. When a new line character is received, a ``ChatMessage`` is
	/// automatically created with received characters and notified to the application
	/// through the ``CoreDelegate`` or ``ChatRoomDelegate`` interfaces. 
	/// - Returns: RFC 4103/T.140 char 
	public var char: UInt32
	{
	
						return linphone_chat_room_get_char(cPtr)

	}
		
	
	/// Gets the list of participants that are currently composing. 
	/// - Returns: List of addresses that are in the is_composing state.   
	public var composingAddresses: [Address]
	{
	
						var swiftList = [Address]()
			let cList = linphone_chat_room_get_composing_addresses(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(Address.getSwiftObject(cObject: data))
				listTemp = UnsafePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
			return swiftList

	}
		
	
	/// Gets the list of participants that are currently composing. 
	/// - Returns: List of ``ComposingParticipant`` that are in the is_composing state.
	///   
	public var composingParticipants: [ComposingParticipant]
	{
	
						var swiftList = [ComposingParticipant]()
			let cList = linphone_chat_room_get_composing_participants(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(ComposingParticipant.getSwiftObject(cObject: data))
				listTemp = UnsafePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
			return swiftList

	}
		
	/// Sets the conference address of a group chat room. 
	/// This function needs to be called from the
	/// LinphoneChatRoomCbsConferenceAddressGenerationCb callback and only there. This
	/// function is meaningful only for server implementation of chatroom, and shall
	/// not by used by client applications. 
	/// - Parameter conferenceAddress: The conference ``Address`` to be used by the
	/// group chat room    
	
	/// Gets the conference address of the chat room. 
	/// - Returns: The conference address of the chat room or nil if this type of chat
	/// room is not conference based.    
	public var conferenceAddress: Address?
	{
	
		get
		{ 
						let cPointer = linphone_chat_room_get_conference_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_chat_room_set_conference_address(cPtr, newValue?.cPtr)
		}
	}
		
	
	/// Gets the conference information associated to the chatroom. 
	/// - Returns: the ``ConferenceInfo``.    
	public var conferenceInfo: ConferenceInfo?
	{
	
						let cPointer = linphone_chat_room_get_conference_info(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = ConferenceInfo.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Returns back pointer to ``Core`` object. 
	/// - Returns: the ``Core`` object this chat room is attached to.    
	public var core: Core?
	{
	
						let cPointer = linphone_chat_room_get_core(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Core.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Returns the creation time for the chat room. 
	/// - Returns: the time at which the chat room was created 
	public var creationTime: time_t
	{
	
						return linphone_chat_room_get_creation_time(cPtr)

	}
		
	
	/// Gets the current ``ChatRoomDelegate``. 
	/// This is meant only to be called from a callback to be able to get the user_data
	/// associated with the ``ChatRoomDelegate`` that is calling the callback. 
	/// - Returns: The ``ChatRoomDelegate`` that has called the last callback    
	public var currentDelegate: ChatRoomDelegate?
	{
	
			let cObject = linphone_chat_room_get_current_callbacks(cPtr)
			let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
			if (result != nil) {
				return Unmanaged<ChatRoomDelegateManager>.fromOpaque(result!).takeUnretainedValue().delegate
				}
			return nil
			
	}
		
	
	/// Returns current parameters associated with the chat room. 
	/// This is typically the parameters passed during the ``ChatRoom`` creation
	/// process to linphone_core_chat_room_create_chat_room() or some default
	/// parameters if no ``ChatRoomParams`` was explicitely passed during ``ChatRoom``
	/// creation. 
	/// - Returns: the current ``ChatRoomParams`` parameters.    
	/// - Deprecated: 17/07/2025. Use ``getParams()`` instead. 
	@available(*, deprecated)
	public var currentParams: ChatRoomParams?
	{
	
						let cPointer = linphone_chat_room_get_current_params(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = ChatRoomParams.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Gets all contents for which content-type starts with either text/ or
	/// application/. 
	/// - Returns: A list of contents considered as "document".      
	public var documentContents: [Content]
	{
	
						var swiftList = [Content]()
			let cList = linphone_chat_room_get_document_contents(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(Content.getSwiftObject(cObject: data))
				listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
				bctbx_list_free_with_data(cList, belle_sip_object_unref)
			return swiftList

	}
		
	
	/// Gets the number of document contents (see ``getDocumentContents()``). 
	/// - Returns: the number of document contents for that ``ChatRoom``. 
	public var documentContentsSize: Int
	{
	
						return Int(linphone_chat_room_get_document_contents_size(cPtr))

	}
		
	/// Enable or disable the ephemeral message feature in the chat room. 
	/// Works only for flexisip-based chat room. An ephemeral message will
	/// automatically disappear from the sender and recipient's chatrooms after a
	/// specified timeout configurable with ``setEphemeralLifetime(time:)``. The timer
	/// starts when the message has been displayed at the recipent, which means:
	/// -at recipient side when ``markAsRead()`` is called.
	/// -at sender side, when the message enters the state
	/// LinphoneChatMessageStateDisplayed (when receiving the displayed IMDN).
	/// - Parameter enable: true if the ephemeral message feature is enabled, false
	/// otherwise. 
	
	/// Returns whether or not the ephemeral message feature is enabled in the chat
	/// room. 
	/// - Returns: true if ephemeral is enabled, false otherwise. 
	public var ephemeralEnabled: Bool
	{
	
		get
		{ 
						return linphone_chat_room_ephemeral_enabled(cPtr) != 0
		}
		set
		{
			linphone_chat_room_enable_ephemeral(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets lifetime (in seconds) for all new ephemeral messages in the chat room. 
	/// After the message is read, it will be deleted after "time" seconds. - See also:
	/// ``ephemeralEnabled()`` 
	/// - Parameter time: The ephemeral lifetime, default is 0 (disabled) 
	/// - Warning: A value of "time" equal to 0 disables ephemeral messages 
	
	/// Gets lifetime (in seconds) for all new ephemeral messages in the chat room. 
	/// After the message is read, it will be deleted after "time" seconds. - See also:
	/// ``ephemeralEnabled()`` 
	/// - Returns: the ephemeral lifetime (in seconds) 
	public var ephemeralLifetime: Int
	{
	
		get
		{ 
						return Int(linphone_chat_room_get_ephemeral_lifetime(cPtr))
		}
		set
		{
			linphone_chat_room_set_ephemeral_lifetime(cPtr, (newValue))
		}
	}
		
	/// Sets the ephemeral mode of the chat room. 
	/// - See also: ``ephemeralEnabled()`` 
	/// - Parameter mode: The ephemeral mode ``EphemeralMode`` 
	/// - Warning: This function only changes the mode of ephemeral messages
	/// ``EphemeralMode``. It is required to manually enable ephemeral messages after
	/// setting the mode by calling ``enableEphemeral(enable:)`` 
	
	/// Gets the ephemeral mode of the chat room. 
	/// - See also: ``ephemeralEnabled()`` 
	/// - Returns: the ephemeral mode ``EphemeralMode`` 
	public var ephemeralMode: ChatRoom.EphemeralMode
	{
	
		get
		{ 
						return ChatRoom.EphemeralMode(rawValue: Int(linphone_chat_room_get_ephemeral_mode(cPtr).rawValue))!
		}
		set
		{
			linphone_chat_room_set_ephemeral_mode(cPtr, LinphoneChatRoomEphemeralMode(rawValue: CUnsignedInt(newValue.rawValue)))
		}
	}
		
	/// Sets not-read lifetime (in seconds) for all new ephemeral messages in the chat
	/// room. 
	/// If the message is not read, it will be deleted after "time" seconds. - See
	/// also: ``ephemeralEnabled()`` 
	/// - Parameter time: The ephemeral not-read lifetime, default is 0 (disabled) 
	/// - Warning: A value of "time" equal to 0 disables the ephemeral not-read
	/// countdowns 
	
	/// Gets not-read lifetime (in seconds) for all new ephemeral messages in the chat
	/// room. 
	/// If the message is not read, it will be deleted after "time" seconds. - See
	/// also: ``ephemeralEnabled()`` 
	/// - Returns: the ephemeral not-read lifetime (in seconds) 
	public var ephemeralNotReadLifetime: Int
	{
	
		get
		{ 
						return Int(linphone_chat_room_get_ephemeral_not_read_lifetime(cPtr))
		}
		set
		{
			linphone_chat_room_set_ephemeral_not_read_lifetime(cPtr, (newValue))
		}
	}
		
	
	/// Gets the number of events in a chat room. 
	/// - Returns: the number of events. 
	public var historyEventsSize: Int
	{
	
						return Int(linphone_chat_room_get_history_events_size(cPtr))

	}
		
	
	/// Gets the number of messages in a chat room. 
	/// - Returns: the number of messages. 
	/// - Deprecated: 30/07/2024. Use ``getHistorySize(filters:)`` instead. 
	@available(*, deprecated)
	public var historySize: Int
	{
	
						return Int(linphone_chat_room_get_history_size(cPtr))

	}
		
	
	/// Returns the chat room identifier. 
	/// - Warning: This method returns a nil pointer if the ChatRoom is in the
	/// Instantiated state 
	/// - Returns: the conference identifier.    
	public var identifier: String?
	{
	
			
			let cPointer = linphone_chat_room_get_identifier(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Returns whether or not a ``ChatRoom`` has at least one ``ChatMessage`` or not. 
	/// - Returns: true if there are no ``ChatMessage``, false otherwise. 
	public var isEmpty: Bool
	{
	
						return linphone_chat_room_is_empty(cPtr) != 0

	}
		
	
	/// Returns whether or not a message can be sent using this chat room. 
	/// A chat room may be read only until it's created, or when it's a group you have
	/// left. 
	/// - Returns: true if a chat message can't be sent in it, false otherwise. 
	public var isReadOnly: Bool
	{
	
						return linphone_chat_room_is_read_only(cPtr) != 0

	}
		
	
	/// Tells whether the remote is currently composing a message. 
	/// - Returns: true if the remote is currently composing a message, false
	/// otherwise. 
	public var isRemoteComposing: Bool
	{
	
						return linphone_chat_room_is_remote_composing(cPtr) != 0

	}
		
	
	/// Gets the last chat message sent or received in this chat room. 
	/// - Returns: the latest ``ChatMessage`` or nil if no message.    
	public var lastMessageInHistory: ChatMessage?
	{
	
						let cPointer = linphone_chat_room_get_last_message_in_history(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = ChatMessage.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Returns the last updated time for the chat room. 
	/// - Returns: the last updated time 
	public var lastUpdateTime: time_t
	{
	
						return linphone_chat_room_get_last_update_time(cPtr)

	}
		
	
	/// Get the local address associated to this chat room. 
	/// - Warning: This method returns a guessed address based on the me participant if
	/// the ChatRoom is in the Instantiated state 
	/// - Returns: The local address.    
	public var localAddress: Address?
	{
	
						let cPointer = linphone_chat_room_get_local_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Gets the participant representing myself in the chat room. 
	/// - Returns: The participant representing myself in the conference or nil if me
	/// is not set.    
	public var me: Participant?
	{
	
						let cPointer = linphone_chat_room_get_me(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Participant.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Gets all contents for which content-type starts with either video/, audio/ or
	/// image/. 
	/// - Returns: A list of contents considered as "media".      
	public var mediaContents: [Content]
	{
	
						var swiftList = [Content]()
			let cList = linphone_chat_room_get_media_contents(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(Content.getSwiftObject(cObject: data))
				listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
				bctbx_list_free_with_data(cList, belle_sip_object_unref)
			return swiftList

	}
		
	
	/// Gets the number of media contents (see ``getMediaContents()``). 
	/// - Returns: the number of media contents for that ``ChatRoom``. 
	public var mediaContentsSize: Int
	{
	
						return Int(linphone_chat_room_get_media_contents_size(cPtr))

	}
		
	/// Sets if a chat room should be considered as muted or not. 
	/// A muted chat room isn't used to compute unread messages total count. 
	/// - Parameter muted: true to flag it as muted, false to un-mute it. 
	
	/// Gets if a chat room has been flagged as muted (not by default). 
	/// A muted chat room isn't used to compute unread messages total count. 
	/// - Returns: true if the chat room is muted, false otherwise. 
	public var muted: Bool
	{
	
		get
		{ 
						return linphone_chat_room_get_muted(cPtr) != 0
		}
		set
		{
			linphone_chat_room_set_muted(cPtr, newValue==true ? 1:0)
		}
	}
		
	
	/// Gets the number of participants in the chat room (that is without ourselves). 
	/// - Returns: The number of participants in the chat room 
	public var nbParticipants: Int
	{
	
						return Int(linphone_chat_room_get_nb_participants(cPtr))

	}
		
	
	/// Returns current parameters associated with the chat room. 
	/// This is typically the parameters passed during the ``ChatRoom`` creation
	/// process to linphone_core_chat_room_create_chat_room() or some default
	/// parameters if no ``ChatRoomParams`` was explicitely passed during ``ChatRoom``
	/// creation. 
	/// - Returns: the current ``ChatRoomParams`` parameters.    
	public var params: ConferenceParams?
	{
	
						let cPointer = linphone_chat_room_get_params(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = ConferenceParams.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Gets the list of participants of a chat room. 
	/// - Returns: A   of the participants       
	public var participants: [Participant]
	{
	
						var swiftList = [Participant]()
			let cList = linphone_chat_room_get_participants(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(Participant.getSwiftObject(cObject: data))
				listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
				bctbx_list_free_with_data(cList, belle_sip_object_unref)
			return swiftList

	}
		
	
	/// Get the peer address associated to this chat room. 
	/// - Warning: This method returns an invalid address if the ChatRoom is in the
	/// Instantiated state 
	/// - Returns: The peer address.    
	public var peerAddress: Address?
	{
	
						let cPointer = linphone_chat_room_get_peer_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Returns the content-type (if set) of what the remote is currently composing. 
	/// - Returns: the content-type of what the remote is currently composing if set
	/// and if it's currently composing, nil otherwise.    
	public var remoteComposingContentType: String?
	{
	
			
			let cPointer = linphone_chat_room_get_remote_composing_content_type(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Gets the security level of a chat room. 
	/// - Returns: The current ``SecurityLevel`` of the chat room 
	public var securityLevel: ChatRoom.SecurityLevel
	{
	
						return ChatRoom.SecurityLevel(rawValue: Int(linphone_chat_room_get_security_level(cPtr).rawValue))!

	}
		
	
	/// Gets the state of the chat room. 
	/// - Returns: The current ``State`` of the chat room 
	public var state: ChatRoom.State
	{
	
						return ChatRoom.State(rawValue: Int(linphone_chat_room_get_state(cPtr).rawValue))!

	}
		
	/// Sets the subject of a chat room. 
	/// - Parameter subject: The new subject to set for the chat room    
	
	/// Gets the subject of a chat room. 
	/// - Returns: The subject of the chat room.    
	public var subject: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_chat_room_get_subject(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_chat_room_set_subject(cPtr, newValue)
		}
	}
		
	/// Sets the subject of a chat room (utf-8 string). 
	/// - Parameter subject: The new subject to set for the chat room    
	
	/// Gets the subject of a chat room (as an UTF8 string) 
	/// - Returns: The subject of the chat room.    
	public var subjectUtf8: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_chat_room_get_subject_utf8(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_chat_room_set_subject_utf8(cPtr, newValue)
		}
	}
		
	
	/// Gets all unread messages for this chat room, sorted from oldest to most recent. 
	/// - Returns: A list of unread chat messages.      
	public var unreadHistory: [ChatMessage]
	{
	
						var swiftList = [ChatMessage]()
			let cList = linphone_chat_room_get_unread_history(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(ChatMessage.getSwiftObject(cObject: data))
				listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
				bctbx_list_free_with_data(cList, belle_sip_object_unref)
			return swiftList

	}
		
	
	/// Gets the number of unread messages in the chatroom. 
	/// - Returns: the number of unread messages. 
	public var unreadMessagesCount: Int
	{
	
						return Int(linphone_chat_room_get_unread_messages_count(cPtr))

	}
		
	/// Assign a user pointer to the chat room. 
	/// - Parameter userData: The user pointer to associate with the chat room.    
	
	/// Retrieve the user pointer associated with the chat room. 
	/// - Returns: The user pointer associated with the chat room.    
	public var userData: UnsafeMutableRawPointer?
	{
	
		get
		{ 
						return linphone_chat_room_get_user_data(cPtr)
		}
		set
		{
			linphone_chat_room_set_user_data(cPtr, newValue)
		}
	}
		
	
	
	/// Adds a participant to a chat room. 
	/// This may fail if this type of chat room does not handle participants. Use
	/// ``canHandleParticipants()`` to know if this chat room handles participants. 
	/// - Parameter addr: The address of the participant to add to the chat room    
	public func addParticipant(addr:Address) 
	{
		linphone_chat_room_add_participant(cPtr, addr.cPtr)
	}
	
	
	
	/// Adds several participants to a chat room at once. 
	/// This may fail if this type of chat room does not handle participants. Use
	/// ``canHandleParticipants()`` to know if this chat room handles participants. 
	/// - Parameter addresses: The participants to add.      
	/// - Returns: true if everything is OK, false otherwise 
	public func addParticipants(addresses:[Address]) -> Bool
	{
		return linphone_chat_room_add_participants(cPtr, ObjectArrayToBctbxList(list: addresses)) != 0
	}
	
	
	
	/// Allow cpim on a basic chat room   . 
	public func allowCpim() 
	{
		linphone_chat_room_allow_cpim(cPtr)
	}
	
	
	
	/// Allow multipart on a basic chat room   . 
	public func allowMultipart() 
	{
		linphone_chat_room_allow_multipart(cPtr)
	}
	
	
	
	/// Tells whether a chat room is able to handle participants. 
	/// - Returns: true if the chat room can handle participants, false otherwise 
	public func canHandleParticipants() -> Bool
	{
		return linphone_chat_room_can_handle_participants(cPtr) != 0
	}
	
	
	
	/// Terminates a chat room and instruct the server to remove all participants. 
	/// - Returns: 0 if the termination is successful, -1 otherwise. 
	public func close() -> Int
	{
		return Int(linphone_chat_room_close(cPtr))
	}
	
	
	
	/// Notifies the destination of the chat message being composed that the user is
	/// typing a new message. 
	/// - Deprecated: 24/06/2025 use ``composeTextMessage()`` instead. 
	@available(*, deprecated)
	public func compose() 
	{
		linphone_chat_room_compose(cPtr)
	}
	
	
	
	/// Notifies the destination of the chat message being composed that the user is
	/// typing a message. 
	public func composeTextMessage() 
	{
		linphone_chat_room_compose_text_message(cPtr)
	}
	
	
	
	/// Notifies the destination of the chat message being composed that the user is
	/// recording a new voice message. 
	public func composeVoiceMessage() 
	{
		linphone_chat_room_compose_voice_message(cPtr)
	}
	
	
	
	/// Creates an empty message attached to the given chat room. 
	/// - Returns: a new ``ChatMessage``    
	public func createEmptyMessage() throws -> ChatMessage
	{
		let cPointer = linphone_chat_room_create_empty_message(cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null ChatMessage value")
		}
		let result = ChatMessage.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a message attached to the given chat room with a particular content. 
	/// Use ``ChatMessage/send()`` to initiate the transfer 
	/// - Parameter initialContent: ``Content`` initial content.    
	/// - Returns: a new ``ChatMessage``    
	public func createFileTransferMessage(initialContent:Content) throws -> ChatMessage
	{
		let cPointer = linphone_chat_room_create_file_transfer_message(cPtr, initialContent.cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null ChatMessage value")
		}
		let result = ChatMessage.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a forward message attached to the given chat room with a particular
	/// message. 
	/// - Parameter message: ``ChatMessage`` message to be forwarded.    
	/// - Returns: a new ``ChatMessage``    
	public func createForwardMessage(message:ChatMessage) throws -> ChatMessage
	{
		let cPointer = linphone_chat_room_create_forward_message(cPtr, message.cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null ChatMessage value")
		}
		let result = ChatMessage.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a message attached to the given chat room with a plain text content
	/// filled with the given message. 
	/// - Parameter message: text message, nil if absent.    
	/// - Returns: a new ``ChatMessage``    
	/// - Deprecated: 01/07/2020. Use ``createMessageFromUtf8(message:)`` instead. 
	@available(*, deprecated)
	public func createMessage(message:String?) throws -> ChatMessage
	{
		let cPointer = linphone_chat_room_create_message(cPtr, message)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null ChatMessage value")
		}
		let result = ChatMessage.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a message attached to the given chat room with a plain text content
	/// filled with the given message. 
	/// Introduced in 01/07/2020 
	/// - Parameter message: text message in UTF8, nil if absent.    
	/// - Returns: a new ``ChatMessage``    
	public func createMessageFromUtf8(message:String?) throws -> ChatMessage
	{
		let cPointer = linphone_chat_room_create_message_from_utf8(cPtr, message)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null ChatMessage value")
		}
		let result = ChatMessage.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a replaces message that will edit the original message. 
	/// - Parameter message: ``ChatMessage`` message to edit.    
	/// - Returns: a new ``ChatMessage``    
	public func createReplacesMessage(message:ChatMessage) throws -> ChatMessage
	{
		let cPointer = linphone_chat_room_create_replaces_message(cPtr, message.cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null ChatMessage value")
		}
		let result = ChatMessage.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a reply message attached to the given chat room with a particular
	/// message. 
	/// - Parameter message: ``ChatMessage`` message to reply to.    
	/// - Returns: a new ``ChatMessage``    
	public func createReplyMessage(message:ChatMessage) throws -> ChatMessage
	{
		let cPointer = linphone_chat_room_create_reply_message(cPtr, message.cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null ChatMessage value")
		}
		let result = ChatMessage.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a chat message with a voice recording attached to the given chat room. 
	/// - Warning: If the recorder isn't in Closed state, it will return an empty
	/// message! 
	/// - Parameter recorder: the ``Recorder`` object used to record the voice message.
	///    
	/// - Returns: a new ``ChatMessage``    
	public func createVoiceRecordingMessage(recorder:Recorder) throws -> ChatMessage
	{
		let cPointer = linphone_chat_room_create_voice_recording_message(cPtr, recorder.cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null ChatMessage value")
		}
		let result = ChatMessage.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Delete all messages from the history. 
	public func deleteHistory() 
	{
		linphone_chat_room_delete_history(cPtr)
	}
	
	
	
	/// Delete a message from the chat room history. 
	/// - Parameter message: The ``ChatMessage`` object to remove.    
	public func deleteMessage(message:ChatMessage) 
	{
		linphone_chat_room_delete_message(cPtr, message.cPtr)
	}
	
	
	
	/// Check if all participants support ephemeral messages. 
	/// It doesn't prevent to send ephemeral messages in the room but those who don't
	/// support it won't delete messages after lifetime has expired. The check is done
	/// by verifying the participant's advertised capabilities (+org.linphone.specs
	/// parameter). - See also: ``ephemeralEnabled()`` 
	/// - Returns: true if all participants in the chat room support ephemeral
	/// messages, false otherwise 
	public func ephemeralSupportedByAllParticipants() -> Bool
	{
		return linphone_chat_room_ephemeral_supported_by_all_participants(cPtr) != 0
	}
	
	
	
	/// Gets the event log sent or received in this chat room that matches a chat
	/// message with the given message_id. 
	/// - Parameter messageId: The id of the message to find    
	/// - Returns: the ``EventLog`` if found or nil.    
	public func findEventLog(messageId:String) -> EventLog?
	{
		let cPointer = linphone_chat_room_find_event_log(cPtr, messageId)
		if (cPointer == nil) {
			return nil
		}
		let result = EventLog.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Gets the chat message sent or received in this chat room that matches the
	/// message_id. 
	/// - Parameter messageId: The id of the message to find    
	/// - Returns: the ``ChatMessage`` if found or nil.    
	public func findMessage(messageId:String) -> ChatMessage?
	{
		let cPointer = linphone_chat_room_find_message(cPtr, messageId)
		if (cPointer == nil) {
			return nil
		}
		let result = ChatMessage.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Finds a participant of a chat room from its address. 
	/// - Parameter address: The ``Address`` to search in the list of participants of
	/// the chat room    
	/// - Returns: The participant if found, nil otherwise.    
	public func findParticipant(address:Address) -> Participant?
	{
		let cPointer = linphone_chat_room_find_participant(cPtr, address.cPtr)
		if (cPointer == nil) {
			return nil
		}
		let result = Participant.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Gets the partial list of contents for which content-type starts with either
	/// text/ or application/. 
	/// - Parameter begin: The first content of the range to be retrieved. Most recent
	/// content has index 0. 
	/// - Parameter end: The last content of the range to be retrieved. Oldest content
	/// has index of size (use ``getDocumentContentsSize()`` to retrieve size) 
	/// - Returns: A list of contents considered as "document".      
	public func getDocumentContentsRange(begin:Int, end:Int) -> [Content]
	{
		var swiftList = [Content]()
		let cList = linphone_chat_room_get_document_contents_range(cPtr, CInt(begin), CInt(end))
		var listTemp = cList
		while (listTemp != nil) {
			let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
			swiftList.append(Content.getSwiftObject(cObject: data))
			listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
		}
			bctbx_list_free_with_data(cList, belle_sip_object_unref)
		return swiftList
	}
	
	
	
	/// Gets nb_message most recent events from chat_room chat room, sorted from oldest
	/// to most recent. 
	/// - Parameter nbMessage: Number of events to retrieve. 0 means everything. 
	/// - Parameter filters: The LinphoneChatRoomHistoryFilterMask mask to filter the
	/// results with ``HistoryFilter`` 
	/// - Returns: A list of      
	public func getHistory(nbMessage:Int, filters:UInt) -> [EventLog]
	{
		var swiftList = [EventLog]()
		let cList = linphone_chat_room_get_history_2(cPtr, CInt(nbMessage), CUnsignedInt(filters))
		var listTemp = cList
		while (listTemp != nil) {
			let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
			swiftList.append(EventLog.getSwiftObject(cObject: data))
			listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
		}
			bctbx_list_free_with_data(cList, belle_sip_object_unref)
		return swiftList
	}
	
	
	
	/// Gets nb_message most recent messages from chat_room chat room, sorted from
	/// oldest to most recent. 
	/// - Parameter nbMessage: Number of message to retrieve. 0 means everything. 
	/// - Returns: A list of      
	/// - Deprecated: 30/07/2024. Use ``getHistory(nbMessage:filters:)`` instead. 
	@available(*, deprecated)
	public func getHistory(nbMessage:Int) -> [ChatMessage]
	{
		var swiftList = [ChatMessage]()
		let cList = linphone_chat_room_get_history(cPtr, CInt(nbMessage))
		var listTemp = cList
		while (listTemp != nil) {
			let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
			swiftList.append(ChatMessage.getSwiftObject(cObject: data))
			listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
		}
			bctbx_list_free_with_data(cList, belle_sip_object_unref)
		return swiftList
	}
	
	
	
	/// Gets nb_events most recent events from chat_room chat room, sorted from oldest
	/// to most recent. 
	/// - Parameter nbEvents: Number of events to retrieve. 0 means everything. 
	/// - Returns: The list of the most recent events.      
	public func getHistoryEvents(nbEvents:Int) -> [EventLog]
	{
		var swiftList = [EventLog]()
		let cList = linphone_chat_room_get_history_events(cPtr, CInt(nbEvents))
		var listTemp = cList
		while (listTemp != nil) {
			let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
			swiftList.append(EventLog.getSwiftObject(cObject: data))
			listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
		}
			bctbx_list_free_with_data(cList, belle_sip_object_unref)
		return swiftList
	}
	
	
	
	/// Gets nb_events most recent chat message events from chat_room chat room, sorted
	/// from oldest to most recent. 
	/// - Parameter nbEvents: Number of events to retrieve. 0 means everything. 
	/// - Returns: A list      
	public func getHistoryMessageEvents(nbEvents:Int) -> [EventLog]
	{
		var swiftList = [EventLog]()
		let cList = linphone_chat_room_get_history_message_events(cPtr, CInt(nbEvents))
		var listTemp = cList
		while (listTemp != nil) {
			let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
			swiftList.append(EventLog.getSwiftObject(cObject: data))
			listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
		}
			bctbx_list_free_with_data(cList, belle_sip_object_unref)
		return swiftList
	}
	
	
	
	/// Gets the partial list of events in the given range, sorted from oldest to most
	/// recent. 
	/// - Parameter begin: The first event of the range to be retrieved. History most
	/// recent message has index 0. 
	/// - Parameter end: The last event of the range to be retrieved. History oldest
	/// message has index of history size (use ``getHistorySize(filters:)`` to retrieve
	/// history size) 
	/// - Parameter filters: The LinphoneChatRoomHistoryFilterMask mask to filter the
	/// results with ``HistoryFilter`` 
	/// - Returns: A list of      
	public func getHistoryRange(begin:Int, end:Int, filters:UInt) -> [EventLog]
	{
		var swiftList = [EventLog]()
		let cList = linphone_chat_room_get_history_range_2(cPtr, CInt(begin), CInt(end), CUnsignedInt(filters))
		var listTemp = cList
		while (listTemp != nil) {
			let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
			swiftList.append(EventLog.getSwiftObject(cObject: data))
			listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
		}
			bctbx_list_free_with_data(cList, belle_sip_object_unref)
		return swiftList
	}
	
	
	
	/// Gets the partial list of messages in the given range, sorted from oldest to
	/// most recent. 
	/// - Parameter begin: The first message of the range to be retrieved. History most
	/// recent message has index 0. 
	/// - Parameter end: The last message of the range to be retrieved. History oldest
	/// message has index of history size (use ``getHistorySize()`` to retrieve history
	/// size) 
	/// - Returns: A list of chat messages.      
	/// - Deprecated: 30/07/2024. Use ``getHistoryRange(begin:end:filters:)`` instead. 
	@available(*, deprecated)
	public func getHistoryRange(begin:Int, end:Int) -> [ChatMessage]
	{
		var swiftList = [ChatMessage]()
		let cList = linphone_chat_room_get_history_range(cPtr, CInt(begin), CInt(end))
		var listTemp = cList
		while (listTemp != nil) {
			let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
			swiftList.append(ChatMessage.getSwiftObject(cObject: data))
			listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
		}
			bctbx_list_free_with_data(cList, belle_sip_object_unref)
		return swiftList
	}
	
	
	
	/// Gets the partial list of messages between two given ``EventLog``, sorted from
	/// oldest to most recent. 
	/// If either first_event or last_event is null, then nothing is returned. 
	/// - Parameter firstEvent: The ``EventLog`` object corresponding to the event.    
	/// - Parameter lastEvent: The ``EventLog`` object corresponding to the event.    
	/// - Parameter filters: The LinphoneChatRoomHistoryFilterMask mask to filter the
	/// results with ``HistoryFilter`` 
	/// - Returns: A list of   between the two provided events, if any.    
	public func getHistoryRangeBetween(firstEvent:EventLog?, lastEvent:EventLog?, filters:UInt) -> [EventLog]
	{
		var swiftList = [EventLog]()
		let cList = linphone_chat_room_get_history_range_between(cPtr, firstEvent?.cPtr, lastEvent?.cPtr, CUnsignedInt(filters))
		var listTemp = cList
		while (listTemp != nil) {
			let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
			swiftList.append(EventLog.getSwiftObject(cObject: data))
			listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
		}
			bctbx_list_free_with_data(cList, belle_sip_object_unref)
		return swiftList
	}
	
	
	
	/// Gets the partial list of events in the given range, sorted from oldest to most
	/// recent. 
	/// - Parameter begin: The first event of the range to be retrieved. History most
	/// recent event has index 0. 
	/// - Parameter end: The last event of the range to be retrieved. History oldest
	/// event has index of history size 
	/// - Returns: The list of the found events.      
	public func getHistoryRangeEvents(begin:Int, end:Int) -> [EventLog]
	{
		var swiftList = [EventLog]()
		let cList = linphone_chat_room_get_history_range_events(cPtr, CInt(begin), CInt(end))
		var listTemp = cList
		while (listTemp != nil) {
			let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
			swiftList.append(EventLog.getSwiftObject(cObject: data))
			listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
		}
			bctbx_list_free_with_data(cList, belle_sip_object_unref)
		return swiftList
	}
	
	
	
	/// Gets the partial list of chat message events in the given range, sorted from
	/// oldest to most recent. 
	/// - Parameter begin: The first event of the range to be retrieved. History most
	/// recent event has index 0. 
	/// - Parameter end: The last event of the range to be retrieved. History oldest
	/// event has index of history size 
	/// - Returns: The list of chat message events.      
	public func getHistoryRangeMessageEvents(begin:Int, end:Int) -> [EventLog]
	{
		var swiftList = [EventLog]()
		let cList = linphone_chat_room_get_history_range_message_events(cPtr, CInt(begin), CInt(end))
		var listTemp = cList
		while (listTemp != nil) {
			let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
			swiftList.append(EventLog.getSwiftObject(cObject: data))
			listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
		}
			bctbx_list_free_with_data(cList, belle_sip_object_unref)
		return swiftList
	}
	
	
	
	/// Gets the partial list of messages in the given range near the ``EventLog``
	/// provided, sorted from oldest to most recent. 
	/// If before and after are both set to 0, the returned list is empty. 
	/// - Parameter before: The number of messages to retrieve before the event
	/// provided. 
	/// - Parameter after: The number of messages to retrieve after the event provided. 
	/// - Parameter event: The ``EventLog`` object corresponding to the event.    
	/// - Parameter filters: The LinphoneChatRoomHistoryFilterMask mask to filter the
	/// results with ``HistoryFilter`` 
	/// - Returns: A list of   near the event provided included.    
	public func getHistoryRangeNear(before:UInt, after:UInt, event:EventLog?, filters:UInt) -> [EventLog]
	{
		var swiftList = [EventLog]()
		let cList = linphone_chat_room_get_history_range_near(cPtr, CUnsignedInt(before), CUnsignedInt(after), event?.cPtr, CUnsignedInt(filters))
		var listTemp = cList
		while (listTemp != nil) {
			let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
			swiftList.append(EventLog.getSwiftObject(cObject: data))
			listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
		}
			bctbx_list_free_with_data(cList, belle_sip_object_unref)
		return swiftList
	}
	
	
	
	/// Gets the number of messages in a chat room. 
	/// - Parameter filters: The LinphoneChatRoomHistoryFilterMask mask to filter the
	/// result with ``HistoryFilter``    
	/// - Returns: the number of messages. 
	public func getHistorySize(filters:UInt) -> Int
	{
		return Int(linphone_chat_room_get_history_size_2(cPtr, CUnsignedInt(filters)))
	}
	
	
	
	/// Gets the partial list of contents for which content-type starts with either
	/// video/, audio/ or image/. 
	/// - Parameter begin: The first content of the range to be retrieved. Most recent
	/// content has index 0. 
	/// - Parameter end: The last content of the range to be retrieved. Oldest content
	/// has index of size (use ``getMediaContentsSize()`` to retrieve size) 
	/// - Returns: A list of contents considered as "media".      
	public func getMediaContentsRange(begin:Int, end:Int) -> [Content]
	{
		var swiftList = [Content]()
		let cList = linphone_chat_room_get_media_contents_range(cPtr, CInt(begin), CInt(end))
		var listTemp = cList
		while (listTemp != nil) {
			let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
			swiftList.append(Content.getSwiftObject(cObject: data))
			listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
		}
			bctbx_list_free_with_data(cList, belle_sip_object_unref)
		return swiftList
	}
	
	
	
	/// Returns whether or not the chat room has been left. 
	/// - Returns: true if the chat room has been left, false otherwise. 
	/// - Deprecated: 16/03/2022 use ``isReadOnly()`` instead. 
	@available(*, deprecated)
	public func hasBeenLeft() -> Bool
	{
		return linphone_chat_room_has_been_left(cPtr) != 0
	}
	
	
	
	/// Checks if a chat room has given capabilities. 
	/// - Parameter mask: a LinphoneChatRoomCapabilitiesMask mask 
	/// - Returns: true if the mask matches, false otherwise 
	public func hasCapability(mask:Int) -> Bool
	{
		return linphone_chat_room_has_capability(cPtr, CInt(mask)) != 0
	}
	
	
	
	/// Leaves a chat room. 
	public func leave() 
	{
		linphone_chat_room_leave(cPtr)
	}
	
	
	
	/// Mark all messages of the conversation as read. 
	public func markAsRead() 
	{
		linphone_chat_room_mark_as_read(cPtr)
	}
	
	
	
	/// Nominates a new admin and then leaves a chat room. 
	/// - Parameter newAdmin: The ``Address`` of the new admin    
	/// - Note: The local participant will not leave the chat room if the new admin
	/// cannot be nominated 
	public func nominateAdminAndLeave(newAdmin:Address) 
	{
		linphone_chat_room_nominate_admin_and_leave(cPtr, newAdmin.cPtr)
	}
	
	
	
	/// Notifies the chatroom that a participant device has just registered. 
	/// This function is meaningful only for server implementation of chatroom, and
	/// shall not by used by client applications. 
	/// - Parameter participantDevice: list of the participant devices to be used by
	/// the group chat room    
	public func notifyParticipantDeviceRegistration(participantDevice:Address) 
	{
		linphone_chat_room_notify_participant_device_registration(cPtr, participantDevice.cPtr)
	}
	
	
	
	/// Used to receive a chat message when using async mechanism with IM
	/// enchat_roomyption engine. 
	/// - Parameter message: ``ChatMessage`` object    
	public func receiveChatMessage(message:ChatMessage) 
	{
		linphone_chat_room_receive_chat_message(cPtr, message.cPtr)
	}
	
	
	
	/// Removes a participant of a chat room. 
	/// - Parameter participant: The participant to remove from the chat room    
	public func removeParticipant(participant:Participant) 
	{
		linphone_chat_room_remove_participant(cPtr, participant.cPtr)
	}
	
	
	
	/// Removes several participants of a chat room at once. 
	/// - Parameter participants: The participants to remove.      
	public func removeParticipants(participants:[Participant]) 
	{
		linphone_chat_room_remove_participants(cPtr, ObjectArrayToBctbxList(list: participants))
	}
	
	
	
	/// Deletes the content of a previously sent message for both sender and receivers. 
	/// Message will still appear in the conversation history but will be empty. You
	/// can still delete it from history using ``deleteMessage(message:)``. 
	/// - Parameter message: The ``ChatMessage`` object to delete.    
	public func retractMessage(message:ChatMessage) 
	{
		linphone_chat_room_retract_message(cPtr, message.cPtr)
	}
	
	
	
	/// Searches chat messages by text. 
	/// - Parameter text: The text to search in messages    
	/// - Parameter from: The ``EventLog`` object corresponding to the event where to
	/// start the search    
	/// - Parameter direction: The ``SearchDirection`` where to search,
	/// LinphoneSearchDirectionUp will search older messages while
	/// LinphoneSearchDirectionDown will search newer messages 
	/// - Returns: the ``EventLog`` corresponding to the message if found or nil.    
	public func searchChatMessageByText(text:String, from:EventLog?, direction:SearchDirection) -> EventLog?
	{
		let cPointer = linphone_chat_room_search_chat_message_by_text(cPtr, text, from?.cPtr, LinphoneSearchDirection(rawValue: CUnsignedInt(direction.rawValue)))
		if (cPointer == nil) {
			return nil
		}
		let result = EventLog.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Changes the admin status of a participant of a chat room (you need to be an
	/// admin yourself to do this). 
	/// - Parameter participant: The Participant for which to change the admin status  
	///  
	/// - Parameter isAdmin: A boolean value telling whether the participant should now
	/// be an admin or not 
	public func setParticipantAdminStatus(participant:Participant, isAdmin:Bool) 
	{
		linphone_chat_room_set_participant_admin_status(cPtr, participant.cPtr, isAdmin==true ? 1:0)
	}
	
	
	
	/// Sets the list of participant devices in the form of SIP URIs with GRUUs for a
	/// given participant. 
	/// This function is meaningful only for server implementation of chatroom, and
	/// shall not by used by client applications. 
	/// - Parameter participantAddress: The participant address    
	/// - Parameter deviceIdentities: List of the participant devices to be used by the
	/// group chat room      
	public func setParticipantDevices(participantAddress:Address, deviceIdentities:[ParticipantDeviceIdentity]) 
	{
		linphone_chat_room_set_participant_devices(cPtr, participantAddress.cPtr, ObjectArrayToBctbxList(list: deviceIdentities))
	}
	
	
	
	/// Notifies the destination of the chat message that the user is no longer
	/// composing. 
	public func stopComposing() 
	{
		linphone_chat_room_stop_composing(cPtr)
	}
}


/// Object defining parameters for a ``ChatRoom``. 
/// Can be created with ``Core/createDefaultChatRoomParams()``. You can use
/// ``isValid()`` to check if your configuration is valid or not.
/// If the ``ChatRoom`` backend is ``Basic``, then no other parameter is required,
/// but ``ChatMessage`` sent and received won't benefit from all features a
/// ``FlexisipChat`` can offer like conversation with multiple participants and a
/// subject, end-to-end encryption, ephemeral messages, etc... but this type is the
/// only one that can interoperate with other SIP clients or with non-flexisip SIP
/// proxies. 
public class ChatRoomParams : LinphoneObject
{

	static public func getSwiftObject(cObject:OpaquePointer) -> ChatRoomParams {
		let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
		if (result != nil) {
			return Unmanaged<ChatRoomParams>.fromOpaque(result!).takeUnretainedValue()
		}
		let sObject = ChatRoomParams(cPointer: cObject)
		belle_sip_object_data_set(UnsafeMutablePointer(cObject), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(sObject).toOpaque()), nil)
		return sObject
	}

	public var getCobject: OpaquePointer? {
		return cPtr
	}
	
	/// Set the backend implementation of these chat room parameters. 
	/// - Parameter backend: The ``ChatRoom.Backend`` enum value 
	/// - Deprecated: 20/05/2024. Use ``ChatParams/setBackend(backend:)`` instead. 
	
	/// Get the backend implementation of the chat room associated with the given
	/// parameters. 
	/// - Returns: the ``ChatRoom.Backend`` 
	/// - Deprecated: 20/05/2024. Use ``ChatParams/getBackend()`` instead. 
	public var backend: ChatRoom.Backend
	{
	@available(*, deprecated)
		get
		{ 
						return ChatRoom.Backend(rawValue: Int(linphone_chat_room_params_get_backend(cPtr).rawValue))
		}
	@available(*, deprecated)
		set
		{
			linphone_chat_room_params_set_backend(cPtr, LinphoneChatRoomBackend(rawValue: CUnsignedInt(newValue.rawValue)))
		}
	}
		
	/// Set the encryption backend implementation of these chat room parameters. 
	/// - Parameter backend: The ``ChatRoom.EncryptionBackend`` enum value 
	/// - Deprecated: 20/05/2024. Use ``ChatParams/setEncryptionBackend(backend:)``
	/// instead. 
	
	/// Get the encryption implementation of the chat room associated with the given
	/// parameters. 
	/// - Returns: the ``ChatRoom.EncryptionBackend`` 
	/// - Deprecated: 20/05/2024. Use ``ChatParams/getEncryptionBackend()`` instead. 
	public var encryptionBackend: ChatRoom.EncryptionBackend
	{
	@available(*, deprecated)
		get
		{ 
						return ChatRoom.EncryptionBackend(rawValue: Int(linphone_chat_room_params_get_encryption_backend(cPtr).rawValue))
		}
	@available(*, deprecated)
		set
		{
			linphone_chat_room_params_set_encryption_backend(cPtr, LinphoneChatRoomEncryptionBackend(rawValue: CUnsignedInt(newValue.rawValue)))
		}
	}
		
	/// Enables or disables encryption for the chat room associated with the given
	/// parameters. 
	/// - Parameter encrypted: true to enable encryption, false to disable. 
	/// - Deprecated: 20/05/2024. Use linphone_conference_params_enable_encryption()
	/// instead. 
	
	/// Get the encryption status of the chat room associated with the given
	/// parameters. 
	/// - Returns: true if encryption is enabled, false otherwise 
	/// - Deprecated: 20/05/2024. Use ``ChatParams/encryptionEnabled()`` instead. 
	public var encryptionEnabled: Bool
	{
	@available(*, deprecated)
		get
		{ 
						return linphone_chat_room_params_encryption_enabled(cPtr) != 0
		}
	@available(*, deprecated)
		set
		{
			linphone_chat_room_params_enable_encryption(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Set lifetime (in seconds) for all new ephemral messages in the chat room. 
	/// After the message is read, it will be deleted after "time" seconds. - See also:
	/// linphone_chat_room_params_ephemeral_enabled() 
	/// - Parameter time: The ephemeral lifetime, default is disabled (0) 
	/// - Deprecated: 20/05/2024. Use ``ChatParams/setEphemeralLifetime(time:)``
	/// instead. 
	
	/// Get lifetime (in seconds) for all new ephemeral messages in the chat room. 
	/// After the message is read, it will be deleted after "time" seconds. - See also:
	/// linphone_chat_room_params_ephemeral_enabled() 
	/// - Returns: the ephemeral lifetime (in seconds) 
	/// - Deprecated: 20/05/2024. Use ``ChatParams/getEphemeralLifetime()`` instead. 
	public var ephemeralLifetime: Int
	{
	@available(*, deprecated)
		get
		{ 
						return Int(linphone_chat_room_params_get_ephemeral_lifetime(cPtr))
		}
	@available(*, deprecated)
		set
		{
			linphone_chat_room_params_set_ephemeral_lifetime(cPtr, (newValue))
		}
	}
		
	/// Enables or disables forcing of ephemeral messages for the chat room associated
	/// with the given parameters. 
	/// - Parameter mode: Ephemeral message mode ``ChatRoom.EphemeralMode``. 
	/// - Deprecated: 20/05/2024. Use ``ChatParams/setEphemeralMode(mode:)`` instead. 
	
	/// Get the ephemeral message mode of the chat room associated with the given
	/// parameters. 
	/// - Returns: the ephemeral message mode ``ChatRoom.EphemeralMode`` 
	/// - Deprecated: 20/05/2024. Use ``ChatParams/getEphemeralMode()`` instead. 
	public var ephemeralMode: ChatRoom.EphemeralMode
	{
	@available(*, deprecated)
		get
		{ 
						return ChatRoom.EphemeralMode(rawValue: Int(linphone_chat_room_params_get_ephemeral_mode(cPtr).rawValue))!
		}
	@available(*, deprecated)
		set
		{
			linphone_chat_room_params_set_ephemeral_mode(cPtr, LinphoneChatRoomEphemeralMode(rawValue: CUnsignedInt(newValue.rawValue)))
		}
	}
		
	/// Enables or disables group chat for the chat room associated with the given
	/// parameters. 
	/// - Parameter group: true to enable group chat, false to disable (resulting in
	/// one-on-one chat room) 
	/// - Deprecated: 20/05/2024. Use ``ConferenceParams/enableGroup(group:)`` instead. 
	
	/// Get the group chat status of the chat room associated with the given
	/// parameters. 
	/// - Returns: true if group chat is enabled, false if one-on-one 
	/// - Deprecated: 20/05/2024. Use ``ConferenceParams/groupEnabled()`` instead. 
	public var groupEnabled: Bool
	{
	@available(*, deprecated)
		get
		{ 
						return linphone_chat_room_params_group_enabled(cPtr) != 0
		}
	@available(*, deprecated)
		set
		{
			linphone_chat_room_params_enable_group(cPtr, newValue==true ? 1:0)
		}
	}
		
	
	/// Returns whether the given parameters are valid or not. 
	/// - Returns: true if the given parameters are valid, false otherwise 
	/// - Deprecated: 20/05/2024. Use ``ConferenceParams/isValid()`` instead. 
	@available(*, deprecated)
	public var isValid: Bool
	{
	
						return linphone_chat_room_params_is_valid(cPtr) != 0

	}
		
	/// Enables or disables real time text for the chat room associated with the given
	/// parameters. 
	/// - Parameter rtt: true to enable real time text, false to disable. 
	/// - Deprecated: 20/05/2024. Use ``ChatParams/enableRtt(rtt:)`` instead. 
	
	/// Get the real time text status of the chat room associated with the given
	/// parameters. 
	/// - Returns: true if real time text is enabled, false otherwise 
	/// - Deprecated: 20/05/2024. Use ``ChatParams/rttEnabled()`` instead. 
	public var rttEnabled: Bool
	{
	@available(*, deprecated)
		get
		{ 
						return linphone_chat_room_params_rtt_enabled(cPtr) != 0
		}
	@available(*, deprecated)
		set
		{
			linphone_chat_room_params_enable_rtt(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Set the subject of the chat room. 
	/// - Parameter subject: The subject to set.    
	/// - Deprecated: 20/05/2024. Use ``ConferenceParams/setSubject(subject:)``
	/// instead. 
	
	/// Get the subject of the chat room. 
	/// - Returns: The subject.    
	/// - Deprecated: 20/05/2024. Use ``ConferenceParams/getSubject()`` instead. 
	public var subject: String?
	{
	@available(*, deprecated)
		get
		{ 
			
			let cPointer = linphone_chat_room_params_get_subject(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
	@available(*, deprecated)
		set
		{
			linphone_chat_room_params_set_subject(cPtr, newValue)
		}
	}
	}


/// Object that represents a ``ChatRoom`` participant that is currently composing. 
public class ComposingParticipant : LinphoneObject
{

	static public func getSwiftObject(cObject:OpaquePointer) -> ComposingParticipant {
		let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
		if (result != nil) {
			return Unmanaged<ComposingParticipant>.fromOpaque(result!).takeUnretainedValue()
		}
		let sObject = ComposingParticipant(cPointer: cObject)
		belle_sip_object_data_set(UnsafeMutablePointer(cObject), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(sObject).toOpaque()), nil)
		return sObject
	}

	public var getCobject: OpaquePointer? {
		return cPtr
	}
	
	
	/// Gets the participant's address. 
	/// - Returns: the ``Address`` of the participant.    
	public var address: Address?
	{
	
						let cPointer = linphone_composing_participant_get_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Gets the content-type of what the participant is being composing. 
	/// - Returns: the content-type set if any, nil otherwise.    
	public var contentType: String?
	{
	
			
			let cPointer = linphone_composing_participant_get_content_type(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	
	/// Clones a composing participant. 
	/// - Returns: The newly created ``ComposingParticipant`` object.    
	public func clone() -> ComposingParticipant?
	{
		let cPointer = linphone_composing_participant_clone(cPtr)
		if (cPointer == nil) {
			return nil
		}
		let result = ComposingParticipant.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
}


/// A conference is the object that allow to make calls when there are 2 or more
/// participants. 
/// To create (or find) a ``Conference``, you first need a ``ConferenceParams``
/// object. ``Core/createConferenceWithParams(params:)`` allows you to create a
/// conference. A conference is uniquely identified by a conference address,
/// meaning you can have more than one conference between two accounts. To find a
/// conference among those a core is part of, you can call
/// ``Core/searchConference(params:localAddr:remoteAddr:participants:)``.
/// A ``Conference`` may be created automatically and implicitely when an outgoing
/// call is made to a conference server. Thanks to the standard 'isfocus' contact
/// parameter, the call is identified as belonging to a conference. The conference
/// object can then be retrieved with ``Call/getConference()``. 
public class Conference : LinphoneObject
{
	var delegateManagers : [ConferenceDelegateManager] = []

	static public func getSwiftObject(cObject:OpaquePointer) -> Conference {
		let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
		if (result != nil) {
			return Unmanaged<Conference>.fromOpaque(result!).takeUnretainedValue()
		}
		let sObject = Conference(cPointer: cObject)
		belle_sip_object_data_set(UnsafeMutablePointer(cObject), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(sObject).toOpaque()), nil)
		return sObject
	}

	public var getCobject: OpaquePointer? {
		return cPtr
	}

	
	///Conference minimum security level. 
	public enum SecurityLevel:Int
	{
		
		/// No security. 
		case None = 0
		/// Point-to-point encryption. 
		case PointToPoint = 1
		/// End-to-end encryption. 
		case EndToEnd = 2
	}

	
	///Type of conference participant list. 
	public enum ParticipantListType:Int
	{
		
		/// Only participants in the initiating INVITE are allowed to join the conference. 
		case Closed = 0
		/// All devices calling the conference URI are allowed to join the conference. 
		case Open = 1
	}

	
	///Mode of joining conference. 
	public enum JoiningMode:Int
	{
		
		/// Participants must dial the conference server. 
		case DialIn = 0
		/// Conference server dials participants. 
		case DialOut = 1
	}

	
	///``Layout`` is used to indicate the layout used by the conference. 
	public enum Layout:Int
	{
		
		/// Grid - each participant is given an equal sized image size. 
		case Grid = 0
		/// Active speaker - participant who speaks is prominently displayed in the center
		/// of the screen and other participants are minimized. 
		case ActiveSpeaker = 1
	}

	
	///``State`` is used to indicate the current state of a conference. 
	public enum State:Int
	{
		
		/// Initial state. 
		case None = 0
		/// Conference is now instantiated locally. 
		case Instantiated = 1
		/// One creation request was sent to the service. 
		case CreationPending = 2
		/// Conference was created on the service. 
		case Created = 3
		/// Conference creation on service failed. 
		case CreationFailed = 4
		/// Wait for conference termination. 
		case TerminationPending = 5
		/// The conference is terminated locally, though it may still exist on the service
		/// for other participants. 
		case Terminated = 6
		/// Conference termination failed. 
		case TerminationFailed = 7
		/// Conference was deleted locally and on the service. 
		case Deleted = 8
	}
	
	
	
	public func addDelegate(delegate: ConferenceDelegate)
	{
		let manager = ConferenceDelegateManager()
		manager.delegate = delegate
		delegateManagers.append(manager)
		linphone_conference_add_callbacks(cPtr, manager.cPtr)
		belle_sip_object_unref(UnsafeMutableRawPointer(manager.cPtr))
		belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(self).toOpaque()), nil)
	}	
	
	
	public func removeDelegate(delegate: ConferenceDelegate)
	{
		if let index = delegateManagers.firstIndex(where: { $0.delegate === delegate as AnyObject}) {
			linphone_conference_remove_callbacks(cPtr, delegateManagers[index].cPtr)
			delegateManagers.remove(at: index)
			belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(self).toOpaque()), nil)
		}
	}	
	
	
	/// A function to convert a ``State`` into a string. 
	static public func stateToString(state:Conference.State) -> String
	{
		let cstr = linphone_conference_state_to_string(LinphoneConferenceState(rawValue: CUnsignedInt(state.rawValue)))
		let result = charArrayToString(charPointer: cstr)
		return result
	}
	
	
	/// Gets the ``Account`` object associated with the conference. 
	/// - Returns: A ``Account`` object.    
	public var account: Account?
	{
	
						let cPointer = linphone_conference_get_account(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Account.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Get the currently active speaker participant device. 
	/// - Returns: the ``ParticipantDevice`` currently displayed as active speaker.    
	public var activeSpeakerParticipantDevice: ParticipantDevice?
	{
	
						let cPointer = linphone_conference_get_active_speaker_participant_device(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = ParticipantDevice.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Gets the call that is controlling a conference. 
	/// -for the local conference, it will return nil
	/// -for the client conference, it will return call associated to the conference
	/// - Returns: the ``Call`` controlling the conference or nil if none or local
	/// conference    
	public var call: Call?
	{
	
						let cPointer = linphone_conference_get_call(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Call.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Returns the ``ChatRoom`` linked to the ``Conference``. 
	/// - Returns: back pointer to ``ChatRoom`` object.    
	public var chatRoom: ChatRoom?
	{
	
						let cPointer = linphone_conference_get_chat_room(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = ChatRoom.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	/// Set the conference address. 
	/// - Parameter address: the conference address to set.    
	/// - Warning: This is only allowed for a client conference if it is in state
	/// CreationPending or Instantiated 
	
	/// Get the conference address of the conference. 
	/// This function may be return a nil pointer if called before the conference
	/// switches to the Created state 
	/// - Returns: The conference address of the conference.    
	public var conferenceAddress: Address?
	{
	
		get
		{ 
						let cPointer = linphone_conference_get_conference_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_conference_set_conference_address(cPtr, newValue?.cPtr)
		}
	}
		
	
	/// Returns core for a ``Conference``. 
	/// - Returns: back pointer to ``Core`` object.    
	public var core: Core?
	{
	
						let cPointer = linphone_conference_get_core(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Core.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Sets the current ``ConferenceDelegate``. 
	/// This is meant only to be called from a callback to be able to get the user_data
	/// associated with the ``ConferenceDelegate`` that is calling the callback. 
	/// - Returns: The ``ConferenceDelegate`` that has called the last callback.    
	public var currentDelegate: ConferenceDelegate?
	{
	
			let cObject = linphone_conference_get_current_callbacks(cPtr)
			let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
			if (result != nil) {
				return Unmanaged<ConferenceDelegateManager>.fromOpaque(result!).takeUnretainedValue().delegate
				}
			return nil
			
	}
		
	
	/// Get current parameters of the conference. 
	/// - Returns: a ``ConferenceParams`` .    
	public var currentParams: ConferenceParams?
	{
	
						let cPointer = linphone_conference_get_current_params(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = ConferenceParams.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Get the conference duration. 
	/// - Returns: conference duration.    
	public var duration: Int?
	{
	
						return Int(linphone_conference_get_duration(cPtr))

	}
		
	
	/// Returns the conference identifier. 
	/// - Warning: This method returns a nil pointer if the Conference is in the
	/// Instantiated state 
	/// - Returns: the conference identifier.    
	public var identifier: String?
	{
	
			
			let cPointer = linphone_conference_get_identifier(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Gets the ``ConferenceInfo`` object associated with a conference. 
	/// - Returns: A ``ConferenceInfo`` object.    
	public var info: ConferenceInfo?
	{
	
						let cPointer = linphone_conference_get_info(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = ConferenceInfo.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	/// Sets the given ``AudioDevice`` as input for this conference only. 
	/// - Parameter audioDevice: The ``AudioDevice``. nil does nothing.    
	
	/// Gets the current input device for this conference. 
	/// - Returns: the ``AudioDevice`` used by this conference as input or nil if there
	/// is currently no soundcard configured (depending on the state of the conference)
	///    
	public var inputAudioDevice: AudioDevice?
	{
	
		get
		{ 
						let cPointer = linphone_conference_get_input_audio_device(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = AudioDevice.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_conference_set_input_audio_device(cPtr, newValue?.cPtr)
		}
	}
		
	
	/// Retrieves the volume of a specific participant. 
	/// - Returns: The volume of the participant expressed in dbm0. 
	public var inputVolume: Float
	{
	
						return linphone_conference_get_input_volume(cPtr)

	}
		
	
	/// For a local conference, it returns whether the local participant is enabled For
	/// a client conference, it return whether the remote participant has left the
	/// conference without bein removed from it. 
	/// - Returns: true if the local participant is in a conference, false otherwise. 
	public var isIn: Bool
	{
	
						return linphone_conference_is_in(cPtr) != 0

	}
		
	
	/// Gets whether the conference is currently being recorded. 
	/// - Returns: true if conference is being recorded, false otherwise. 
	public var isRecording: Bool
	{
	
						return linphone_conference_is_recording(cPtr) != 0

	}
		
	
	/// For a local audio video conference, this function returns the participant
	/// hosting the conference For a remote audio video conference, this function
	/// returns the local participant of the conference. 
	/// - Returns: a ``Participant`` .    
	public var me: Participant?
	{
	
						let cPointer = linphone_conference_get_me(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Participant.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	/// Get microphone muted state. 
	/// Note that the microphone may be disabled globally if false was given to
	/// ``Core/enableMic(enable:)``. 
	/// - Parameter muted: The microphone muted state 
	
	/// Retrieves the volume of a specific participant. 
	/// - Returns: true if the microphone is muted, false otherwise 
	public var microphoneMuted: Bool
	{
	
		get
		{ 
						return linphone_conference_get_microphone_muted(cPtr) != 0
		}
		set
		{
			linphone_conference_set_microphone_muted(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets the given ``AudioDevice`` as output for this conference only. 
	/// - Parameter audioDevice: The ``AudioDevice``. nil does nothing.    
	
	/// Gets the current output device for this conference. 
	/// - Returns: the ``AudioDevice`` used by this conference as output or nil if
	/// there is currently no soundcard configured (depending on the state of the
	/// conference)    
	public var outputAudioDevice: AudioDevice?
	{
	
		get
		{ 
						let cPointer = linphone_conference_get_output_audio_device(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = AudioDevice.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_conference_set_output_audio_device(cPtr, newValue?.cPtr)
		}
	}
		
	
	/// Get number of participants without me. 
	/// - Returns: the number of participants excluding me in a ``Conference`` 
	public var participantCount: Int
	{
	
						return Int(linphone_conference_get_participant_count(cPtr))

	}
		
	
	/// Get list of all participant devices of a conference including me if it is in. 
	/// - Returns: The list of participant devices of the conference.         
	public var participantDeviceList: [ParticipantDevice]
	{
	
						var swiftList = [ParticipantDevice]()
			let cList = linphone_conference_get_participant_device_list(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(ParticipantDevice.getSwiftObject(cObject: data))
				listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
				bctbx_list_free_with_data(cList, belle_sip_object_unref)
			return swiftList

	}
		
	
	/// Get list of all participants of a conference. 
	/// - Warning: The returned list does not include me. 
	/// - Returns: The list of participants of the conference.         
	public var participantList: [Participant]
	{
	
						var swiftList = [Participant]()
			let cList = linphone_conference_get_participant_list(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(Participant.getSwiftObject(cObject: data))
				listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
				bctbx_list_free_with_data(cList, belle_sip_object_unref)
			return swiftList

	}
		
	
	/// Get URIs of all participants of one conference The returned bctbx_list_t
	/// contains URIs of all participants. 
	/// That list must be freed after use and each URI must be unref with
	/// linphone_address_unref - Warning: The returned list does not include me. 
	/// - Returns: The list of the participants' address active in the conference.     
	///    
	/// - Deprecated: 10/07/2020 Use ``getParticipantList()`` instead. 
	@available(*, deprecated)
	public var participants: [Address]
	{
	
						var swiftList = [Address]()
			let cList = linphone_conference_get_participants(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(Address.getSwiftObject(cObject: data))
				listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
				bctbx_list_free_with_data(cList, belle_sip_object_unref)
			return swiftList

	}
		
	
	/// Gets a player associated with the conference to play a local file and stream it
	/// to the remote peers. 
	/// - Returns: A ``Player`` object.    
	public var player: Player?
	{
	
						let cPointer = linphone_conference_get_player(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Player.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Get the participant that is currently screen sharing. 
	/// - Returns: a pointer to the participant found or nullptr.    
	public var screenSharingParticipant: Participant?
	{
	
						let cPointer = linphone_conference_get_screen_sharing_participant(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Participant.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Get the participant device that is currently screen sharing. 
	/// - Returns: a pointer to the participant device found or nullptr.    
	public var screenSharingParticipantDevice: ParticipantDevice?
	{
	
						let cPointer = linphone_conference_get_screen_sharing_participant_device(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = ParticipantDevice.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Get the conference start time. 
	/// - Returns: conference start time.    
	public var startTime: time_t?
	{
	
						return linphone_conference_get_start_time(cPtr)

	}
		
	
	/// Get the current state of the conference. 
	/// - Returns: the ``State`` of the conference. 
	public var state: Conference.State
	{
	
						return Conference.State(rawValue: Int(linphone_conference_get_state(cPtr).rawValue))!

	}
		
	/// Set the conference subject. 
	/// - Parameter subject: conference subject    
	
	/// Get the conference subject. 
	/// - Returns: conference subject.    
	public var subject: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_conference_get_subject(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_conference_set_subject(cPtr, newValue)
		}
	}
		
	/// Set the conference subject as an UTF-8 string. 
	/// - Parameter subject: conference subject    
	
	/// Get the conference subject as an UTF-8 string. 
	/// - Returns: conference subject.    
	public var subjectUtf8: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_conference_get_subject_utf8(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_conference_set_subject_utf8(cPtr, newValue)
		}
	}
		
	/// Associate a user pointer to the linphone conference. 
	/// - Parameter userData: The user data to associate with the ``Conference``
	/// object.    
	
	/// Retrieves the user pointer that was given to ``setUserData(userData:)`` 
	/// - Returns: The user data associated with the ``Conference`` object.    
	public var userData: UnsafeMutableRawPointer?
	{
	
		get
		{ 
						return linphone_conference_get_user_data(cPtr)
		}
		set
		{
			linphone_conference_set_user_data(cPtr, newValue)
		}
	}
		
	/// Set the conference username. 
	/// - Parameter username: conference subject    
	
	/// Get the conference username. 
	/// - Returns: conference subject.    
	public var username: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_conference_get_username(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_conference_set_username(cPtr, newValue)
		}
	}
		
	
	
	/// Join an existing call to the conference. 
	/// If the conference is in the state LinphoneConferenceStateCreationPending, then
	/// the conference will start on the input and output audio devices used for the
	/// currently active call, if any 
	/// - Parameter call: a ``Call`` that has to be added to the conference.    
	/// - Warning: This function guarantees that the local endpoint is added to the
	/// conference only if one of calls added is in state StreamsRunning. It is highly
	/// recommended to call linphone_confererence_enter() to guarantee that the local
	/// endpoint is added to the conference. 
	public func addParticipant(call:Call) throws 
	{
		let exception_result = linphone_conference_add_participant(cPtr, call.cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "addParticipant returned value \(exception_result)")
		}
	}
	
	
	
	/// Join a participant to the conference. 
	/// - Parameter uri: a ``Address`` that has to be added to the conference.    
	/// - Warning: This function guarantees that the local endpoint is added to the
	/// conference only if there is a call state StreamsRunning towards one of the
	/// addresses. It is highly recommended to call linphone_confererence_enter() to
	/// guarantee that the local endpoint is added to the conference. 
	public func addParticipant(uri:Address) throws 
	{
		let exception_result = linphone_conference_add_participant_2(cPtr, uri.cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "addParticipant returned value \(exception_result)")
		}
	}
	
	
	
	/// Add participants to the conference, by supplying a list of ``Call``. 
	/// If the conference is in the state LinphoneConferenceStateCreationPending, then
	/// the conference will start on the input and output audio devices used for the
	/// currently active call, if any 
	/// - Parameter calls: A list of calls to add to the conference.      
	public func addParticipants(calls:[Call]) throws 
	{
		let exception_result = linphone_conference_add_participants(cPtr, ObjectArrayToBctbxList(list: calls))
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "addParticipants returned value \(exception_result)")
		}
	}
	
	
	
	/// Add participants to the conference, by supplying a list of ``Address``. 
	/// - Parameter addresses: A list of calls to add to the conference.      
	public func addParticipants(addresses:[Address]) throws 
	{
		let exception_result = linphone_conference_add_participants_2(cPtr, ObjectArrayToBctbxList(list: addresses))
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "addParticipants returned value \(exception_result)")
		}
	}
	
	
	
	/// Terminates a conference. 
	/// If no media is supported, it instructs the conference server to remove all
	/// participants 
	/// - Returns: 0 if the termination is successful, -1 otherwise. 
	public func close() -> Int
	{
		return Int(linphone_conference_close(cPtr))
	}
	
	
	
	/// For a local conference, the local participant joins the conference For a client
	/// conference, the participant rejoins the conference after leaving it earlier on. 
	/// - Returns: 0 if succeeded. Negative number if failed 
	public func enter() -> Int
	{
		return Int(linphone_conference_enter(cPtr))
	}
	
	
	
	/// Find a participant from a conference. 
	/// - Parameter uri: SIP URI of the participant to search.    
	/// - Returns: a pointer to the participant found or nullptr.    
	public func findParticipant(uri:Address) -> Participant?
	{
		let cPointer = linphone_conference_find_participant(cPtr, uri.cPtr)
		if (cPointer == nil) {
			return nil
		}
		let result = Participant.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Retrieves the volume of a specific participant. 
	/// - Parameter device: The Participant    
	/// - Returns: The volume of the participant expressed in dbm0. 
	public func getParticipantDeviceVolume(device:ParticipantDevice) -> Int
	{
		return Int(linphone_conference_get_participant_device_volume(cPtr, device.cPtr))
	}
	
	
	
	/// Invite participants to the conference, by supplying a list of ``Address`` If
	/// the conference is in the state LinphoneConferenceStateCreationPending, then the
	/// conference will start on the input and output audio devices used for the
	/// currently active call, if any This method will call every address supplied and
	/// add it to a conference on devices hosting a conference. 
	/// On the other hand, if the conference is created on a server, this method will
	/// allow to create either a chat only conference or an ad-hoc audio video
	/// conference 
	/// - Parameter addresses: A list of SIP addresses to invite.      
	/// - Parameter params: ``CallParams`` to use for inviting the participants.    
	/// - Warning: The ``CallParams`` are only honored by conference servers and
	/// devices that host the conference locally. 
	public func inviteParticipants(addresses:[Address], params:CallParams?) throws 
	{
		let exception_result = linphone_conference_invite_participants(cPtr, ObjectArrayToBctbxList(list: addresses), params?.cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "inviteParticipants returned value \(exception_result)")
		}
	}
	
	
	
	/// For a local audio video conference, this function compares the address provided
	/// as argument with that of participant hosting the conference For a remote audio
	/// video conference, this function compares the address provided as argument with
	/// that of the local participant of the conference. 
	/// - Parameter uri: A ``Address`` object    
	/// - Returns: true if the participant is me, false otherwise. 
	public func isMe(uri:Address) -> Bool
	{
		return linphone_conference_is_me(cPtr, uri.cPtr) != 0
	}
	
	
	
	/// For a local conference, the local participant leaves the conference For a
	/// client conference, the participant leaves the conference after joining it
	/// earlier on. 
	/// - Returns: 0 if succeeded. Negative number if failed 
	public func leave() -> Int
	{
		return Int(linphone_conference_leave(cPtr))
	}
	
	
	
	/// Nominates a new admin and then leaves a conference. 
	/// - Parameter newAdmin: The ``Address`` of the new admin    
	/// - Returns: 0 if succeeded. Negative number if failed 
	/// - Note: The local participant will not leave the chat room if the new admin
	/// cannot be nominated 
	/// - Warning: It is not applicable to conference servers. 
	public func nominateAdminAndLeave(newAdmin:Address) -> Int
	{
		return Int(linphone_conference_nominate_admin_and_leave(cPtr, newAdmin.cPtr))
	}
	
	
	
	/// - Parameter uri: URI of the participant to remove    
	/// - Warning: The passed participant uri must be one of those returned by
	/// ``getParticipants()`` 
	/// - Returns: 0 if succeeded, -1 if failed 
	/// - Deprecated: 10/07/2020 Use ``removeParticipant(participant:)`` instead. 
	@available(*, deprecated)
	public func removeParticipant(uri:Address) throws 
	{
		let exception_result = linphone_conference_remove_participant(cPtr, uri.cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "removeParticipant returned value \(exception_result)")
		}
	}
	
	
	
	/// - Parameter participant: participant to remove    
	/// - Warning: The passed participant must be one of those returned by
	/// ``getParticipantList()`` 
	/// - Warning: This method may destroy the conference if the only remaining
	/// participant had an existing call to the local participant before the conference
	/// was created 
	/// - Returns: 0 if succeeded, -1 if failed 
	public func removeParticipant(participant:Participant) throws 
	{
		let exception_result = linphone_conference_remove_participant_2(cPtr, participant.cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "removeParticipant returned value \(exception_result)")
		}
	}
	
	
	
	/// - Parameter call: call to remove    
	/// - Returns: 0 if succeeded, -1 if failed 
	/// - Deprecated: 10/07/2020 Use ``removeParticipant(participant:)`` instead. 
	@available(*, deprecated)
	public func removeParticipant(call:Call) throws 
	{
		let exception_result = linphone_conference_remove_participant_3(cPtr, call.cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "removeParticipant returned value \(exception_result)")
		}
	}
	
	
	
	/// Set stream capability on 'me' device of a local conference. 
	/// - Parameter direction: the direction of stream of type stream_type 
	/// - Parameter streamType: A ``StreamType`` 
	public func setLocalParticipantStreamCapability(direction:MediaDirection, streamType:StreamType) 
	{
		linphone_conference_set_local_participant_stream_capability(cPtr, LinphoneMediaDirection(rawValue: CInt(direction.rawValue)), LinphoneStreamType(rawValue: CUnsignedInt(streamType.rawValue)))
	}
	
	
	
	/// Change the admin status of a participant of a conference (you need to be an
	/// admin yourself to do this). 
	/// - Parameter participant: The Participant for which to change the admin status  
	///  
	/// - Parameter isAdmin: A boolean value telling whether the participant should now
	/// be an admin or not 
	public func setParticipantAdminStatus(participant:Participant, isAdmin:Bool) 
	{
		linphone_conference_set_participant_admin_status(cPtr, participant.cPtr, isAdmin==true ? 1:0)
	}
	
	
	
	/// Starts recording the conference. 
	/// - Parameter path: Where to record the conference    
	/// - Returns: 0 if succeeded. Negative number in case of failure. 
	public func startRecording(path:String) -> Int
	{
		return Int(linphone_conference_start_recording(cPtr, path))
	}
	
	
	
	/// Stops the conference recording. 
	/// - Returns: 0 if succeeded. Negative number in case of failure. 
	public func stopRecording() -> Int
	{
		return Int(linphone_conference_stop_recording(cPtr))
	}
	
	
	
	/// Terminates a conference. 
	/// - Returns: 0 if the termination is successful, -1 otherwise. 
	public func terminate() -> Int
	{
		return Int(linphone_conference_terminate(cPtr))
	}
	
	
	
	/// Update parameters of the conference. 
	/// This is typically used enable or disable the video stream in the conference. 
	/// - Parameter params: the new parameters to apply.    
	public func updateParams(params:ConferenceParams) -> Int
	{
		return Int(linphone_conference_update_params(cPtr, params.cPtr))
	}
}


/// Object defining all information related to a past or future conference. 
public class ConferenceInfo : LinphoneObject
{

	static public func getSwiftObject(cObject:OpaquePointer) -> ConferenceInfo {
		let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
		if (result != nil) {
			return Unmanaged<ConferenceInfo>.fromOpaque(result!).takeUnretainedValue()
		}
		let sObject = ConferenceInfo(cPointer: cObject)
		belle_sip_object_data_set(UnsafeMutablePointer(cObject), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(sObject).toOpaque()), nil)
		return sObject
	}

	public var getCobject: OpaquePointer? {
		return cPtr
	}

	
	///``State`` is used to list all states of a conference info object 
	public enum State:Int
	{
		
		/// New conference. 
		case New = 0
		/// Conference has been updated. 
		case Updated = 1
		/// Canceling a conference. 
		case Cancelled = 2
	}
	
	/// Sets the CCMP URI of the conference. 
	/// - Parameter uri: The URI of the conference in the CCMP server.    
	
	/// Retrieve the CCMP URI of the conference. 
	/// - Returns: The URI of the conference stored in the CCMP server.    
	public var ccmpUri: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_conference_info_get_ccmp_uri(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_conference_info_set_ccmp_uri(cPtr, newValue)
		}
	}
		
	/// Sets the date and time of the conference. 
	/// - Parameter datetime: The date and time of the conference. 
	
	/// Retrieve the date and time of the conference. 
	/// - Returns: The date and time of the conference. 
	public var dateTime: time_t
	{
	
		get
		{ 
						return linphone_conference_info_get_date_time(cPtr)
		}
		set
		{
			linphone_conference_info_set_date_time(cPtr, newValue)
		}
	}
		
	/// Sets the description of the conference. 
	/// - Parameter description: The description of the conference.    
	
	/// Retrieve the description of the conference. 
	/// - Returns: The description of the conference.    
	public var description: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_conference_info_get_description(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_conference_info_set_description(cPtr, newValue)
		}
	}
		
	/// Sets the description of the conference. 
	/// - Parameter description: The description of the conference.    
	
	/// Retrieve the description of the conference. 
	/// - Returns: The description of the conference.    
	public var descriptionUtf8: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_conference_info_get_description_utf8(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_conference_info_set_description_utf8(cPtr, newValue)
		}
	}
		
	/// Sets the duration (in minutes) of the conference. 
	/// - Parameter duration: The duration of the conference. 
	
	/// Retrieve the duration (in minutes) of the conference. 
	/// - Returns: The duration of the conference. 
	public var duration: UInt
	{
	
		get
		{ 
						return UInt(linphone_conference_info_get_duration(cPtr))
		}
		set
		{
			linphone_conference_info_set_duration(cPtr, CUnsignedInt(newValue))
		}
	}
		
	
	/// Retrieve the conference as an Icalendar string. 
	/// - Returns: The conference as an Icalendar string. The returned char* must be
	/// freed by the caller.       
	public var icalendarString: String?
	{
	
			
			let cPointer = linphone_conference_info_get_icalendar_string(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			if (cPointer != nil) {
				bctbx_free(cPointer)
			}
			return result

	}
		
	/// Store the ICS UID in the conference info. 
	/// - Parameter uid: the ICS UID to be associated to the ``ConferenceInfo`` object.
	///    
	
	/// Retrieve the ICS UID linked to a conference info. 
	/// - Returns: the ICS UID.    
	public var icsUid: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_conference_info_get_ics_uid(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_conference_info_set_ics_uid(cPtr, newValue)
		}
	}
		
	/// Sets the organizer of the conference. 
	/// - Parameter organizer: The ``Address`` of the conference's organizer.    
	
	/// Retrieve the organizer of the conference. 
	/// - Returns: The ``Address`` of the conference's organizer.    
	public var organizer: Address?
	{
	
		get
		{ 
						let cPointer = linphone_conference_info_get_organizer(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_conference_info_set_organizer(cPtr, newValue?.cPtr)
		}
	}
		
	/// Sets the organizer information of the conference. 
	/// - Parameter organizer: The ``ParticipantInfo`` of the conference's organizer.  
	///  
	
	/// Retrieve the organizer of the conference. 
	/// - Returns: The ``ParticipantInfo`` of the conference's organizer.    
	public var organizerInfo: ParticipantInfo?
	{
	
		get
		{ 
						let cPointer = linphone_conference_info_get_organizer_info(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = ParticipantInfo.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_conference_info_set_organizer_info(cPtr, newValue?.cPtr)
		}
	}
		
	/// Sets the list of participants. 
	/// - Parameter participantInfos: The list of participant informations to set.      
	
	/// Retrieve the list of participants as list of participant infos. 
	/// - Returns: The list of participant informations.      
	public var participantInfos: [ParticipantInfo]
	{
	
		get
		{ 
						var swiftList = [ParticipantInfo]()
			let cList = linphone_conference_info_get_participant_infos(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(ParticipantInfo.getSwiftObject(cObject: data))
				listTemp = UnsafePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
			return swiftList
		}
		set
		{
			withExtendedLifetime(newValue) {
				var cList: UnsafeMutablePointer<bctbx_list_t>? = nil
				for data in newValue {
					cList = bctbx_list_append(cList, UnsafeMutableRawPointer(data.cPtr))
				}
				linphone_conference_info_set_participant_infos(cPtr, cList)
			}
		}
	}
		
	/// Sets the list of participants. 
	/// - Parameter participants: The list of participants to set.      
	/// - Deprecated: 24/08/2023 use linphone_conference_info_set_participant_infos
	/// instead 
	
	/// Retrieve the list of participants as list of addresses. 
	/// - Returns: The list of participants.      
	/// - Deprecated: 24/08/2023 use linphone_conference_info_get_participant_infos
	/// instead 
	public var participants: [Address]
	{
	@available(*, deprecated)
		get
		{ 
						var swiftList = [Address]()
			let cList = linphone_conference_info_get_participants(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(Address.getSwiftObject(cObject: data))
				listTemp = UnsafePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
			return swiftList
		}
	@available(*, deprecated)
		set
		{
			withExtendedLifetime(newValue) {
				var cList: UnsafeMutablePointer<bctbx_list_t>? = nil
				for data in newValue {
					cList = bctbx_list_append(cList, UnsafeMutableRawPointer(data.cPtr))
				}
				linphone_conference_info_set_participants(cPtr, cList)
			}
		}
	}
		
	/// Sets the desired security level of the conference. 
	/// - Parameter securityLevel: The desired security level of the conference. 
	
	/// Retrieve the desired security level of the conference. 
	/// - Returns: The desired security level of the conference. 
	public var securityLevel: Conference.SecurityLevel
	{
	
		get
		{ 
						return Conference.SecurityLevel(rawValue: Int(linphone_conference_info_get_security_level(cPtr).rawValue))!
		}
		set
		{
			linphone_conference_info_set_security_level(cPtr, LinphoneConferenceSecurityLevel(rawValue: CUnsignedInt(newValue.rawValue)))
		}
	}
		
	
	/// Retrieve the state of the conference info. 
	/// - Returns: ``State`` object.    
	public var state: ConferenceInfo.State?
	{
	
						return ConferenceInfo.State(rawValue: Int(linphone_conference_info_get_state(cPtr).rawValue))!

	}
		
	/// Sets the subject of the conference. 
	/// - Parameter subject: The subject of the conference.    
	
	/// Retrieve the subject of the conference. 
	/// - Returns: The subject of the conference.    
	public var subject: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_conference_info_get_subject(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_conference_info_set_subject(cPtr, newValue)
		}
	}
		
	/// Sets the subject of the conference. 
	/// - Parameter subject: The subject of the conference.    
	
	/// Retrieve the subject of the conference. 
	/// - Returns: The subject of the conference.    
	public var subjectUtf8: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_conference_info_get_subject_utf8(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_conference_info_set_subject_utf8(cPtr, newValue)
		}
	}
		
	
	/// Retrieve the URI of the conference. 
	/// - Returns: The URI of the conference (``Address``).    
	public var uri: Address?
	{
	
						let cPointer = linphone_conference_info_get_uri(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	
	/// Add a participant to the conference. 
	/// - Parameter participant: The participant (``Address``) to add.    
	public func addParticipant(participant:Address) 
	{
		linphone_conference_info_add_participant(cPtr, participant.cPtr)
	}
	
	
	
	/// Add a participant to the conference. 
	/// - Parameter participantInfo: The participant information (``ParticipantInfo``)
	/// to add. This method can be called to set attributes such as the role to the
	/// organizer of the conference    
	public func addParticipant(participantInfo:ParticipantInfo) 
	{
		linphone_conference_info_add_participant_2(cPtr, participantInfo.cPtr)
	}
	
	
	
	/// Add a list of participants. 
	/// - Parameter participantInfos: The list of participant informations to add.      
	public func addParticipantInfos(participantInfos:[ParticipantInfo]?) 
	{
		linphone_conference_info_add_participant_infos(cPtr, ObjectArrayToBctbxList(list: participantInfos))
	}
	
	
	
	/// Clone an object ``ConferenceInfo``. 
	/// - Returns: the cloned ``ConferenceInfo`` object.    
	public func clone() -> ConferenceInfo?
	{
		let cPointer = linphone_conference_info_clone(cPtr)
		if (cPointer == nil) {
			return nil
		}
		let result = ConferenceInfo.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Find a participant information in the conference information. 
	/// - Parameter participant: The participant (``Address``) to search.    
	/// - Returns: The participant information (``ParticipantInfo``).    
	public func findParticipant(participant:Address) -> ParticipantInfo?
	{
		let cPointer = linphone_conference_info_find_participant(cPtr, participant.cPtr)
		if (cPointer == nil) {
			return nil
		}
		let result = ParticipantInfo.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Get the capability of the conference. 
	/// The capability information represents the capability for the conference linked
	/// to the ``ConferenceInfo`` to handle a given stream type (audio, video or text). 
	/// - Parameter streamType: A ``StreamType`` 
	/// - Returns: the capability of the conference linked to conference information
	/// ``ConferenceInfo`` 
	public func getCapability(streamType:StreamType) -> Bool
	{
		return linphone_conference_info_get_capability(cPtr, LinphoneStreamType(rawValue: CUnsignedInt(streamType.rawValue))) != 0
	}
	
	
	
	/// Remove a participant from the conference. 
	/// - Parameter participant: The participant (``Address``) to remove.    
	public func removeParticipant(participant:Address) 
	{
		linphone_conference_info_remove_participant(cPtr, participant.cPtr)
	}
	
	
	
	/// Sets the capability of the conference. 
	/// The capability information represents the capability for the conference linked
	/// to the ``ConferenceInfo`` to handle a given stream type (audio, video or text). 
	/// - Parameter streamType: A ``StreamType`` 
	/// - Parameter enable: the capability of the conference linked to conference
	/// information ``ConferenceInfo`` 
	public func setCapability(streamType:StreamType, enable:Bool) 
	{
		linphone_conference_info_set_capability(cPtr, LinphoneStreamType(rawValue: CUnsignedInt(streamType.rawValue)), enable==true ? 1:0)
	}
	
	
	
	/// Update the participant information in the conference informations. 
	/// - Parameter participantInfo: The participant information (``ParticipantInfo``)
	/// to update. This method can be called to change attributes such as the role to
	/// the organizer of the conference    
	public func updateParticipant(participantInfo:ParticipantInfo) 
	{
		linphone_conference_info_update_participant(cPtr, participantInfo.cPtr)
	}
}


/// Object defining parameters for a ``Conference``. 
/// Can be created by calling function
/// ``Core/createConferenceParams(conference:)``. 
public class ConferenceParams : LinphoneObject
{

	static public func getSwiftObject(cObject:OpaquePointer) -> ConferenceParams {
		let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
		if (result != nil) {
			return Unmanaged<ConferenceParams>.fromOpaque(result!).takeUnretainedValue()
		}
		let sObject = ConferenceParams(cPointer: cObject)
		belle_sip_object_data_set(UnsafeMutablePointer(cObject), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(sObject).toOpaque()), nil)
		return sObject
	}

	public var getCobject: OpaquePointer? {
		return cPtr
	}
	
	/// Sets the account for the conference. 
	/// - Parameter account: a pointer to the account.    
	/// - Warning: This account is used to configure some conference parameter field,
	/// therefore it is recommended to set it after selecting the capabilities. For
	/// instance capabilities will be used when defaulting the factory address. If
	/// audio or video one is enabled, then the default factory address will be the one
	/// associated to audio video conferencing, otherwise it will be the chat one.
	/// linphone_core_create_conference_with_params 
	
	/// Returns the account for the conference. 
	/// - Returns: a pointer to the account or nil if it is not set.    
	public var account: Account?
	{
	
		get
		{ 
						let cPointer = linphone_conference_params_get_account(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Account.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_conference_params_set_account(cPtr, newValue?.cPtr)
		}
	}
		
	/// Enable audio capabilities. 
	/// - Parameter enable: If true, audio will be enabled during conference 
	
	/// Check whether audio capabilities are enabled. 
	/// - Returns: true if the conference supports audio capabilities, false otherwise 
	public var audioEnabled: Bool
	{
	
		get
		{ 
						return linphone_conference_params_audio_enabled(cPtr) != 0
		}
		set
		{
			linphone_conference_params_enable_audio(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Enable chat capabilities. 
	/// - Parameter enable: If true, chat is enabled during conference 
	
	/// Check whether chat capabilities are enabled. 
	/// - Returns: true if the conference supports chat capabilities, false otherwise 
	public var chatEnabled: Bool
	{
	
		get
		{ 
						return linphone_conference_params_chat_enabled(cPtr) != 0
		}
		set
		{
			linphone_conference_params_enable_chat(cPtr, newValue==true ? 1:0)
		}
	}
		
	
	/// Get the chat parameters. 
	/// - Returns: the chat parameters if chat capabilities are on, nil otherwise    
	public var chatParams: ChatParams?
	{
	
						let cPointer = linphone_conference_params_get_chat_params(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = ChatParams.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	/// Set the conference factory address of the conference. 
	/// By default when creating a new conference, the factory address will come from
	/// the current proxy configuration. If nil then the conference will be local else
	/// it will be a client conference. 
	/// - Parameter address: the conference factory address.    
	
	/// Get the conference factory address of the conference that has been set. 
	/// - Returns: the factory address conference description.    
	public var conferenceFactoryAddress: Address?
	{
	
		get
		{ 
						let cPointer = linphone_conference_params_get_conference_factory_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_conference_params_set_conference_factory_address(cPtr, newValue?.cPtr)
		}
	}
		
	/// Set the description of the conference (utf8) 
	/// - Parameter description: the conference description.    
	
	/// Get conference description (utf8). 
	/// - Returns: the conference description.    
	public var descriptionUtf8: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_conference_params_get_description_utf8(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_conference_params_set_description_utf8(cPtr, newValue)
		}
	}
		
	/// Enables or disables group chat for the text capabilities of the conference
	/// associated with the given parameters. 
	/// - Parameter group: true to enable group chat, false to disable (resulting in
	/// one-on-one text capabilities of the conference) 
	
	/// Get the group chat status of the text capabilities of the conference associated
	/// with the given parameters. 
	/// - Returns: true if group chat is enabled, false if one-on-one 
	public var groupEnabled: Bool
	{
	
		get
		{ 
						return linphone_conference_params_group_enabled(cPtr) != 0
		}
		set
		{
			linphone_conference_params_enable_group(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Set the conference as hidden. 
	/// This means that the contact address will not have any conference releated
	/// attribute such as isfocus, the conference ID and the admin status. 
	/// - Parameter hidden: Boolean that states whether the conference is hidden or not 
	
	public var hidden: Bool?
	{
	
		willSet
		{
			linphone_conference_params_set_hidden(cPtr, newValue==true ? 1:0)
		}
	}
		
	
	/// Check whether audio capabilities are enabled. 
	/// - Returns: true if the conference supports audio capabilities, false otherwise 
	/// - Deprecated: 16/12/2021 Use ``audioEnabled()`` instead. 
	@available(*, deprecated)
	public var isAudioEnabled: Bool
	{
	
						return linphone_conference_params_is_audio_enabled(cPtr) != 0

	}
		
	
	/// Check whether chat capabilities are enabled. 
	/// - Returns: true if the conference supports chat capabilities, false otherwise 
	/// - Deprecated: 16/12/2021 Use ``chatEnabled()`` instead. 
	@available(*, deprecated)
	public var isChatEnabled: Bool
	{
	
						return linphone_conference_params_is_chat_enabled(cPtr) != 0

	}
		
	
	/// Get the value of the hidden flag. 
	/// - Returns: whether the conference is hidden or not 
	public var isHidden: Bool
	{
	
						return linphone_conference_params_is_hidden(cPtr) != 0

	}
		
	
	/// Returns whether local participant has to enter the conference. 
	/// - Returns: true if local participant is by default part of the conference,
	/// false otherwise 
	/// - Deprecated: 16/12/2021 Use ``localParticipantEnabled()`` instead. 
	@available(*, deprecated)
	public var isLocalParticipantEnabled: Bool
	{
	
						return linphone_conference_params_is_local_participant_enabled(cPtr) != 0

	}
		
	
	/// Returns whether conference can have only one participant. 
	/// - Returns: true if the conference can have only one participant, false
	/// otherwise 
	/// - Deprecated: 16/12/2021 Use ``oneParticipantConferenceEnabled()`` instead. 
	@available(*, deprecated)
	public var isOneParticipantConferenceEnabled: Bool
	{
	
						return linphone_conference_params_is_one_participant_conference_enabled(cPtr) != 0

	}
		
	
	/// Returns whether the given parameters are valid or not. 
	/// - Returns: true if the given parameters are valid, false otherwise 
	public var isValid: Bool
	{
	
						return linphone_conference_params_is_valid(cPtr) != 0

	}
		
	
	/// Check whether video capabilities are enabled. 
	/// - Returns: true if the conference supports video capabilities, false otherwise 
	/// - Deprecated: 16/12/2021 Use ``videoEnabled()`` instead. 
	@available(*, deprecated)
	public var isVideoEnabled: Bool
	{
	
						return linphone_conference_params_is_video_enabled(cPtr) != 0

	}
		
	/// Enable local participant to enter the conference. 
	/// The local participant is the one driving the local ``Core``. It uses the local
	/// sound devices. The default value is true. Setting to false is mostly helpful
	/// when using liblinphone on a server application. 
	/// - Parameter enable: true if local participant is automatically added to the
	/// conference, false otherwise 
	
	/// Returns whether local participant has to enter the conference. 
	/// - Returns: true if local participant is by default part of the conference,
	/// false otherwise 
	public var localParticipantEnabled: Bool
	{
	
		get
		{ 
						return linphone_conference_params_local_participant_enabled(cPtr) != 0
		}
		set
		{
			linphone_conference_params_enable_local_participant(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Enable conference with one participant. 
	/// - Parameter enable: true if conference can have only one participant, false
	/// otherwise 
	
	/// Returns whether conference can have only one participant. 
	/// - Returns: true if the conference can have only one participant, false
	/// otherwise 
	public var oneParticipantConferenceEnabled: Bool
	{
	
		get
		{ 
						return linphone_conference_params_one_participant_conference_enabled(cPtr) != 0
		}
		set
		{
			linphone_conference_params_enable_one_participant_conference(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Set the participant list type. 
	/// - Parameter type: Participant list type ``Conference.ParticipantListType``.
	/// This allows to restrict the access to the conference to a selected set of
	/// participants 
	
	/// Get the participant list type. 
	/// - Returns: participant list type ``Conference.ParticipantListType``. 
	public var participantListType: Conference.ParticipantListType
	{
	
		get
		{ 
						return Conference.ParticipantListType(rawValue: Int(linphone_conference_params_get_participant_list_type(cPtr).rawValue))!
		}
		set
		{
			linphone_conference_params_set_participant_list_type(cPtr, LinphoneConferenceParticipantListType(rawValue: CUnsignedInt(newValue.rawValue)))
		}
	}
		
	
	/// Returns the proxy configuration for the conference. 
	/// - Returns: a pointer to the proxy configuration or nil if it is not set.    
	/// - Deprecated: 11/01/2022 Use ``getAccount()`` instead. 
	@available(*, deprecated)
	public var proxyCfg: ProxyConfig?
	{
	
						let cPointer = linphone_conference_params_get_proxy_cfg(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = ProxyConfig.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	/// Set the desired security level of the conference. 
	/// - Parameter securityLevel: The desired security level of the conference. 
	
	/// Retrieve the desired security level of the conference. 
	/// - Returns: The desired security level of the conference. 
	public var securityLevel: Conference.SecurityLevel
	{
	
		get
		{ 
						return Conference.SecurityLevel(rawValue: Int(linphone_conference_params_get_security_level(cPtr).rawValue))!
		}
		set
		{
			linphone_conference_params_set_security_level(cPtr, LinphoneConferenceSecurityLevel(rawValue: CUnsignedInt(newValue.rawValue)))
		}
	}
		
	/// Set the conference subject. 
	/// - Parameter subject: conference subject    
	
	/// Get the conference subject. 
	/// - Returns: conference subject.    
	public var subject: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_conference_params_get_subject(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_conference_params_set_subject(cPtr, newValue)
		}
	}
		
	/// Set the conference subject as an UTF8 string. 
	/// - Parameter subject: conference subject    
	
	/// Get the conference subject as an UTF-8 string. 
	/// - Returns: conference subject.    
	public var subjectUtf8: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_conference_params_get_subject_utf8(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_conference_params_set_subject_utf8(cPtr, newValue)
		}
	}
		
	/// Enable video capabilities. 
	/// - Parameter enable: If true, video will be enabled during conference 
	
	/// Check whether video capabilities are enabled. 
	/// - Returns: true if the conference supports video capabilities, false otherwise 
	public var videoEnabled: Bool
	{
	
		get
		{ 
						return linphone_conference_params_video_enabled(cPtr) != 0
		}
		set
		{
			linphone_conference_params_enable_video(cPtr, newValue==true ? 1:0)
		}
	}
		
	
	
	/// Clone a ``ConferenceParams``. 
	/// - Returns: An allocated ``ConferenceParams`` with the same parameters than
	/// params    
	public func clone() -> ConferenceParams?
	{
		let cPointer = linphone_conference_params_clone(cPtr)
		if (cPointer == nil) {
			return nil
		}
		let result = ConferenceParams.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
}


/// Creates and manages conferences on a conferenceing service, and send conference
/// invitations to notify participants. 
public class ConferenceScheduler : LinphoneObject
{
	var delegateManagers : [ConferenceSchedulerDelegateManager] = []

	static public func getSwiftObject(cObject:OpaquePointer) -> ConferenceScheduler {
		let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
		if (result != nil) {
			return Unmanaged<ConferenceScheduler>.fromOpaque(result!).takeUnretainedValue()
		}
		let sObject = ConferenceScheduler(cPointer: cObject)
		belle_sip_object_data_set(UnsafeMutablePointer(cObject), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(sObject).toOpaque()), nil)
		return sObject
	}

	public var getCobject: OpaquePointer? {
		return cPtr
	}

	
	///``Kind`` is used to know which conference scheduler type the core is using 
	public enum Kind:Int
	{
		
		/// SIP conference scheduler type. 
		case SIP = 0
		/// Database conference scheduler type. 
		case DB = 1
		/// CCMP conference scheduler type. 
		case CCMP = 2
	}

	
	///Describes conference scheduler possible states. 
	public enum State:Int
	{
		
		/// Default state of a freshly created ``ConferenceScheduler``. 
		case Idle = 0
		/// An error has happened during conference creation. 
		case Error = 1
		/// Conference creation is in progress. 
		case AllocationPending = 2
		/// Confererence has been created. 
		case Ready = 3
		/// Conference has been updated. 
		case Updating = 4
	}
	
	
	
	public func addDelegate(delegate: ConferenceSchedulerDelegate)
	{
		let manager = ConferenceSchedulerDelegateManager()
		manager.delegate = delegate
		delegateManagers.append(manager)
		linphone_conference_scheduler_add_callbacks(cPtr, manager.cPtr)
		belle_sip_object_unref(UnsafeMutableRawPointer(manager.cPtr))
		belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(self).toOpaque()), nil)
	}	
	
	
	public func removeDelegate(delegate: ConferenceSchedulerDelegate)
	{
		if let index = delegateManagers.firstIndex(where: { $0.delegate === delegate as AnyObject}) {
			linphone_conference_scheduler_remove_callbacks(cPtr, delegateManagers[index].cPtr)
			delegateManagers.remove(at: index)
			belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(self).toOpaque()), nil)
		}
	}	
	/// Set the ``Account`` to use for the conference scheduler. 
	/// - Parameter account: The ``Account`` to use, or nil if none has been selected.
	/// The ``ConferenceScheduler`` keeps a reference to it and removes the previous
	/// one, if any.    
	
	/// Get the ``Account`` that is used for the conference scheduler. 
	/// - Returns: The selected ``Account`` for the call, or nil if none has been
	/// selected.    
	public var account: Account?
	{
	
		get
		{ 
						let cPointer = linphone_conference_scheduler_get_account(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Account.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_conference_scheduler_set_account(cPtr, newValue?.cPtr)
		}
	}
		
	
	/// Gets the ``Core`` from a ``ConferenceScheduler`` object. 
	/// - Returns: the ``Core`` object.    
	public var core: Core?
	{
	
						let cPointer = linphone_conference_scheduler_get_core(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Core.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Gets the current ``ConferenceSchedulerDelegate``. 
	/// This is meant only to be called from a callback to be able to get the user_data
	/// associated with the ``ConferenceSchedulerDelegate`` that is calling the
	/// callback. 
	/// - Returns: The ``ConferenceSchedulerDelegate`` that has called the last
	/// callback.    
	public var currentDelegate: ConferenceSchedulerDelegate?
	{
	
			let cObject = linphone_conference_scheduler_get_current_callbacks(cPtr)
			let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
			if (result != nil) {
				return Unmanaged<ConferenceSchedulerDelegateManager>.fromOpaque(result!).takeUnretainedValue().delegate
				}
			return nil
			
	}
		
	/// Sets the ``ConferenceInfo`` to use to create/update the conference, which will
	/// be done right away. 
	/// - Parameter conferenceInfo: the ``ConferenceInfo`` object to use to start
	/// creating/updating the client conference.    
	
	/// Returns the ``ConferenceInfo`` currently set in this scheduler. 
	/// - Returns: the currently configured ``ConferenceInfo`` or nil if none is set.  
	///  
	public var info: ConferenceInfo?
	{
	
		get
		{ 
						let cPointer = linphone_conference_scheduler_get_info(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = ConferenceInfo.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_conference_scheduler_set_info(cPtr, newValue?.cPtr)
		}
	}
		
	
	
	/// Cancel the conference linked to the ``ConferenceInfo`` provided as argument. 
	/// - Parameter conferenceInfo: the ``ConferenceInfo`` object to linked to the
	/// conference to cancel.    
	public func cancelConference(conferenceInfo:ConferenceInfo?) 
	{
		linphone_conference_scheduler_cancel_conference(cPtr, conferenceInfo?.cPtr)
	}
	
	
	
	/// Sends an invitation to the scheduled conference to each participant by chat,
	/// using given chat rooms params to use/create the chat room in which to send it. 
	/// - Parameter chatRoomParams: the ``ChatRoomParams`` object to use to use/create
	/// the ``ChatRoom`` that will be used to send the invite.    
	/// - Deprecated: 28/08/2024 Use ``sendInvitations(conferenceParams:)`` instead. 
	@available(*, deprecated)
	public func sendInvitations(chatRoomParams:ChatRoomParams) 
	{
		linphone_conference_scheduler_send_invitations(cPtr, chatRoomParams.cPtr)
	}
	
	
	
	/// Sends an invitation to the scheduled conference to each participant by chat,
	/// using given conference params to use/create the chat room in which to send it. 
	/// - Parameter conferenceParams: the ``ConferenceParams`` object to use to
	/// use/create the ``ChatRoom`` that will be used to send the invite.    
	public func sendInvitations(conferenceParams:ConferenceParams) 
	{
		linphone_conference_scheduler_send_invitations_2(cPtr, conferenceParams.cPtr)
	}
}


/// This object is used to manipulate a configuration file. 
/// The format of the configuration file is a .ini like format:
/// -sections are defined in []
/// -each section contains a sequence of key=value pairs
/// -each line starting by a # is a comment
/// Various types can be used: strings and lists of strings, integers, floats,
/// booleans (written as 0 or 1) and range of integers.
/// Usually a ``Core`` is initialized using two ``Config``, one default (where
/// configuration changes through API calls will be saved) and one named 'factory'
/// which is read-only and overwrites any setting that may exists in the default
/// one.
/// It is also possible to use only one (either default or factory) or even none. 
public class Config : LinphoneObject
{

	static public func getSwiftObject(cObject:OpaquePointer) -> Config {
		let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
		if (result != nil) {
			return Unmanaged<Config>.fromOpaque(result!).takeUnretainedValue()
		}
		let sObject = Config(cPointer: cObject)
		belle_sip_object_data_set(UnsafeMutablePointer(cObject), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(sObject).toOpaque()), nil)
		return sObject
	}

	public var getCobject: OpaquePointer? {
		return cPtr
	}
	
	
	
	/// Instantiates a ``Config`` object from a user config file name, group id and a
	/// factory config file. 
	/// The "group id" is the string that identify the "App group" capability of the
	/// iOS application. App group gives access to a shared file system where all the
	/// configuration files for shared core are stored. Both iOS application and iOS
	/// app extension that need shared core must activate the "App group" capability
	/// with the SAME "group id" in the project settings. The caller of this
	/// constructor owns a reference. linphone_config_unref must be called when this
	/// object is no longer needed.
	/// - Parameter appGroupId: used to compute the path of the config file in the file
	/// system shared by the shared Cores    
	/// - Parameter configFilename: the filename of the user config file to read to
	/// fill the instantiated ``Config``    
	/// - Parameter factoryConfigFilename: the filename of the factory config file to
	/// read to fill the instantiated ``Config``    
	/// - See also: linphone_config_new 
	/// - Returns: a ``Config`` object   
	/// The user config file is read first to fill the ``Config`` and then the factory
	/// config file is read. Therefore the configuration parameters defined in the user
	/// config file will be overwritten by the parameters defined in the factory config
	/// file. 
	static public func newForSharedCore(appGroupId:String, configFilename:String?, factoryConfigFilename:String?) -> Config?
	{
		let cPointer = linphone_config_new_for_shared_core(appGroupId, configFilename, factoryConfigFilename)
		if (cPointer == nil) {
			return nil
		}
		let result = Config.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Instantiates a ``Config`` object from a user provided buffer. 
	/// The caller of this constructor owns a reference. linphone_config_unref must be
	/// called when this object is no longer needed.
	/// - Parameter buffer: the buffer from which the ``Config`` will be retrieved. We
	/// expect the buffer to be null-terminated.    
	/// - See also: ``newWithFactory(configFilename:factoryConfigFilename:)`` 
	/// - See also: linphone_config_new 
	/// - Returns: a ``Config`` object    
	static public func newFromBuffer(buffer:String) -> Config?
	{
		let cPointer = linphone_config_new_from_buffer(buffer)
		if (cPointer == nil) {
			return nil
		}
		let result = Config.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Instantiates a ``Config`` object from a user config file and a factory config
	/// file. 
	/// The caller of this constructor owns a reference. linphone_config_unref must be
	/// called when this object is no longer needed.
	/// - Parameter configFilename: the filename of the user config file to read to
	/// fill the instantiated ``Config``    
	/// - Parameter factoryConfigFilename: the filename of the factory config file to
	/// read to fill the instantiated ``Config``    
	/// - See also: linphone_config_new 
	/// - Returns: a ``Config`` object   
	/// The user config file is read first to fill the ``Config`` and then the factory
	/// config file is read. Therefore the configuration parameters defined in the user
	/// config file will be overwritten by the parameters defined in the factory config
	/// file. 
	static public func newWithFactory(configFilename:String?, factoryConfigFilename:String?) -> Config?
	{
		let cPointer = linphone_config_new_with_factory(configFilename, factoryConfigFilename)
		if (cPointer == nil) {
			return nil
		}
		let result = Config.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	/// Indicates whether the ``Config`` object is readonly, in other words it has no
	/// file backend or file is opened without write permission. 
	/// - Returns: a boolean. 
	public var isReadonly: Bool
	{
	
						return linphone_config_is_readonly(cPtr) != 0

	}
		
	
	/// Returns the list of sections' names in the ``Config``. 
	/// - Returns: A list of strings.      
	public var sectionsNamesList: [String]
	{
	
						var swiftList = [String]()
			let cList = linphone_config_get_sections_names_list(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				swiftList.append(String(cString: unsafeBitCast(listTemp!.pointee.data, to: UnsafePointer<CChar>.self)))
				listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp!.pointee.next)
			}
				bctbx_list_free(cList)
			return swiftList

	}
		
	
	
	/// Removes entries for key,value in a section. 
	/// - Parameter section: the section for which to clean the key entry    
	/// - Parameter key: the key to clean    
	public func cleanEntry(section:String, key:String) 
	{
		linphone_config_clean_entry(cPtr, section, key)
	}
	
	
	
	/// Removes every pair of key,value in a section and remove the section. 
	/// - Parameter section: the section to clean    
	public func cleanSection(section:String) 
	{
		linphone_config_clean_section(cPtr, section)
	}
	
	
	
	/// Removes a suite of sections whose name is derived from section argument,
	/// suffixed with _0, _1, _2 etc. 
	/// This is a common representation for configuration objects that can have
	/// multiple instances. 
	public func cleanSectionSuite(section:String) 
	{
		linphone_config_clean_section_suite(cPtr, section)
	}
	
	
	
	/// Dumps the ``Config`` as INI into a buffer. 
	/// - Returns: The buffer that contains the config dump       
	public func dump() -> String
	{
		let cstr = linphone_config_dump(cPtr)
		let result = charArrayToString(charPointer: cstr)
		if (cstr != nil) {
			bctbx_free(cstr)
		}
		return result
	}
	
	
	
	/// Dumps the ``Config`` as XML into a buffer. 
	/// - Returns: The buffer that contains the XML dump       
	public func dumpAsXml() -> String
	{
		let cstr = linphone_config_dump_as_xml(cPtr)
		let result = charArrayToString(charPointer: cstr)
		if (cstr != nil) {
			bctbx_free(cstr)
		}
		return result
	}
	
	
	
	/// Retrieves a configuration item as a boolean, given its section, key, and
	/// default value. 
	/// The default boolean value is returned if the config item isn't found. 
	/// - Parameter section: The section from which to retrieve a configuration item    
	/// - Parameter key: The name of the configuration item to retrieve    
	/// - Parameter defaultValue: The default value to return if not found 
	/// - Returns: the found value or default_value if not found. 
	public func getBool(section:String, key:String, defaultValue:Bool) -> Bool
	{
		return linphone_config_get_bool(cPtr, section, key, defaultValue==true ? 1:0) != 0
	}
	
	
	
	/// Retrieves a default configuration item as a float, given its section, key, and
	/// default value. 
	/// The default float value is returned if the config item isn't found. 
	/// - Parameter section: The section from which to retrieve the default value    
	/// - Parameter key: The name of the configuration item to retrieve    
	/// - Parameter defaultValue: The default value to return if not found 
	/// - Returns: the found default value or default_value if not found. 
	public func getDefaultFloat(section:String, key:String, defaultValue:Float) -> Float
	{
		return linphone_config_get_default_float(cPtr, section, key, defaultValue)
	}
	
	
	
	/// Retrieves a default configuration item as an integer, given its section, key,
	/// and default value. 
	/// The default integer value is returned if the config item isn't found. 
	/// - Parameter section: The section from which to retrieve the default value    
	/// - Parameter key: The name of the configuration item to retrieve    
	/// - Parameter defaultValue: The default value to return if not found 
	/// - Returns: the found default value or default_value if not found. 
	public func getDefaultInt(section:String, key:String, defaultValue:Int) -> Int
	{
		return Int(linphone_config_get_default_int(cPtr, section, key, CInt(defaultValue)))
	}
	
	
	
	/// Retrieves a default configuration item as a 64 bit integer, given its section,
	/// key, and default value. 
	/// The default integer value is returned if the config item isn't found. 
	/// - Parameter section: The section from which to retrieve the default value    
	/// - Parameter key: The name of the configuration item to retrieve    
	/// - Parameter defaultValue: The default value to return if not found 
	/// - Returns: the found default value or default_value if not found. 
	public func getDefaultInt64(section:String, key:String, defaultValue:Int64) -> Int64
	{
		return linphone_config_get_default_int64(cPtr, section, key, defaultValue)
	}
	
	
	
	/// Retrieves a configuration item as a float, given its section, key, and default
	/// value. 
	/// The default float value is returned if the config item isn't found. 
	/// - Parameter section: The section from which to retrieve a configuration item    
	/// - Parameter key: The name of the configuration item to retrieve    
	/// - Parameter defaultValue: The default value to return if not found 
	/// - Returns: the found value or default_value if not found. 
	public func getFloat(section:String, key:String, defaultValue:Float) -> Float
	{
		return linphone_config_get_float(cPtr, section, key, defaultValue)
	}
	
	
	
	/// Retrieves a configuration item as an integer, given its section, key, and
	/// default value. 
	/// The default integer value is returned if the config item isn't found. 
	/// - Parameter section: The section from which to retrieve a configuration item    
	/// - Parameter key: The name of the configuration item to retrieve    
	/// - Parameter defaultValue: The default value to return if not found 
	/// - Returns: the found value or default_value if not found. 
	public func getInt(section:String, key:String, defaultValue:Int) -> Int
	{
		return Int(linphone_config_get_int(cPtr, section, key, CInt(defaultValue)))
	}
	
	
	
	/// Retrieves a configuration item as a 64 bit integer, given its section, key, and
	/// default value. 
	/// The default integer value is returned if the config item isn't found. 
	/// - Parameter section: The section from which to retrieve a configuration item    
	/// - Parameter key: The name of the configuration item to retrieve    
	/// - Parameter defaultValue: The default value to return if not found 
	/// - Returns: the found value or default_value if not found. 
	public func getInt64(section:String, key:String, defaultValue:Int64) -> Int64
	{
		return linphone_config_get_int64(cPtr, section, key, defaultValue)
	}
	
	
	
	/// Returns the list of keys' names for a section in the ``Config``. 
	/// - Parameter section: The section name    
	/// - Returns: A list of strings.      
	public func getKeysNamesList(section:String) -> [String]
	{
		var swiftList = [String]()
		let cList = linphone_config_get_keys_names_list(cPtr, section)
		var listTemp = cList
		while (listTemp != nil) {
			swiftList.append(String(cString: unsafeBitCast(listTemp!.pointee.data, to: UnsafePointer<CChar>.self)))
			listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp!.pointee.next)
		}
			bctbx_list_free(cList)
		return swiftList
	}
	
	
	
	/// Retrieves the overwrite flag for a config item. 
	/// - Parameter section: The section from which to retrieve the overwrite flag    
	/// - Parameter key: The name of the configuration item to retrieve the overwrite
	/// flag from.    
	/// - Returns: true if overwrite flag is set, false otherwise. 
	public func getOverwriteFlagForEntry(section:String, key:String) -> Bool
	{
		return linphone_config_get_overwrite_flag_for_entry(cPtr, section, key) != 0
	}
	
	
	
	/// Retrieves the overwrite flag for a config section. 
	/// - Parameter section: The section from which to retrieve the overwrite flag    
	/// - Returns: true if overwrite flag is set, false otherwise. 
	public func getOverwriteFlagForSection(section:String) -> Bool
	{
		return linphone_config_get_overwrite_flag_for_section(cPtr, section) != 0
	}
	
	
	
	/// Retrieves a configuration item as a range, given its section, key, and default
	/// min and max values. 
	/// - Parameter section: The section from which to retrieve a configuration item    
	/// - Parameter key: The name of the configuration item to retrieve    
	/// - Parameter min: The min value found or default_min    
	/// - Parameter max: The max value found or default_max    
	/// - Parameter defaultMin: the default min value to return if not found 
	/// - Parameter defaultMax: the default max value to return if not found. 
	/// - Returns: true if the value is successfully parsed as a range, false
	/// otherwise. If false is returned, min and max are filled respectively with
	/// default_min and default_max values. 
	public func getRange(section:String, key:String, min:UnsafeMutablePointer<Int32>, max:UnsafeMutablePointer<Int32>, defaultMin:Int, defaultMax:Int) -> Bool
	{
		return linphone_config_get_range(cPtr, section, key, min, max, CInt(defaultMin), CInt(defaultMax)) != 0
	}
	
	
	
	/// Retrieves the skip flag for a config item. 
	/// - Parameter section: The section from which to retrieve the skip flag    
	/// - Parameter key: The name of the configuration item to retrieve the skip flag
	/// from 
	/// - Returns: true if skip flag is set, false otherwise. 
	public func getSkipFlagForEntry(section:String, key:String) -> Bool
	{
		return linphone_config_get_skip_flag_for_entry(cPtr, section, key) != 0
	}
	
	
	
	/// Retrieves the skip flag for a config section. 
	/// - Parameter section: The section from which to retrieve the skip flag    
	/// - Returns: true if skip flag is set, false otherwise. 
	public func getSkipFlagForSection(section:String) -> Bool
	{
		return linphone_config_get_skip_flag_for_section(cPtr, section) != 0
	}
	
	
	
	/// Retrieves a configuration item as a list of strings, given its section, key,
	/// and default value. 
	/// The default value is returned if the config item is not found. 
	/// - Parameter section: The section from which to retrieve a configuration item    
	/// - Parameter key: The name of the configuration item to retrieve    
	/// - Parameter defaultList: The list to return when the key doesn't exist.      
	/// - Returns: A list of strings.      
	public func getStringList(section:String, key:String, defaultList:[String]?) -> [String]
	{
		var swiftList = [String]()
		let cList = linphone_config_get_string_list(cPtr, section, key, StringArrayToBctbxList(list:defaultList))
		var listTemp = cList
		while (listTemp != nil) {
			swiftList.append(String(cString: unsafeBitCast(listTemp!.pointee.data, to: UnsafePointer<CChar>.self)))
			listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp!.pointee.next)
		}
			bctbx_list_free(cList)
		return swiftList
	}
	
	
	
	/// Returns if a given section with a given key is present in the configuration. 
	/// - Parameter section: to check if the given entry exists    
	/// - Parameter key: to check if it exists    
	/// - Returns: 1 if it exists, 0 otherwise 
	public func hasEntry(section:String, key:String) -> Int
	{
		return Int(linphone_config_has_entry(cPtr, section, key))
	}
	
	
	
	/// Returns if a given section is present in the configuration. 
	/// - Parameter section: the section to check if exists    
	/// - Returns: 1 if it exists, 0 otherwise 
	public func hasSection(section:String) -> Int
	{
		return Int(linphone_config_has_section(cPtr, section))
	}
	
	
	
	/// Reads a xml config file and fill the ``Config`` with the read config dynamic
	/// values. 
	/// - Parameter filename: The filename of the config file to read to fill the
	/// ``Config``    
	public func loadFromXmlFile(filename:String) -> String
	{
		let cstr = linphone_config_load_from_xml_file(cPtr, filename)
		let result = charArrayToString(charPointer: cstr)
		return result
	}
	
	
	
	/// Reads a xml config string and fill the ``Config`` with the read config dynamic
	/// values. 
	/// - Parameter buffer: The string of the config file to fill the ``Config``    
	/// - Returns: 0 in case of success 
	public func loadFromXmlString(buffer:String) throws 
	{
		let exception_result = linphone_config_load_from_xml_string(cPtr, buffer)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "loadFromXmlString returned value \(exception_result)")
		}
	}
	
	
	
	/// Reads a user config file and fill the ``Config`` with the read config values. 
	/// - Parameter filename: The filename of the config file to read to fill the
	/// ``Config``    
	public func readFile(filename:String) throws 
	{
		let exception_result = linphone_config_read_file(cPtr, filename)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "readFile returned value \(exception_result)")
		}
	}
	
	
	
	/// Check if given file name exists relatively to the current location. 
	/// - Parameter filename: The file name to check if exists    
	/// - Returns: true if file exists relative to the to the current location 
	public func relativeFileExists(filename:String) -> Bool
	{
		return linphone_config_relative_file_exists(cPtr, filename) != 0
	}
	
	
	
	/// Reload the config from the file. 
	public func reload() 
	{
		linphone_config_reload(cPtr)
	}
	
	
	
	/// Sets a boolean config item. 
	/// - Parameter section: The section from which to retrieve a configuration item    
	/// - Parameter key: The name of the configuration item to retrieve    
	/// - Parameter value: the value to set 
	public func setBool(section:String, key:String, value:Bool) 
	{
		linphone_config_set_bool(cPtr, section, key, value==true ? 1:0)
	}
	
	
	
	/// Sets a float config item. 
	/// - Parameter section: The section from which to retrieve a configuration item    
	/// - Parameter key: The name of the configuration item to retrieve    
	/// - Parameter value: the value to set 
	public func setFloat(section:String, key:String, value:Float) 
	{
		linphone_config_set_float(cPtr, section, key, value)
	}
	
	
	
	/// Sets an integer config item. 
	/// - Parameter section: The section from which to retrieve a configuration item    
	/// - Parameter key: The name of the configuration item to retrieve    
	/// - Parameter value: the value to set 
	public func setInt(section:String, key:String, value:Int) 
	{
		linphone_config_set_int(cPtr, section, key, CInt(value))
	}
	
	
	
	/// Sets a 64 bits integer config item. 
	/// - Parameter section: The section from which to retrieve a configuration item    
	/// - Parameter key: The name of the configuration item to retrieve    
	/// - Parameter value: the value to set 
	public func setInt64(section:String, key:String, value:Int64) 
	{
		linphone_config_set_int64(cPtr, section, key, value)
	}
	
	
	
	/// Sets an integer config item, but store it as hexadecimal. 
	/// - Parameter section: The section from which to retrieve a configuration item    
	/// - Parameter key: The name of the configuration item to retrieve    
	/// - Parameter value: the value to set 
	public func setIntHex(section:String, key:String, value:Int) 
	{
		linphone_config_set_int_hex(cPtr, section, key, CInt(value))
	}
	
	
	
	/// Sets the overwrite flag for a config item (used when dumping config as xml) 
	/// - Parameter section: The section from which to set the overwrite flag    
	/// - Parameter key: The name of the configuration item to set the overwrite flag
	/// from    
	/// - Parameter value: The overwrite flag value to set 
	public func setOverwriteFlagForEntry(section:String, key:String, value:Bool) 
	{
		linphone_config_set_overwrite_flag_for_entry(cPtr, section, key, value==true ? 1:0)
	}
	
	
	
	/// Sets the overwrite flag for a config section (used when dumping config as xml) 
	/// - Parameter section: The section from which to set the overwrite flag    
	/// - Parameter value: The overwrite flag value to set 
	public func setOverwriteFlagForSection(section:String, value:Bool) 
	{
		linphone_config_set_overwrite_flag_for_section(cPtr, section, value==true ? 1:0)
	}
	
	
	
	/// Sets a range config item. 
	/// - Parameter section: The section from which to retrieve a configuration item    
	/// - Parameter key: The name of the configuration item to retrieve    
	/// - Parameter minValue: the min value to set 
	/// - Parameter maxValue: the max value to set 
	public func setRange(section:String, key:String, minValue:Int, maxValue:Int) 
	{
		linphone_config_set_range(cPtr, section, key, CInt(minValue), CInt(maxValue))
	}
	
	
	
	/// Sets the skip flag for a config item (used when dumping config as xml) 
	/// - Parameter section: The section from which to set the skip flag    
	/// - Parameter key: The name of the configuration item to set the skip flag from  
	///  
	/// - Parameter value: The skip flag value to set 
	public func setSkipFlagForEntry(section:String, key:String, value:Bool) 
	{
		linphone_config_set_skip_flag_for_entry(cPtr, section, key, value==true ? 1:0)
	}
	
	
	
	/// Sets the skip flag for a config section (used when dumping config as xml) 
	/// - Parameter section: The section from which to set the skip flag    
	/// - Parameter value: The skip flag value to set 
	public func setSkipFlagForSection(section:String, value:Bool) 
	{
		linphone_config_set_skip_flag_for_section(cPtr, section, value==true ? 1:0)
	}
	
	
	
	/// Sets a string config item. 
	/// - Parameter section: The section from which to retrieve a configuration item    
	/// - Parameter key: The name of the configuration item to retrieve    
	/// - Parameter value: The value to set    
	public func setString(section:String, key:String, value:String?) 
	{
		linphone_config_set_string(cPtr, section, key, value)
	}
	
	
	
	/// Sets a string list config item. 
	/// - Parameter section: The name of the section to put the configuration item into
	///    
	/// - Parameter key: The name of the configuration item to set    
	/// - Parameter value: The value to set.      
	public func setStringList(section:String, key:String, value:[String]?) 
	{
		linphone_config_set_string_list(cPtr, section, key, StringArrayToBctbxList(list:value))
	}
	
	
	
	/// Writes the config file to disk. 
	/// - Returns: 0 if successful, -1 otherwise 
	public func sync() throws 
	{
		let exception_result = linphone_config_sync(cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "sync returned value \(exception_result)")
		}
	}
	
	
	
	/// Write a string in a file placed relatively with the Linphone configuration
	/// file. 
	/// - Parameter filename: Name of the file where to write data. The name is
	/// relative to the place of the config file    
	/// - Parameter data: String to write    
	public func writeRelativeFile(filename:String, data:String) 
	{
		linphone_config_write_relative_file(cPtr, filename, data)
	}
}


/// This object holds data that can be embedded in a signaling message or IM
/// message. 
/// Use ``Core/createContent()`` to create it, and then you should set at least
/// it's type and subtype and fill the buffer with your data.
/// A ``Content`` can be multipart (contain other contents), have file information
/// (name, path, size), be encrypted, have custom headers, etc... 
public class Content : LinphoneObject
{

	static public func getSwiftObject(cObject:OpaquePointer) -> Content {
		let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
		if (result != nil) {
			return Unmanaged<Content>.fromOpaque(result!).takeUnretainedValue()
		}
		let sObject = Content(cPointer: cObject)
		belle_sip_object_data_set(UnsafeMutablePointer(cObject), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(sObject).toOpaque()), nil)
		return sObject
	}

	public var getCobject: OpaquePointer? {
		return cPtr
	}
	
	
	/// Get the content data buffer, usually a string. 
	/// - Returns: The content data buffer.    
	public var buffer: UnsafePointer<UInt8>
	{
	
						return linphone_content_get_buffer(cPtr)

	}
		
	
	/// Returns the creation timestamp if this content is a FileContent (received or
	/// sent by chat). 
	/// - Returns: The timestamp at which this content was created if available, -1
	/// otherwise. 
	public var creationTimestamp: time_t
	{
	
						return linphone_content_get_creation_timestamp(cPtr)

	}
		
	/// Set the disposition of the Content, for example "recipient-list". 
	/// - Parameter disposition: The disposition of the Content.    
	
	/// Get the disposition of the Content, for example "recipient-list". 
	/// - Returns: The disposition of the Content.    
	public var disposition: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_content_get_disposition(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_content_set_disposition(cPtr, newValue)
		}
	}
		
	/// Set the encoding of the data buffer, for example "gzip". 
	/// - Parameter encoding: The encoding of the data buffer.    
	
	/// Get the encoding of the data buffer, for example "gzip". 
	/// - Returns: The encoding of the data buffer.    
	public var encoding: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_content_get_encoding(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_content_set_encoding(cPtr, newValue)
		}
	}
		
	/// Sets the file duration in seconds (useful for media files for example). 
	/// - Parameter duration: the duration of the file, in milliseconds. 
	
	/// Gets the file duration in seconds, if information is available. 
	/// - Returns: The duration of the file in milliseconds or -1 if information isn't
	/// available. 
	public var fileDuration: Int
	{
	
		get
		{ 
						return Int(linphone_content_get_file_duration(cPtr))
		}
		set
		{
			linphone_content_set_file_duration(cPtr, CInt(newValue))
		}
	}
		
	/// Set the file transfer filepath for this content (replace
	/// linphone_chat_message_set_file_transfer_filepath). 
	/// - Parameter filePath: the file transfer filepath.    
	
	/// Get the file transfer filepath set for this content (replace
	/// linphone_chat_message_get_file_transfer_filepath). 
	/// - Returns: The file path set for this content if it has been set, nil
	/// otherwise.    
	public var filePath: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_content_get_file_path(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_content_set_file_path(cPtr, newValue)
		}
	}
		
	
	/// Get the file size if content is either a FileContent or a FileTransferContent. 
	/// - Returns: The represented file size. 
	public var fileSize: Int
	{
	
						return Int(linphone_content_get_file_size(cPtr))

	}
		
	
	/// Tells whether or not this content contains a file. 
	/// - Returns: true if this content contains a file, false otherwise. 
	public var isFile: Bool
	{
	
						return linphone_content_is_file(cPtr) != 0

	}
		
	
	/// Tells whether or not this content contains an encrypted file. 
	/// - Returns: True is this content contains a file and this file is encrypted,
	/// false otherwise. 
	public var isFileEncrypted: Bool
	{
	
						return linphone_content_is_file_encrypted(cPtr) != 0

	}
		
	
	/// Tells whether or not this content is a file transfer. 
	/// - Returns: true if this content is a file transfer, false otherwise. 
	public var isFileTransfer: Bool
	{
	
						return linphone_content_is_file_transfer(cPtr) != 0

	}
		
	
	/// Tells whether or not this content contains an icalendar by checking it's
	/// content type. 
	/// - Returns: true if this content type is 'text/calendar;conference-event=yes',
	/// false otherwise. 
	public var isIcalendar: Bool
	{
	
						return linphone_content_is_icalendar(cPtr) != 0

	}
		
	
	/// Tell whether a content is a multipart content. 
	/// - Returns: A boolean value telling whether the content is multipart or not. 
	public var isMultipart: Bool
	{
	
						return linphone_content_is_multipart(cPtr) != 0

	}
		
	
	/// Tells whether or not this content contains text. 
	/// - Returns: true if this content contains plain text, false otherwise. 
	public var isText: Bool
	{
	
						return linphone_content_is_text(cPtr) != 0

	}
		
	
	/// Tells whether or not this content contains a voice recording by checking it's
	/// content type. 
	/// - Returns: true if this content type is 'audio/wav;voice-recording=yes', false
	/// otherwise. 
	public var isVoiceRecording: Bool
	{
	
						return linphone_content_is_voice_recording(cPtr) != 0

	}
		
	
	/// Get the key associated with a RCS file transfer message if encrypted. 
	/// - Returns: The key to encrypt/decrypt the file associated to this content.    
	public var key: String?
	{
	
			
			let cPointer = linphone_content_get_key(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Get the size of key associated with a RCS file transfer message if encrypted. 
	/// - Returns: The key size in bytes 
	public var keySize: Int
	{
	
						return Int(linphone_content_get_key_size(cPtr))

	}
		
	/// Set the name associated with a RCS file transfer message. 
	/// It is used to store the original filename of the file to be downloaded from
	/// server. 
	/// - Parameter name: The name of the content.    
	
	/// Get the name associated with a RCS file transfer message. 
	/// It is used to store the original filename of the file to be downloaded from
	/// server. 
	/// - Returns: The name of the content.    
	public var name: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_content_get_name(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_content_set_name(cPtr, newValue)
		}
	}
		
	
	/// Get all the parts from a multipart content. 
	/// - Returns: A   object holding the part if found, nil otherwise.       
	public var parts: [Content]
	{
	
						var swiftList = [Content]()
			let cList = linphone_content_get_parts(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(Content.getSwiftObject(cObject: data))
				listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
				bctbx_list_free_with_data(cList, belle_sip_object_unref)
			return swiftList

	}
		
	
	/// Generates a temporary plain copy of the file and returns its paths The caller
	/// is responsible to then delete this temporary copy and the returned string. 
	/// - Returns: The file path set for this content if it has been set, nil
	/// otherwise.    
	/// - Deprecated: 2022-01-07. Use ``exportPlainFile()`` instead. 
	@available(*, deprecated)
	public var plainFilePath: String
	{
	
			
			let cPointer = linphone_content_get_plain_file_path(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			if (cPointer != nil) {
				bctbx_free(cPointer)
			}
			return result

	}
		
	
	/// Returns the chat message id for which this content is related to, if any. 
	/// - Returns: The chat message ID if this content is related to a chat message,
	/// nil otherwise.    
	public var relatedChatMessageId: String?
	{
	
			
			let cPointer = linphone_content_get_related_chat_message_id(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	/// Set the content data size, excluding null character despite null character is
	/// always set for convenience. 
	/// - Parameter size: The content data buffer size. 
	
	/// Get the content data buffer size, excluding null character despite null
	/// character is always set for convenience. 
	/// - Returns: The content data buffer size. 
	public var size: Int
	{
	
		get
		{ 
						return Int(linphone_content_get_size(cPtr))
		}
		set
		{
			linphone_content_set_size(cPtr, (newValue))
		}
	}
		
	/// Set the string content data buffer. 
	/// - Parameter buffer: The string content data buffer in UTF8.    
	/// - Deprecated: 2020-07-01. Use ``setUtf8Text(buffer:)`` instead. 
	
	/// Get the string content data buffer. 
	/// - Returns: The string content data buffer.    
	/// - Deprecated: 2020-07-01. Use ``getUtf8Text()`` instead. 
	public var stringBuffer: String
	{
	@available(*, deprecated)
		get
		{ 
			
			let cPointer = linphone_content_get_string_buffer(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
	@available(*, deprecated)
		set
		{
			linphone_content_set_string_buffer(cPtr, newValue)
		}
	}
		
	/// Set the mime subtype of the content data. 
	/// - Parameter subtype: The mime subtype of the content data, for example "html". 
	///   
	
	/// Get the mime subtype of the content data. 
	/// - Returns: The mime subtype of the content data, for example "html".    
	public var subtype: String
	{
	
		get
		{ 
			
			let cPointer = linphone_content_get_subtype(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_content_set_subtype(cPtr, newValue)
		}
	}
		
	/// Set the mime type of the content data. 
	/// - Parameter type: The mime type of the content data, for example "application".
	///    
	
	/// Get the mime type of the content data. 
	/// - Returns: The mime type of the content data, for example "application".    
	public var type: String
	{
	
		get
		{ 
			
			let cPointer = linphone_content_get_type(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_content_set_type(cPtr, newValue)
		}
	}
		
	/// Assign a user pointer to the content. 
	/// - Parameter userData: The user pointer to associate with the content.    
	
	/// Retrieve the user pointer associated with the content. 
	/// - Returns: The user pointer associated with the content.    
	public var userData: UnsafeMutableRawPointer?
	{
	
		get
		{ 
						return linphone_content_get_user_data(cPtr)
		}
		set
		{
			linphone_content_set_user_data(cPtr, newValue)
		}
	}
		
	/// Get the string content data buffer. 
	/// Introduced in 01/07/2020 
	/// - Parameter buffer: The string content data buffer in UTF8.    
	
	/// Get the string content data buffer. 
	/// Introduced in 01/07/2020 
	/// - Returns: The string content data buffer in UTF8.    
	public var utf8Text: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_content_get_utf8_text(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_content_set_utf8_text(cPtr, newValue)
		}
	}
		
	
	
	/// Adds a parameter to the ContentType header. 
	/// - Parameter name: the name of the parameter to add.    
	/// - Parameter value: the value of the parameter to add.    
	public func addContentTypeParameter(name:String, value:String?) 
	{
		linphone_content_add_content_type_parameter(cPtr, name, value)
	}
	
	
	
	/// Adds a custom header in a content. 
	/// - Parameter headerName: The name of the header to add.    
	/// - Parameter headerValue: The value of the header to add.    
	public func addCustomHeader(headerName:String, headerValue:String) 
	{
		linphone_content_add_custom_header(cPtr, headerName, headerValue)
	}
	
	
	
	/// Generates a temporary plain copy of the file and returns its paths The caller
	/// is responsible to then delete this temporary copy and the returned string. 
	/// - Returns: The file path set for this content if it has been set, nil
	/// otherwise.    
	public func exportPlainFile() -> String
	{
		let cstr = linphone_content_export_plain_file(cPtr)
		let result = charArrayToString(charPointer: cstr)
		if (cstr != nil) {
			bctbx_free(cstr)
		}
		return result
	}
	
	
	
	/// Find a part from a multipart content looking for a part header with a specified
	/// value. 
	/// - Parameter headerName: The name of the header to look for.    
	/// - Parameter headerValue: The value of the header to look for.    
	/// - Returns: A ``Content`` object object the part if found, nil otherwise.    
	public func findPartByHeader(headerName:String, headerValue:String) -> Content?
	{
		let cPointer = linphone_content_find_part_by_header(cPtr, headerName, headerValue)
		if (cPointer == nil) {
			return nil
		}
		let result = Content.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Get a custom header value of a content. 
	/// - Parameter headerName: The name of the header to get the value from.    
	/// - Returns: The value of the header if found, nil otherwise.    
	public func getCustomHeader(headerName:String) -> String
	{
		let cstr = linphone_content_get_custom_header(cPtr, headerName)
		let result = charArrayToString(charPointer: cstr)
		return result
	}
	
	
	
	/// Get a part from a multipart content according to its index. 
	/// - Parameter index: The index of the part to get. 
	/// - Returns: A ``Content`` object holding the part if found, nil otherwise.    
	public func getPart(index:Int) -> Content?
	{
		let cPointer = linphone_content_get_part(cPtr, CInt(index))
		if (cPointer == nil) {
			return nil
		}
		let result = Content.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Set the content data buffer, usually a string. 
	/// - Parameter buffer: The content data buffer.    
	/// - Parameter size: The size of the content data buffer. 
	public func setBuffer(buffer:UnsafePointer<UInt8>, size:Int) 
	{
		linphone_content_set_buffer(cPtr, buffer, size)
	}
	
	
	
	/// Set the key associated with a RCS file transfer message if encrypted. 
	/// - Parameter key: The key to be used to encrypt/decrypt file associated to this
	/// content.    
	/// - Parameter keyLength: The lengh of the key. 
	public func setKey(key:String, keyLength:Int) 
	{
		linphone_content_set_key(cPtr, key, keyLength)
	}
}


/// Main object to instanciate and on which to keep a reference. 
/// This object is the first object to instanciante, and will allow you to perform
/// all kind of tasks. To create it, use either
/// ``Factory/createCore(configPath:factoryConfigPath:systemContext:)`` or
/// ``Factory/createCoreWithConfig(config:systemContext:)``, see ``Config`` for
/// more information about factory and default config. On some platforms like
/// Android or iOS you will need to give it the Context of your application.
/// Once the ``Core`` is in state ``Ready``, use ``start()``. It will then go to
/// state ``On`` and from that you can start using it for calls and chat messages.
/// It is recommended to add a ``CoreDelegate`` listener using
/// ``addDelegate(cbs:)`` to monitor different events.
/// To be able to receive events from the network, you must schedule a call
/// ``iterate()`` often, like every 20ms. On Android & iOS
/// ``isAutoIterateEnabled()`` is enabled by default so you don't have to worry
/// about that unless you disable it using ``enableAutoIterate(enable:)`` or by
/// setting in the [misc] section of your configuration auto_iterate=0. - Warning:
/// Our API isn't thread-safe but also isn't blocking, so it is strongly recommend
/// to always call our methods from the main thread.
/// Once you don't need it anymore, call ``stop()`` and release the reference on it
/// so it can gracefully shutdown. 
public class Core : LinphoneObject
{
	var delegateManagers : [CoreDelegateManager] = []

	static public func getSwiftObject(cObject:OpaquePointer) -> Core {
		let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
		if (result != nil) {
			return Unmanaged<Core>.fromOpaque(result!).takeUnretainedValue()
		}
		let sObject = Core(cPointer: cObject)
		belle_sip_object_data_set(UnsafeMutablePointer(cObject), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(sObject).toOpaque()), nil)
		return sObject
	}

	public var getCobject: OpaquePointer? {
		return cPtr
	}
	/// Get the UIView in which the video is be rendered. Use nativeVideoWindowId for direct pointer management.
	/// - Returns: The UIView in which the video will be rendered.
#if os(iOS) || os(tvOS)
	public var nativeVideoWindow: UIView?
	{
		get
		{
			return Unmanaged<UIView>.fromOpaque(linphone_core_get_native_video_window_id(cPtr)).takeUnretainedValue()
		}
		set
		{
			if let previousViewPtr = linphone_core_get_native_video_window_id(cPtr)
			{
				Unmanaged<UIView>.fromOpaque(previousViewPtr).release()
			}
			if let view = newValue {
				linphone_core_set_native_video_window_id(cPtr, UnsafeMutableRawPointer(Unmanaged.passRetained(view).toOpaque()))
			} else {
				linphone_core_set_native_video_window_id(cPtr, nil)
			}
		}
	}
#else
	public var nativeVideoWindow: UnsafeMutableRawPointer?

	{
		get
		{
			return linphone_core_get_native_video_window_id(cPtr)
		}
		set
		{
			linphone_core_set_native_video_window_id(cPtr, newValue)
		}
	}
#endif
	/// Get the UIView in which the camera preview is be rendered. Use nativePreviewWindowId for direct pointer management.
	/// - Returns: The UIView in which the camera preview will be rendered.
#if os(iOS) || os(tvOS)
	public var nativePreviewWindow: UIView?
	{
		get
		{
			return Unmanaged<UIView>.fromOpaque(linphone_core_get_native_preview_window_id(cPtr)).takeUnretainedValue()
		}
		set
		{
			if let previousViewPtr = linphone_core_get_native_preview_window_id(cPtr)
			{
				Unmanaged<UIView>.fromOpaque(previousViewPtr).release()
			}
			if let view = newValue {
				linphone_core_set_native_preview_window_id(cPtr, UnsafeMutableRawPointer(Unmanaged.passRetained(view).toOpaque()))
			} else {
				linphone_core_set_native_preview_window_id(cPtr, nil)
			}
		}
	}
#else
	public var nativePreviewWindow: UnsafeMutableRawPointer?

	{
		get
		{
			return linphone_core_get_native_preview_window_id(cPtr)
		}
		set
		{
			linphone_core_set_native_preview_window_id(cPtr, newValue)
		}
	}
#endif

	
	///Used to notify if log collection upload have been successfully delivered or
	///not. 
	public enum LogCollectionUploadState:Int
	{
		
		/// Delivery in progress. 
		case InProgress = 0
		/// Log collection upload successfully delivered and acknowledged by remote end
		/// point. 
		case Delivered = 1
		/// Log collection upload was not delivered. 
		case NotDelivered = 2
	}
	
	
	
	public func addDelegate(delegate: CoreDelegate)
	{
		let manager = CoreDelegateManager()
		manager.delegate = delegate
		delegateManagers.append(manager)
		linphone_core_add_callbacks(cPtr, manager.cPtr)
		belle_sip_object_unref(UnsafeMutableRawPointer(manager.cPtr))
		belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(self).toOpaque()), nil)
	}	
	
	
	public func removeDelegate(delegate: CoreDelegate)
	{
		if let index = delegateManagers.firstIndex(where: { $0.delegate === delegate as AnyObject}) {
			linphone_core_remove_callbacks(cPtr, delegateManagers[index].cPtr)
			delegateManagers.remove(at: index)
			belle_sip_object_data_set(UnsafeMutablePointer(cPtr), "swiftRef",  UnsafeMutableRawPointer(Unmanaged.passUnretained(self).toOpaque()), nil)
		}
	}	
	
	
	/// Compresses the log collection in a single file. 
	/// - Returns: The path of the compressed log collection file.       
	static public func compressLogCollection() -> String
	{
		let cstr = linphone_core_compress_log_collection()
		let result = charArrayToString(charPointer: cstr)
		if (cstr != nil) {
			bctbx_free(cstr)
		}
		return result
	}
	
	
	
	/// Enables the linphone core log collection to upload logs on a server. 
	/// - Parameter state: ``LogCollectionState`` value telling whether to enable log
	/// collection or not. 
	static public func enableLogCollection(state:LogCollectionState) 
	{
		linphone_core_enable_log_collection(LinphoneLogCollectionState(rawValue: CUnsignedInt(state.rawValue)))
	}
	
	
	/// Gets the max file size in bytes of the files used for log collection. 
	/// - Returns: The max file size in bytes of the files used for log collection.    
	static public var getLogCollectionMaxFileSize: Int
	{
	
						return Int(linphone_core_get_log_collection_max_file_size())

	}
		
	
	/// Gets the path where the log files will be written for log collection. 
	/// - Returns: The path where the log files will be written.    
	static public var getLogCollectionPath: String
	{
	
			
			let cPointer = linphone_core_get_log_collection_path()
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Gets the prefix of the filenames that will be used for log collection. 
	/// - Returns: The prefix of the filenames used for log collection.    
	static public var getLogCollectionPrefix: String
	{
	
			
			let cPointer = linphone_core_get_log_collection_prefix()
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Are PostQuantum algoritms available. 
	/// - Returns: true if Post Quantum algorithms are available false otherwise 
	static public var getPostQuantumAvailable: Bool
	{
	
						return linphone_core_get_post_quantum_available() != 0

	}
		
	
	/// Returns liblinphone's version as a string. 
	/// - Returns: the current version of the ``Core``    
	static public var getVersion: String
	{
	
			
			let cPointer = linphone_core_get_version()
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	
	/// Tells whether the linphone core log collection is enabled. 
	/// - Returns: The ``LogCollectionState`` of the ``Core`` log collection. 
	static public func logCollectionEnabled() -> LogCollectionState
	{
		return LogCollectionState(rawValue: Int(linphone_core_log_collection_enabled().rawValue))!
	}
	
	
	
	/// Resets the log collection by removing the log files. 
	static public func resetLogCollection() 
	{
		linphone_core_reset_log_collection()
	}
	
	
	
	/// Enables logs serialization (output logs from either the thread that creates the
	/// linphone core or the thread that calls ``iterate()``). 
	/// Must be called before creating the ``Core``. 
	static public func serializeLogs() 
	{
		linphone_core_serialize_logs()
	}
	
	
	
	/// Sets the max file size in bytes of the files used for log collection. 
	/// Warning: this function should only not be used to change size dynamically but
	/// instead only before calling ``enableLogCollection(state:)``. If you increase
	/// max size on runtime, logs chronological order COULD be broken. 
	/// - Parameter size: The max file size in bytes of the files used for log
	/// collection. 
	static public func setLogCollectionMaxFileSize(size:Int) 
	{
		linphone_core_set_log_collection_max_file_size(size)
	}
	
	
	
	/// Sets the path of a directory where the log files will be written for log
	/// collection. 
	/// When log collection is enabled, the function will close the file with the
	/// current prefix in the old path and it will open the new file with current
	/// prefix in the new path. If you need to change the path and the file at the same
	/// time, then you should deactivate log collection with
	/// ``enableLogCollection(state:)`` before doing modifications. 
	/// - Parameter path: The path where the log files will be written.    
	static public func setLogCollectionPath(path:String) 
	{
		linphone_core_set_log_collection_path(path)
	}
	
	
	
	/// Sets the prefix of the filenames that will be used for log collection. 
	/// When log collection is enabled, the function will close the old file and it
	/// will open the new one in the current path. If you need to change the path and
	/// the file at the same time, then you should deactivate log collection with
	/// ``enableLogCollection(state:)`` before doing modifications. 
	/// - Parameter prefix: The prefix to use for the filenames for log collection.    
	static public func setLogCollectionPrefix(prefix:String) 
	{
		linphone_core_set_log_collection_prefix(prefix)
	}
	
	
	
	/// True if tunnel extension was compiled. 
	/// - Returns: true if library was built with tunnel, false otherwise 
	static public func tunnelAvailable() -> Bool
	{
		return linphone_core_tunnel_available() != 0
	}
	
	
	
	/// Return the availability of uPnP. 
	/// - Returns: true if uPnP is available otherwise return false. 
	/// - Warning: UPNP support has been removed, and proved to be useless. 
	static public func upnpAvailable() -> Bool
	{
		return linphone_core_upnp_available() != 0
	}
	
	
	
	/// Tells whether VCARD support is builtin. 
	/// - Returns: true if VCARD is supported, false otherwise. 
	static public func vcardSupported() -> Bool
	{
		return linphone_core_vcard_supported() != 0
	}
	
	/// Sets the ``AccountCreator`` backend on the ``Core``. 
	/// - Parameter backend: The ``AccountCreator.Backend`` 
	
	/// Gets the ``AccountCreator`` backend set for the ``Core``. 
	/// - Returns: The ``AccountCreator.Backend`` 
	public var accountCreatorBackend: AccountCreator.Backend
	{
	
		get
		{ 
						return AccountCreator.Backend(rawValue: Int(linphone_core_get_account_creator_backend(cPtr).rawValue))!
		}
		set
		{
			linphone_core_set_account_creator_backend(cPtr, LinphoneAccountCreatorBackend(rawValue: CUnsignedInt(newValue.rawValue)))
		}
	}
		
	/// Sets the ``AccountCreator`` url on the ``Core``. 
	/// - Parameter url: The URL to reach    
	
	/// Gets the ``AccountCreator`` url on the ``Core``. 
	/// - Returns: url The URL to reach.    
	public var accountCreatorUrl: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_core_get_account_creator_url(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_core_set_account_creator_url(cPtr, newValue)
		}
	}
		
	
	/// Returns an unmodifiable list of entered accounts. 
	/// - Returns:      
	public var accountList: [Account]
	{
	
						var swiftList = [Account]()
			let cList = linphone_core_get_account_list(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(Account.getSwiftObject(cObject: data))
				listTemp = UnsafePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
			return swiftList

	}
		
	/// Sets adaptive rate algorithm. 
	/// It will be used for each new calls starting from now. Calls already started
	/// will not be updated. 
	/// - Parameter algorithm: the adaptive rate control algorithm. Currently two
	/// values are supported: 'advanced', which is the default value, or 'basic'.    
	
	/// Returns which adaptive rate algorithm is currently configured for future calls. 
	/// - See also: ``setAdaptiveRateAlgorithm(algorithm:)`` 
	/// - Returns: the adaptive rate algorithm. Currently two values are supported:
	/// 'advanced', which is the default value, or 'basic'.    
	public var adaptiveRateAlgorithm: String
	{
	
		get
		{ 
			
			let cPointer = linphone_core_get_adaptive_rate_algorithm(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_core_set_adaptive_rate_algorithm(cPtr, newValue)
		}
	}
		
	/// Enable adaptive rate control. 
	/// Adaptive rate control consists in using RTCP feedback provided information to
	/// dynamically control the output bitrate of the audio and video encoders, so that
	/// we can adapt to the network conditions and available bandwidth. Control of the
	/// audio encoder is done in case of audio-only call, and control of the video
	/// encoder is done for audio & video calls. Adaptive rate control feature is
	/// enabled by default.
	/// - Parameter enabled: true to enable adaptive rate control, false otherwise 
	
	/// Returns whether adaptive rate control is enabled. 
	/// - See also: ``enableAdaptiveRateControl(enabled:)``
	/// - Returns: true if adaptive rate control is enabled, false otherwise 
	public var adaptiveRateControlEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_adaptive_rate_control_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_adaptive_rate_control(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Enables the experimental software Automatic Gain Control. 
	/// This algorithm is very experimental, not usable in its current state. Automatic
	/// Gain Control is usally provided by sound devices, typically on iOS and Android. 
	/// - Parameter val: a boolean value 
	
	/// Tells whether the experimental software Automatic Gain Control is activated. 
	/// This algorithm is very experimental, not usable in its current state. 
	/// - Returns: true if the AGC is enabled, false otherwise. 
	public var agcEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_agc_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_agc(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Enables alerts. 
	/// See ``Alert`` for more details. 
	/// - Parameter enable: whether alert reporting is enabled or not. 
	
	/// Returns whether alert reporting is enabled. 
	/// See ``Alert`` for more details. 
	/// - Returns: whether alert reporting is enabled. 
	public var alertsEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_alerts_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_alerts(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Enables or disables the audio adaptive jitter compensation. 
	/// It is enabled by default and should be disabled only to verify how
	/// communication over IP is a disaster without jitter compensation. 
	/// - Parameter enable: true to enable the audio adaptive jitter compensation,
	/// false to disable it. 
	
	/// Tells whether the audio adaptive jitter compensation is enabled. 
	/// - Returns: true if the audio adaptive jitter compensation is enabled, false
	/// otherwise. 
	public var audioAdaptiveJittcompEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_audio_adaptive_jittcomp_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_audio_adaptive_jittcomp(cPtr, newValue==true ? 1:0)
		}
	}
		
	
	/// Returns a list of audio devices, with only the first device for each type To
	/// have the list of all audio devices, use ``getExtendedAudioDevices()`` 
	/// - Returns:   A list with the first ``AudioDevice`` of each type       
	public var audioDevices: [AudioDevice]
	{
	
						var swiftList = [AudioDevice]()
			let cList = linphone_core_get_audio_devices(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(AudioDevice.getSwiftObject(cObject: data))
				listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
				bctbx_list_free_with_data(cList, belle_sip_object_unref)
			return swiftList

	}
		
	/// Sets the DSCP field for outgoing audio streams. 
	/// The DSCP defines the quality of service in IP packets. - Note: It is usually
	/// useless or a bad idea to try to play with DSCP bits unless having full control
	/// on the network. 
	/// - Warning: Setting the DSCP bits is more or less well supported by operating
	/// systems and sometimes requires to disable IPv6. 
	/// - Parameter dscp: The DSCP value to set 
	
	/// Gets the DSCP field for outgoing audio streams. 
	/// The DSCP defines the quality of service in IP packets. 
	/// - Returns: The current DSCP value 
	public var audioDscp: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_audio_dscp(cPtr))
		}
		set
		{
			linphone_core_set_audio_dscp(cPtr, CInt(newValue))
		}
	}
		
	/// Sets the nominal audio jitter buffer size in milliseconds. 
	/// The value takes effect immediately for all running and pending calls, if any. A
	/// value of 0 disables the jitter buffer.
	/// - Parameter milliseconds: the audio jitter buffer size to set in milliseconds 
	
	/// Returns the nominal audio jitter buffer size in milliseconds. 
	/// - Returns: The nominal audio jitter buffer size in milliseconds 
	public var audioJittcomp: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_audio_jittcomp(cPtr))
		}
		set
		{
			linphone_core_set_audio_jittcomp(cPtr, CInt(newValue))
		}
	}
		
	/// Use to set multicast address to be used for audio stream. 
	/// - Parameter ip: an ipv4/6 multicast address.    
	/// - Returns: 0 in case of success 
	
	/// Use to get multicast address to be used for audio stream. 
	/// - Returns: an ipv4/6 multicast address or default value.    
	public var audioMulticastAddr: String?
	{
	
			
			let cPointer = linphone_core_get_audio_multicast_addr(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
	public func setAudiomulticastaddr(newValue: String?) throws
	{
		let exception_result = linphone_core_set_audio_multicast_addr(cPtr, newValue)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "username setter returned value \(exception_result)")
		}
	}
		
	/// Use to enable multicast rtp for audio stream. 
	/// If enabled, outgoing calls put a multicast address from
	/// ``getVideoMulticastAddr()`` into audio cline. In case of outgoing call audio
	/// stream is sent to this multicast address. For incoming calls behavior is
	/// unchanged. 
	/// - Parameter yesno: if yes, subsequent calls will propose multicast ip set by
	/// ``setAudioMulticastAddr(ip:)`` 
	
	/// Use to get multicast state of audio stream. 
	/// - Returns: true if subsequent calls will propose multicast ip set by
	/// ``setAudioMulticastAddr(ip:)`` 
	public var audioMulticastEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_audio_multicast_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_audio_multicast(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Use to set multicast ttl to be used for audio stream. 
	/// - Parameter ttl: value or -1 if not used. [0..255] default value is 1 
	/// - Returns: 0 in case of success 
	
	/// Use to get multicast ttl to be used for audio stream. 
	/// - Returns: a time to leave value 
	public var audioMulticastTtl: Int
	{
	
						return Int(linphone_core_get_audio_multicast_ttl(cPtr))

	}
	public func setAudiomulticastttl(newValue: Int) throws
	{
		let exception_result = linphone_core_set_audio_multicast_ttl(cPtr, CInt(newValue))
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "username setter returned value \(exception_result)")
		}
	}
		
	/// Redefines the list of the available payload types (codecs), in their order of
	/// preference. 
	/// The payload type are listed from higher priority to lowest priority. It is
	/// worth to note that the set of ``PayloadType`` objects assigned here need to be
	/// the same as the one returned by ``getAudioPayloadTypes()``. The purpose of the
	/// setter is to let application modify their order of preference. In particular,
	/// including in the provided list a payload type not supported has no effect.
	/// Removing specific payload types from the original list has no effect too:
	/// missing payload types will be automatically added. In order to disable a
	/// specific codec, applications shall use ``PayloadType/enable(enabled:)`` instead.
	/// - Parameter payloadTypes: The new list of payload types.      
	
	/// Returns the list of the available audio payload types. 
	/// - Returns: A freshly allocated list of the available payload types.         
	public var audioPayloadTypes: [PayloadType]
	{
	
		get
		{ 
						var swiftList = [PayloadType]()
			let cList = linphone_core_get_audio_payload_types(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(PayloadType.getSwiftObject(cObject: data))
				listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
				bctbx_list_free_with_data(cList, belle_sip_object_unref)
			return swiftList
		}
		set
		{
			withExtendedLifetime(newValue) {
				var cList: UnsafeMutablePointer<bctbx_list_t>? = nil
				for data in newValue {
					cList = bctbx_list_append(cList, UnsafeMutableRawPointer(data.cPtr))
				}
				linphone_core_set_audio_payload_types(cPtr, cList)
			}
		}
	}
		
	/// Sets the UDP port used for audio streaming. 
	/// A value of -1 will request the system to allocate the local port randomly. This
	/// is recommended in order to avoid firewall warnings. 
	/// - Parameter port: The UDP port to use for audio streaming 
	
	/// Gets the UDP port used for audio streaming. 
	/// - Returns: The UDP port used for audio streaming 
	public var audioPort: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_audio_port(cPtr))
		}
		set
		{
			linphone_core_set_audio_port(cPtr, CInt(newValue))
		}
	}
		
	
	/// Get the audio port range from which is randomly chosen the UDP port used for
	/// audio streaming. 
	/// - Returns: a ``Range`` object       
	public var audioPortsRange: Range?
	{
	
						let cPointer = linphone_core_get_audio_ports_range(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Range.getSwiftObject(cObject:cPointer!)
			belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
			return result

	}
		
	
	/// Returns an unmodifiable list of currently entered ``AuthInfo``. 
	/// - Returns: A list of ``AuthInfo``.      
	public var authInfoList: [AuthInfo]
	{
	
						var swiftList = [AuthInfo]()
			let cList = linphone_core_get_auth_info_list(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(AuthInfo.getSwiftObject(cObject: data))
				listTemp = UnsafePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
			return swiftList

	}
		
	/// Automatically downloads files attached to a chat message if it's content type
	/// matches the one we use for icalendars. 
	/// - Parameter autoDownloadIcalendars: true to automatically download incoming
	/// icalendars, false to disable it. 
	
	/// Gets if the automatic download of incoming icalendars is enabled or not. 
	/// - Returns: true if icalendars will be automatically downloaded, false
	/// otherwise. 
	public var autoDownloadIcalendarsEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_auto_download_icalendars_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_auto_download_icalendars(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Enables automatic download of files attached to a chat message if it's content
	/// type matches the one we use for voice recordings. 
	/// - Parameter autoDownloadVoiceRecordings: true to automatically download
	/// incoming voice recordings, false to disable it. 
	
	/// Gets if the auto download for incoming voice recordings is enabled or not. 
	/// - Returns: true if voice recordings will be automatically downloaded, false
	/// otherwise. 
	public var autoDownloadVoiceRecordingsEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_auto_download_voice_recordings_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_auto_download_voice_recordings(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets the timer used to schedule the call to core.iterate() method when in
	/// background (Android only). 
	/// - Parameter schedule: The timing in milliseconds used to schedule the call
	/// while in background. 
	
	/// Gets the timer used to schedule the call to core.iterate() method when in
	/// background (Android only). 
	/// This is only used when ``autoIterateEnabled()`` returns true. 
	/// - Returns: The timing in milliseconds used to schedule the call while in
	/// background (default is 500ms). 
	public var autoIterateBackgroundSchedule: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_auto_iterate_background_schedule(cPtr))
		}
		set
		{
			linphone_core_set_auto_iterate_background_schedule(cPtr, CInt(newValue))
		}
	}
		
	/// Enable or disable the automatic schedule of ``iterate()`` method on Android &
	/// iOS. 
	/// If enabled, ``iterate()`` will be called on the main thread every 20ms
	/// automatically. If disabled, it is the application that must do this job. 
	/// - Parameter enable: true to enable auto iterate, false to disable 
	
	/// Gets whether auto iterate is enabled or not (Android & iOS only). 
	/// - Returns: true if ``iterate()`` is scheduled automatically, false otherwise 
	public var autoIterateEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_auto_iterate_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_auto_iterate(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets the timer used to schedule the call to core.iterate() method when in
	/// foreground (Android only). 
	/// - Parameter schedule: The timing in milliseconds used to schedule the call
	/// while in foreground. 
	
	/// Gets the timer used to schedule the call to core.iterate() method when in
	/// foreground (Android only). 
	/// This is only used when ``autoIterateEnabled()`` returns true. 
	/// - Returns: The timing in milliseconds used to schedule the call while in
	/// foreground (default is 20ms). 
	public var autoIterateForegroundSchedule: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_auto_iterate_foreground_schedule(cPtr))
		}
		set
		{
			linphone_core_set_auto_iterate_foreground_schedule(cPtr, CInt(newValue))
		}
	}
		
	/// Enables the automatic sending of 180 Ringing when receiving a call. 
	/// - Parameter enable: true to activate the automatic 180 Ringing, false to
	/// disable it. 
	
	/// Gets if the automatic sending of 180 Ringing is enabled or not. 
	/// - Returns: true if the automatic sending of 180 Ringing is enabled, false
	/// otherwise. 
	public var autoSendRingingEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_auto_send_ringing_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_auto_send_ringing(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Enables or disables automatic http proxy detection. 
	/// - Parameter enable: true if automatic http proxy is enabled or false. 
	
	/// Returns whether automatic http proxy is enabled. 
	/// - Returns: true if automatic http proxy is enabled or false. 
	public var automaticHttpProxyDetectionEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_automatic_http_proxy_detection_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_automatic_http_proxy_detection(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Enables RTCP feedback (also known as RTP/AVPF profile). 
	/// Setting ``Default`` is equivalent to LinphoneAVPFDisabled. This setting can be
	/// overriden per ``Account`` with ``AccountParams/setAvpfMode(mode:)``. The value
	/// set here is used for calls placed or received out of any account configured, or
	/// if the account is configured with LinphoneAVPFDefault. 
	/// - Parameter mode: The AVPF mode to use. 
	
	/// Returns AVPF enablement. 
	/// See ``setAvpfMode(mode:)`` . 
	/// - Returns: The current ``AVPFMode`` mode 
	public var avpfMode: AVPFMode
	{
	
		get
		{ 
						return AVPFMode(rawValue: Int(linphone_core_get_avpf_mode(cPtr).rawValue))!
		}
		set
		{
			linphone_core_set_avpf_mode(cPtr, LinphoneAVPFMode(rawValue: CInt(newValue.rawValue)))
		}
	}
		
	/// Sets the avpf report interval in seconds. 
	/// This value can be overriden at Account level using
	/// linphone_account_params__set_avpf_rr_interval(). 
	/// - Parameter interval: The report interval in seconds 
	
	/// Returns the avpf report interval in seconds. 
	/// - Returns: The current AVPF report interval in seconds 
	public var avpfRrInterval: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_avpf_rr_interval(cPtr))
		}
		set
		{
			linphone_core_set_avpf_rr_interval(cPtr, CInt(newValue))
		}
	}
		
	/// Enable text sending via Baudot tones in the audio stream. 
	/// It is disabled by default. Enablement requires a SDK built with full Baudot
	/// support: -DENABLE_BAUDOT=ON . 
	/// - Parameter enabled: true if enabled, false otherwise. 
	
	/// Returns enablement of text sending via Baudot tones in the audio stream. 
	/// - Returns: true if text sending via Baudot tones in the audio stream is
	/// enabled, false otherwise. 
	public var baudotEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_baudot_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_baudot(cPtr, newValue==true ? 1:0)
		}
	}
		
	
	/// Get the list of call logs (past calls). 
	/// - Returns: A list of ``CallLog``.      
	public var callLogs: [CallLog]
	{
	
						var swiftList = [CallLog]()
			let cList = linphone_core_get_call_logs(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(CallLog.getSwiftObject(cObject: data))
				listTemp = UnsafePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
			return swiftList

	}
		
	/// Sets the database filename where call logs will be stored. 
	/// If the file does not exist, it will be created.
	/// - Parameter path: filesystem path    
	/// - Deprecated: 07/12/2021: Use only for migration purposes 
	
	/// Gets the database filename where call logs will be stored. 
	/// - Returns: filesystem path.    
	/// - Deprecated: 07/12/2021: Use only for migration purposes 
	public var callLogsDatabasePath: String?
	{
	@available(*, deprecated)
		get
		{ 
			
			let cPointer = linphone_core_get_call_logs_database_path(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
	@available(*, deprecated)
		set
		{
			linphone_core_set_call_logs_database_path(cPtr, newValue)
		}
	}
		
	/// Sets a wav file to be played when putting somebody on hold (see
	/// ``setUseFiles(yesno:)``). 
	/// The file must be a 16 bit linear wav file. 
	/// - Parameter file: The path to the file to be played when putting somebody on
	/// hold.    
	
	/// Get the wav file played when putting a call on hold with ``Call/pause()`` 
	/// The file is a 16 bit linear file. 
	/// - Returns: The path to the file that is played when putting a call on hold.    
	public var callOnHoldMusicFile: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_core_get_call_on_hold_music_file(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_core_set_call_on_hold_music_file(cPtr, newValue)
		}
	}
		
	/// Enables or disables call tone indications. 
	/// This value is taken into account from next time call parameters are created.
	/// This feature can also be enabled per-call using ``CallParams``. 
	/// - Parameter yesno: a boolean to indicate whether the feature is to be enabled. 
	
	/// Check whether tone indications of calls are enabled. 
	/// - Returns: true if call tone indications are enabled 
	public var callToneIndicationsEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_call_tone_indications_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_call_tone_indications(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Special function to enable the callkit. 
	/// - Parameter enabled: true to enable callkit, false to disable it. 
	
	/// Special function to check if the callkit is enabled, False by default. 
	/// - Returns: true if callkit is enabled, false otherwise. 
	public var callkitEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_callkit_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_callkit(cPtr, newValue==true ? 1:0)
		}
	}
		
	
	/// Gets the current list of calls. 
	/// Note that this list is read-only and might be changed by the core after a
	/// function call to ``iterate()``. Similarly the ``Call`` objects inside it might
	/// be destroyed without prior notice. To hold references to ``Call`` object into
	/// your program, you must use linphone_call_ref. 
	/// - Returns: A list of ``Call``      
	public var calls: [Call]
	{
	
						var swiftList = [Call]()
			let cList = linphone_core_get_calls(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(Call.getSwiftObject(cObject: data))
				listTemp = UnsafePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
			return swiftList

	}
		
	
	/// Gets the number of Call. 
	/// - Returns: The current number of calls 
	public var callsNb: Int
	{
	
						return Int(linphone_core_get_calls_nb(cPtr))

	}
		
	
	/// Get the camera sensor rotation. 
	/// This is needed on some mobile platforms to get the number of degrees the camera
	/// sensor is rotated relative to the screen. 
	/// - Returns: The camera sensor rotation in degrees (0 to 360) or -1 if it could
	/// not be retrieved 
	public var cameraSensorRotation: Int
	{
	
						return Int(linphone_core_get_camera_sensor_rotation(cPtr))

	}
		
	/// Sets the whitebalance of the camera (currently only supported by Android). 
	/// - Parameter whitebalance: The whitebalance of the camera. 
	
	/// Gets the whitebalance of the camera (currently only supported by Android). 
	/// - Returns: The whitebalance of the camera, default is -1 (disabled). 
	public var cameraWhitebalance: Int
	{
	
						return Int(linphone_core_get_camera_whitebalance(cPtr))

	}
	public func setCamerawhitebalance(newValue: Int) throws
	{
		let exception_result = linphone_core_set_camera_whitebalance(cPtr, CInt(newValue))
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "username setter returned value \(exception_result)")
		}
	}
		
	/// Defines whether capability negotiation (RFC5939) is supported. 
	/// - Parameter enable: true to support RFC5939; false otherwise. 
	
	/// Checks if the capability negotiation (RFC5939) is supported or not. 
	/// - Returns: true if capability negotiation is supported; false otherwise. 
	public var capabilityNegociationEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_capability_negociation_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_capability_negociation(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Defines whether capability negotiation (RFC5939) reINVITE is enabled. 
	/// - Parameter enable: true to enable capability negotiation reINVITE; false
	/// otherwise. 
	
	/// Checks if the capability negotiation (RFC5939) reINVITE is enabled or not. 
	/// - Returns: true if capability negotiation reINVITE is enabled; false otherwise. 
	public var capabilityNegotiationReinviteEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_capability_negotiation_reinvite_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_capability_negotiation_reinvite(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets the sound device used for capture. 
	/// - Parameter devid: The device name as returned by
	/// linphone_core_get_sound_devices    
	/// - Returns: 0 
	/// - Deprecated: 11/09/2024 use ``setInputAudioDevice(audioDevice:)`` or
	/// ``setDefaultInputAudioDevice(audioDevice:)`` instead.
	
	/// Gets the name of the currently assigned sound device for capture. 
	/// - Returns: The name of the currently assigned sound device for capture.    
	/// - Deprecated: 11/09/2024 use ``getInputAudioDevice()`` or
	/// ``getDefaultInputAudioDevice()`` instead.
	@available(*, deprecated)
	public var captureDevice: String?
	{
	
			
			let cPointer = linphone_core_get_capture_device(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
	@available(*, deprecated)
	public func setCapturedevice(newValue: String?) throws
	{
		let exception_result = linphone_core_set_capture_device(cPtr, newValue)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "username setter returned value \(exception_result)")
		}
	}
		
	/// Defines whether cfg lines are going to be merged if capability negotiation
	/// (RFC5939) is supported. 
	/// - Parameter merge: true to merge acfg and pcfg lines with consecutive indexes;
	/// false otherwise. 
	
	/// Checks if cfg lines are going to the merged if the capability negotiation
	/// (RFC5939) is supported or not. 
	/// - Returns: true if acfg and pcfg lines with consecutive indexes are going to be
	/// merged; false otherwise. 
	public var cfgLinesMergingEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_cfg_lines_merging_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_cfg_lines_merging(cPtr, newValue==true ? 1:0)
		}
	}
		
	
	/// Returns whether chat is enabled. 
	/// - Returns: true if chat is enabled, false otherwise 
	public var chatEnabled: Bool
	{
	
						return linphone_core_chat_enabled(cPtr) != 0

	}
		
	/// Sets whether chat messages grouping is enabled or not. 
	/// This optimisation is turned on by default. It allows to receive bulks of
	/// incoming message faster, and notify them to the application in a row. Set [sip]
	/// chat_messages_aggregation in your configuration file for the timer, default is
	/// 500ms. 
	/// - Parameter enabled: true to wait for chat messages and notify them as at once,
	/// false to keep legacy behavior. 
	
	/// End of group ldap. 
	/// Returns whether chat messages grouping is enabled or not. 
	/// - Returns: true if received chat messages will be notified as a bundle, false
	/// otherwise. 
	public var chatMessagesAggregationEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_get_chat_messages_aggregation_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_set_chat_messages_aggregation_enabled(cPtr, newValue==true ? 1:0)
		}
	}
		
	
	/// Returns a list of chat rooms. 
	/// - Returns: List of chat rooms.      
	public var chatRooms: [ChatRoom]
	{
	
						var swiftList = [ChatRoom]()
			let cList = linphone_core_get_chat_rooms(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(ChatRoom.getSwiftObject(cObject: data))
				listTemp = UnsafePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
			return swiftList

	}
		
	/// Set the conference availability before start. 
	/// It is the number of seconds clients can join the conference before its actual
	/// start time. 
	/// - Parameter seconds: number of seconds the conference can be joined early. A
	/// negative value means always
	/// - Warning: This setting is only applicable to conference servers 
	
	/// Gets the conference availability before start. 
	/// - Returns: the number of seconds the conference can be joined early
	/// - Warning: This setting is only applicable to conference servers 
	public var conferenceAvailabilityBeforeStart: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_conference_availability_before_start(cPtr))
		}
		set
		{
			linphone_core_set_conference_availability_before_start(cPtr, (newValue))
		}
	}
		
	/// Set the conference cleanup timer period. 
	/// This timer helps managing the automatic deletion of ended conferences. In fact,
	/// under normal circumstances a conference is deleted only if it transition from
	/// an active to an inactive state after its end time. Nonetheless a side effect is
	/// that there may be a conference that never became active or it was terminate
	/// before its due date and time. This timer, if setup, therefore periodically
	/// looks for expired conference and cleans then up 
	/// - Parameter seconds: period of the timer. A 0 or negative value stops the timer 
	
	/// Gets the conference cleanup timer period. 
	/// - Returns: the period of the conference cleanup timer 
	public var conferenceCleanupPeriod: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_conference_cleanup_period(cPtr))
		}
		set
		{
			linphone_core_set_conference_cleanup_period(cPtr, (newValue))
		}
	}
		
	/// Set the conference expire period. 
	/// It is the number of seconds after the end time or the last participant joined -
	/// whichever is later - the conference cannot be joined anymore 
	/// - Parameter seconds: number of seconds before the conference expires. A 0 or
	/// negative value means immediately after the end
	/// - Warning: This setting is only applicable to conference servers 
	
	/// Gets the conference expire period. 
	/// - Returns: the number of second after which the conference cannot be joined
	/// - Warning: This setting is only applicable to conference servers 
	public var conferenceExpirePeriod: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_conference_expire_period(cPtr))
		}
		set
		{
			linphone_core_set_conference_expire_period(cPtr, (newValue))
		}
	}
		
	/// Enable sending conference invitations in the chat message body instead of as a
	/// file attachment. 
	/// - Parameter enable: true to send ICS as message body, false to send it as file
	/// transfer 
	
	/// Gets wether conference invitations will be sent in the chat message body or as
	/// a file attachment. 
	/// - Returns: true if ICS will be sent in the message body (by default), false if
	/// it will be sent as a file attachment. 
	public var conferenceIcsInMessageBodyEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_conference_ics_in_message_body_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_conference_ics_in_message_body(cPtr, newValue==true ? 1:0)
		}
	}
		
	
	/// Retrieve the list of conference information on DB. 
	/// - Returns: The list of conference infos  .       
	public var conferenceInformationList: [ConferenceInfo]
	{
	
						var swiftList = [ConferenceInfo]()
			let cList = linphone_core_get_conference_information_list(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(ConferenceInfo.getSwiftObject(cObject: data))
				listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
				bctbx_list_free_with_data(cList, belle_sip_object_unref)
			return swiftList

	}
		
	
	/// Returns the input volume of the local participant. 
	/// - Returns: A value inside [0.0 ; 1.0] 
	/// - Deprecated: 23/01/2025 Use ``Conference/getInputVolume()`` instead. 
	@available(*, deprecated)
	public var conferenceLocalInputVolume: Float
	{
	
						return linphone_core_get_conference_local_input_volume(cPtr)

	}
		
	/// Sets the maximum number of thumbnails requested in the SDP during a conference
	/// call. 
	/// - Parameter max: the maximum number of thumbnails requested in the SDP during a
	/// conference call 
	
	/// Gets the maximum number of thumbnails requested in the SDP during a conference
	/// call ``Account/getCallLogs()``. 
	/// - Returns: the maximum number of thumbnails requested in the SDP during a
	/// conference call 
	public var conferenceMaxThumbnails: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_conference_max_thumbnails(cPtr))
		}
		set
		{
			linphone_core_set_conference_max_thumbnails(cPtr, CInt(newValue))
		}
	}
		
	/// Selects whether the default conference participant list is open or closed. 
	/// - Parameter type: A ``Conference.ParticipantListType`` participant list type 
	
	/// Tells whether the default conference participant list is open or closed. 
	/// - Returns: A ``Conference.ParticipantListType`` participant list type 
	public var conferenceParticipantListType: Conference.ParticipantListType
	{
	
		get
		{ 
						return Conference.ParticipantListType(rawValue: Int(linphone_core_get_conference_participant_list_type(cPtr).rawValue))!
		}
		set
		{
			linphone_core_set_conference_participant_list_type(cPtr, LinphoneConferenceParticipantListType(rawValue: CUnsignedInt(newValue.rawValue)))
		}
	}
		
	/// Enables the conference server mode. 
	/// This has the effect to listen of the conference factory uri to create new
	/// conferences when receiving INVITE messages there. Enabling this mode is
	/// required when using liblinphone in a conference server application, but shall
	/// not be be employed in any client application. 
	/// - Parameter enable: A boolean value telling whether to enable or disable the
	/// conference server feature 
	
	/// Tells whether the conference server feature is enabled. 
	/// - Returns: A boolean value telling whether the conference server feature is
	/// enabled or not 
	public var conferenceServerEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_conference_server_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_conference_server(cPtr, newValue==true ? 1:0)
		}
	}
		
	
	/// Returns the config object used to manage the storage (config) file. 
	/// - Returns: a ``Config`` object.    
	public var config: Config?
	{
	
						let cPointer = linphone_core_get_config(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Config.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	/// Sets my consolidated presence. 
	/// - Parameter presence: ``ConsolidatedPresence`` value 
	
	/// Gets my consolidated presence. 
	/// - Returns: My ``ConsolidatedPresence`` presence 
	public var consolidatedPresence: ConsolidatedPresence
	{
	
		get
		{ 
						return ConsolidatedPresence(rawValue: Int(linphone_core_get_consolidated_presence(cPtr).rawValue))!
		}
		set
		{
			linphone_core_set_consolidated_presence(cPtr, LinphoneConsolidatedPresence(rawValue: CUnsignedInt(newValue.rawValue)))
		}
	}
		
	
	/// Gets the current call. 
	/// The current call is defined as follows: If multiple concurrent calls exist, it
	/// is the only one that currelty uses the microphone or speaker, otherwise it is
	/// the unique call that exists unless this call is in ``Paused`` state. 
	/// - Returns: The current call or nil if no call is running.    
	public var currentCall: Call?
	{
	
						let cPointer = linphone_core_get_current_call(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Call.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Gets the remote address of the current call. 
	/// - Returns: The remote address of the current call or nil if there is no current
	/// call.    
	public var currentCallRemoteAddress: Address?
	{
	
						let cPointer = linphone_core_get_current_call_remote_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Gets the current ``CoreDelegate`` being invoked, if any. 
	/// This is meant only to be called from a callback to be able to get the user_data
	/// associated with the ``CoreDelegate`` that is calling the callback. 
	/// - Returns: the ``CoreDelegate`` that has called the last callback    
	public var currentDelegate: CoreDelegate?
	{
	
			let cObject = linphone_core_get_current_callbacks(cPtr)
			let result = belle_sip_object_data_get(UnsafeMutablePointer(cObject), "swiftRef")
			if (result != nil) {
				return Unmanaged<CoreDelegateManager>.fromOpaque(result!).takeUnretainedValue().delegate
				}
			return nil
			
	}
		
	
	/// Get the effective video definition provided by the camera for the captured
	/// video. 
	/// When preview is disabled or not yet started this function returns a 0x0 video
	/// definition. 
	/// - Returns: The captured ``VideoDefinition``.   
	/// - See also: ``setPreviewVideoDefinition(videoDefinition:)`` 
	public var currentPreviewVideoDefinition: VideoDefinition?
	{
	
						let cPointer = linphone_core_get_current_preview_video_definition(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = VideoDefinition.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	/// Enables or disables database usage. 
	/// This function can only be called before starting the core up 
	/// - Parameter value: a boolean to indicate whether the database is to be enabled. 
	
	/// Returns whether the database is enabled. 
	/// - Returns: a boolean indicating the enablement of the database. 
	public var databaseEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_database_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_database(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets the default account. 
	/// This default account must be part of the list of already entered ``Account``.
	/// Toggling it as default will make ``Core`` default to the identity associated
	/// with the account in all incoming and outgoing calls if the destination SIP uri
	/// does not explicitely match any other accounts. 
	/// - Parameter account: The account to use as the default one.    
	
	/// Returns the default account, that is the one used to determine the current
	/// identity. 
	/// - Returns: The default account.    
	public var defaultAccount: Account?
	{
	
		get
		{ 
						let cPointer = linphone_core_get_default_account(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Account.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_core_set_default_account(cPtr, newValue?.cPtr)
		}
	}
		
	/// Set the default conference layout. 
	/// - Parameter value: layout 
	
	/// Gets the default conference layout. 
	/// - Returns: conference layout 
	public var defaultConferenceLayout: Conference.Layout
	{
	
		get
		{ 
						return Conference.Layout(rawValue: Int(linphone_core_get_default_conference_layout(cPtr).rawValue))!
		}
		set
		{
			linphone_core_set_default_conference_layout(cPtr, LinphoneConferenceLayout(rawValue: CUnsignedInt(newValue.rawValue)))
		}
	}
		
	/// Set the default ephemeral lifetime in seconds once read. 
	/// - Parameter value: lifetime of ephemeral messages in seconds once read 
	
	/// Gets the default lifetime of ephemeral messages in seconds once they are read. 
	/// - Returns: lifetime of ephemeral messages in seconds once read 
	public var defaultEphemeralLifetime: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_default_ephemeral_lifetime(cPtr))
		}
		set
		{
			linphone_core_set_default_ephemeral_lifetime(cPtr, (newValue))
		}
	}
		
	/// Set the default ephemeral lifetime in seconds when not read. 
	/// - Parameter value: lifetime of ephemeral messages in seconds when not read 
	
	/// Gets the default lifetime of ephemeral messages in seconds when they are not
	/// read. 
	/// - Returns: lifetime of ephemeral messages in seconds when not read 
	public var defaultEphemeralNotReadLifetime: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_default_ephemeral_not_read_lifetime(cPtr))
		}
		set
		{
			linphone_core_set_default_ephemeral_not_read_lifetime(cPtr, (newValue))
		}
	}
		
	
	/// Retrieves the first list of ``Friend`` from the core. 
	/// - Returns: the first ``FriendList`` object or nil.    
	public var defaultFriendList: FriendList?
	{
	
						let cPointer = linphone_core_get_default_friend_list(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = FriendList.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	/// Sets the given ``AudioDevice`` as default input for next calls. 
	/// - Parameter audioDevice: The ``AudioDevice``    
	
	/// Gets the default input audio device. 
	/// - Returns: The default input audio device    
	public var defaultInputAudioDevice: AudioDevice?
	{
	
		get
		{ 
						let cPointer = linphone_core_get_default_input_audio_device(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = AudioDevice.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_core_set_default_input_audio_device(cPtr, newValue?.cPtr)
		}
	}
		
	/// Sets the given ``AudioDevice`` as default output for next calls. 
	/// - Parameter audioDevice: The ``AudioDevice``    
	
	/// Gets the default output audio device. 
	/// - Returns: The default output audio device    
	public var defaultOutputAudioDevice: AudioDevice?
	{
	
		get
		{ 
						let cPointer = linphone_core_get_default_output_audio_device(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = AudioDevice.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_core_set_default_output_audio_device(cPtr, newValue?.cPtr)
		}
	}
		
	/// Sets the default proxy. 
	/// This default proxy must be part of the list of already entered ``ProxyConfig``.
	/// Toggling it as default will make ``Core`` use the identity associated with the
	/// proxy configuration in all incoming and outgoing calls. 
	/// - Parameter config: The proxy configuration to use as the default one.    
	/// - Deprecated: 04/09/2024 Use ``setDefaultAccount(account:)`` 
	
	/// Returns the default proxy configuration, that is the one used to determine the
	/// current identity. 
	/// - Returns: The default proxy configuration.    
	/// - Deprecated: 04/09/2024 Use ``getDefaultAccount()`` 
	public var defaultProxyConfig: ProxyConfig?
	{
	@available(*, deprecated)
		get
		{ 
						let cPointer = linphone_core_get_default_proxy_config(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = ProxyConfig.getSwiftObject(cObject:cPointer!)
			return result
		}
	@available(*, deprecated)
		set
		{
			linphone_core_set_default_proxy_config(cPtr, newValue?.cPtr)
		}
	}
		
	
	/// Gets the name of the default mediastreamer2 filter used for rendering video on
	/// the current platform. 
	/// This is for advanced users of the library, mainly to expose mediastreamer video
	/// filter name and status. 
	/// - Returns: The default video display filter.    
	public var defaultVideoDisplayFilter: String
	{
	
			
			let cPointer = linphone_core_get_default_video_display_filter(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	/// Set the in delayed timeout in seconds. 
	/// After this timeout period, a delayed call (internal call initialisation or
	/// resolution) is resumed. 
	/// - Parameter seconds: The new delayed timeout
	/// - Deprecated: 04/09/2024 Obscure. 
	
	/// Gets the delayed timeout See ``setDelayedTimeout(seconds:)`` for details. 
	/// - Returns: The current delayed timeout in seconds
	/// - Deprecated: 04/09/2024 Obscure. 
	public var delayedTimeout: Int
	{
	@available(*, deprecated)
		get
		{ 
						return Int(linphone_core_get_delayed_timeout(cPtr))
		}
	@available(*, deprecated)
		set
		{
			linphone_core_set_delayed_timeout(cPtr, CInt(newValue))
		}
	}
		
	/// Tells the core the device current orientation. 
	/// This can be used by capture filters on mobile devices to select between
	/// portrait/landscape mode and to produce properly oriented images. The exact
	/// meaning of the value in rotation if left to each device specific
	/// implementations. IOS supported values are 0 for UIInterfaceOrientationPortrait
	/// and 270 for UIInterfaceOrientationLandscapeRight. 
	/// - Parameter rotation: The orientation to use 
	
	/// Gets the current device orientation. 
	/// - Returns: The current device orientation
	/// - See also: ``setDeviceRotation(rotation:)`` 
	public var deviceRotation: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_device_rotation(cPtr))
		}
		set
		{
			linphone_core_set_device_rotation(cPtr, CInt(newValue))
		}
	}
		
	/// Setup a new digest authentication policy applicable for SIP and HTTP. 
	/// - Parameter policy: a ``DigestAuthenticationPolicy``    
	/// - See also: ``Factory/createDigestAuthenticationPolicy()`` 
	
	/// Get the current digest authentication policy applicable for SIP and HTTP. 
	/// - Returns: The current digest authentication policy.    
	public var digestAuthenticationPolicy: DigestAuthenticationPolicy?
	{
	
		get
		{ 
						let cPointer = linphone_core_get_digest_authentication_policy(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = DigestAuthenticationPolicy.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_core_set_digest_authentication_policy(cPtr, newValue?.cPtr)
		}
	}
		
	/// Calling this method with disable=true will cause the microhone to be completely
	/// deactivated when muted, when given possible by the implementation on the
	/// platform on which liblinphone is running. 
	/// Otherwise, sound recording remains active but silence is sent instead of
	/// recorded audio. Playback sound will be briefly interrupted while the audio is
	/// reconfigured. Currently only implemented for IOS, it will also disable Apple's
	/// microphone recording indicator when microphone is muted.
	/// - Parameter disable: True if you wish to entirely stop the audio recording when
	/// muting the microphone. 
	
	/// Get whether the microphone will be completely deactivated when muted. 
	/// Please refer to ``setDisableRecordOnMute(disable:)``.
	/// - Returns: True if you wish to entirely stop the audio recording when muting
	/// the microphone. 
	public var disableRecordOnMute: Bool
	{
	
		get
		{ 
						return linphone_core_get_disable_record_on_mute(cPtr) != 0
		}
		set
		{
			linphone_core_set_disable_record_on_mute(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Enable or disable DNS search (use of local domain if the fully qualified name
	/// did return results). 
	/// - Parameter enable: true to enable DNS search, false to disable it. 
	
	/// Tells whether DNS search (use of local domain if the fully qualified name did
	/// return results) is enabled. 
	/// - Returns: true if DNS search is enabled, false if disabled. 
	public var dnsSearchEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_dns_search_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_dns_search(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Forces liblinphone to use the supplied list of dns servers, instead of system's
	/// ones. 
	/// - Parameter servers: A list of strings containing the IP addresses of DNS
	/// servers to be used.   Setting to nil restores default behaviour, which is to
	/// use the DNS server list provided by the system. The list is copied internally. 
	///   
	
	public var dnsServers: [String] = []
	{
	
		willSet
		{
			var cList: UnsafeMutablePointer<bctbx_list_t>? = nil
			for data in newValue {
				let sData:NSString = data as NSString
				cList = bctbx_list_append(cList, unsafeBitCast(sData.utf8String, to: UnsafeMutablePointer<CChar>.self))
			}
			linphone_core_set_dns_servers(cPtr, cList)
		}
	}
		
	/// Forces liblinphone to use the supplied list of dns servers, instead of system's
	/// ones and set dns_set_by_app at true or false according to value of servers
	/// list. 
	/// - See also: ``getDnsSetByApp()`` 
	/// - Parameter servers: A list of strings containing the IP addresses of DNS
	/// servers to be used.   Setting to nil restores default behaviour, which is to
	/// use the DNS server list provided by the system. The list is copied internally. 
	///   
	
	public var dnsServersApp: [String] = []
	{
	
		willSet
		{
			var cList: UnsafeMutablePointer<bctbx_list_t>? = nil
			for data in newValue {
				let sData:NSString = data as NSString
				cList = bctbx_list_append(cList, unsafeBitCast(sData.utf8String, to: UnsafeMutablePointer<CChar>.self))
			}
			linphone_core_set_dns_servers_app(cPtr, cList)
		}
	}
		
	
	/// Tells if the DNS was set by an application. 
	/// - Returns: true if DNS was set by app, false otherwise. 
	public var dnsSetByApp: Bool
	{
	
						return linphone_core_get_dns_set_by_app(cPtr) != 0

	}
		
	/// Enable or disable DNS SRV resolution. 
	/// - Parameter enable: true to enable DNS SRV resolution, false to disable it. 
	
	/// Tells whether DNS SRV resolution is enabled. 
	/// - Returns: true if DNS SRV resolution is enabled, false if disabled. 
	public var dnsSrvEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_dns_srv_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_dns_srv(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets maximum available download bandwidth This is IP bandwidth, in kbit/s. 
	/// This information is signaled to other parties during calls (within SDP
	/// messages) so that the remote end can have sufficient knowledge to properly
	/// configure its audio & video codec output bitrate to not overflow available
	/// bandwidth. By default, the download and upload bandwidth are unknowns (set to
	/// zero), in which case adaptive algorithms are run during calls in order to
	/// detect available bandwidth and adapt audio and video bitrate usage. - See also:
	/// ``enableAdaptiveRateControl(enabled:)``.
	/// - Parameter bandwidth: the bandwidth in kbits/s, 0 if unknown. 
	
	/// Retrieve the maximum available download bandwidth. 
	/// This value was set by ``setDownloadBandwidth(bandwidth:)``. 
	/// - Returns: the download bandiwdth in kbits/s, 0 for unknown. 
	public var downloadBandwidth: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_download_bandwidth(cPtr))
		}
		set
		{
			linphone_core_set_download_bandwidth(cPtr, CInt(newValue))
		}
	}
		
	/// Set audio packetization time linphone expects to receive from peer. 
	/// A value of zero means that ptime is not specified. 
	/// - Parameter ptime: the download packetization time to set 
	
	/// Get audio packetization time linphone expects to receive from peer. 
	/// A value of zero means that ptime is not specified. 
	/// - Returns: the download packetization time set 
	public var downloadPtime: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_download_ptime(cPtr))
		}
		set
		{
			linphone_core_set_download_ptime(cPtr, CInt(newValue))
		}
	}
		
	
	/// Gets the currently stored calibration delay for the software echo cancellation. 
	/// - Returns: the current calibration value, -1 if it failed, 0 if not done or not
	/// needed, a positive value if a software echo canceller is required after running
	/// ``startEchoCancellerCalibration()``. 
	public var echoCancellationCalibration: Int
	{
	
						return Int(linphone_core_get_echo_cancellation_calibration(cPtr))

	}
		
	/// Enables or disables echo cancellation. 
	/// Value is saved and used for subsequent calls. This actually controls software
	/// echo cancellation only. When 'enable' is set to false, software echo
	/// cancellation is disabled, but hardware one, if available, remains activated.
	/// When set to true, software echo cancellation is activated in either of these
	/// two conditions:
	/// -hardware echo cancellation is not available
	/// -an echo calibration procedure was run successfully, which implicitly cause the
	/// software echo canceller to be preferred over hardware one. 
	
	/// Returns true if echo cancellation is enabled. 
	/// - Returns: A boolean value telling whether echo cancellation is enabled or
	/// disabled 
	public var echoCancellationEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_echo_cancellation_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_echo_cancellation(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets the name of the mediastreamer2 filter to be used for echo cancelling. 
	/// - Parameter filtername: The name of the mediastreamer2 filter to be used for
	/// echo cancellation.    
	
	/// Gets the name of the mediastreamer2 filter used for echo cancelling. 
	/// - Returns: The name of the mediastreamer2 filter used for echo cancellation.    
	public var echoCancellerFilterName: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_core_get_echo_canceller_filter_name(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_core_set_echo_canceller_filter_name(cPtr, newValue)
		}
	}
		
	/// Enables or disable echo limiter. 
	/// "Echo limiter" refers to an algorithm that creates half-duplex conversation in
	/// order to suppress echo. It is experimental and shall be used only in rare cases
	/// where echo cancellation cannot perform because of non-linear speaker/mic
	/// coupling. You shall not expected good audio quality with the echo limiter. 
	/// - Parameter enable: true to enable echo limiter, false to disable it. 
	
	/// Tells whether echo limiter is enabled. 
	/// Enables or disable echo limiter. "Echo limiter" refers to an algorithm that
	/// creates half-duplex conversation in order to suppress echo. It is experimental
	/// and shall be used only in rare cases where echo cancellation cannot perform
	/// because of non-linear speaker/mic coupling. You shall not expected good audio
	/// quality with the echo limiter. 
	/// - Returns: true if the echo limiter is enabled, false otherwise. 
	public var echoLimiterEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_echo_limiter_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_echo_limiter(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// sets the state of the EKT plugin in the Linphone core instance. 
	/// - Parameter ektPluginLoaded: whether the EKT plugin is loaded or not 
	
	public var ektPluginLoaded: Bool?
	{
	
		willSet
		{
			linphone_core_set_ekt_plugin_loaded(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Enable the deletion of empty chatrooms (i.e. 
	/// chatrooms with no participants) 
	/// - Parameter enable: A boolean value telling whether to enable or disable the
	/// deletion of chat rooms with no participants in (Applicable to servers only) it 
	
	/// Tells whether empty chat rooms are deleted or not. 
	/// - Returns: A boolean value telling whether the deletion of empty chatrooms is
	/// enabled or not (Applicable to servers only) 
	public var emptyChatroomsDeletionEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_empty_chatrooms_deletion_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_empty_chatrooms_deletion(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Enable or disable the UPDATE method support. 
	/// - Parameter value: Enable or disable it 
	
	public var enableSipUpdate: Int = 0
	{
	
		willSet
		{
			linphone_core_set_enable_sip_update(cPtr, CInt(newValue))
		}
	}
		
	/// Sets the policy for ephemeral chat messages. 
	/// See ``EphemeralChatMessagePolicy`` for more details. 
	/// - Parameter policy: the ``EphemeralChatMessagePolicy`` to use. 
	
	/// Gets the current policy for ephemeral chat messages. 
	/// See ``EphemeralChatMessagePolicy`` for more details. 
	/// - Returns: the current ``EphemeralChatMessagePolicy``. 
	public var ephemeralChatMessagePolicy: EphemeralChatMessagePolicy
	{
	
		get
		{ 
						return EphemeralChatMessagePolicy(rawValue: Int(linphone_core_get_ephemeral_chat_message_policy(cPtr).rawValue))!
		}
		set
		{
			linphone_core_set_ephemeral_chat_message_policy(cPtr, LinphoneEphemeralChatMessagePolicy(rawValue: CUnsignedInt(newValue.rawValue)))
		}
	}
		
	/// Do not use, this function does nothing. 
	/// - Parameter bandwidth: the bandwidth in kbits/s, 0 for infinite
	/// - Deprecated: 04/09/2024 this function does nothing. 
	
	public var expectedBandwidth: Int = 0
	{
	
	@available(*, deprecated)
		willSet
		{
			linphone_core_set_expected_bandwidth(cPtr, CInt(newValue))
		}
	}
		
	
	/// Returns the list of all audio devices. 
	/// - Returns:   A list of all ``AudioDevice``       
	public var extendedAudioDevices: [AudioDevice]
	{
	
						var swiftList = [AudioDevice]()
			let cList = linphone_core_get_extended_audio_devices(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(AudioDevice.getSwiftObject(cObject: data))
				listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
				bctbx_list_free_with_data(cList, belle_sip_object_unref)
			return swiftList

	}
		
	/// Enables the flexible FEC feature (RFC8627) for video calls to recover RTP
	/// packet loss. 
	/// - Parameter enable: A boolean value telling whether to enable or disable the
	/// flexible FEC feature 
	
	/// Tells whether the flexible FEC feature (RFC8627) is enabled for this ``Core``
	/// object. 
	/// - Returns: A boolean value telling whether the flexible FEC feature is enabled
	/// or not for this ``Core`` object 
	public var fecEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_fec_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_fec(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Globally sets an http file transfer server to be used for content type
	/// application/vnd.gsma.rcs-ft-http+xml. 
	/// Url may be like: "https://file.linphone.org/upload.php". This value can also be
	/// set for a dedicated account using
	/// linphone_account_params_set_file_transfer_server. 
	/// - Parameter serverUrl: URL of the file server.    
	
	/// Gets the globaly set http file transfer server to be used for content type
	/// application/vnd.gsma.rcs-ft-http+xml. 
	/// Url may be like: "https://file.linphone.org/upload.php". 
	/// - Returns: URL of the file server.    
	public var fileTransferServer: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_core_get_file_transfer_server(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_core_set_file_transfer_server(cPtr, newValue)
		}
	}
		
	/// Artificially cause the relay path to be selected when ICE is used. 
	/// This is mainly a function for test, for example to validate that the relay
	/// service (ever TURN or media-aware SIP proxy) is working as expected. Indeed, in
	/// many cases a path through host or server reflexive candidate will be found by
	/// ICE, which makes difficult to make sure that the relay service is working as
	/// expected. 
	/// - Parameter enable: boolean value 
	
	/// Indicates whether the ICE relay path is forcibly selected. 
	/// - Returns: a boolean value indicating whether forced relay is enabled.
	/// - See also: ``enableForcedIceRelay(enable:)``. 
	public var forcedIceRelayEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_forced_ice_relay_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_forced_ice_relay(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets whether or not to start friend lists subscription when in foreground. 
	/// - Parameter enable: whether or not to enable the feature 
	
	/// Returns whether or not friend lists subscription are enabled. 
	/// - Returns: whether or not the feature is enabled 
	public var friendListSubscriptionEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_friend_list_subscription_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_friend_list_subscription(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets the database filename where friends will be stored. 
	/// If the file does not exist, it will be created.
	/// - Parameter path: filesystem path.    
	/// - Deprecated: 27/10/2023 Friends are now stored in the main db 
	
	/// Gets the database filename where friends will be stored. 
	/// - Returns: filesystem path.    
	/// - Deprecated: 27/10/2023 Friends are now stored in the main db 
	public var friendsDatabasePath: String?
	{
	@available(*, deprecated)
		get
		{ 
			
			let cPointer = linphone_core_get_friends_database_path(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
	@available(*, deprecated)
		set
		{
			linphone_core_set_friends_database_path(cPtr, newValue)
		}
	}
		
	
	/// Retrieves the list of ``FriendList`` from the core. 
	/// - Returns: A list of ``FriendList``.      
	public var friendsLists: [FriendList]
	{
	
						var swiftList = [FriendList]()
			let cList = linphone_core_get_friends_lists(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(FriendList.getSwiftObject(cObject: data))
				listTemp = UnsafePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
			return swiftList

	}
		
	
	/// Retrieve the list of future conference information on DB. 
	/// - Returns: The list of future conference infos  .       
	public var futureConferenceInformationList: [ConferenceInfo]
	{
	
						var swiftList = [ConferenceInfo]()
			let cList = linphone_core_get_future_conference_information_list(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(ConferenceInfo.getSwiftObject(cObject: data))
				listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
				bctbx_list_free_with_data(cList, belle_sip_object_unref)
			return swiftList

	}
		
	/// Enable RFC3389 generic comfort noise algorithm (CN payload type). 
	/// It is disabled by default, because this algorithm is only relevant for legacy
	/// codecs (PCMU, PCMA, G722). Enablement requires a SDK built with full G729
	/// support: -DENABLE_G729=ON -DENABLE_G729B_CNG=ON . - Warning: : the G729 support
	/// is not included in Liblinphone default licence - the purchase of a license
	/// extension is required. 
	/// - Parameter enabled: true if enabled, false otherwise. 
	
	/// Returns enablement of RFC3389 generic comfort noise algorithm. 
	/// - Returns: true if generic comfort noise is enabled, false otherwise. 
	public var genericComfortNoiseEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_generic_comfort_noise_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_generic_comfort_noise(cPtr, newValue==true ? 1:0)
		}
	}
		
	
	/// Returns the global state of core. 
	/// - Returns: a ``GlobalState`` enum.    
	public var globalState: GlobalState
	{
	
						return GlobalState(rawValue: Int(linphone_core_get_global_state(cPtr).rawValue))!

	}
		
	/// Sets whether to keep GRUU parameter in the conference addresses. 
	/// - Parameter enabled: true if enabled, false otherwise. 
	/// - Warning: This setting will also remove the GRUU parameter from all conference
	/// and chat room addresses stored in the database at startup. Setting it to false
	/// after it being set to true earlier on does not restore the previous state of
	/// the database 
	
	/// Returns whether the gr parameter is kept in the conference address. 
	/// - Returns: true if the "gr" parameter is kept in the conference address, false
	/// otherwise. 
	/// - See also: ``enableGruuInConferenceAddress(enabled:)`` for more informations 
	public var gruuInConferenceAddressEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_gruu_in_conference_address_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_gruu_in_conference_address(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Tells ``Core`` to guess local hostname automatically in primary contact. 
	/// - Parameter enable: whether to enable the guess hostname feature or not 
	
	/// Returns true if hostname part of primary contact is guessed automatically. 
	/// - Returns: true if guess hostname enabled, false otherwise. 
	public var guessHostname: Bool
	{
	
		get
		{ 
						return linphone_core_get_guess_hostname(cPtr) != 0
		}
		set
		{
			linphone_core_set_guess_hostname(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets http proxy address to be used for signaling during next channel
	/// connection. 
	/// Use ``setNetworkReachable(reachable:)`` FASLE/true to force channel restart. 
	/// - Parameter host: Hostname of IP adress of the http proxy (can be nil to
	/// disable).    
	
	/// Gets http proxy address to be used for signaling. 
	/// - Returns: hostname of IP adress of the http proxy (can be nil to disable).    
	public var httpProxyHost: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_core_get_http_proxy_host(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_core_set_http_proxy_host(cPtr, newValue)
		}
	}
		
	/// Sets http proxy port to be used for signaling. 
	/// - Parameter port: of the http proxy. 
	
	/// Gets http proxy port to be used for signaling. 
	/// - Returns: port of the http proxy. 
	public var httpProxyPort: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_http_proxy_port(cPtr))
		}
		set
		{
			linphone_core_set_http_proxy_port(cPtr, CInt(newValue))
		}
	}
		
	
	/// Gets the default identity SIP address. 
	/// This is an helper function. If no default proxy is set, this will return the
	/// primary contact ( see ``getPrimaryContact()`` ). If a default proxy is set it
	/// returns the registered identity on the proxy. 
	/// - Returns: The default identity SIP address.    
	public var identity: String
	{
	
			
			let cPointer = linphone_core_get_identity(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Gets the ``ImNotifPolicy`` object controlling the instant messaging delivery
	/// and displayed service notifications. 
	/// - Returns: A ``ImNotifPolicy`` object.    
	public var imNotifPolicy: ImNotifPolicy?
	{
	
						let cPointer = linphone_core_get_im_notif_policy(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = ImNotifPolicy.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	/// Set the IMDN resend period. 
	/// It is the number of seconds after the first attempt to send, an IMDN message is
	/// sent again on startup if failed earlier on 
	/// - Parameter seconds: number of seconds after the first attempt to send an IMDN,
	/// it is retried at startup. A negative value means all IMDNs are resent at
	/// startup. 
	
	/// Gets the IMDN resend period. 
	/// - Returns: the number of second to resend an failed IMDN message 
	public var imdnResendPeriod: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_imdn_resend_period(cPtr))
		}
		set
		{
			linphone_core_set_imdn_resend_period(cPtr, (newValue))
		}
	}
		
	/// Sets the threshold for sending IMDN to all participants to a ``ChatRoom``. 
	/// - Parameter threshold: the threshold for sending IMDN to all participants to a
	/// ``ChatRoom`` 
	
	/// Returns the threshold for sending IMDN to all participants to a ``ChatRoom``. 
	/// - Returns: An integer value telling the threshold for sending IMDN to all
	/// participants to a ``ChatRoom`` 
	public var imdnToEverybodyThreshold: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_imdn_to_everybody_threshold(cPtr))
		}
		set
		{
			linphone_core_set_imdn_to_everybody_threshold(cPtr, CInt(newValue))
		}
	}
		
	/// Sets the maximum duration of a call in seconds. 
	/// After this timeout period, the call is automatically hangup. A value of 0
	/// disables this feature. 
	/// - Parameter seconds: The new timeout in seconds 
	
	/// Gets the maximum duration of a call. 
	/// See ``setInCallTimeout(seconds:)`` for details. 
	/// - Returns: The current in call timeout in seconds 
	public var inCallTimeout: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_in_call_timeout(cPtr))
		}
		set
		{
			linphone_core_set_in_call_timeout(cPtr, CInt(newValue))
		}
	}
		
	/// Sets the incoming call timeout in seconds. 
	/// If an incoming call isn't answered for this timeout period, it is automatically
	/// declined. 
	/// - Parameter seconds: The new timeout in seconds 
	
	/// Returns the incoming call timeout See ``setIncTimeout(seconds:)`` for details. 
	/// - Returns: The current incoming call timeout in seconds 
	public var incTimeout: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_inc_timeout(cPtr))
		}
		set
		{
			linphone_core_set_inc_timeout(cPtr, CInt(newValue))
		}
	}
		
	/// Sets the given ``AudioDevice`` as input for all active calls. 
	/// - Parameter audioDevice: The ``AudioDevice``. nil does nothing.    
	
	/// Gets the input audio device for the current call. 
	/// - Returns: The input audio device for the current or first call, nil if there
	/// is no call.    
	public var inputAudioDevice: AudioDevice?
	{
	
		get
		{ 
						let cPointer = linphone_core_get_input_audio_device(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = AudioDevice.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_core_set_input_audio_device(cPtr, newValue?.cPtr)
		}
	}
		
	/// Turns IPv6 support on or off. 
	/// - Parameter enable: A boolean value telling whether to enable IPv6 support 
	
	/// Tells whether IPv6 is enabled or not. 
	/// - Returns: A boolean value telling whether IPv6 is enabled or not 
	public var ipv6Enabled: Bool
	{
	
		get
		{ 
						return linphone_core_ipv6_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_ipv6(cPtr, newValue==true ? 1:0)
		}
	}
		
	
	/// Gets if the automatic download of incoming icalendars is enabled or not. 
	/// - Returns: true if icalendars will be automatically downloaded, false otherwise.
	/// - Deprecated: 16/12/2021 Use ``autoDownloadIcalendarsEnabled()`` instead. 
	@available(*, deprecated)
	public var isAutoDownloadIcalendarsEnabled: Bool
	{
	
						return linphone_core_is_auto_download_icalendars_enabled(cPtr) != 0

	}
		
	
	/// Gets if the auto download for incoming voice recordings is enabled or not. 
	/// - Returns: true if voice recordings will be automatically downloaded, false
	/// otherwise.
	/// - Deprecated: 16/12/2021 Use ``autoDownloadVoiceRecordingsEnabled()`` instead. 
	@available(*, deprecated)
	public var isAutoDownloadVoiceRecordingsEnabled: Bool
	{
	
						return linphone_core_is_auto_download_voice_recordings_enabled(cPtr) != 0

	}
		
	
	/// Gets whether auto iterate is enabled or not (Android & iOS only). 
	/// - Returns: true if ``iterate()`` is scheduled automatically, false otherwise
	/// - Deprecated: 16/12/2021 Use ``autoIterateEnabled()`` instead. 
	@available(*, deprecated)
	public var isAutoIterateEnabled: Bool
	{
	
						return linphone_core_is_auto_iterate_enabled(cPtr) != 0

	}
		
	
	/// Check whether the device is echo canceller calibration is required. 
	/// - Returns: true if it is required, false otherwise 
	public var isEchoCancellerCalibrationRequired: Bool
	{
	
						return linphone_core_is_echo_canceller_calibration_required(cPtr) != 0

	}
		
	
	/// Gets if the EKT plugin is currently loaded in the Linphone core instance. 
	/// - Returns: true if the EKT plugin is loaded 
	public var isEktPluginLoaded: Bool
	{
	
						return linphone_core_is_ekt_plugin_loaded(cPtr) != 0

	}
		
	
	/// Returns whether or not friend lists subscription are enabled. 
	/// - Returns: whether or not the feature is enabled 
	/// - Deprecated: 16/12/2021 Use ``friendListSubscriptionEnabled()`` instead. 
	@available(*, deprecated)
	public var isFriendListSubscriptionEnabled: Bool
	{
	
						return linphone_core_is_friend_list_subscription_enabled(cPtr) != 0

	}
		
	
	/// Gets whether the Core is considering itself in background or not. 
	/// The Core foreground/background state depends on the last call made to
	/// ``enterBackground()`` or ``enterForeground()``. 
	/// - Returns: true if the Core is in background, false otherwise. 
	public var isInBackground: Bool
	{
	
						return linphone_core_is_in_background(cPtr) != 0

	}
		
	
	/// Indicates whether the local participant is part of a conference. 
	/// - Warning: That function automatically fails in the case of conferences using a
	/// conferencet server (focus). If you use such a conference, you should use
	/// ``Conference/removeParticipant(uri:)`` instead. 
	/// - Returns: true if the local participant is in a conference, false otherwise. 
	/// - Deprecated: 09/03/2021 Use ``Conference/isIn()`` instead. 
	@available(*, deprecated)
	public var isInConference: Bool
	{
	
						return linphone_core_is_in_conference(cPtr) != 0

	}
		
	
	/// Tells whether there is an incoming invite pending. 
	/// - Returns: A boolean telling whether an incoming invite is pending or not. 
	public var isIncomingInvitePending: Bool
	{
	
						return linphone_core_is_incoming_invite_pending(cPtr) != 0

	}
		
	
	/// Checks if the configured media encryption is mandatory or not. 
	/// - Returns: true if media encryption is mandatory; false otherwise. 
	public var isMediaEncryptionMandatory: Bool
	{
	
						return linphone_core_is_media_encryption_mandatory(cPtr) != 0

	}
		
	
	/// Returns whether the native ringing is enabled or not. 
	/// - Returns: True if we use the native ringing, false otherwise
	/// - Deprecated: 16/12/2021 Use ``nativeRingingEnabled()`` instead. 
	@available(*, deprecated)
	public var isNativeRingingEnabled: Bool
	{
	
						return linphone_core_is_native_ringing_enabled(cPtr) != 0

	}
		
	
	/// return network state either as positioned by the application or by linphone
	/// itself. 
	/// - Returns: true if network is reachable, false otherwise 
	public var isNetworkReachable: Bool
	{
	
						return linphone_core_is_network_reachable(cPtr) != 0

	}
		
	
	/// Gets whether push notifications are available or not (Android & iOS only). 
	/// - Returns: true if push notifications are available, false otherwise 
	public var isPushNotificationAvailable: Bool
	{
	
						return linphone_core_is_push_notification_available(cPtr) != 0

	}
		
	
	/// Gets whether push notifications are enabled or not (Android & iOS only). 
	/// If not, the app will have to handle all the push-related settings for each
	/// accounts 
	/// - Returns: true if push notifications are enabled, false otherwise
	/// - Deprecated: 16/12/2021 Use ``pushNotificationEnabled()`` instead. 
	@available(*, deprecated)
	public var isPushNotificationEnabled: Bool
	{
	
						return linphone_core_is_push_notification_enabled(cPtr) != 0

	}
		
	
	/// Gets if the record aware feature is enabled or not. 
	/// - Returns: true if the record aware feature is enabled, false otherwise. 
	/// - Deprecated: 16/12/2021 Use ``recordAwareEnabled()`` instead. 
	@available(*, deprecated)
	public var isRecordAwareEnabled: Bool
	{
	
						return linphone_core_is_record_aware_enabled(cPtr) != 0

	}
		
	
	/// Returns whether or not sender name is hidden in a forwarded message. 
	/// - Returns: whether or not the feature 
	public var isSenderNameHiddenInForwardMessage: Bool
	{
	
						return linphone_core_is_sender_name_hidden_in_forward_message(cPtr) != 0

	}
		
	
	/// Get whether the tls server certificate must be verified when connecting to a
	/// SIP/TLS server. 
	/// - Returns: True if the tls server certificate must be verified 
	public var isVerifyServerCertificates: Bool
	{
	
						return linphone_core_is_verify_server_certificates(cPtr) != 0

	}
		
	
	/// Get whether the tls server certificate common name must be verified when
	/// connecting to a SIP/TLS server. 
	/// - Returns: True if the tls server certificate common name must be verified 
	public var isVerifyServerCn: Bool
	{
	
						return linphone_core_is_verify_server_cn(cPtr) != 0

	}
		
	
	/// Gets whether the device will vibrate while an incoming call is ringing (Android
	/// only). 
	/// - Returns: true if the device will vibrate (if possible), false otherwise
	/// - Deprecated: 16/12/2021 Use ``vibrationOnIncomingCallEnabled()`` instead. 
	@available(*, deprecated)
	public var isVibrationOnIncomingCallEnabled: Bool
	{
	
						return linphone_core_is_vibration_on_incoming_call_enabled(cPtr) != 0

	}
		
	/// Enables signaling keep alive, small udp packet sent periodically to keep udp
	/// NAT association. 
	/// - Parameter enable: A boolean value telling whether signaling keep alive is to
	/// be enabled 
	
	/// Is signaling keep alive enabled. 
	/// - Returns: A boolean value telling whether signaling keep alive is enabled 
	public var keepAliveEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_keep_alive_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_keep_alive(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Set a label - for logging/troubleshooting purpose - to the core object. 
	/// This label is used by the logger to give context. When running an application
	/// with several ``Core`` objects, (such as a test), it is useful to enhance the
	/// log's readability'. 
	/// - Parameter label: a developer-friendly label. 
	
	/// Get the label assigned to the ``Core``. 
	/// The default value is nil (no label). 
	/// - Returns: the assigned label. 
	public var label: String
	{
	
		get
		{ 
			
			let cPointer = linphone_core_get_label(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_core_set_label(cPtr, newValue)
		}
	}
		
	
	/// Gets the latest outgoing call log. 
	/// Conference calls are not returned by this function! Requires ENABLE_DB_STORAGE
	/// to work. 
	/// - Returns: The last outgoing call log if any.    
	public var lastOutgoingCallLog: CallLog?
	{
	
						let cPointer = linphone_core_get_last_outgoing_call_log(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = CallLog.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Returns a list of entered LDAPs. 
	/// Items must be freed with linphone_ldap_unref 
	/// - Returns:         
	/// - Deprecated: 18/11/2024 use ``getRemoteContactDirectories()`` instead. 
	@available(*, deprecated)
	public var ldapList: [Ldap]
	{
	
						var swiftList = [Ldap]()
			let cList = linphone_core_get_ldap_list(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(Ldap.getSwiftObject(cObject: data))
				listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
				bctbx_list_free_with_data(cList, belle_sip_object_unref)
			return swiftList

	}
		
	/// Tells to ``Core`` to use LIME X3DH. 
	/// - Parameter enable: A boolean value telling whether to enable or disable LIME
	/// X3DH 
	
	/// Tells wether LIME X3DH is enabled or not. 
	/// - Returns: The current lime state 
	public var limeX3DhEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_lime_x3dh_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_lime_x3dh(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Set the x3dh server url. 
	/// If empty, this function will disable LIME X3DH from core. Otherwise, or if
	/// different from the existing value, this will (re-)initialize the LIME X3DH
	/// engine. 
	/// - Parameter url: The x3dh server url.   
	/// - Deprecated: 26/08/2022 Use ``AccountParams/setLimeServerUrl(url:)`` instead. 
	
	/// Get the x3dh server url. 
	/// - Returns: The x3dh server url.   
	/// - Deprecated: 26/08/2022 Use ``AccountParams/getLimeServerUrl()`` instead. 
	public var limeX3DhServerUrl: String?
	{
	@available(*, deprecated)
		get
		{ 
			
			let cPointer = linphone_core_get_lime_x3dh_server_url(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
	@available(*, deprecated)
		set
		{
			linphone_core_set_lime_x3dh_server_url(cPtr, newValue)
		}
	}
		
	/// Sets the linphone specs list value telling what functionalities the linphone
	/// client supports. 
	/// The said "specs" (specifications) are pair of keyword/version that helps
	/// advertising the level feature supported by liblinphone. An application usually
	/// does not need to set this information, that is automatically assigned at
	/// ``Core`` start. Setting this is mainly for internal tests of backward
	/// compatibility. 
	/// - Parameter specs: The list of string specs to set.      
	
	/// Get the list of linphone specs string values representing what functionalities
	/// the linphone client supports. 
	/// - See also: ``setLinphoneSpecsList(specs:)`` 
	/// - Returns: A list of supported specs. The list must be freed with
	/// bctbx_list_free() after usage.         
	public var linphoneSpecsList: [String]
	{
	
		get
		{ 
						var swiftList = [String]()
			let cList = linphone_core_get_linphone_specs_list(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				swiftList.append(String(cString: unsafeBitCast(listTemp!.pointee.data, to: UnsafePointer<CChar>.self)))
				listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp!.pointee.next)
			}
				bctbx_list_free_with_data(cList, bctbx_free)
			return swiftList
		}
		set
		{
			var cList: UnsafeMutablePointer<bctbx_list_t>? = nil
			for data in newValue {
				let sData:NSString = data as NSString
				cList = bctbx_list_append(cList, unsafeBitCast(sData.utf8String, to: UnsafeMutablePointer<CChar>.self))
			}
			linphone_core_set_linphone_specs_list(cPtr, cList)
		}
	}
		
	
	/// Return the list of loaded plugins. 
	/// - Returns: the list of loaded plugins      
	public var loadedPlugins: [String]
	{
	
						var swiftList = [String]()
			let cList = linphone_core_get_loaded_plugins(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				swiftList.append(String(cString: unsafeBitCast(listTemp!.pointee.data, to: UnsafePointer<CChar>.self)))
				listTemp = UnsafePointer<bctbx_list_t>(listTemp!.pointee.next)
			}
			return swiftList

	}
		
	
	/// Special function to check if the local network permission has been granted by
	/// the user (useful for iOS). 
	/// The test performed by this function may popup the local network permission
	/// dialog, for that reason it could be a good idea to check it twice to conclude
	/// that the user has deny the permission. 
	/// - Returns: true if local permission request is granted, false otherwise. 
	public var localPermissionEnabled: Bool
	{
	
						return linphone_core_local_permission_enabled(cPtr) != 0

	}
		
	/// Sets the url of the server where to upload the collected log files. 
	/// - Parameter serverUrl: The url of the server where to upload the collected log
	/// files.    
	
	/// Gets the url of the server where to upload the collected log files. 
	/// - Returns: The url of the server where to upload the collected log files.    
	public var logCollectionUploadServerUrl: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_core_get_log_collection_upload_server_url(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_core_set_log_collection_upload_server_url(cPtr, newValue)
		}
	}
		
	/// Sets the maximum number of call logs to retrieve when using ``getCallLogs()``
	/// or ``Account/getCallLogs()``. 
	/// - Parameter max: the maximum number of call logs, use -1 to get them all. 
	
	/// Gets the maximum number of call logs retrieved when using ``getCallLogs()`` or
	/// ``Account/getCallLogs()``. 
	/// - Returns: the maximum number of call logs that will be returned. -1 will
	/// return them all. 
	public var maxCallLogs: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_max_call_logs(cPtr))
		}
		set
		{
			linphone_core_set_max_call_logs(cPtr, CInt(newValue))
		}
	}
		
	/// Sets the maximum number of simultaneous calls Linphone core can manage at a
	/// time. 
	/// All new call above this limit are declined with a busy answer 
	/// - Parameter max: number of simultaneous calls 
	
	/// Gets the maximum number of simultaneous calls Linphone core can manage at a
	/// time. 
	/// All new call above this limit are declined with a busy answer 
	/// - Returns: maximum number of simultaneous calls 
	public var maxCalls: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_max_calls(cPtr))
		}
		set
		{
			linphone_core_set_max_calls(cPtr, CInt(newValue))
		}
	}
		
	/// Sets the size under which incoming files in chat messages will be downloaded
	/// automatically. 
	/// - Parameter size: The size in bytes, -1 to disable the autodownload feature, 0
	/// to download them all no matter the size 
	
	/// Gets the size under which incoming files in chat messages will be downloaded
	/// automatically. 
	/// - Returns: The size in bytes, -1 if autodownload feature is disabled, 0 to
	/// download them all no matter the size 
	public var maxSizeForAutoDownloadIncomingFiles: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_max_size_for_auto_download_incoming_files(cPtr))
		}
		set
		{
			linphone_core_set_max_size_for_auto_download_incoming_files(cPtr, CInt(newValue))
		}
	}
		
	/// Sets the sound device used for media. 
	/// - Parameter devid: The device name as returned by
	/// linphone_core_get_sound_devices    
	/// - Returns: 0 
	
	/// Gets the name of the currently assigned sound device for media. 
	/// - Returns: The name of the currently assigned sound device for capture.    
	public var mediaDevice: String?
	{
	
			
			let cPointer = linphone_core_get_media_device(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
	public func setMediadevice(newValue: String?) throws
	{
		let exception_result = linphone_core_set_media_device(cPtr, newValue)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "username setter returned value \(exception_result)")
		}
	}
		
	/// Chooses the media encryption type to be used for RTP packets. 
	/// - Parameter menc: The media encryption policy to be used. 
	/// - Returns: 0 if successful, any other value otherwise. 
	
	/// Gets the media encryption type being used for RTP packets. 
	/// - Returns: The ``MediaEncryption`` policy being used. 
	public var mediaEncryption: MediaEncryption
	{
	
						return MediaEncryption(rawValue: Int(linphone_core_get_media_encryption(cPtr).rawValue))!

	}
	public func setMediaencryption(newValue: MediaEncryption) throws
	{
		let exception_result = linphone_core_set_media_encryption(cPtr, LinphoneMediaEncryption(rawValue: CUnsignedInt(newValue.rawValue)))
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "username setter returned value \(exception_result)")
		}
	}
		
	/// Defines whether the configured media encryption is mandatory, if it is and the
	/// negotation cannot result in the desired media encryption then the call will
	/// fail. 
	/// If not an INVITE will be resent with encryption disabled. 
	/// - Parameter mandatory: true to set it mandatory; false otherwise. 
	
	public var mediaEncryptionMandatory: Bool?
	{
	
		willSet
		{
			linphone_core_set_media_encryption_mandatory(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// This method is called by the application to notify the linphone core library
	/// when the media (RTP) network is reachable. 
	/// This is for advanced usage, when SIP and RTP layers are required to use
	/// different interfaces. Most applications just need
	/// ``setNetworkReachable(reachable:)``.
	/// - Parameter reachable: true if network is reachable, false otherwise 
	
	public var mediaNetworkReachable: Bool?
	{
	
		willSet
		{
			linphone_core_set_media_network_reachable(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets the media resources mode. 
	/// Value values are: unique and shared. When the mode is set to unique, then only
	/// one call in the state StreamsRunning is allowed. While accepting a call, the
	/// core will try to free media resource used by the current call. If it is
	/// unsuccessful, then the call is not accepted. If mode is set to shared, then the
	/// media resources of the current call (if any) are not emptied when taking a new
	/// call. If the user whishes to free them, he/she is responsible to call
	/// linphone_core_preempt_sound_resources himself/herself 
	/// - Parameter mode: the chosen mode 
	
	/// This function returns the media resource mode for this core. 
	/// - Returns: The media resource mode 
	public var mediaResourceMode: MediaResourceMode
	{
	
		get
		{ 
						return MediaResourceMode(rawValue: Int(linphone_core_get_media_resource_mode(cPtr).rawValue))!
		}
		set
		{
			linphone_core_set_media_resource_mode(cPtr, LinphoneMediaResourceMode(rawValue: CUnsignedInt(newValue.rawValue)))
		}
	}
		
	/// It sets the duration of the timer to resend a message when the channel is
	/// broken (i.e. 
	/// the core gets an NoResponse or IOError response)
	/// - Parameter duration: the duration of the timer in seconds. A 0 or negative
	/// number means that the feature is deactivated. 
	
	/// Returns the duration of the timer that delays the automatic resending of chat
	/// messages. 
	/// - Returns: the duration of the timer in seconds 
	public var messageAutomaticResendingDelay: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_message_automatic_resending_delay(cPtr))
		}
		set
		{
			linphone_core_set_message_automatic_resending_delay(cPtr, CInt(newValue))
		}
	}
		
	/// It sets the duration of the timer that starts just after the SUBSCRIBE is sent
	/// to delay the sending of chat messages in group chats. 
	/// - Parameter duration: the duration of the timer in seconds. A 0 or negative
	/// number deactivates the feature. 
	/// - Warning: it is only useful to set this property if
	/// linphone_core_send_message_after_notify_enabled returns false 
	
	/// Returns the duration of the timer that delays the sending of chat messages. 
	/// - Returns: the duration of the timer in seconds 
	public var messageSendingDelay: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_message_sending_delay(cPtr))
		}
		set
		{
			linphone_core_set_message_sending_delay(cPtr, CInt(newValue))
		}
	}
		
	/// Enables or disables the microphone. 
	/// This effectively enable or disable microphone (mute) for currently the running
	/// call or conference if any, as well as it applies to future currently running
	/// calls or conferences. 
	/// - Parameter enable: true to enable the microphone, false to disable it. 
	
	/// Tells whether the microphone is enabled. 
	/// - Returns: true if the microphone is enabled, false if disabled. 
	public var micEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_mic_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_mic(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Allow to control microphone level: gain in db. 
	/// - Parameter level: The new microphone level 
	
	/// Get microphone gain in db. 
	/// - Returns: The current microphone gain 
	public var micGainDb: Float
	{
	
		get
		{ 
						return linphone_core_get_mic_gain_db(cPtr)
		}
		set
		{
			linphone_core_set_mic_gain_db(cPtr, newValue)
		}
	}
		
	
	/// Gets the number of missed calls. 
	/// Once checked, this counter can be reset with ``resetMissedCallsCount()``. 
	/// - Returns: The number of missed calls. 
	public var missedCallsCount: Int
	{
	
						return Int(linphone_core_get_missed_calls_count(cPtr))

	}
		
	/// Sets the maximum transmission unit size in bytes. 
	/// This information is useful for sending RTP packets. Default value is 1500. 
	/// - Parameter mtu: The MTU in bytes 
	
	/// Returns the maximum transmission unit size in bytes. 
	public var mtu: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_mtu(cPtr))
		}
		set
		{
			linphone_core_set_mtu(cPtr, CInt(newValue))
		}
	}
		
	/// Deprecated. 
	/// This function was used to force a given IP address to appear in SDP.
	/// Unfortunately, this cannot work as explained by
	/// https://www.rfc-editor.org/rfc/rfc5389#section-2 . 
	/// This function was used to force a given IP address to appear in SDP.
	/// Unfortunately, this cannot work as explained by
	/// https://www.rfc-editor.org/rfc/rfc5389#section-2 . 
	/// - Parameter addr: The public IP address of NAT to use.    
	/// - Deprecated: 12/10/2022 Use ``setNatPolicy(policy:)``
	
	/// Deprecated. 
	/// Get the public IP address of NAT being used. 
	/// - Returns: The public IP address of NAT being used.    
	/// - Deprecated: 12/10/2022 Use ``getNatPolicy()``
	public var natAddress: String?
	{
	@available(*, deprecated)
		get
		{ 
			
			let cPointer = linphone_core_get_nat_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
	@available(*, deprecated)
		set
		{
			linphone_core_set_nat_address(cPtr, newValue)
		}
	}
		
	/// Set the policy to use to pass through NATs/firewalls. 
	/// It may be overridden by a NAT policy for a specific proxy config. 
	/// - Parameter policy: ``NatPolicy`` object   
	/// - See also: ``AccountParams/setNatPolicy(policy:)`` 
	
	/// Get The policy that is used to pass through NATs/firewalls. 
	/// It may be overridden by a NAT policy for a specific proxy config. 
	/// - Returns: ``NatPolicy`` object in use.   
	/// - See also: ``AccountParams/getNatPolicy()`` 
	public var natPolicy: NatPolicy?
	{
	
		get
		{ 
						let cPointer = linphone_core_get_nat_policy(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = NatPolicy.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_core_set_nat_policy(cPtr, newValue?.cPtr)
		}
	}
		
	/// Set the native window id where the preview video (local camera) is to be
	/// displayed. 
	/// This has to be used in conjunction with ``usePreviewWindow(yesno:)``. See
	/// ``setNativeVideoWindowId(windowId:)`` for general details about window_id
	/// This has to be used in conjunction with ``usePreviewWindow(yesno:)``. See
	/// ``setNativeVideoWindowId(windowId:)`` for general details about window_id
	/// On Android : org.linphone.mediastream.video.capture.CaptureTextureView is used
	/// for ``setNativePreviewWindowId(windowId:)``. It is inherited from TextureView
	/// and takes care of rotating the captured image from the camera and scale it to
	/// keep it's ratio.
	/// On Android : org.linphone.mediastream.video.capture.CaptureTextureView is used
	/// for ``setNativePreviewWindowId(windowId:)``. It is inherited from TextureView
	/// and takes care of rotating the captured image from the camera and scale it to
	/// keep it's ratio.
	/// On Android : org.linphone.mediastream.video.capture.CaptureTextureView is used
	/// for ``setNativePreviewWindowId(windowId:)``. It is inherited from TextureView
	/// and takes care of rotating the captured image from the camera and scale it to
	/// keep it's ratio.
	/// - Parameter windowId: The native window id where the preview video is to be
	/// displayed.    
	
	/// Get the native window handle of the video preview window. 
	/// see ``setNativeVideoWindowId(windowId:)`` for details about window_id
	/// see ``setNativeVideoWindowId(windowId:)`` for details about window_id
	/// There is a special case for Qt : ``getNativePreviewWindowId()`` returns a
	/// QQuickFramebufferObject::Renderer. Note : Qt blocks GUI thread when calling
	/// createRenderer(), so it is safe to call linphone functions there if needed.
	/// There is a special case for Qt : ``getNativePreviewWindowId()`` returns a
	/// QQuickFramebufferObject::Renderer. Note : Qt blocks GUI thread when calling
	/// createRenderer(), so it is safe to call linphone functions there if needed.
	/// - Returns: The native window handle of the video preview window.    
	public var nativePreviewWindowId: UnsafeMutableRawPointer?
	{
	
		get
		{ 
						return linphone_core_get_native_preview_window_id(cPtr)
		}
		set
		{
			linphone_core_set_native_preview_window_id(cPtr, newValue)
		}
	}
		
	/// Sets whether to use the platform-dependent ringing. 
	/// This property is meaningful for Android platform only. When set to true, the
	/// incoming call's ring tone is played by a Android MediaPlayer object playing the
	/// phone's default ringtone, and manages vibrator as well. When set to false, the
	/// incoming call's ring tone is played using liblinphone's internal ring tone
	/// player, that is generic for all platforms.
	/// - Parameter enable: True to enable native ringing, false otherwise 
	
	/// Returns whether the native ringing is enabled or not. 
	/// This property is meaningful for Android platform only. When set to true, the
	/// incoming call's ring tone is played by a Android MediaPlayer object playing the
	/// phone's default ringtone, and manages vibrator as well. When set to false, the
	/// incoming call's ring tone is played using liblinphone's internal ring tone
	/// player, that is generic for all platforms. 
	/// - Returns: True if we use the native ringing, false otherwise 
	public var nativeRingingEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_native_ringing_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_native_ringing(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Set the native video window id where the video is to be displayed. 
	/// On Desktop platforms(MacOS, Linux, Windows), the display filter is "MSOGL" by
	/// default. That means : If window_id is not set or set to
	/// LINPHONE_VIDEO_DISPLAY_NONE, then the core will not create its own window,
	/// unless the special id LINPHONE_VIDEO_DISPLAY_AUTO is given. This is currently
	/// only supported for Linux X11 (Window type), Windows UWP (SwapChainPanel type)
	/// and Windows Win32 (HWND type).
	/// On Desktop platforms(MacOS, Linux, Windows), the display filter is "MSOGL" by
	/// default. That means : If window_id is not set or set to
	/// LINPHONE_VIDEO_DISPLAY_NONE, then the core will not create its own window,
	/// unless the special id LINPHONE_VIDEO_DISPLAY_AUTO is given. This is currently
	/// only supported for Linux X11 (Window type), Windows UWP (SwapChainPanel type)
	/// and Windows Win32 (HWND type).
	/// The C# Wrapper on Windows for UWP takes directly a SwapChainPane without
	/// Marshalling. On other platforms, window_id is a MSOglContextInfo defined in
	/// msogl.h of mediastreamer2 There is a special case for Qt : The "MSQOGL" filter
	/// must be selected by using ``setVideoDisplayFilter(filterName:)``. Setting
	/// window id is only used to stop rendering by passing
	/// LINPHONE_VIDEO_DISPLAY_NONE. ``getNativeVideoWindowId()`` returns a
	/// QQuickFramebufferObject::Renderer and ``createNativeVideoWindowId(context:)``
	/// creates one. After a creation, ``setNativeVideoWindowId(windowId:)`` must be
	/// called with the new object.
	/// The C# Wrapper on Windows for UWP takes directly a SwapChainPane without
	/// Marshalling. On other platforms, window_id is a MSOglContextInfo defined in
	/// msogl.h of mediastreamer2 There is a special case for Qt : The "MSQOGL" filter
	/// must be selected by using ``setVideoDisplayFilter(filterName:)``. Setting
	/// window id is only used to stop rendering by passing
	/// LINPHONE_VIDEO_DISPLAY_NONE. ``getNativeVideoWindowId()`` returns a
	/// QQuickFramebufferObject::Renderer and ``createNativeVideoWindowId(context:)``
	/// creates one. After a creation, ``setNativeVideoWindowId(windowId:)`` must be
	/// called with the new object.
	/// The C# Wrapper on Windows for UWP takes directly a SwapChainPane without
	/// Marshalling. On other platforms, window_id is a MSOglContextInfo defined in
	/// msogl.h of mediastreamer2 There is a special case for Qt : The "MSQOGL" filter
	/// must be selected by using ``setVideoDisplayFilter(filterName:)``. Setting
	/// window id is only used to stop rendering by passing
	/// LINPHONE_VIDEO_DISPLAY_NONE. ``getNativeVideoWindowId()`` returns a
	/// QQuickFramebufferObject::Renderer and ``createNativeVideoWindowId(context:)``
	/// creates one. After a creation, ``setNativeVideoWindowId(windowId:)`` must be
	/// called with the new object.
	/// The C# Wrapper on Windows for UWP takes directly a SwapChainPane without
	/// Marshalling. On other platforms, window_id is a MSOglContextInfo defined in
	/// msogl.h of mediastreamer2 There is a special case for Qt : The "MSQOGL" filter
	/// must be selected by using ``setVideoDisplayFilter(filterName:)``. Setting
	/// window id is only used to stop rendering by passing
	/// LINPHONE_VIDEO_DISPLAY_NONE. ``getNativeVideoWindowId()`` returns a
	/// QQuickFramebufferObject::Renderer and ``createNativeVideoWindowId(context:)``
	/// creates one. After a creation, ``setNativeVideoWindowId(windowId:)`` must be
	/// called with the new object.
	/// The C# Wrapper on Windows for UWP takes directly a SwapChainPane without
	/// Marshalling. On other platforms, window_id is a MSOglContextInfo defined in
	/// msogl.h of mediastreamer2 There is a special case for Qt : The "MSQOGL" filter
	/// must be selected by using ``setVideoDisplayFilter(filterName:)``. Setting
	/// window id is only used to stop rendering by passing
	/// LINPHONE_VIDEO_DISPLAY_NONE. ``getNativeVideoWindowId()`` returns a
	/// QQuickFramebufferObject::Renderer and ``createNativeVideoWindowId(context:)``
	/// creates one. After a creation, ``setNativeVideoWindowId(windowId:)`` must be
	/// called with the new object.
	/// On mobile operating systems, LINPHONE_VIDEO_DISPLAY_AUTO is not supported and
	/// window_id depends of the platform : iOS : It is a UIView. Android : It is a
	/// TextureView.
	/// On mobile operating systems, LINPHONE_VIDEO_DISPLAY_AUTO is not supported and
	/// window_id depends of the platform : iOS : It is a UIView. Android : It is a
	/// TextureView.
	/// - Parameter windowId: The native window id where the remote video is to be
	/// displayed.    
	
	/// Get the native window handle of the video window. 
	/// see linphone_core_set_native_video_window_id for details about window_id
	/// see linphone_core_set_native_video_window_id for details about window_id
	/// There is a special case for Qt : ``getNativeVideoWindowId()`` returns a
	/// QQuickFramebufferObject::Renderer. Note : Qt blocks GUI thread when calling
	/// createRenderer(), so it is safe to call linphone functions there if needed.
	/// There is a special case for Qt : ``getNativeVideoWindowId()`` returns a
	/// QQuickFramebufferObject::Renderer. Note : Qt blocks GUI thread when calling
	/// createRenderer(), so it is safe to call linphone functions there if needed.
	/// - Returns: The native window handle of the video window.    
	public var nativeVideoWindowId: UnsafeMutableRawPointer?
	{
	
		get
		{ 
						return linphone_core_get_native_video_window_id(cPtr)
		}
		set
		{
			linphone_core_set_native_video_window_id(cPtr, newValue)
		}
	}
		
	/// This method is called by the application to notify the linphone core library
	/// when network is reachable. 
	/// Calling this method with true trigger linphone to initiate a registration
	/// process for all proxies. Calling this method disables the automatic network
	/// detection mode. It means you must call this method after each network state
	/// changes.
	/// - Parameter reachable: true if network is reachable, false otherwise 
	
	public var networkReachable: Bool?
	{
	
		willSet
		{
			linphone_core_set_network_reachable(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets the no-rtp timeout value in seconds when the call is on hold. 
	/// - Parameter seconds: The no-rtp timeout value to use in seconds when the call
	/// is on hold
	/// - See also: linphone_core_get_nortp_on_hold_timeout() for details. 
	
	/// Gets the value of the no-rtp timeout when the call is on hold. 
	/// When no RTP or RTCP packets have been received for a while when the call is on
	/// hold ``Core`` will consider the call is broken (remote end crashed or
	/// disconnected from the network), and thus will terminate the call. The no-rtp
	/// timeout is the duration above which the call is considered broken. 
	/// - Returns: The value of the no-rtp timeout in seconds when the call is on hold 
	public var nortpOnholdTimeout: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_nortp_onhold_timeout(cPtr))
		}
		set
		{
			linphone_core_set_nortp_onhold_timeout(cPtr, CInt(newValue))
		}
	}
		
	/// Sets the no-rtp timeout value in seconds. 
	/// - Parameter seconds: The no-rtp timeout value to use in seconds
	/// - See also: ``getNortpTimeout()`` for details. 
	
	/// Gets the value of the no-rtp timeout. 
	/// When no RTP or RTCP packets have been received for a while ``Core`` will
	/// consider the call is broken (remote end crashed or disconnected from the
	/// network), and thus will terminate the call. The no-rtp timeout is the duration
	/// above which the call is considered broken. 
	/// - Returns: The value of the no-rtp timeout in seconds 
	public var nortpTimeout: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_nortp_timeout(cPtr))
		}
		set
		{
			linphone_core_set_nortp_timeout(cPtr, CInt(newValue))
		}
	}
		
	/// Sets the given ``AudioDevice`` as output for all active calls. 
	/// - Parameter audioDevice: The ``AudioDevice``. nil does nothing.    
	
	/// Gets the output audio device for the current call. 
	/// - Returns: The output audio device for the current or first call, nil if there
	/// is no call.    
	public var outputAudioDevice: AudioDevice?
	{
	
		get
		{ 
						let cPointer = linphone_core_get_output_audio_device(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = AudioDevice.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_core_set_output_audio_device(cPtr, newValue?.cPtr)
		}
	}
		
	/// Sets a wav file to be played when files are used instead of soundcards (see
	/// ``setUseFiles(yesno:)``). 
	/// The file must be a 16 bit linear wav file. 
	/// - Parameter file: The path to the file to be played when files are used instead
	/// of soundcards.    
	
	/// Get the wav file that is played when files are used instead of soundcards (see
	/// ``setUseFiles(yesno:)``). 
	/// The file is a 16 bit linear wav file. 
	/// - Returns: The path to the file that is played when files are used instead of
	/// soundcards.    
	public var playFile: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_core_get_play_file(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_core_set_play_file(cPtr, newValue)
		}
	}
		
	/// Sets the sound device used for playback. 
	/// - Parameter devid: The device name as returned by
	/// linphone_core_get_sound_devices    
	/// - Returns: 0 
	/// - Deprecated: 11/09/2024 use ``setOutputAudioDevice(audioDevice:)`` or
	/// ``setDefaultOutputAudioDevice(audioDevice:)`` instead.
	
	/// Gets the name of the currently assigned sound device for playback. 
	/// - Returns: The name of the currently assigned sound device for playback.    
	/// - Deprecated: 11/09/2024 use ``getOutputAudioDevice()`` or
	/// ``getDefaultOutputAudioDevice()`` instead.
	@available(*, deprecated)
	public var playbackDevice: String?
	{
	
			
			let cPointer = linphone_core_get_playback_device(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
	@available(*, deprecated)
	public func setPlaybackdevice(newValue: String?) throws
	{
		let exception_result = linphone_core_set_playback_device(cPtr, newValue)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "username setter returned value \(exception_result)")
		}
	}
		
	/// Allow to control play level before entering sound card: gain in db. 
	/// - Parameter level: The new play level 
	
	/// Gets playback gain in db (before entering sound card). 
	/// - Returns: The current playback gain 
	public var playbackGainDb: Float
	{
	
		get
		{ 
						return linphone_core_get_playback_gain_db(cPtr)
		}
		set
		{
			linphone_core_set_playback_gain_db(cPtr, newValue)
		}
	}
		
	/// Set the preferred frame rate for video. 
	/// Based on the available bandwidth constraints and network conditions, the video
	/// encoder remains free to lower the framerate. There is no warranty that the
	/// preferred frame rate be the actual framerate. used during a call. Default value
	/// is 0, which means "use encoder's default fps value". 
	/// - Parameter fps: the target frame rate in number of frames per seconds. 
	
	/// Returns the preferred video framerate, previously set by
	/// ``setPreferredFramerate(fps:)``. 
	/// - Returns: frame rate in number of frames per seconds. 
	public var preferredFramerate: Float
	{
	
		get
		{ 
						return linphone_core_get_preferred_framerate(cPtr)
		}
		set
		{
			linphone_core_set_preferred_framerate(cPtr, newValue)
		}
	}
		
	/// Set the preferred video definition for the stream that is captured and sent to
	/// the remote party. 
	/// All standard video definitions are accepted on the receive path. 
	/// - Parameter videoDefinition: ``VideoDefinition`` object    
	
	/// Get the preferred video definition for the stream that is captured and sent to
	/// the remote party. 
	/// - Returns: The preferred ``VideoDefinition``    
	public var preferredVideoDefinition: VideoDefinition?
	{
	
		get
		{ 
						let cPointer = linphone_core_get_preferred_video_definition(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = VideoDefinition.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_core_set_preferred_video_definition(cPtr, newValue?.cPtr)
		}
	}
		
	/// Sets the preferred video definition by its name. 
	/// Call ``Factory/getSupportedVideoDefinitions()`` to have a list of supported
	/// video definitions.
	/// - Parameter name: The name of the definition to set    
	
	public var preferredVideoDefinitionByName: String = ""
	{
	
		willSet
		{
			linphone_core_set_preferred_video_definition_by_name(cPtr, newValue)
		}
	}
		
	/// Sets my presence model. 
	/// - Parameter presence: ``PresenceModel``    
	
	/// Gets my presence model. 
	/// - Returns: A ``PresenceModel`` object, or nil if no presence model has been
	/// set.    
	public var presenceModel: PresenceModel?
	{
	
		get
		{ 
						let cPointer = linphone_core_get_presence_model(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = PresenceModel.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_core_set_presence_model(cPtr, newValue?.cPtr)
		}
	}
		
	/// Set the video definition for the captured (preview) video. 
	/// This method is for advanced usage where a video capture must be set
	/// independently of the definition of the stream actually sent through the call.
	/// This allows for example to have the preview window in High Definition even if
	/// due to bandwidth constraint the sent video definition is small. Using this
	/// feature increases the CPU consumption, since a rescaling will be done
	/// internally. 
	/// - Parameter videoDefinition: ``VideoDefinition`` object    
	
	/// Get the definition of the captured video. 
	/// - Returns: The captured ``VideoDefinition`` if it was previously set by
	/// ``setPreviewVideoDefinition(videoDefinition:)``, otherwise a 0x0
	/// ``VideoDefinition``.    
	/// - See also: ``setPreviewVideoDefinition(videoDefinition:)`` 
	public var previewVideoDefinition: VideoDefinition?
	{
	
		get
		{ 
						let cPointer = linphone_core_get_preview_video_definition(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = VideoDefinition.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_core_set_preview_video_definition(cPtr, newValue?.cPtr)
		}
	}
		
	/// -Set the video definition for the captured (preview) video by its name. Call
	/// ``Factory/getSupportedVideoDefinitions()`` to have a list of supported video
	/// definitions. Video resolution names are: qcif, svga, cif, vga, 4cif, svga ...
	/// - Parameter name: The name of the definition to set    
	
	public var previewVideoDefinitionByName: String = ""
	{
	
		willSet
		{
			linphone_core_set_preview_video_definition_by_name(cPtr, newValue)
		}
	}
		
	/// Sets the local "from" SIP identity used for calls made out of any configured
	/// ``Account``. 
	/// Not using a SIP account is not recommended.
	/// This data is used in absence of any proxy configuration or when no account
	/// configuration is set. See ``Account`` 
	/// - Parameter contact: the contact to set    
	/// - Returns: 0 if successful, -1 otherwise 
	
	/// Returns the default identity when no account is used. 
	/// This SIP address usually contains a private ip address, and may not be routable
	/// globally.
	/// - Returns: the primary contact identity    
	public var primaryContact: String
	{
	
			
			let cPointer = linphone_core_get_primary_contact(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
	public func setPrimarycontact(newValue: String) throws
	{
		let exception_result = linphone_core_set_primary_contact(cPtr, newValue)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "username setter returned value \(exception_result)")
		}
	}
		
	
	/// Same as ``getPrimaryContact()`` but the result is a ``Address`` object instead
	/// of a string. 
	/// - Returns: a ``Address`` object.       
	public var primaryContactAddress: Address?
	{
	
						let cPointer = linphone_core_get_primary_contact_address(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
			return result

	}
		
	
	/// Same as ``getPrimaryContact()`` but the result is a ``Address`` object instead
	/// of const char *. 
	/// - Returns: a ``Address`` object.      
	/// - Deprecated: 22/10/2018 Use ``createPrimaryContactParsed()`` instead. 
	@available(*, deprecated)
	public var primaryContactParsed: Address?
	{
	
						let cPointer = linphone_core_get_primary_contact_parsed(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Address.getSwiftObject(cObject:cPointer!)
			belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
			return result

	}
		
	/// Sets the URI where to download xml configuration file at startup. 
	/// http://, https:// and file:// uris are supported. This can also be set from
	/// configuration file or factory config file, from [misc] section, item
	/// "config-uri". Calling this function does not load the configuration. It will
	/// write the value into configuration so that configuration from URI will take
	/// place during next ``start()`` invocation. The format the xml file is briefly
	/// documented here:
	/// https://wiki.linphone.org/xwiki/wiki/public/view/Lib/Features/Remote%20Provisioning/ 
	/// http://, https:// and file:// uris are supported. This can also be set from
	/// configuration file or factory config file, from [misc] section, item
	/// "config-uri". Calling this function does not load the configuration. It will
	/// write the value into configuration so that configuration from URI will take
	/// place during next ``start()`` invocation. The format the xml file is briefly
	/// documented here:
	/// https://wiki.linphone.org/xwiki/wiki/public/view/Lib/Features/Remote%20Provisioning/ 
	/// http://, https:// and file:// uris are supported. This can also be set from
	/// configuration file or factory config file, from [misc] section, item
	/// "config-uri". Calling this function does not load the configuration. It will
	/// write the value into configuration so that configuration from URI will take
	/// place during next ``start()`` invocation. The format the xml file is briefly
	/// documented here:
	/// https://wiki.linphone.org/xwiki/wiki/public/view/Lib/Features/Remote%20Provisioning/ 
	/// http://, https:// and file:// uris are supported. This can also be set from
	/// configuration file or factory config file, from [misc] section, item
	/// "config-uri". Calling this function does not load the configuration. It will
	/// write the value into configuration so that configuration from URI will take
	/// place during next ``start()`` invocation. The format the xml file is briefly
	/// documented here:
	/// https://wiki.linphone.org/xwiki/wiki/public/view/Lib/Features/Remote%20Provisioning/ 
	/// http://, https:// and file:// uris are supported. This can also be set from
	/// configuration file or factory config file, from [misc] section, item
	/// "config-uri". Calling this function does not load the configuration. It will
	/// write the value into configuration so that configuration from URI will take
	/// place during next ``start()`` invocation. The format the xml file is briefly
	/// documented here:
	/// https://wiki.linphone.org/xwiki/wiki/public/view/Lib/Features/Remote%20Provisioning/ 
	/// - Parameter uri: the uri to use in order to obtain the configuration. Passing
	/// nil will disable remote provisioning.    
	/// - Returns: -1 if uri could not be parsed, 0 otherwise. Note that this does not
	/// check validity of URI endpoint nor scheme and download may still fail. 
	
	/// Get provisioning URI. 
	/// - Returns: the provisioning URI.    
	public var provisioningUri: String?
	{
	
			
			let cPointer = linphone_core_get_provisioning_uri(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
	public func setProvisioninguri(newValue: String?) throws
	{
		let exception_result = linphone_core_set_provisioning_uri(cPtr, newValue)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "username setter returned value \(exception_result)")
		}
	}
		
	
	/// Returns an unmodifiable list of entered proxy configurations. 
	/// - Returns: A list of ``ProxyConfig``.      
	/// - Deprecated: 04/09/2024 Use ``getAccountList()`` 
	@available(*, deprecated)
	public var proxyConfigList: [ProxyConfig]
	{
	
						var swiftList = [ProxyConfig]()
			let cList = linphone_core_get_proxy_config_list(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(ProxyConfig.getSwiftObject(cObject: data))
				listTemp = UnsafePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
			return swiftList

	}
		
	/// Configures the minimum interval between a push notification and the
	/// corresponding incoming INVITE. 
	/// If exceeded, Linphone Call is transitioned to CallError and further incoming
	/// invite associated to this push is declined if any. 
	/// - Parameter seconds: The new timeout in seconds 
	
	/// Returns the push incoming call timeout See
	/// ``setPushIncomingCallTimeout(seconds:)`` for details. 
	/// - Returns: The current push incoming call timeout in seconds 
	public var pushIncomingCallTimeout: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_push_incoming_call_timeout(cPtr))
		}
		set
		{
			linphone_core_set_push_incoming_call_timeout(cPtr, CInt(newValue))
		}
	}
		
	
	/// Gets the push notification configuration object if it exists. 
	/// - Returns: the ``PushNotificationConfig`` if it exists, nil otherwise.    
	public var pushNotificationConfig: PushNotificationConfig?
	{
	
						let cPointer = linphone_core_get_push_notification_config(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = PushNotificationConfig.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	/// Enables or disables push notifications on Android & iOS. 
	/// If enabled, it will try to get the push token add configure each account with
	/// push_notification_allowed set to true with push parameters. IOS: will also
	/// instanciate a PushRegistry, so make sure that your app does not instanciate one
	/// too or there will be a conflict. 
	/// - Parameter enable: true to enable push notifications, false to disable 
	
	/// Gets whether push notifications are enabled or not (Android & iOS only). 
	/// If not, the app will have to handle all the push-related settings for each
	/// accounts 
	/// - Returns: true if push notifications are enabled, false otherwise 
	public var pushNotificationEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_push_notification_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_push_notification(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Controls QRCode scanning enablement. 
	/// When enabled, several QRCodes can be decoded. The recognized QR code are
	/// notified through the ``CoreDelegate`` interface (see
	/// linphone_core_cbs_set_qrcode_found) 
	/// - Parameter enable: A boolean value telling whether to enable QRCode in the
	/// preview. 
	
	/// Tells whether QRCode is enabled in the preview. 
	/// - Returns: A boolean value telling whether QRCode is enabled in the preview. 
	public var qrcodeVideoPreviewEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_qrcode_video_preview_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_qrcode_video_preview(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Set the queued message resend period. 
	/// It is the number of seconds after the first attempt to send the message. 
	/// - Parameter seconds: number of seconds after the first attempt to send a
	/// message. A negative value means all queued messages are resent at startup. 
	
	/// Gets the queued message resend period. 
	/// - Returns: the number of second to resend a queued message 
	public var queuedMessageResendPeriod: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_queued_message_resend_period(cPtr))
		}
		set
		{
			linphone_core_set_queued_message_resend_period(cPtr, (newValue))
		}
	}
		
	
	/// Gets if realtime text is enabled or not (RFC4103). 
	/// - Returns: true if realtime text is enabled, false otherwise 
	public var realtimeTextEnabled: Bool
	{
	
						return linphone_core_realtime_text_enabled(cPtr) != 0

	}
		
	/// Set keep alive interval for real time text (RFC4103). 
	/// - Parameter interval: The keep alive interval of real time text, 25000 by
	/// default. 
	
	/// Gets keep alive interval of real time text (RFC4103). 
	/// - Returns: keep alive interval of real time text. 
	public var realtimeTextKeepaliveInterval: UInt
	{
	
		get
		{ 
						return UInt(linphone_core_get_realtime_text_keepalive_interval(cPtr))
		}
		set
		{
			linphone_core_set_realtime_text_keepalive_interval(cPtr, CUnsignedInt(newValue))
		}
	}
		
	/// Enables the record-aware feature that will warn other users when doing
	/// recording during a call. 
	/// See ``CallDelegate`` for being notified when a call is being recorded. 
	/// - Parameter enable: true to activate the record aware feature, false to disable
	/// it. 
	
	/// Gets if the record aware feature is enabled or not. 
	/// - Returns: true if the record aware feature is enabled, false otherwise. 
	public var recordAwareEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_record_aware_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_record_aware(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets a wav file where incoming stream is to be recorded, when files are used
	/// instead of soundcards (see ``setUseFiles(yesno:)``). 
	/// This feature is different from call recording
	/// (``CallParams/setRecordFile(path:)``) The file will be a 16 bit linear wav
	/// file. 
	/// - Parameter file: The path to the file where incoming stream is to be recorded.
	///    
	
	/// Get the wav file where incoming stream is recorded, when files are used instead
	/// of soundcards (see ``setUseFiles(yesno:)``). 
	/// This feature is different from call recording
	/// (``CallParams/setRecordFile(path:)``) The file is a 16 bit linear wav file. 
	/// - Returns: The path to the file where incoming stream is recorded.    
	public var recordFile: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_core_get_record_file(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_core_set_record_file(cPtr, newValue)
		}
	}
		
	/// Sets if accounts will wait for network to be connected before trying to
	/// REGISTER. 
	/// - Parameter registerOnlyWhenNetworkIsUp: true to wait for an internet
	/// connection before trying to REGISTER, false to do it no matter the network
	/// state. 
	
	/// Gets if accounts will wait for network to be connected before trying to
	/// REGISTER or not. 
	/// - Returns: true if accounts will wait for internet connection before trying to
	/// REGISTER, false otherwise. 
	public var registerOnlyWhenNetworkIsUp: Bool
	{
	
		get
		{ 
						return linphone_core_get_register_only_when_network_is_up(cPtr) != 0
		}
		set
		{
			linphone_core_set_register_only_when_network_is_up(cPtr, newValue==true ? 1:0)
		}
	}
		
	
	/// Returns how many attachments are yet to be downloaded. 
	/// - Returns: how many attachments are yet to be downloaded. 
	public var remainingDownloadFileCount: UInt
	{
	
						return UInt(linphone_core_get_remaining_download_file_count(cPtr))

	}
		
	
	/// Returns how many attachments are yet to be uploaded. 
	/// - Returns: how many attachments are yet to be uploaded. 
	public var remainingUploadFileCount: UInt
	{
	
						return UInt(linphone_core_get_remaining_upload_file_count(cPtr))

	}
		
	
	/// Gets the list of currently configured LDAP or CardDAV remote servers used by
	/// ``MagicSearch``. 
	/// - Returns: the list of ``RemoteContactDirectory`` currently configured, if any.
	///         
	public var remoteContactDirectories: [RemoteContactDirectory]
	{
	
						var swiftList = [RemoteContactDirectory]()
			let cList = linphone_core_get_remote_contact_directories(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(RemoteContactDirectory.getSwiftObject(cObject: data))
				listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
				bctbx_list_free_with_data(cList, belle_sip_object_unref)
			return swiftList

	}
		
	/// Specify a ring back tone to be played to far end during incoming calls. 
	/// - Parameter ring: The path to the remote ring back tone to be played.    
	
	/// Get the ring back tone played to far end during incoming calls. 
	/// - Returns: the path to the remote ring back tone to be played.    
	public var remoteRingbackTone: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_core_get_remote_ringback_tone(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_core_set_remote_ringback_tone(cPtr, newValue)
		}
	}
		
	/// Sets whether the use RTCP NACK for reliability of video transmission is enabled
	/// or not. 
	/// Using RTCP NACK feedback is one of the available techniques to help mitigate
	/// the loss of video RTP packets. It gives good results when round trip time is
	/// low. It is disabled by default. 
	/// - Parameter enable: A boolean value telling whether to enable NACK context 
	
	/// Gets whether the use RTCP NACK for reliability of video transmission is enabled
	/// or not. 
	/// Using RTCP NACK feedback is one of the available techniques to help mitigate
	/// the loss of video RTP packets. It gives good results when round trip time is
	/// low. It is disabled by default. 
	/// - Returns: A boolean value telling whether NACK usage is enabled or not 
	public var retransmissionOnNackEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_retransmission_on_nack_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_retransmission_on_nack(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets the path to a wav file used for ringing. 
	/// The file must be a wav 16bit linear. If null, ringing is disable unless [sound]
	/// use_native_ringing is enabled, in which case we use the device ringtone. 
	/// The file must be a wav 16bit linear. If null, ringing is disable unless [sound]
	/// use_native_ringing is enabled, in which case we use the device ringtone. 
	/// - Parameter path: The path to a wav file to be used for ringing, null to
	/// disable or use device ringing depending on [sound] use_native_ringing.    
	
	/// Returns the path to the wav file used for ringing. 
	/// - Returns: The path to the wav file used for ringing.    
	public var ring: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_core_get_ring(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_core_set_ring(cPtr, newValue)
		}
	}
		
	/// Enable or disable the ring play during an incoming early media call. 
	/// - Parameter enable: A boolean value telling whether to enable ringing during an
	/// incoming early media call. 
	
	/// Tells whether the ring play is enabled during an incoming early media call. 
	public var ringDuringIncomingEarlyMedia: Bool
	{
	
		get
		{ 
						return linphone_core_get_ring_during_incoming_early_media(cPtr) != 0
		}
		set
		{
			linphone_core_set_ring_during_incoming_early_media(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets the path to a wav file used for ringing back. 
	/// Ringback means the ring that is heard when it's ringing at the remote party.
	/// The file must be a wav 16bit linear. 
	/// - Parameter path: The path to a wav file to be used for ringing back.    
	
	/// Returns the path to the wav file used for ringing back. 
	/// - Returns: The path to the wav file used for ringing back.    
	public var ringback: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_core_get_ringback(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_core_set_ringback(cPtr, newValue)
		}
	}
		
	/// Sets the sound device used for ringing. 
	/// - Parameter devid: The device name as returned by
	/// linphone_core_get_sound_devices    
	/// - Returns: 0
	/// - Deprecated: 29/08/2025 Use a combination of ``getExtendedAudioDevices()`` and
	/// ``AudioDevice/setUseForRinging(useForRinging:)`` instead. 
	
	/// Gets the name of the currently assigned sound device for ringing. 
	/// - Returns: The name of the currently assigned sound device for ringing.   
	/// - Deprecated: 29/08/2025 Use a combination of ``getExtendedAudioDevices()`` and
	/// ``AudioDevice/getUseForRinging()`` instead. 
	@available(*, deprecated)
	public var ringerDevice: String?
	{
	
			
			let cPointer = linphone_core_get_ringer_device(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
	@available(*, deprecated)
	public func setRingerdevice(newValue: String?) throws
	{
		let exception_result = linphone_core_set_ringer_device(cPtr, newValue)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "username setter returned value \(exception_result)")
		}
	}
		
	/// Sets the path to a file or folder containing trusted root CAs (PEM format) 
	/// - Parameter path: The path to a file or folder containing trusted root CAs.    
	
	/// Gets the path to a file or folder containing the trusted root CAs (PEM format) 
	/// - Returns: The path to a file or folder containing the trusted root CAs.    
	public var rootCa: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_core_get_root_ca(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_core_set_root_ca(cPtr, newValue)
		}
	}
		
	/// Sets the trusted root CAs (PEM format) 
	/// - Parameter data: The trusted root CAs as a string    
	
	public var rootCaData: String = ""
	{
	
		willSet
		{
			linphone_core_set_root_ca_data(cPtr, newValue)
		}
	}
		
	/// Enables or disables RTP bundle mode (Media Multiplexing). 
	/// See https://datatracker.ietf.org/doc/html/rfc8843 for more information about
	/// the feature. When enabled, liblinphone will try to negociate the use of a
	/// single port for all streams when doing an outgoing call. It automatically
	/// enables rtcp-mux. This feature can also be enabled per-call using
	/// ``CallParams``. RTP Bundle mode is required for video conferencing. 
	/// See https://datatracker.ietf.org/doc/html/rfc8843 for more information about
	/// the feature. When enabled, liblinphone will try to negociate the use of a
	/// single port for all streams when doing an outgoing call. It automatically
	/// enables rtcp-mux. This feature can also be enabled per-call using
	/// ``CallParams``. RTP Bundle mode is required for video conferencing. 
	/// - Parameter value: a boolean to indicate whether the feature is to be enabled. 
	
	/// Returns whether RTP bundle mode (also known as Media Multiplexing) is enabled. 
	/// See https://datatracker.ietf.org/doc/html/rfc8843 for more information. 
	/// See https://datatracker.ietf.org/doc/html/rfc8843 for more information. 
	/// - Returns: a boolean indicating the enablement of rtp bundle mode. 
	public var rtpBundleEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_rtp_bundle_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_rtp_bundle(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Control when media offer is sent in SIP INVITE. 
	/// Enabling this type of SIP call flow is not recommended. This setting is mainly
	/// for internal testing. 
	/// - Parameter enable: true if INVITE has to be sent whitout SDP. 
	
	/// Media offer control param for SIP INVITE. 
	/// - Returns: true if INVITE has to be sent whitout SDP. 
	public var sdp200AckEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_sdp_200_ack_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_sdp_200_ack(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Enables or disable self view during calls. 
	/// - Parameter enable: A boolean value telling whether to enable self view
	/// Self-view refers to having local webcam image inserted in corner of the video
	/// window during calls. This function works at any time, including during calls.
	/// - Deprecated: 04/09/2024 Prefer using ``setNativePreviewWindowId(windowId:)``
	/// to assign a view to render the local image. 
	
	/// Tells whether video self view during call is enabled or not. 
	/// - Returns: A boolean value telling whether self view is enabled 
	/// - See also: ``enableSelfView(enable:)`` for details. 
	public var selfViewEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_self_view_enabled(cPtr) != 0
		}
	@available(*, deprecated)
		set
		{
			linphone_core_enable_self_view(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Enable sending of chat message on group chats only after receiving the NOTIFY
	/// full state If it is disabled, as it is the default value, message will be sent
	/// after the delay set by linphone_core_get_message_sending_delay 
	/// Enable sending of chat message on group chats only after receiving the NOTIFY
	/// full state If it is disabled, as it is the default value, message will be sent
	/// after the delay set by linphone_core_get_message_sending_delay 
	/// - Parameter enabled: true if enabled, false otherwise. 
	
	/// Returns enablement of sending chat messages on group chats after receiving the
	/// NOTIFY full state. 
	/// - Returns: true if the core waits for the NOTIFY full statet before sending
	/// messages to group chats, false otherwise. 
	public var sendMessageAfterNotifyEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_send_message_after_notify_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_send_message_after_notify(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Enables whether or not to hide sender name in forwarded message. 
	/// - Parameter enable: whether or not to enable the feature 
	
	public var senderNameHiddenInForwardMessageEnabled: Bool?
	{
	
		willSet
		{
			linphone_core_enable_sender_name_hidden_in_forward_message(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Enable the Session Timers support. 
	/// - Parameter enabled: Enable or disable it 
	
	/// Check if the Session Timers feature is enabled. 
	/// - Returns: true if session timers are enabled, false otherwise 
	public var sessionExpiresEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_session_expires_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_session_expires(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets the session expires minSE value, forced to a minimum of 90 by default. 
	/// - Parameter min: The minSE value 
	
	/// Returns the session expires min value, 90 by default. 
	/// - Returns: The minSE value 
	public var sessionExpiresMinValue: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_session_expires_min_value(cPtr))
		}
		set
		{
			linphone_core_set_session_expires_min_value(cPtr, CInt(newValue))
		}
	}
		
	/// Sets the session expires refresher value. 
	/// - Parameter refresher: The ``SessionExpiresRefresher`` configuration value 
	
	/// Returns the session expires refresher value. 
	/// - Returns: The ``SessionExpiresRefresher`` configuration refresher value 
	public var sessionExpiresRefresherValue: SessionExpiresRefresher
	{
	
		get
		{ 
						return SessionExpiresRefresher(rawValue: Int(linphone_core_get_session_expires_refresher_value(cPtr).rawValue))!
		}
		set
		{
			linphone_core_set_session_expires_refresher_value(cPtr, LinphoneSessionExpiresRefresher(rawValue: CUnsignedInt(newValue.rawValue)))
		}
	}
		
	/// Sets the session expires value, 0 by default. 
	/// - Parameter expires: The session expires value 
	
	/// Returns the session expires value. 
	/// - Returns: The session timer expire value 
	public var sessionExpiresValue: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_session_expires_value(cPtr))
		}
		set
		{
			linphone_core_set_session_expires_value(cPtr, CInt(newValue))
		}
	}
		
	/// Sets the DSCP field for SIP signaling channel. 
	/// The DSCP defines the quality of service in IP packets. - Note: It is usually
	/// useless or a bad idea to try to play with DSCP bits unless having full control
	/// on the network. 
	/// - Warning: Setting the DSCP bits is more or less well supported by operating
	/// systems and sometimes requires to disable IPv6. 
	/// - Parameter dscp: The DSCP value to set 
	
	/// Gets the DSCP field for SIP signaling channel. 
	/// The DSCP defines the quality of service in IP packets. 
	/// - Returns: The current DSCP value 
	public var sipDscp: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_sip_dscp(cPtr))
		}
		set
		{
			linphone_core_set_sip_dscp(cPtr, CInt(newValue))
		}
	}
		
	/// This method is called by the application to notify the linphone core library
	/// when the SIP network is reachable. 
	/// This is for advanced usage, when SIP and RTP layers are required to use
	/// different interfaces. Most applications just need
	/// ``setNetworkReachable(reachable:)``.
	/// - Parameter reachable: true if network is reachable, false otherwise 
	
	public var sipNetworkReachable: Bool?
	{
	
		willSet
		{
			linphone_core_set_sip_network_reachable(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets the SIP transport timeout, which represents the maximum time permitted to
	/// establish a connection to a SIP server. 
	/// - Parameter timeoutMs: The SIP transport timeout in milliseconds. 
	
	/// Gets the SIP transport timeout, which represents the maximum time permitted to
	/// establish a connection to a SIP server. 
	/// - Returns: The SIP transport timeout in milliseconds. 
	public var sipTransportTimeout: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_sip_transport_timeout(cPtr))
		}
		set
		{
			linphone_core_set_sip_transport_timeout(cPtr, CInt(newValue))
		}
	}
		
	
	/// Gets the list of the available sound devices. 
	/// - Returns: An unmodifiable array of strings contanining the names of the
	/// available sound devices that is nil terminated.         
	/// - Deprecated: 10/04/2021 Use ``getAudioDevices()`` instead.
	@available(*, deprecated)
	public var soundDevicesList: [String]
	{
	
						var swiftList = [String]()
			let cList = linphone_core_get_sound_devices_list(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				swiftList.append(String(cString: unsafeBitCast(listTemp!.pointee.data, to: UnsafePointer<CChar>.self)))
				listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp!.pointee.next)
			}
				bctbx_list_free_with_data(cList, bctbx_free)
			return swiftList

	}
		
	/// Sets the crypto suites available to the core. 
	/// - Parameter suites: comma separated list of supported suites    
	
	/// Gets the crypto suites available to the core. 
	/// - Returns: a comma separated list of supported suites    
	public var srtpCryptoSuites: String
	{
	
		get
		{ 
			
			let cPointer = linphone_core_get_srtp_crypto_suites(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_core_set_srtp_crypto_suites(cPtr, newValue)
		}
	}
		
	/// Set the path to the image file to stream when "Static picture" is set as the
	/// video device. 
	/// - Parameter path: The path to the image file to use.    
	
	/// Get the path to the image file streamed when "Static picture" is set as the
	/// video device. 
	/// - Returns: The path to the image file streamed when "Static picture" is set as
	/// the video device.    
	public var staticPicture: String?
	{
	
			
			let cPointer = linphone_core_get_static_picture(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
	public func setStaticpicture(newValue: String?) throws
	{
		let exception_result = linphone_core_set_static_picture(cPtr, newValue)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "username setter returned value \(exception_result)")
		}
	}
		
	/// Set the frame rate for static picture. 
	/// - Parameter fps: The new frame rate to use for static picture. 
	
	/// Get the frame rate for static picture. 
	/// - Returns: The frame rate used for static picture. 
	public var staticPictureFps: Float
	{
	
						return linphone_core_get_static_picture_fps(cPtr)

	}
	public func setStaticpicturefps(newValue: Float) throws
	{
		let exception_result = linphone_core_set_static_picture_fps(cPtr, newValue)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "username setter returned value \(exception_result)")
		}
	}
		
	/// Set the STUN server address to use when the firewall policy is set to STUN. 
	/// - Parameter server: The STUN server address to use.    
	/// - Deprecated: 04/09/2024 use ``setNatPolicy(policy:)`` or
	/// ``AccountParams/setNatPolicy(policy:)``.
	
	/// Get the STUN server address being used. 
	/// - Returns: The STUN server address being used.    
	/// - Deprecated: 04/09/2024 use ``getNatPolicy()`` or
	/// ``AccountParams/getNatPolicy()``.
	public var stunServer: String?
	{
	@available(*, deprecated)
		get
		{ 
			
			let cPointer = linphone_core_get_stun_server(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
	@available(*, deprecated)
		set
		{
			linphone_core_set_stun_server(cPtr, newValue)
		}
	}
		
	
	/// Returns a list of strings containing the file format types supported for call
	/// recording. 
	/// - Returns: The supported formats, typically 'wav', 'mka', 'mkv', 'smff'.       
	///  
	public var supportedFileFormatsList: [String]
	{
	
						var swiftList = [String]()
			let cList = linphone_core_get_supported_file_formats_list(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				swiftList.append(String(cString: unsafeBitCast(listTemp!.pointee.data, to: UnsafePointer<CChar>.self)))
				listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp!.pointee.next)
			}
				bctbx_list_free_with_data(cList, bctbx_free)
			return swiftList

	}
		
	/// Set the SIP supported tags. 
	/// - Parameter tags: The SIP feature tags to set    
	
	public var supportedTag: String = ""
	{
	
		willSet
		{
			linphone_core_set_supported_tag(cPtr, newValue)
		}
	}
		
	/// Defines what level of support is provided to the 100rel attribute. 
	/// - Parameter level: support level of 100rel 
	
	/// Gets the support level of the 100rel attribute. 
	/// - Returns: The 100 rel support level 
	public var tag100RelSupportLevel: SupportLevel
	{
	
		get
		{ 
						return SupportLevel(rawValue: Int(linphone_core_get_tag_100rel_support_level(cPtr).rawValue))!
		}
		set
		{
			linphone_core_set_tag_100rel_support_level(cPtr, LinphoneSupportLevel(rawValue: CUnsignedInt(newValue.rawValue)))
		}
	}
		
	/// Defines whether tcap lines are going to be merged if capability negotiation
	/// (RFC5939) is supported. 
	/// - Parameter merge: true to merge tcap lines with consecutive indexes; false
	/// otherwise. 
	
	public var tcapLineMergingEnabled: Bool?
	{
	
		willSet
		{
			linphone_core_enable_tcap_line_merging(cPtr, newValue==true ? 1:0)
		}
	}
		
	
	/// Checks if tcap lines are going to the merged if the capability negotiation
	/// (RFC5939) is supported or not. 
	/// - Returns: true if tcap lines with consecutive indexes are going to be merged;
	/// false otherwise. 
	public var tcapLinesMergingEnabled: Bool
	{
	
						return linphone_core_tcap_lines_merging_enabled(cPtr) != 0

	}
		
	/// Redefines the list of the available payload types for real-time text. 
	/// - Parameter payloadTypes: The new list of payload types.      
	
	/// Returns the list of the available real-time text payload types. 
	/// - Returns: A freshly allocated list of the available payload types.         
	public var textPayloadTypes: [PayloadType]
	{
	
		get
		{ 
						var swiftList = [PayloadType]()
			let cList = linphone_core_get_text_payload_types(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(PayloadType.getSwiftObject(cObject: data))
				listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
				bctbx_list_free_with_data(cList, belle_sip_object_unref)
			return swiftList
		}
		set
		{
			withExtendedLifetime(newValue) {
				var cList: UnsafeMutablePointer<bctbx_list_t>? = nil
				for data in newValue {
					cList = bctbx_list_append(cList, UnsafeMutableRawPointer(data.cPtr))
				}
				linphone_core_set_text_payload_types(cPtr, cList)
			}
		}
	}
		
	/// Sets the UDP port used for text streaming. 
	/// A value if -1 will request the system to allocate the local port randomly. This
	/// is recommended in order to avoid firewall warnings. 
	/// - Parameter port: The UDP port to use for text streaming 
	
	/// Gets the UDP port used for text streaming. 
	/// - Returns: The UDP port used for text streaming 
	public var textPort: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_text_port(cPtr))
		}
		set
		{
			linphone_core_set_text_port(cPtr, CInt(newValue))
		}
	}
		
	
	/// Get the text port range from which is randomly chosen the UDP port used for
	/// text streaming. 
	/// - Returns: a ``Range`` object       
	public var textPortsRange: Range?
	{
	
						let cPointer = linphone_core_get_text_ports_range(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Range.getSwiftObject(cObject:cPointer!)
			belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
			return result

	}
		
	/// Sets a TLS certificate used for TLS authentication The certificate won't be
	/// stored, you have to set it after each ``Core`` startup. 
	/// - Parameter tlsCert: the TLS certificate.    
	
	/// Gets the TLS certificate. 
	/// - Returns: the TLS certificate or nil if not set yet.    
	public var tlsCert: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_core_get_tls_cert(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_core_set_tls_cert(cPtr, newValue)
		}
	}
		
	/// Sets a TLS certificate path used for TLS authentication The path will be stored
	/// in the rc file and automatically restored on startup. 
	/// - Parameter tlsCertPath: path to the TLS certificate.    
	
	/// Gets the path to the TLS certificate file. 
	/// - Returns: the TLS certificate path or nil if not set yet.    
	public var tlsCertPath: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_core_get_tls_cert_path(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_core_set_tls_cert_path(cPtr, newValue)
		}
	}
		
	/// Sets a TLS key used for TLS authentication The key won't be stored, you have to
	/// set it after each ``Core`` startup. 
	/// - Parameter tlsKey: the TLS key.    
	
	/// Gets the TLS key. 
	/// - Returns: the TLS key or nil if not set yet.    
	public var tlsKey: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_core_get_tls_key(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_core_set_tls_key(cPtr, newValue)
		}
	}
		
	/// Sets a TLS key path used for TLS authentication The path will be stored in the
	/// rc file and automatically restored on startup. 
	/// - Parameter tlsKeyPath: path to the TLS key.    
	
	/// Gets the path to the TLS key file. 
	/// - Returns: the TLS key path or nil if not set yet.    
	public var tlsKeyPath: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_core_get_tls_key_path(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_core_set_tls_key_path(cPtr, newValue)
		}
	}
		
	/// Sets the ports to be used for each of transport (UDP or TCP) A zero value port
	/// for a given transport means the transport is not used. 
	/// A value of LC_SIP_TRANSPORT_RANDOM (-1) means the port is to be chosen randomly
	/// by the system. A value of LC_SIP_TRANSPORT_DONTBIND (-2) means that the socket
	/// will not be bound explicitly, in other words liblinphone won't listen for
	/// incoming connections at all. This mode is suitable for a pure client
	/// application (ex: a mobile application). 
	/// - Parameter transports: A LinphoneSipTransports structure giving the ports to
	/// use    
	/// - Returns: 0 
	
	/// Retrieves the port configuration used for each transport (udp, tcp, tls). 
	/// A zero value port for a given transport means the transport is not used. A
	/// value of LC_SIP_TRANSPORT_RANDOM (-1) means the port is to be chosen randomly
	/// by the system. A value of LC_SIP_TRANSPORT_DONTBIND (-2) means that the socket
	/// will not be bound explicitely, in other words liblinphone won't listen for
	/// incoming connections at all. This mode is suitable for a pure client
	/// application (ex: a mobile application). 
	/// - Returns: A ``Transports`` structure with the configured ports       
	public var transports: Transports?
	{
	
						let cPointer = linphone_core_get_transports(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Transports.getSwiftObject(cObject:cPointer!)
			belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
			return result

	}
	public func setTransports(newValue: Transports) throws
	{
		let exception_result = linphone_core_set_transports(cPtr, newValue.cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "username setter returned value \(exception_result)")
		}
	}
		
	
	/// Retrieves the real port number assigned for each sip transport (udp, tcp, tls). 
	/// A zero value means that the transport is not activated. If
	/// LC_SIP_TRANSPORT_RANDOM was passed to linphone_core_set_sip_transports, the
	/// random port choosed by the system is returned. 
	/// - Returns: A ``Transports`` structure with the ports being used       
	public var transportsUsed: Transports?
	{
	
						let cPointer = linphone_core_get_transports_used(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Transports.getSwiftObject(cObject:cPointer!)
			belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
			return result

	}
		
	
	/// Gets tunnel instance if available. 
	/// - Returns: ``Tunnel`` or nil if not available.    
	public var tunnel: Tunnel?
	{
	
						let cPointer = linphone_core_get_tunnel(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Tunnel.getSwiftObject(cObject:cPointer!)
			return result

	}
		
	
	/// Returns the global unread chat message count. 
	/// - Returns: The global unread chat message count. 
	public var unreadChatMessageCount: Int
	{
	
						return Int(linphone_core_get_unread_chat_message_count(cPtr))

	}
		
	
	/// Returns the unread chat message count for all active local address. 
	/// (Primary contact + proxy configs.) 
	/// - Returns: The unread chat message count. 
	public var unreadChatMessageCountFromActiveLocals: Int
	{
	
						return Int(linphone_core_get_unread_chat_message_count_from_active_locals(cPtr))

	}
		
	/// Sets maximum available upload bandwidth This is IP bandwidth, in kbit/s. 
	/// This information is used by liblinphone together with remote side available
	/// bandwidth signaled in SDP messages to properly configure audio & video codec's
	/// output bitrate. By default, the download and upload bandwidth are unknowns (set
	/// to zero), in which case adaptive algorithms are run during calls in order to
	/// detect available bandwidth and adapt audio and video bitrate usage. - See also:
	/// ``enableAdaptiveRateControl(enabled:)``.
	/// - Parameter bandwidth: the bandwidth in kbits/s, 0 for unknown. 
	
	/// Retrieve the maximum available upload bandwidth. 
	/// This value was set by ``setUploadBandwidth(bandwidth:)``. 
	/// - Returns: the upload bandiwdth in kbits/s, 0 for unknown. 
	public var uploadBandwidth: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_upload_bandwidth(cPtr))
		}
		set
		{
			linphone_core_set_upload_bandwidth(cPtr, CInt(newValue))
		}
	}
		
	/// Sets audio packetization time linphone will send (in absence of requirement
	/// from peer) A value of 0 stands for the current codec default packetization
	/// time. 
	/// - Parameter ptime: the upload packetization time to set 
	
	/// Gets audio packetization time linphone will send (in absence of requirement
	/// from peer) A value of 0 stands for the current codec default packetization
	/// time. 
	/// - Returns: the upload packetization time set 
	public var uploadPtime: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_upload_ptime(cPtr))
		}
		set
		{
			linphone_core_set_upload_ptime(cPtr, CInt(newValue))
		}
	}
		
	
	/// Return the external ip address of router. 
	/// In some cases the uPnP can have an external ip address but not a usable uPnP
	/// (state different of Ok).
	/// - Returns: a null terminated string containing the external ip address. If the
	/// the external ip address is not available return null.    
	public var upnpExternalIpaddress: String?
	{
	
			
			let cPointer = linphone_core_get_upnp_external_ipaddress(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	
	/// Return the internal state of uPnP. 
	/// - Returns: an ``UpnpState``. 
	public var upnpState: UpnpState
	{
	
						return UpnpState(rawValue: Int(linphone_core_get_upnp_state(cPtr).rawValue))!

	}
		
	/// Ask the core to stream audio from and to files, instead of using the soundcard. 
	/// - Parameter yesno: A boolean value asking to stream audio from and to files or
	/// not. 
	
	/// Gets whether linphone is currently streaming audio from and to files, rather
	/// than using the soundcard. 
	/// - Returns: A boolean value representing whether linphone is streaming audio
	/// from and to files or not. 
	public var useFiles: Bool
	{
	
		get
		{ 
						return linphone_core_get_use_files(cPtr) != 0
		}
		set
		{
			linphone_core_set_use_files(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets whether SIP INFO method can be used to send digits. 
	/// This non-standard but common practice. 
	/// - Parameter useInfo: A boolean value telling whether to use SIP INFO to send
	/// digits 
	
	/// Indicates whether SIP INFO can be used to send digits. 
	/// - Returns: A boolean value telling whether SIP INFO is used to send digits 
	public var useInfoForDtmf: Bool
	{
	
		get
		{ 
						return linphone_core_get_use_info_for_dtmf(cPtr) != 0
		}
		set
		{
			linphone_core_set_use_info_for_dtmf(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets whether RFC2833 or RFC4633 can be to be used to send digits. 
	/// This is preferred method to reliabily transmit DTMFs codes. There are two
	/// settings relevant to dtmf sending: ``setUseRfc2833ForDtmf(useRfc2833:)`` and
	/// ``setUseInfoForDtmf(useInfo:)``; Resulting in 4 cases:
	/// -If neither are enabled, don't send anything.
	/// -If one is enabled but not the other, then send the DTMF using the one the that
	/// is enabled.
	/// -If both are enabled, use RFC2833, then SIP INFO as fallback only if the media
	/// does not support telephone-events. In that last sub-case, note that the DTMF
	/// will also be sent modulated into the audio signal.
	/// - Parameter useRfc2833: A boolean value telling whether to use RFC2833 to send
	/// digits 
	
	/// Indicates whether RFC2833/RFC4633 can be used to send digits. 
	/// - Returns: A boolean value telling whether RFC2833 is used to send digits 
	public var useRfc2833ForDtmf: Bool
	{
	
		get
		{ 
						return linphone_core_get_use_rfc2833_for_dtmf(cPtr) != 0
		}
		set
		{
			linphone_core_set_use_rfc2833_for_dtmf(cPtr, newValue==true ? 1:0)
		}
	}
		
	
	/// Gets the user-agent as a string. 
	/// - Returns: liblinphone's user agent as a string.    
	public var userAgent: String
	{
	
			
			let cPointer = linphone_core_get_user_agent(cPtr)
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
		
	/// Set the path to the directory storing the user's x509 certificates (used by
	/// dtls) 
	/// - Parameter path: The path to the directory to use to store the user's
	/// certificates.    
	
	/// Get the path to the directory storing the user's certificates. 
	/// - Returns: The path to the directory storing the user's certificates.    
	public var userCertificatesPath: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_core_get_user_certificates_path(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_core_set_user_certificates_path(cPtr, newValue)
		}
	}
		
	/// Associate a user pointer to the linphone core. 
	/// - Parameter userData: The user data to associate with the ``Core`` object.    
	
	/// Retrieves the user pointer that was given to linphone_core_new 
	/// - Returns: The user data associated with the ``Core`` object.    
	public var userData: UnsafeMutableRawPointer?
	{
	
		get
		{ 
						return linphone_core_get_user_data(cPtr)
		}
		set
		{
			linphone_core_set_user_data(cPtr, newValue)
		}
	}
		
	/// Enable vibration will incoming call is ringing (Android only). 
	/// - Parameter enable: true to enable the vibration on incoming call, false
	/// otherwise 
	
	/// Gets whether the device will vibrate while an incoming call is ringing (Android
	/// only). 
	/// - Returns: true if the device will vibrate (if possible), false otherwise 
	public var vibrationOnIncomingCallEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_vibration_on_incoming_call_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_vibration_on_incoming_call(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets the default policy for video. 
	/// This policy defines whether:
	/// -video shall be initiated by default for outgoing calls
	/// -video shall be accepted by default for incoming calls
	/// -if video shall be accepted by default, what direction do we want
	/// (send/receive, receive only, etc...)
	/// - Parameter policy: The ``VideoActivationPolicy`` to use    
	
	/// Get the default policy for video. 
	/// See ``setVideoActivationPolicy(policy:)`` for more details. 
	/// - Returns: The currently used video policy    
	public var videoActivationPolicy: VideoActivationPolicy?
	{
	
		get
		{ 
						let cPointer = linphone_core_get_video_activation_policy(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = VideoActivationPolicy.getSwiftObject(cObject:cPointer!)
			return result
		}
		set
		{
			linphone_core_set_video_activation_policy(cPtr, newValue?.cPtr)
		}
	}
		
	/// Enables or disables the video adaptive jitter compensation. 
	/// - Parameter enable: true to enable the video adaptive jitter compensation,
	/// false to disable it. 
	
	/// Tells whether the video adaptive jitter compensation is enabled. 
	/// - Returns: true if the video adaptive jitter compensation is enabled, false
	/// otherwise. 
	public var videoAdaptiveJittcompEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_video_adaptive_jittcomp_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_video_adaptive_jittcomp(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Enable or disable video capture. 
	/// This function does not have any effect during calls. It just indicates the
	/// ``Core`` to initiate future calls with video capture or not. 
	/// - Parameter enable: true to enable video capture, false to disable it. 
	
	/// Tells whether video capture is enabled. 
	/// - Returns: true if video capture is enabled, false if disabled. 
	public var videoCaptureEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_video_capture_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_video_capture(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets the priority policy for video codecs (payload types). 
	/// Since version 5.3, the default value is ``Auto`` unless the core's
	/// configuration file describes a list of video payload type to use. This is to
	/// preserve backward compatibility for upgrading applications. See
	/// ``CodecPriorityPolicy`` for more details. 
	/// - Parameter policy: the ``CodecPriorityPolicy`` to apply 
	
	/// Gets the current priority policy for video codecs (payload types). 
	/// See ``CodecPriorityPolicy`` for more details. 
	/// - Returns: the current ``CodecPriorityPolicy`` 
	public var videoCodecPriorityPolicy: CodecPriorityPolicy
	{
	
		get
		{ 
						return CodecPriorityPolicy(rawValue: Int(linphone_core_get_video_codec_priority_policy(cPtr).rawValue))!
		}
		set
		{
			linphone_core_set_video_codec_priority_policy(cPtr, LinphoneCodecPriorityPolicy(rawValue: CUnsignedInt(newValue.rawValue)))
		}
	}
		
	/// Sets the active video device. 
	/// - Parameter id: The name of the video device to use as returned by
	/// linphone_core_get_video_devices    
	
	/// Returns the name of the currently active video device. 
	/// - Returns: The name of the currently active video device.    
	public var videoDevice: String?
	{
	
			
			let cPointer = linphone_core_get_video_device(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
	public func setVideodevice(newValue: String?) throws
	{
		let exception_result = linphone_core_set_video_device(cPtr, newValue)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "username setter returned value \(exception_result)")
		}
	}
		
	
	/// Gets the list of the available video capture devices. 
	/// - Returns: An unmodifiable array of strings contanining the names of the
	/// available video capture devices that is nil terminated.         
	public var videoDevicesList: [String]
	{
	
						var swiftList = [String]()
			let cList = linphone_core_get_video_devices_list(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				swiftList.append(String(cString: unsafeBitCast(listTemp!.pointee.data, to: UnsafePointer<CChar>.self)))
				listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp!.pointee.next)
			}
				bctbx_list_free_with_data(cList, bctbx_free)
			return swiftList

	}
		
	/// Enable or disable video display. 
	/// This function does not have any effect during calls. It just indicates the
	/// ``Core`` to initiate future calls with video display or not. 
	/// - Parameter enable: true to enable video display, false to disable it. 
	
	/// Tells whether video display is enabled. 
	/// - Returns: true if video display is enabled, false if disabled. 
	public var videoDisplayEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_video_display_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_video_display(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets the name of the mediastreamer2 engine (filter) to be used for rendering
	/// video. 
	/// - Parameter filterName: the filter name to use or nil to use default.    
	
	/// Gets the name of the mediastreamer2 engine (filter) used for rendering video. 
	/// - Returns: The currently selected video display filter.    
	public var videoDisplayFilter: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_core_get_video_display_filter(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_core_set_video_display_filter(cPtr, newValue)
		}
	}
		
	/// Sets the DSCP field for outgoing video streams. 
	/// The DSCP defines the quality of service in IP packets. When RTP bundling is
	/// negociated during the call (see ``enableRtpBundle(value:)``), the video packets
	/// are sent through the audio RTP/UDP connection, which leaves the video dscp
	/// setting wihtout effect. - Note: It is usually useless or a bad idea to try to
	/// play with DSCP bits unless having full control on the network. 
	/// - Warning: Setting the DSCP bits is more or less well supported by operating
	/// systems and sometimes requires to disable IPv6. 
	/// - Parameter dscp: The DSCP value to set 
	
	/// Gets the DSCP field for outgoing video streams. 
	/// The DSCP defines the quality of service in IP packets. 
	/// - Returns: The current DSCP value 
	public var videoDscp: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_video_dscp(cPtr))
		}
		set
		{
			linphone_core_set_video_dscp(cPtr, CInt(newValue))
		}
	}
		
	
	/// Returns true if either capture or display is enabled, false otherwise. 
	/// same as ( ``videoCaptureEnabled()`` | ``videoDisplayEnabled()`` )
	/// - Returns: true if either capture or display is enabled, false otherwise. 
	public var videoEnabled: Bool
	{
	
						return linphone_core_video_enabled(cPtr) != 0

	}
		
	/// Sets the nominal video jitter buffer size in milliseconds. 
	/// The value takes effect immediately for all running and pending calls, if any. A
	/// value of 0 disables the jitter buffer.
	/// - Parameter milliseconds: the jitter buffer size in milliseconds 
	
	/// Returns the nominal video jitter buffer size in milliseconds. 
	/// - Returns: The nominal video jitter buffer size in milliseconds 
	public var videoJittcomp: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_video_jittcomp(cPtr))
		}
		set
		{
			linphone_core_set_video_jittcomp(cPtr, CInt(newValue))
		}
	}
		
	/// Use to set multicast address to be used for video stream. 
	/// - Parameter ip: an ipv4/6 multicast address.    
	/// - Returns: 0 in case of success 
	
	/// Use to get multicast address to be used for video stream. 
	/// - Returns: an ipv4/6 multicast address, or default value.    
	public var videoMulticastAddr: String?
	{
	
			
			let cPointer = linphone_core_get_video_multicast_addr(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result

	}
	public func setVideomulticastaddr(newValue: String?) throws
	{
		let exception_result = linphone_core_set_video_multicast_addr(cPtr, newValue)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "username setter returned value \(exception_result)")
		}
	}
		
	/// Use to enable multicast rtp for video stream. 
	/// If enabled, outgoing calls put a multicast address from
	/// ``getVideoMulticastAddr()`` into video cline. In case of outgoing call video
	/// stream is sent to this multicast address. For incoming calls behavior is
	/// unchanged. 
	/// - Parameter yesno: if yes, subsequent outgoing calls will propose multicast ip
	/// set by ``setVideoMulticastAddr(ip:)`` 
	
	/// Use to get multicast state of video stream. 
	/// - Returns: true if subsequent calls will propose multicast ip set by
	/// ``setVideoMulticastAddr(ip:)`` 
	public var videoMulticastEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_video_multicast_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_video_multicast(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Use to set multicast ttl to be used for video stream. 
	/// - Parameter ttl: value or -1 if not used. [0..255] default value is 1 
	/// - Returns: 0 in case of success 
	
	/// Use to get multicast ttl to be used for video stream. 
	/// - Returns: a time to leave value 
	public var videoMulticastTtl: Int
	{
	
						return Int(linphone_core_get_video_multicast_ttl(cPtr))

	}
	public func setVideomulticastttl(newValue: Int) throws
	{
		let exception_result = linphone_core_set_video_multicast_ttl(cPtr, CInt(newValue))
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "username setter returned value \(exception_result)")
		}
	}
		
	/// Redefines the list of the available video payload types (codecs). 
	/// The payload type are listed from higher priority to lowest priority. It is
	/// worth to note that the set of ``PayloadType`` objects assigned here need to be
	/// the same as the one returned by ``getVideoPayloadTypes()``. The purpose of the
	/// setter is to let application modify their order of preference. In particular,
	/// including in the provided list a payload type not supported has no effect.
	/// Removing specific payload types from the original list has no effect too:
	/// missing payload types will be automatically added. In order to disable a
	/// specific codec, applications shall use ``PayloadType/enable(enabled:)`` instead.
	/// Calling this function if the video codec priority policy is
	/// LinphoneCodecPriorityPolicyAuto turns video codec priority policy to basic
	/// scheme, since application is not supposed to control the order of video codecs
	/// when LinphoneCodecPriorityPolicyAuto is selected, by definition.
	/// - See also: ``setVideoCodecPriorityPolicy(policy:)`` 
	/// - Parameter payloadTypes: The new list of codecs. The core does not take
	/// ownership on it.      
	
	/// Returns the list of the available video payload types (codecs), in their order
	/// of preference. 
	/// - Returns: A freshly allocated list of the available payload types.         
	public var videoPayloadTypes: [PayloadType]
	{
	
		get
		{ 
						var swiftList = [PayloadType]()
			let cList = linphone_core_get_video_payload_types(cPtr)
			var listTemp = cList
			while (listTemp != nil) {
				let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
				swiftList.append(PayloadType.getSwiftObject(cObject: data))
				listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
			}
				bctbx_list_free_with_data(cList, belle_sip_object_unref)
			return swiftList
		}
		set
		{
			withExtendedLifetime(newValue) {
				var cList: UnsafeMutablePointer<bctbx_list_t>? = nil
				for data in newValue {
					cList = bctbx_list_append(cList, UnsafeMutableRawPointer(data.cPtr))
				}
				linphone_core_set_video_payload_types(cPtr, cList)
			}
		}
	}
		
	/// Sets the UDP port used for video streaming. 
	/// A value of -1 will request the system to allocate the local port randomly. This
	/// is recommended in order to avoid firewall warnings. 
	/// - Parameter port: The UDP port to use for video streaming 
	
	/// Gets the UDP port used for video streaming. 
	/// - Returns: The UDP port used for video streaming 
	public var videoPort: Int
	{
	
		get
		{ 
						return Int(linphone_core_get_video_port(cPtr))
		}
		set
		{
			linphone_core_set_video_port(cPtr, CInt(newValue))
		}
	}
		
	
	/// Get the video port range from which is randomly chosen the UDP port used for
	/// video streaming. 
	/// - Returns: a ``Range`` object       
	public var videoPortsRange: Range?
	{
	
						let cPointer = linphone_core_get_video_ports_range(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = Range.getSwiftObject(cObject:cPointer!)
			belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
			return result

	}
		
	/// Set the video preset to be used for video calls. 
	/// - Parameter preset: The name of the video preset to be used (can be nil to use
	/// the default video preset).    
	
	/// Get the video preset used for video calls. 
	/// - Returns: The name of the video preset used for video calls (can be nil if the
	/// default video preset is used).    
	public var videoPreset: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_core_get_video_preset(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_core_set_video_preset(cPtr, newValue)
		}
	}
		
	/// Controls video preview enablement. 
	/// - Parameter enable: A boolean value telling whether the video preview is to be
	/// shown Video preview refers to the action of displaying the local webcam image
	/// to the user while not in call. 
	
	/// Tells whether video preview is enabled. 
	/// - Returns: A boolean value telling whether video preview is enabled 
	public var videoPreviewEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_video_preview_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_video_preview(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Enable or disable video source reuse when switching from preview to actual
	/// video call. 
	/// This source reuse is useful when you always display the preview, even before
	/// calls are initiated. By keeping the video source for the transition to a real
	/// video call, you will smooth out the source close/reopen cycle.
	/// This function does not have any effect during calls. It just indicates the
	/// ``Core`` to initiate future calls with video source reuse or not. Also, at the
	/// end of a video call, the source will be closed whatsoever for now. 
	/// - Parameter enable: true to enable video source reuse. false to disable it for
	/// subsequent calls. 
	
	public var videoSourceReuseEnabled: Bool?
	{
	
		willSet
		{
			linphone_core_enable_video_source_reuse(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Turns Wifi only mode on or off. 
	/// If enabled, app won't register when active network isn't WiFi or Ethernet. -
	/// Warning: Only works for Android platform. 
	/// - Parameter enable: A boolean value telling whether to enable IPv6 support 
	
	/// Tells whether Wifi only mode is enabled or not. 
	/// - Warning: Only works for Android platform. 
	/// - Returns: A boolean value telling whether Wifi only mode is enabled or not 
	public var wifiOnlyEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_wifi_only_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_wifi_only(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Defines whether RTP port is set to 0 when a stream is inactive. 
	/// - Parameter enable: true to set the RTP port to 0 if the stream direction is
	/// inactive; false otherwise. 
	
	/// Checks if RTP port is set to 0 when a stream is inactive. 
	/// - Returns: true if the RTP port is set to 0 if the stream direction is
	/// inactive; false otherwise. 
	public var zeroRtpPortForStreamInactiveEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_zero_rtp_port_for_stream_inactive_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_zero_rtp_port_for_stream_inactive(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Defines whether ZRTP go clear is enabled. 
	/// - Parameter enabled: true to enable ZRTP go clear; false otherwise. 
	
	/// Checks if the ZRTP go clear is enabled or not. 
	/// - Returns: true if ZTRP go clear is enabled; false otherwise. 
	public var zrtpGoClearEnabled: Bool
	{
	
		get
		{ 
						return linphone_core_zrtp_go_clear_enabled(cPtr) != 0
		}
		set
		{
			linphone_core_enable_zrtp_go_clear(cPtr, newValue==true ? 1:0)
		}
	}
		
	/// Sets the path to the file storing the zrtp secrets cache. 
	/// - Parameter file: The path to the file to use to store the zrtp secrets cache. 
	///   
	
	/// Gets the path to the file storing the zrtp secrets cache. 
	/// - Returns: The path to the file storing the zrtp secrets cache.    
	public var zrtpSecretsFile: String?
	{
	
		get
		{ 
			
			let cPointer = linphone_core_get_zrtp_secrets_file(cPtr)
			if (cPointer == nil) {
				return nil
			}
			let result = charArrayToString(charPointer: cPointer)
			return result
		}
		set
		{
			linphone_core_set_zrtp_secrets_file(cPtr, newValue)
		}
	}
		
	
	
	/// This method is used to abort a user authentication request initiated by
	/// ``Core`` from the authentication_requested callback of ``CoreDelegate``. 
	/// - Note: this function currently does not take into account the 'info'
	/// parameter. All pending authentication requests are aborted. 
	/// - Parameter info: the ``AuthInfo`` for which to abort authentication    
	public func abortAuthentication(info:AuthInfo?) 
	{
		linphone_core_abort_authentication(cPtr, info?.cPtr)
	}
	
	
	
	/// Special function to indicate if the audio session is activated. 
	/// Must be called when ProviderDelegate of the callkit notifies that the audio
	/// session is activated or deactivated. 
	/// - Parameter activated: true to activate the audio session, false to disable it. 
	public func activateAudioSession(activated:Bool) 
	{
		linphone_core_activate_audio_session(cPtr, activated==true ? 1:0)
	}
	
	
	
	/// Adds an account. 
	/// This will start registration on the proxy, if registration is enabled. 
	/// - Parameter account: the ``Account`` to add    
	/// - Returns: 0 if successful, -1 otherwise 
	public func addAccount(account:Account) throws 
	{
		let exception_result = linphone_core_add_account(cPtr, account.cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "addAccount returned value \(exception_result)")
		}
	}
	
	
	
	/// Adds all calls into the conference. 
	/// If no conference is running a new internal conference context is created and
	/// all current calls are added to it. 
	/// - Returns: 0 if succeeded. Negative number if failed 
	/// - Warning: This function guarantees that the local endpoint is added to the
	/// conference. 
	/// - Deprecated: 23/01/2025 Use
	/// ``Conference/inviteParticipants(addresses:params:)`` instead. 
	@available(*, deprecated)
	public func addAllToConference() throws 
	{
		let exception_result = linphone_core_add_all_to_conference(cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "addAllToConference returned value \(exception_result)")
		}
	}
	
	
	
	/// Adds authentication information to the ``Core``. 
	/// These nformation will be used during all SIP or HTTP transactions that require
	/// authentication. 
	/// - Parameter info: The ``AuthInfo`` to add.    
	public func addAuthInfo(info:AuthInfo) 
	{
		linphone_core_add_auth_info(cPtr, info.cPtr)
	}
	
	
	
	/// Add support for the specified content type. 
	/// It is the application responsibility to handle it correctly afterwards. 
	/// - Parameter contentType: The content type to add support for    
	public func addContentTypeSupport(contentType:String) 
	{
		linphone_core_add_content_type_support(cPtr, contentType)
	}
	
	
	
	/// Add a friend list. 
	/// - Parameter list: ``FriendList`` object    
	public func addFriendList(list:FriendList) 
	{
		linphone_core_add_friend_list(cPtr, list.cPtr)
	}
	
	
	
	/// Add or update a LDAP server and save it to the configuration. 
	/// - Parameter ldap: The ``Ldap`` object to add/update.    
	/// - Deprecated: 18/11/2024 use
	/// ``addRemoteContactDirectory(remoteContactDirectory:)`` instead. 
	@available(*, deprecated)
	public func addLdap(ldap:Ldap) 
	{
		linphone_core_add_ldap(cPtr, ldap.cPtr)
	}
	
	
	
	/// Add the given linphone specs to the list of functionalities the linphone client
	/// supports. 
	/// - See also: ``setLinphoneSpecsList(specs:)`` 
	/// - Parameter spec: The spec to add    
	public func addLinphoneSpec(spec:String) 
	{
		linphone_core_add_linphone_spec(cPtr, spec)
	}
	
	
	
	/// Add an extra header for retrieving the remote provisioning (check
	/// ``setProvisioningUri(uri:)``). 
	/// This can also be set from configuration file or factory config file, from[misc]
	/// section, item "config-uri-headers_X" where X is the index of the header
	/// starting by 0. 
	/// - Parameter headerName: the header to use when downloading the configuration.  
	///  
	/// - Parameter value: the value to use when downloading the configuration.    
	public func addProvisioningHeader(headerName:String, value:String) 
	{
		linphone_core_add_provisioning_header(cPtr, headerName, value)
	}
	
	
	
	/// Add a proxy configuration. 
	/// This will start registration on the proxy, if registration is enabled. 
	/// - Parameter config: the ``ProxyConfig`` to add    
	/// - Returns: 0 if successful, -1 otherwise 
	/// - Deprecated: 04/09/2024 Use ``addAccount(account:)`` 
	@available(*, deprecated)
	public func addProxyConfig(config:ProxyConfig) throws 
	{
		let exception_result = linphone_core_add_proxy_config(cPtr, config.cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "addProxyConfig returned value \(exception_result)")
		}
	}
	
	
	
	/// Adds a ``RemoteContactDirectory`` object previously created to the Core, to be
	/// used later by ``MagicSearch`` to query contacts using either LDAP or CardDAV. 
	/// - Parameter remoteContactDirectory: the newly created
	/// ``RemoteContactDirectory`` to add.    
	public func addRemoteContactDirectory(remoteContactDirectory:RemoteContactDirectory) 
	{
		linphone_core_add_remote_contact_directory(cPtr, remoteContactDirectory.cPtr)
	}
	
	
	
	/// This function controls signaling features supported by the core. 
	/// They are typically included in a SIP Supported header. 
	/// - Parameter tag: The feature tag name    
	public func addSupportedTag(tag:String) 
	{
		linphone_core_add_supported_tag(cPtr, tag)
	}
	
	
	
	/// Adds a participant to the conference. 
	/// If no conference is going on a new internal conference context is created and
	/// the participant is added to it. 
	/// - Parameter call: The current call with the participant to add    
	/// - Returns: 0 if succeeded. Negative number if failed 
	/// - Deprecated: 23/01/2025 Use ``Conference/addParticipant(call:)`` instead. 
	@available(*, deprecated)
	public func addToConference(call:Call) throws 
	{
		let exception_result = linphone_core_add_to_conference(cPtr, call.cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "addToConference returned value \(exception_result)")
		}
	}
	
	
	
	/// Special function to indicate if the audio route is changed. 
	/// Must be called in the callback of AVAudioSessionRouteChangeNotification. 
	/// - Deprecated: 07/01/2020 now handled in the linphone SDK directly 
	@available(*, deprecated)
	public func audioRouteChanged() 
	{
		linphone_core_audio_route_changed(cPtr)
	}
	
	
	
	/// Check whether ringing of calls is disabled. 
	/// - Returns: true if call ringing is disabled 
	public func callRingingDisabled() -> Bool
	{
		return linphone_core_call_ringing_disabled(cPtr) != 0
	}
	
	
	
	/// Gets the default ephemeral message mode. 
	/// - Returns: the default ephemeral message mode ``ChatRoom.EphemeralMode`` 
	public func chatRoomGetDefaultEphemeralMode() -> ChatRoom.EphemeralMode
	{
		return ChatRoom.EphemeralMode(rawValue: Int(linphone_core_chat_room_get_default_ephemeral_mode(cPtr).rawValue))!
	}
	
	
	
	/// Sets the default ephemeral message mode. 
	/// - Parameter mode: default ephemeral message mode ``ChatRoom.EphemeralMode`` 
	public func chatRoomSetDefaultEphemeralMode(mode:ChatRoom.EphemeralMode) 
	{
		linphone_core_chat_room_set_default_ephemeral_mode(cPtr, LinphoneChatRoomEphemeralMode(rawValue: CUnsignedInt(mode.rawValue)))
	}
	
	
	
	/// Asynchronously checks if a new version of the application is available from a
	/// well-known http server URI given by ``Core`` 's configuration. 
	/// The result of the check is given through the ``CoreDelegate`` interface, see
	/// linphone_core_cbs_set_version_update_check_result_received. The http URI has to
	/// be given in [misc] section as key 'version_check_url_root'. The subdirectory is
	/// appended to this root URI, per platform, and a "VERSION" file is fetched. For
	/// example:
	/// -https://download.linphone.org/releases/android/RELEASE
	/// -https://download.linphone.org/releases/windows/RELEASE The RELEASE file is
	/// expected to contain the most recent version number available followed by an
	/// http uri where this version can be retrieved. For example: 5.2.5
	/// https://play.google.com/store/apps/details?id=org.linphone 
	public func checkForUpdate(currentVersion:String) 
	{
		linphone_core_check_for_update(cPtr, currentVersion)
	}
	
	
	
	/// Erases all accounts from config. 
	public func clearAccounts() 
	{
		linphone_core_clear_accounts(cPtr)
	}
	
	
	
	/// Clears all authentication information. 
	public func clearAllAuthInfo() 
	{
		linphone_core_clear_all_auth_info(cPtr)
	}
	
	
	
	/// Erases the call log list. 
	public func clearCallLogs() 
	{
		linphone_core_clear_call_logs(cPtr)
	}
	
	
	
	/// Erases all LDAP from the configuration. 
	/// - Deprecated: 18/11/2024 use
	/// ``removeRemoteContactDirectory(remoteContactDirectory:)`` instead.
	@available(*, deprecated)
	public func clearLdaps() 
	{
		linphone_core_clear_ldaps(cPtr)
	}
	
	
	
	/// Clear all headers that were added with
	/// ``addProvisioningHeader(headerName:value:)``. 
	public func clearProvisioningHeaders() 
	{
		linphone_core_clear_provisioning_headers(cPtr)
	}
	
	
	
	/// Erase all proxies from config. 
	/// - Deprecated: 04/09/2024 Use ``clearAccounts()`` 
	@available(*, deprecated)
	public func clearProxyConfig() 
	{
		linphone_core_clear_proxy_config(cPtr)
	}
	
	
	
	/// Forces a flush of the config to disk. 
	/// - Returns: 0 if successful, -1 otherwise 
	public func configSync() throws 
	{
		let exception_result = linphone_core_config_sync(cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "configSync returned value \(exception_result)")
		}
	}
	
	
	
	/// Special function to configure audio session with default settings. 
	/// Must be called in ProviderDelegate's callbacks when answer an incoming call and
	/// start an outgoing call. 
	public func configureAudioSession() 
	{
		linphone_core_configure_audio_session(cPtr)
	}
	
	
	
	/// Creates an account using given parameters, see ``createAccountParams()``. 
	/// Once created, the account must be added to the ``Core`` in order to be used for
	/// registration, calls, messages etc. Use ``addAccount(account:)`` to add it to
	/// the ``Core``. 
	/// - Parameter params: ``AccountParams`` object    
	/// - Returns: ``Account`` with default values set    
	public func createAccount(params:AccountParams) throws -> Account
	{
		let cPointer = linphone_core_create_account(cPtr, params.cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null Account value")
		}
		let result = Account.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Create a ``AccountCreator`` and set Linphone Request callbacks. 
	/// - Parameter xmlrpcUrl: The URL to the XML-RPC server.    
	/// - Returns: The new ``AccountCreator`` object.   
	/// - Deprecated: 04/09/2024 : The ``AccountCreator`` interface is replaced by the
	/// ``AccountManagerServices`` interface. 
	@available(*, deprecated)
	public func createAccountCreator(xmlrpcUrl:String?) throws -> AccountCreator
	{
		let cPointer = linphone_core_create_account_creator(cPtr, xmlrpcUrl)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null AccountCreator value")
		}
		let result = AccountCreator.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a ``AccountManagerServices``. 
	/// - Returns: The new ``AccountManagerServices`` object.    
	public func createAccountManagerServices() throws -> AccountManagerServices
	{
		let cPointer = linphone_core_create_account_manager_services(cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null AccountManagerServices value")
		}
		let result = AccountManagerServices.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Create an account params using default values from Linphone core. 
	/// - Returns: ``AccountParams`` with default values set    
	public func createAccountParams() throws -> AccountParams
	{
		let cPointer = linphone_core_create_account_params(cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null AccountParams value")
		}
		let result = AccountParams.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a ``Address`` object by parsing the user supplied address, given as a
	/// string. 
	/// - Parameter address: String containing the user supplied address    
	/// - Returns: The created ``Address`` object    
	/// - Deprecated: 04/06/2024 use ``Factory/createAddress(addr:)``.
	@available(*, deprecated)
	public func createAddress(address:String?) throws -> Address
	{
		let cPointer = linphone_core_create_address(cPtr, address)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null Address value")
		}
		let result = Address.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a fake ``CallLog``. 
	/// - Parameter from: ``Address`` of caller    
	/// - Parameter to: ``Address`` of callee    
	/// - Parameter dir: ``Call.Dir`` of call 
	/// - Parameter duration: call length in seconds 
	/// - Parameter startTime: timestamp of call start time 
	/// - Parameter connectedTime: timestamp of call connection 
	/// - Parameter status: ``Call.Status`` of call 
	/// - Parameter videoEnabled: whether video was enabled or not for this call 
	/// - Parameter quality: call quality 
	/// - Returns: a ``CallLog`` object    
	public func createCallLog(from:Address, to:Address, dir:Call.Dir, duration:Int, startTime:time_t, connectedTime:time_t, status:Call.Status, videoEnabled:Bool, quality:Float) throws -> CallLog
	{
		let cPointer = linphone_core_create_call_log(cPtr, from.cPtr, to.cPtr, LinphoneCallDir(rawValue: CUnsignedInt(dir.rawValue)), CInt(duration), startTime, connectedTime, LinphoneCallStatus(rawValue: CUnsignedInt(status.rawValue)), videoEnabled==true ? 1:0, quality)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null CallLog value")
		}
		let result = CallLog.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a ``CallParams`` suitable for ``inviteWithParams(url:params:)``,
	/// ``Call/acceptWithParams(params:)``,
	/// ``Call/acceptEarlyMediaWithParams(params:)``, ``Call/update(params:)`` or
	/// linphone_call_accept_update_with_params(). 
	/// The parameters are initialized according to the current ``Core`` configuration
	/// and the current state of the call if any. In order to create a ``CallParams``
	/// for an already created call, the call argument must not be null. 
	/// - Parameter call: ``Call`` for which the parameters are to be built, or nil in
	/// the case where the parameters are to be used for a new outgoing call.    
	/// - Returns: A new ``CallParams`` object.    
	public func createCallParams(call:Call?) throws -> CallParams
	{
		let cPointer = linphone_core_create_call_params(cPtr, call?.cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null CallParams value")
		}
		let result = CallParams.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a new CardDAV server params object that may be used later by
	/// ``MagicSearch`` to query contacts. 
	/// - Returns: a new ``CardDavParams`` object to configure, and then use it to
	/// create a ``RemoteContactDirectory`` using
	/// ``createCardDavRemoteContactDirectory(params:)``.    
	public func createCardDavParams() throws -> CardDavParams
	{
		let cPointer = linphone_core_create_card_dav_params(cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null CardDavParams value")
		}
		let result = CardDavParams.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a new CardDAV server params object that may be used later by
	/// ``MagicSearch`` to query contacts. 
	/// - Parameter params: the ``CardDavParams`` to configure this remote contact
	/// directory.    
	/// - Returns: a new ``RemoteContactDirectory`` object to configure, and then add
	/// using ``addRemoteContactDirectory(remoteContactDirectory:)``.    
	public func createCardDavRemoteContactDirectory(params:CardDavParams) throws -> RemoteContactDirectory
	{
		let cPointer = linphone_core_create_card_dav_remote_contact_directory(cPtr, params.cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null RemoteContactDirectory value")
		}
		let result = RemoteContactDirectory.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Create a SIP conference scheduler that can be used to create client conferences
	/// for now or later and then send conference info as an ICS through chat. 
	/// A CCMPConferenceScheduler creates a conference on a server by using the CCMP
	/// protocol 
	/// - Parameter account: The ``Account`` to use in the ``ConferenceScheduler``.    
	/// - Returns: A pointer on the freshly created ``ConferenceScheduler``.    
	public func createCcmpConferenceScheduler(account:Account?) throws -> ConferenceScheduler
	{
		let cPointer = linphone_core_create_ccmp_conference_scheduler(cPtr, account?.cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null ConferenceScheduler value")
		}
		let result = ConferenceScheduler.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Create a chat room. 
	/// - Parameter params: The chat room creation parameters ``ConferenceParams``    
	/// - Parameter participants: The initial list of participants of the chat room.   
	///   
	/// - Returns: The newly created chat room (can be an existing one if backend is
	/// Basic) or nil.    
	public func createChatRoom(params:ConferenceParams, participants:[Address]) throws -> ChatRoom
	{
		let cPointer = linphone_core_create_chat_room_7(cPtr, params.cPtr, ObjectArrayToBctbxList(list: participants))
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null ChatRoom value")
		}
		let result = ChatRoom.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Create a chat room. 
	/// - Parameter params: The chat room creation parameters ``ChatRoomParams``    
	/// - Parameter localAddr: ``Address`` representing the local proxy configuration
	/// to use for the chat room creation    
	/// - Parameter subject: The subject of the group chat room    
	/// - Parameter participants: The initial list of participants of the chat room    
	///  
	/// - Returns: The newly created chat room.    
	/// - Deprecated: 02/07/2020, use ``createChatRoom(params:participants:)`` instead
	@available(*, deprecated)
	public func createChatRoom(params:ChatRoomParams, localAddr:Address, subject:String, participants:[Address]) throws -> ChatRoom
	{
		let cPointer = linphone_core_create_chat_room(cPtr, params.cPtr, localAddr.cPtr, subject, ObjectArrayToBctbxList(list: participants))
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null ChatRoom value")
		}
		let result = ChatRoom.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Create a chat room. 
	/// - Parameter params: The chat room creation parameters ``ChatRoomParams``    
	/// - Parameter subject: The subject of the group chat room    
	/// - Parameter participants: The initial list of participants of the chat room.   
	///   
	/// - Returns: The newly created chat room.    
	/// - Deprecated: 02/07/2020, use ``createChatRoom(params:participants:)`` instead
	@available(*, deprecated)
	public func createChatRoom(params:ChatRoomParams, subject:String, participants:[Address]) throws -> ChatRoom
	{
		let cPointer = linphone_core_create_chat_room_2(cPtr, params.cPtr, subject, ObjectArrayToBctbxList(list: participants))
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null ChatRoom value")
		}
		let result = ChatRoom.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// - Parameter subject: The subject of the group chat room    
	/// - Parameter participants: The initial list of participants of the chat room.   
	///   
	/// - Returns: The newly created chat room.    
	/// - Deprecated: 02/07/2020, use ``createChatRoom(params:participants:)`` instead
	@available(*, deprecated)
	public func createChatRoom(subject:String, participants:[Address]) throws -> ChatRoom
	{
		let cPointer = linphone_core_create_chat_room_3(cPtr, subject, ObjectArrayToBctbxList(list: participants))
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null ChatRoom value")
		}
		let result = ChatRoom.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// - Parameter params: The chat room creation parameters ``ChatRoomParams``    
	/// - Parameter localAddr: ``Address`` representing the local proxy configuration
	/// to use for the chat room creation    
	/// - Parameter participant: ``Address`` representing the initial participant to
	/// add to the chat room    
	/// - Returns: The newly created chat room.    
	/// - Deprecated: 02/07/2020, use ``createChatRoom(params:participants:)`` instead
	@available(*, deprecated)
	public func createChatRoom(params:ChatRoomParams, localAddr:Address, participant:Address) throws -> ChatRoom
	{
		let cPointer = linphone_core_create_chat_room_4(cPtr, params.cPtr, localAddr.cPtr, participant.cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null ChatRoom value")
		}
		let result = ChatRoom.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// - Parameter participant: ``Address`` representing the initial participant to
	/// add to the chat room    
	/// - Returns: The newly created chat room.    
	/// - Deprecated: 02/07/2020, use ``createChatRoom(params:participants:)`` instead
	@available(*, deprecated)
	public func createChatRoom(participant:Address) throws -> ChatRoom
	{
		let cPointer = linphone_core_create_chat_room_5(cPtr, participant.cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null ChatRoom value")
		}
		let result = ChatRoom.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Create a chat room. 
	/// - Parameter params: The chat room creation parameters ``ChatRoomParams``    
	/// - Parameter localAddr: ``Address`` of a local ``Account`` identity or nil    
	/// - Parameter participants: The initial list of participants of the chat room.   
	///   
	/// - Returns: The newly created chat room (can be an existing one if backend is
	/// Basic) or nil.    
	/// - Deprecated: 22/10/2024, use ``createChatRoom(params:participants:)`` instead
	@available(*, deprecated)
	public func createChatRoom(params:ChatRoomParams, localAddr:Address?, participants:[Address]) throws -> ChatRoom
	{
		let cPointer = linphone_core_create_chat_room_6(cPtr, params.cPtr, localAddr?.cPtr, ObjectArrayToBctbxList(list: participants))
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null ChatRoom value")
		}
		let result = ChatRoom.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates some default conference parameters for instanciating a conference with
	/// ``createConferenceWithParams(params:)``. 
	/// - Parameter conference: ``Conference`` for which the parameters are to be
	/// build, or nil in the case where the parameters are to be used for a new
	/// conference.    
	/// - Returns: a ``ConferenceParams`` object.    
	public func createConferenceParams(conference:Conference?) throws -> ConferenceParams
	{
		let cPointer = linphone_core_create_conference_params_2(cPtr, conference?.cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null ConferenceParams value")
		}
		let result = ConferenceParams.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Create a conference scheduler that can be used to schedule conferences on a
	/// client conference service and then send conference information invitation as an
	/// ICS object through chat. 
	/// The default account (see ``getDefaultAccount()`` ) is used to determine which
	/// kind of conference scheduler is created. 
	/// - Returns: A pointer on the freshly created ``ConferenceScheduler``.    
	public func createConferenceScheduler() throws -> ConferenceScheduler
	{
		let cPointer = linphone_core_create_conference_scheduler(cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null ConferenceScheduler value")
		}
		let result = ConferenceScheduler.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Create a conference scheduler that can be used to create client conferences for
	/// now or later and then send conference info as an ICS through chat. 
	/// A SIP-based implementation is created if the ``Account`` has not defined the
	/// URL of a CCMP server, other it will create a implementation relying on CCMP
	/// protocol. 
	/// - Parameter account: The ``Account`` to use in the ``ConferenceScheduler``.    
	/// - Returns: A pointer on the freshly created ``ConferenceScheduler``.    
	public func createConferenceScheduler(account:Account?) throws -> ConferenceScheduler
	{
		let cPointer = linphone_core_create_conference_scheduler_2(cPtr, account?.cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null ConferenceScheduler value")
		}
		let result = ConferenceScheduler.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Create a conference scheduler that can be used to create client conferences for
	/// now or later and then send conference info as an ICS through chat by specifying
	/// its type. 
	/// - Parameter account: The ``Account`` to use in the ``ConferenceScheduler``.    
	/// - Parameter schedulingType: The type of the ``ConferenceScheduler``. 
	/// - Returns: A pointer on the freshly created ``ConferenceScheduler``.    
	public func createConferenceSchedulerWithType(account:Account?, schedulingType:ConferenceScheduler.Kind) throws -> ConferenceScheduler
	{
		let cPointer = linphone_core_create_conference_scheduler_with_type(cPtr, account?.cPtr, LinphoneConferenceSchedulerType(rawValue: CUnsignedInt(schedulingType.rawValue)))
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null ConferenceScheduler value")
		}
		let result = ConferenceScheduler.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Create a conference. 
	/// Local or client conference is determinated from the 'conference_type' variable
	/// in the 'misc' section of the configuration, or by the factory address
	/// parameter. See ``ConferenceParams/setConferenceFactoryAddress(address:)`` for
	/// more details. 
	/// - Parameter params: Parameters of the conference. See ``ConferenceParams``.    
	/// - Returns: A pointer on the freshly created conference ``Conference``. That
	/// object will be automatically freed by the core after calling
	/// ``terminateConference()``.    
	/// - Warning: To guarantee the backward comatibility, this method will assign the
	/// created conference to the conference context held by the core. Nonetheless
	/// starting from release 5.4, the conference context will be override at every
	/// conference created by calling this method 
	public func createConferenceWithParams(params:ConferenceParams) throws -> Conference
	{
		let cPointer = linphone_core_create_conference_with_params(cPtr, params.cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null Conference value")
		}
		let result = Conference.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Create a ``Config`` object from a user config file. 
	/// - Parameter filename: The filename of the config file to read to fill the
	/// instantiated ``Config``    
	/// - Returns: a ``Config`` object.    
	public func createConfig(filename:String?) throws -> Config
	{
		let cPointer = linphone_core_create_config(cPtr, filename)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null Config value")
		}
		let result = Config.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a content with default values from Linphone core. 
	/// - Returns: ``Content`` object with default values set    
	public func createContent() throws -> Content
	{
		let cPointer = linphone_core_create_content(cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null Content value")
		}
		let result = Content.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Create a database conference scheduler that can be used to create client
	/// conferences for now or later and then send conference info as an ICS through
	/// chat. 
	/// The DBConferenceScheduler only creates a conference info to be stored in the
	/// database of the linphone core 
	/// - Parameter account: The ``Account`` to use in the ``ConferenceScheduler``.    
	/// - Returns: A pointer on the freshly created ``ConferenceScheduler``.    
	public func createDbConferenceScheduler(account:Account?) throws -> ConferenceScheduler
	{
		let cPointer = linphone_core_create_db_conference_scheduler(cPtr, account?.cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null ConferenceScheduler value")
		}
		let result = ConferenceScheduler.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates and returns the default chat room parameters. 
	/// - Returns: A ``ChatRoomParams`` object    
	public func createDefaultChatRoomParams() throws -> ChatRoomParams
	{
		let cPointer = linphone_core_create_default_chat_room_params(cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null ChatRoomParams value")
		}
		let result = ChatRoomParams.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Gets a ``EktInfo`` from an XML body. 
	/// - Parameter xmlBody: the string containing the XML body    
	/// - Returns: The ``EktInfo``    
	public func createEktInfoFromXml(xmlBody:String) throws -> EktInfo
	{
		let cPointer = linphone_core_create_ekt_info_from_xml(cPtr, xmlBody)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null EktInfo value")
		}
		let result = EktInfo.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates an empty ``Friend``. 
	/// - Returns: The created ``Friend`` object    
	public func createFriend() throws -> Friend
	{
		let cPointer = linphone_core_create_friend(cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null Friend value")
		}
		let result = Friend.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a ``Friend`` from ai ``Vcard``. 
	/// - Parameter vcard: a ``Vcard`` object    
	/// - Returns: A new ``Friend`` object which has its vCard attribute initialized
	/// from the given vCard, accessible using ``Friend/getVcard()``.    
	public func createFriendFromVcard(vcard:Vcard) throws -> Friend
	{
		let cPointer = linphone_core_create_friend_from_vcard(cPtr, vcard.cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null Friend value")
		}
		let result = Friend.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a new empty ``FriendList`` object. 
	/// - Returns: A new ``FriendList`` object.    
	public func createFriendList() throws -> FriendList
	{
		let cPointer = linphone_core_create_friend_list(cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null FriendList value")
		}
		let result = FriendList.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a ``Friend`` from the given address. 
	/// - Parameter address: A string containing the address to create the ``Friend``
	/// from    
	/// - Returns: The created ``Friend`` object.    
	public func createFriendWithAddress(address:String) throws -> Friend
	{
		let cPointer = linphone_core_create_friend_with_address(cPtr, address)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null Friend value")
		}
		let result = Friend.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates an empty info message. 
	/// - Returns: a new ``InfoMessage``.   
	/// The info message can later be filled with information using
	/// ``InfoMessage/addHeader(name:value:)`` or ``InfoMessage/setContent(content:)``,
	/// and finally sent with ``Call/sendInfoMessage(info:)``. 
	public func createInfoMessage() throws -> InfoMessage
	{
		let cPointer = linphone_core_create_info_message(cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null InfoMessage value")
		}
		let result = InfoMessage.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates an empty LDAP search. 
	/// ``Ldap/setParams(params:)`` must be call to save the parameters in the
	/// configuration file.
	/// - Returns: ``Ldap`` with default values set       
	/// - Deprecated: 18/11/2024 use ``createLdapRemoteContactDirectory(params:)``
	/// instead. 
	@available(*, deprecated)
	public func createLdap() throws -> Ldap
	{
		let cPointer = linphone_core_create_ldap(cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null Ldap value")
		}
		let result = Ldap.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Create a LDAP params using default values from Linphone core. 
	/// Check ``LdapParams`` to update values. In order to add a new LDAP configuration
	/// to ``MagicSearch``, these parameters must be passed to
	/// linphone_core_create_ldap_with_params. Or, use ``Ldap/setParams(params:)``.
	/// - Returns: ``LdapParams`` with default values set.       
	public func createLdapParams() throws -> LdapParams
	{
		let cPointer = linphone_core_create_ldap_params(cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null LdapParams value")
		}
		let result = LdapParams.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a new CardDAV server params object that may be used later by
	/// ``MagicSearch`` to query contacts. 
	/// - Parameter params: the ``LdapParams`` to configure this remote contact
	/// directory.    
	/// - Returns: a new ``RemoteContactDirectory`` object to configure, and then add
	/// using ``addRemoteContactDirectory(remoteContactDirectory:)``.    
	public func createLdapRemoteContactDirectory(params:LdapParams) throws -> RemoteContactDirectory
	{
		let cPointer = linphone_core_create_ldap_remote_contact_directory(cPtr, params.cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null RemoteContactDirectory value")
		}
		let result = RemoteContactDirectory.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a LDAP search using given parameters, adds it to the core list and
	/// stores them in the configuration file. 
	/// - Parameter params: ``LdapParams`` object    
	/// - Returns: ``Ldap`` object       
	/// - Deprecated: 18/11/2024 use ``createLdapRemoteContactDirectory(params:)``
	/// instead. 
	@available(*, deprecated)
	public func createLdapWithParams(params:LdapParams) throws -> Ldap
	{
		let cPointer = linphone_core_create_ldap_with_params(cPtr, params.cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null Ldap value")
		}
		let result = Ldap.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a media file player, that can be used to play audio and video to the
	/// user, outside of any call or conference. 
	/// See ``getSupportedFileFormatsList()`` for supported multimedia file types. 
	/// - Parameter soundCardName: Playback sound card. If nil, the ringer sound card
	/// set in ``Core`` will be used    
	/// - Parameter videoDisplayName: Video display. If nil, the video display set in
	/// ``Core`` will be used    
	/// - Parameter windowId: Id of the drawing window. See
	/// ``setNativeVideoWindowId(windowId:)`` for a discussion about supported native
	/// video window types.    
	/// - Returns: A pointer on the new instance. nil if failed.    
	public func createLocalPlayer(soundCardName:String?, videoDisplayName:String?, windowId:UnsafeMutableRawPointer?) throws -> Player
	{
		let cPointer = linphone_core_create_local_player(cPtr, soundCardName, videoDisplayName, windowId)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null Player value")
		}
		let result = Player.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a ``MagicSearch`` object. 
	/// - Returns: The created ``MagicSearch`` object    
	public func createMagicSearch() throws -> MagicSearch
	{
		let cPointer = linphone_core_create_magic_search(cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null MagicSearch value")
		}
		let result = MagicSearch.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Create a new ``NatPolicy`` object with every policies being disabled. 
	/// - Returns: A new ``NatPolicy`` object.    
	public func createNatPolicy() throws -> NatPolicy
	{
		let cPointer = linphone_core_create_nat_policy(cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null NatPolicy value")
		}
		let result = NatPolicy.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Create a Window ID for the video preview window. 
	/// Available for MSQOGL and MSOGL. see ``setNativeVideoWindowId(windowId:)`` for
	/// details about window_id
	/// Available for MSQOGL and MSOGL. see ``setNativeVideoWindowId(windowId:)`` for
	/// details about window_id
	/// MSQOgl can be used for the creation. ``createNativePreviewWindowId(context:)``
	/// returns a QQuickFramebufferObject::Renderer. This object must be returned by
	/// your QQuickFramebufferObject::createRenderer() overload for Qt.
	/// linphone_core_set_native_preview_window_id_2() must be called with this object
	/// after the creation. Note : Qt blocks GUI thread when calling createRenderer(),
	/// so it is safe to call linphone functions there if needed.
	/// MSQOgl can be used for the creation. ``createNativePreviewWindowId(context:)``
	/// returns a QQuickFramebufferObject::Renderer. This object must be returned by
	/// your QQuickFramebufferObject::createRenderer() overload for Qt.
	/// linphone_core_set_native_preview_window_id_2() must be called with this object
	/// after the creation. Note : Qt blocks GUI thread when calling createRenderer(),
	/// so it is safe to call linphone functions there if needed.
	/// A context can be used to prevent Linphone from allocating the container
	/// (MSOglContextInfo for MSOGL). nil if not used.
	/// A context can be used to prevent Linphone from allocating the container
	/// (MSOglContextInfo for MSOGL). nil if not used.
	/// - Parameter context: preallocated Window ID (Used only for MSOGL)    
	/// - Returns: The created Window ID.    
	public func createNativePreviewWindowId(context:UnsafeMutableRawPointer?) throws -> UnsafeMutableRawPointer
	{
		return linphone_core_create_native_preview_window_id_2(cPtr, context)
	}
	
	
	
	/// Create a native window handle for the video preview window. 
	/// see ``createNativePreviewWindowId(context:)`` for details
	/// - Returns: The native window handle of the video preview window.    
	public func createNativePreviewWindowId() throws -> UnsafeMutableRawPointer
	{
		return linphone_core_create_native_preview_window_id(cPtr)
	}
	
	
	
	/// Create a Window ID from the current call. 
	/// Available for MSQOGL and MSOGL. see ``setNativeVideoWindowId(windowId:)`` for
	/// details about window_id
	/// Available for MSQOGL and MSOGL. see ``setNativeVideoWindowId(windowId:)`` for
	/// details about window_id
	/// When MSQOgl can be used for the creation:
	/// ``createNativeVideoWindowId(context:)`` returns a
	/// QQuickFramebufferObject::Renderer. This object must be returned by your
	/// QQuickFramebufferObject::createRenderer() overload for Qt.
	/// ``setNativeVideoWindowId(windowId:)`` must be called with this object after the
	/// creation. Note : Qt blocks GUI thread when calling createRenderer(), so it is
	/// safe to call linphone functions there if needed.
	/// When MSQOgl can be used for the creation:
	/// ``createNativeVideoWindowId(context:)`` returns a
	/// QQuickFramebufferObject::Renderer. This object must be returned by your
	/// QQuickFramebufferObject::createRenderer() overload for Qt.
	/// ``setNativeVideoWindowId(windowId:)`` must be called with this object after the
	/// creation. Note : Qt blocks GUI thread when calling createRenderer(), so it is
	/// safe to call linphone functions there if needed.
	/// A context can be used to prevent Linphone from allocating the container
	/// (MSOglContextInfo for MSOGL). nil if not used.
	/// A context can be used to prevent Linphone from allocating the container
	/// (MSOglContextInfo for MSOGL). nil if not used.
	/// - Parameter context: preallocated Window ID (Used only for MSOGL)    
	/// - Returns: The created Window ID    
	public func createNativeVideoWindowId(context:UnsafeMutableRawPointer?) throws -> UnsafeMutableRawPointer
	{
		return linphone_core_create_native_video_window_id_2(cPtr, context)
	}
	
	
	
	/// Create a native window handle for the video window from the current call. 
	/// see ``createNativeVideoWindowId(context:)`` for details
	/// - Returns: The native window handle of the video window.    
	public func createNativeVideoWindowId() throws -> UnsafeMutableRawPointer
	{
		return linphone_core_create_native_video_window_id(cPtr)
	}
	
	
	
	/// Creates an out-of-dialog notification, specifying the destination resource, the
	/// event name. 
	/// The notification can be send with ``Event/notify(body:)``. 
	/// - Parameter resource: the destination resource    
	/// - Parameter event: the event name    
	/// - Returns: a ``Event`` holding the context of the notification.    
	public func createNotify(resource:Address, event:String) throws -> Event
	{
		let cPointer = linphone_core_create_notify(cPtr, resource.cPtr, event)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null Event value")
		}
		let result = Event.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a publish context for a one-shot publish. 
	/// After being created, the publish must be sent using
	/// ``Event/sendPublish(body:)``. The ``Event`` is automatically terminated when
	/// the publish transaction is finished, either with success or failure. The
	/// application must not call ``Event/terminate()`` for such one-shot publish. 
	/// - Parameter resource: the resource uri for the event    
	/// - Parameter event: the event name    
	/// - Returns: the ``Event`` holding the context of the publish.    
	public func createOneShotPublish(resource:Address, event:String) throws -> Event
	{
		let cPointer = linphone_core_create_one_shot_publish(cPtr, resource.cPtr, event)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null Event value")
		}
		let result = Event.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a ``PresenceActivity`` with the given type and description. 
	/// - Parameter acttype: The ``PresenceActivity.Kind`` to set for the activity. 
	/// - Parameter description: An additional description of the activity to set for
	/// the activity. Can be nil if no additional description is to be added.    
	/// - Returns: The created ``PresenceActivity`` object.    
	public func createPresenceActivity(acttype:PresenceActivity.Kind, description:String?) throws -> PresenceActivity
	{
		let cPointer = linphone_core_create_presence_activity(cPtr, LinphonePresenceActivityType(rawValue: CUnsignedInt(acttype.rawValue)), description)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null PresenceActivity value")
		}
		let result = PresenceActivity.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a default ``PresenceModel``. 
	/// - Returns: The created ``PresenceModel`` object.    
	public func createPresenceModel() throws -> PresenceModel
	{
		let cPointer = linphone_core_create_presence_model(cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null PresenceModel value")
		}
		let result = PresenceModel.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a ``PresenceModel`` with the given activity type and activity
	/// description. 
	/// - Parameter acttype: The ``PresenceActivity.Kind`` to set for the activity of
	/// the created model. 
	/// - Parameter description: An additional description of the activity to set for
	/// the activity. Can be nil if no additional description is to be added.    
	/// - Returns: The created ``PresenceModel`` object.    
	public func createPresenceModelWithActivity(acttype:PresenceActivity.Kind, description:String?) throws -> PresenceModel
	{
		let cPointer = linphone_core_create_presence_model_with_activity(cPtr, LinphonePresenceActivityType(rawValue: CUnsignedInt(acttype.rawValue)), description)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null PresenceModel value")
		}
		let result = PresenceModel.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a ``PresenceModel`` with the given activity type, activity description,
	/// note content and note language. 
	/// - Parameter acttype: The ``PresenceActivity.Kind`` to set for the activity of
	/// the created model. 
	/// - Parameter description: An additional description of the activity to set for
	/// the activity. Can be nil if no additional description is to be added.    
	/// - Parameter note: The content of the note to be added to the created model.    
	/// - Parameter lang: The language of the note to be added to the created model.    
	/// - Returns: The created ``PresenceModel`` object.    
	public func createPresenceModelWithActivityAndNote(acttype:PresenceActivity.Kind, description:String?, note:String, lang:String?) throws -> PresenceModel
	{
		let cPointer = linphone_core_create_presence_model_with_activity_and_note(cPtr, LinphonePresenceActivityType(rawValue: CUnsignedInt(acttype.rawValue)), description, note, lang)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null PresenceModel value")
		}
		let result = PresenceModel.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a ``PresenceNote`` with the given content and language. 
	/// - Parameter content: The content of the note to be created.    
	/// - Parameter lang: The language of the note to be created.    
	/// - Returns: The created ``PresenceNote`` object.    
	public func createPresenceNote(content:String, lang:String?) throws -> PresenceNote
	{
		let cPointer = linphone_core_create_presence_note(cPtr, content, lang)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null PresenceNote value")
		}
		let result = PresenceNote.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a ``PresencePerson`` with the given id. 
	/// - Parameter id: The id of the person to be created.    
	/// - Returns: The created ``PresencePerson`` object.    
	public func createPresencePerson(id:String) throws -> PresencePerson
	{
		let cPointer = linphone_core_create_presence_person(cPtr, id)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null PresencePerson value")
		}
		let result = PresencePerson.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a ``PresenceService`` with the given id, basic status and contact. 
	/// - Parameter id: The id of the service to be created.    
	/// - Parameter basicStatus: The basic status of the service to be created. 
	/// - Parameter contact: A string containing a contact information corresponding to
	/// the service to be created.    
	/// - Returns: The created ``PresenceService`` object.    
	public func createPresenceService(id:String, basicStatus:PresenceBasicStatus, contact:String) throws -> PresenceService
	{
		let cPointer = linphone_core_create_presence_service(cPtr, id, LinphonePresenceBasicStatus(rawValue: CUnsignedInt(basicStatus.rawValue)), contact)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null PresenceService value")
		}
		let result = PresenceService.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Same as ``getPrimaryContact()`` but the result is a ``Address`` object instead
	/// of const char *. 
	/// - Returns: a ``Address`` object.   
	/// - Deprecated: prefer using ``getPrimaryContactAddress()`` 
	@available(*, deprecated)
	public func createPrimaryContactParsed() throws -> Address
	{
		let cPointer = linphone_core_create_primary_contact_parsed(cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null Address value")
		}
		let result = Address.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Create a proxy config with default values from Linphone core. 
	/// - Returns: ``ProxyConfig`` with default values set    
	/// - Deprecated: 04/09/2024 Use ``createAccount(params:)`` 
	@available(*, deprecated)
	public func createProxyConfig() throws -> ProxyConfig
	{
		let cPointer = linphone_core_create_proxy_config(cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null ProxyConfig value")
		}
		let result = ProxyConfig.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a publish context for an event state. 
	/// After being created, the publish must be sent using
	/// ``Event/sendPublish(body:)``. After expiry, the publication is refreshed unless
	/// it is terminated before. 
	/// - Parameter resource: the resource uri for the event    
	/// - Parameter event: the event name    
	/// - Parameter expires: the lifetime of event being published, -1 if no associated
	/// duration, in which case it will not be refreshed. 
	/// - Returns: the ``Event`` holding the context of the publish.    
	public func createPublish(resource:Address, event:String, expires:Int) throws -> Event
	{
		let cPointer = linphone_core_create_publish(cPtr, resource.cPtr, event, CInt(expires))
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null Event value")
		}
		let result = Event.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates an independant media file recorder, that can be used to record user's
	/// voice or video outside of any call or conference. 
	/// See ``getSupportedFileFormatsList()`` for supported multimedia file types. 
	/// - Parameter params: The ``RecorderParams`` that will contains all recorder
	/// parameters.    
	/// - Returns: A pointer on the new instance. nil if failed.    
	public func createRecorder(params:RecorderParams) throws -> Recorder
	{
		let cPointer = linphone_core_create_recorder(cPtr, params.cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null Recorder value")
		}
		let result = Recorder.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates an object that holds recording parameters. 
	/// - See also: ``createRecorder(params:)`` 
	/// - Returns: A pointer on the newly created instance of ``RecorderParams``    
	public func createRecorderParams() throws -> RecorderParams
	{
		let cPointer = linphone_core_create_recorder_params(cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null RecorderParams value")
		}
		let result = RecorderParams.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Create a SIP conference scheduler that can be used to create client conferences
	/// for now or later and then send conference info as an ICS through chat. 
	/// A SipConferenceScheduler creates a conference on a server by using a SIP dialog 
	/// - Parameter account: The ``Account`` to use in the ``ConferenceScheduler``.    
	/// - Returns: A pointer on the freshly created ``ConferenceScheduler``.    
	public func createSipConferenceScheduler(account:Account?) throws -> ConferenceScheduler
	{
		let cPointer = linphone_core_create_sip_conference_scheduler(cPtr, account?.cPtr)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null ConferenceScheduler value")
		}
		let result = ConferenceScheduler.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates an outgoing subscription, specifying the destination resource, the
	/// event name, and an optional content body. 
	/// If accepted, the subscription runs for a finite period, but is automatically
	/// renewed if not terminated before. Unlike
	/// ``subscribe(resource:event:expires:body:)`` the subscription isn't sent
	/// immediately. It will be send when calling ``Event/sendSubscribe(body:)``. 
	/// - Parameter resource: the destination resource    
	/// - Parameter event: the event name    
	/// - Parameter expires: the whished duration of the subscription 
	/// - Returns: a ``Event`` holding the context of the created subcription.    
	public func createSubscribe(resource:Address, event:String, expires:Int) throws -> Event
	{
		let cPointer = linphone_core_create_subscribe(cPtr, resource.cPtr, event, CInt(expires))
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null Event value")
		}
		let result = Event.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates an outgoing subscription, specifying the destination resource, the
	/// event name, and an optional content body. 
	/// If accepted, the subscription runs for a finite period, but is automatically
	/// renewed if not terminated before. Unlike
	/// ``subscribe(resource:event:expires:body:)`` the subscription isn't sent
	/// immediately. It will be send when calling ``Event/sendSubscribe(body:)``. 
	/// - Parameter resource: the destination resource    
	/// - Parameter proxy: the proxy configuration to use    
	/// - Parameter event: the event name    
	/// - Parameter expires: the whished duration of the subscription 
	/// - Returns: a ``Event`` holding the context of the created subcription.    
	public func createSubscribe(resource:Address, proxy:ProxyConfig, event:String, expires:Int) throws -> Event
	{
		let cPointer = linphone_core_create_subscribe_2(cPtr, resource.cPtr, proxy.cPtr, event, CInt(expires))
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null Event value")
		}
		let result = Event.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Creates a vCard from a text, if possible. 
	/// - Parameter input: the vCard raw text to parse.    
	/// - Returns: a new ``Vcard`` object if the parsing of the input succeeded, nil
	/// otherwise.    
	public func createVcardFromText(input:String) throws -> Vcard
	{
		let cPointer = linphone_core_create_vcard_from_text(cPtr, input)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null Vcard value")
		}
		let result = Vcard.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Gets an XML body. 
	/// - Parameter ektInfo: the ``EktInfo``    
	/// - Returns: The XML body       
	/// - Deprecated: 06/02/2025 use ``createXmlFromEktInfo(ektInfo:account:)``.
	@available(*, deprecated)
	public func createXmlFromEktInfo(ektInfo:EktInfo) throws -> String
	{
		let cstr = linphone_core_create_xml_from_ekt_info(cPtr, ektInfo.cPtr)
		let result = charArrayToString(charPointer: cstr)
		if (cstr != nil) {
			bctbx_free(cstr)
		}
		return result
	}
	
	
	
	/// Gets an XML body using a specific account. 
	/// - Parameter ektInfo: the ``EktInfo``    
	/// - Parameter account: the ``Account`` associated with the conference    
	/// - Returns: The XML body       
	public func createXmlFromEktInfo(ektInfo:EktInfo, account:Account?) throws -> String
	{
		let cstr = linphone_core_create_xml_from_ekt_info_2(cPtr, ektInfo.cPtr, account?.cPtr)
		let result = charArrayToString(charPointer: cstr)
		if (cstr != nil) {
			bctbx_free(cstr)
		}
		return result
	}
	
	
	
	/// Create a ``XmlRpcSession`` for a given url. 
	/// - Parameter url: The URL to the XML-RPC server. Must be NON nil.    
	/// - Returns: The new ``XmlRpcSession`` object.    
	public func createXmlRpcSession(url:String) throws -> XmlRpcSession
	{
		let cPointer = linphone_core_create_xml_rpc_session(cPtr, url)
		if (cPointer == nil) {
			throw LinphoneError.exception(result: "create null XmlRpcSession value")
		}
		let result = XmlRpcSession.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Removes a chatroom including all message history from the ``Core``. 
	/// - Parameter chatRoom: A ``ChatRoom`` object    
	public func deleteChatRoom(chatRoom:ChatRoom) 
	{
		linphone_core_delete_chat_room(cPtr, chatRoom.cPtr)
	}
	
	
	
	/// Deletes a conference information from DB. 
	/// - Parameter conferenceInfo: the ``ConferenceInfo`` to delete.    
	public func deleteConferenceInformation(conferenceInfo:ConferenceInfo) 
	{
		linphone_core_delete_conference_information(cPtr, conferenceInfo.cPtr)
	}
	
	
	
	/// Sets device_token when application
	/// didRegisterForRemoteNotificationsWithDeviceToken (IOS only). 
	/// - Parameter deviceToken: format (NSData *).    
	public func didRegisterForRemotePush(deviceToken:UnsafeMutableRawPointer?) 
	{
		linphone_core_did_register_for_remote_push(cPtr, deviceToken)
	}
	
	
	
	/// Sets device_token when application
	/// didRegisterForRemoteNotificationsWithDeviceToken (IOS only). 
	/// - Parameter deviceTokenStr: string extracted from the Data objectf received in
	/// didRegisterForRemoteNotificationsWithDeviceToken ios function. Append ":remote"
	/// after data formating..    
	public func didRegisterForRemotePushWithStringifiedToken(deviceTokenStr:String?) 
	{
		linphone_core_did_register_for_remote_push_with_stringified_token(cPtr, deviceTokenStr)
	}
	
	
	
	/// Enables or disables call ringing. 
	/// This value is taken into account from next time call parameters are created.
	/// This feature can also be enabled per-call using ``CallParams``. 
	/// - Parameter yesno: a boolean to indicate whether the feature is to be disabled. 
	public func disableCallRinging(yesno:Bool) 
	{
		linphone_core_disable_call_ringing(cPtr, yesno==true ? 1:0)
	}
	
	
	
	/// Inconditionnaly disables incoming chat messages. 
	/// - Parameter denyReason: the deny reason (``None`` has no effect). 
	public func disableChat(denyReason:Reason) 
	{
		linphone_core_disable_chat(cPtr, LinphoneReason(rawValue: CUnsignedInt(denyReason.rawValue)))
	}
	
	
	
	/// Enables reception of incoming chat messages. 
	/// By default it is enabled but it can be disabled with
	/// ``disableChat(denyReason:)``. 
	public func enableChat() 
	{
		linphone_core_enable_chat(cPtr)
	}
	
	
	
	/// Call this method when you receive a push notification (if you handle push
	/// notifications manually). 
	/// It will ensure the proxy configs are correctly registered to the proxy server,
	/// so the call or the message will be correctly delivered. 
	/// - Deprecated: 09/03/2022 See ``processPushNotification(callId:)`` instead. 
	@available(*, deprecated)
	public func ensureRegistered() 
	{
		linphone_core_ensure_registered(cPtr)
	}
	
	
	
	/// This method is called by the application to notify the linphone core library
	/// when it enters background mode. 
	public func enterBackground() 
	{
		linphone_core_enter_background(cPtr)
	}
	
	
	
	/// Joins the local participant to the running conference. 
	/// - Returns: 0 if succeeded. Negative number if failed 
	/// - Deprecated: 09/03/2021 Use ``Conference/enter()`` instead. 
	@available(*, deprecated)
	public func enterConference() throws 
	{
		let exception_result = linphone_core_enter_conference(cPtr)
		guard exception_result == 0 else {
			throw LinphoneError.exception(result: "enterConference returned value \(exception_result)")
		}
	}
	
	
	
	/// This method is called by the application to notify the linphone core library
	/// when it enters foreground mode. 
	public func enterForeground() 
	{
		linphone_core_enter_foreground(cPtr)
	}
	
	
	
	/// Returns whether a specific file format is supported. 
	/// - See also: linphone_core_get_supported_file_formats 
	/// - Parameter fmt: The format extension (wav, mkv).    
	/// - Returns: true if the file format is supported, false otherwise 
	public func fileFormatSupported(fmt:String) -> Bool
	{
		return linphone_core_file_format_supported(cPtr, fmt) != 0
	}
	
	
	
	/// Finds authentication info matching realm, username, domain criteria. 
	/// First of all, (realm,username) pair are searched. If multiple results (which
	/// should not happen because realm are supposed to be unique), then domain is
	/// added to the search. 
	/// - Parameter realm: the authentication 'realm' (optional)    
	/// - Parameter username: the SIP username to be authenticated (mandatory)    
	/// - Parameter sipDomain: the SIP domain name (optional)    
	/// - Returns: a ``AuthInfo`` if found.    
	public func findAuthInfo(realm:String?, username:String, sipDomain:String?) -> AuthInfo?
	{
		let cPointer = linphone_core_find_auth_info(cPtr, realm, username, sipDomain)
		if (cPointer == nil) {
			return nil
		}
		let result = AuthInfo.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Search from the list of current calls if a remote address match uri. 
	/// - Parameter uri: which should match call remote uri    
	/// - Returns: ``Call`` or nil if no match is found.    
	/// - Deprecated: 27/10/2020. Use ``getCallByRemoteAddress2(remoteAddress:)``
	/// instead. 
	@available(*, deprecated)
	public func findCallFromUri(uri:String) -> Call?
	{
		let cPointer = linphone_core_find_call_from_uri(cPtr, uri)
		if (cPointer == nil) {
			return nil
		}
		let result = Call.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Gets the call log matching the call id, or nil if can't be found. 
	/// - Parameter callId: Call id of the call log to find    
	/// - Parameter limit: Search limit of the most recent call logs to find    
	/// - Returns: A call log matching the call id if any.       
	public func findCallLog(callId:String, limit:Int) -> CallLog?
	{
		let cPointer = linphone_core_find_call_log(cPtr, callId, CInt(limit))
		if (cPointer == nil) {
			return nil
		}
		let result = CallLog.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Gets the call log matching the call id, or nil if can't be found. 
	/// - Parameter callId: Call id of the call log to find    
	/// - Returns: A call log matching the call id if any.       
	public func findCallLogFromCallId(callId:String) -> CallLog?
	{
		let cPointer = linphone_core_find_call_log_from_call_id(cPtr, callId)
		if (cPointer == nil) {
			return nil
		}
		let result = CallLog.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Find a chat room. 
	/// No reference is transferred to the application. The ``Core`` keeps a reference
	/// on the chat room. 
	/// - Parameter peerAddr: a linphone address.    
	/// - Parameter localAddr: a linphone address.    
	/// - Returns: ``ChatRoom`` where messaging can take place.    
	/// - Deprecated: 02/07/2020, use
	/// ``searchChatRoom(params:localAddr:remoteAddr:participants:)`` instead
	@available(*, deprecated)
	public func findChatRoom(peerAddr:Address, localAddr:Address) -> ChatRoom?
	{
		let cPointer = linphone_core_find_chat_room(cPtr, peerAddr.cPtr, localAddr.cPtr)
		if (cPointer == nil) {
			return nil
		}
		let result = ChatRoom.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Retrieve the conference information linked to the provided URI if any. 
	/// - Parameter uri: Uri of the conference as provided by the CCMP server backend. 
	///   
	/// - Returns: The ``ConferenceInfo`` found if any, nil otherwise.       
	public func findConferenceInformationFromCcmpUri(uri:String) -> ConferenceInfo?
	{
		let cPointer = linphone_core_find_conference_information_from_ccmp_uri(cPtr, uri)
		if (cPointer == nil) {
			return nil
		}
		let result = ConferenceInfo.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Retrieve the conference information linked to the provided URI if any. 
	/// - Parameter uri: ``Address`` of the uri.    
	/// - Returns: The ``ConferenceInfo`` found if any, nil otherwise.       
	public func findConferenceInformationFromUri(uri:Address) -> ConferenceInfo?
	{
		let cPointer = linphone_core_find_conference_information_from_uri(cPtr, uri.cPtr)
		if (cPointer == nil) {
			return nil
		}
		let result = ConferenceInfo.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Retrieves a list of ``Address`` sort and filter. 
	/// - Parameter filter: Chars used for the filter*    
	/// - Parameter sipOnly: Only sip address or not 
	/// - Returns: A list of filtered ``Address`` + the ``Address`` created with the
	/// filter.      
	public func findContactsByChar(filter:String, sipOnly:Bool) -> [Address]
	{
		var swiftList = [Address]()
		let cList = linphone_core_find_contacts_by_char(cPtr, filter, sipOnly==true ? 1:0)
		var listTemp = cList
		while (listTemp != nil) {
			let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
			swiftList.append(Address.getSwiftObject(cObject: data))
			listTemp = UnsafePointer<bctbx_list_t>(listTemp?.pointee.next)
		}
		return swiftList
	}
	
	
	
	/// Searches a ``Friend`` by its address. 
	/// - Parameter address: The ``Address`` to use to search the friend.    
	/// - Returns: The ``Friend`` object corresponding to the given address or nil if
	/// not found.    
	public func findFriend(address:Address) -> Friend?
	{
		let cPointer = linphone_core_find_friend(cPtr, address.cPtr)
		if (cPointer == nil) {
			return nil
		}
		let result = Friend.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Searches a ``Friend`` by its phone number. 
	/// - Parameter phoneNumber: The phone number to use to search the friend.    
	/// - Returns: The ``Friend`` object corresponding to the given phone number or nil
	/// if not found.    
	public func findFriendByPhoneNumber(phoneNumber:String) -> Friend?
	{
		let cPointer = linphone_core_find_friend_by_phone_number(cPtr, phoneNumber)
		if (cPointer == nil) {
			return nil
		}
		let result = Friend.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Searches all ``Friend`` matching an address. 
	/// - Parameter address: The address to use to search the friends.    
	/// - Returns: A list of ``Friend`` corresponding to the given address.         
	public func findFriends(address:Address) -> [Friend]
	{
		var swiftList = [Friend]()
		let cList = linphone_core_find_friends(cPtr, address.cPtr)
		var listTemp = cList
		while (listTemp != nil) {
			let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
			swiftList.append(Friend.getSwiftObject(cObject: data))
			listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
		}
			bctbx_list_free_with_data(cList, belle_sip_object_unref)
		return swiftList
	}
	
	
	
	/// Find a one-on-one chat room. 
	/// No reference is transferred to the application. The ``Core`` keeps a reference
	/// on the chat room. 
	/// - Parameter localAddr: a linphone address.    
	/// - Parameter participantAddr: a linphone address.    
	/// - Parameter encrypted: whether to look for an encrypted chat room or not 
	/// - Returns: ``ChatRoom`` where messaging can take place.    
	/// - Deprecated: 02/07/2020, use
	/// ``searchChatRoom(params:localAddr:remoteAddr:participants:)`` instead
	@available(*, deprecated)
	public func findOneToOneChatRoom(localAddr:Address, participantAddr:Address, encrypted:Bool) -> ChatRoom?
	{
		let cPointer = linphone_core_find_one_to_one_chat_room_2(cPtr, localAddr.cPtr, participantAddr.cPtr, encrypted==true ? 1:0)
		if (cPointer == nil) {
			return nil
		}
		let result = ChatRoom.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Search for a ``Account`` by it's idkey. 
	/// - Parameter idkey: An arbitrary idkey string associated to an account.    
	/// - Returns: the ``Account`` object for the given idkey value, or nil if none
	/// found    
	public func getAccountByIdkey(idkey:String?) -> Account?
	{
		let cPointer = linphone_core_get_account_by_idkey(cPtr, idkey)
		if (cPointer == nil) {
			return nil
		}
		let result = Account.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Gets the call by its SIP callid. 
	/// - Parameter callId: of call    
	/// - Returns: call ``Call``, return null if there is no call find.    
	public func getCallByCallid(callId:String) -> Call?
	{
		let cPointer = linphone_core_get_call_by_callid(cPtr, callId)
		if (cPointer == nil) {
			return nil
		}
		let result = Call.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Get the call with the remote_address specified. 
	/// - Parameter remoteAddress: The remote address of the call that we want to get  
	///  
	/// - Returns: The call if it has been found, nil otherwise.   
	/// - Deprecated: 08/07/2020 use ``getCallByRemoteAddress2(remoteAddress:)``
	/// instead 
	@available(*, deprecated)
	public func getCallByRemoteAddress(remoteAddress:String) -> Call?
	{
		let cPointer = linphone_core_get_call_by_remote_address(cPtr, remoteAddress)
		if (cPointer == nil) {
			return nil
		}
		let result = Call.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Get the call with the specified ``Address``. 
	/// - Parameter remoteAddress: the ``Address`` for which the call remote address
	/// must match    
	/// - Returns: the ``Call`` of the call if found.    
	public func getCallByRemoteAddress2(remoteAddress:Address) -> Call?
	{
		let cPointer = linphone_core_get_call_by_remote_address2(cPtr, remoteAddress.cPtr)
		if (cPointer == nil) {
			return nil
		}
		let result = Call.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Gets the list of call logs (past calls). 
	/// At the contrary of linphone_core_get_call_logs, it is your responsibility to
	/// unref the logs and free this list once you are done using it. Requires
	/// ENABLE_DB_STORAGE to work. 
	/// - Parameter peerAddress: The remote ``Address`` object.    
	/// - Parameter localAddress: The local ``Address`` object    
	/// - Returns: A list of ``CallLog``.         
	public func getCallHistory(peerAddress:Address, localAddress:Address) -> [CallLog]
	{
		var swiftList = [CallLog]()
		let cList = linphone_core_get_call_history_2(cPtr, peerAddress.cPtr, localAddress.cPtr)
		var listTemp = cList
		while (listTemp != nil) {
			let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
			swiftList.append(CallLog.getSwiftObject(cObject: data))
			listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
		}
			bctbx_list_free_with_data(cList, belle_sip_object_unref)
		return swiftList
	}
	
	
	
	/// Get a chat room whose peer is the supplied address. 
	/// If it does not exist yet, it will be created as a basic chat room. No reference
	/// is transferred to the application. The ``Core`` keeps a reference on the chat
	/// room. - Warning: This method is prone to errors, use
	/// ``searchChatRoom(params:localAddr:remoteAddr:participants:)`` instead 
	/// - Parameter addr: a linphone address.    
	/// - Returns: ``ChatRoom`` where messaging can take place.    
	/// - Deprecated: 02/07/2020, use
	/// ``searchChatRoom(params:localAddr:remoteAddr:participants:)`` instead
	@available(*, deprecated)
	public func getChatRoom(addr:Address) -> ChatRoom?
	{
		let cPointer = linphone_core_get_chat_room(cPtr, addr.cPtr)
		if (cPointer == nil) {
			return nil
		}
		let result = ChatRoom.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Get a chat room. 
	/// If it does not exist yet, it will be created as a basic chat room. No reference
	/// is transferred to the application. The ``Core`` keeps a reference on the chat
	/// room. - Warning: This method is prone to errors, use
	/// ``searchChatRoom(params:localAddr:remoteAddr:participants:)`` instead 
	/// - Parameter peerAddr: a linphone address.    
	/// - Parameter localAddr: a linphone address.    
	/// - Returns: ``ChatRoom`` where messaging can take place.    
	/// - Deprecated: 02/07/2020, use
	/// ``searchChatRoom(params:localAddr:remoteAddr:participants:)`` instead
	@available(*, deprecated)
	public func getChatRoom(peerAddr:Address, localAddr:Address) -> ChatRoom?
	{
		let cPointer = linphone_core_get_chat_room_2(cPtr, peerAddr.cPtr, localAddr.cPtr)
		if (cPointer == nil) {
			return nil
		}
		let result = ChatRoom.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Get a chat room for messaging from a sip uri like sip:joe@sip.linphone.org. 
	/// Get a chat room for messaging from a sip uri like sip:joe@sip.linphone.org. 
	/// If it does not exist yet, it will be created as a basic chat room. No reference
	/// is transferred to the application. The ``Core`` keeps a reference on the chat
	/// room. - Warning: This method is prone to errors, use
	/// ``searchChatRoom(params:localAddr:remoteAddr:participants:)`` instead 
	/// - Parameter to: The destination address for messages.    
	/// - Returns: ``ChatRoom`` where messaging can take place.    
	/// - Deprecated: 02/07/2020, use
	/// ``searchChatRoom(params:localAddr:remoteAddr:participants:)`` instead
	@available(*, deprecated)
	public func getChatRoomFromUri(to:String) -> ChatRoom?
	{
		let cPointer = linphone_core_get_chat_room_from_uri(cPtr, to)
		if (cPointer == nil) {
			return nil
		}
		let result = ChatRoom.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Retrieve the list of conference information on DB after a certain time. 
	/// - Parameter time: Time to retrieve conference info. 
	/// - Returns: The list of conference infos  .       
	public func getConferenceInformationListAfterTime(time:time_t) -> [ConferenceInfo]
	{
		var swiftList = [ConferenceInfo]()
		let cList = linphone_core_get_conference_information_list_after_time(cPtr, time)
		var listTemp = cList
		while (listTemp != nil) {
			let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
			swiftList.append(ConferenceInfo.getSwiftObject(cObject: data))
			listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
		}
			bctbx_list_free_with_data(cList, belle_sip_object_unref)
		return swiftList
	}
	
	
	
	/// Retrieve the list of conference information on DB where the address passed as
	/// argument is either the organizer or a participant. 
	/// - Parameter uri: ``Address`` of the participant.    
	/// - Returns: The list of conference infos   where the address passed as argument
	/// is either the organizer or a participant.       
	public func getConferenceInformationsWithParticipant(uri:Address) -> [ConferenceInfo]
	{
		var swiftList = [ConferenceInfo]()
		let cList = linphone_core_get_conference_informations_with_participant(cPtr, uri.cPtr)
		var listTemp = cList
		while (listTemp != nil) {
			let data = unsafeBitCast(listTemp?.pointee.data, to: OpaquePointer.self)
			swiftList.append(ConferenceInfo.getSwiftObject(cObject: data))
			listTemp = UnsafeMutablePointer<bctbx_list_t>(listTemp?.pointee.next)
		}
			bctbx_list_free_with_data(cList, belle_sip_object_unref)
		return swiftList
	}
	
	
	
	/// Searches a ``Friend`` by its reference key. 
	/// - Parameter key: The reference key to use to search the friend.    
	/// - Returns: The ``Friend`` object corresponding to the given reference key.    
	public func getFriendByRefKey(key:String) -> Friend?
	{
		let cPointer = linphone_core_get_friend_by_ref_key(cPtr, key)
		if (cPointer == nil) {
			return nil
		}
		let result = Friend.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Retrieves the list of ``Friend`` from the core that has the given display name. 
	/// - Parameter name: the name of the list    
	/// - Returns: the first ``FriendList`` object or nil.    
	public func getFriendListByName(name:String) -> FriendList?
	{
		let cPointer = linphone_core_get_friend_list_by_name(cPtr, name)
		if (cPointer == nil) {
			return nil
		}
		let result = FriendList.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Gets the chat room we have been added into using the chat_room_addr included in
	/// the push notification body This will start the core given in parameter, iterate
	/// until the new chat room is received and return it. 
	/// By default, after 25 seconds the function returns because iOS kills the app
	/// extension after 30 seconds. 
	/// - Parameter chatRoomAddr: The sip address of the chat room    
	/// - Returns: The ``ChatRoom`` object.    
	public func getNewChatRoomFromConfAddr(chatRoomAddr:String) -> ChatRoom?
	{
		let cPointer = linphone_core_get_new_chat_room_from_conf_addr(cPtr, chatRoomAddr)
		if (cPointer == nil) {
			return nil
		}
		let result = ChatRoom.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Gets the chat message with the call_id included in the push notification body
	/// This will start the core given in parameter, iterate until the message is
	/// received and return it. 
	/// By default, after 25 seconds the function returns because iOS kills the app
	/// extension after 30 seconds. - Warning: Specific to iOS platform to satisfy
	/// contraints of notification service extension. 
	/// - Parameter callId: The callId of the Message SIP transaction    
	/// - Returns: The ``ChatMessage`` object.    
	public func getNewMessageFromCallid(callId:String) -> PushNotificationMessage?
	{
		let cPointer = linphone_core_get_new_message_from_callid(cPtr, callId)
		if (cPointer == nil) {
			return nil
		}
		let result = PushNotificationMessage.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Get payload type from mime type and clock rate. 
	/// This function searches in audio and video codecs for the given payload type
	/// name and clockrate. 
	/// - Parameter type: payload mime type (I.E "speex", "pcmu", "h264")    
	/// - Parameter rate: can be LINPHONE_FIND_PAYLOAD_IGNORE_RATE (-1) 
	/// - Parameter channels: number of channels, can be
	/// LINPHONE_FIND_PAYLOAD_IGNORE_CHANNELS (-1) 
	/// - Returns: Returns nil if not found. If a ``PayloadType`` is returned, it must
	/// be released with linphone_payload_type_unref after using it.       
	public func getPayloadType(type:String, rate:Int, channels:Int) -> PayloadType?
	{
		let cPointer = linphone_core_get_payload_type(cPtr, type, CInt(rate), CInt(channels))
		if (cPointer == nil) {
			return nil
		}
		let result = PayloadType.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Searches for a ``ProxyConfig`` by it's idkey. 
	/// - Parameter idkey: An arbitrary idkey string associated to a proxy
	/// configuration 
	/// - Returns: the ``ProxyConfig`` object for the given idkey value, or nil if none
	/// found    
	/// - Deprecated: 04/09/2024 Use ``getAccountByIdkey(idkey:)`` 
	@available(*, deprecated)
	public func getProxyConfigByIdkey(idkey:String) -> ProxyConfig?
	{
		let cPointer = linphone_core_get_proxy_config_by_idkey(cPtr, idkey)
		if (cPointer == nil) {
			return nil
		}
		let result = ProxyConfig.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Get the zrtp sas validation status for a peer uri. 
	/// Once the SAS has been validated or rejected, the status will never return to
	/// Unknown (unless you delete your cache) 
	/// - Parameter addr: the peer uri   
	/// - Returns: - LinphoneZrtpPeerStatusUnknown: this uri is not present in cache OR
	/// during calls with the active device, SAS never was validated or rejected
	public func getZrtpStatus(addr:String) -> ZrtpPeerStatus
	{
		return ZrtpPeerStatus(rawValue: Int(linphone_core_get_zrtp_status(cPtr, addr).rawValue))!
	}
	
	
	
	/// Check whether the device has a hardware echo canceller. 
	/// - Returns: true if it does, false otherwise 
	public func hasBuiltinEchoCanceller() -> Bool
	{
		return linphone_core_has_builtin_echo_canceller(cPtr) != 0
	}
	
	
	
	/// Check whether the device is flagged has crappy opengl. 
	/// - Returns: true if crappy opengl flag is set, false otherwise 
	public func hasCrappyOpengl() -> Bool
	{
		return linphone_core_has_crappy_opengl(cPtr) != 0
	}
	
	
	
	/// Tells whether there is a call running. 
	/// - Returns: A boolean value telling whether a call is currently running or not 
	public func inCall() -> Bool
	{
		return linphone_core_in_call(cPtr) != 0
	}
	
	
	
	/// Constructs a ``Address`` from the given string if possible. 
	/// In case of just a username, characters will be unescaped. If a phone number is
	/// detected, it will be flattened. sip: or sips: prefix will be added if not
	/// present. Finally, @domain will be added if not present using default proxy
	/// config. - See also: ``ProxyConfig/normalizeSipUri(username:)`` for
	/// documentation. 
	/// - Parameter url: the url to parse    
	/// - Returns: the ``Address`` matching the url or nil in case of failure.      
	/// - Deprecated: on 18/07/2022, use
	/// ``interpretUrl(url:applyInternationalPrefix:)`` instead. 
	@available(*, deprecated)
	public func interpretUrl(url:String) -> Address?
	{
		let cPointer = linphone_core_interpret_url(cPtr, url)
		if (cPointer == nil) {
			return nil
		}
		let result = Address.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Constructs a ``Address`` from the given string if possible. 
	/// In case of just a username, characters will be unescaped. If a phone number is
	/// detected, it will be flattened. sip: or sips: prefix will be added if not
	/// present. Finally, @domain will be added if not present using the default
	/// ``Account``. - See also: ``Account/normalizeSipUri(username:)`` for a similar
	/// function. 
	/// - Parameter url: the url to parse    
	/// - Parameter applyInternationalPrefix: whether or not to try to format url as
	/// phone number using default account prefix if it set (and if url is a number). 
	/// - Returns: the ``Address`` matching the url or nil in case of failure.       
	public func interpretUrl(url:String, applyInternationalPrefix:Bool) -> Address?
	{
		let cPointer = linphone_core_interpret_url_2(cPtr, url, applyInternationalPrefix==true ? 1:0)
		if (cPointer == nil) {
			return nil
		}
		let result = Address.getSwiftObject(cObject: cPointer!)
		belle_sip_object_unref(UnsafeMutableRawPointer(cPointer))
		return result
	}
	
	
	
	/// Initiates an outgoing call. 
	/// The application doesn't own a reference to the returned ``Call`` object. Use
	/// linphone_call_ref to safely keep the ``Call`` pointer valid within your
	/// application.
	/// - Parameter url: The destination of the call (sip address, or phone number).    
	/// - Returns: A ``Call`` object or nil in case of failure.    
	public func invite(url:String) -> Call?
	{
		let cPointer = linphone_core_invite(cPtr, url)
		if (cPointer == nil) {
			return nil
		}
		let result = Call.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Initiates an outgoing call given a destination ``Address`` The ``Address`` can
	/// be constructed directly using ``Factory/createAddress(addr:)``, or created by
	/// ``interpretUrl(url:applyInternationalPrefix:)``. 
	/// In C, the application doesn't own a reference to the returned ``Call`` object.
	/// Use linphone_call_ref to safely keep the ``Call`` pointer valid within your
	/// application. 
	/// - Parameter addr: The destination of the call (sip address).    
	/// - Returns: A ``Call`` object or nil in case of failure.    
	public func inviteAddress(addr:Address) -> Call?
	{
		let cPointer = linphone_core_invite_address(cPtr, addr.cPtr)
		if (cPointer == nil) {
			return nil
		}
		let result = Call.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Initiates an outgoing call given a destination ``Address`` The ``Address`` can
	/// be constructed directly using ``Factory/createAddress(addr:)``, or created by
	/// ``interpretUrl(url:applyInternationalPrefix:)``. 
	/// In C, the application doesn't own a reference to the returned ``Call`` object.
	/// Use linphone_call_ref to safely keep the ``Call`` pointer valid within your
	/// application. If the ``Account`` is not specified in parameters, the caller's
	/// account will be automatically selected by finding what is the best to reach the
	/// destination of the call. 
	/// - Parameter addr: The destination of the call (sip address).    
	/// - Parameter params: Call parameters    
	/// - Returns: A ``Call`` object or nil in case of failure.    
	public func inviteAddressWithParams(addr:Address, params:CallParams) -> Call?
	{
		let cPointer = linphone_core_invite_address_with_params(cPtr, addr.cPtr, params.cPtr)
		if (cPointer == nil) {
			return nil
		}
		let result = Call.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Initiates an outgoing call given a destination ``Address`` The ``Address`` can
	/// be constructed directly using ``Factory/createAddress(addr:)``, or created by
	/// ``interpretUrl(url:applyInternationalPrefix:)``. 
	/// In C, the application doesn't own a reference to the returned ``Call`` object.
	/// Use linphone_call_ref to safely keep the ``Call`` pointer valid within your
	/// application. If the ``Account`` is not specified in parameters, the caller's
	/// account will be automatically selected by finding what is the best to reach the
	/// destination of the call. 
	/// - Parameter addr: The destination of the call (sip address).    
	/// - Parameter params: Call parameters    
	/// - Parameter subject: Subject of the call, UTF-8 encoded    
	/// - Parameter content: Body of the SIP INVITE    
	/// - Returns: A ``Call`` object or nil in case of failure.    
	public func inviteAddressWithParams(addr:Address, params:CallParams, subject:String?, content:Content?) -> Call?
	{
		let cPointer = linphone_core_invite_address_with_params_2(cPtr, addr.cPtr, params.cPtr, subject, content?.cPtr)
		if (cPointer == nil) {
			return nil
		}
		let result = Call.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Initiates an outgoing call according to supplied call parameters In C, the
	/// application doesn't own a reference to the returned ``Call`` object. 
	/// Use linphone_call_ref to safely keep the ``Call`` pointer valid within your
	/// application. 
	/// - Parameter url: The destination of the call (sip address, or phone number).    
	/// - Parameter params: the ``CallParams`` call parameters    
	/// - Returns: A ``Call`` object or nil in case of failure.    
	public func inviteWithParams(url:String, params:CallParams) -> Call?
	{
		let cPointer = linphone_core_invite_with_params(cPtr, url, params.cPtr)
		if (cPointer == nil) {
			return nil
		}
		let result = Call.getSwiftObject(cObject: cPointer!)
		return result
	}
	
	
	
	/// Tells whether a content type is supported. 
	/// - Parameter contentType: The content type to check    
	/// - Returns: A boolean value telling whether the specified content type is
	/// supported or not. 
	public func isContentTypeSupported(contentType:String) -> Bool
	{
		return linphone_core_is_content_type_supported(cPtr, contentType) != 0
	}
	
	
	
	/// Checks if given media encryption is supported. 
	/// - Parameter menc: The media encryption policy to be used. 
	/// - Returns: true if the media encryption is supported, false otherwise 
	public func isMediaEncryptionSupported(menc:MediaEncryption) -> Bool
	{
		return linphone_core_is_media_encryption_supported(cPtr, LinphoneMediaEncryption(rawValue: CUnsignedInt(menc.rawValue))) != 0
	}
	
	
	
	/// Checks if the given media filter is loaded and usable. 
	/// This is for advanced users of the library, mainly to expose mediastreamer video
	/// filter status. 
	/// - Parameter filtername: the filter name    
	/// - Returns: true if the filter is loaded and usable, false otherwise 
	public func isMediaFilterSupported(filtername:String) -> Bool
	{
		return linphone_core_is_media_filter_supported(cPtr, filtername) != 0
	}
	
	
	
	/// Tells whether a plugin is loaded or not. 
	/// - Parameter name: name of the plugin    
	/// - Returns: A boolean value telling whether the plugin has been loaded 
	public func isPluginLoaded(name:String) -> Bool
	{
		return linphone_core_is_plugin_loaded(cPtr, name) != 0
	}
	
	
	
	/// Main loop integration. 
	/// Unless auto-iterate mode is provided ( see ``enableAutoIterate(enable:)`` ), it
	/// is 