/*
LinphoneWrapper.cs
Copyright (c) 2010-2022 Belledonne Communications SARL.

This file is part of Liblinphone 
(see https://gitlab.linphone.org/BC/public/liblinphone).

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU Affero General Public License as
published by the Free Software Foundation, either version 3 of the
License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Affero General Public License for more details.

You should have received a copy of the GNU Affero General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.
*/

using System;
using System.Runtime.InteropServices;
using System.Collections.Generic;
#if IOS
using ObjCRuntime;
#endif


namespace Linphone
{
#region Wrapper specifics
	/// <summary>
	/// Only contains the LIB_NAME value that represents the library in which all DllImport are made
	/// </summary>
	public class LinphoneWrapper
	{
		public const string VERSION = "5.4.77-2-gfff9b7eb18";
#if WINDOWS_UWP
		public const UnmanagedType ByteStringMarshalling = UnmanagedType.LPStr;
#else
		public const UnmanagedType ByteStringMarshalling = UnmanagedType.LPUTF8Str;
#endif		

#if IOS
		public const string BELLE_SIP_LIB_NAME = "__Internal";
		public const string BCTOOLBOX_LIB_NAME = "__Internal";
		public const string MEDIASTREAMER_LIB_NAME = "__Internal";
		public const string LIB_NAME = "__Internal";
#elif ANDROID
		public const string LIB_NAME = "liblinphone.so";
		public const string BELLE_SIP_LIB_NAME = LIB_NAME;
		public const string BCTOOLBOX_LIB_NAME = LIB_NAME;
		public const string MEDIASTREAMER_LIB_NAME = LIB_NAME;
#else
		public const string BELLE_SIP_LIB_NAME = "belle-sip";
		public const string BCTOOLBOX_LIB_NAME = "bctoolbox";
		public const string MEDIASTREAMER_LIB_NAME = "mediastreamer2";
		public const string LIB_NAME = "liblinphone"; // With this, it automatically finds liblinphone.dll
#endif
/// https://docs.microsoft.com/fr-fr/xamarin/cross-platform/app-fundamentals/building-cross-platform-applications/platform-divergence-abstraction-divergent-implementation#android
#if ANDROID
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void setAndroidLogHandler();
#endif
#if IOS
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_iphone_enable_logs();
#endif

		/// <summary>
		/// Registers the native log handler in Linphone.
		/// </summary>
		public static void setNativeLogHandler()
		{
#if ANDROID
			setAndroidLogHandler();
#elif IOS
			linphone_iphone_enable_logs();
#endif
		}
		
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_force_utf8();
		public static void forceUtf8(){
			linphone_force_utf8();
		}
	}

	/// <summary>
	/// All methods that returns a LinphoneStatus with a value != 0 as an error code in C are translated in C# by throwing a LinphoneException
	/// </summary>
#if !(ANDROID || IOS)
	public class LinphoneException : System.Exception
	{
		public LinphoneException() : base() { }
		public LinphoneException(string message) : base(message) { }
		public LinphoneException(string message, System.Exception inner) : base(message, inner) { }
	}
#else
	[Serializable()]
	public class LinphoneException : System.Exception
	{
		public LinphoneException() : base() { }
		public LinphoneException(string message) : base(message) { }
		public LinphoneException(string message, System.Exception inner) : base(message, inner) { }
		protected LinphoneException(System.Runtime.Serialization.SerializationInfo info, System.Runtime.Serialization.StreamingContext context) { }
	}
#endif

	[StructLayout(LayoutKind.Sequential)]
	/// <summary>
	/// Parent class for a Linphone public objects
	/// </summary>
	public class LinphoneObject
	{
		internal IntPtr nativePtr;

		internal GCHandle handle;

		internal bool isListener;

		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		internal delegate void OnLinphoneObjectDataDestroyed(IntPtr data);

		[DllImport(LinphoneWrapper.BELLE_SIP_LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(IOS || ANDROID)
		internal static extern int belle_sip_object_data_set(IntPtr ptr, string name, IntPtr data, IntPtr cb);
#else
		internal static extern int belle_sip_object_data_set(IntPtr ptr, string name, IntPtr data, OnLinphoneObjectDataDestroyed cb);
#endif

		[DllImport(LinphoneWrapper.BELLE_SIP_LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		internal static extern IntPtr belle_sip_object_data_get(IntPtr ptr, string name);

		[DllImport(LinphoneWrapper.BELLE_SIP_LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		internal static extern IntPtr belle_sip_object_ref(IntPtr ptr);

		[DllImport(LinphoneWrapper.BELLE_SIP_LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		internal static extern void belle_sip_object_unref(IntPtr ptr);

		[DllImport(LinphoneWrapper.BELLE_SIP_LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		internal static extern void belle_sip_object_data_remove(IntPtr ptr, string name);

		[DllImport(LinphoneWrapper.BCTOOLBOX_LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		internal static extern IntPtr bctbx_list_next(IntPtr ptr);

		[DllImport(LinphoneWrapper.BCTOOLBOX_LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		internal static extern IntPtr bctbx_list_get_data(IntPtr ptr);

		[DllImport(LinphoneWrapper.BCTOOLBOX_LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		internal static extern void bctbx_list_free(IntPtr ptr);

		[DllImport(LinphoneWrapper.BCTOOLBOX_LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		internal static extern IntPtr bctbx_list_append(IntPtr elem, IntPtr data);
		
		[DllImport(LinphoneWrapper.BCTOOLBOX_LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		internal static extern void bctbx_free(IntPtr elem);

		internal OnLinphoneObjectDataDestroyed onLinphoneListenerObjectDestroyed;

#if IOS
		[MonoPInvokeCallback((typeof(OnLinphoneObjectDataDestroyed)))]
		internal static void on_linphone_cbs_data_destroyed(IntPtr data) {
#else
		internal void on_linphone_cbs_data_destroyed(IntPtr data) {
#endif
			if (data != IntPtr.Zero)
			{
				GCHandle handle = GCHandle.FromIntPtr(data);
				if (handle.IsAllocated) {
					handle.Free();
				}
			}
		}

		~LinphoneObject()
		{
			if (nativePtr != IntPtr.Zero && !isListener) {
				belle_sip_object_data_remove(nativePtr, "cs_obj");
				belle_sip_object_unref(nativePtr);
				handle.Free();
			}
		}

		internal static T fromNativePtr<T>(IntPtr ptr, bool takeRef=true, bool isListener=false) where T : LinphoneObject, new()
		{
			if (ptr == IntPtr.Zero) return null;
			IntPtr objPtr = belle_sip_object_data_get(ptr, "cs_obj");
			if (objPtr != IntPtr.Zero)
			{
				T obj = null;
				GCHandle handle = GCHandle.FromIntPtr(objPtr);
				if (handle.IsAllocated)
				{
					obj = (T)handle.Target;
				}
				if (obj == null)
				{
					objPtr = IntPtr.Zero;
				}
				else
				{
					return obj;
				}
			}
			if (objPtr == IntPtr.Zero)
			{
				T obj = new T();
				if (takeRef)
				{
					ptr = belle_sip_object_ref(ptr);
				}
				obj.nativePtr = ptr;
				obj.isListener = isListener;

				if (isListener) {
					obj.handle = GCHandle.Alloc(obj, GCHandleType.Normal);
				} else {
					obj.handle = GCHandle.Alloc(obj, GCHandleType.WeakTrackResurrection);
				}
				objPtr = GCHandle.ToIntPtr(obj.handle);

				if (isListener) {
#if IOS
					belle_sip_object_data_set(ptr, "cs_obj", objPtr, on_linphone_cbs_data_destroyed);
#elif ANDROID
					belle_sip_object_data_set(ptr, "cs_obj", objPtr, obj.on_linphone_cbs_data_destroyed);
#else
					obj.onLinphoneListenerObjectDestroyed = obj.on_linphone_cbs_data_destroyed;
					IntPtr cb = Marshal.GetFunctionPointerForDelegate(obj.onLinphoneListenerObjectDestroyed);
					belle_sip_object_data_set(ptr, "cs_obj", objPtr, cb);
#endif
				} else {
#if !(IOS || ANDROID)
					belle_sip_object_data_set(ptr, "cs_obj", objPtr, IntPtr.Zero);
#else
					belle_sip_object_data_set(ptr, "cs_obj", objPtr, null);
#endif
				}
				return obj;
			}
			return null;
		}
		
		/* This is a ugly hack because Marshall.PtrToStringUTF8() and its converse fonction
		are not available in any version of .NET Framework */
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		[return: MarshalAs(LinphoneWrapper.ByteStringMarshalling)]
		internal protected static extern String linphone_pointer_to_string(IntPtr ptr);
		
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		internal protected static extern IntPtr linphone_string_to_pointer([MarshalAs(LinphoneWrapper.ByteStringMarshalling)] String str);

		internal static IEnumerable<string> MarshalStringArray(IntPtr listPtr)
		{
			if (listPtr != IntPtr.Zero)
			{
				IntPtr ptr = listPtr;
				while (ptr != IntPtr.Zero)
				{
					IntPtr dataPtr = bctbx_list_get_data(ptr);
					if (dataPtr == IntPtr.Zero)
					{
						break;
					}
					string key = linphone_pointer_to_string(dataPtr);
					yield return key;
					ptr = bctbx_list_next(ptr);
				}
			}
		}

		internal static IEnumerable<T> MarshalBctbxList<T>(IntPtr listPtr, bool takeRef=true) where T : LinphoneObject, new()
		{
			if (listPtr != IntPtr.Zero)
			{
				IntPtr ptr = listPtr;
				while (ptr != IntPtr.Zero)
				{
					IntPtr dataPtr = bctbx_list_get_data(ptr);
					if (dataPtr == IntPtr.Zero)
					{
						break;
					}
					T obj = fromNativePtr<T>(dataPtr, takeRef);
					yield return obj;
					ptr = bctbx_list_next(ptr);
				}
			}
		}

		internal protected IntPtr StringArrayToBctbxList(IEnumerable<string> stringlist)
		{
			IntPtr bctbx_list = IntPtr.Zero;
			foreach (string s in stringlist)
			{
				IntPtr string_ptr = linphone_string_to_pointer(s);
				bctbx_list = bctbx_list_append(bctbx_list, string_ptr);
			}
			return bctbx_list;
		}

		internal protected void CleanStringArrayPtrs(IntPtr l)
		{
			IntPtr elem;
			for (elem = l; elem != IntPtr.Zero; elem = bctbx_list_next(elem)){
				bctbx_free(bctbx_list_get_data(elem));
			}
			bctbx_list_free(l);
		}

		internal static IntPtr ObjectArrayToBctbxList<T>(IEnumerable<T> objlist) where T : LinphoneObject, new()
		{
			IntPtr bctbx_list = IntPtr.Zero;
			foreach (T ptr in objlist)
			{
				bctbx_list = bctbx_list_append(bctbx_list, ptr.nativePtr);
			}
			return bctbx_list;
		}

	}

	public class MediastreamerFactory
	{
		public IntPtr nativePtr;

		[DllImport(LinphoneWrapper.MEDIASTREAMER_LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int ms_factory_enable_filter_from_name(IntPtr nativePtr, string name, char enabled);

		public void enableFilterFromName(string name, bool enabled)
		{
			ms_factory_enable_filter_from_name(nativePtr, name, enabled ? (char)1 : (char)0);
		}

		[DllImport(LinphoneWrapper.MEDIASTREAMER_LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void ms_devices_info_add(IntPtr devices_info, string manufacturer, string model, string platform, uint flags, int delay, int recommended_rate);

		[DllImport(LinphoneWrapper.MEDIASTREAMER_LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr ms_factory_get_devices_info(IntPtr factory);

		public void addDevicesInfo(string manufacturer, string model, string platform, uint flags, int delay, int recommended_rate)
		{
			ms_devices_info_add(ms_factory_get_devices_info(nativePtr), manufacturer, model, platform, flags, delay, recommended_rate);
		}
	}
#endregion

#region Enums

	/// <summary>
	/// Enum describing RTP AVPF activation modes. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum AVPFMode
	{
		/// <summary>
		/// Use default value defined at upper level. 
		/// <para>
		/// </para>
		/// </summary>
		Default = -1,
		/// <summary>
		/// AVPF is disabled. 
		/// <para>
		/// </para>
		/// </summary>
		Disabled = 0,
		/// <summary>
		/// AVPF is enabled. 
		/// <para>
		/// </para>
		/// </summary>
		Enabled = 1,
	}

	/// <summary>
	/// Enum describing activation code checking, used by the <see cref="Linphone.AccountCreator">
	/// Linphone.AccountCreator</see>. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum AccountCreatorActivationCodeStatus
	{
		/// <summary>
		/// Activation code ok. 
		/// <para>
		/// </para>
		/// </summary>
		Ok = 0,
		/// <summary>
		/// Activation code too short. 
		/// <para>
		/// </para>
		/// </summary>
		TooShort = 1,
		/// <summary>
		/// Activation code too long. 
		/// <para>
		/// </para>
		/// </summary>
		TooLong = 2,
		/// <summary>
		/// Contain invalid characters. 
		/// <para>
		/// </para>
		/// </summary>
		InvalidCharacters = 3,
	}

	/// <summary>
	/// Enum describing algorithm checking, used by the <see cref="Linphone.AccountCreator">
	/// Linphone.AccountCreator</see>. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum AccountCreatorAlgoStatus
	{
		/// <summary>
		/// Algorithm ok. 
		/// <para>
		/// </para>
		/// </summary>
		Ok = 0,
		/// <summary>
		/// Algorithm not supported. 
		/// <para>
		/// </para>
		/// </summary>
		NotSupported = 1,
	}

	/// <summary>
	/// Enum describing backend used in the <see cref="Linphone.AccountCreator">
	/// Linphone.AccountCreator</see>. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum AccountCreatorBackend
	{
		/// <summary>
		/// XMLRPC Backend. 
		/// <para>
		/// </para>
		/// </summary>
		XMLRPC = 0,
		/// <summary>
		/// FlexiAPI Backend. 
		/// <para>
		/// </para>
		/// </summary>
		FlexiAPI = 1,
	}

	/// <summary>
	/// Enum describing domain checking, used by the <see cref="Linphone.AccountCreator">
	/// Linphone.AccountCreator</see>. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum AccountCreatorDomainStatus
	{
		/// <summary>
		/// Domain ok. 
		/// <para>
		/// </para>
		/// </summary>
		Ok = 0,
		/// <summary>
		/// Domain invalid. 
		/// <para>
		/// </para>
		/// </summary>
		Invalid = 1,
	}

	/// <summary>
	/// Enum describing email checking, used by the <see cref="Linphone.AccountCreator">
	/// Linphone.AccountCreator</see>. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum AccountCreatorEmailStatus
	{
		/// <summary>
		/// Email ok. 
		/// <para>
		/// </para>
		/// </summary>
		Ok = 0,
		/// <summary>
		/// Email malformed. 
		/// <para>
		/// </para>
		/// </summary>
		Malformed = 1,
		/// <summary>
		/// Contain invalid characters. 
		/// <para>
		/// </para>
		/// </summary>
		InvalidCharacters = 2,
	}

	/// <summary>
	/// Enum describing language checking, used by the <see cref="Linphone.AccountCreator">
	/// Linphone.AccountCreator</see>. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum AccountCreatorLanguageStatus
	{
		/// <summary>
		/// Language ok. 
		/// <para>
		/// </para>
		/// </summary>
		Ok = 0,
	}

	/// <summary>
	/// Enum describing password checking, used by the <see cref="Linphone.AccountCreator">
	/// Linphone.AccountCreator</see>. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum AccountCreatorPasswordStatus
	{
		/// <summary>
		/// Password ok. 
		/// <para>
		/// </para>
		/// </summary>
		Ok = 0,
		/// <summary>
		/// Password too short. 
		/// <para>
		/// </para>
		/// </summary>
		TooShort = 1,
		/// <summary>
		/// Password too long. 
		/// <para>
		/// </para>
		/// </summary>
		TooLong = 2,
		/// <summary>
		/// Contain invalid characters. 
		/// <para>
		/// </para>
		/// </summary>
		InvalidCharacters = 3,
		/// <summary>
		/// Missing specific characters. 
		/// <para>
		/// </para>
		/// </summary>
		MissingCharacters = 4,
	}

	/// <summary>
	/// Enum describing phone number checking, used by the <see cref="Linphone.AccountCreator">
	/// Linphone.AccountCreator</see>. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum AccountCreatorPhoneNumberStatus
	{
		/// <summary>
		/// Phone number ok. 
		/// <para>
		/// </para>
		/// </summary>
		Ok = 1,
		/// <summary>
		/// Phone number too short. 
		/// <para>
		/// </para>
		/// </summary>
		TooShort = 2,
		/// <summary>
		/// Phone number too long. 
		/// <para>
		/// </para>
		/// </summary>
		TooLong = 4,
		/// <summary>
		/// Country code invalid. 
		/// <para>
		/// </para>
		/// </summary>
		InvalidCountryCode = 8,
		/// <summary>
		/// Phone number invalid. 
		/// <para>
		/// </para>
		/// </summary>
		Invalid = 16,
	}

	/// <summary>
	/// Enum describing the status of server request, used by the <see cref="Linphone.AccountCreator">
	/// Linphone.AccountCreator</see>. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum AccountCreatorStatus
	{
		/// <summary>
		/// Request status. 
		/// <para>
		/// Request passed 
		/// </para>
		/// </summary>
		RequestOk = 0,
		/// <summary>
		/// Request failed. 
		/// <para>
		/// </para>
		/// </summary>
		RequestFailed = 1,
		/// <summary>
		/// Request failed due to missing argument(s) 
		/// <para>
		/// </para>
		/// </summary>
		MissingArguments = 2,
		/// <summary>
		/// Request failed due to missing callback(s) 
		/// <para>
		/// </para>
		/// </summary>
		MissingCallbacks = 3,
		/// <summary>
		/// Account status. 
		/// <para>
		/// Account created 
		/// </para>
		/// </summary>
		AccountCreated = 4,
		/// <summary>
		/// Account not created. 
		/// <para>
		/// </para>
		/// </summary>
		AccountNotCreated = 5,
		/// <summary>
		/// Account exist. 
		/// <para>
		/// </para>
		/// </summary>
		AccountExist = 6,
		/// <summary>
		/// Account exist with alias. 
		/// <para>
		/// </para>
		/// </summary>
		AccountExistWithAlias = 7,
		/// <summary>
		/// Account not exist. 
		/// <para>
		/// </para>
		/// </summary>
		AccountNotExist = 8,
		/// <summary>
		/// Account was created with Alias. 
		/// <para>
		/// </para>
		/// </summary>
		AliasIsAccount = 9,
		/// <summary>
		/// Alias exist. 
		/// <para>
		/// </para>
		/// </summary>
		AliasExist = 10,
		/// <summary>
		/// Alias not exist. 
		/// <para>
		/// </para>
		/// </summary>
		AliasNotExist = 11,
		/// <summary>
		/// Account activated. 
		/// <para>
		/// </para>
		/// </summary>
		AccountActivated = 12,
		/// <summary>
		/// Account already activated. 
		/// <para>
		/// </para>
		/// </summary>
		AccountAlreadyActivated = 13,
		/// <summary>
		/// Account not activated. 
		/// <para>
		/// </para>
		/// </summary>
		AccountNotActivated = 14,
		/// <summary>
		/// Account linked. 
		/// <para>
		/// </para>
		/// </summary>
		AccountLinked = 15,
		/// <summary>
		/// Account not linked. 
		/// <para>
		/// </para>
		/// </summary>
		AccountNotLinked = 16,
		/// <summary>
		/// Server. 
		/// <para>
		/// Error server 
		/// </para>
		/// </summary>
		ServerError = 17,
		/// <summary>
		/// Error cannot send SMS. 
		/// <para>
		/// </para>
		/// </summary>
		PhoneNumberInvalid = 18,
		/// <summary>
		/// Error key doesn&apos;t match. 
		/// <para>
		/// </para>
		/// </summary>
		WrongActivationCode = 19,
		/// <summary>
		/// Error too many SMS sent. 
		/// <para>
		/// </para>
		/// </summary>
		PhoneNumberOverused = 20,
		/// <summary>
		/// Error algo isn&apos;t MD5 or SHA-256. 
		/// <para>
		/// </para>
		/// </summary>
		AlgoNotSupported = 21,
		/// <summary>
		/// Generic error. 
		/// <para>
		/// </para>
		/// </summary>
		UnexpectedError = 22,
		/// <summary>
		/// This API isn&apos;t implemented in the current backend. 
		/// <para>
		/// </para>
		/// </summary>
		NotImplementedError = 23,
		/// <summary>
		/// Request has been denied, probably due to invalid auth token. 
		/// <para>
		/// </para>
		/// </summary>
		RequestNotAuthorized = 24,
		/// <summary>
		/// Request has been denied, due to too many requests sent in given period. 
		/// <para>
		/// </para>
		/// </summary>
		RequestTooManyRequests = 25,
	}

	/// <summary>
	/// Enum describing transport checking, used by the <see cref="Linphone.AccountCreator">
	/// Linphone.AccountCreator</see>. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum AccountCreatorTransportStatus
	{
		/// <summary>
		/// Transport ok. 
		/// <para>
		/// </para>
		/// </summary>
		Ok = 0,
		/// <summary>
		/// Transport invalid. 
		/// <para>
		/// </para>
		/// </summary>
		Unsupported = 1,
	}

	/// <summary>
	/// Enum describing username checking, used by the <see cref="Linphone.AccountCreator">
	/// Linphone.AccountCreator</see>. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum AccountCreatorUsernameStatus
	{
		/// <summary>
		/// Username ok. 
		/// <para>
		/// </para>
		/// </summary>
		Ok = 0,
		/// <summary>
		/// Username too short. 
		/// <para>
		/// </para>
		/// </summary>
		TooShort = 1,
		/// <summary>
		/// Username too long. 
		/// <para>
		/// </para>
		/// </summary>
		TooLong = 2,
		/// <summary>
		/// Contain invalid characters. 
		/// <para>
		/// </para>
		/// </summary>
		InvalidCharacters = 3,
		/// <summary>
		/// Invalid username. 
		/// <para>
		/// </para>
		/// </summary>
		Invalid = 4,
	}

	/// <summary>
	/// List of all supported <see cref="Linphone.AccountManagerServicesRequest">
	/// Linphone.AccountManagerServicesRequest</see> requests, allowing to know which
	/// one triggered either a callback from the <see cref="Linphone.AccountManagerServicesRequestListener">
	/// Linphone.AccountManagerServicesRequestListener</see>. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum AccountManagerServicesRequestType
	{
		/// <summary>
		/// Asks the account manager to send us an account creation token by push
		/// notification. 
		/// <para>
		/// </para>
		/// </summary>
		SendAccountCreationTokenByPush = 0,
		/// <summary>
		/// Asks the account manager to create a request token to be validated in a web
		/// browser (captcha) and that can later be used to create an account creation
		/// token. 
		/// <para>
		/// </para>
		/// </summary>
		AccountCreationRequestToken = 1,
		/// <summary>
		/// Asks the account manager to consume the account creation request token to
		/// create an account creation token. 
		/// <para>
		/// </para>
		/// </summary>
		AccountCreationTokenFromAccountCreationRequestToken = 2,
		/// <summary>
		/// Uses an account creation token to create the account. 
		/// <para>
		/// </para>
		/// </summary>
		CreateAccountUsingToken = 3,
		/// <summary>
		/// Asks for a code to be sent by SMS to a given phone number to link that phone
		/// number with an account. 
		/// <para>
		/// </para>
		/// </summary>
		SendPhoneNumberLinkingCodeBySms = 4,
		/// <summary>
		/// Uses the code received by SMS to confirm the link between an account and a
		/// phone number. 
		/// <para>
		/// </para>
		/// </summary>
		LinkPhoneNumberUsingCode = 5,
		/// <summary>
		/// Asks for a code to be sent by email to a given email address to link that
		/// address with an account. 
		/// <para>
		/// </para>
		/// </summary>
		SendEmailLinkingCodeByEmail = 6,
		/// <summary>
		/// Uses the code received by email to confirm the link between an account and an
		/// email address. 
		/// <para>
		/// </para>
		/// </summary>
		LinkEmailUsingCode = 7,
		/// <summary>
		/// Gets the list of devices for account. 
		/// <para>
		/// </para>
		/// </summary>
		GetDevicesList = 8,
		/// <summary>
		/// Removes an account device. 
		/// <para>
		/// </para>
		/// </summary>
		DeleteDevice = 9,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		GetCreationTokenAsAdmin = 100,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		GetAccountInfoAsAdmin = 101,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		DeleteAccountAsAdmin = 102,
	}

	/// <summary>
	/// Enum describing Ip family. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum AddressFamily
	{
		/// <summary>
		/// IpV4. 
		/// <para>
		/// </para>
		/// </summary>
		Inet = 0,
		/// <summary>
		/// IpV6. 
		/// <para>
		/// </para>
		/// </summary>
		Inet6 = 1,
		/// <summary>
		/// Unknown. 
		/// <para>
		/// </para>
		/// </summary>
		Unspec = 2,
	}

	/// <summary>
	/// All kinds of alerts. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum AlertType
	{
		/// <summary>
		/// Camera is not working. 
		/// <para>
		/// No other information </para>
		/// </summary>
		/// <remarks>Note : Use the key &quot;camera_misfunction_interval&quot; in the
		/// section &quot;alerts::camera&quot; to set the interval at which the problem is
		/// checked in a <see cref="Linphone.Config">Linphone.Config</see>. </remarks>
		QoSCameraMisfunction = 0,
		/// <summary>
		/// Camera is capturing low framerate. 
		/// <para>
		/// Information supplied : float framerate; </para>
		/// </summary>
		/// <remarks>Note : Use the key &quot;low_framerate_interval&quot; in the section
		/// &quot;alerts::camera&quot; to set or get the interval at which the problem is
		/// checked in a <see cref="Linphone.Config">Linphone.Config</see>. </remarks>
		QoSCameraLowFramerate = 1,
		/// <summary>
		/// Video decoding has stopped for a given period (10 s by default). 
		/// <para>
		/// No other information. </para>
		/// </summary>
		/// <remarks>Note : Use the key &quot;video_stalled_interval&quot; in the section
		/// &quot;alerts::camera&quot; to set or get the interval at which the problem is
		/// checked in a <see cref="Linphone.Config">Linphone.Config</see>. </remarks>
		QoSVideoStalled = 2,
		/// <summary>
		/// A received media stream suffers from high loss or late rate. 
		/// <para>
		/// Information provided is:
		/// </para>
		/// </summary>
		QoSHighLossLateRate = 3,
		/// <summary>
		/// A report of high loss rate is received from remote party. 
		/// <para>
		/// Information provided: loss-rate (float). </para>
		/// </summary>
		/// <remarks>Note : Use the key &quot;remote_loss_rate_interval&quot; in the
		/// section &quot;alerts::network&quot; to set or get the interval at which the
		/// problem is checked in a <see cref="Linphone.Config">Linphone.Config</see>.
		/// </remarks>
		QoSHighRemoteLossRate = 4,
		/// <summary>
		/// Packet Burst phenomenon. 
		/// <para>
		/// </para>
		/// </summary>
		/// <remarks>Note : Use the key &quot;burst_occured_interval&quot; in the section
		/// &quot;alerts::network&quot; to set or get the interval at which the problem is
		/// checked in a <see cref="Linphone.Config">Linphone.Config</see>. </remarks>
		QoSBurstOccured = 5,
		/// <summary>
		/// Loss rate is significant but retransmissions fail to arrive on time. 
		/// <para>
		/// Information provided: nack-performance (float) the fraction of lost packets
		/// recovered thanks to nack-triggered retransmissions. </para>
		/// </summary>
		/// <remarks>Note : Use the key &quot;nack_check_interval&quot; in the section
		/// &quot;alerts::network&quot; to set or get the interval at which the problem is
		/// checked in a <see cref="Linphone.Config">Linphone.Config</see>. </remarks>
		QoSRetransmissionFailures = 6,
		/// <summary>
		/// Low bandwidth detected. 
		/// <para>
		/// Information provided: bandwidth (float) in kbit/s. </para>
		/// </summary>
		/// <remarks>Note : Use the key &quot;download_bandwidth_interval&quot; in the
		/// section &quot;alerts::video&quot; to set or get the interval at which the
		/// problem is checked in a <see cref="Linphone.Config">Linphone.Config</see>.
		/// </remarks>
		QoSLowDownloadBandwidthEstimation = 7,
		/// <summary>
		/// Low quality (bitrate) video received. 
		/// <para>
		/// Information provided: bitrate (float) in kbit/s, width (integer), int height
		/// (integer). </para>
		/// </summary>
		/// <remarks>Note : Use the key &quot;low_quality_received_interval&quot; in the
		/// section &quot;alerts::video&quot; to set or get the interval at which the
		/// problem is checked in a <see cref="Linphone.Config">Linphone.Config</see>.
		/// </remarks>
		QoSLowQualityReceivedVideo = 8,
		/// <summary>
		/// Low quality video is being sent. 
		/// <para>
		/// Information provided: bitrate (float)in kbit/s, width (integer), height
		/// (integer). </para>
		/// </summary>
		/// <remarks>Note : Use the key &quot;quality_sent_interval&quot; in the section
		/// &quot;alerts::camera&quot; to set or get the interval at which the problem is
		/// checked in a <see cref="Linphone.Config">Linphone.Config</see>. </remarks>
		QoSLowQualitySentVideo = 9,
		/// <summary>
		/// The operating system reports a low radio signal (wifi or mobile) 
		/// <para>
		/// </para>
		/// </summary>
		/// <remarks>Note : Use the key &quot;low_signal_interval&quot; in the section
		/// &quot;alerts::network&quot; to set or get the interval at which the problem is
		/// checked in a <see cref="Linphone.Config">Linphone.Config</see>. </remarks>
		QoSLowSignal = 10,
		/// <summary>
		/// The operating system reports a loss of radio signal (wifi or mobile). 
		/// <para>
		/// Information provided: rssi-value (float), signal-type (string) with values
		/// {&quot;wifi&quot;, &quot;mobile&quot;, &quot;other&quot;}. </para>
		/// </summary>
		/// <remarks>Note : Use the key &quot;lost_signal_interval&quot; in the section
		/// &quot;alerts::network&quot; to set or get the interval at which the problem is
		/// checked in a <see cref="Linphone.Config">Linphone.Config</see>. </remarks>
		QoSLostSignal = 11,
	}

	/// <summary>
	/// <see cref="Linphone.AudioDeviceCapabilities">
	/// Linphone.AudioDeviceCapabilities</see> enum represents whether a device can
	/// record audio, play audio or both 
	/// <para>
	/// </para>
	/// </summary>
	[Flags]
	public enum AudioDeviceCapabilities
	{
		/// <summary>
		/// Can record audio. 
		/// <para>
		/// </para>
		/// </summary>
		CapabilityRecord = 1<<0,
		/// <summary>
		/// Can play audio. 
		/// <para>
		/// </para>
		/// </summary>
		CapabilityPlay = 1<<1,
		/// <summary>
		/// Can play and record audio. 
		/// <para>
		/// </para>
		/// </summary>
		CapabilityAll = 3,
	}

	/// <summary>
	/// <see cref="Linphone.AudioDeviceType">Linphone.AudioDeviceType</see> enum
	/// represents the different types of an audio device. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum AudioDeviceType
	{
		/// <summary>
		/// Unknown. 
		/// <para>
		/// </para>
		/// </summary>
		Unknown = 0,
		/// <summary>
		/// Microphone. 
		/// <para>
		/// </para>
		/// </summary>
		Microphone = 1,
		/// <summary>
		/// Earpiece. 
		/// <para>
		/// </para>
		/// </summary>
		Earpiece = 2,
		/// <summary>
		/// Speaker. 
		/// <para>
		/// </para>
		/// </summary>
		Speaker = 3,
		/// <summary>
		/// Bluetooth. 
		/// <para>
		/// </para>
		/// </summary>
		Bluetooth = 4,
		/// <summary>
		/// Bluetooth A2DP. 
		/// <para>
		/// </para>
		/// </summary>
		BluetoothA2DP = 5,
		/// <summary>
		/// Telephony. 
		/// <para>
		/// </para>
		/// </summary>
		Telephony = 6,
		/// <summary>
		/// AuxLine. 
		/// <para>
		/// </para>
		/// </summary>
		AuxLine = 7,
		/// <summary>
		/// GenericUsb. 
		/// <para>
		/// </para>
		/// </summary>
		GenericUsb = 8,
		/// <summary>
		/// Headset. 
		/// <para>
		/// </para>
		/// </summary>
		Headset = 9,
		/// <summary>
		/// Headphones. 
		/// <para>
		/// </para>
		/// </summary>
		Headphones = 10,
		/// <summary>
		/// Hearing Aid. 
		/// <para>
		/// </para>
		/// </summary>
		HearingAid = 11,
	}

	/// <summary>
	/// Enum describing the authentication methods. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum AuthMethod
	{
		/// <summary>
		/// Digest authentication requested. 
		/// <para>
		/// </para>
		/// </summary>
		HttpDigest = 0,
		/// <summary>
		/// Client certificate requested. 
		/// <para>
		/// </para>
		/// </summary>
		Tls = 1,
		/// <summary>
		/// Bearer authentication. 
		/// <para>
		/// </para>
		/// </summary>
		Bearer = 2,
		/// <summary>
		/// Basic authentication. 
		/// <para>
		/// </para>
		/// </summary>
		Basic = 3,
	}

	/// <summary>
	/// <see cref="Linphone.BaudotMode">Linphone.BaudotMode</see> enum represents the
	/// Baudot mode to use for the call. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum BaudotMode
	{
		/// <summary>
		/// Send and receive audio. 
		/// <para>
		/// </para>
		/// </summary>
		Voice = 0,
		/// <summary>
		/// Send and receive Baudot tones. 
		/// <para>
		/// </para>
		/// </summary>
		Tty = 1,
		/// <summary>
		/// Send Baudot tones, but receive audio. 
		/// <para>
		/// </para>
		/// </summary>
		HearingCarryOver = 2,
		/// <summary>
		/// Send audio, but receive Baudot tones. 
		/// <para>
		/// </para>
		/// </summary>
		VoiceCarryOver = 3,
	}

	/// <summary>
	/// <see cref="Linphone.BaudotStandard">Linphone.BaudotStandard</see> enum
	/// represents the Baudot standard to use to send Baudot tones in the call. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum BaudotStandard
	{
		/// <summary>
		/// Send 45.45 baud US Baudot tones. 
		/// <para>
		/// </para>
		/// </summary>
		Us = 0,
		/// <summary>
		/// Send 50 baud European Baudot tones. 
		/// <para>
		/// </para>
		/// </summary>
		Europe = 1,
	}

	/// <summary>
	/// Enum representing the direction of a call. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum CallDir
	{
		/// <summary>
		/// outgoing calls 
		/// <para>
		/// </para>
		/// </summary>
		Outgoing = 0,
		/// <summary>
		/// incoming calls 
		/// <para>
		/// </para>
		/// </summary>
		Incoming = 1,
	}

	/// <summary>
	/// <see cref="Linphone.CallState">Linphone.CallState</see> enum represents the
	/// different states a call can reach into. 
	/// <para>
	/// The application is notified of a state change through the
	/// LinphoneCoreVTable::call_state_changed callback. 
	/// </para>
	/// </summary>
	
	public enum CallState
	{
		/// <summary>
		/// Initial state. 
		/// <para>
		/// </para>
		/// </summary>
		Idle = 0,
		/// <summary>
		/// Incoming call received. 
		/// <para>
		/// </para>
		/// </summary>
		IncomingReceived = 1,
		/// <summary>
		/// PushIncoming call received. 
		/// <para>
		/// </para>
		/// </summary>
		PushIncomingReceived = 2,
		/// <summary>
		/// Outgoing call initialized. 
		/// <para>
		/// </para>
		/// </summary>
		OutgoingInit = 3,
		/// <summary>
		/// Outgoing call in progress. 
		/// <para>
		/// </para>
		/// </summary>
		OutgoingProgress = 4,
		/// <summary>
		/// Outgoing call ringing. 
		/// <para>
		/// </para>
		/// </summary>
		OutgoingRinging = 5,
		/// <summary>
		/// Outgoing call early media. 
		/// <para>
		/// </para>
		/// </summary>
		OutgoingEarlyMedia = 6,
		/// <summary>
		/// Connected. 
		/// <para>
		/// </para>
		/// </summary>
		Connected = 7,
		/// <summary>
		/// Streams running. 
		/// <para>
		/// </para>
		/// </summary>
		StreamsRunning = 8,
		/// <summary>
		/// Pausing. 
		/// <para>
		/// </para>
		/// </summary>
		Pausing = 9,
		/// <summary>
		/// Paused. 
		/// <para>
		/// </para>
		/// </summary>
		Paused = 10,
		/// <summary>
		/// Resuming. 
		/// <para>
		/// </para>
		/// </summary>
		Resuming = 11,
		/// <summary>
		/// Referred. 
		/// <para>
		/// </para>
		/// </summary>
		Referred = 12,
		/// <summary>
		/// Error. 
		/// <para>
		/// </para>
		/// </summary>
		Error = 13,
		/// <summary>
		/// Call end. 
		/// <para>
		/// </para>
		/// </summary>
		End = 14,
		/// <summary>
		/// Paused by remote. 
		/// <para>
		/// </para>
		/// </summary>
		PausedByRemote = 15,
		/// <summary>
		/// The call&apos;s parameters are updated for example when video is asked by
		/// remote. 
		/// <para>
		/// </para>
		/// </summary>
		UpdatedByRemote = 16,
		/// <summary>
		/// We are proposing early media to an incoming call. 
		/// <para>
		/// </para>
		/// </summary>
		IncomingEarlyMedia = 17,
		/// <summary>
		/// We have initiated a call update. 
		/// <para>
		/// </para>
		/// </summary>
		Updating = 18,
		/// <summary>
		/// The call object is now released. 
		/// <para>
		/// </para>
		/// </summary>
		Released = 19,
		/// <summary>
		/// The call is updated by remote while not yet answered (SIP UPDATE in early
		/// dialog received) 
		/// <para>
		/// </para>
		/// </summary>
		EarlyUpdatedByRemote = 20,
		/// <summary>
		/// We are updating the call while not yet answered (SIP UPDATE in early dialog
		/// sent) 
		/// <para>
		/// </para>
		/// </summary>
		EarlyUpdating = 21,
	}

	/// <summary>
	/// Enum representing the status of a call. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum CallStatus
	{
		/// <summary>
		/// The call was sucessful. 
		/// <para>
		/// </para>
		/// </summary>
		Success = 0,
		/// <summary>
		/// The call was aborted (caller hanged up) 
		/// <para>
		/// </para>
		/// </summary>
		Aborted = 1,
		/// <summary>
		/// The call was missed (incoming call timed out without being answered or hanged
		/// up) 
		/// <para>
		/// </para>
		/// </summary>
		Missed = 2,
		/// <summary>
		/// The call was declined, either locally or by remote end. 
		/// <para>
		/// </para>
		/// </summary>
		Declined = 3,
		/// <summary>
		/// The call was aborted before being advertised to the application - for protocol
		/// reasons. 
		/// <para>
		/// </para>
		/// </summary>
		EarlyAborted = 4,
		/// <summary>
		/// The call was answered on another device. 
		/// <para>
		/// </para>
		/// </summary>
		AcceptedElsewhere = 5,
		/// <summary>
		/// The call was declined on another device. 
		/// <para>
		/// </para>
		/// </summary>
		DeclinedElsewhere = 6,
	}

	/// <summary>
	/// <see cref="Linphone.ChatMessageDirection">Linphone.ChatMessageDirection</see>
	/// is used to indicate if a message is outgoing or incoming. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum ChatMessageDirection
	{
		/// <summary>
		/// Incoming message. 
		/// <para>
		/// </para>
		/// </summary>
		Incoming = 0,
		/// <summary>
		/// Outgoing message. 
		/// <para>
		/// </para>
		/// </summary>
		Outgoing = 1,
	}

	/// <summary>
	/// <see cref="Linphone.ChatMessageState">Linphone.ChatMessageState</see> is used
	/// to notify if messages have been successfully delivered or not. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum ChatMessageState
	{
		/// <summary>
		/// Initial state. 
		/// <para>
		/// </para>
		/// </summary>
		Idle = 0,
		/// <summary>
		/// Delivery in progress. 
		/// <para>
		/// </para>
		/// </summary>
		InProgress = 1,
		/// <summary>
		/// Message successfully delivered and acknowledged by the server. 
		/// <para>
		/// </para>
		/// </summary>
		Delivered = 2,
		/// <summary>
		/// Message was not delivered. 
		/// <para>
		/// </para>
		/// </summary>
		NotDelivered = 3,
		/// <summary>
		/// Message was received and acknowledged but cannot get file from server. 
		/// <para>
		/// </para>
		/// </summary>
		FileTransferError = 4,
		/// <summary>
		/// File transfer has been completed successfully. 
		/// <para>
		/// </para>
		/// </summary>
		FileTransferDone = 5,
		/// <summary>
		/// Message successfully delivered an acknowledged by the remote user. 
		/// <para>
		/// </para>
		/// </summary>
		DeliveredToUser = 6,
		/// <summary>
		/// Message successfully displayed to the remote user. 
		/// <para>
		/// </para>
		/// </summary>
		Displayed = 7,
		/// <summary>
		/// File transfer is in progress. 
		/// <para>
		/// If message is incoming it&apos;s a download, otherwise it&apos;s an upload. 
		/// </para>
		/// </summary>
		FileTransferInProgress = 8,
		/// <summary>
		/// Message is pending delivery. 
		/// <para>
		/// </para>
		/// </summary>
		PendingDelivery = 9,
		/// <summary>
		/// The user cancelled the file transfer. 
		/// <para>
		/// </para>
		/// </summary>
		FileTransferCancelling = 10,
	}

	/// <summary>
	/// <see cref="Linphone.ChatRoomBackend">Linphone.ChatRoomBackend</see> is used to
	/// indicate the backend implementation of a chat room. 
	/// <para>
	/// </para>
	/// </summary>
	[Flags]
	public enum ChatRoomBackend
	{
		/// <summary>
		/// Basic (client-to-client) chat room. 
		/// <para>
		/// </para>
		/// </summary>
		Basic = 1<<0,
		/// <summary>
		/// Server-based chat room. 
		/// <para>
		/// </para>
		/// </summary>
		FlexisipChat = 1<<1,
	}

	/// <summary>
	/// <see cref="Linphone.ChatRoomCapabilities">Linphone.ChatRoomCapabilities</see>
	/// is used to indicate the capabilities of a chat room. 
	/// <para>
	/// </para>
	/// </summary>
	[Flags]
	public enum ChatRoomCapabilities
	{
		/// <summary>
		/// No capabilities. 
		/// <para>
		/// </para>
		/// </summary>
		None = 0,
		/// <summary>
		/// No server. 
		/// <para>
		/// It&apos;s a direct communication 
		/// </para>
		/// </summary>
		Basic = 1<<0,
		/// <summary>
		/// Supports RTT. 
		/// <para>
		/// </para>
		/// </summary>
		RealTimeText = 1<<1,
		/// <summary>
		/// Use server (supports group chat) 
		/// <para>
		/// </para>
		/// </summary>
		Conference = 1<<2,
		/// <summary>
		/// Special proxy chat room flag. 
		/// <para>
		/// </para>
		/// </summary>
		Proxy = 1<<3,
		/// <summary>
		/// Chat room migratable from Basic to Conference. 
		/// <para>
		/// </para>
		/// </summary>
		Migratable = 1<<4,
		/// <summary>
		/// A communication between two participants (can be Basic or Conference) 
		/// <para>
		/// </para>
		/// </summary>
		OneToOne = 1<<5,
		/// <summary>
		/// Chat room is encrypted. 
		/// <para>
		/// </para>
		/// </summary>
		Encrypted = 1<<6,
		/// <summary>
		/// Chat room can enable ephemeral messages. 
		/// <para>
		/// </para>
		/// </summary>
		Ephemeral = 1<<7,
	}

	/// <summary>
	/// <see cref="Linphone.ChatRoomEncryptionBackend">
	/// Linphone.ChatRoomEncryptionBackend</see> is used to indicate the encryption
	/// engine used by a chat room. 
	/// <para>
	/// </para>
	/// </summary>
	[Flags]
	public enum ChatRoomEncryptionBackend
	{
		/// <summary>
		/// No encryption. 
		/// <para>
		/// </para>
		/// </summary>
		None = 0,
		/// <summary>
		/// Lime x3dh encryption. 
		/// <para>
		/// </para>
		/// </summary>
		Lime = 1<<0,
	}

	/// <summary>
	/// <see cref="Linphone.ChatRoomEphemeralMode">Linphone.ChatRoomEphemeralMode</see>
	/// is used to the ephemeral message mode used by a chat room. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum ChatRoomEphemeralMode
	{
		/// <summary>
		/// Each device manages its own ephemeral settings. 
		/// <para>
		/// </para>
		/// </summary>
		DeviceManaged = 0,
		/// <summary>
		/// Ephemeral settings are chatroom wide and only admins can change them. 
		/// <para>
		/// </para>
		/// </summary>
		AdminManaged = 1,
	}

	/// <summary>
	/// <see cref="Linphone.ChatRoomHistoryFilter">Linphone.ChatRoomHistoryFilter</see>
	/// is used to determine which filter to apply to history functions. 
	/// <para>
	/// </para>
	/// </summary>
	[Flags]
	public enum ChatRoomHistoryFilter
	{
		/// <summary>
		/// No filter. 
		/// <para>
		/// </para>
		/// </summary>
		None = 0,
		/// <summary>
		/// Retrieve calls. 
		/// <para>
		/// </para>
		/// </summary>
		Call = 1<<0,
		/// <summary>
		/// Retrieve chat messages. 
		/// <para>
		/// </para>
		/// </summary>
		ChatMessage = 1<<1,
		/// <summary>
		/// Retrieve chat messages and security events. 
		/// <para>
		/// </para>
		/// </summary>
		ChatMessageSecurity = 1<<2,
		/// <summary>
		/// Retrieve all chat room events. 
		/// <para>
		/// </para>
		/// </summary>
		Info = 1<<3,
		/// <summary>
		/// Retrieve all chat room events without device events. 
		/// <para>
		/// </para>
		/// </summary>
		InfoNoDevice = 1<<4,
	}

	/// <summary>
	/// TODO move to encryption engine object when available <see cref="Linphone.ChatRoomSecurityLevel">
	/// Linphone.ChatRoomSecurityLevel</see> is used to indicate the encryption
	/// security level of a chat room. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum ChatRoomSecurityLevel
	{
		/// <summary>
		/// Security failure. 
		/// <para>
		/// </para>
		/// </summary>
		Unsafe = 0,
		/// <summary>
		/// No encryption. 
		/// <para>
		/// </para>
		/// </summary>
		ClearText = 1,
		/// <summary>
		/// Encrypted. 
		/// <para>
		/// </para>
		/// </summary>
		Encrypted = 2,
		/// <summary>
		/// Encrypted and verified. 
		/// <para>
		/// </para>
		/// </summary>
		Safe = 3,
	}

	/// <summary>
	/// <see cref="Linphone.ChatRoomState">Linphone.ChatRoomState</see> is used to
	/// indicate the current state of a chat room. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum ChatRoomState
	{
		/// <summary>
		/// Initial state. 
		/// <para>
		/// </para>
		/// </summary>
		None = 0,
		/// <summary>
		/// Chat room is now instantiated on local. 
		/// <para>
		/// </para>
		/// </summary>
		Instantiated = 1,
		/// <summary>
		/// One creation request was sent to the server. 
		/// <para>
		/// </para>
		/// </summary>
		CreationPending = 2,
		/// <summary>
		/// Chat room was created on the server. 
		/// <para>
		/// </para>
		/// </summary>
		Created = 3,
		/// <summary>
		/// Chat room creation failed. 
		/// <para>
		/// </para>
		/// </summary>
		CreationFailed = 4,
		/// <summary>
		/// Wait for chat room termination. 
		/// <para>
		/// </para>
		/// </summary>
		TerminationPending = 5,
		/// <summary>
		/// Chat room exists on server but not in local. 
		/// <para>
		/// </para>
		/// </summary>
		Terminated = 6,
		/// <summary>
		/// The chat room termination failed. 
		/// <para>
		/// </para>
		/// </summary>
		TerminationFailed = 7,
		/// <summary>
		/// Chat room was deleted on the server. 
		/// <para>
		/// </para>
		/// </summary>
		Deleted = 8,
	}

	/// <summary>
	/// Codec priority policies. 
	/// <para>
	/// This enum represents different policies for managing offered codec lists during
	/// calls, as well as the offer-answer logic. Currently, policies can be applied
	/// only for video codecs. 
	/// </para>
	/// </summary>
	
	public enum CodecPriorityPolicy
	{
		/// <summary>
		/// In this mode, codecs have initial default ordering, that can be changed by the
		/// application The answerer of a call accepts codecs with the order given in the
		/// offer. 
		/// <para>
		/// </para>
		/// </summary>
		Basic = 0,
		/// <summary>
		/// In this mode, the codec list is managed by the <see cref="Linphone.Core">
		/// Linphone.Core</see> according to hardware capabilities in the goal of
		/// optimizing video quality and user experience. 
		/// <para>
		/// The answerer of call may re-order the offerer&apos;s list in its answer in
		/// order to give preference to certain codecs. 
		/// </para>
		/// </summary>
		Auto = 1,
	}

	/// <summary>
	/// <see cref="Linphone.ConferenceInfoState">Linphone.ConferenceInfoState</see> is
	/// used to list all states of a conference info object 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum ConferenceInfoState
	{
		/// <summary>
		/// New conference. 
		/// <para>
		/// </para>
		/// </summary>
		New = 0,
		/// <summary>
		/// Conference has been updated. 
		/// <para>
		/// </para>
		/// </summary>
		Updated = 1,
		/// <summary>
		/// Canceling a conference. 
		/// <para>
		/// </para>
		/// </summary>
		Cancelled = 2,
	}

	/// <summary>
	/// Mode of joining conference. 
	/// <para>
	/// </para>
	/// </summary>
	/// <remarks>Warning : It is only valid at the creation of the conferece </remarks> 
	
	public enum ConferenceJoiningMode
	{
		/// <summary>
		/// Participants must dial the conference server. 
		/// <para>
		/// </para>
		/// </summary>
		DialIn = 0,
		/// <summary>
		/// Conference server dials participants. 
		/// <para>
		/// </para>
		/// </summary>
		DialOut = 1,
	}

	/// <summary>
	/// <see cref="Linphone.ConferenceLayout">Linphone.ConferenceLayout</see> is used
	/// to indicate the layout used by the conference. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum ConferenceLayout
	{
		/// <summary>
		/// Grid - each participant is given an equal sized image size. 
		/// <para>
		/// </para>
		/// </summary>
		Grid = 0,
		/// <summary>
		/// Active speaker - participant who speaks is prominently displayed in the center
		/// of the screen and other participants are minimized. 
		/// <para>
		/// </para>
		/// </summary>
		ActiveSpeaker = 1,
	}

	/// <summary>
	/// Type of conference participant list. 
	/// <para>
	/// When participant list is closed, no more participants can be added other than
	/// those declared when creating the conference When participant list is open,
	/// other people can join the conference upon invitation of a participant no more
	/// participants can be added other than those declared when creating the
	/// conference 
	/// </para>
	/// </summary>
	
	public enum ConferenceParticipantListType
	{
		/// <summary>
		/// Only participants in the initiating INVITE are allowed to join the conference. 
		/// <para>
		/// </para>
		/// </summary>
		Closed = 0,
		/// <summary>
		/// All devices calling the conference URI are allowed to join the conference. 
		/// <para>
		/// </para>
		/// </summary>
		Open = 1,
	}

	/// <summary>
	/// Describes conference scheduler possible states. 
	/// <para>
	/// It is notified via the conference_scheduler_state_changed callback in <see cref="Linphone.ConferenceSchedulerListener">
	/// Linphone.ConferenceSchedulerListener</see>. 
	/// </para>
	/// </summary>
	
	public enum ConferenceSchedulerState
	{
		/// <summary>
		/// Default state of a freshly created <see cref="Linphone.ConferenceScheduler">
		/// Linphone.ConferenceScheduler</see>. 
		/// <para>
		/// </para>
		/// </summary>
		Idle = 0,
		/// <summary>
		/// An error has happened during conference creation. 
		/// <para>
		/// </para>
		/// </summary>
		Error = 1,
		/// <summary>
		/// Conference creation is in progress. 
		/// <para>
		/// </para>
		/// </summary>
		AllocationPending = 2,
		/// <summary>
		/// Confererence has been created. 
		/// <para>
		/// </para>
		/// </summary>
		Ready = 3,
		/// <summary>
		/// Conference has been updated. 
		/// <para>
		/// </para>
		/// </summary>
		Updating = 4,
	}

	/// <summary>
	/// <see cref="Linphone.ConferenceSchedulerType">
	/// Linphone.ConferenceSchedulerType</see> is used to know which conference
	/// scheduler type the core is using 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum ConferenceSchedulerType
	{
		/// <summary>
		/// SIP conference scheduler type. 
		/// <para>
		/// </para>
		/// </summary>
		SIP = 0,
		/// <summary>
		/// Database conference scheduler type. 
		/// <para>
		/// </para>
		/// </summary>
		DB = 1,
		/// <summary>
		/// CCMP conference scheduler type. 
		/// <para>
		/// </para>
		/// </summary>
		CCMP = 2,
	}

	/// <summary>
	/// Conference minimum security level. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum ConferenceSecurityLevel
	{
		/// <summary>
		/// No security. 
		/// <para>
		/// </para>
		/// </summary>
		None = 0,
		/// <summary>
		/// Point-to-point encryption. 
		/// <para>
		/// </para>
		/// </summary>
		PointToPoint = 1,
		/// <summary>
		/// End-to-end encryption. 
		/// <para>
		/// </para>
		/// </summary>
		EndToEnd = 2,
	}

	/// <summary>
	/// <see cref="Linphone.ConferenceState">Linphone.ConferenceState</see> is used to
	/// indicate the current state of a conference. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum ConferenceState
	{
		/// <summary>
		/// Initial state. 
		/// <para>
		/// </para>
		/// </summary>
		None = 0,
		/// <summary>
		/// Conference is now instantiated locally. 
		/// <para>
		/// </para>
		/// </summary>
		Instantiated = 1,
		/// <summary>
		/// One creation request was sent to the service. 
		/// <para>
		/// </para>
		/// </summary>
		CreationPending = 2,
		/// <summary>
		/// Conference was created on the service. 
		/// <para>
		/// </para>
		/// </summary>
		Created = 3,
		/// <summary>
		/// Conference creation on service failed. 
		/// <para>
		/// </para>
		/// </summary>
		CreationFailed = 4,
		/// <summary>
		/// Wait for conference termination. 
		/// <para>
		/// </para>
		/// </summary>
		TerminationPending = 5,
		/// <summary>
		/// The conference is terminated locally, though it may still exist on the service
		/// for other participants. 
		/// <para>
		/// </para>
		/// </summary>
		Terminated = 6,
		/// <summary>
		/// Conference termination failed. 
		/// <para>
		/// </para>
		/// </summary>
		TerminationFailed = 7,
		/// <summary>
		/// Conference was deleted locally and on the service. 
		/// <para>
		/// </para>
		/// </summary>
		Deleted = 8,
	}

	/// <summary>
	/// Describes the state of the remote configuring process of the <see cref="Linphone.Core">
	/// Linphone.Core</see> object, &apos;Skipped&apos; when the feature is disabled. 
	/// <para>
	/// It is notified via the configuring_status() callback in <see cref="Linphone.CoreListener">
	/// Linphone.CoreListener</see>. 
	/// </para>
	/// </summary>
	
	public enum ConfiguringState
	{
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Successful = 0,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Failed = 1,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Skipped = 2,
	}

	/// <summary>
	/// Consolidated presence information: &apos;online&apos; means the user is open
	/// for communication, &apos;busy&apos; means the user is open for communication
	/// but involved in an other activity, &apos;do not disturb&apos; means the user is
	/// not open for communication, and &apos;offline&apos; means that no presence
	/// information is available. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum ConsolidatedPresence
	{
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Online = 0,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Busy = 1,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		DoNotDisturb = 2,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Offline = 3,
	}

	/// <summary>
	/// Used to notify if log collection upload have been successfully delivered or
	/// not. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum CoreLogCollectionUploadState
	{
		/// <summary>
		/// Delivery in progress. 
		/// <para>
		/// </para>
		/// </summary>
		InProgress = 0,
		/// <summary>
		/// Log collection upload successfully delivered and acknowledged by remote end
		/// point. 
		/// <para>
		/// </para>
		/// </summary>
		Delivered = 1,
		/// <summary>
		/// Log collection upload was not delivered. 
		/// <para>
		/// </para>
		/// </summary>
		NotDelivered = 2,
	}

	/// <summary>
	/// Enum describing the result of the echo canceller calibration process. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum EcCalibratorStatus
	{
		/// <summary>
		/// The echo canceller calibration process is on going. 
		/// <para>
		/// </para>
		/// </summary>
		InProgress = 0,
		/// <summary>
		/// The echo canceller calibration has been performed and produced an echo delay
		/// measure. 
		/// <para>
		/// </para>
		/// </summary>
		Done = 1,
		/// <summary>
		/// The echo canceller calibration process has failed. 
		/// <para>
		/// </para>
		/// </summary>
		Failed = 2,
		/// <summary>
		/// The echo canceller calibration has been performed and no echo has been
		/// detected. 
		/// <para>
		/// </para>
		/// </summary>
		DoneNoEcho = 3,
	}

	/// <summary>
	/// <see cref="Linphone.EventLogType">Linphone.EventLogType</see> is used to
	/// indicate the type of an event. 
	/// <para>
	/// Useful for cast. 
	/// </para>
	/// </summary>
	
	public enum EventLogType
	{
		/// <summary>
		/// No defined event. 
		/// <para>
		/// </para>
		/// </summary>
		None = 0,
		/// <summary>
		/// Conference (created) event. 
		/// <para>
		/// </para>
		/// </summary>
		ConferenceCreated = 1,
		/// <summary>
		/// Conference (terminated) event. 
		/// <para>
		/// </para>
		/// </summary>
		ConferenceTerminated = 2,
		/// <summary>
		/// Conference call (start) event. 
		/// <para>
		/// </para>
		/// </summary>
		ConferenceCallStarted = 3,
		/// <summary>
		/// Conference call (connected) event. 
		/// <para>
		/// </para>
		/// </summary>
		ConferenceConnected = 21,
		/// <summary>
		/// Conference call (end) event. 
		/// <para>
		/// </para>
		/// </summary>
		ConferenceCallEnded = 4,
		/// <summary>
		/// Conference chat message event. 
		/// <para>
		/// </para>
		/// </summary>
		ConferenceChatMessage = 5,
		/// <summary>
		/// Conference allowed participant changed event. 
		/// <para>
		/// </para>
		/// </summary>
		ConferenceAllowedParticipantListChanged = 28,
		/// <summary>
		/// Conference participant (added) event. 
		/// <para>
		/// </para>
		/// </summary>
		ConferenceParticipantAdded = 6,
		/// <summary>
		/// Conference participant (removed) event. 
		/// <para>
		/// </para>
		/// </summary>
		ConferenceParticipantRemoved = 7,
		/// <summary>
		/// Conference participant (role unknown) event. 
		/// <para>
		/// </para>
		/// </summary>
		ConferenceParticipantRoleUnknown = 25,
		/// <summary>
		/// Conference participant (role speaker) event. 
		/// <para>
		/// </para>
		/// </summary>
		ConferenceParticipantRoleSpeaker = 26,
		/// <summary>
		/// Conference participant (role listener) event. 
		/// <para>
		/// </para>
		/// </summary>
		ConferenceParticipantRoleListener = 27,
		/// <summary>
		/// Conference participant (set admin) event. 
		/// <para>
		/// </para>
		/// </summary>
		ConferenceParticipantSetAdmin = 8,
		/// <summary>
		/// Conference participant (unset admin) event. 
		/// <para>
		/// </para>
		/// </summary>
		ConferenceParticipantUnsetAdmin = 9,
		/// <summary>
		/// Conference participant device (added) event. 
		/// <para>
		/// </para>
		/// </summary>
		ConferenceParticipantDeviceAdded = 10,
		/// <summary>
		/// Conference participant device (removed) event. 
		/// <para>
		/// </para>
		/// </summary>
		ConferenceParticipantDeviceRemoved = 11,
		/// <summary>
		/// Conference participant device (joining request) event. 
		/// <para>
		/// </para>
		/// </summary>
		ConferenceParticipantDeviceJoiningRequest = 29,
		/// <summary>
		/// Conference participant device (media capability changed) event. 
		/// <para>
		/// </para>
		/// </summary>
		ConferenceParticipantDeviceMediaCapabilityChanged = 17,
		/// <summary>
		/// Conference participant device (media availability changed) event. 
		/// <para>
		/// </para>
		/// </summary>
		ConferenceParticipantDeviceMediaAvailabilityChanged = 23,
		/// <summary>
		/// Conference participant device (left) event. 
		/// <para>
		/// </para>
		/// </summary>
		ConferenceParticipantDeviceStatusChanged = 22,
		/// <summary>
		/// Conference subject event. 
		/// <para>
		/// </para>
		/// </summary>
		ConferenceSubjectChanged = 12,
		/// <summary>
		/// Conference available media event. 
		/// <para>
		/// </para>
		/// </summary>
		ConferenceAvailableMediaChanged = 18,
		/// <summary>
		/// Conference encryption security event. 
		/// <para>
		/// </para>
		/// </summary>
		ConferenceSecurityEvent = 13,
		/// <summary>
		/// Conference ephemeral message (ephemeral message lifetime changed) event. 
		/// <para>
		/// </para>
		/// </summary>
		ConferenceEphemeralMessageLifetimeChanged = 14,
		/// <summary>
		/// Conference ephemeral message (ephemeral message enabled) event. 
		/// <para>
		/// </para>
		/// </summary>
		ConferenceEphemeralMessageEnabled = 15,
		/// <summary>
		/// Conference ephemeral message (ephemeral message disabled) event. 
		/// <para>
		/// </para>
		/// </summary>
		ConferenceEphemeralMessageDisabled = 16,
		/// <summary>
		/// Conference ephemeral message (ephemeral message settings managed by admin)
		/// event. 
		/// <para>
		/// </para>
		/// </summary>
		ConferenceEphemeralMessageManagedByAdmin = 19,
		/// <summary>
		/// Conference ephemeral message (ephemeral message settings managed by
		/// participants) event. 
		/// <para>
		/// </para>
		/// </summary>
		ConferenceEphemeralMessageManagedByParticipants = 20,
		/// <summary>
		/// Reaction event to a chat message. 
		/// <para>
		/// </para>
		/// </summary>
		ConferenceChatMessageReaction = 24,
	}

	/// <summary>
	/// Enum describing the capabilities of a <see cref="Linphone.Friend">
	/// Linphone.Friend</see>, populated through presence subscribe/notify process. 
	/// <para>
	/// </para>
	/// </summary>
	[Flags]
	public enum FriendCapability
	{
		/// <summary>
		/// No capabilities populated. 
		/// <para>
		/// </para>
		/// </summary>
		None = 0,
		/// <summary>
		/// This friend can be invited in a Flexisip backend <see cref="Linphone.ChatRoom">
		/// Linphone.ChatRoom</see>. 
		/// <para>
		/// </para>
		/// </summary>
		GroupChat = 1<<0,
		/// <summary>
		/// This friend can be invited in a Flexisip backend end-to-end encrypted <see cref="Linphone.ChatRoom">
		/// Linphone.ChatRoom</see>. 
		/// <para>
		/// </para>
		/// </summary>
		LimeX3Dh = 1<<1,
		/// <summary>
		/// This friend is able to delete ephemeral messages once they have expired. 
		/// <para>
		/// </para>
		/// </summary>
		EphemeralMessages = 1<<2,
	}

	/// <summary>
	/// Enum describing the status of a LinphoneFriendList operation. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum FriendListStatus
	{
		/// <summary>
		/// Operation went fine. 
		/// <para>
		/// </para>
		/// </summary>
		OK = 0,
		/// <summary>
		/// <see cref="Linphone.Friend">Linphone.Friend</see> wasn&apos;t found in the <see cref="Linphone.FriendList">
		/// Linphone.FriendList</see> 
		/// <para>
		/// </para>
		/// </summary>
		NonExistentFriend = 1,
		/// <summary>
		/// <see cref="Linphone.Friend">Linphone.Friend</see> is already present in a <see cref="Linphone.FriendList">
		/// Linphone.FriendList</see> 
		/// <para>
		/// </para>
		/// </summary>
		InvalidFriend = 2,
	}

	/// <summary>
	/// Enum describing the status of a CardDAV synchronization. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum FriendListSyncStatus
	{
		/// <summary>
		/// Synchronization started. 
		/// <para>
		/// </para>
		/// </summary>
		Started = 0,
		/// <summary>
		/// Synchronization finished successfuly. 
		/// <para>
		/// </para>
		/// </summary>
		Successful = 1,
		/// <summary>
		/// Synchronization failed. 
		/// <para>
		/// </para>
		/// </summary>
		Failure = 2,
	}

	/// <summary>
	/// The types of FriendList. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum FriendListType
	{
		/// <summary>
		/// Default value, used when no other type applies. 
		/// <para>
		/// </para>
		/// </summary>
		Default = -1,
		/// <summary>
		/// Used when list is synchronized with a remote CardDAV server. 
		/// <para>
		/// </para>
		/// </summary>
		CardDAV = 0,
		/// <summary>
		/// Used for simple vCards list remotely provisionned by a server. 
		/// <para>
		/// </para>
		/// </summary>
		VCard4 = 1,
		/// <summary>
		/// Friend list used by app for cache purposes, friends added in this list will be
		/// ignored by <see cref="Linphone.MagicSearch">Linphone.MagicSearch</see>  
		/// <para>
		/// </para>
		/// </summary>
		ApplicationCache = 2,
	}

	/// <summary>
	/// Describes the global state of the <see cref="Linphone.Core">Linphone.Core</see>
	/// object. 
	/// <para>
	/// It is notified via the global_state_changed() callback in <see cref="Linphone.CoreListener">
	/// Linphone.CoreListener</see>. 
	/// </para>
	/// </summary>
	
	public enum GlobalState
	{
		/// <summary>
		/// State in which we&apos;re in after <see cref="Linphone.Core.Stop()">
		/// Linphone.Core.Stop()</see>. 
		/// <para>
		/// Do not call any method while in this state except for <see cref="Linphone.Core.Start()">
		/// Linphone.Core.Start()</see> 
		/// </para>
		/// </summary>
		Off = 0,
		/// <summary>
		/// Transient state for when we call <see cref="Linphone.Core.Start()">
		/// Linphone.Core.Start()</see> 
		/// <para>
		/// </para>
		/// </summary>
		Startup = 1,
		/// <summary>
		/// Indicates <see cref="Linphone.Core">Linphone.Core</see> has been started and is
		/// up and running. 
		/// <para>
		/// </para>
		/// </summary>
		On = 2,
		/// <summary>
		/// Transient state for when we call <see cref="Linphone.Core.Stop()">
		/// Linphone.Core.Stop()</see> 
		/// <para>
		/// </para>
		/// </summary>
		Shutdown = 3,
		/// <summary>
		/// Transient state between Startup and On if there is a remote provisionning URI
		/// configured. 
		/// <para>
		/// </para>
		/// </summary>
		Configuring = 4,
		/// <summary>
		/// <see cref="Linphone.Core">Linphone.Core</see> state after being created by
		/// linphone_factory_create_core, generally followed by a call to <see cref="Linphone.Core.Start()">
		/// Linphone.Core.Start()</see> 
		/// <para>
		/// </para>
		/// </summary>
		Ready = 5,
	}

	/// <summary>
	/// Enum describing ICE states. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum IceState
	{
		/// <summary>
		/// ICE has not been activated for this call or stream. 
		/// <para>
		/// </para>
		/// </summary>
		NotActivated = 0,
		/// <summary>
		/// ICE processing has failed. 
		/// <para>
		/// </para>
		/// </summary>
		Failed = 1,
		/// <summary>
		/// ICE process is in progress. 
		/// <para>
		/// </para>
		/// </summary>
		InProgress = 2,
		/// <summary>
		/// ICE has established a direct connection to the remote host. 
		/// <para>
		/// </para>
		/// </summary>
		HostConnection = 3,
		/// <summary>
		/// ICE has established a connection to the remote host through one or several
		/// NATs. 
		/// <para>
		/// </para>
		/// </summary>
		ReflexiveConnection = 4,
		/// <summary>
		/// ICE has established a connection through a relay. 
		/// <para>
		/// </para>
		/// </summary>
		RelayConnection = 5,
	}

	/// <summary>
	/// Enum describing how the authentification will be made. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum LdapAuthMethod
	{
		/// <summary>
		/// Connection without passwords. 
		/// <para>
		/// </para>
		/// </summary>
		Anonymous = 0,
		/// <summary>
		/// Connection with username/password. 
		/// <para>
		/// </para>
		/// </summary>
		Simple = 1,
	}

	/// <summary>
	/// Enum describing server certificates verification modes. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum LdapCertVerificationMode
	{
		/// <summary>
		/// Use default value defined on core. 
		/// <para>
		/// </para>
		/// </summary>
		Default = -1,
		/// <summary>
		/// Verification is disabled. 
		/// <para>
		/// </para>
		/// </summary>
		Disabled = 0,
		/// <summary>
		/// Verification is enabled. 
		/// <para>
		/// </para>
		/// </summary>
		Enabled = 1,
	}

	/// <summary>
	/// Enum describing errors in LDAP parameters. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum LdapCheck
	{
		/// <summary>
		/// No error. 
		/// <para>
		/// </para>
		/// </summary>
		Ok = 0,
		/// <summary>
		/// Server field is empty. 
		/// <para>
		/// </para>
		/// </summary>
		ServerEmpty = 1,
		/// <summary>
		/// The server is not an url. 
		/// <para>
		/// </para>
		/// </summary>
		ServerNotUrl = 2,
		/// <summary>
		/// The server doesn&apos;t contain a scheme. 
		/// <para>
		/// </para>
		/// </summary>
		ServerNoScheme = 4,
		/// <summary>
		/// The server is not a LDAP scheme. 
		/// <para>
		/// </para>
		/// </summary>
		ServerNotLdap = 8,
		/// <summary>
		/// LDAP over SSL is non-standardized and deprecated: ldaps has been specified. 
		/// <para>
		/// </para>
		/// </summary>
		ServerLdaps = 16,
		/// <summary>
		/// Base Object has been specified. 
		/// <para>
		/// </para>
		/// </summary>
		BaseObjectEmpty = 32,
		/// <summary>
		/// Some required fields are missing. 
		/// <para>
		/// </para>
		/// </summary>
		MissingFields = 64,
	}

	/// <summary>
	/// Enum Debug verbosity for OpenLdap. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum LdapDebugLevel
	{
		/// <summary>
		/// Set OpenLdap verbosity to none. 
		/// <para>
		/// </para>
		/// </summary>
		Off = 0,
		/// <summary>
		/// Set OpenLdap verbosity to debug level. 
		/// <para>
		/// </para>
		/// </summary>
		Verbose = 1,
	}

	/// <summary>
	/// <para>
	/// </para>
	/// </summary>
	
	public enum LimeState
	{
		/// <summary>
		/// Lime is not used at all. 
		/// <para>
		/// </para>
		/// </summary>
		Disabled = 0,
		/// <summary>
		/// Lime is always used. 
		/// <para>
		/// </para>
		/// </summary>
		Mandatory = 1,
		/// <summary>
		/// Lime is used only if we already shared a secret with remote. 
		/// <para>
		/// </para>
		/// </summary>
		Preferred = 2,
	}

	/// <summary>
	/// Whether or not to keep a file with the logs. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum LogCollectionState
	{
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Disabled = 0,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Enabled = 1,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		EnabledWithoutPreviousLogHandler = 2,
	}

	/// <summary>
	/// Verbosity levels of log messages. 
	/// <para>
	/// </para>
	/// </summary>
	[Flags]
	public enum LogLevel
	{
		/// <summary>
		/// Level for debug messages. 
		/// <para>
		/// </para>
		/// </summary>
		Debug = 1<<0,
		/// <summary>
		/// Level for traces. 
		/// <para>
		/// </para>
		/// </summary>
		Trace = 1<<1,
		/// <summary>
		/// Level for information messages. 
		/// <para>
		/// </para>
		/// </summary>
		Message = 1<<2,
		/// <summary>
		/// Level for warning messages. 
		/// <para>
		/// </para>
		/// </summary>
		Warning = 1<<3,
		/// <summary>
		/// Level for error messages. 
		/// <para>
		/// </para>
		/// </summary>
		Error = 1<<4,
		/// <summary>
		/// Level for fatal error messages. 
		/// <para>
		/// </para>
		/// </summary>
		Fatal = 1<<5,
	}

	/// <summary>
	/// Enum describing how to merge <see cref="Linphone.SearchResult">
	/// Linphone.SearchResult</see> from <see cref="Linphone.MagicSearch">
	/// Linphone.MagicSearch</see>. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum MagicSearchAggregation
	{
		/// <summary>
		/// No aggregation is done, you can have multiple SearchResult with the same
		/// Friend. 
		/// <para>
		/// </para>
		/// </summary>
		None = 0,
		/// <summary>
		/// Aggregation is done by friend, you will have at most a SearchResult per Friend. 
		/// <para>
		/// </para>
		/// </summary>
		Friend = 1,
	}

	/// <summary>
	/// Enum describing the search categories for Magic Search. 
	/// <para>
	/// </para>
	/// </summary>
	[Flags]
	public enum MagicSearchSource
	{
		/// <summary>
		/// no Source specified. 
		/// <para>
		/// If requested in search, the list should be empty 
		/// </para>
		/// </summary>
		None = 0,
		/// <summary>
		/// Search in friends only. 
		/// <para>
		/// </para>
		/// </summary>
		Friends = 1<<0,
		/// <summary>
		/// Search in Call Logs. 
		/// <para>
		/// </para>
		/// </summary>
		CallLogs = 1<<1,
		/// <summary>
		/// Search in LDAP servers. 
		/// <para>
		/// </para>
		/// </summary>
		LdapServers = 1<<2,
		/// <summary>
		/// Search in Chat rooms participants. 
		/// <para>
		/// </para>
		/// </summary>
		ChatRooms = 1<<3,
		/// <summary>
		/// Search from request : it is usually an address built from the request. 
		/// <para>
		/// </para>
		/// </summary>
		Request = 1<<4,
		/// <summary>
		/// Search in &quot;starred&quot; friends only. 
		/// <para>
		/// </para>
		/// </summary>
		FavoriteFriends = 1<<5,
		/// <summary>
		/// Search in conferences info (organizer and participants) 
		/// <para>
		/// </para>
		/// </summary>
		ConferencesInfo = 1<<6,
		/// <summary>
		/// Search in remote CardDAV servers (not locally synched ones) if any. 
		/// <para>
		/// </para>
		/// </summary>
		RemoteCardDAV = 1<<7,
		/// <summary>
		/// Search in all sources. 
		/// <para>
		/// </para>
		/// </summary>
		All = -1,
	}

	/// <summary>
	/// Indicates for a given media the stream direction. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum MediaDirection
	{
		/// <summary>
		/// Default value, shouldn&apos;t be used. 
		/// <para>
		/// </para>
		/// </summary>
		Invalid = -1,
		/// <summary>
		/// No active media not supported yet. 
		/// <para>
		/// </para>
		/// </summary>
		Inactive = 0,
		/// <summary>
		/// Media is only being sent, it won&apos;t be received. 
		/// <para>
		/// </para>
		/// </summary>
		SendOnly = 1,
		/// <summary>
		/// Media will only be received, nothing will be sent. 
		/// <para>
		/// </para>
		/// </summary>
		RecvOnly = 2,
		/// <summary>
		/// Media will be sent and received. 
		/// <para>
		/// </para>
		/// </summary>
		SendRecv = 3,
	}

	/// <summary>
	/// Enum describing type of media encryption types. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum MediaEncryption
	{
		/// <summary>
		/// No media encryption is used. 
		/// <para>
		/// </para>
		/// </summary>
		None = 0,
		/// <summary>
		/// Use SRTP media encryption. 
		/// <para>
		/// </para>
		/// </summary>
		SRTP = 1,
		/// <summary>
		/// Use ZRTP media encryption. 
		/// <para>
		/// </para>
		/// </summary>
		ZRTP = 2,
		/// <summary>
		/// Use DTLS media encryption. 
		/// <para>
		/// </para>
		/// </summary>
		DTLS = 3,
	}

	/// <summary>
	/// Enum representing the file format of a recording. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum MediaFileFormat
	{
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Unknown = 0,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Wav = 1,
		/// <summary>
		/// &lt; WAVE file format, .wav file extension. 
		/// <para>
		/// </para>
		/// </summary>
		Mkv = 2,
		/// <summary>
		/// &lt; Standard Matroska file format, supports video, .mkv or .mka file
		/// extension. 
		/// <para>
		/// &lt; Simple Multimedia File Format, a proprietary format that supports video,
		/// .smff file extension. 
		/// </para>
		/// </summary>
		Smff = 3,
	}

	/// <summary>
	/// Media resource usage. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum MediaResourceMode
	{
		/// <summary>
		/// Media resources are not shared. 
		/// <para>
		/// </para>
		/// </summary>
		Exclusive = 0,
		/// <summary>
		/// Media resources are shared. 
		/// <para>
		/// </para>
		/// </summary>
		Shared = 1,
	}

	/// <summary>
	/// Enum describing the different contexts for the <see cref="Linphone.MessageWaitingIndicationSummary">
	/// Linphone.MessageWaitingIndicationSummary</see>. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum MessageWaitingIndicationContextClass
	{
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Voice = 0,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Fax = 1,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Pager = 2,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Multimedia = 3,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Text = 4,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		None = 5,
	}

	/// <summary>
	/// <see cref="Linphone.ParticipantDeviceDisconnectionMethod">
	/// Linphone.ParticipantDeviceDisconnectionMethod</see> is used to indicate how a
	/// participant left a conference. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum ParticipantDeviceDisconnectionMethod
	{
		/// <summary>
		/// an admin removes the device from a conference 
		/// <para>
		/// </para>
		/// </summary>
		Booted = 0,
		/// <summary>
		/// the device disconnects from the conference 
		/// <para>
		/// </para>
		/// </summary>
		Departed = 1,
		/// <summary>
		/// device is busy 
		/// <para>
		/// </para>
		/// </summary>
		Busy = 2,
		/// <summary>
		/// an error occurred while the device is leaving the conference or he declined a
		/// call from the server 
		/// <para>
		/// </para>
		/// </summary>
		Failed = 3,
	}

	/// <summary>
	/// <see cref="Linphone.ParticipantDeviceJoiningMethod">
	/// Linphone.ParticipantDeviceJoiningMethod</see> is used to indicate how a
	/// participant joined a conference or if it is the focus. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum ParticipantDeviceJoiningMethod
	{
		/// <summary>
		/// device called the conference 
		/// <para>
		/// </para>
		/// </summary>
		DialedIn = 0,
		/// <summary>
		/// device is called the conference 
		/// <para>
		/// </para>
		/// </summary>
		DialedOut = 1,
		/// <summary>
		/// device is the focus 
		/// <para>
		/// </para>
		/// </summary>
		FocusOwner = 2,
	}

	/// <summary>
	/// <see cref="Linphone.ParticipantDeviceState">
	/// Linphone.ParticipantDeviceState</see> is used to list all states a participant
	/// device can be in 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum ParticipantDeviceState
	{
		/// <summary>
		/// an INVITE has been sent 
		/// <para>
		/// </para>
		/// </summary>
		Joining = 0,
		/// <summary>
		/// the SIP session has been concluded, participant is part of the conference 
		/// <para>
		/// </para>
		/// </summary>
		Present = 1,
		/// <summary>
		/// A BYE is pending. 
		/// <para>
		/// </para>
		/// </summary>
		Leaving = 2,
		/// <summary>
		/// The Session is terminated. 
		/// <para>
		/// </para>
		/// </summary>
		Left = 3,
		/// <summary>
		/// Initial state for the server group chatroom, when the participant has not yet
		/// been INVITEd. 
		/// <para>
		/// </para>
		/// </summary>
		ScheduledForJoining = 4,
		/// <summary>
		/// Transitional state for a participant that will receive a BYE shortly. 
		/// <para>
		/// </para>
		/// </summary>
		ScheduledForLeaving = 5,
		/// <summary>
		/// the SIP session has been concluded, participant is not media mixed 
		/// <para>
		/// </para>
		/// </summary>
		OnHold = 6,
		/// <summary>
		/// 180 Ringing 
		/// <para>
		/// </para>
		/// </summary>
		Alerting = 7,
		/// <summary>
		/// Some medias have been muted by the focus. 
		/// <para>
		/// </para>
		/// </summary>
		MutedByFocus = 8,
		/// <summary>
		/// the participant has sent a request to join the conference as he/she didn&apos;t
		/// receive any invitation for 
		/// <para>
		/// </para>
		/// </summary>
		RequestingToJoin = 9,
	}

	/// <summary>
	/// <see cref="Linphone.ParticipantRole">Linphone.ParticipantRole</see> is used to
	/// define a role of a participant within a conference 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum ParticipantRole
	{
		/// <summary>
		/// participant is a speaker in the conference 
		/// <para>
		/// </para>
		/// </summary>
		Speaker = 0,
		/// <summary>
		/// participant is a listener in the conference. 
		/// <para>
		/// He/She cannot speak 
		/// </para>
		/// </summary>
		Listener = 1,
		/// <summary>
		/// participant role is unknown 
		/// <para>
		/// </para>
		/// </summary>
		Unknown = 2,
	}

	/// <summary>
	/// The state of a <see cref="Linphone.Player">Linphone.Player</see>. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum PlayerState
	{
		/// <summary>
		/// No file is opened for playing. 
		/// <para>
		/// </para>
		/// </summary>
		Closed = 0,
		/// <summary>
		/// The player is paused. 
		/// <para>
		/// </para>
		/// </summary>
		Paused = 1,
		/// <summary>
		/// The player is playing. 
		/// <para>
		/// </para>
		/// </summary>
		Playing = 2,
	}

	/// <summary>
	/// Activities as defined in section 3.2 of RFC 4480. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum PresenceActivityType
	{
		/// <summary>
		/// The person has a calendar appointment, without specifying exactly of what type. 
		/// <para>
		/// This activity is indicated if more detailed information is not available or the
		/// person chooses not to reveal more information. 
		/// </para>
		/// </summary>
		Appointment = 0,
		/// <summary>
		/// The person is physically away from all interactive communication devices. 
		/// <para>
		/// </para>
		/// </summary>
		Away = 1,
		/// <summary>
		/// The person is eating the first meal of the day, usually eaten in the morning. 
		/// <para>
		/// </para>
		/// </summary>
		Breakfast = 2,
		/// <summary>
		/// The person is busy, without further details. 
		/// <para>
		/// </para>
		/// </summary>
		Busy = 3,
		/// <summary>
		/// The person is having his or her main meal of the day, eaten in the evening or
		/// at midday. 
		/// <para>
		/// </para>
		/// </summary>
		Dinner = 4,
		/// <summary>
		/// This is a scheduled national or local holiday. 
		/// <para>
		/// </para>
		/// </summary>
		Holiday = 5,
		/// <summary>
		/// The person is riding in a vehicle, such as a car, but not steering. 
		/// <para>
		/// </para>
		/// </summary>
		InTransit = 6,
		/// <summary>
		/// The person is looking for (paid) work. 
		/// <para>
		/// </para>
		/// </summary>
		LookingForWork = 7,
		/// <summary>
		/// The person is eating his or her midday meal. 
		/// <para>
		/// </para>
		/// </summary>
		Lunch = 8,
		/// <summary>
		/// The person is scheduled for a meal, without specifying whether it is breakfast,
		/// lunch, or dinner, or some other meal. 
		/// <para>
		/// </para>
		/// </summary>
		Meal = 9,
		/// <summary>
		/// The person is in an assembly or gathering of people, as for a business, social,
		/// or religious purpose. 
		/// <para>
		/// A meeting is a sub-class of an appointment. 
		/// </para>
		/// </summary>
		Meeting = 10,
		/// <summary>
		/// The person is talking on the telephone. 
		/// <para>
		/// </para>
		/// </summary>
		OnThePhone = 11,
		/// <summary>
		/// The person is engaged in an activity with no defined representation. 
		/// <para>
		/// A string describing the activity in plain text SHOULD be provided. 
		/// </para>
		/// </summary>
		Other = 12,
		/// <summary>
		/// A performance is a sub-class of an appointment and includes musical,
		/// theatrical, and cinematic performances as well as lectures. 
		/// <para>
		/// It is distinguished from a meeting by the fact that the person may either be
		/// lecturing or be in the audience, with a potentially large number of other
		/// people, making interruptions particularly noticeable. 
		/// </para>
		/// </summary>
		Performance = 13,
		/// <summary>
		/// The person will not return for the foreseeable future, e.g., because it is no
		/// longer working for the company. 
		/// <para>
		/// </para>
		/// </summary>
		PermanentAbsence = 14,
		/// <summary>
		/// The person is occupying himself or herself in amusement, sport, or other
		/// recreation. 
		/// <para>
		/// </para>
		/// </summary>
		Playing = 15,
		/// <summary>
		/// The person is giving a presentation, lecture, or participating in a formal
		/// round-table discussion. 
		/// <para>
		/// </para>
		/// </summary>
		Presentation = 16,
		/// <summary>
		/// The person is visiting stores in search of goods or services. 
		/// <para>
		/// </para>
		/// </summary>
		Shopping = 17,
		/// <summary>
		/// The person is sleeping. 
		/// <para>
		/// </para>
		/// </summary>
		Sleeping = 18,
		/// <summary>
		/// The person is observing an event, such as a sports event. 
		/// <para>
		/// </para>
		/// </summary>
		Spectator = 19,
		/// <summary>
		/// The person is controlling a vehicle, watercraft, or plane. 
		/// <para>
		/// </para>
		/// </summary>
		Steering = 20,
		/// <summary>
		/// The person is on a business or personal trip, but not necessarily in-transit. 
		/// <para>
		/// </para>
		/// </summary>
		Travel = 21,
		/// <summary>
		/// The person is watching television. 
		/// <para>
		/// </para>
		/// </summary>
		TV = 22,
		/// <summary>
		/// The activity of the person is unknown. 
		/// <para>
		/// </para>
		/// </summary>
		Unknown = 23,
		/// <summary>
		/// A period of time devoted to pleasure, rest, or relaxation. 
		/// <para>
		/// </para>
		/// </summary>
		Vacation = 24,
		/// <summary>
		/// The person is engaged in, typically paid, labor, as part of a profession or
		/// job. 
		/// <para>
		/// </para>
		/// </summary>
		Working = 25,
		/// <summary>
		/// The person is participating in religious rites. 
		/// <para>
		/// </para>
		/// </summary>
		Worship = 26,
	}

	/// <summary>
	/// Basic status as defined in section 4.1.4 of RFC 3863. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum PresenceBasicStatus
	{
		/// <summary>
		/// This value means that the associated contact element, if any, is ready to
		/// accept communication. 
		/// <para>
		/// </para>
		/// </summary>
		Open = 0,
		/// <summary>
		/// This value means that the associated contact element, if any, is unable to
		/// accept communication. 
		/// <para>
		/// </para>
		/// </summary>
		Closed = 1,
	}

	/// <summary>
	/// Defines privacy policy to apply as described by rfc3323. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum Privacy
	{
		/// <summary>
		/// Privacy services must not perform any privacy function. 
		/// <para>
		/// </para>
		/// </summary>
		None = 0,
		/// <summary>
		/// Request that privacy services provide a user-level privacy function. 
		/// <para>
		/// With this mode, &quot;from&quot; header is hidden, usually replaced by From:
		/// &quot;Anonymous&quot; &lt;sip:anonymous@anonymous.invalid&gt; 
		/// </para>
		/// </summary>
		User = 1,
		/// <summary>
		/// Request that privacy services modify headers that cannot be set arbitrarily by
		/// the user (Contact/Via). 
		/// <para>
		/// </para>
		/// </summary>
		Header = 2,
		/// <summary>
		/// Request that privacy services provide privacy for session media. 
		/// <para>
		/// </para>
		/// </summary>
		Session = 4,
		/// <summary>
		/// rfc3325 The presence of this privacy type in a Privacy header field indicates
		/// that the user would like the Network Asserted Identity to be kept private with
		/// respect to SIP entities outside the Trust Domain with which the user
		/// authenticated. 
		/// <para>
		/// Note that a user requesting multiple types of privacy MUST include all of the
		/// requested privacy types in its Privacy header field value 
		/// </para>
		/// </summary>
		Id = 8,
		/// <summary>
		/// Privacy service must perform the specified services or fail the request. 
		/// <para>
		/// </para>
		/// </summary>
		Critical = 16,
		/// <summary>
		/// Special keyword to use privacy as defined either globally or by proxy using
		/// <see cref="Linphone.ProxyConfig.Privacy">Linphone.ProxyConfig.Privacy</see> 
		/// <para>
		/// </para>
		/// </summary>
		Default = 32768,
	}

	/// <summary>
	/// Enum for publish states. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum PublishState
	{
		/// <summary>
		/// Initial state, do not use. 
		/// <para>
		/// </para>
		/// </summary>
		None = 0,
		/// <summary>
		/// An incoming publish is received. 
		/// <para>
		/// </para>
		/// </summary>
		IncomingReceived = 1,
		/// <summary>
		/// Publish is accepted. 
		/// <para>
		/// </para>
		/// </summary>
		Ok = 2,
		/// <summary>
		/// Publish encoutered an error, <see cref="Linphone.Event.Reason">
		/// Linphone.Event.Reason</see> gives reason code. 
		/// <para>
		/// </para>
		/// </summary>
		Error = 3,
		/// <summary>
		/// Publish is about to expire, only sent if [sip]-&gt;refresh_generic_publish
		/// property is set to 0. 
		/// <para>
		/// </para>
		/// </summary>
		Expiring = 4,
		/// <summary>
		/// Event has been un published. 
		/// <para>
		/// </para>
		/// </summary>
		Cleared = 5,
		/// <summary>
		/// Publish is about to terminate. 
		/// <para>
		/// </para>
		/// </summary>
		Terminating = 6,
		/// <summary>
		/// An outgoing publish was created and submitted. 
		/// <para>
		/// </para>
		/// </summary>
		OutgoingProgress = 7,
		/// <summary>
		/// Publish is about to refresh. 
		/// <para>
		/// </para>
		/// </summary>
		Refreshing = 8,
	}

	/// <summary>
	/// Enum describing various failure reasons or contextual information for some
	/// events. 
	/// <para>
	/// </para>
	/// </summary>
	/// <remarks>See : <see cref="Linphone.Call.Reason">Linphone.Call.Reason</see>
	/// </remarks>
	/// <remarks>See : <see cref="Linphone.ProxyConfig.Error">
	/// Linphone.ProxyConfig.Error</see> </remarks>
	/// <remarks>See : <see cref="Linphone.ErrorInfo.Reason">
	/// Linphone.ErrorInfo.Reason</see> </remarks>
	
	public enum Reason
	{
		/// <summary>
		/// No reason has been set by the core. 
		/// <para>
		/// </para>
		/// </summary>
		None = 0,
		/// <summary>
		/// No response received from remote. 
		/// <para>
		/// </para>
		/// </summary>
		NoResponse = 1,
		/// <summary>
		/// Authentication failed due to bad credentials or resource forbidden. 
		/// <para>
		/// </para>
		/// </summary>
		Forbidden = 2,
		/// <summary>
		/// The call has been declined. 
		/// <para>
		/// </para>
		/// </summary>
		Declined = 3,
		/// <summary>
		/// Destination of the call was not found. 
		/// <para>
		/// </para>
		/// </summary>
		NotFound = 4,
		/// <summary>
		/// The call was not answered in time (request timeout) 
		/// <para>
		/// </para>
		/// </summary>
		NotAnswered = 5,
		/// <summary>
		/// Phone line was busy. 
		/// <para>
		/// </para>
		/// </summary>
		Busy = 6,
		/// <summary>
		/// Unsupported content. 
		/// <para>
		/// </para>
		/// </summary>
		UnsupportedContent = 7,
		/// <summary>
		/// Bad event. 
		/// <para>
		/// </para>
		/// </summary>
		BadEvent = 8,
		/// <summary>
		/// Transport error: connection failures, disconnections etc... 
		/// <para>
		/// </para>
		/// </summary>
		IOError = 9,
		/// <summary>
		/// Do not disturb reason. 
		/// <para>
		/// </para>
		/// </summary>
		DoNotDisturb = 10,
		/// <summary>
		/// Operation is unauthorized because missing credential. 
		/// <para>
		/// </para>
		/// </summary>
		Unauthorized = 11,
		/// <summary>
		/// Operation is rejected due to incompatible or unsupported media parameters. 
		/// <para>
		/// </para>
		/// </summary>
		NotAcceptable = 12,
		/// <summary>
		/// Operation could not be executed by server or remote client because it
		/// didn&apos;t have any context for it. 
		/// <para>
		/// </para>
		/// </summary>
		NoMatch = 13,
		/// <summary>
		/// Resource moved permanently. 
		/// <para>
		/// </para>
		/// </summary>
		MovedPermanently = 14,
		/// <summary>
		/// Resource no longer exists. 
		/// <para>
		/// </para>
		/// </summary>
		Gone = 15,
		/// <summary>
		/// Temporarily unavailable. 
		/// <para>
		/// </para>
		/// </summary>
		TemporarilyUnavailable = 16,
		/// <summary>
		/// Address incomplete. 
		/// <para>
		/// </para>
		/// </summary>
		AddressIncomplete = 17,
		/// <summary>
		/// Not implemented. 
		/// <para>
		/// </para>
		/// </summary>
		NotImplemented = 18,
		/// <summary>
		/// Bad gateway. 
		/// <para>
		/// </para>
		/// </summary>
		BadGateway = 19,
		/// <summary>
		/// The received request contains a Session-Expires header field with a duration
		/// below the minimum timer. 
		/// <para>
		/// </para>
		/// </summary>
		SessionIntervalTooSmall = 20,
		/// <summary>
		/// Server timeout. 
		/// <para>
		/// </para>
		/// </summary>
		ServerTimeout = 21,
		/// <summary>
		/// Unknown reason. 
		/// <para>
		/// </para>
		/// </summary>
		Unknown = 22,
		/// <summary>
		/// The call has been transferred. 
		/// <para>
		/// </para>
		/// </summary>
		Transferred = 23,
		/// <summary>
		/// Conditional Request Failed. 
		/// <para>
		/// </para>
		/// </summary>
		ConditionalRequestFailed = 24,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		SasCheckRequired = 25,
	}

	/// <summary>
	/// Enum representing the state of a recording. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum RecorderState
	{
		/// <summary>
		/// No file is opened for recording. 
		/// <para>
		/// </para>
		/// </summary>
		Closed = 0,
		/// <summary>
		/// The recorder is paused. 
		/// <para>
		/// </para>
		/// </summary>
		Paused = 1,
		/// <summary>
		/// The recorder is running. 
		/// <para>
		/// </para>
		/// </summary>
		Running = 2,
	}

	/// <summary>
	/// Describes proxy registration states. 
	/// <para>
	/// It is notified via the registration_state_changed() callback in <see cref="Linphone.CoreListener">
	/// Linphone.CoreListener</see>. 
	/// </para>
	/// </summary>
	
	public enum RegistrationState
	{
		/// <summary>
		/// Initial state for registrations. 
		/// <para>
		/// </para>
		/// </summary>
		None = 0,
		/// <summary>
		/// Registration is in progress. 
		/// <para>
		/// </para>
		/// </summary>
		Progress = 1,
		/// <summary>
		/// Registration is successful. 
		/// <para>
		/// </para>
		/// </summary>
		Ok = 2,
		/// <summary>
		/// Unregistration succeeded. 
		/// <para>
		/// </para>
		/// </summary>
		Cleared = 3,
		/// <summary>
		/// Registration failed. 
		/// <para>
		/// </para>
		/// </summary>
		Failed = 4,
		/// <summary>
		/// Registration refreshing. 
		/// <para>
		/// </para>
		/// </summary>
		Refreshing = 5,
	}

	/// <summary>
	/// Enum describing the type of #RemoteContactDirectory (currently CardDAV or
	/// LDAP). 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum RemoteContactDirectoryType
	{
		/// <summary>
		/// Remote contact directory will use #CardDavParams. 
		/// <para>
		/// </para>
		/// </summary>
		CardDav = 0,
		/// <summary>
		/// Remote contact directory will use #LdapParams. 
		/// <para>
		/// </para>
		/// </summary>
		Ldap = 1,
	}

	/// <summary>
	/// Enum for search direction. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum SearchDirection
	{
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Up = 0,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Down = 1,
	}

	/// <summary>
	/// <see cref="Linphone.SecurityEventType">Linphone.SecurityEventType</see> is used
	/// to indicate the type of security event. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum SecurityEventType
	{
		/// <summary>
		/// Event is not a security event. 
		/// <para>
		/// </para>
		/// </summary>
		None = 0,
		/// <summary>
		/// Chatroom security level downgraded event. 
		/// <para>
		/// </para>
		/// </summary>
		SecurityLevelDowngraded = 1,
		/// <summary>
		/// Participant has exceeded the maximum number of device event. 
		/// <para>
		/// </para>
		/// </summary>
		ParticipantMaxDeviceCountExceeded = 2,
		/// <summary>
		/// Peer device instant messaging encryption identity key has changed event. 
		/// <para>
		/// </para>
		/// </summary>
		EncryptionIdentityKeyChanged = 3,
		/// <summary>
		/// Man in the middle detected event. 
		/// <para>
		/// </para>
		/// </summary>
		ManInTheMiddleDetected = 4,
	}

	/// <summary>
	/// Security level determined by type of encryption (point-to-point, end-to-end,
	/// etc...) and whether or not a SAS validation was made with the remote(s) end(s). 
	/// <para>
	/// A <see cref="Linphone.SecurityLevel.EndToEndEncryptedAndVerified">
	/// Linphone.SecurityLevel.EndToEndEncryptedAndVerified</see> level means it&apos;s
	/// end-to-end encrypted and SAS validation was made. An <see cref="Linphone.SecurityLevel.Unsafe">
	/// Linphone.SecurityLevel.Unsafe</see> level means end-to-end-encrypted but
	/// it&apos;s likely a man-in-the-middle exists between you and one device. 
	/// </para>
	/// </summary>
	
	public enum SecurityLevel
	{
		/// <summary>
		/// Security failure. 
		/// <para>
		/// </para>
		/// </summary>
		Unsafe = 0,
		/// <summary>
		/// No encryption. 
		/// <para>
		/// </para>
		/// </summary>
		None = 1,
		/// <summary>
		/// End-to-end encrypted. 
		/// <para>
		/// </para>
		/// </summary>
		EndToEndEncrypted = 2,
		/// <summary>
		/// End-to-end encrypted and verified. 
		/// <para>
		/// </para>
		/// </summary>
		EndToEndEncryptedAndVerified = 3,
		/// <summary>
		/// Point-to-point encrypted. 
		/// <para>
		/// </para>
		/// </summary>
		PointToPointEncrypted = 4,
	}

	/// <summary>
	/// Session Timers refresher. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum SessionExpiresRefresher
	{
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Unspecified = 0,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		UAS = 1,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		UAC = 2,
	}

	/// <summary>
	/// All signal units that a device can use. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum SignalStrengthUnit
	{
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Rssi = 0,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Dbm = 1,
	}

	/// <summary>
	/// All signal types that a device can use. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum SignalType
	{
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Wifi = 0,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Mobile = 1,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Other = 2,
	}

	/// <summary>
	/// Enum describing type of SRTP encryption suite. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum SrtpSuite
	{
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		AESCM128HMACSHA180 = 0,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		AESCM128HMACSHA132 = 1,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		AES192CMHMACSHA180 = 2,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		AES192CMHMACSHA132 = 3,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		AES256CMHMACSHA180 = 4,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		AES256CMHMACSHA132 = 5,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		AEADAES128GCM = 6,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		AEADAES256GCM = 7,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Invalid = 255,
	}

	/// <summary>
	/// Enum describing the stream types. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum StreamType
	{
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Audio = 0,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Video = 1,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Text = 2,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Unknown = 3,
	}

	/// <summary>
	/// Enum controlling behavior for incoming subscription request. 
	/// <para>
	/// Use by <see cref="Linphone.Friend.IncSubscribePolicy">
	/// Linphone.Friend.IncSubscribePolicy</see> 
	/// </para>
	/// </summary>
	
	public enum SubscribePolicy
	{
		/// <summary>
		/// Does not automatically accept an incoming subscription request. 
		/// <para>
		/// This policy implies that a decision has to be taken for each incoming
		/// subscription request notified by callback
		/// LinphoneCoreVTable.new_subscription_requested 
		/// </para>
		/// </summary>
		SPWait = 0,
		/// <summary>
		/// Rejects incoming subscription request. 
		/// <para>
		/// </para>
		/// </summary>
		SPDeny = 1,
		/// <summary>
		/// Automatically accepts a subscription request. 
		/// <para>
		/// </para>
		/// </summary>
		SPAccept = 2,
	}

	/// <summary>
	/// Enum for subscription direction (incoming or outgoing). 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum SubscriptionDir
	{
		/// <summary>
		/// Incoming subscription. 
		/// <para>
		/// </para>
		/// </summary>
		Incoming = 0,
		/// <summary>
		/// Outgoing subscription. 
		/// <para>
		/// </para>
		/// </summary>
		Outgoing = 1,
		/// <summary>
		/// Invalid subscription direction. 
		/// <para>
		/// </para>
		/// </summary>
		InvalidDir = 2,
	}

	/// <summary>
	/// Enum for subscription states. 
	/// <para>
	/// <see cref="Linphone.SubscriptionState.Terminated">
	/// Linphone.SubscriptionState.Terminated</see> and <see cref="Linphone.SubscriptionState.Error">
	/// Linphone.SubscriptionState.Error</see> are final states. 
	/// </para>
	/// </summary>
	
	public enum SubscriptionState
	{
		/// <summary>
		/// Initial state, should not be used. 
		/// <para>
		/// </para>
		/// </summary>
		None = 0,
		/// <summary>
		/// An outgoing subcription was sent. 
		/// <para>
		/// </para>
		/// </summary>
		OutgoingProgress = 1,
		/// <summary>
		/// An incoming subcription is received. 
		/// <para>
		/// </para>
		/// </summary>
		IncomingReceived = 2,
		/// <summary>
		/// Subscription is pending, waiting for user approval. 
		/// <para>
		/// </para>
		/// </summary>
		Pending = 3,
		/// <summary>
		/// Subscription is accepted. 
		/// <para>
		/// </para>
		/// </summary>
		Active = 4,
		/// <summary>
		/// Subscription is terminated normally. 
		/// <para>
		/// </para>
		/// </summary>
		Terminated = 5,
		/// <summary>
		/// Subscription was terminated by an error, indicated by <see cref="Linphone.Event.Reason">
		/// Linphone.Event.Reason</see> 
		/// <para>
		/// </para>
		/// </summary>
		Error = 6,
		/// <summary>
		/// Subscription is about to expire, only sent if
		/// [sip]-&gt;refresh_generic_subscribe property is set to 0. 
		/// <para>
		/// </para>
		/// </summary>
		Expiring = 7,
	}

	/// <summary>
	/// <para>
	/// </para>
	/// </summary>
	
	public enum SupportLevel
	{
		/// <summary>
		/// No support for the feature. 
		/// <para>
		/// </para>
		/// </summary>
		NoSupport = 0,
		/// <summary>
		/// Optional support for the feature. 
		/// <para>
		/// </para>
		/// </summary>
		Optional = 1,
		/// <summary>
		/// Mandatory support for the feature. 
		/// <para>
		/// </para>
		/// </summary>
		Mandatory = 2,
	}

	/// <summary>
	/// Enum listing frequent telephony tones. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum ToneID
	{
		/// <summary>
		/// Not a tone. 
		/// <para>
		/// </para>
		/// </summary>
		Undefined = 0,
		/// <summary>
		/// Busy tone. 
		/// <para>
		/// </para>
		/// </summary>
		Busy = 1,
		/// <summary>
		/// Call waiting tone. 
		/// <para>
		/// </para>
		/// </summary>
		CallWaiting = 2,
		/// <summary>
		/// Call on hold tone. 
		/// <para>
		/// </para>
		/// </summary>
		CallOnHold = 3,
		/// <summary>
		/// Tone played when call is abruptly disconnected (media lost) 
		/// <para>
		/// </para>
		/// </summary>
		CallLost = 4,
		/// <summary>
		/// When the call end for any reason but lost. 
		/// <para>
		/// </para>
		/// </summary>
		CallEnd = 5,
		/// <summary>
		/// When the call is not answered. 
		/// <para>
		/// </para>
		/// </summary>
		CallNotAnswered = 6,
		/// <summary>
		/// When the SAS check is required. 
		/// <para>
		/// </para>
		/// </summary>
		SasCheckRequired = 7,
	}

	/// <summary>
	/// Enum describing transport type for LinphoneAddress. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum TransportType
	{
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Udp = 0,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Tcp = 1,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Tls = 2,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Dtls = 3,
	}

	/// <summary>
	/// Enum describing the tunnel modes. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum TunnelMode
	{
		/// <summary>
		/// The tunnel is disabled. 
		/// <para>
		/// </para>
		/// </summary>
		Disable = 0,
		/// <summary>
		/// The tunnel is enabled. 
		/// <para>
		/// </para>
		/// </summary>
		Enable = 1,
		/// <summary>
		/// The tunnel is enabled automatically if it is required. 
		/// <para>
		/// </para>
		/// </summary>
		Auto = 2,
	}

	/// <summary>
	/// Enum describing uPnP states. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum UpnpState
	{
		/// <summary>
		/// uPnP is not activate 
		/// <para>
		/// </para>
		/// </summary>
		Idle = 0,
		/// <summary>
		/// uPnP process is in progress 
		/// <para>
		/// </para>
		/// </summary>
		Pending = 1,
		/// <summary>
		/// Internal use: Only used by port binding. 
		/// <para>
		/// </para>
		/// </summary>
		Adding = 2,
		/// <summary>
		/// Internal use: Only used by port binding. 
		/// <para>
		/// </para>
		/// </summary>
		Removing = 3,
		/// <summary>
		/// uPnP is not available 
		/// <para>
		/// </para>
		/// </summary>
		NotAvailable = 4,
		/// <summary>
		/// uPnP is enabled 
		/// <para>
		/// </para>
		/// </summary>
		Ok = 5,
		/// <summary>
		/// uPnP processing has failed 
		/// <para>
		/// </para>
		/// </summary>
		Ko = 6,
		/// <summary>
		/// IGD router is blacklisted. 
		/// <para>
		/// </para>
		/// </summary>
		Blacklisted = 7,
	}

	/// <summary>
	/// Enum describing the result of a version update check. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum VersionUpdateCheckResult
	{
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		UpToDate = 0,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		NewVersionAvailable = 1,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Error = 2,
	}

	/// <summary>
	/// Enum representing the sub type of the screen sharing. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum VideoSourceScreenSharingType
	{
		/// <summary>
		/// The screen sharing is done from a display. 
		/// <para>
		/// </para>
		/// </summary>
		Display = 0,
		/// <summary>
		/// The screen sharing is done from a window. 
		/// <para>
		/// </para>
		/// </summary>
		Window = 1,
		/// <summary>
		/// The screen sharing is done from an area. 
		/// <para>
		/// </para>
		/// </summary>
		Area = 2,
	}

	/// <summary>
	/// Enum representing the type of a video source. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum VideoSourceType
	{
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Unknown = 0,
		/// <summary>
		/// The video source is another call. 
		/// <para>
		/// </para>
		/// </summary>
		Call = 1,
		/// <summary>
		/// The video source is a camera. 
		/// <para>
		/// </para>
		/// </summary>
		Camera = 2,
		/// <summary>
		/// The video source is an image. 
		/// <para>
		/// </para>
		/// </summary>
		Image = 3,
		/// <summary>
		/// The video source is a screen sharing. 
		/// <para>
		/// </para>
		/// </summary>
		ScreenSharing = 4,
	}

	/// <summary>
	/// Enum describing the types of argument for LinphoneXmlRpcRequest. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum XmlRpcArgType
	{
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		None = 0,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Int = 1,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		String = 2,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		StringStruct = 3,
	}

	/// <summary>
	/// Enum describing the status of a LinphoneXmlRpcRequest. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum XmlRpcStatus
	{
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Pending = 0,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Ok = 1,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Failed = 2,
	}

	/// <summary>
	/// Enum describing the ZRTP key exchange algorithns. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum ZrtpKeyAgreement
	{
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Invalid = 0,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Dh2K = 1,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Dh3K = 2,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Ec25 = 3,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Ec38 = 4,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Ec52 = 5,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		X255 = 6,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		X448 = 7,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		K255 = 8,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		K448 = 9,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Kyb1 = 10,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Kyb2 = 11,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Kyb3 = 12,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Hqc1 = 13,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Hqc2 = 14,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Hqc3 = 15,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		K255Kyb512 = 16,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		K255Hqc128 = 17,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		K448Kyb1024 = 18,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		K448Hqc256 = 19,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		K255Kyb512Hqc128 = 20,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		K448Kyb1024Hqc256 = 21,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Mlk1 = 22,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Mlk2 = 23,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		Mlk3 = 24,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		K255Mlk512 = 25,
		/// <summary>
		/// <para>
		/// </para>
		/// </summary>
		K448Mlk1024 = 26,
	}

	/// <summary>
	/// Enum describing the ZRTP SAS validation status of a peer URI. 
	/// <para>
	/// </para>
	/// </summary>
	
	public enum ZrtpPeerStatus
	{
		/// <summary>
		/// Peer URI unkown or never validated/invalidated the SAS. 
		/// <para>
		/// </para>
		/// </summary>
		Unknown = 0,
		/// <summary>
		/// Peer URI SAS rejected in database. 
		/// <para>
		/// </para>
		/// </summary>
		Invalid = 1,
		/// <summary>
		/// Peer URI SAS validated in database. 
		/// <para>
		/// </para>
		/// </summary>
		Valid = 2,
	}

#endregion

#region Listeners
	/// <summary>
	/// An object to handle the callbacks for the handling of <see cref="Linphone.Account">
	/// Linphone.Account</see> objects. 
	/// <para>
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class AccountListener : LinphoneObject
	{
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_get_current_callbacks(IntPtr thiz);

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_account_cbs_set_registration_state_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_account_cbs_set_registration_state_changed(IntPtr thiz, OnRegistrationStateChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnRegistrationStateChangedDelegatePrivate(IntPtr account, int state, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string message);

		/// <summary>
		/// Callback for notifying when a registration state has changed for the account. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="account">LinphoneAccount object whose registration state changed. 
		///   </param>
		/// <param name="state">The current LinphoneRegistrationState. </param>
		/// <param name="message">A non null informational message about the state.   
		/// </param>
		public delegate void OnRegistrationStateChangedDelegate(Linphone.Account account, Linphone.RegistrationState state, string message);
		private OnRegistrationStateChangedDelegatePrivate on_registration_state_changed_private;
		private OnRegistrationStateChangedDelegate on_registration_state_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnRegistrationStateChangedDelegatePrivate))]
#endif
		private static void on_registration_state_changed(IntPtr account, int state, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string message)
		{
			Account thiz = fromNativePtr<Account>(account);

			IntPtr ptr = linphone_account_get_current_callbacks(thiz.nativePtr);
			AccountListener current_listener = fromNativePtr<AccountListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_registration_state_changed_public?.Invoke(thiz, (Linphone.RegistrationState)state, message);
			}
		}

		public OnRegistrationStateChangedDelegate OnRegistrationStateChanged
		{
			get
			{
				return on_registration_state_changed_public;
			}
			set
			{
				on_registration_state_changed_public = value;
				on_registration_state_changed_private = on_registration_state_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_registration_state_changed_private);
				linphone_account_cbs_set_registration_state_changed(nativePtr, cb);
#else
				linphone_account_cbs_set_registration_state_changed(nativePtr, on_registration_state_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_account_cbs_set_message_waiting_indication_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_account_cbs_set_message_waiting_indication_changed(IntPtr thiz, OnMessageWaitingIndicationChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnMessageWaitingIndicationChangedDelegatePrivate(IntPtr account, IntPtr mwi);

		/// <summary>
		/// Callback for notifying a message waiting indication change for the account. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="account">LinphoneAccount object whose message waiting indication
		/// changed.    </param>
		/// <param name="mwi">The current LinphoneMessageWaitingIndication.    </param>
		public delegate void OnMessageWaitingIndicationChangedDelegate(Linphone.Account account, Linphone.MessageWaitingIndication mwi);
		private OnMessageWaitingIndicationChangedDelegatePrivate on_message_waiting_indication_changed_private;
		private OnMessageWaitingIndicationChangedDelegate on_message_waiting_indication_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnMessageWaitingIndicationChangedDelegatePrivate))]
#endif
		private static void on_message_waiting_indication_changed(IntPtr account, IntPtr mwi)
		{
			Account thiz = fromNativePtr<Account>(account);

			IntPtr ptr = linphone_account_get_current_callbacks(thiz.nativePtr);
			AccountListener current_listener = fromNativePtr<AccountListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_message_waiting_indication_changed_public?.Invoke(thiz, fromNativePtr<Linphone.MessageWaitingIndication>(mwi));
			}
		}

		public OnMessageWaitingIndicationChangedDelegate OnMessageWaitingIndicationChanged
		{
			get
			{
				return on_message_waiting_indication_changed_public;
			}
			set
			{
				on_message_waiting_indication_changed_public = value;
				on_message_waiting_indication_changed_private = on_message_waiting_indication_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_message_waiting_indication_changed_private);
				linphone_account_cbs_set_message_waiting_indication_changed(nativePtr, cb);
#else
				linphone_account_cbs_set_message_waiting_indication_changed(nativePtr, on_message_waiting_indication_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_account_cbs_set_conference_information_updated(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_account_cbs_set_conference_information_updated(IntPtr thiz, OnConferenceInformationUpdatedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnConferenceInformationUpdatedDelegatePrivate(IntPtr account, IntPtr infos);

		/// <summary>
		/// Callback for notifying that the conference information list of an account has
		/// been updated. 
		/// <para>
		/// This is generally the case when retrieving conference informations from a CCMP
		/// server as the request might take a little bit of time to be responded. In order
		/// to allow the core to perform its tasks while the conference information are
		/// being sent, the core will send first the list of conference information that
		/// are locally stored and then this callback is called once it is updated with the
		/// information received by the CCMP server. 
		/// </para>
		/// </summary>
		/// <param name="account"><see cref="Linphone.Account">Linphone.Account</see>
		/// object whose message waiting indication changed.    </param>
		/// <param name="infos">The updated list of conference informations      </param>
		public delegate void OnConferenceInformationUpdatedDelegate(Linphone.Account account, IEnumerable<Linphone.ConferenceInfo> infos);
		private OnConferenceInformationUpdatedDelegatePrivate on_conference_information_updated_private;
		private OnConferenceInformationUpdatedDelegate on_conference_information_updated_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnConferenceInformationUpdatedDelegatePrivate))]
#endif
		private static void on_conference_information_updated(IntPtr account, IntPtr infos)
		{
			Account thiz = fromNativePtr<Account>(account);

			IntPtr ptr = linphone_account_get_current_callbacks(thiz.nativePtr);
			AccountListener current_listener = fromNativePtr<AccountListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_conference_information_updated_public?.Invoke(thiz, MarshalBctbxList<Linphone.ConferenceInfo>(infos));
			}
		}

		public OnConferenceInformationUpdatedDelegate OnConferenceInformationUpdated
		{
			get
			{
				return on_conference_information_updated_public;
			}
			set
			{
				on_conference_information_updated_public = value;
				on_conference_information_updated_private = on_conference_information_updated;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_conference_information_updated_private);
				linphone_account_cbs_set_conference_information_updated(nativePtr, cb);
#else
				linphone_account_cbs_set_conference_information_updated(nativePtr, on_conference_information_updated_private);
#endif
			}
		}
	}

	/// <summary>
	/// An object to handle the responses callbacks for handling the <see cref="Linphone.AccountCreator">
	/// Linphone.AccountCreator</see> operations. 
	/// <para>
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class AccountCreatorListener : LinphoneObject
	{
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_creator_get_current_callbacks(IntPtr thiz);

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_account_creator_cbs_set_create_account(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_account_creator_cbs_set_create_account(IntPtr thiz, OnCreateAccountDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnCreateAccountDelegatePrivate(IntPtr creator, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string response);

		/// <summary>
		/// Callback to notify a response of server. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="creator"><see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see> object    </param>
		/// <param name="status">The status of the <see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see> test existence operation that has just finished
		/// </param>
		/// <param name="response">The response has a string    </param>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public delegate void OnCreateAccountDelegate(Linphone.AccountCreator creator, Linphone.AccountCreatorStatus status, string response);
		private OnCreateAccountDelegatePrivate on_create_account_private;
		private OnCreateAccountDelegate on_create_account_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnCreateAccountDelegatePrivate))]
#endif
		private static void on_create_account(IntPtr creator, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string response)
		{
			AccountCreator thiz = fromNativePtr<AccountCreator>(creator);

			IntPtr ptr = linphone_account_creator_get_current_callbacks(thiz.nativePtr);
			AccountCreatorListener current_listener = fromNativePtr<AccountCreatorListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_create_account_public?.Invoke(thiz, (Linphone.AccountCreatorStatus)status, response);
			}
		}

		public OnCreateAccountDelegate OnCreateAccount
		{
			get
			{
				return on_create_account_public;
			}
			set
			{
				on_create_account_public = value;
				on_create_account_private = on_create_account;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_create_account_private);
				linphone_account_creator_cbs_set_create_account(nativePtr, cb);
#else
				linphone_account_creator_cbs_set_create_account(nativePtr, on_create_account_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_account_creator_cbs_set_is_account_exist(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_account_creator_cbs_set_is_account_exist(IntPtr thiz, OnIsAccountExistDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnIsAccountExistDelegatePrivate(IntPtr creator, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string response);

		/// <summary>
		/// Callback to notify a response of server. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="creator"><see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see> object    </param>
		/// <param name="status">The status of the <see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see> test existence operation that has just finished
		/// </param>
		/// <param name="response">The response has a string    </param>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public delegate void OnIsAccountExistDelegate(Linphone.AccountCreator creator, Linphone.AccountCreatorStatus status, string response);
		private OnIsAccountExistDelegatePrivate on_is_account_exist_private;
		private OnIsAccountExistDelegate on_is_account_exist_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnIsAccountExistDelegatePrivate))]
#endif
		private static void on_is_account_exist(IntPtr creator, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string response)
		{
			AccountCreator thiz = fromNativePtr<AccountCreator>(creator);

			IntPtr ptr = linphone_account_creator_get_current_callbacks(thiz.nativePtr);
			AccountCreatorListener current_listener = fromNativePtr<AccountCreatorListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_is_account_exist_public?.Invoke(thiz, (Linphone.AccountCreatorStatus)status, response);
			}
		}

		public OnIsAccountExistDelegate OnIsAccountExist
		{
			get
			{
				return on_is_account_exist_public;
			}
			set
			{
				on_is_account_exist_public = value;
				on_is_account_exist_private = on_is_account_exist;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_is_account_exist_private);
				linphone_account_creator_cbs_set_is_account_exist(nativePtr, cb);
#else
				linphone_account_creator_cbs_set_is_account_exist(nativePtr, on_is_account_exist_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_account_creator_cbs_set_activate_account(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_account_creator_cbs_set_activate_account(IntPtr thiz, OnActivateAccountDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnActivateAccountDelegatePrivate(IntPtr creator, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string response);

		/// <summary>
		/// Callback to notify a response of server. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="creator"><see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see> object    </param>
		/// <param name="status">The status of the <see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see> test existence operation that has just finished
		/// </param>
		/// <param name="response">The response has a string    </param>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public delegate void OnActivateAccountDelegate(Linphone.AccountCreator creator, Linphone.AccountCreatorStatus status, string response);
		private OnActivateAccountDelegatePrivate on_activate_account_private;
		private OnActivateAccountDelegate on_activate_account_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnActivateAccountDelegatePrivate))]
#endif
		private static void on_activate_account(IntPtr creator, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string response)
		{
			AccountCreator thiz = fromNativePtr<AccountCreator>(creator);

			IntPtr ptr = linphone_account_creator_get_current_callbacks(thiz.nativePtr);
			AccountCreatorListener current_listener = fromNativePtr<AccountCreatorListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_activate_account_public?.Invoke(thiz, (Linphone.AccountCreatorStatus)status, response);
			}
		}

		public OnActivateAccountDelegate OnActivateAccount
		{
			get
			{
				return on_activate_account_public;
			}
			set
			{
				on_activate_account_public = value;
				on_activate_account_private = on_activate_account;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_activate_account_private);
				linphone_account_creator_cbs_set_activate_account(nativePtr, cb);
#else
				linphone_account_creator_cbs_set_activate_account(nativePtr, on_activate_account_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_account_creator_cbs_set_send_token(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_account_creator_cbs_set_send_token(IntPtr thiz, OnSendTokenDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnSendTokenDelegatePrivate(IntPtr creator, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string response);

		/// <summary>
		/// Callback to notify a response of server. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="creator"><see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see> object    </param>
		/// <param name="status">The status of the <see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see> test existence operation that has just finished
		/// </param>
		/// <param name="response">The response has a string    </param>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public delegate void OnSendTokenDelegate(Linphone.AccountCreator creator, Linphone.AccountCreatorStatus status, string response);
		private OnSendTokenDelegatePrivate on_send_token_private;
		private OnSendTokenDelegate on_send_token_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnSendTokenDelegatePrivate))]
#endif
		private static void on_send_token(IntPtr creator, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string response)
		{
			AccountCreator thiz = fromNativePtr<AccountCreator>(creator);

			IntPtr ptr = linphone_account_creator_get_current_callbacks(thiz.nativePtr);
			AccountCreatorListener current_listener = fromNativePtr<AccountCreatorListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_send_token_public?.Invoke(thiz, (Linphone.AccountCreatorStatus)status, response);
			}
		}

		public OnSendTokenDelegate OnSendToken
		{
			get
			{
				return on_send_token_public;
			}
			set
			{
				on_send_token_public = value;
				on_send_token_private = on_send_token;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_send_token_private);
				linphone_account_creator_cbs_set_send_token(nativePtr, cb);
#else
				linphone_account_creator_cbs_set_send_token(nativePtr, on_send_token_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_account_creator_cbs_set_account_creation_request_token(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_account_creator_cbs_set_account_creation_request_token(IntPtr thiz, OnAccountCreationRequestTokenDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnAccountCreationRequestTokenDelegatePrivate(IntPtr creator, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string response);

		/// <summary>
		/// Callback to notify a response of server. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="creator"><see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see> object    </param>
		/// <param name="status">The status of the <see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see> test existence operation that has just finished
		/// </param>
		/// <param name="response">The response has a string    </param>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public delegate void OnAccountCreationRequestTokenDelegate(Linphone.AccountCreator creator, Linphone.AccountCreatorStatus status, string response);
		private OnAccountCreationRequestTokenDelegatePrivate on_account_creation_request_token_private;
		private OnAccountCreationRequestTokenDelegate on_account_creation_request_token_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnAccountCreationRequestTokenDelegatePrivate))]
#endif
		private static void on_account_creation_request_token(IntPtr creator, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string response)
		{
			AccountCreator thiz = fromNativePtr<AccountCreator>(creator);

			IntPtr ptr = linphone_account_creator_get_current_callbacks(thiz.nativePtr);
			AccountCreatorListener current_listener = fromNativePtr<AccountCreatorListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_account_creation_request_token_public?.Invoke(thiz, (Linphone.AccountCreatorStatus)status, response);
			}
		}

		public OnAccountCreationRequestTokenDelegate OnAccountCreationRequestToken
		{
			get
			{
				return on_account_creation_request_token_public;
			}
			set
			{
				on_account_creation_request_token_public = value;
				on_account_creation_request_token_private = on_account_creation_request_token;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_account_creation_request_token_private);
				linphone_account_creator_cbs_set_account_creation_request_token(nativePtr, cb);
#else
				linphone_account_creator_cbs_set_account_creation_request_token(nativePtr, on_account_creation_request_token_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_account_creator_cbs_set_account_creation_token_using_request_token(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_account_creator_cbs_set_account_creation_token_using_request_token(IntPtr thiz, OnAccountCreationTokenUsingRequestTokenDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnAccountCreationTokenUsingRequestTokenDelegatePrivate(IntPtr creator, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string response);

		/// <summary>
		/// Callback to notify a response of server. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="creator"><see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see> object    </param>
		/// <param name="status">The status of the <see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see> test existence operation that has just finished
		/// </param>
		/// <param name="response">The response has a string    </param>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public delegate void OnAccountCreationTokenUsingRequestTokenDelegate(Linphone.AccountCreator creator, Linphone.AccountCreatorStatus status, string response);
		private OnAccountCreationTokenUsingRequestTokenDelegatePrivate on_account_creation_token_using_request_token_private;
		private OnAccountCreationTokenUsingRequestTokenDelegate on_account_creation_token_using_request_token_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnAccountCreationTokenUsingRequestTokenDelegatePrivate))]
#endif
		private static void on_account_creation_token_using_request_token(IntPtr creator, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string response)
		{
			AccountCreator thiz = fromNativePtr<AccountCreator>(creator);

			IntPtr ptr = linphone_account_creator_get_current_callbacks(thiz.nativePtr);
			AccountCreatorListener current_listener = fromNativePtr<AccountCreatorListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_account_creation_token_using_request_token_public?.Invoke(thiz, (Linphone.AccountCreatorStatus)status, response);
			}
		}

		public OnAccountCreationTokenUsingRequestTokenDelegate OnAccountCreationTokenUsingRequestToken
		{
			get
			{
				return on_account_creation_token_using_request_token_public;
			}
			set
			{
				on_account_creation_token_using_request_token_public = value;
				on_account_creation_token_using_request_token_private = on_account_creation_token_using_request_token;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_account_creation_token_using_request_token_private);
				linphone_account_creator_cbs_set_account_creation_token_using_request_token(nativePtr, cb);
#else
				linphone_account_creator_cbs_set_account_creation_token_using_request_token(nativePtr, on_account_creation_token_using_request_token_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_account_creator_cbs_set_is_account_activated(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_account_creator_cbs_set_is_account_activated(IntPtr thiz, OnIsAccountActivatedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnIsAccountActivatedDelegatePrivate(IntPtr creator, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string response);

		/// <summary>
		/// Callback to notify a response of server. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="creator"><see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see> object    </param>
		/// <param name="status">The status of the <see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see> test existence operation that has just finished
		/// </param>
		/// <param name="response">The response has a string    </param>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public delegate void OnIsAccountActivatedDelegate(Linphone.AccountCreator creator, Linphone.AccountCreatorStatus status, string response);
		private OnIsAccountActivatedDelegatePrivate on_is_account_activated_private;
		private OnIsAccountActivatedDelegate on_is_account_activated_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnIsAccountActivatedDelegatePrivate))]
#endif
		private static void on_is_account_activated(IntPtr creator, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string response)
		{
			AccountCreator thiz = fromNativePtr<AccountCreator>(creator);

			IntPtr ptr = linphone_account_creator_get_current_callbacks(thiz.nativePtr);
			AccountCreatorListener current_listener = fromNativePtr<AccountCreatorListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_is_account_activated_public?.Invoke(thiz, (Linphone.AccountCreatorStatus)status, response);
			}
		}

		public OnIsAccountActivatedDelegate OnIsAccountActivated
		{
			get
			{
				return on_is_account_activated_public;
			}
			set
			{
				on_is_account_activated_public = value;
				on_is_account_activated_private = on_is_account_activated;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_is_account_activated_private);
				linphone_account_creator_cbs_set_is_account_activated(nativePtr, cb);
#else
				linphone_account_creator_cbs_set_is_account_activated(nativePtr, on_is_account_activated_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_account_creator_cbs_set_link_account(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_account_creator_cbs_set_link_account(IntPtr thiz, OnLinkAccountDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnLinkAccountDelegatePrivate(IntPtr creator, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string response);

		/// <summary>
		/// Callback to notify a response of server. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="creator"><see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see> object    </param>
		/// <param name="status">The status of the <see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see> test existence operation that has just finished
		/// </param>
		/// <param name="response">The response has a string    </param>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public delegate void OnLinkAccountDelegate(Linphone.AccountCreator creator, Linphone.AccountCreatorStatus status, string response);
		private OnLinkAccountDelegatePrivate on_link_account_private;
		private OnLinkAccountDelegate on_link_account_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnLinkAccountDelegatePrivate))]
#endif
		private static void on_link_account(IntPtr creator, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string response)
		{
			AccountCreator thiz = fromNativePtr<AccountCreator>(creator);

			IntPtr ptr = linphone_account_creator_get_current_callbacks(thiz.nativePtr);
			AccountCreatorListener current_listener = fromNativePtr<AccountCreatorListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_link_account_public?.Invoke(thiz, (Linphone.AccountCreatorStatus)status, response);
			}
		}

		public OnLinkAccountDelegate OnLinkAccount
		{
			get
			{
				return on_link_account_public;
			}
			set
			{
				on_link_account_public = value;
				on_link_account_private = on_link_account;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_link_account_private);
				linphone_account_creator_cbs_set_link_account(nativePtr, cb);
#else
				linphone_account_creator_cbs_set_link_account(nativePtr, on_link_account_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_account_creator_cbs_set_activate_alias(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_account_creator_cbs_set_activate_alias(IntPtr thiz, OnActivateAliasDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnActivateAliasDelegatePrivate(IntPtr creator, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string response);

		/// <summary>
		/// Callback to notify a response of server. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="creator"><see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see> object    </param>
		/// <param name="status">The status of the <see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see> test existence operation that has just finished
		/// </param>
		/// <param name="response">The response has a string    </param>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public delegate void OnActivateAliasDelegate(Linphone.AccountCreator creator, Linphone.AccountCreatorStatus status, string response);
		private OnActivateAliasDelegatePrivate on_activate_alias_private;
		private OnActivateAliasDelegate on_activate_alias_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnActivateAliasDelegatePrivate))]
#endif
		private static void on_activate_alias(IntPtr creator, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string response)
		{
			AccountCreator thiz = fromNativePtr<AccountCreator>(creator);

			IntPtr ptr = linphone_account_creator_get_current_callbacks(thiz.nativePtr);
			AccountCreatorListener current_listener = fromNativePtr<AccountCreatorListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_activate_alias_public?.Invoke(thiz, (Linphone.AccountCreatorStatus)status, response);
			}
		}

		public OnActivateAliasDelegate OnActivateAlias
		{
			get
			{
				return on_activate_alias_public;
			}
			set
			{
				on_activate_alias_public = value;
				on_activate_alias_private = on_activate_alias;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_activate_alias_private);
				linphone_account_creator_cbs_set_activate_alias(nativePtr, cb);
#else
				linphone_account_creator_cbs_set_activate_alias(nativePtr, on_activate_alias_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_account_creator_cbs_set_is_alias_used(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_account_creator_cbs_set_is_alias_used(IntPtr thiz, OnIsAliasUsedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnIsAliasUsedDelegatePrivate(IntPtr creator, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string response);

		/// <summary>
		/// Callback to notify a response of server. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="creator"><see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see> object    </param>
		/// <param name="status">The status of the <see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see> test existence operation that has just finished
		/// </param>
		/// <param name="response">The response has a string    </param>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public delegate void OnIsAliasUsedDelegate(Linphone.AccountCreator creator, Linphone.AccountCreatorStatus status, string response);
		private OnIsAliasUsedDelegatePrivate on_is_alias_used_private;
		private OnIsAliasUsedDelegate on_is_alias_used_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnIsAliasUsedDelegatePrivate))]
#endif
		private static void on_is_alias_used(IntPtr creator, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string response)
		{
			AccountCreator thiz = fromNativePtr<AccountCreator>(creator);

			IntPtr ptr = linphone_account_creator_get_current_callbacks(thiz.nativePtr);
			AccountCreatorListener current_listener = fromNativePtr<AccountCreatorListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_is_alias_used_public?.Invoke(thiz, (Linphone.AccountCreatorStatus)status, response);
			}
		}

		public OnIsAliasUsedDelegate OnIsAliasUsed
		{
			get
			{
				return on_is_alias_used_public;
			}
			set
			{
				on_is_alias_used_public = value;
				on_is_alias_used_private = on_is_alias_used;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_is_alias_used_private);
				linphone_account_creator_cbs_set_is_alias_used(nativePtr, cb);
#else
				linphone_account_creator_cbs_set_is_alias_used(nativePtr, on_is_alias_used_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_account_creator_cbs_set_is_account_linked(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_account_creator_cbs_set_is_account_linked(IntPtr thiz, OnIsAccountLinkedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnIsAccountLinkedDelegatePrivate(IntPtr creator, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string response);

		/// <summary>
		/// Callback to notify a response of server. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="creator"><see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see> object    </param>
		/// <param name="status">The status of the <see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see> test existence operation that has just finished
		/// </param>
		/// <param name="response">The response has a string    </param>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public delegate void OnIsAccountLinkedDelegate(Linphone.AccountCreator creator, Linphone.AccountCreatorStatus status, string response);
		private OnIsAccountLinkedDelegatePrivate on_is_account_linked_private;
		private OnIsAccountLinkedDelegate on_is_account_linked_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnIsAccountLinkedDelegatePrivate))]
#endif
		private static void on_is_account_linked(IntPtr creator, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string response)
		{
			AccountCreator thiz = fromNativePtr<AccountCreator>(creator);

			IntPtr ptr = linphone_account_creator_get_current_callbacks(thiz.nativePtr);
			AccountCreatorListener current_listener = fromNativePtr<AccountCreatorListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_is_account_linked_public?.Invoke(thiz, (Linphone.AccountCreatorStatus)status, response);
			}
		}

		public OnIsAccountLinkedDelegate OnIsAccountLinked
		{
			get
			{
				return on_is_account_linked_public;
			}
			set
			{
				on_is_account_linked_public = value;
				on_is_account_linked_private = on_is_account_linked;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_is_account_linked_private);
				linphone_account_creator_cbs_set_is_account_linked(nativePtr, cb);
#else
				linphone_account_creator_cbs_set_is_account_linked(nativePtr, on_is_account_linked_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_account_creator_cbs_set_recover_account(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_account_creator_cbs_set_recover_account(IntPtr thiz, OnRecoverAccountDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnRecoverAccountDelegatePrivate(IntPtr creator, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string response);

		/// <summary>
		/// Callback to notify a response of server. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="creator"><see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see> object    </param>
		/// <param name="status">The status of the <see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see> test existence operation that has just finished
		/// </param>
		/// <param name="response">The response has a string    </param>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public delegate void OnRecoverAccountDelegate(Linphone.AccountCreator creator, Linphone.AccountCreatorStatus status, string response);
		private OnRecoverAccountDelegatePrivate on_recover_account_private;
		private OnRecoverAccountDelegate on_recover_account_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnRecoverAccountDelegatePrivate))]
#endif
		private static void on_recover_account(IntPtr creator, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string response)
		{
			AccountCreator thiz = fromNativePtr<AccountCreator>(creator);

			IntPtr ptr = linphone_account_creator_get_current_callbacks(thiz.nativePtr);
			AccountCreatorListener current_listener = fromNativePtr<AccountCreatorListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_recover_account_public?.Invoke(thiz, (Linphone.AccountCreatorStatus)status, response);
			}
		}

		public OnRecoverAccountDelegate OnRecoverAccount
		{
			get
			{
				return on_recover_account_public;
			}
			set
			{
				on_recover_account_public = value;
				on_recover_account_private = on_recover_account;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_recover_account_private);
				linphone_account_creator_cbs_set_recover_account(nativePtr, cb);
#else
				linphone_account_creator_cbs_set_recover_account(nativePtr, on_recover_account_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_account_creator_cbs_set_update_account(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_account_creator_cbs_set_update_account(IntPtr thiz, OnUpdateAccountDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnUpdateAccountDelegatePrivate(IntPtr creator, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string response);

		/// <summary>
		/// Callback to notify a response of server. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="creator"><see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see> object    </param>
		/// <param name="status">The status of the <see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see> test existence operation that has just finished
		/// </param>
		/// <param name="response">The response has a string    </param>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public delegate void OnUpdateAccountDelegate(Linphone.AccountCreator creator, Linphone.AccountCreatorStatus status, string response);
		private OnUpdateAccountDelegatePrivate on_update_account_private;
		private OnUpdateAccountDelegate on_update_account_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnUpdateAccountDelegatePrivate))]
#endif
		private static void on_update_account(IntPtr creator, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string response)
		{
			AccountCreator thiz = fromNativePtr<AccountCreator>(creator);

			IntPtr ptr = linphone_account_creator_get_current_callbacks(thiz.nativePtr);
			AccountCreatorListener current_listener = fromNativePtr<AccountCreatorListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_update_account_public?.Invoke(thiz, (Linphone.AccountCreatorStatus)status, response);
			}
		}

		public OnUpdateAccountDelegate OnUpdateAccount
		{
			get
			{
				return on_update_account_public;
			}
			set
			{
				on_update_account_public = value;
				on_update_account_private = on_update_account;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_update_account_private);
				linphone_account_creator_cbs_set_update_account(nativePtr, cb);
#else
				linphone_account_creator_cbs_set_update_account(nativePtr, on_update_account_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_account_creator_cbs_set_login_linphone_account(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_account_creator_cbs_set_login_linphone_account(IntPtr thiz, OnLoginLinphoneAccountDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnLoginLinphoneAccountDelegatePrivate(IntPtr creator, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string response);

		/// <summary>
		/// Callback to notify a response of server. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="creator"><see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see> object    </param>
		/// <param name="status">The status of the <see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see> test existence operation that has just finished
		/// </param>
		/// <param name="response">The response has a string    </param>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead  : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public delegate void OnLoginLinphoneAccountDelegate(Linphone.AccountCreator creator, Linphone.AccountCreatorStatus status, string response);
		private OnLoginLinphoneAccountDelegatePrivate on_login_linphone_account_private;
		private OnLoginLinphoneAccountDelegate on_login_linphone_account_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnLoginLinphoneAccountDelegatePrivate))]
#endif
		private static void on_login_linphone_account(IntPtr creator, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string response)
		{
			AccountCreator thiz = fromNativePtr<AccountCreator>(creator);

			IntPtr ptr = linphone_account_creator_get_current_callbacks(thiz.nativePtr);
			AccountCreatorListener current_listener = fromNativePtr<AccountCreatorListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_login_linphone_account_public?.Invoke(thiz, (Linphone.AccountCreatorStatus)status, response);
			}
		}

		public OnLoginLinphoneAccountDelegate OnLoginLinphoneAccount
		{
			get
			{
				return on_login_linphone_account_public;
			}
			set
			{
				on_login_linphone_account_public = value;
				on_login_linphone_account_private = on_login_linphone_account;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_login_linphone_account_private);
				linphone_account_creator_cbs_set_login_linphone_account(nativePtr, cb);
#else
				linphone_account_creator_cbs_set_login_linphone_account(nativePtr, on_login_linphone_account_private);
#endif
			}
		}
	}

	/// <summary>
	/// An object to handle the callbacks for <see cref="Linphone.AccountManagerServicesRequest">
	/// Linphone.AccountManagerServicesRequest</see> object. 
	/// <para>
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class AccountManagerServicesRequestListener : LinphoneObject
	{
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_manager_services_request_get_current_callbacks(IntPtr thiz);

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_account_manager_services_request_cbs_set_request_successful(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_account_manager_services_request_cbs_set_request_successful(IntPtr thiz, OnRequestSuccessfulDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnRequestSuccessfulDelegatePrivate(IntPtr request, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string data);

		/// <summary>
		/// Callback for notifying a request was successful. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="request"><see cref="Linphone.AccountManagerServicesRequest">
		/// Linphone.AccountManagerServicesRequest</see> object.    </param>
		/// <param name="data">any relevant data depending on the request (for example SIP
		/// identity for account creation).    </param>
		public delegate void OnRequestSuccessfulDelegate(Linphone.AccountManagerServicesRequest request, string data);
		private OnRequestSuccessfulDelegatePrivate on_request_successful_private;
		private OnRequestSuccessfulDelegate on_request_successful_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnRequestSuccessfulDelegatePrivate))]
#endif
		private static void on_request_successful(IntPtr request, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string data)
		{
			AccountManagerServicesRequest thiz = fromNativePtr<AccountManagerServicesRequest>(request);

			IntPtr ptr = linphone_account_manager_services_request_get_current_callbacks(thiz.nativePtr);
			AccountManagerServicesRequestListener current_listener = fromNativePtr<AccountManagerServicesRequestListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_request_successful_public?.Invoke(thiz, data);
			}
		}

		public OnRequestSuccessfulDelegate OnRequestSuccessful
		{
			get
			{
				return on_request_successful_public;
			}
			set
			{
				on_request_successful_public = value;
				on_request_successful_private = on_request_successful;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_request_successful_private);
				linphone_account_manager_services_request_cbs_set_request_successful(nativePtr, cb);
#else
				linphone_account_manager_services_request_cbs_set_request_successful(nativePtr, on_request_successful_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_account_manager_services_request_cbs_set_request_error(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_account_manager_services_request_cbs_set_request_error(IntPtr thiz, OnRequestErrorDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnRequestErrorDelegatePrivate(IntPtr request, int statusCode, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string errorMessage, IntPtr parameterErrors);

		/// <summary>
		/// Callback for notifying a request has failed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="request"><see cref="Linphone.AccountManagerServicesRequest">
		/// Linphone.AccountManagerServicesRequest</see> object.    </param>
		/// <param name="statusCode">The error status code. </param>
		/// <param name="errorMessage">An error message, if any.    </param>
		/// <param name="parameterErrors">A <see cref="Linphone.Dictionary">
		/// Linphone.Dictionary</see> with parameter specific errors, if any.    </param>
		public delegate void OnRequestErrorDelegate(Linphone.AccountManagerServicesRequest request, int statusCode, string errorMessage, Linphone.Dictionary parameterErrors);
		private OnRequestErrorDelegatePrivate on_request_error_private;
		private OnRequestErrorDelegate on_request_error_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnRequestErrorDelegatePrivate))]
#endif
		private static void on_request_error(IntPtr request, int statusCode, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string errorMessage, IntPtr parameterErrors)
		{
			AccountManagerServicesRequest thiz = fromNativePtr<AccountManagerServicesRequest>(request);

			IntPtr ptr = linphone_account_manager_services_request_get_current_callbacks(thiz.nativePtr);
			AccountManagerServicesRequestListener current_listener = fromNativePtr<AccountManagerServicesRequestListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_request_error_public?.Invoke(thiz, statusCode, errorMessage, fromNativePtr<Linphone.Dictionary>(parameterErrors));
			}
		}

		public OnRequestErrorDelegate OnRequestError
		{
			get
			{
				return on_request_error_public;
			}
			set
			{
				on_request_error_public = value;
				on_request_error_private = on_request_error;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_request_error_private);
				linphone_account_manager_services_request_cbs_set_request_error(nativePtr, cb);
#else
				linphone_account_manager_services_request_cbs_set_request_error(nativePtr, on_request_error_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_account_manager_services_request_cbs_set_devices_list_fetched(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_account_manager_services_request_cbs_set_devices_list_fetched(IntPtr thiz, OnDevicesListFetchedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnDevicesListFetchedDelegatePrivate(IntPtr request, IntPtr devicesList);

		/// <summary>
		/// Callback for notifying when the
		/// #LinphoneAccountManagerServicesRequestGetDevicesList request has results
		/// available. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="request"><see cref="Linphone.AccountManagerServicesRequest">
		/// Linphone.AccountManagerServicesRequest</see> object.    </param>
		/// <param name="devicesList">the   of fetched devices.    </param>
		public delegate void OnDevicesListFetchedDelegate(Linphone.AccountManagerServicesRequest request, IEnumerable<Linphone.AccountDevice> devicesList);
		private OnDevicesListFetchedDelegatePrivate on_devices_list_fetched_private;
		private OnDevicesListFetchedDelegate on_devices_list_fetched_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnDevicesListFetchedDelegatePrivate))]
#endif
		private static void on_devices_list_fetched(IntPtr request, IntPtr devicesList)
		{
			AccountManagerServicesRequest thiz = fromNativePtr<AccountManagerServicesRequest>(request);

			IntPtr ptr = linphone_account_manager_services_request_get_current_callbacks(thiz.nativePtr);
			AccountManagerServicesRequestListener current_listener = fromNativePtr<AccountManagerServicesRequestListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_devices_list_fetched_public?.Invoke(thiz, MarshalBctbxList<Linphone.AccountDevice>(devicesList));
			}
		}

		public OnDevicesListFetchedDelegate OnDevicesListFetched
		{
			get
			{
				return on_devices_list_fetched_public;
			}
			set
			{
				on_devices_list_fetched_public = value;
				on_devices_list_fetched_private = on_devices_list_fetched;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_devices_list_fetched_private);
				linphone_account_manager_services_request_cbs_set_devices_list_fetched(nativePtr, cb);
#else
				linphone_account_manager_services_request_cbs_set_devices_list_fetched(nativePtr, on_devices_list_fetched_private);
#endif
			}
		}
	}

	/// <summary>
	/// Object that represents a callback attached to an alert. 
	/// <para>
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class AlertListener : LinphoneObject
	{
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_alert_get_current_callbacks(IntPtr thiz);

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_alert_cbs_set_terminated(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_alert_cbs_set_terminated(IntPtr thiz, OnTerminatedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnTerminatedDelegatePrivate(IntPtr alert);

		/// <summary>
		/// Callback to know if an alert stops. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="alert">the alert that stops    </param>
		public delegate void OnTerminatedDelegate(Linphone.Alert alert);
		private OnTerminatedDelegatePrivate on_terminated_private;
		private OnTerminatedDelegate on_terminated_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnTerminatedDelegatePrivate))]
#endif
		private static void on_terminated(IntPtr alert)
		{
			Alert thiz = fromNativePtr<Alert>(alert);

			IntPtr ptr = linphone_alert_get_current_callbacks(thiz.nativePtr);
			AlertListener current_listener = fromNativePtr<AlertListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_terminated_public?.Invoke(thiz);
			}
		}

		public OnTerminatedDelegate OnTerminated
		{
			get
			{
				return on_terminated_public;
			}
			set
			{
				on_terminated_public = value;
				on_terminated_private = on_terminated;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_terminated_private);
				linphone_alert_cbs_set_terminated(nativePtr, cb);
#else
				linphone_alert_cbs_set_terminated(nativePtr, on_terminated_private);
#endif
			}
		}
	}

	/// <summary>
	/// That class holds all the callbacks which are called by <see cref="Linphone.Call">
	/// Linphone.Call</see> objects. 
	/// <para>
	/// Use <see cref="Linphone.Factory.CreateCallCbs()">
	/// Linphone.Factory.CreateCallCbs()</see> to create an instance. Then, call the
	/// callback setters on the events you need to monitor and pass the object to a
	/// <see cref="Linphone.Call">Linphone.Call</see> instance through <see cref="Linphone.Call.AddListener()">
	/// Linphone.Call.AddListener()</see>. 
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class CallListener : LinphoneObject
	{
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_current_callbacks(IntPtr thiz);

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_call_cbs_set_dtmf_received(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_call_cbs_set_dtmf_received(IntPtr thiz, OnDtmfReceivedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnDtmfReceivedDelegatePrivate(IntPtr call, int dtmf);

		/// <summary>
		/// Callback for being notified of received DTMFs. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="call">LinphoneCall object that received the dtmf    </param>
		/// <param name="dtmf">The ascii code of the dtmf </param>
		public delegate void OnDtmfReceivedDelegate(Linphone.Call call, int dtmf);
		private OnDtmfReceivedDelegatePrivate on_dtmf_received_private;
		private OnDtmfReceivedDelegate on_dtmf_received_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnDtmfReceivedDelegatePrivate))]
#endif
		private static void on_dtmf_received(IntPtr call, int dtmf)
		{
			Call thiz = fromNativePtr<Call>(call);

			IntPtr ptr = linphone_call_get_current_callbacks(thiz.nativePtr);
			CallListener current_listener = fromNativePtr<CallListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_dtmf_received_public?.Invoke(thiz, dtmf);
			}
		}

		public OnDtmfReceivedDelegate OnDtmfReceived
		{
			get
			{
				return on_dtmf_received_public;
			}
			set
			{
				on_dtmf_received_public = value;
				on_dtmf_received_private = on_dtmf_received;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_dtmf_received_private);
				linphone_call_cbs_set_dtmf_received(nativePtr, cb);
#else
				linphone_call_cbs_set_dtmf_received(nativePtr, on_dtmf_received_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_call_cbs_set_goclear_ack_sent(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_call_cbs_set_goclear_ack_sent(IntPtr thiz, OnGoclearAckSentDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnGoclearAckSentDelegatePrivate(IntPtr call);

		/// <summary>
		/// GoClear ACK sent callback. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="call">the <see cref="Linphone.Call">Linphone.Call</see> on which
		/// the GoClear ACK was sent.    </param>
		public delegate void OnGoclearAckSentDelegate(Linphone.Call call);
		private OnGoclearAckSentDelegatePrivate on_goclear_ack_sent_private;
		private OnGoclearAckSentDelegate on_goclear_ack_sent_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnGoclearAckSentDelegatePrivate))]
#endif
		private static void on_goclear_ack_sent(IntPtr call)
		{
			Call thiz = fromNativePtr<Call>(call);

			IntPtr ptr = linphone_call_get_current_callbacks(thiz.nativePtr);
			CallListener current_listener = fromNativePtr<CallListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_goclear_ack_sent_public?.Invoke(thiz);
			}
		}

		public OnGoclearAckSentDelegate OnGoclearAckSent
		{
			get
			{
				return on_goclear_ack_sent_public;
			}
			set
			{
				on_goclear_ack_sent_public = value;
				on_goclear_ack_sent_private = on_goclear_ack_sent;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_goclear_ack_sent_private);
				linphone_call_cbs_set_goclear_ack_sent(nativePtr, cb);
#else
				linphone_call_cbs_set_goclear_ack_sent(nativePtr, on_goclear_ack_sent_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_call_cbs_set_security_level_downgraded(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_call_cbs_set_security_level_downgraded(IntPtr thiz, OnSecurityLevelDowngradedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnSecurityLevelDowngradedDelegatePrivate(IntPtr call);

		/// <summary>
		/// Call security level downgraded callback. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="call"><see cref="Linphone.Call">Linphone.Call</see> object whose
		/// security level is downgraded.    </param>
		public delegate void OnSecurityLevelDowngradedDelegate(Linphone.Call call);
		private OnSecurityLevelDowngradedDelegatePrivate on_security_level_downgraded_private;
		private OnSecurityLevelDowngradedDelegate on_security_level_downgraded_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnSecurityLevelDowngradedDelegatePrivate))]
#endif
		private static void on_security_level_downgraded(IntPtr call)
		{
			Call thiz = fromNativePtr<Call>(call);

			IntPtr ptr = linphone_call_get_current_callbacks(thiz.nativePtr);
			CallListener current_listener = fromNativePtr<CallListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_security_level_downgraded_public?.Invoke(thiz);
			}
		}

		public OnSecurityLevelDowngradedDelegate OnSecurityLevelDowngraded
		{
			get
			{
				return on_security_level_downgraded_public;
			}
			set
			{
				on_security_level_downgraded_public = value;
				on_security_level_downgraded_private = on_security_level_downgraded;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_security_level_downgraded_private);
				linphone_call_cbs_set_security_level_downgraded(nativePtr, cb);
#else
				linphone_call_cbs_set_security_level_downgraded(nativePtr, on_security_level_downgraded_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_call_cbs_set_encryption_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_call_cbs_set_encryption_changed(IntPtr thiz, OnEncryptionChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnEncryptionChangedDelegatePrivate(IntPtr call, char on, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string authenticationToken);

		/// <summary>
		/// Call encryption changed callback. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="call"><see cref="Linphone.Call">Linphone.Call</see> object whose
		/// encryption is changed.    </param>
		/// <param name="on">Whether encryption is activated. </param>
		/// <param name="authenticationToken">An authentication_token, currently set for
		/// ZRTP kind of encryption only.    </param>
		public delegate void OnEncryptionChangedDelegate(Linphone.Call call, bool on, string authenticationToken);
		private OnEncryptionChangedDelegatePrivate on_encryption_changed_private;
		private OnEncryptionChangedDelegate on_encryption_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnEncryptionChangedDelegatePrivate))]
#endif
		private static void on_encryption_changed(IntPtr call, char on, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string authenticationToken)
		{
			Call thiz = fromNativePtr<Call>(call);

			IntPtr ptr = linphone_call_get_current_callbacks(thiz.nativePtr);
			CallListener current_listener = fromNativePtr<CallListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_encryption_changed_public?.Invoke(thiz, on == 0, authenticationToken);
			}
		}

		public OnEncryptionChangedDelegate OnEncryptionChanged
		{
			get
			{
				return on_encryption_changed_public;
			}
			set
			{
				on_encryption_changed_public = value;
				on_encryption_changed_private = on_encryption_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_encryption_changed_private);
				linphone_call_cbs_set_encryption_changed(nativePtr, cb);
#else
				linphone_call_cbs_set_encryption_changed(nativePtr, on_encryption_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_call_cbs_set_authentication_token_verified(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_call_cbs_set_authentication_token_verified(IntPtr thiz, OnAuthenticationTokenVerifiedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnAuthenticationTokenVerifiedDelegatePrivate(IntPtr call, char verified);

		/// <summary>
		/// Call authentication token verified callback. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="call"><see cref="Linphone.Call">Linphone.Call</see> object whose
		/// authentication is verified.    </param>
		/// <param name="verified">Whether encryption is verified. </param>
		public delegate void OnAuthenticationTokenVerifiedDelegate(Linphone.Call call, bool verified);
		private OnAuthenticationTokenVerifiedDelegatePrivate on_authentication_token_verified_private;
		private OnAuthenticationTokenVerifiedDelegate on_authentication_token_verified_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnAuthenticationTokenVerifiedDelegatePrivate))]
#endif
		private static void on_authentication_token_verified(IntPtr call, char verified)
		{
			Call thiz = fromNativePtr<Call>(call);

			IntPtr ptr = linphone_call_get_current_callbacks(thiz.nativePtr);
			CallListener current_listener = fromNativePtr<CallListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_authentication_token_verified_public?.Invoke(thiz, verified == 0);
			}
		}

		public OnAuthenticationTokenVerifiedDelegate OnAuthenticationTokenVerified
		{
			get
			{
				return on_authentication_token_verified_public;
			}
			set
			{
				on_authentication_token_verified_public = value;
				on_authentication_token_verified_private = on_authentication_token_verified;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_authentication_token_verified_private);
				linphone_call_cbs_set_authentication_token_verified(nativePtr, cb);
#else
				linphone_call_cbs_set_authentication_token_verified(nativePtr, on_authentication_token_verified_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_call_cbs_set_send_master_key_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_call_cbs_set_send_master_key_changed(IntPtr thiz, OnSendMasterKeyChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnSendMasterKeyChangedDelegatePrivate(IntPtr call, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string sendMasterKey);

		/// <summary>
		/// Call send master key changed callback. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="call"><see cref="Linphone.Call">Linphone.Call</see> object whose
		/// encryption is changed.    </param>
		/// <param name="sendMasterKey">The send master key of the SRTP session.    </param>
		public delegate void OnSendMasterKeyChangedDelegate(Linphone.Call call, string sendMasterKey);
		private OnSendMasterKeyChangedDelegatePrivate on_send_master_key_changed_private;
		private OnSendMasterKeyChangedDelegate on_send_master_key_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnSendMasterKeyChangedDelegatePrivate))]
#endif
		private static void on_send_master_key_changed(IntPtr call, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string sendMasterKey)
		{
			Call thiz = fromNativePtr<Call>(call);

			IntPtr ptr = linphone_call_get_current_callbacks(thiz.nativePtr);
			CallListener current_listener = fromNativePtr<CallListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_send_master_key_changed_public?.Invoke(thiz, sendMasterKey);
			}
		}

		public OnSendMasterKeyChangedDelegate OnSendMasterKeyChanged
		{
			get
			{
				return on_send_master_key_changed_public;
			}
			set
			{
				on_send_master_key_changed_public = value;
				on_send_master_key_changed_private = on_send_master_key_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_send_master_key_changed_private);
				linphone_call_cbs_set_send_master_key_changed(nativePtr, cb);
#else
				linphone_call_cbs_set_send_master_key_changed(nativePtr, on_send_master_key_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_call_cbs_set_receive_master_key_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_call_cbs_set_receive_master_key_changed(IntPtr thiz, OnReceiveMasterKeyChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnReceiveMasterKeyChangedDelegatePrivate(IntPtr call, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string receiveMasterKey);

		/// <summary>
		/// Call receive master key changed callback. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="call"><see cref="Linphone.Call">Linphone.Call</see> object whose
		/// encryption is changed.    </param>
		/// <param name="receiveMasterKey">The receive master key of the SRTP session.   
		/// </param>
		public delegate void OnReceiveMasterKeyChangedDelegate(Linphone.Call call, string receiveMasterKey);
		private OnReceiveMasterKeyChangedDelegatePrivate on_receive_master_key_changed_private;
		private OnReceiveMasterKeyChangedDelegate on_receive_master_key_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnReceiveMasterKeyChangedDelegatePrivate))]
#endif
		private static void on_receive_master_key_changed(IntPtr call, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string receiveMasterKey)
		{
			Call thiz = fromNativePtr<Call>(call);

			IntPtr ptr = linphone_call_get_current_callbacks(thiz.nativePtr);
			CallListener current_listener = fromNativePtr<CallListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_receive_master_key_changed_public?.Invoke(thiz, receiveMasterKey);
			}
		}

		public OnReceiveMasterKeyChangedDelegate OnReceiveMasterKeyChanged
		{
			get
			{
				return on_receive_master_key_changed_public;
			}
			set
			{
				on_receive_master_key_changed_public = value;
				on_receive_master_key_changed_private = on_receive_master_key_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_receive_master_key_changed_private);
				linphone_call_cbs_set_receive_master_key_changed(nativePtr, cb);
#else
				linphone_call_cbs_set_receive_master_key_changed(nativePtr, on_receive_master_key_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_call_cbs_set_info_message_received(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_call_cbs_set_info_message_received(IntPtr thiz, OnInfoMessageReceivedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnInfoMessageReceivedDelegatePrivate(IntPtr call, IntPtr message);

		/// <summary>
		/// Callback for receiving info messages. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="call"><see cref="Linphone.Call">Linphone.Call</see> whose info
		/// message belongs to.    </param>
		/// <param name="message"><see cref="Linphone.InfoMessage">
		/// Linphone.InfoMessage</see> object.    </param>
		public delegate void OnInfoMessageReceivedDelegate(Linphone.Call call, Linphone.InfoMessage message);
		private OnInfoMessageReceivedDelegatePrivate on_info_message_received_private;
		private OnInfoMessageReceivedDelegate on_info_message_received_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnInfoMessageReceivedDelegatePrivate))]
#endif
		private static void on_info_message_received(IntPtr call, IntPtr message)
		{
			Call thiz = fromNativePtr<Call>(call);

			IntPtr ptr = linphone_call_get_current_callbacks(thiz.nativePtr);
			CallListener current_listener = fromNativePtr<CallListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_info_message_received_public?.Invoke(thiz, fromNativePtr<Linphone.InfoMessage>(message));
			}
		}

		public OnInfoMessageReceivedDelegate OnInfoMessageReceived
		{
			get
			{
				return on_info_message_received_public;
			}
			set
			{
				on_info_message_received_public = value;
				on_info_message_received_private = on_info_message_received;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_info_message_received_private);
				linphone_call_cbs_set_info_message_received(nativePtr, cb);
#else
				linphone_call_cbs_set_info_message_received(nativePtr, on_info_message_received_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_call_cbs_set_state_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_call_cbs_set_state_changed(IntPtr thiz, OnStateChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnStateChangedDelegatePrivate(IntPtr call, int state, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string message);

		/// <summary>
		/// Call state notification callback. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="call"><see cref="Linphone.Call">Linphone.Call</see> whose state is
		/// changed.    </param>
		/// <param name="state">The new <see cref="Linphone.CallState">
		/// Linphone.CallState</see> of the call </param>
		/// <param name="message">An informational message about the state.    </param>
		public delegate void OnStateChangedDelegate(Linphone.Call call, Linphone.CallState state, string message);
		private OnStateChangedDelegatePrivate on_state_changed_private;
		private OnStateChangedDelegate on_state_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnStateChangedDelegatePrivate))]
#endif
		private static void on_state_changed(IntPtr call, int state, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string message)
		{
			Call thiz = fromNativePtr<Call>(call);

			IntPtr ptr = linphone_call_get_current_callbacks(thiz.nativePtr);
			CallListener current_listener = fromNativePtr<CallListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_state_changed_public?.Invoke(thiz, (Linphone.CallState)state, message);
			}
		}

		public OnStateChangedDelegate OnStateChanged
		{
			get
			{
				return on_state_changed_public;
			}
			set
			{
				on_state_changed_public = value;
				on_state_changed_private = on_state_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_state_changed_private);
				linphone_call_cbs_set_state_changed(nativePtr, cb);
#else
				linphone_call_cbs_set_state_changed(nativePtr, on_state_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_call_cbs_set_stats_updated(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_call_cbs_set_stats_updated(IntPtr thiz, OnStatsUpdatedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnStatsUpdatedDelegatePrivate(IntPtr call, IntPtr stats);

		/// <summary>
		/// Callback for receiving quality statistics for calls. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="call"><see cref="Linphone.Call">Linphone.Call</see> object whose
		/// statistics are notified    </param>
		/// <param name="stats"><see cref="Linphone.CallStats">Linphone.CallStats</see>
		/// object    </param>
		public delegate void OnStatsUpdatedDelegate(Linphone.Call call, Linphone.CallStats stats);
		private OnStatsUpdatedDelegatePrivate on_stats_updated_private;
		private OnStatsUpdatedDelegate on_stats_updated_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnStatsUpdatedDelegatePrivate))]
#endif
		private static void on_stats_updated(IntPtr call, IntPtr stats)
		{
			Call thiz = fromNativePtr<Call>(call);

			IntPtr ptr = linphone_call_get_current_callbacks(thiz.nativePtr);
			CallListener current_listener = fromNativePtr<CallListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_stats_updated_public?.Invoke(thiz, fromNativePtr<Linphone.CallStats>(stats));
			}
		}

		public OnStatsUpdatedDelegate OnStatsUpdated
		{
			get
			{
				return on_stats_updated_public;
			}
			set
			{
				on_stats_updated_public = value;
				on_stats_updated_private = on_stats_updated;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_stats_updated_private);
				linphone_call_cbs_set_stats_updated(nativePtr, cb);
#else
				linphone_call_cbs_set_stats_updated(nativePtr, on_stats_updated_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_call_cbs_set_transfer_state_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_call_cbs_set_transfer_state_changed(IntPtr thiz, OnTransferStateChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnTransferStateChangedDelegatePrivate(IntPtr call, int state);

		/// <summary>
		/// Callback for notifying progresses of transfers. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="call">LinphoneCall that was transferred    </param>
		/// <param name="state">The LinphoneCallState of the call to transfer target at the
		/// far end. </param>
		public delegate void OnTransferStateChangedDelegate(Linphone.Call call, Linphone.CallState state);
		private OnTransferStateChangedDelegatePrivate on_transfer_state_changed_private;
		private OnTransferStateChangedDelegate on_transfer_state_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnTransferStateChangedDelegatePrivate))]
#endif
		private static void on_transfer_state_changed(IntPtr call, int state)
		{
			Call thiz = fromNativePtr<Call>(call);

			IntPtr ptr = linphone_call_get_current_callbacks(thiz.nativePtr);
			CallListener current_listener = fromNativePtr<CallListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_transfer_state_changed_public?.Invoke(thiz, (Linphone.CallState)state);
			}
		}

		public OnTransferStateChangedDelegate OnTransferStateChanged
		{
			get
			{
				return on_transfer_state_changed_public;
			}
			set
			{
				on_transfer_state_changed_public = value;
				on_transfer_state_changed_private = on_transfer_state_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_transfer_state_changed_private);
				linphone_call_cbs_set_transfer_state_changed(nativePtr, cb);
#else
				linphone_call_cbs_set_transfer_state_changed(nativePtr, on_transfer_state_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_call_cbs_set_refer_requested(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_call_cbs_set_refer_requested(IntPtr thiz, OnReferRequestedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnReferRequestedDelegatePrivate(IntPtr call, IntPtr referTo);

		/// <summary>
		/// Callback for notifying when a call transfer (refer) is requested. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="call"><see cref="Linphone.Call">Linphone.Call</see> associated
		/// with the transfer request.    </param>
		/// <param name="referTo">The target <see cref="Linphone.Address">
		/// Linphone.Address</see> to which the call is being transferred.    </param>
		public delegate void OnReferRequestedDelegate(Linphone.Call call, Linphone.Address referTo);
		private OnReferRequestedDelegatePrivate on_refer_requested_private;
		private OnReferRequestedDelegate on_refer_requested_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnReferRequestedDelegatePrivate))]
#endif
		private static void on_refer_requested(IntPtr call, IntPtr referTo)
		{
			Call thiz = fromNativePtr<Call>(call);

			IntPtr ptr = linphone_call_get_current_callbacks(thiz.nativePtr);
			CallListener current_listener = fromNativePtr<CallListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_refer_requested_public?.Invoke(thiz, fromNativePtr<Linphone.Address>(referTo));
			}
		}

		public OnReferRequestedDelegate OnReferRequested
		{
			get
			{
				return on_refer_requested_public;
			}
			set
			{
				on_refer_requested_public = value;
				on_refer_requested_private = on_refer_requested;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_refer_requested_private);
				linphone_call_cbs_set_refer_requested(nativePtr, cb);
#else
				linphone_call_cbs_set_refer_requested(nativePtr, on_refer_requested_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_call_cbs_set_ack_processing(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_call_cbs_set_ack_processing(IntPtr thiz, OnAckProcessingDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnAckProcessingDelegatePrivate(IntPtr call, IntPtr ack, char isReceived);

		/// <summary>
		/// Callback for notifying the processing SIP ACK messages. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="call"><see cref="Linphone.Call">Linphone.Call</see> for which an
		/// ACK is being received or sent    </param>
		/// <param name="ack">the ACK <see cref="Linphone.Headers">Linphone.Headers</see>  
		///  </param>
		/// <param name="isReceived">if true this ACK is an incoming one, otherwise it is
		/// an ACK about to be sent. </param>
		public delegate void OnAckProcessingDelegate(Linphone.Call call, Linphone.Headers ack, bool isReceived);
		private OnAckProcessingDelegatePrivate on_ack_processing_private;
		private OnAckProcessingDelegate on_ack_processing_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnAckProcessingDelegatePrivate))]
#endif
		private static void on_ack_processing(IntPtr call, IntPtr ack, char isReceived)
		{
			Call thiz = fromNativePtr<Call>(call);

			IntPtr ptr = linphone_call_get_current_callbacks(thiz.nativePtr);
			CallListener current_listener = fromNativePtr<CallListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_ack_processing_public?.Invoke(thiz, fromNativePtr<Linphone.Headers>(ack), isReceived == 0);
			}
		}

		public OnAckProcessingDelegate OnAckProcessing
		{
			get
			{
				return on_ack_processing_public;
			}
			set
			{
				on_ack_processing_public = value;
				on_ack_processing_private = on_ack_processing;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_ack_processing_private);
				linphone_call_cbs_set_ack_processing(nativePtr, cb);
#else
				linphone_call_cbs_set_ack_processing(nativePtr, on_ack_processing_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_call_cbs_set_tmmbr_received(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_call_cbs_set_tmmbr_received(IntPtr thiz, OnTmmbrReceivedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnTmmbrReceivedDelegatePrivate(IntPtr call, int streamIndex, int tmmbr);

		/// <summary>
		/// Callback for notifying a received TMMBR. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="call">LinphoneCall for which the TMMBR has changed    </param>
		/// <param name="streamIndex">the index of the current stream </param>
		/// <param name="tmmbr">the value of the received TMMBR </param>
		public delegate void OnTmmbrReceivedDelegate(Linphone.Call call, int streamIndex, int tmmbr);
		private OnTmmbrReceivedDelegatePrivate on_tmmbr_received_private;
		private OnTmmbrReceivedDelegate on_tmmbr_received_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnTmmbrReceivedDelegatePrivate))]
#endif
		private static void on_tmmbr_received(IntPtr call, int streamIndex, int tmmbr)
		{
			Call thiz = fromNativePtr<Call>(call);

			IntPtr ptr = linphone_call_get_current_callbacks(thiz.nativePtr);
			CallListener current_listener = fromNativePtr<CallListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_tmmbr_received_public?.Invoke(thiz, streamIndex, tmmbr);
			}
		}

		public OnTmmbrReceivedDelegate OnTmmbrReceived
		{
			get
			{
				return on_tmmbr_received_public;
			}
			set
			{
				on_tmmbr_received_public = value;
				on_tmmbr_received_private = on_tmmbr_received;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_tmmbr_received_private);
				linphone_call_cbs_set_tmmbr_received(nativePtr, cb);
#else
				linphone_call_cbs_set_tmmbr_received(nativePtr, on_tmmbr_received_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_call_cbs_set_snapshot_taken(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_call_cbs_set_snapshot_taken(IntPtr thiz, OnSnapshotTakenDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnSnapshotTakenDelegatePrivate(IntPtr call, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string filePath);

		/// <summary>
		/// Callback for notifying a snapshot taken. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="call">LinphoneCall for which the snapshot was taken    </param>
		/// <param name="filePath">the name of the saved file    </param>
		public delegate void OnSnapshotTakenDelegate(Linphone.Call call, string filePath);
		private OnSnapshotTakenDelegatePrivate on_snapshot_taken_private;
		private OnSnapshotTakenDelegate on_snapshot_taken_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnSnapshotTakenDelegatePrivate))]
#endif
		private static void on_snapshot_taken(IntPtr call, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string filePath)
		{
			Call thiz = fromNativePtr<Call>(call);

			IntPtr ptr = linphone_call_get_current_callbacks(thiz.nativePtr);
			CallListener current_listener = fromNativePtr<CallListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_snapshot_taken_public?.Invoke(thiz, filePath);
			}
		}

		public OnSnapshotTakenDelegate OnSnapshotTaken
		{
			get
			{
				return on_snapshot_taken_public;
			}
			set
			{
				on_snapshot_taken_public = value;
				on_snapshot_taken_private = on_snapshot_taken;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_snapshot_taken_private);
				linphone_call_cbs_set_snapshot_taken(nativePtr, cb);
#else
				linphone_call_cbs_set_snapshot_taken(nativePtr, on_snapshot_taken_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_call_cbs_set_next_video_frame_decoded(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_call_cbs_set_next_video_frame_decoded(IntPtr thiz, OnNextVideoFrameDecodedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnNextVideoFrameDecodedDelegatePrivate(IntPtr call);

		/// <summary>
		/// Callback to notify a next video frame has been decoded. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="call">LinphoneCall for which the next video frame has been decoded
		///    </param>
		public delegate void OnNextVideoFrameDecodedDelegate(Linphone.Call call);
		private OnNextVideoFrameDecodedDelegatePrivate on_next_video_frame_decoded_private;
		private OnNextVideoFrameDecodedDelegate on_next_video_frame_decoded_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnNextVideoFrameDecodedDelegatePrivate))]
#endif
		private static void on_next_video_frame_decoded(IntPtr call)
		{
			Call thiz = fromNativePtr<Call>(call);

			IntPtr ptr = linphone_call_get_current_callbacks(thiz.nativePtr);
			CallListener current_listener = fromNativePtr<CallListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_next_video_frame_decoded_public?.Invoke(thiz);
			}
		}

		public OnNextVideoFrameDecodedDelegate OnNextVideoFrameDecoded
		{
			get
			{
				return on_next_video_frame_decoded_public;
			}
			set
			{
				on_next_video_frame_decoded_public = value;
				on_next_video_frame_decoded_private = on_next_video_frame_decoded;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_next_video_frame_decoded_private);
				linphone_call_cbs_set_next_video_frame_decoded(nativePtr, cb);
#else
				linphone_call_cbs_set_next_video_frame_decoded(nativePtr, on_next_video_frame_decoded_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_call_cbs_set_camera_not_working(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_call_cbs_set_camera_not_working(IntPtr thiz, OnCameraNotWorkingDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnCameraNotWorkingDelegatePrivate(IntPtr call, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string cameraName);

		/// <summary>
		/// Callback to notify that the camera is not working and has been changed to
		/// &quot;No Webcam&quot;. 
		/// <para>
		/// A camera is detected as mis-functionning as soon as it outputs no frames at all
		/// during a period of 5 seconds. This check is only performed on desktop
		/// platforms, in the purpose of notifying camera failures, for example if when a
		/// usb cable gets disconnected.
		/// </para>
		/// </summary>
		/// <param name="call">LinphoneCall for which the next video frame has been decoded
		///    </param>
		/// <param name="cameraName">the name of the non-working camera    </param>
		public delegate void OnCameraNotWorkingDelegate(Linphone.Call call, string cameraName);
		private OnCameraNotWorkingDelegatePrivate on_camera_not_working_private;
		private OnCameraNotWorkingDelegate on_camera_not_working_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnCameraNotWorkingDelegatePrivate))]
#endif
		private static void on_camera_not_working(IntPtr call, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string cameraName)
		{
			Call thiz = fromNativePtr<Call>(call);

			IntPtr ptr = linphone_call_get_current_callbacks(thiz.nativePtr);
			CallListener current_listener = fromNativePtr<CallListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_camera_not_working_public?.Invoke(thiz, cameraName);
			}
		}

		public OnCameraNotWorkingDelegate OnCameraNotWorking
		{
			get
			{
				return on_camera_not_working_public;
			}
			set
			{
				on_camera_not_working_public = value;
				on_camera_not_working_private = on_camera_not_working;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_camera_not_working_private);
				linphone_call_cbs_set_camera_not_working(nativePtr, cb);
#else
				linphone_call_cbs_set_camera_not_working(nativePtr, on_camera_not_working_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_call_cbs_set_video_display_error_occurred(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_call_cbs_set_video_display_error_occurred(IntPtr thiz, OnVideoDisplayErrorOccurredDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnVideoDisplayErrorOccurredDelegatePrivate(IntPtr call, int errorCode);

		/// <summary>
		/// Callback to notify that there are errors from the video rendering. 
		/// <para>
		/// The error code depends of the implementation.
		/// </para>
		/// </summary>
		/// <param name="call">LinphoneCall    </param>
		/// <param name="errorCode">error code from render. It depends of the renderer.
		/// </param>
		public delegate void OnVideoDisplayErrorOccurredDelegate(Linphone.Call call, int errorCode);
		private OnVideoDisplayErrorOccurredDelegatePrivate on_video_display_error_occurred_private;
		private OnVideoDisplayErrorOccurredDelegate on_video_display_error_occurred_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnVideoDisplayErrorOccurredDelegatePrivate))]
#endif
		private static void on_video_display_error_occurred(IntPtr call, int errorCode)
		{
			Call thiz = fromNativePtr<Call>(call);

			IntPtr ptr = linphone_call_get_current_callbacks(thiz.nativePtr);
			CallListener current_listener = fromNativePtr<CallListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_video_display_error_occurred_public?.Invoke(thiz, errorCode);
			}
		}

		public OnVideoDisplayErrorOccurredDelegate OnVideoDisplayErrorOccurred
		{
			get
			{
				return on_video_display_error_occurred_public;
			}
			set
			{
				on_video_display_error_occurred_public = value;
				on_video_display_error_occurred_private = on_video_display_error_occurred;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_video_display_error_occurred_private);
				linphone_call_cbs_set_video_display_error_occurred(nativePtr, cb);
#else
				linphone_call_cbs_set_video_display_error_occurred(nativePtr, on_video_display_error_occurred_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_call_cbs_set_audio_device_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_call_cbs_set_audio_device_changed(IntPtr thiz, OnAudioDeviceChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnAudioDeviceChangedDelegatePrivate(IntPtr call, IntPtr audioDevice);

		/// <summary>
		/// Callback to notify that the audio device has been changed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="call">LinphoneCall for which the audio device has changed   
		/// </param>
		/// <param name="audioDevice">the new audio device used for this call    </param>
		public delegate void OnAudioDeviceChangedDelegate(Linphone.Call call, Linphone.AudioDevice audioDevice);
		private OnAudioDeviceChangedDelegatePrivate on_audio_device_changed_private;
		private OnAudioDeviceChangedDelegate on_audio_device_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnAudioDeviceChangedDelegatePrivate))]
#endif
		private static void on_audio_device_changed(IntPtr call, IntPtr audioDevice)
		{
			Call thiz = fromNativePtr<Call>(call);

			IntPtr ptr = linphone_call_get_current_callbacks(thiz.nativePtr);
			CallListener current_listener = fromNativePtr<CallListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_audio_device_changed_public?.Invoke(thiz, fromNativePtr<Linphone.AudioDevice>(audioDevice));
			}
		}

		public OnAudioDeviceChangedDelegate OnAudioDeviceChanged
		{
			get
			{
				return on_audio_device_changed_public;
			}
			set
			{
				on_audio_device_changed_public = value;
				on_audio_device_changed_private = on_audio_device_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_audio_device_changed_private);
				linphone_call_cbs_set_audio_device_changed(nativePtr, cb);
#else
				linphone_call_cbs_set_audio_device_changed(nativePtr, on_audio_device_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_call_cbs_set_remote_recording(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_call_cbs_set_remote_recording(IntPtr thiz, OnRemoteRecordingDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnRemoteRecordingDelegatePrivate(IntPtr call, char recording);

		/// <summary>
		/// Callback to notify that the call is being recorded by the remote. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="call">LinphoneCall for which the audio is recorded    </param>
		/// <param name="recording">true if the call is being recorded by the remote, false
		/// otherwise </param>
		public delegate void OnRemoteRecordingDelegate(Linphone.Call call, bool recording);
		private OnRemoteRecordingDelegatePrivate on_remote_recording_private;
		private OnRemoteRecordingDelegate on_remote_recording_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnRemoteRecordingDelegatePrivate))]
#endif
		private static void on_remote_recording(IntPtr call, char recording)
		{
			Call thiz = fromNativePtr<Call>(call);

			IntPtr ptr = linphone_call_get_current_callbacks(thiz.nativePtr);
			CallListener current_listener = fromNativePtr<CallListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_remote_recording_public?.Invoke(thiz, recording == 0);
			}
		}

		public OnRemoteRecordingDelegate OnRemoteRecording
		{
			get
			{
				return on_remote_recording_public;
			}
			set
			{
				on_remote_recording_public = value;
				on_remote_recording_private = on_remote_recording;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_remote_recording_private);
				linphone_call_cbs_set_remote_recording(nativePtr, cb);
#else
				linphone_call_cbs_set_remote_recording(nativePtr, on_remote_recording_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_call_cbs_set_baudot_detected(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_call_cbs_set_baudot_detected(IntPtr thiz, OnBaudotDetectedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnBaudotDetectedDelegatePrivate(IntPtr call, int standard);

		/// <summary>
		/// Callback to notify that Baudot tones have been detected in the audio received
		/// from the remote. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="call">LinphoneCall where Baudot tones have been detected   
		/// </param>
		/// <param name="standard">The Baudot standard of the detected tones. </param>
		public delegate void OnBaudotDetectedDelegate(Linphone.Call call, Linphone.BaudotStandard standard);
		private OnBaudotDetectedDelegatePrivate on_baudot_detected_private;
		private OnBaudotDetectedDelegate on_baudot_detected_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnBaudotDetectedDelegatePrivate))]
#endif
		private static void on_baudot_detected(IntPtr call, int standard)
		{
			Call thiz = fromNativePtr<Call>(call);

			IntPtr ptr = linphone_call_get_current_callbacks(thiz.nativePtr);
			CallListener current_listener = fromNativePtr<CallListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_baudot_detected_public?.Invoke(thiz, (Linphone.BaudotStandard)standard);
			}
		}

		public OnBaudotDetectedDelegate OnBaudotDetected
		{
			get
			{
				return on_baudot_detected_public;
			}
			set
			{
				on_baudot_detected_public = value;
				on_baudot_detected_private = on_baudot_detected;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_baudot_detected_private);
				linphone_call_cbs_set_baudot_detected(nativePtr, cb);
#else
				linphone_call_cbs_set_baudot_detected(nativePtr, on_baudot_detected_private);
#endif
			}
		}
	}

	/// <summary>
	/// An object to handle the callbacks for the handling a <see cref="Linphone.ChatMessage">
	/// Linphone.ChatMessage</see> objects. 
	/// <para>
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class ChatMessageListener : LinphoneObject
	{
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_chat_message_get_current_callbacks(IntPtr thiz);

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_message_cbs_set_msg_state_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_message_cbs_set_msg_state_changed(IntPtr thiz, OnMsgStateChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnMsgStateChangedDelegatePrivate(IntPtr message, int state);

		/// <summary>
		/// Call back used to notify message delivery status. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="message"><see cref="Linphone.ChatMessage">
		/// Linphone.ChatMessage</see> object    </param>
		/// <param name="state"><see cref="Linphone.ChatMessageState">
		/// Linphone.ChatMessageState</see> </param>
		public delegate void OnMsgStateChangedDelegate(Linphone.ChatMessage message, Linphone.ChatMessageState state);
		private OnMsgStateChangedDelegatePrivate on_msg_state_changed_private;
		private OnMsgStateChangedDelegate on_msg_state_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnMsgStateChangedDelegatePrivate))]
#endif
		private static void on_msg_state_changed(IntPtr message, int state)
		{
			ChatMessage thiz = fromNativePtr<ChatMessage>(message);

			IntPtr ptr = linphone_chat_message_get_current_callbacks(thiz.nativePtr);
			ChatMessageListener current_listener = fromNativePtr<ChatMessageListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_msg_state_changed_public?.Invoke(thiz, (Linphone.ChatMessageState)state);
			}
		}

		public OnMsgStateChangedDelegate OnMsgStateChanged
		{
			get
			{
				return on_msg_state_changed_public;
			}
			set
			{
				on_msg_state_changed_public = value;
				on_msg_state_changed_private = on_msg_state_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_msg_state_changed_private);
				linphone_chat_message_cbs_set_msg_state_changed(nativePtr, cb);
#else
				linphone_chat_message_cbs_set_msg_state_changed(nativePtr, on_msg_state_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_message_cbs_set_new_message_reaction(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_message_cbs_set_new_message_reaction(IntPtr thiz, OnNewMessageReactionDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnNewMessageReactionDelegatePrivate(IntPtr message, IntPtr reaction);

		/// <summary>
		/// Callback used to notify a reaction has been received or sent for a given
		/// message. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="message">LinphoneChatMessage object    </param>
		/// <param name="reaction">the LinphoneChatMessageReaction reaction that was sent
		/// or received    </param>
		public delegate void OnNewMessageReactionDelegate(Linphone.ChatMessage message, Linphone.ChatMessageReaction reaction);
		private OnNewMessageReactionDelegatePrivate on_new_message_reaction_private;
		private OnNewMessageReactionDelegate on_new_message_reaction_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnNewMessageReactionDelegatePrivate))]
#endif
		private static void on_new_message_reaction(IntPtr message, IntPtr reaction)
		{
			ChatMessage thiz = fromNativePtr<ChatMessage>(message);

			IntPtr ptr = linphone_chat_message_get_current_callbacks(thiz.nativePtr);
			ChatMessageListener current_listener = fromNativePtr<ChatMessageListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_new_message_reaction_public?.Invoke(thiz, fromNativePtr<Linphone.ChatMessageReaction>(reaction));
			}
		}

		public OnNewMessageReactionDelegate OnNewMessageReaction
		{
			get
			{
				return on_new_message_reaction_public;
			}
			set
			{
				on_new_message_reaction_public = value;
				on_new_message_reaction_private = on_new_message_reaction;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_new_message_reaction_private);
				linphone_chat_message_cbs_set_new_message_reaction(nativePtr, cb);
#else
				linphone_chat_message_cbs_set_new_message_reaction(nativePtr, on_new_message_reaction_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_message_cbs_set_reaction_removed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_message_cbs_set_reaction_removed(IntPtr thiz, OnReactionRemovedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnReactionRemovedDelegatePrivate(IntPtr message, IntPtr address);

		/// <summary>
		/// Callback used to notify a reaction has been removed from a given message. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="message">LinphoneChatMessage object    </param>
		/// <param name="address">the LinphoneAddress of the person that removed it&apos;s
		/// reaction    </param>
		public delegate void OnReactionRemovedDelegate(Linphone.ChatMessage message, Linphone.Address address);
		private OnReactionRemovedDelegatePrivate on_reaction_removed_private;
		private OnReactionRemovedDelegate on_reaction_removed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnReactionRemovedDelegatePrivate))]
#endif
		private static void on_reaction_removed(IntPtr message, IntPtr address)
		{
			ChatMessage thiz = fromNativePtr<ChatMessage>(message);

			IntPtr ptr = linphone_chat_message_get_current_callbacks(thiz.nativePtr);
			ChatMessageListener current_listener = fromNativePtr<ChatMessageListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_reaction_removed_public?.Invoke(thiz, fromNativePtr<Linphone.Address>(address));
			}
		}

		public OnReactionRemovedDelegate OnReactionRemoved
		{
			get
			{
				return on_reaction_removed_public;
			}
			set
			{
				on_reaction_removed_public = value;
				on_reaction_removed_private = on_reaction_removed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_reaction_removed_private);
				linphone_chat_message_cbs_set_reaction_removed(nativePtr, cb);
#else
				linphone_chat_message_cbs_set_reaction_removed(nativePtr, on_reaction_removed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_message_cbs_set_file_transfer_terminated(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_message_cbs_set_file_transfer_terminated(IntPtr thiz, OnFileTransferTerminatedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnFileTransferTerminatedDelegatePrivate(IntPtr message, IntPtr content);

		/// <summary>
		/// File transfer terminated callback prototype. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="message"><see cref="Linphone.ChatMessage">
		/// Linphone.ChatMessage</see> message from which the body is received.    </param>
		/// <param name="content"><see cref="Linphone.Content">Linphone.Content</see>
		/// incoming content information    </param>
		public delegate void OnFileTransferTerminatedDelegate(Linphone.ChatMessage message, Linphone.Content content);
		private OnFileTransferTerminatedDelegatePrivate on_file_transfer_terminated_private;
		private OnFileTransferTerminatedDelegate on_file_transfer_terminated_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnFileTransferTerminatedDelegatePrivate))]
#endif
		private static void on_file_transfer_terminated(IntPtr message, IntPtr content)
		{
			ChatMessage thiz = fromNativePtr<ChatMessage>(message);

			IntPtr ptr = linphone_chat_message_get_current_callbacks(thiz.nativePtr);
			ChatMessageListener current_listener = fromNativePtr<ChatMessageListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_file_transfer_terminated_public?.Invoke(thiz, fromNativePtr<Linphone.Content>(content));
			}
		}

		public OnFileTransferTerminatedDelegate OnFileTransferTerminated
		{
			get
			{
				return on_file_transfer_terminated_public;
			}
			set
			{
				on_file_transfer_terminated_public = value;
				on_file_transfer_terminated_private = on_file_transfer_terminated;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_file_transfer_terminated_private);
				linphone_chat_message_cbs_set_file_transfer_terminated(nativePtr, cb);
#else
				linphone_chat_message_cbs_set_file_transfer_terminated(nativePtr, on_file_transfer_terminated_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_message_cbs_set_file_transfer_recv(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_message_cbs_set_file_transfer_recv(IntPtr thiz, OnFileTransferRecvDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnFileTransferRecvDelegatePrivate(IntPtr message, IntPtr content, IntPtr buffer);

		/// <summary>
		/// File transfer receive callback prototype. 
		/// <para>
		/// This function is called by the core upon an incoming File transfer is started.
		/// This function may be call several time for the same file in case of large file. 
		/// </para>
		/// </summary>
		/// <param name="message"><see cref="Linphone.ChatMessage">
		/// Linphone.ChatMessage</see> message from which the body is received.    </param>
		/// <param name="content"><see cref="Linphone.Content">Linphone.Content</see>
		/// incoming content information    </param>
		/// <param name="buffer"><see cref="Linphone.Buffer">Linphone.Buffer</see> holding
		/// the received data. Empty buffer means end of file.    </param>
		public delegate void OnFileTransferRecvDelegate(Linphone.ChatMessage message, Linphone.Content content, Linphone.Buffer buffer);
		private OnFileTransferRecvDelegatePrivate on_file_transfer_recv_private;
		private OnFileTransferRecvDelegate on_file_transfer_recv_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnFileTransferRecvDelegatePrivate))]
#endif
		private static void on_file_transfer_recv(IntPtr message, IntPtr content, IntPtr buffer)
		{
			ChatMessage thiz = fromNativePtr<ChatMessage>(message);

			IntPtr ptr = linphone_chat_message_get_current_callbacks(thiz.nativePtr);
			ChatMessageListener current_listener = fromNativePtr<ChatMessageListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_file_transfer_recv_public?.Invoke(thiz, fromNativePtr<Linphone.Content>(content), fromNativePtr<Linphone.Buffer>(buffer));
			}
		}

		public OnFileTransferRecvDelegate OnFileTransferRecv
		{
			get
			{
				return on_file_transfer_recv_public;
			}
			set
			{
				on_file_transfer_recv_public = value;
				on_file_transfer_recv_private = on_file_transfer_recv;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_file_transfer_recv_private);
				linphone_chat_message_cbs_set_file_transfer_recv(nativePtr, cb);
#else
				linphone_chat_message_cbs_set_file_transfer_recv(nativePtr, on_file_transfer_recv_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_message_cbs_set_file_transfer_send(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_message_cbs_set_file_transfer_send(IntPtr thiz, OnFileTransferSendDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnFileTransferSendDelegatePrivate(IntPtr message, IntPtr content, long offset, long size);

		/// <summary>
		/// File transfer send callback prototype. 
		/// <para>
		/// This function is called by the core when an outgoing file transfer is started.
		/// This function is called until size is set to 0. 
		/// </para>
		/// </summary>
		/// <param name="message"><see cref="Linphone.ChatMessage">
		/// Linphone.ChatMessage</see> message from which the body is received.    </param>
		/// <param name="content"><see cref="Linphone.Content">Linphone.Content</see>
		/// outgoing content    </param>
		/// <param name="offset">the offset in the file from where to get the data to be
		/// sent </param>
		/// <param name="size">the number of bytes expected by the framework </param>
		/// <returns>A <see cref="Linphone.Buffer">Linphone.Buffer</see> object holding the
		/// data written by the application. An empty buffer means end of file.   
		/// </returns>
		/// <remarks>Warning : The returned value isn&apos;t used, hence the deprecation!
		/// </remarks> 
		/// deprecated : 17/08/2020 Use LinphoneChatMessageCbsFileTransferSendChunkCb
		/// instead. 
		public delegate void OnFileTransferSendDelegate(Linphone.ChatMessage message, Linphone.Content content, long offset, long size);
		private OnFileTransferSendDelegatePrivate on_file_transfer_send_private;
		private OnFileTransferSendDelegate on_file_transfer_send_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnFileTransferSendDelegatePrivate))]
#endif
		private static void on_file_transfer_send(IntPtr message, IntPtr content, long offset, long size)
		{
			ChatMessage thiz = fromNativePtr<ChatMessage>(message);

			IntPtr ptr = linphone_chat_message_get_current_callbacks(thiz.nativePtr);
			ChatMessageListener current_listener = fromNativePtr<ChatMessageListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_file_transfer_send_public?.Invoke(thiz, fromNativePtr<Linphone.Content>(content), offset, size);
			}
		}

		public OnFileTransferSendDelegate OnFileTransferSend
		{
			get
			{
				return on_file_transfer_send_public;
			}
			set
			{
				on_file_transfer_send_public = value;
				on_file_transfer_send_private = on_file_transfer_send;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_file_transfer_send_private);
				linphone_chat_message_cbs_set_file_transfer_send(nativePtr, cb);
#else
				linphone_chat_message_cbs_set_file_transfer_send(nativePtr, on_file_transfer_send_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_message_cbs_set_file_transfer_send_chunk(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_message_cbs_set_file_transfer_send_chunk(IntPtr thiz, OnFileTransferSendChunkDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnFileTransferSendChunkDelegatePrivate(IntPtr message, IntPtr content, long offset, long size, IntPtr buffer);

		/// <summary>
		/// File transfer send callback prototype. 
		/// <para>
		/// This function is called by the core when an outgoing file transfer is started.
		/// This function is called until size is set to 0. 
		/// </para>
		/// </summary>
		/// <param name="message"><see cref="Linphone.ChatMessage">
		/// Linphone.ChatMessage</see> message from which the body is received.    </param>
		/// <param name="content"><see cref="Linphone.Content">Linphone.Content</see>
		/// outgoing content    </param>
		/// <param name="offset">the offset in the file from where to get the data to be
		/// sent </param>
		/// <param name="size">the number of bytes expected by the framework </param>
		/// <param name="buffer">A <see cref="Linphone.Buffer">Linphone.Buffer</see> to be
		/// filled. Leave it empty when end of file has been reached.    </param>
		public delegate void OnFileTransferSendChunkDelegate(Linphone.ChatMessage message, Linphone.Content content, long offset, long size, Linphone.Buffer buffer);
		private OnFileTransferSendChunkDelegatePrivate on_file_transfer_send_chunk_private;
		private OnFileTransferSendChunkDelegate on_file_transfer_send_chunk_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnFileTransferSendChunkDelegatePrivate))]
#endif
		private static void on_file_transfer_send_chunk(IntPtr message, IntPtr content, long offset, long size, IntPtr buffer)
		{
			ChatMessage thiz = fromNativePtr<ChatMessage>(message);

			IntPtr ptr = linphone_chat_message_get_current_callbacks(thiz.nativePtr);
			ChatMessageListener current_listener = fromNativePtr<ChatMessageListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_file_transfer_send_chunk_public?.Invoke(thiz, fromNativePtr<Linphone.Content>(content), offset, size, fromNativePtr<Linphone.Buffer>(buffer));
			}
		}

		public OnFileTransferSendChunkDelegate OnFileTransferSendChunk
		{
			get
			{
				return on_file_transfer_send_chunk_public;
			}
			set
			{
				on_file_transfer_send_chunk_public = value;
				on_file_transfer_send_chunk_private = on_file_transfer_send_chunk;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_file_transfer_send_chunk_private);
				linphone_chat_message_cbs_set_file_transfer_send_chunk(nativePtr, cb);
#else
				linphone_chat_message_cbs_set_file_transfer_send_chunk(nativePtr, on_file_transfer_send_chunk_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_message_cbs_set_file_transfer_progress_indication(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_message_cbs_set_file_transfer_progress_indication(IntPtr thiz, OnFileTransferProgressIndicationDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnFileTransferProgressIndicationDelegatePrivate(IntPtr message, IntPtr content, long offset, long total);

		/// <summary>
		/// File transfer progress indication callback prototype. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="message"><see cref="Linphone.ChatMessage">
		/// Linphone.ChatMessage</see> message from which the body is received.    </param>
		/// <param name="content"><see cref="Linphone.Content">Linphone.Content</see>
		/// incoming content information    </param>
		/// <param name="offset">The number of bytes sent/received since the beginning of
		/// the transfer. </param>
		/// <param name="total">The total number of bytes to be sent/received. </param>
		public delegate void OnFileTransferProgressIndicationDelegate(Linphone.ChatMessage message, Linphone.Content content, long offset, long total);
		private OnFileTransferProgressIndicationDelegatePrivate on_file_transfer_progress_indication_private;
		private OnFileTransferProgressIndicationDelegate on_file_transfer_progress_indication_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnFileTransferProgressIndicationDelegatePrivate))]
#endif
		private static void on_file_transfer_progress_indication(IntPtr message, IntPtr content, long offset, long total)
		{
			ChatMessage thiz = fromNativePtr<ChatMessage>(message);

			IntPtr ptr = linphone_chat_message_get_current_callbacks(thiz.nativePtr);
			ChatMessageListener current_listener = fromNativePtr<ChatMessageListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_file_transfer_progress_indication_public?.Invoke(thiz, fromNativePtr<Linphone.Content>(content), offset, total);
			}
		}

		public OnFileTransferProgressIndicationDelegate OnFileTransferProgressIndication
		{
			get
			{
				return on_file_transfer_progress_indication_public;
			}
			set
			{
				on_file_transfer_progress_indication_public = value;
				on_file_transfer_progress_indication_private = on_file_transfer_progress_indication;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_file_transfer_progress_indication_private);
				linphone_chat_message_cbs_set_file_transfer_progress_indication(nativePtr, cb);
#else
				linphone_chat_message_cbs_set_file_transfer_progress_indication(nativePtr, on_file_transfer_progress_indication_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_message_cbs_set_participant_imdn_state_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_message_cbs_set_participant_imdn_state_changed(IntPtr thiz, OnParticipantImdnStateChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnParticipantImdnStateChangedDelegatePrivate(IntPtr message, IntPtr state);

		/// <summary>
		/// Call back used to notify participant IMDN state. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="message"><see cref="Linphone.ChatMessage">
		/// Linphone.ChatMessage</see> object    </param>
		/// <param name="state"><see cref="Linphone.ParticipantImdnState">
		/// Linphone.ParticipantImdnState</see>    </param>
		public delegate void OnParticipantImdnStateChangedDelegate(Linphone.ChatMessage message, Linphone.ParticipantImdnState state);
		private OnParticipantImdnStateChangedDelegatePrivate on_participant_imdn_state_changed_private;
		private OnParticipantImdnStateChangedDelegate on_participant_imdn_state_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnParticipantImdnStateChangedDelegatePrivate))]
#endif
		private static void on_participant_imdn_state_changed(IntPtr message, IntPtr state)
		{
			ChatMessage thiz = fromNativePtr<ChatMessage>(message);

			IntPtr ptr = linphone_chat_message_get_current_callbacks(thiz.nativePtr);
			ChatMessageListener current_listener = fromNativePtr<ChatMessageListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_participant_imdn_state_changed_public?.Invoke(thiz, fromNativePtr<Linphone.ParticipantImdnState>(state));
			}
		}

		public OnParticipantImdnStateChangedDelegate OnParticipantImdnStateChanged
		{
			get
			{
				return on_participant_imdn_state_changed_public;
			}
			set
			{
				on_participant_imdn_state_changed_public = value;
				on_participant_imdn_state_changed_private = on_participant_imdn_state_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_participant_imdn_state_changed_private);
				linphone_chat_message_cbs_set_participant_imdn_state_changed(nativePtr, cb);
#else
				linphone_chat_message_cbs_set_participant_imdn_state_changed(nativePtr, on_participant_imdn_state_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_message_cbs_set_ephemeral_message_timer_started(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_message_cbs_set_ephemeral_message_timer_started(IntPtr thiz, OnEphemeralMessageTimerStartedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnEphemeralMessageTimerStartedDelegatePrivate(IntPtr message);

		/// <summary>
		/// Callback used to notify an ephemeral message that its lifespan before
		/// disappearing has started to decrease. 
		/// <para>
		/// This callback is called when the ephemeral message is read by the receiver. 
		/// </para>
		/// </summary>
		/// <param name="message">LinphoneChatMessage object    </param>
		public delegate void OnEphemeralMessageTimerStartedDelegate(Linphone.ChatMessage message);
		private OnEphemeralMessageTimerStartedDelegatePrivate on_ephemeral_message_timer_started_private;
		private OnEphemeralMessageTimerStartedDelegate on_ephemeral_message_timer_started_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnEphemeralMessageTimerStartedDelegatePrivate))]
#endif
		private static void on_ephemeral_message_timer_started(IntPtr message)
		{
			ChatMessage thiz = fromNativePtr<ChatMessage>(message);

			IntPtr ptr = linphone_chat_message_get_current_callbacks(thiz.nativePtr);
			ChatMessageListener current_listener = fromNativePtr<ChatMessageListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_ephemeral_message_timer_started_public?.Invoke(thiz);
			}
		}

		public OnEphemeralMessageTimerStartedDelegate OnEphemeralMessageTimerStarted
		{
			get
			{
				return on_ephemeral_message_timer_started_public;
			}
			set
			{
				on_ephemeral_message_timer_started_public = value;
				on_ephemeral_message_timer_started_private = on_ephemeral_message_timer_started;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_ephemeral_message_timer_started_private);
				linphone_chat_message_cbs_set_ephemeral_message_timer_started(nativePtr, cb);
#else
				linphone_chat_message_cbs_set_ephemeral_message_timer_started(nativePtr, on_ephemeral_message_timer_started_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_message_cbs_set_ephemeral_message_deleted(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_message_cbs_set_ephemeral_message_deleted(IntPtr thiz, OnEphemeralMessageDeletedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnEphemeralMessageDeletedDelegatePrivate(IntPtr message);

		/// <summary>
		/// Call back used to notify ephemeral message is deleted. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="message">LinphoneChatMessage object    </param>
		public delegate void OnEphemeralMessageDeletedDelegate(Linphone.ChatMessage message);
		private OnEphemeralMessageDeletedDelegatePrivate on_ephemeral_message_deleted_private;
		private OnEphemeralMessageDeletedDelegate on_ephemeral_message_deleted_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnEphemeralMessageDeletedDelegatePrivate))]
#endif
		private static void on_ephemeral_message_deleted(IntPtr message)
		{
			ChatMessage thiz = fromNativePtr<ChatMessage>(message);

			IntPtr ptr = linphone_chat_message_get_current_callbacks(thiz.nativePtr);
			ChatMessageListener current_listener = fromNativePtr<ChatMessageListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_ephemeral_message_deleted_public?.Invoke(thiz);
			}
		}

		public OnEphemeralMessageDeletedDelegate OnEphemeralMessageDeleted
		{
			get
			{
				return on_ephemeral_message_deleted_public;
			}
			set
			{
				on_ephemeral_message_deleted_public = value;
				on_ephemeral_message_deleted_private = on_ephemeral_message_deleted;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_ephemeral_message_deleted_private);
				linphone_chat_message_cbs_set_ephemeral_message_deleted(nativePtr, cb);
#else
				linphone_chat_message_cbs_set_ephemeral_message_deleted(nativePtr, on_ephemeral_message_deleted_private);
#endif
			}
		}
	}

	/// <summary>
	/// An object to handle the callbacks for the handling a <see cref="Linphone.ChatRoom">
	/// Linphone.ChatRoom</see> objects. 
	/// <para>
	/// Use <see cref="Linphone.Factory.CreateChatRoomCbs()">
	/// Linphone.Factory.CreateChatRoomCbs()</see> to create an instance. Then pass the
	/// object to a <see cref="Linphone.ChatRoom">Linphone.ChatRoom</see> instance
	/// through <see cref="Linphone.ChatRoom.AddListener()">
	/// Linphone.ChatRoom.AddListener()</see>. 
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class ChatRoomListener : LinphoneObject
	{
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_chat_room_get_current_callbacks(IntPtr thiz);

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_is_composing_received(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_is_composing_received(IntPtr thiz, OnIsComposingReceivedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnIsComposingReceivedDelegatePrivate(IntPtr chatRoom, IntPtr remoteAddress, char isComposing);

		/// <summary>
		/// Is composing notification callback prototype. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="chatRoom">LinphoneChatRoom involved in the conversation    </param>
		/// <param name="remoteAddress">The LinphoneAddress that has sent the is-composing
		/// notification    </param>
		/// <param name="isComposing">A boolean value telling whether the remote is
		/// composing or not </param>
		public delegate void OnIsComposingReceivedDelegate(Linphone.ChatRoom chatRoom, Linphone.Address remoteAddress, bool isComposing);
		private OnIsComposingReceivedDelegatePrivate on_is_composing_received_private;
		private OnIsComposingReceivedDelegate on_is_composing_received_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnIsComposingReceivedDelegatePrivate))]
#endif
		private static void on_is_composing_received(IntPtr chatRoom, IntPtr remoteAddress, char isComposing)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_is_composing_received_public?.Invoke(thiz, fromNativePtr<Linphone.Address>(remoteAddress), isComposing == 0);
			}
		}

		public OnIsComposingReceivedDelegate OnIsComposingReceived
		{
			get
			{
				return on_is_composing_received_public;
			}
			set
			{
				on_is_composing_received_public = value;
				on_is_composing_received_private = on_is_composing_received;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_is_composing_received_private);
				linphone_chat_room_cbs_set_is_composing_received(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_is_composing_received(nativePtr, on_is_composing_received_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_message_received(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_message_received(IntPtr thiz, OnMessageReceivedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnMessageReceivedDelegatePrivate(IntPtr chatRoom, IntPtr message);

		/// <summary>
		/// Callback used to notify a chat room that a message has been received. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="chatRoom">LinphoneChatRoom object    </param>
		/// <param name="message">The LinphoneChatMessage that has been received    </param>
		public delegate void OnMessageReceivedDelegate(Linphone.ChatRoom chatRoom, Linphone.ChatMessage message);
		private OnMessageReceivedDelegatePrivate on_message_received_private;
		private OnMessageReceivedDelegate on_message_received_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnMessageReceivedDelegatePrivate))]
#endif
		private static void on_message_received(IntPtr chatRoom, IntPtr message)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_message_received_public?.Invoke(thiz, fromNativePtr<Linphone.ChatMessage>(message));
			}
		}

		public OnMessageReceivedDelegate OnMessageReceived
		{
			get
			{
				return on_message_received_public;
			}
			set
			{
				on_message_received_public = value;
				on_message_received_private = on_message_received;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_message_received_private);
				linphone_chat_room_cbs_set_message_received(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_message_received(nativePtr, on_message_received_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_messages_received(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_messages_received(IntPtr thiz, OnMessagesReceivedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnMessagesReceivedDelegatePrivate(IntPtr chatRoom, IntPtr chatMessages);

		/// <summary>
		/// Callback used to notify a chat room that many chat messages have been received. 
		/// <para>
		/// Only called when aggregation is enabled (aka [sip] chat_messages_aggregation ==
		/// 1 or using linphone_core_set_chat_messages_aggregation_enabled), it replaces
		/// the single message received callback. 
		/// </para>
		/// </summary>
		/// <param name="chatRoom">LinphoneChatRoom object    </param>
		/// <param name="chatMessages">The   list of events to be notified    </param>
		public delegate void OnMessagesReceivedDelegate(Linphone.ChatRoom chatRoom, IEnumerable<Linphone.ChatMessage> chatMessages);
		private OnMessagesReceivedDelegatePrivate on_messages_received_private;
		private OnMessagesReceivedDelegate on_messages_received_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnMessagesReceivedDelegatePrivate))]
#endif
		private static void on_messages_received(IntPtr chatRoom, IntPtr chatMessages)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_messages_received_public?.Invoke(thiz, MarshalBctbxList<Linphone.ChatMessage>(chatMessages));
			}
		}

		public OnMessagesReceivedDelegate OnMessagesReceived
		{
			get
			{
				return on_messages_received_public;
			}
			set
			{
				on_messages_received_public = value;
				on_messages_received_private = on_messages_received;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_messages_received_private);
				linphone_chat_room_cbs_set_messages_received(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_messages_received(nativePtr, on_messages_received_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_new_event(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_new_event(IntPtr thiz, OnNewEventDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnNewEventDelegatePrivate(IntPtr chatRoom, IntPtr eventLog);

		/// <summary>
		/// Callback used to notify a chat room that an event log has been created. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// object    </param>
		/// <param name="eventLog"><see cref="Linphone.EventLog">Linphone.EventLog</see>
		/// The event to be notified    </param>
		public delegate void OnNewEventDelegate(Linphone.ChatRoom chatRoom, Linphone.EventLog eventLog);
		private OnNewEventDelegatePrivate on_new_event_private;
		private OnNewEventDelegate on_new_event_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnNewEventDelegatePrivate))]
#endif
		private static void on_new_event(IntPtr chatRoom, IntPtr eventLog)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_new_event_public?.Invoke(thiz, fromNativePtr<Linphone.EventLog>(eventLog));
			}
		}

		public OnNewEventDelegate OnNewEvent
		{
			get
			{
				return on_new_event_public;
			}
			set
			{
				on_new_event_public = value;
				on_new_event_private = on_new_event;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_new_event_private);
				linphone_chat_room_cbs_set_new_event(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_new_event(nativePtr, on_new_event_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_new_events(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_new_events(IntPtr thiz, OnNewEventsDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnNewEventsDelegatePrivate(IntPtr chatRoom, IntPtr eventLogs);

		/// <summary>
		/// Callback used to notify a chat room that many event logs have been created. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// object    </param>
		/// <param name="eventLogs">The   list of events to be notified    </param>
		public delegate void OnNewEventsDelegate(Linphone.ChatRoom chatRoom, IEnumerable<Linphone.EventLog> eventLogs);
		private OnNewEventsDelegatePrivate on_new_events_private;
		private OnNewEventsDelegate on_new_events_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnNewEventsDelegatePrivate))]
#endif
		private static void on_new_events(IntPtr chatRoom, IntPtr eventLogs)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_new_events_public?.Invoke(thiz, MarshalBctbxList<Linphone.EventLog>(eventLogs));
			}
		}

		public OnNewEventsDelegate OnNewEvents
		{
			get
			{
				return on_new_events_public;
			}
			set
			{
				on_new_events_public = value;
				on_new_events_private = on_new_events;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_new_events_private);
				linphone_chat_room_cbs_set_new_events(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_new_events(nativePtr, on_new_events_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_chat_message_received(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_chat_message_received(IntPtr thiz, OnChatMessageReceivedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnChatMessageReceivedDelegatePrivate(IntPtr chatRoom, IntPtr eventLog);

		/// <summary>
		/// Callback used to notify a chat room that a chat message has been received. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// object    </param>
		/// <param name="eventLog"><see cref="Linphone.EventLog">Linphone.EventLog</see>
		/// The event to be notified    </param>
		public delegate void OnChatMessageReceivedDelegate(Linphone.ChatRoom chatRoom, Linphone.EventLog eventLog);
		private OnChatMessageReceivedDelegatePrivate on_chat_message_received_private;
		private OnChatMessageReceivedDelegate on_chat_message_received_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnChatMessageReceivedDelegatePrivate))]
#endif
		private static void on_chat_message_received(IntPtr chatRoom, IntPtr eventLog)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_chat_message_received_public?.Invoke(thiz, fromNativePtr<Linphone.EventLog>(eventLog));
			}
		}

		public OnChatMessageReceivedDelegate OnChatMessageReceived
		{
			get
			{
				return on_chat_message_received_public;
			}
			set
			{
				on_chat_message_received_public = value;
				on_chat_message_received_private = on_chat_message_received;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_chat_message_received_private);
				linphone_chat_room_cbs_set_chat_message_received(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_chat_message_received(nativePtr, on_chat_message_received_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_chat_messages_received(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_chat_messages_received(IntPtr thiz, OnChatMessagesReceivedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnChatMessagesReceivedDelegatePrivate(IntPtr chatRoom, IntPtr eventLogs);

		/// <summary>
		/// Callback used to notify a chat room that one or many chat messages have been
		/// received. 
		/// <para>
		/// Only called when aggregation is enabled (aka [sip] chat_messages_aggregation ==
		/// 1 or using <see cref="Linphone.Core.ChatMessagesAggregationEnabled">
		/// Linphone.Core.ChatMessagesAggregationEnabled</see>), it replaces the single
		/// chat message received callback. 
		/// </para>
		/// </summary>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// object    </param>
		/// <param name="eventLogs">The   list of events to be notified    </param>
		public delegate void OnChatMessagesReceivedDelegate(Linphone.ChatRoom chatRoom, IEnumerable<Linphone.EventLog> eventLogs);
		private OnChatMessagesReceivedDelegatePrivate on_chat_messages_received_private;
		private OnChatMessagesReceivedDelegate on_chat_messages_received_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnChatMessagesReceivedDelegatePrivate))]
#endif
		private static void on_chat_messages_received(IntPtr chatRoom, IntPtr eventLogs)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_chat_messages_received_public?.Invoke(thiz, MarshalBctbxList<Linphone.EventLog>(eventLogs));
			}
		}

		public OnChatMessagesReceivedDelegate OnChatMessagesReceived
		{
			get
			{
				return on_chat_messages_received_public;
			}
			set
			{
				on_chat_messages_received_public = value;
				on_chat_messages_received_private = on_chat_messages_received;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_chat_messages_received_private);
				linphone_chat_room_cbs_set_chat_messages_received(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_chat_messages_received(nativePtr, on_chat_messages_received_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_chat_message_sending(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_chat_message_sending(IntPtr thiz, OnChatMessageSendingDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnChatMessageSendingDelegatePrivate(IntPtr chatRoom, IntPtr eventLog);

		/// <summary>
		/// Callback used to notify a chat room that a chat message is being sent. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// object    </param>
		/// <param name="eventLog"><see cref="Linphone.EventLog">Linphone.EventLog</see>
		/// The event to be notified    </param>
		public delegate void OnChatMessageSendingDelegate(Linphone.ChatRoom chatRoom, Linphone.EventLog eventLog);
		private OnChatMessageSendingDelegatePrivate on_chat_message_sending_private;
		private OnChatMessageSendingDelegate on_chat_message_sending_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnChatMessageSendingDelegatePrivate))]
#endif
		private static void on_chat_message_sending(IntPtr chatRoom, IntPtr eventLog)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_chat_message_sending_public?.Invoke(thiz, fromNativePtr<Linphone.EventLog>(eventLog));
			}
		}

		public OnChatMessageSendingDelegate OnChatMessageSending
		{
			get
			{
				return on_chat_message_sending_public;
			}
			set
			{
				on_chat_message_sending_public = value;
				on_chat_message_sending_private = on_chat_message_sending;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_chat_message_sending_private);
				linphone_chat_room_cbs_set_chat_message_sending(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_chat_message_sending(nativePtr, on_chat_message_sending_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_chat_message_sent(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_chat_message_sent(IntPtr thiz, OnChatMessageSentDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnChatMessageSentDelegatePrivate(IntPtr chatRoom, IntPtr eventLog);

		/// <summary>
		/// Callback used to notify a chat room that a chat message has been sent. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// object    </param>
		/// <param name="eventLog"><see cref="Linphone.EventLog">Linphone.EventLog</see>
		/// The event to be notified    </param>
		public delegate void OnChatMessageSentDelegate(Linphone.ChatRoom chatRoom, Linphone.EventLog eventLog);
		private OnChatMessageSentDelegatePrivate on_chat_message_sent_private;
		private OnChatMessageSentDelegate on_chat_message_sent_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnChatMessageSentDelegatePrivate))]
#endif
		private static void on_chat_message_sent(IntPtr chatRoom, IntPtr eventLog)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_chat_message_sent_public?.Invoke(thiz, fromNativePtr<Linphone.EventLog>(eventLog));
			}
		}

		public OnChatMessageSentDelegate OnChatMessageSent
		{
			get
			{
				return on_chat_message_sent_public;
			}
			set
			{
				on_chat_message_sent_public = value;
				on_chat_message_sent_private = on_chat_message_sent;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_chat_message_sent_private);
				linphone_chat_room_cbs_set_chat_message_sent(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_chat_message_sent(nativePtr, on_chat_message_sent_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_participant_added(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_participant_added(IntPtr thiz, OnParticipantAddedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnParticipantAddedDelegatePrivate(IntPtr chatRoom, IntPtr eventLog);

		/// <summary>
		/// Callback used to notify a chat room that a participant has been added. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// object    </param>
		/// <param name="eventLog"><see cref="Linphone.EventLog">Linphone.EventLog</see>
		/// The event to be notified    </param>
		public delegate void OnParticipantAddedDelegate(Linphone.ChatRoom chatRoom, Linphone.EventLog eventLog);
		private OnParticipantAddedDelegatePrivate on_participant_added_private;
		private OnParticipantAddedDelegate on_participant_added_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnParticipantAddedDelegatePrivate))]
#endif
		private static void on_participant_added(IntPtr chatRoom, IntPtr eventLog)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_participant_added_public?.Invoke(thiz, fromNativePtr<Linphone.EventLog>(eventLog));
			}
		}

		public OnParticipantAddedDelegate OnParticipantAdded
		{
			get
			{
				return on_participant_added_public;
			}
			set
			{
				on_participant_added_public = value;
				on_participant_added_private = on_participant_added;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_participant_added_private);
				linphone_chat_room_cbs_set_participant_added(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_participant_added(nativePtr, on_participant_added_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_participant_removed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_participant_removed(IntPtr thiz, OnParticipantRemovedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnParticipantRemovedDelegatePrivate(IntPtr chatRoom, IntPtr eventLog);

		/// <summary>
		/// Callback used to notify a chat room that a participant has been removed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// object    </param>
		/// <param name="eventLog"><see cref="Linphone.EventLog">Linphone.EventLog</see>
		/// The event to be notified    </param>
		public delegate void OnParticipantRemovedDelegate(Linphone.ChatRoom chatRoom, Linphone.EventLog eventLog);
		private OnParticipantRemovedDelegatePrivate on_participant_removed_private;
		private OnParticipantRemovedDelegate on_participant_removed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnParticipantRemovedDelegatePrivate))]
#endif
		private static void on_participant_removed(IntPtr chatRoom, IntPtr eventLog)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_participant_removed_public?.Invoke(thiz, fromNativePtr<Linphone.EventLog>(eventLog));
			}
		}

		public OnParticipantRemovedDelegate OnParticipantRemoved
		{
			get
			{
				return on_participant_removed_public;
			}
			set
			{
				on_participant_removed_public = value;
				on_participant_removed_private = on_participant_removed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_participant_removed_private);
				linphone_chat_room_cbs_set_participant_removed(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_participant_removed(nativePtr, on_participant_removed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_participant_admin_status_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_participant_admin_status_changed(IntPtr thiz, OnParticipantAdminStatusChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnParticipantAdminStatusChangedDelegatePrivate(IntPtr chatRoom, IntPtr eventLog);

		/// <summary>
		/// Callback used to notify a chat room that the admin status of a participant has
		/// been changed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// object    </param>
		/// <param name="eventLog"><see cref="Linphone.EventLog">Linphone.EventLog</see>
		/// The event to be notified    </param>
		public delegate void OnParticipantAdminStatusChangedDelegate(Linphone.ChatRoom chatRoom, Linphone.EventLog eventLog);
		private OnParticipantAdminStatusChangedDelegatePrivate on_participant_admin_status_changed_private;
		private OnParticipantAdminStatusChangedDelegate on_participant_admin_status_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnParticipantAdminStatusChangedDelegatePrivate))]
#endif
		private static void on_participant_admin_status_changed(IntPtr chatRoom, IntPtr eventLog)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_participant_admin_status_changed_public?.Invoke(thiz, fromNativePtr<Linphone.EventLog>(eventLog));
			}
		}

		public OnParticipantAdminStatusChangedDelegate OnParticipantAdminStatusChanged
		{
			get
			{
				return on_participant_admin_status_changed_public;
			}
			set
			{
				on_participant_admin_status_changed_public = value;
				on_participant_admin_status_changed_private = on_participant_admin_status_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_participant_admin_status_changed_private);
				linphone_chat_room_cbs_set_participant_admin_status_changed(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_participant_admin_status_changed(nativePtr, on_participant_admin_status_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_state_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_state_changed(IntPtr thiz, OnStateChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnStateChangedDelegatePrivate(IntPtr chatRoom, int newState);

		/// <summary>
		/// Callback used to notify a chat room state has changed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="chatRoom">LinphoneChatRoom object    </param>
		/// <param name="newState">The new LinphoneChatRoomState of the chat room </param>
		public delegate void OnStateChangedDelegate(Linphone.ChatRoom chatRoom, Linphone.ChatRoomState newState);
		private OnStateChangedDelegatePrivate on_state_changed_private;
		private OnStateChangedDelegate on_state_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnStateChangedDelegatePrivate))]
#endif
		private static void on_state_changed(IntPtr chatRoom, int newState)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_state_changed_public?.Invoke(thiz, (Linphone.ChatRoomState)newState);
			}
		}

		public OnStateChangedDelegate OnStateChanged
		{
			get
			{
				return on_state_changed_public;
			}
			set
			{
				on_state_changed_public = value;
				on_state_changed_private = on_state_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_state_changed_private);
				linphone_chat_room_cbs_set_state_changed(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_state_changed(nativePtr, on_state_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_security_event(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_security_event(IntPtr thiz, OnSecurityEventDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnSecurityEventDelegatePrivate(IntPtr chatRoom, IntPtr eventLog);

		/// <summary>
		/// Callback used to notify a security event in the chat room. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// object    </param>
		/// <param name="eventLog"><see cref="Linphone.EventLog">Linphone.EventLog</see>
		/// The event to be notified    </param>
		public delegate void OnSecurityEventDelegate(Linphone.ChatRoom chatRoom, Linphone.EventLog eventLog);
		private OnSecurityEventDelegatePrivate on_security_event_private;
		private OnSecurityEventDelegate on_security_event_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnSecurityEventDelegatePrivate))]
#endif
		private static void on_security_event(IntPtr chatRoom, IntPtr eventLog)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_security_event_public?.Invoke(thiz, fromNativePtr<Linphone.EventLog>(eventLog));
			}
		}

		public OnSecurityEventDelegate OnSecurityEvent
		{
			get
			{
				return on_security_event_public;
			}
			set
			{
				on_security_event_public = value;
				on_security_event_private = on_security_event;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_security_event_private);
				linphone_chat_room_cbs_set_security_event(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_security_event(nativePtr, on_security_event_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_subject_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_subject_changed(IntPtr thiz, OnSubjectChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnSubjectChangedDelegatePrivate(IntPtr chatRoom, IntPtr eventLog);

		/// <summary>
		/// Callback used to notify that the subject of a chat room has changed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// object    </param>
		/// <param name="eventLog"><see cref="Linphone.EventLog">Linphone.EventLog</see>
		/// The event to be notified    </param>
		public delegate void OnSubjectChangedDelegate(Linphone.ChatRoom chatRoom, Linphone.EventLog eventLog);
		private OnSubjectChangedDelegatePrivate on_subject_changed_private;
		private OnSubjectChangedDelegate on_subject_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnSubjectChangedDelegatePrivate))]
#endif
		private static void on_subject_changed(IntPtr chatRoom, IntPtr eventLog)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_subject_changed_public?.Invoke(thiz, fromNativePtr<Linphone.EventLog>(eventLog));
			}
		}

		public OnSubjectChangedDelegate OnSubjectChanged
		{
			get
			{
				return on_subject_changed_public;
			}
			set
			{
				on_subject_changed_public = value;
				on_subject_changed_private = on_subject_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_subject_changed_private);
				linphone_chat_room_cbs_set_subject_changed(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_subject_changed(nativePtr, on_subject_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_undecryptable_message_received(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_undecryptable_message_received(IntPtr thiz, OnUndecryptableMessageReceivedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnUndecryptableMessageReceivedDelegatePrivate(IntPtr chatRoom, IntPtr message);

		/// <summary>
		/// Callback used to notify a chat room that a message has been received but we
		/// were unable to decrypt it. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// involved in this conversation    </param>
		/// <param name="message">The <see cref="Linphone.ChatMessage">
		/// Linphone.ChatMessage</see> that has been received    </param>
		public delegate void OnUndecryptableMessageReceivedDelegate(Linphone.ChatRoom chatRoom, Linphone.ChatMessage message);
		private OnUndecryptableMessageReceivedDelegatePrivate on_undecryptable_message_received_private;
		private OnUndecryptableMessageReceivedDelegate on_undecryptable_message_received_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnUndecryptableMessageReceivedDelegatePrivate))]
#endif
		private static void on_undecryptable_message_received(IntPtr chatRoom, IntPtr message)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_undecryptable_message_received_public?.Invoke(thiz, fromNativePtr<Linphone.ChatMessage>(message));
			}
		}

		public OnUndecryptableMessageReceivedDelegate OnUndecryptableMessageReceived
		{
			get
			{
				return on_undecryptable_message_received_public;
			}
			set
			{
				on_undecryptable_message_received_public = value;
				on_undecryptable_message_received_private = on_undecryptable_message_received;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_undecryptable_message_received_private);
				linphone_chat_room_cbs_set_undecryptable_message_received(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_undecryptable_message_received(nativePtr, on_undecryptable_message_received_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_participant_device_added(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_participant_device_added(IntPtr thiz, OnParticipantDeviceAddedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnParticipantDeviceAddedDelegatePrivate(IntPtr chatRoom, IntPtr eventLog);

		/// <summary>
		/// Callback used to notify a chat room that a participant has been added. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// object    </param>
		/// <param name="eventLog"><see cref="Linphone.EventLog">Linphone.EventLog</see>
		/// The event to be notified    </param>
		public delegate void OnParticipantDeviceAddedDelegate(Linphone.ChatRoom chatRoom, Linphone.EventLog eventLog);
		private OnParticipantDeviceAddedDelegatePrivate on_participant_device_added_private;
		private OnParticipantDeviceAddedDelegate on_participant_device_added_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnParticipantDeviceAddedDelegatePrivate))]
#endif
		private static void on_participant_device_added(IntPtr chatRoom, IntPtr eventLog)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_participant_device_added_public?.Invoke(thiz, fromNativePtr<Linphone.EventLog>(eventLog));
			}
		}

		public OnParticipantDeviceAddedDelegate OnParticipantDeviceAdded
		{
			get
			{
				return on_participant_device_added_public;
			}
			set
			{
				on_participant_device_added_public = value;
				on_participant_device_added_private = on_participant_device_added;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_participant_device_added_private);
				linphone_chat_room_cbs_set_participant_device_added(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_participant_device_added(nativePtr, on_participant_device_added_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_participant_device_removed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_participant_device_removed(IntPtr thiz, OnParticipantDeviceRemovedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnParticipantDeviceRemovedDelegatePrivate(IntPtr chatRoom, IntPtr eventLog);

		/// <summary>
		/// Callback used to notify a chat room that a participant has been removed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// object    </param>
		/// <param name="eventLog"><see cref="Linphone.EventLog">Linphone.EventLog</see>
		/// The event to be notified    </param>
		public delegate void OnParticipantDeviceRemovedDelegate(Linphone.ChatRoom chatRoom, Linphone.EventLog eventLog);
		private OnParticipantDeviceRemovedDelegatePrivate on_participant_device_removed_private;
		private OnParticipantDeviceRemovedDelegate on_participant_device_removed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnParticipantDeviceRemovedDelegatePrivate))]
#endif
		private static void on_participant_device_removed(IntPtr chatRoom, IntPtr eventLog)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_participant_device_removed_public?.Invoke(thiz, fromNativePtr<Linphone.EventLog>(eventLog));
			}
		}

		public OnParticipantDeviceRemovedDelegate OnParticipantDeviceRemoved
		{
			get
			{
				return on_participant_device_removed_public;
			}
			set
			{
				on_participant_device_removed_public = value;
				on_participant_device_removed_private = on_participant_device_removed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_participant_device_removed_private);
				linphone_chat_room_cbs_set_participant_device_removed(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_participant_device_removed(nativePtr, on_participant_device_removed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_participant_device_state_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_participant_device_state_changed(IntPtr thiz, OnParticipantDeviceStateChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnParticipantDeviceStateChangedDelegatePrivate(IntPtr chatRoom, IntPtr eventLog, int state);

		/// <summary>
		/// Callback used to notify a conference that a participant device has changed
		/// state. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// object    </param>
		/// <param name="eventLog"><see cref="Linphone.EventLog">Linphone.EventLog</see>
		/// The event to be notified    </param>
		/// <param name="state">new participant device state </param>
		public delegate void OnParticipantDeviceStateChangedDelegate(Linphone.ChatRoom chatRoom, Linphone.EventLog eventLog, Linphone.ParticipantDeviceState state);
		private OnParticipantDeviceStateChangedDelegatePrivate on_participant_device_state_changed_private;
		private OnParticipantDeviceStateChangedDelegate on_participant_device_state_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnParticipantDeviceStateChangedDelegatePrivate))]
#endif
		private static void on_participant_device_state_changed(IntPtr chatRoom, IntPtr eventLog, int state)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_participant_device_state_changed_public?.Invoke(thiz, fromNativePtr<Linphone.EventLog>(eventLog), (Linphone.ParticipantDeviceState)state);
			}
		}

		public OnParticipantDeviceStateChangedDelegate OnParticipantDeviceStateChanged
		{
			get
			{
				return on_participant_device_state_changed_public;
			}
			set
			{
				on_participant_device_state_changed_public = value;
				on_participant_device_state_changed_private = on_participant_device_state_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_participant_device_state_changed_private);
				linphone_chat_room_cbs_set_participant_device_state_changed(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_participant_device_state_changed(nativePtr, on_participant_device_state_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_participant_device_media_availability_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_participant_device_media_availability_changed(IntPtr thiz, OnParticipantDeviceMediaAvailabilityChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnParticipantDeviceMediaAvailabilityChangedDelegatePrivate(IntPtr chatRoom, IntPtr eventLog);

		/// <summary>
		/// Callback used to notify a conference that the media availability of a
		/// participant device has been changed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// object    </param>
		/// <param name="eventLog"><see cref="Linphone.EventLog">Linphone.EventLog</see>
		/// The event to be notified    </param>
		public delegate void OnParticipantDeviceMediaAvailabilityChangedDelegate(Linphone.ChatRoom chatRoom, Linphone.EventLog eventLog);
		private OnParticipantDeviceMediaAvailabilityChangedDelegatePrivate on_participant_device_media_availability_changed_private;
		private OnParticipantDeviceMediaAvailabilityChangedDelegate on_participant_device_media_availability_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnParticipantDeviceMediaAvailabilityChangedDelegatePrivate))]
#endif
		private static void on_participant_device_media_availability_changed(IntPtr chatRoom, IntPtr eventLog)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_participant_device_media_availability_changed_public?.Invoke(thiz, fromNativePtr<Linphone.EventLog>(eventLog));
			}
		}

		public OnParticipantDeviceMediaAvailabilityChangedDelegate OnParticipantDeviceMediaAvailabilityChanged
		{
			get
			{
				return on_participant_device_media_availability_changed_public;
			}
			set
			{
				on_participant_device_media_availability_changed_public = value;
				on_participant_device_media_availability_changed_private = on_participant_device_media_availability_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_participant_device_media_availability_changed_private);
				linphone_chat_room_cbs_set_participant_device_media_availability_changed(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_participant_device_media_availability_changed(nativePtr, on_participant_device_media_availability_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_conference_joined(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_conference_joined(IntPtr thiz, OnConferenceJoinedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnConferenceJoinedDelegatePrivate(IntPtr chatRoom, IntPtr eventLog);

		/// <summary>
		/// Callback used to notify a chat room has been joined. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// object    </param>
		/// <param name="eventLog"><see cref="Linphone.EventLog">Linphone.EventLog</see>
		/// The event to be notified    </param>
		public delegate void OnConferenceJoinedDelegate(Linphone.ChatRoom chatRoom, Linphone.EventLog eventLog);
		private OnConferenceJoinedDelegatePrivate on_conference_joined_private;
		private OnConferenceJoinedDelegate on_conference_joined_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnConferenceJoinedDelegatePrivate))]
#endif
		private static void on_conference_joined(IntPtr chatRoom, IntPtr eventLog)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_conference_joined_public?.Invoke(thiz, fromNativePtr<Linphone.EventLog>(eventLog));
			}
		}

		public OnConferenceJoinedDelegate OnConferenceJoined
		{
			get
			{
				return on_conference_joined_public;
			}
			set
			{
				on_conference_joined_public = value;
				on_conference_joined_private = on_conference_joined;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_conference_joined_private);
				linphone_chat_room_cbs_set_conference_joined(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_conference_joined(nativePtr, on_conference_joined_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_conference_left(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_conference_left(IntPtr thiz, OnConferenceLeftDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnConferenceLeftDelegatePrivate(IntPtr chatRoom, IntPtr eventLog);

		/// <summary>
		/// Callback used to notify a chat room has been left. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// object    </param>
		/// <param name="eventLog"><see cref="Linphone.EventLog">Linphone.EventLog</see>
		/// The event to be notified    </param>
		public delegate void OnConferenceLeftDelegate(Linphone.ChatRoom chatRoom, Linphone.EventLog eventLog);
		private OnConferenceLeftDelegatePrivate on_conference_left_private;
		private OnConferenceLeftDelegate on_conference_left_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnConferenceLeftDelegatePrivate))]
#endif
		private static void on_conference_left(IntPtr chatRoom, IntPtr eventLog)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_conference_left_public?.Invoke(thiz, fromNativePtr<Linphone.EventLog>(eventLog));
			}
		}

		public OnConferenceLeftDelegate OnConferenceLeft
		{
			get
			{
				return on_conference_left_public;
			}
			set
			{
				on_conference_left_public = value;
				on_conference_left_private = on_conference_left;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_conference_left_private);
				linphone_chat_room_cbs_set_conference_left(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_conference_left(nativePtr, on_conference_left_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_ephemeral_event(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_ephemeral_event(IntPtr thiz, OnEphemeralEventDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnEphemeralEventDelegatePrivate(IntPtr chatRoom, IntPtr eventLog);

		/// <summary>
		/// Callback used to notify a chat room that an ephemeral related event has been
		/// generated. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// object    </param>
		/// <param name="eventLog"><see cref="Linphone.EventLog">Linphone.EventLog</see>
		/// The event to be notified    </param>
		public delegate void OnEphemeralEventDelegate(Linphone.ChatRoom chatRoom, Linphone.EventLog eventLog);
		private OnEphemeralEventDelegatePrivate on_ephemeral_event_private;
		private OnEphemeralEventDelegate on_ephemeral_event_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnEphemeralEventDelegatePrivate))]
#endif
		private static void on_ephemeral_event(IntPtr chatRoom, IntPtr eventLog)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_ephemeral_event_public?.Invoke(thiz, fromNativePtr<Linphone.EventLog>(eventLog));
			}
		}

		public OnEphemeralEventDelegate OnEphemeralEvent
		{
			get
			{
				return on_ephemeral_event_public;
			}
			set
			{
				on_ephemeral_event_public = value;
				on_ephemeral_event_private = on_ephemeral_event;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_ephemeral_event_private);
				linphone_chat_room_cbs_set_ephemeral_event(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_ephemeral_event(nativePtr, on_ephemeral_event_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_ephemeral_message_timer_started(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_ephemeral_message_timer_started(IntPtr thiz, OnEphemeralMessageTimerStartedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnEphemeralMessageTimerStartedDelegatePrivate(IntPtr chatRoom, IntPtr eventLog);

		/// <summary>
		/// Callback used to notify a chat room that the lifespan of an ephemeral message
		/// before disappearing has started to decrease. 
		/// <para>
		/// This callback is called when the ephemeral message is read by the receiver. 
		/// </para>
		/// </summary>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// object    </param>
		/// <param name="eventLog"><see cref="Linphone.EventLog">Linphone.EventLog</see>
		/// The event to be notified    </param>
		public delegate void OnEphemeralMessageTimerStartedDelegate(Linphone.ChatRoom chatRoom, Linphone.EventLog eventLog);
		private OnEphemeralMessageTimerStartedDelegatePrivate on_ephemeral_message_timer_started_private;
		private OnEphemeralMessageTimerStartedDelegate on_ephemeral_message_timer_started_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnEphemeralMessageTimerStartedDelegatePrivate))]
#endif
		private static void on_ephemeral_message_timer_started(IntPtr chatRoom, IntPtr eventLog)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_ephemeral_message_timer_started_public?.Invoke(thiz, fromNativePtr<Linphone.EventLog>(eventLog));
			}
		}

		public OnEphemeralMessageTimerStartedDelegate OnEphemeralMessageTimerStarted
		{
			get
			{
				return on_ephemeral_message_timer_started_public;
			}
			set
			{
				on_ephemeral_message_timer_started_public = value;
				on_ephemeral_message_timer_started_private = on_ephemeral_message_timer_started;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_ephemeral_message_timer_started_private);
				linphone_chat_room_cbs_set_ephemeral_message_timer_started(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_ephemeral_message_timer_started(nativePtr, on_ephemeral_message_timer_started_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_ephemeral_message_deleted(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_ephemeral_message_deleted(IntPtr thiz, OnEphemeralMessageDeletedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnEphemeralMessageDeletedDelegatePrivate(IntPtr chatRoom, IntPtr eventLog);

		/// <summary>
		/// Callback used to notify a chat room that an ephemeral message has been deleted. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// object    </param>
		/// <param name="eventLog"><see cref="Linphone.EventLog">Linphone.EventLog</see>
		/// The event to be notified    </param>
		public delegate void OnEphemeralMessageDeletedDelegate(Linphone.ChatRoom chatRoom, Linphone.EventLog eventLog);
		private OnEphemeralMessageDeletedDelegatePrivate on_ephemeral_message_deleted_private;
		private OnEphemeralMessageDeletedDelegate on_ephemeral_message_deleted_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnEphemeralMessageDeletedDelegatePrivate))]
#endif
		private static void on_ephemeral_message_deleted(IntPtr chatRoom, IntPtr eventLog)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_ephemeral_message_deleted_public?.Invoke(thiz, fromNativePtr<Linphone.EventLog>(eventLog));
			}
		}

		public OnEphemeralMessageDeletedDelegate OnEphemeralMessageDeleted
		{
			get
			{
				return on_ephemeral_message_deleted_public;
			}
			set
			{
				on_ephemeral_message_deleted_public = value;
				on_ephemeral_message_deleted_private = on_ephemeral_message_deleted;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_ephemeral_message_deleted_private);
				linphone_chat_room_cbs_set_ephemeral_message_deleted(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_ephemeral_message_deleted(nativePtr, on_ephemeral_message_deleted_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_conference_address_generation(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_conference_address_generation(IntPtr thiz, OnConferenceAddressGenerationDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnConferenceAddressGenerationDelegatePrivate(IntPtr chatRoom);

		/// <summary>
		/// Callback used when a group chat room is created server-side to generate the
		/// address of the chat room. 
		/// <para>
		/// The function <see cref="Linphone.ChatRoom.ConferenceAddress">
		/// Linphone.ChatRoom.ConferenceAddress</see> needs to be called by this callback. 
		/// </para>
		/// </summary>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// object    </param>
		public delegate void OnConferenceAddressGenerationDelegate(Linphone.ChatRoom chatRoom);
		private OnConferenceAddressGenerationDelegatePrivate on_conference_address_generation_private;
		private OnConferenceAddressGenerationDelegate on_conference_address_generation_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnConferenceAddressGenerationDelegatePrivate))]
#endif
		private static void on_conference_address_generation(IntPtr chatRoom)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_conference_address_generation_public?.Invoke(thiz);
			}
		}

		public OnConferenceAddressGenerationDelegate OnConferenceAddressGeneration
		{
			get
			{
				return on_conference_address_generation_public;
			}
			set
			{
				on_conference_address_generation_public = value;
				on_conference_address_generation_private = on_conference_address_generation;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_conference_address_generation_private);
				linphone_chat_room_cbs_set_conference_address_generation(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_conference_address_generation(nativePtr, on_conference_address_generation_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_participant_registration_subscription_requested(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_participant_registration_subscription_requested(IntPtr thiz, OnParticipantRegistrationSubscriptionRequestedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnParticipantRegistrationSubscriptionRequestedDelegatePrivate(IntPtr chatRoom, IntPtr participantAddress);

		/// <summary>
		/// Callback used when a group chat room server is subscribing to registration
		/// state of a participant. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// object    </param>
		/// <param name="participantAddress"><see cref="Linphone.Address">
		/// Linphone.Address</see> object    </param>
		public delegate void OnParticipantRegistrationSubscriptionRequestedDelegate(Linphone.ChatRoom chatRoom, Linphone.Address participantAddress);
		private OnParticipantRegistrationSubscriptionRequestedDelegatePrivate on_participant_registration_subscription_requested_private;
		private OnParticipantRegistrationSubscriptionRequestedDelegate on_participant_registration_subscription_requested_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnParticipantRegistrationSubscriptionRequestedDelegatePrivate))]
#endif
		private static void on_participant_registration_subscription_requested(IntPtr chatRoom, IntPtr participantAddress)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_participant_registration_subscription_requested_public?.Invoke(thiz, fromNativePtr<Linphone.Address>(participantAddress));
			}
		}

		public OnParticipantRegistrationSubscriptionRequestedDelegate OnParticipantRegistrationSubscriptionRequested
		{
			get
			{
				return on_participant_registration_subscription_requested_public;
			}
			set
			{
				on_participant_registration_subscription_requested_public = value;
				on_participant_registration_subscription_requested_private = on_participant_registration_subscription_requested;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_participant_registration_subscription_requested_private);
				linphone_chat_room_cbs_set_participant_registration_subscription_requested(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_participant_registration_subscription_requested(nativePtr, on_participant_registration_subscription_requested_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_participant_registration_unsubscription_requested(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_participant_registration_unsubscription_requested(IntPtr thiz, OnParticipantRegistrationUnsubscriptionRequestedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnParticipantRegistrationUnsubscriptionRequestedDelegatePrivate(IntPtr chatRoom, IntPtr participantAddress);

		/// <summary>
		/// Callback used when a group chat room server is unsubscribing to registration
		/// state of a participant. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// object    </param>
		/// <param name="participantAddress"><see cref="Linphone.Address">
		/// Linphone.Address</see> object    </param>
		public delegate void OnParticipantRegistrationUnsubscriptionRequestedDelegate(Linphone.ChatRoom chatRoom, Linphone.Address participantAddress);
		private OnParticipantRegistrationUnsubscriptionRequestedDelegatePrivate on_participant_registration_unsubscription_requested_private;
		private OnParticipantRegistrationUnsubscriptionRequestedDelegate on_participant_registration_unsubscription_requested_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnParticipantRegistrationUnsubscriptionRequestedDelegatePrivate))]
#endif
		private static void on_participant_registration_unsubscription_requested(IntPtr chatRoom, IntPtr participantAddress)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_participant_registration_unsubscription_requested_public?.Invoke(thiz, fromNativePtr<Linphone.Address>(participantAddress));
			}
		}

		public OnParticipantRegistrationUnsubscriptionRequestedDelegate OnParticipantRegistrationUnsubscriptionRequested
		{
			get
			{
				return on_participant_registration_unsubscription_requested_public;
			}
			set
			{
				on_participant_registration_unsubscription_requested_public = value;
				on_participant_registration_unsubscription_requested_private = on_participant_registration_unsubscription_requested;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_participant_registration_unsubscription_requested_private);
				linphone_chat_room_cbs_set_participant_registration_unsubscription_requested(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_participant_registration_unsubscription_requested(nativePtr, on_participant_registration_unsubscription_requested_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_chat_message_should_be_stored(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_chat_message_should_be_stored(IntPtr thiz, OnChatMessageShouldBeStoredDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnChatMessageShouldBeStoredDelegatePrivate(IntPtr chatRoom, IntPtr message);

		/// <summary>
		/// Callback used to tell the core whether or not to store the incoming message in
		/// db or not using <see cref="Linphone.ChatMessage.ToBeStored">
		/// Linphone.ChatMessage.ToBeStored</see>. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// object    </param>
		/// <param name="message">The <see cref="Linphone.ChatMessage">
		/// Linphone.ChatMessage</see> that is being received    </param>
		public delegate void OnChatMessageShouldBeStoredDelegate(Linphone.ChatRoom chatRoom, Linphone.ChatMessage message);
		private OnChatMessageShouldBeStoredDelegatePrivate on_chat_message_should_be_stored_private;
		private OnChatMessageShouldBeStoredDelegate on_chat_message_should_be_stored_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnChatMessageShouldBeStoredDelegatePrivate))]
#endif
		private static void on_chat_message_should_be_stored(IntPtr chatRoom, IntPtr message)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_chat_message_should_be_stored_public?.Invoke(thiz, fromNativePtr<Linphone.ChatMessage>(message));
			}
		}

		public OnChatMessageShouldBeStoredDelegate OnChatMessageShouldBeStored
		{
			get
			{
				return on_chat_message_should_be_stored_public;
			}
			set
			{
				on_chat_message_should_be_stored_public = value;
				on_chat_message_should_be_stored_private = on_chat_message_should_be_stored;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_chat_message_should_be_stored_private);
				linphone_chat_room_cbs_set_chat_message_should_be_stored(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_chat_message_should_be_stored(nativePtr, on_chat_message_should_be_stored_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_chat_message_participant_imdn_state_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_chat_message_participant_imdn_state_changed(IntPtr thiz, OnChatMessageParticipantImdnStateChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnChatMessageParticipantImdnStateChangedDelegatePrivate(IntPtr chatRoom, IntPtr message, IntPtr state);

		/// <summary>
		/// Callback used to notify a participant state has changed in a message of this
		/// chat room. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// object    </param>
		/// <param name="message">The <see cref="Linphone.ChatMessage">
		/// Linphone.ChatMessage</see> for which a participant has it&apos;s state changed 
		///   </param>
		/// <param name="state">The <see cref="Linphone.ParticipantImdnState">
		/// Linphone.ParticipantImdnState</see>    </param>
		public delegate void OnChatMessageParticipantImdnStateChangedDelegate(Linphone.ChatRoom chatRoom, Linphone.ChatMessage message, Linphone.ParticipantImdnState state);
		private OnChatMessageParticipantImdnStateChangedDelegatePrivate on_chat_message_participant_imdn_state_changed_private;
		private OnChatMessageParticipantImdnStateChangedDelegate on_chat_message_participant_imdn_state_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnChatMessageParticipantImdnStateChangedDelegatePrivate))]
#endif
		private static void on_chat_message_participant_imdn_state_changed(IntPtr chatRoom, IntPtr message, IntPtr state)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_chat_message_participant_imdn_state_changed_public?.Invoke(thiz, fromNativePtr<Linphone.ChatMessage>(message), fromNativePtr<Linphone.ParticipantImdnState>(state));
			}
		}

		public OnChatMessageParticipantImdnStateChangedDelegate OnChatMessageParticipantImdnStateChanged
		{
			get
			{
				return on_chat_message_participant_imdn_state_changed_public;
			}
			set
			{
				on_chat_message_participant_imdn_state_changed_public = value;
				on_chat_message_participant_imdn_state_changed_private = on_chat_message_participant_imdn_state_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_chat_message_participant_imdn_state_changed_private);
				linphone_chat_room_cbs_set_chat_message_participant_imdn_state_changed(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_chat_message_participant_imdn_state_changed(nativePtr, on_chat_message_participant_imdn_state_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_chat_room_read(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_chat_room_read(IntPtr thiz, OnChatRoomReadDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnChatRoomReadDelegatePrivate(IntPtr chatRoom);

		/// <summary>
		/// Callback used to notify a chat room was &quot;marked as read&quot;. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="chatRoom">The LinphoneChatRoom object that was marked as read   
		/// </param>
		public delegate void OnChatRoomReadDelegate(Linphone.ChatRoom chatRoom);
		private OnChatRoomReadDelegatePrivate on_chat_room_read_private;
		private OnChatRoomReadDelegate on_chat_room_read_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnChatRoomReadDelegatePrivate))]
#endif
		private static void on_chat_room_read(IntPtr chatRoom)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_chat_room_read_public?.Invoke(thiz);
			}
		}

		public OnChatRoomReadDelegate OnChatRoomRead
		{
			get
			{
				return on_chat_room_read_public;
			}
			set
			{
				on_chat_room_read_public = value;
				on_chat_room_read_private = on_chat_room_read;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_chat_room_read_private);
				linphone_chat_room_cbs_set_chat_room_read(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_chat_room_read(nativePtr, on_chat_room_read_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_chat_room_cbs_set_new_message_reaction(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_chat_room_cbs_set_new_message_reaction(IntPtr thiz, OnNewMessageReactionDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnNewMessageReactionDelegatePrivate(IntPtr chatRoom, IntPtr message, IntPtr reaction);

		/// <summary>
		/// Callback used to notify a reaction has been received or sent for a given
		/// message. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="chatRoom">LinphoneChatRoom object    </param>
		/// <param name="message">LinphoneChatMessage object for which we received a
		/// reaction    </param>
		/// <param name="reaction">the LinphoneChatMessageReaction reaction that was sent
		/// or received    </param>
		public delegate void OnNewMessageReactionDelegate(Linphone.ChatRoom chatRoom, Linphone.ChatMessage message, Linphone.ChatMessageReaction reaction);
		private OnNewMessageReactionDelegatePrivate on_new_message_reaction_private;
		private OnNewMessageReactionDelegate on_new_message_reaction_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnNewMessageReactionDelegatePrivate))]
#endif
		private static void on_new_message_reaction(IntPtr chatRoom, IntPtr message, IntPtr reaction)
		{
			ChatRoom thiz = fromNativePtr<ChatRoom>(chatRoom);

			IntPtr ptr = linphone_chat_room_get_current_callbacks(thiz.nativePtr);
			ChatRoomListener current_listener = fromNativePtr<ChatRoomListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_new_message_reaction_public?.Invoke(thiz, fromNativePtr<Linphone.ChatMessage>(message), fromNativePtr<Linphone.ChatMessageReaction>(reaction));
			}
		}

		public OnNewMessageReactionDelegate OnNewMessageReaction
		{
			get
			{
				return on_new_message_reaction_public;
			}
			set
			{
				on_new_message_reaction_public = value;
				on_new_message_reaction_private = on_new_message_reaction;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_new_message_reaction_private);
				linphone_chat_room_cbs_set_new_message_reaction(nativePtr, cb);
#else
				linphone_chat_room_cbs_set_new_message_reaction(nativePtr, on_new_message_reaction_private);
#endif
			}
		}
	}

	/// <summary>
	/// An object to handle the callbacks for the handling a <see cref="Linphone.Conference">
	/// Linphone.Conference</see> objects. 
	/// <para>
	/// Use <see cref="Linphone.Factory.CreateConferenceCbs()">
	/// Linphone.Factory.CreateConferenceCbs()</see> to create an instance. Then pass
	/// the object to a <see cref="Linphone.Conference">Linphone.Conference</see>
	/// instance through <see cref="Linphone.Conference.AddListener()">
	/// Linphone.Conference.AddListener()</see>. 
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class ConferenceListener : LinphoneObject
	{
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_conference_get_current_callbacks(IntPtr thiz);

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_conference_cbs_set_allowed_participant_list_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_conference_cbs_set_allowed_participant_list_changed(IntPtr thiz, OnAllowedParticipantListChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnAllowedParticipantListChangedDelegatePrivate(IntPtr conference);

		/// <summary>
		/// Callback used to notify a conference that the list of participants allowed to
		/// join the conference has changed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="conference"><see cref="Linphone.Conference">
		/// Linphone.Conference</see> object    </param>
		public delegate void OnAllowedParticipantListChangedDelegate(Linphone.Conference conference);
		private OnAllowedParticipantListChangedDelegatePrivate on_allowed_participant_list_changed_private;
		private OnAllowedParticipantListChangedDelegate on_allowed_participant_list_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnAllowedParticipantListChangedDelegatePrivate))]
#endif
		private static void on_allowed_participant_list_changed(IntPtr conference)
		{
			Conference thiz = fromNativePtr<Conference>(conference);

			IntPtr ptr = linphone_conference_get_current_callbacks(thiz.nativePtr);
			ConferenceListener current_listener = fromNativePtr<ConferenceListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_allowed_participant_list_changed_public?.Invoke(thiz);
			}
		}

		public OnAllowedParticipantListChangedDelegate OnAllowedParticipantListChanged
		{
			get
			{
				return on_allowed_participant_list_changed_public;
			}
			set
			{
				on_allowed_participant_list_changed_public = value;
				on_allowed_participant_list_changed_private = on_allowed_participant_list_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_allowed_participant_list_changed_private);
				linphone_conference_cbs_set_allowed_participant_list_changed(nativePtr, cb);
#else
				linphone_conference_cbs_set_allowed_participant_list_changed(nativePtr, on_allowed_participant_list_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_conference_cbs_set_participant_added(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_conference_cbs_set_participant_added(IntPtr thiz, OnParticipantAddedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnParticipantAddedDelegatePrivate(IntPtr conference, IntPtr participant);

		/// <summary>
		/// Callback used to notify a conference that a participant has been added. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="conference">LinphoneConference object    </param>
		/// <param name="participant">LinphoneParticipant that has been added to the
		/// conference    </param>
		public delegate void OnParticipantAddedDelegate(Linphone.Conference conference, Linphone.Participant participant);
		private OnParticipantAddedDelegatePrivate on_participant_added_private;
		private OnParticipantAddedDelegate on_participant_added_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnParticipantAddedDelegatePrivate))]
#endif
		private static void on_participant_added(IntPtr conference, IntPtr participant)
		{
			Conference thiz = fromNativePtr<Conference>(conference);

			IntPtr ptr = linphone_conference_get_current_callbacks(thiz.nativePtr);
			ConferenceListener current_listener = fromNativePtr<ConferenceListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_participant_added_public?.Invoke(thiz, fromNativePtr<Linphone.Participant>(participant));
			}
		}

		public OnParticipantAddedDelegate OnParticipantAdded
		{
			get
			{
				return on_participant_added_public;
			}
			set
			{
				on_participant_added_public = value;
				on_participant_added_private = on_participant_added;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_participant_added_private);
				linphone_conference_cbs_set_participant_added(nativePtr, cb);
#else
				linphone_conference_cbs_set_participant_added(nativePtr, on_participant_added_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_conference_cbs_set_participant_removed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_conference_cbs_set_participant_removed(IntPtr thiz, OnParticipantRemovedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnParticipantRemovedDelegatePrivate(IntPtr conference, IntPtr participant);

		/// <summary>
		/// Callback used to notify a conference that a participant has been removed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="conference">LinphoneConference object    </param>
		/// <param name="participant">LinphoneParticipant that has been removed to the
		/// conference    </param>
		public delegate void OnParticipantRemovedDelegate(Linphone.Conference conference, Linphone.Participant participant);
		private OnParticipantRemovedDelegatePrivate on_participant_removed_private;
		private OnParticipantRemovedDelegate on_participant_removed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnParticipantRemovedDelegatePrivate))]
#endif
		private static void on_participant_removed(IntPtr conference, IntPtr participant)
		{
			Conference thiz = fromNativePtr<Conference>(conference);

			IntPtr ptr = linphone_conference_get_current_callbacks(thiz.nativePtr);
			ConferenceListener current_listener = fromNativePtr<ConferenceListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_participant_removed_public?.Invoke(thiz, fromNativePtr<Linphone.Participant>(participant));
			}
		}

		public OnParticipantRemovedDelegate OnParticipantRemoved
		{
			get
			{
				return on_participant_removed_public;
			}
			set
			{
				on_participant_removed_public = value;
				on_participant_removed_private = on_participant_removed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_participant_removed_private);
				linphone_conference_cbs_set_participant_removed(nativePtr, cb);
#else
				linphone_conference_cbs_set_participant_removed(nativePtr, on_participant_removed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_conference_cbs_set_participant_device_added(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_conference_cbs_set_participant_device_added(IntPtr thiz, OnParticipantDeviceAddedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnParticipantDeviceAddedDelegatePrivate(IntPtr conference, IntPtr participantDevice);

		/// <summary>
		/// Callback used to notify a conference that a participant device has been added. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="conference">LinphoneConference object    </param>
		/// <param name="participantDevice">LinphoneParticipantDevice that has been added
		/// to the conference    </param>
		public delegate void OnParticipantDeviceAddedDelegate(Linphone.Conference conference, Linphone.ParticipantDevice participantDevice);
		private OnParticipantDeviceAddedDelegatePrivate on_participant_device_added_private;
		private OnParticipantDeviceAddedDelegate on_participant_device_added_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnParticipantDeviceAddedDelegatePrivate))]
#endif
		private static void on_participant_device_added(IntPtr conference, IntPtr participantDevice)
		{
			Conference thiz = fromNativePtr<Conference>(conference);

			IntPtr ptr = linphone_conference_get_current_callbacks(thiz.nativePtr);
			ConferenceListener current_listener = fromNativePtr<ConferenceListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_participant_device_added_public?.Invoke(thiz, fromNativePtr<Linphone.ParticipantDevice>(participantDevice));
			}
		}

		public OnParticipantDeviceAddedDelegate OnParticipantDeviceAdded
		{
			get
			{
				return on_participant_device_added_public;
			}
			set
			{
				on_participant_device_added_public = value;
				on_participant_device_added_private = on_participant_device_added;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_participant_device_added_private);
				linphone_conference_cbs_set_participant_device_added(nativePtr, cb);
#else
				linphone_conference_cbs_set_participant_device_added(nativePtr, on_participant_device_added_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_conference_cbs_set_participant_device_removed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_conference_cbs_set_participant_device_removed(IntPtr thiz, OnParticipantDeviceRemovedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnParticipantDeviceRemovedDelegatePrivate(IntPtr conference, IntPtr participantDevice);

		/// <summary>
		/// Callback used to notify a conference that a participant device has been
		/// removed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="conference">LinphoneConference object    </param>
		/// <param name="participantDevice">LinphoneParticipantDevice that has been removed
		/// to the conference    </param>
		public delegate void OnParticipantDeviceRemovedDelegate(Linphone.Conference conference, Linphone.ParticipantDevice participantDevice);
		private OnParticipantDeviceRemovedDelegatePrivate on_participant_device_removed_private;
		private OnParticipantDeviceRemovedDelegate on_participant_device_removed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnParticipantDeviceRemovedDelegatePrivate))]
#endif
		private static void on_participant_device_removed(IntPtr conference, IntPtr participantDevice)
		{
			Conference thiz = fromNativePtr<Conference>(conference);

			IntPtr ptr = linphone_conference_get_current_callbacks(thiz.nativePtr);
			ConferenceListener current_listener = fromNativePtr<ConferenceListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_participant_device_removed_public?.Invoke(thiz, fromNativePtr<Linphone.ParticipantDevice>(participantDevice));
			}
		}

		public OnParticipantDeviceRemovedDelegate OnParticipantDeviceRemoved
		{
			get
			{
				return on_participant_device_removed_public;
			}
			set
			{
				on_participant_device_removed_public = value;
				on_participant_device_removed_private = on_participant_device_removed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_participant_device_removed_private);
				linphone_conference_cbs_set_participant_device_removed(nativePtr, cb);
#else
				linphone_conference_cbs_set_participant_device_removed(nativePtr, on_participant_device_removed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_conference_cbs_set_participant_device_joining_request(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_conference_cbs_set_participant_device_joining_request(IntPtr thiz, OnParticipantDeviceJoiningRequestDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnParticipantDeviceJoiningRequestDelegatePrivate(IntPtr conference, IntPtr participantDevice);

		/// <summary>
		/// Callback used to notify a conference that a participant has requested to join
		/// the conference. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="conference"><see cref="Linphone.Conference">
		/// Linphone.Conference</see> object    </param>
		/// <param name="participantDevice"><see cref="Linphone.ParticipantDevice">
		/// Linphone.ParticipantDevice</see> that has requested to join the conference   
		/// </param>
		public delegate void OnParticipantDeviceJoiningRequestDelegate(Linphone.Conference conference, Linphone.ParticipantDevice participantDevice);
		private OnParticipantDeviceJoiningRequestDelegatePrivate on_participant_device_joining_request_private;
		private OnParticipantDeviceJoiningRequestDelegate on_participant_device_joining_request_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnParticipantDeviceJoiningRequestDelegatePrivate))]
#endif
		private static void on_participant_device_joining_request(IntPtr conference, IntPtr participantDevice)
		{
			Conference thiz = fromNativePtr<Conference>(conference);

			IntPtr ptr = linphone_conference_get_current_callbacks(thiz.nativePtr);
			ConferenceListener current_listener = fromNativePtr<ConferenceListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_participant_device_joining_request_public?.Invoke(thiz, fromNativePtr<Linphone.ParticipantDevice>(participantDevice));
			}
		}

		public OnParticipantDeviceJoiningRequestDelegate OnParticipantDeviceJoiningRequest
		{
			get
			{
				return on_participant_device_joining_request_public;
			}
			set
			{
				on_participant_device_joining_request_public = value;
				on_participant_device_joining_request_private = on_participant_device_joining_request;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_participant_device_joining_request_private);
				linphone_conference_cbs_set_participant_device_joining_request(nativePtr, cb);
#else
				linphone_conference_cbs_set_participant_device_joining_request(nativePtr, on_participant_device_joining_request_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_conference_cbs_set_participant_role_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_conference_cbs_set_participant_role_changed(IntPtr thiz, OnParticipantRoleChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnParticipantRoleChangedDelegatePrivate(IntPtr conference, IntPtr participant);

		/// <summary>
		/// Callback used to notify a conference that the role of a participant has been
		/// changed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="conference"><see cref="Linphone.Conference">
		/// Linphone.Conference</see> object    </param>
		/// <param name="participant"><see cref="Linphone.Participant">
		/// Linphone.Participant</see> whose role has changed    </param>
		public delegate void OnParticipantRoleChangedDelegate(Linphone.Conference conference, Linphone.Participant participant);
		private OnParticipantRoleChangedDelegatePrivate on_participant_role_changed_private;
		private OnParticipantRoleChangedDelegate on_participant_role_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnParticipantRoleChangedDelegatePrivate))]
#endif
		private static void on_participant_role_changed(IntPtr conference, IntPtr participant)
		{
			Conference thiz = fromNativePtr<Conference>(conference);

			IntPtr ptr = linphone_conference_get_current_callbacks(thiz.nativePtr);
			ConferenceListener current_listener = fromNativePtr<ConferenceListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_participant_role_changed_public?.Invoke(thiz, fromNativePtr<Linphone.Participant>(participant));
			}
		}

		public OnParticipantRoleChangedDelegate OnParticipantRoleChanged
		{
			get
			{
				return on_participant_role_changed_public;
			}
			set
			{
				on_participant_role_changed_public = value;
				on_participant_role_changed_private = on_participant_role_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_participant_role_changed_private);
				linphone_conference_cbs_set_participant_role_changed(nativePtr, cb);
#else
				linphone_conference_cbs_set_participant_role_changed(nativePtr, on_participant_role_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_conference_cbs_set_participant_admin_status_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_conference_cbs_set_participant_admin_status_changed(IntPtr thiz, OnParticipantAdminStatusChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnParticipantAdminStatusChangedDelegatePrivate(IntPtr conference, IntPtr participant);

		/// <summary>
		/// Callback used to notify a conference that the admin status of a participant has
		/// been changed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="conference">LinphoneConference object    </param>
		/// <param name="participant">LinphoneParticipant whose admin status has changed   
		/// </param>
		public delegate void OnParticipantAdminStatusChangedDelegate(Linphone.Conference conference, Linphone.Participant participant);
		private OnParticipantAdminStatusChangedDelegatePrivate on_participant_admin_status_changed_private;
		private OnParticipantAdminStatusChangedDelegate on_participant_admin_status_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnParticipantAdminStatusChangedDelegatePrivate))]
#endif
		private static void on_participant_admin_status_changed(IntPtr conference, IntPtr participant)
		{
			Conference thiz = fromNativePtr<Conference>(conference);

			IntPtr ptr = linphone_conference_get_current_callbacks(thiz.nativePtr);
			ConferenceListener current_listener = fromNativePtr<ConferenceListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_participant_admin_status_changed_public?.Invoke(thiz, fromNativePtr<Linphone.Participant>(participant));
			}
		}

		public OnParticipantAdminStatusChangedDelegate OnParticipantAdminStatusChanged
		{
			get
			{
				return on_participant_admin_status_changed_public;
			}
			set
			{
				on_participant_admin_status_changed_public = value;
				on_participant_admin_status_changed_private = on_participant_admin_status_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_participant_admin_status_changed_private);
				linphone_conference_cbs_set_participant_admin_status_changed(nativePtr, cb);
#else
				linphone_conference_cbs_set_participant_admin_status_changed(nativePtr, on_participant_admin_status_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_conference_cbs_set_participant_device_state_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_conference_cbs_set_participant_device_state_changed(IntPtr thiz, OnParticipantDeviceStateChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnParticipantDeviceStateChangedDelegatePrivate(IntPtr conference, IntPtr device, int state);

		/// <summary>
		/// Callback used to notify a conference that a participant device has changed
		/// state. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="conference">LinphoneConference object    </param>
		/// <param name="device">LinphoneParticipantDevice who change state    </param>
		/// <param name="state">new participant device state </param>
		public delegate void OnParticipantDeviceStateChangedDelegate(Linphone.Conference conference, Linphone.ParticipantDevice device, Linphone.ParticipantDeviceState state);
		private OnParticipantDeviceStateChangedDelegatePrivate on_participant_device_state_changed_private;
		private OnParticipantDeviceStateChangedDelegate on_participant_device_state_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnParticipantDeviceStateChangedDelegatePrivate))]
#endif
		private static void on_participant_device_state_changed(IntPtr conference, IntPtr device, int state)
		{
			Conference thiz = fromNativePtr<Conference>(conference);

			IntPtr ptr = linphone_conference_get_current_callbacks(thiz.nativePtr);
			ConferenceListener current_listener = fromNativePtr<ConferenceListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_participant_device_state_changed_public?.Invoke(thiz, fromNativePtr<Linphone.ParticipantDevice>(device), (Linphone.ParticipantDeviceState)state);
			}
		}

		public OnParticipantDeviceStateChangedDelegate OnParticipantDeviceStateChanged
		{
			get
			{
				return on_participant_device_state_changed_public;
			}
			set
			{
				on_participant_device_state_changed_public = value;
				on_participant_device_state_changed_private = on_participant_device_state_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_participant_device_state_changed_private);
				linphone_conference_cbs_set_participant_device_state_changed(nativePtr, cb);
#else
				linphone_conference_cbs_set_participant_device_state_changed(nativePtr, on_participant_device_state_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_conference_cbs_set_participant_device_screen_sharing_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_conference_cbs_set_participant_device_screen_sharing_changed(IntPtr thiz, OnParticipantDeviceScreenSharingChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnParticipantDeviceScreenSharingChangedDelegatePrivate(IntPtr conference, IntPtr device, char enabled);

		/// <summary>
		/// Callback used to notify a conference that a participant device starts or stops
		/// screen sharing. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="conference"><see cref="Linphone.Conference">
		/// Linphone.Conference</see> object    </param>
		/// <param name="device"><see cref="Linphone.ParticipantDevice">
		/// Linphone.ParticipantDevice</see> who starts or stops screen sharing    </param>
		/// <param name="enabled">whether the screen sharing is enabled or disabled </param>
		public delegate void OnParticipantDeviceScreenSharingChangedDelegate(Linphone.Conference conference, Linphone.ParticipantDevice device, bool enabled);
		private OnParticipantDeviceScreenSharingChangedDelegatePrivate on_participant_device_screen_sharing_changed_private;
		private OnParticipantDeviceScreenSharingChangedDelegate on_participant_device_screen_sharing_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnParticipantDeviceScreenSharingChangedDelegatePrivate))]
#endif
		private static void on_participant_device_screen_sharing_changed(IntPtr conference, IntPtr device, char enabled)
		{
			Conference thiz = fromNativePtr<Conference>(conference);

			IntPtr ptr = linphone_conference_get_current_callbacks(thiz.nativePtr);
			ConferenceListener current_listener = fromNativePtr<ConferenceListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_participant_device_screen_sharing_changed_public?.Invoke(thiz, fromNativePtr<Linphone.ParticipantDevice>(device), enabled == 0);
			}
		}

		public OnParticipantDeviceScreenSharingChangedDelegate OnParticipantDeviceScreenSharingChanged
		{
			get
			{
				return on_participant_device_screen_sharing_changed_public;
			}
			set
			{
				on_participant_device_screen_sharing_changed_public = value;
				on_participant_device_screen_sharing_changed_private = on_participant_device_screen_sharing_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_participant_device_screen_sharing_changed_private);
				linphone_conference_cbs_set_participant_device_screen_sharing_changed(nativePtr, cb);
#else
				linphone_conference_cbs_set_participant_device_screen_sharing_changed(nativePtr, on_participant_device_screen_sharing_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_conference_cbs_set_participant_device_media_availability_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_conference_cbs_set_participant_device_media_availability_changed(IntPtr thiz, OnParticipantDeviceMediaAvailabilityChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnParticipantDeviceMediaAvailabilityChangedDelegatePrivate(IntPtr conference, IntPtr device);

		/// <summary>
		/// Callback used to notify a conference that the media availability of a
		/// participant device has been changed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="conference">LinphoneConference object    </param>
		/// <param name="device">LinphoneParticipantDevice whose media availability changed
		/// has changed    </param>
		public delegate void OnParticipantDeviceMediaAvailabilityChangedDelegate(Linphone.Conference conference, Linphone.ParticipantDevice device);
		private OnParticipantDeviceMediaAvailabilityChangedDelegatePrivate on_participant_device_media_availability_changed_private;
		private OnParticipantDeviceMediaAvailabilityChangedDelegate on_participant_device_media_availability_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnParticipantDeviceMediaAvailabilityChangedDelegatePrivate))]
#endif
		private static void on_participant_device_media_availability_changed(IntPtr conference, IntPtr device)
		{
			Conference thiz = fromNativePtr<Conference>(conference);

			IntPtr ptr = linphone_conference_get_current_callbacks(thiz.nativePtr);
			ConferenceListener current_listener = fromNativePtr<ConferenceListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_participant_device_media_availability_changed_public?.Invoke(thiz, fromNativePtr<Linphone.ParticipantDevice>(device));
			}
		}

		public OnParticipantDeviceMediaAvailabilityChangedDelegate OnParticipantDeviceMediaAvailabilityChanged
		{
			get
			{
				return on_participant_device_media_availability_changed_public;
			}
			set
			{
				on_participant_device_media_availability_changed_public = value;
				on_participant_device_media_availability_changed_private = on_participant_device_media_availability_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_participant_device_media_availability_changed_private);
				linphone_conference_cbs_set_participant_device_media_availability_changed(nativePtr, cb);
#else
				linphone_conference_cbs_set_participant_device_media_availability_changed(nativePtr, on_participant_device_media_availability_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_conference_cbs_set_participant_device_media_capability_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_conference_cbs_set_participant_device_media_capability_changed(IntPtr thiz, OnParticipantDeviceMediaCapabilityChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnParticipantDeviceMediaCapabilityChangedDelegatePrivate(IntPtr conference, IntPtr device);

		/// <summary>
		/// Callback used to notify a conference that the media capability of a participant
		/// device has been changed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="conference"><see cref="Linphone.Conference">
		/// Linphone.Conference</see> object    </param>
		/// <param name="device"><see cref="Linphone.ParticipantDevice">
		/// Linphone.ParticipantDevice</see> whose media capability changed has changed   
		/// </param>
		public delegate void OnParticipantDeviceMediaCapabilityChangedDelegate(Linphone.Conference conference, Linphone.ParticipantDevice device);
		private OnParticipantDeviceMediaCapabilityChangedDelegatePrivate on_participant_device_media_capability_changed_private;
		private OnParticipantDeviceMediaCapabilityChangedDelegate on_participant_device_media_capability_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnParticipantDeviceMediaCapabilityChangedDelegatePrivate))]
#endif
		private static void on_participant_device_media_capability_changed(IntPtr conference, IntPtr device)
		{
			Conference thiz = fromNativePtr<Conference>(conference);

			IntPtr ptr = linphone_conference_get_current_callbacks(thiz.nativePtr);
			ConferenceListener current_listener = fromNativePtr<ConferenceListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_participant_device_media_capability_changed_public?.Invoke(thiz, fromNativePtr<Linphone.ParticipantDevice>(device));
			}
		}

		public OnParticipantDeviceMediaCapabilityChangedDelegate OnParticipantDeviceMediaCapabilityChanged
		{
			get
			{
				return on_participant_device_media_capability_changed_public;
			}
			set
			{
				on_participant_device_media_capability_changed_public = value;
				on_participant_device_media_capability_changed_private = on_participant_device_media_capability_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_participant_device_media_capability_changed_private);
				linphone_conference_cbs_set_participant_device_media_capability_changed(nativePtr, cb);
#else
				linphone_conference_cbs_set_participant_device_media_capability_changed(nativePtr, on_participant_device_media_capability_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_conference_cbs_set_state_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_conference_cbs_set_state_changed(IntPtr thiz, OnStateChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnStateChangedDelegatePrivate(IntPtr conference, int newState);

		/// <summary>
		/// Callback used to notify a conference state has changed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="conference">LinphoneConference object    </param>
		/// <param name="newState">The new state of the conference </param>
		public delegate void OnStateChangedDelegate(Linphone.Conference conference, Linphone.ConferenceState newState);
		private OnStateChangedDelegatePrivate on_state_changed_private;
		private OnStateChangedDelegate on_state_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnStateChangedDelegatePrivate))]
#endif
		private static void on_state_changed(IntPtr conference, int newState)
		{
			Conference thiz = fromNativePtr<Conference>(conference);

			IntPtr ptr = linphone_conference_get_current_callbacks(thiz.nativePtr);
			ConferenceListener current_listener = fromNativePtr<ConferenceListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_state_changed_public?.Invoke(thiz, (Linphone.ConferenceState)newState);
			}
		}

		public OnStateChangedDelegate OnStateChanged
		{
			get
			{
				return on_state_changed_public;
			}
			set
			{
				on_state_changed_public = value;
				on_state_changed_private = on_state_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_state_changed_private);
				linphone_conference_cbs_set_state_changed(nativePtr, cb);
#else
				linphone_conference_cbs_set_state_changed(nativePtr, on_state_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_conference_cbs_set_available_media_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_conference_cbs_set_available_media_changed(IntPtr thiz, OnAvailableMediaChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnAvailableMediaChangedDelegatePrivate(IntPtr conference);

		/// <summary>
		/// Callback used to notify that the available media of a conference has changed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="conference"><see cref="Linphone.Conference">
		/// Linphone.Conference</see> object    </param>
		public delegate void OnAvailableMediaChangedDelegate(Linphone.Conference conference);
		private OnAvailableMediaChangedDelegatePrivate on_available_media_changed_private;
		private OnAvailableMediaChangedDelegate on_available_media_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnAvailableMediaChangedDelegatePrivate))]
#endif
		private static void on_available_media_changed(IntPtr conference)
		{
			Conference thiz = fromNativePtr<Conference>(conference);

			IntPtr ptr = linphone_conference_get_current_callbacks(thiz.nativePtr);
			ConferenceListener current_listener = fromNativePtr<ConferenceListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_available_media_changed_public?.Invoke(thiz);
			}
		}

		public OnAvailableMediaChangedDelegate OnAvailableMediaChanged
		{
			get
			{
				return on_available_media_changed_public;
			}
			set
			{
				on_available_media_changed_public = value;
				on_available_media_changed_private = on_available_media_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_available_media_changed_private);
				linphone_conference_cbs_set_available_media_changed(nativePtr, cb);
#else
				linphone_conference_cbs_set_available_media_changed(nativePtr, on_available_media_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_conference_cbs_set_subject_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_conference_cbs_set_subject_changed(IntPtr thiz, OnSubjectChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnSubjectChangedDelegatePrivate(IntPtr conference, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string subject);

		/// <summary>
		/// Callback used to notify that the subject of a conference has changed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="conference">LinphoneConference object    </param>
		/// <param name="subject">subject of the conference    </param>
		public delegate void OnSubjectChangedDelegate(Linphone.Conference conference, string subject);
		private OnSubjectChangedDelegatePrivate on_subject_changed_private;
		private OnSubjectChangedDelegate on_subject_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnSubjectChangedDelegatePrivate))]
#endif
		private static void on_subject_changed(IntPtr conference, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string subject)
		{
			Conference thiz = fromNativePtr<Conference>(conference);

			IntPtr ptr = linphone_conference_get_current_callbacks(thiz.nativePtr);
			ConferenceListener current_listener = fromNativePtr<ConferenceListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_subject_changed_public?.Invoke(thiz, subject);
			}
		}

		public OnSubjectChangedDelegate OnSubjectChanged
		{
			get
			{
				return on_subject_changed_public;
			}
			set
			{
				on_subject_changed_public = value;
				on_subject_changed_private = on_subject_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_subject_changed_private);
				linphone_conference_cbs_set_subject_changed(nativePtr, cb);
#else
				linphone_conference_cbs_set_subject_changed(nativePtr, on_subject_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_conference_cbs_set_participant_device_is_speaking_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_conference_cbs_set_participant_device_is_speaking_changed(IntPtr thiz, OnParticipantDeviceIsSpeakingChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnParticipantDeviceIsSpeakingChangedDelegatePrivate(IntPtr conference, IntPtr participantDevice, char isSpeaking);

		/// <summary>
		/// Callback used to notify that a participant device is speaking or isn&apos;t
		/// speaking anymore. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="conference"><see cref="Linphone.Conference">
		/// Linphone.Conference</see> object    </param>
		/// <param name="participantDevice">the participant device    </param>
		/// <param name="isSpeaking">true if is speaking, false otherwise </param>
		public delegate void OnParticipantDeviceIsSpeakingChangedDelegate(Linphone.Conference conference, Linphone.ParticipantDevice participantDevice, bool isSpeaking);
		private OnParticipantDeviceIsSpeakingChangedDelegatePrivate on_participant_device_is_speaking_changed_private;
		private OnParticipantDeviceIsSpeakingChangedDelegate on_participant_device_is_speaking_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnParticipantDeviceIsSpeakingChangedDelegatePrivate))]
#endif
		private static void on_participant_device_is_speaking_changed(IntPtr conference, IntPtr participantDevice, char isSpeaking)
		{
			Conference thiz = fromNativePtr<Conference>(conference);

			IntPtr ptr = linphone_conference_get_current_callbacks(thiz.nativePtr);
			ConferenceListener current_listener = fromNativePtr<ConferenceListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_participant_device_is_speaking_changed_public?.Invoke(thiz, fromNativePtr<Linphone.ParticipantDevice>(participantDevice), isSpeaking == 0);
			}
		}

		public OnParticipantDeviceIsSpeakingChangedDelegate OnParticipantDeviceIsSpeakingChanged
		{
			get
			{
				return on_participant_device_is_speaking_changed_public;
			}
			set
			{
				on_participant_device_is_speaking_changed_public = value;
				on_participant_device_is_speaking_changed_private = on_participant_device_is_speaking_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_participant_device_is_speaking_changed_private);
				linphone_conference_cbs_set_participant_device_is_speaking_changed(nativePtr, cb);
#else
				linphone_conference_cbs_set_participant_device_is_speaking_changed(nativePtr, on_participant_device_is_speaking_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_conference_cbs_set_participant_device_is_muted(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_conference_cbs_set_participant_device_is_muted(IntPtr thiz, OnParticipantDeviceIsMutedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnParticipantDeviceIsMutedDelegatePrivate(IntPtr conference, IntPtr participantDevice, char isMuted);

		/// <summary>
		/// Callback used to notify that a participant device is muted or is no longer
		/// muted. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="conference"><see cref="Linphone.Conference">
		/// Linphone.Conference</see> object    </param>
		/// <param name="participantDevice">the participant device    </param>
		/// <param name="isMuted">true if is muted, false otherwise </param>
		public delegate void OnParticipantDeviceIsMutedDelegate(Linphone.Conference conference, Linphone.ParticipantDevice participantDevice, bool isMuted);
		private OnParticipantDeviceIsMutedDelegatePrivate on_participant_device_is_muted_private;
		private OnParticipantDeviceIsMutedDelegate on_participant_device_is_muted_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnParticipantDeviceIsMutedDelegatePrivate))]
#endif
		private static void on_participant_device_is_muted(IntPtr conference, IntPtr participantDevice, char isMuted)
		{
			Conference thiz = fromNativePtr<Conference>(conference);

			IntPtr ptr = linphone_conference_get_current_callbacks(thiz.nativePtr);
			ConferenceListener current_listener = fromNativePtr<ConferenceListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_participant_device_is_muted_public?.Invoke(thiz, fromNativePtr<Linphone.ParticipantDevice>(participantDevice), isMuted == 0);
			}
		}

		public OnParticipantDeviceIsMutedDelegate OnParticipantDeviceIsMuted
		{
			get
			{
				return on_participant_device_is_muted_public;
			}
			set
			{
				on_participant_device_is_muted_public = value;
				on_participant_device_is_muted_private = on_participant_device_is_muted;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_participant_device_is_muted_private);
				linphone_conference_cbs_set_participant_device_is_muted(nativePtr, cb);
#else
				linphone_conference_cbs_set_participant_device_is_muted(nativePtr, on_participant_device_is_muted_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_conference_cbs_set_audio_device_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_conference_cbs_set_audio_device_changed(IntPtr thiz, OnAudioDeviceChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnAudioDeviceChangedDelegatePrivate(IntPtr conference, IntPtr audioDevice);

		/// <summary>
		/// Callback used to notify that the audio device of a conference has changed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="conference">LinphoneConference object    </param>
		/// <param name="audioDevice">audio device of the conference    </param>
		public delegate void OnAudioDeviceChangedDelegate(Linphone.Conference conference, Linphone.AudioDevice audioDevice);
		private OnAudioDeviceChangedDelegatePrivate on_audio_device_changed_private;
		private OnAudioDeviceChangedDelegate on_audio_device_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnAudioDeviceChangedDelegatePrivate))]
#endif
		private static void on_audio_device_changed(IntPtr conference, IntPtr audioDevice)
		{
			Conference thiz = fromNativePtr<Conference>(conference);

			IntPtr ptr = linphone_conference_get_current_callbacks(thiz.nativePtr);
			ConferenceListener current_listener = fromNativePtr<ConferenceListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_audio_device_changed_public?.Invoke(thiz, fromNativePtr<Linphone.AudioDevice>(audioDevice));
			}
		}

		public OnAudioDeviceChangedDelegate OnAudioDeviceChanged
		{
			get
			{
				return on_audio_device_changed_public;
			}
			set
			{
				on_audio_device_changed_public = value;
				on_audio_device_changed_private = on_audio_device_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_audio_device_changed_private);
				linphone_conference_cbs_set_audio_device_changed(nativePtr, cb);
#else
				linphone_conference_cbs_set_audio_device_changed(nativePtr, on_audio_device_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_conference_cbs_set_active_speaker_participant_device(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_conference_cbs_set_active_speaker_participant_device(IntPtr thiz, OnActiveSpeakerParticipantDeviceDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnActiveSpeakerParticipantDeviceDelegatePrivate(IntPtr conference, IntPtr participantDevice);

		/// <summary>
		/// Callback used to notify which participant device video is being displayed as
		/// &quot;actively speaking&quot;. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="conference"><see cref="Linphone.Conference">
		/// Linphone.Conference</see> object    </param>
		/// <param name="participantDevice">the participant device currently displayed as
		/// active speaker    </param>
		public delegate void OnActiveSpeakerParticipantDeviceDelegate(Linphone.Conference conference, Linphone.ParticipantDevice participantDevice);
		private OnActiveSpeakerParticipantDeviceDelegatePrivate on_active_speaker_participant_device_private;
		private OnActiveSpeakerParticipantDeviceDelegate on_active_speaker_participant_device_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnActiveSpeakerParticipantDeviceDelegatePrivate))]
#endif
		private static void on_active_speaker_participant_device(IntPtr conference, IntPtr participantDevice)
		{
			Conference thiz = fromNativePtr<Conference>(conference);

			IntPtr ptr = linphone_conference_get_current_callbacks(thiz.nativePtr);
			ConferenceListener current_listener = fromNativePtr<ConferenceListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_active_speaker_participant_device_public?.Invoke(thiz, fromNativePtr<Linphone.ParticipantDevice>(participantDevice));
			}
		}

		public OnActiveSpeakerParticipantDeviceDelegate OnActiveSpeakerParticipantDevice
		{
			get
			{
				return on_active_speaker_participant_device_public;
			}
			set
			{
				on_active_speaker_participant_device_public = value;
				on_active_speaker_participant_device_private = on_active_speaker_participant_device;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_active_speaker_participant_device_private);
				linphone_conference_cbs_set_active_speaker_participant_device(nativePtr, cb);
#else
				linphone_conference_cbs_set_active_speaker_participant_device(nativePtr, on_active_speaker_participant_device_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_conference_cbs_set_full_state_received(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_conference_cbs_set_full_state_received(IntPtr thiz, OnFullStateReceivedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnFullStateReceivedDelegatePrivate(IntPtr conference);

		/// <summary>
		/// Callback used to notify when a notify full state has been received. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="conference"><see cref="Linphone.Conference">
		/// Linphone.Conference</see> object    </param>
		public delegate void OnFullStateReceivedDelegate(Linphone.Conference conference);
		private OnFullStateReceivedDelegatePrivate on_full_state_received_private;
		private OnFullStateReceivedDelegate on_full_state_received_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnFullStateReceivedDelegatePrivate))]
#endif
		private static void on_full_state_received(IntPtr conference)
		{
			Conference thiz = fromNativePtr<Conference>(conference);

			IntPtr ptr = linphone_conference_get_current_callbacks(thiz.nativePtr);
			ConferenceListener current_listener = fromNativePtr<ConferenceListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_full_state_received_public?.Invoke(thiz);
			}
		}

		public OnFullStateReceivedDelegate OnFullStateReceived
		{
			get
			{
				return on_full_state_received_public;
			}
			set
			{
				on_full_state_received_public = value;
				on_full_state_received_private = on_full_state_received;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_full_state_received_private);
				linphone_conference_cbs_set_full_state_received(nativePtr, cb);
#else
				linphone_conference_cbs_set_full_state_received(nativePtr, on_full_state_received_private);
#endif
			}
		}
	}

	/// <summary>
	/// Callbacks of <see cref="Linphone.ConferenceScheduler">
	/// Linphone.ConferenceScheduler</see> object. 
	/// <para>
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class ConferenceSchedulerListener : LinphoneObject
	{
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_conference_scheduler_get_current_callbacks(IntPtr thiz);

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_conference_scheduler_cbs_set_state_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_conference_scheduler_cbs_set_state_changed(IntPtr thiz, OnStateChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnStateChangedDelegatePrivate(IntPtr conferenceScheduler, int state);

		/// <summary>
		/// Callback for notifying when a registration state has changed for the conference
		/// scheduler. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="conferenceScheduler">LinphoneConferenceScheduler object whose
		/// state has changed.    </param>
		/// <param name="state">The current LinphoneConferenceSchedulerState. </param>
		public delegate void OnStateChangedDelegate(Linphone.ConferenceScheduler conferenceScheduler, Linphone.ConferenceSchedulerState state);
		private OnStateChangedDelegatePrivate on_state_changed_private;
		private OnStateChangedDelegate on_state_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnStateChangedDelegatePrivate))]
#endif
		private static void on_state_changed(IntPtr conferenceScheduler, int state)
		{
			ConferenceScheduler thiz = fromNativePtr<ConferenceScheduler>(conferenceScheduler);

			IntPtr ptr = linphone_conference_scheduler_get_current_callbacks(thiz.nativePtr);
			ConferenceSchedulerListener current_listener = fromNativePtr<ConferenceSchedulerListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_state_changed_public?.Invoke(thiz, (Linphone.ConferenceSchedulerState)state);
			}
		}

		public OnStateChangedDelegate OnStateChanged
		{
			get
			{
				return on_state_changed_public;
			}
			set
			{
				on_state_changed_public = value;
				on_state_changed_private = on_state_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_state_changed_private);
				linphone_conference_scheduler_cbs_set_state_changed(nativePtr, cb);
#else
				linphone_conference_scheduler_cbs_set_state_changed(nativePtr, on_state_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_conference_scheduler_cbs_set_invitations_sent(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_conference_scheduler_cbs_set_invitations_sent(IntPtr thiz, OnInvitationsSentDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnInvitationsSentDelegatePrivate(IntPtr conferenceScheduler, IntPtr failedInvitations);

		/// <summary>
		/// Callback for notifying when conference invitations have been sent. 
		/// <para>
		/// In case of error for some participants, their addresses will be given as
		/// parameter. 
		/// </para>
		/// </summary>
		/// <param name="conferenceScheduler"><see cref="Linphone.ConferenceScheduler">
		/// Linphone.ConferenceScheduler</see> object whose state has changed.    </param>
		/// <param name="failedInvitations">a list of addresses for which invitation
		/// couldn&apos;t be sent.      </param>
		public delegate void OnInvitationsSentDelegate(Linphone.ConferenceScheduler conferenceScheduler, IEnumerable<Linphone.Address> failedInvitations);
		private OnInvitationsSentDelegatePrivate on_invitations_sent_private;
		private OnInvitationsSentDelegate on_invitations_sent_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnInvitationsSentDelegatePrivate))]
#endif
		private static void on_invitations_sent(IntPtr conferenceScheduler, IntPtr failedInvitations)
		{
			ConferenceScheduler thiz = fromNativePtr<ConferenceScheduler>(conferenceScheduler);

			IntPtr ptr = linphone_conference_scheduler_get_current_callbacks(thiz.nativePtr);
			ConferenceSchedulerListener current_listener = fromNativePtr<ConferenceSchedulerListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_invitations_sent_public?.Invoke(thiz, MarshalBctbxList<Linphone.Address>(failedInvitations));
			}
		}

		public OnInvitationsSentDelegate OnInvitationsSent
		{
			get
			{
				return on_invitations_sent_public;
			}
			set
			{
				on_invitations_sent_public = value;
				on_invitations_sent_private = on_invitations_sent;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_invitations_sent_private);
				linphone_conference_scheduler_cbs_set_invitations_sent(nativePtr, cb);
#else
				linphone_conference_scheduler_cbs_set_invitations_sent(nativePtr, on_invitations_sent_private);
#endif
			}
		}
	}

	/// <summary>
	/// That class holds all the callbacks which are called by <see cref="Linphone.Core">
	/// Linphone.Core</see>. 
	/// <para>
	/// Once created, add your <see cref="Linphone.CoreListener">
	/// Linphone.CoreListener</see> using <see cref="Linphone.Core.AddListener()">
	/// Linphone.Core.AddListener()</see>. Keep a reference on it as long as you need
	/// it. You can use <see cref="Linphone.Core.RemoveListener()">
	/// Linphone.Core.RemoveListener()</see> to remove it but that isn&apos;t mandatory.
	/// The same applies to all listeners in our API. 
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class CoreListener : LinphoneObject
	{
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_core_get_current_callbacks(IntPtr thiz);

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_new_alert_triggered(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_new_alert_triggered(IntPtr thiz, OnNewAlertTriggeredDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnNewAlertTriggeredDelegatePrivate(IntPtr core, IntPtr alert);

		/// <summary>
		/// Callback for notifying about an alert (e.g on Qos) 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="alert"><see cref="Linphone.Alert">Linphone.Alert</see> to notify  
		///  </param>
		public delegate void OnNewAlertTriggeredDelegate(Linphone.Core core, Linphone.Alert alert);
		private OnNewAlertTriggeredDelegatePrivate on_new_alert_triggered_private;
		private OnNewAlertTriggeredDelegate on_new_alert_triggered_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnNewAlertTriggeredDelegatePrivate))]
#endif
		private static void on_new_alert_triggered(IntPtr core, IntPtr alert)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_new_alert_triggered_public?.Invoke(thiz, fromNativePtr<Linphone.Alert>(alert));
			}
		}

		public OnNewAlertTriggeredDelegate OnNewAlertTriggered
		{
			get
			{
				return on_new_alert_triggered_public;
			}
			set
			{
				on_new_alert_triggered_public = value;
				on_new_alert_triggered_private = on_new_alert_triggered;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_new_alert_triggered_private);
				linphone_core_cbs_set_new_alert_triggered(nativePtr, cb);
#else
				linphone_core_cbs_set_new_alert_triggered(nativePtr, on_new_alert_triggered_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_global_state_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_global_state_changed(IntPtr thiz, OnGlobalStateChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnGlobalStateChangedDelegatePrivate(IntPtr core, int state, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string message);

		/// <summary>
		/// Global state notification callback. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core">the <see cref="Linphone.Core">Linphone.Core</see>.   
		/// </param>
		/// <param name="state">the <see cref="Linphone.GlobalState">
		/// Linphone.GlobalState</see> </param>
		/// <param name="message">informational message.    </param>
		public delegate void OnGlobalStateChangedDelegate(Linphone.Core core, Linphone.GlobalState state, string message);
		private OnGlobalStateChangedDelegatePrivate on_global_state_changed_private;
		private OnGlobalStateChangedDelegate on_global_state_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnGlobalStateChangedDelegatePrivate))]
#endif
		private static void on_global_state_changed(IntPtr core, int state, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string message)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_global_state_changed_public?.Invoke(thiz, (Linphone.GlobalState)state, message);
			}
		}

		public OnGlobalStateChangedDelegate OnGlobalStateChanged
		{
			get
			{
				return on_global_state_changed_public;
			}
			set
			{
				on_global_state_changed_public = value;
				on_global_state_changed_private = on_global_state_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_global_state_changed_private);
				linphone_core_cbs_set_global_state_changed(nativePtr, cb);
#else
				linphone_core_cbs_set_global_state_changed(nativePtr, on_global_state_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_registration_state_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_registration_state_changed(IntPtr thiz, OnRegistrationStateChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnRegistrationStateChangedDelegatePrivate(IntPtr core, IntPtr proxyConfig, int state, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string message);

		/// <summary>
		/// Registration state notification callback prototype. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core">the <see cref="Linphone.Core">Linphone.Core</see>    </param>
		/// <param name="proxyConfig">the <see cref="Linphone.ProxyConfig">
		/// Linphone.ProxyConfig</see> which state has changed    </param>
		/// <param name="state">the current <see cref="Linphone.RegistrationState">
		/// Linphone.RegistrationState</see> </param>
		/// <param name="message">a non null informational message about the state   
		/// </param>
		/// deprecated : 06/04/2020 Use LinphoneCoreCbsAccountRegistrationStateChangedCb
		/// instead
		public delegate void OnRegistrationStateChangedDelegate(Linphone.Core core, Linphone.ProxyConfig proxyConfig, Linphone.RegistrationState state, string message);
		private OnRegistrationStateChangedDelegatePrivate on_registration_state_changed_private;
		private OnRegistrationStateChangedDelegate on_registration_state_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnRegistrationStateChangedDelegatePrivate))]
#endif
		private static void on_registration_state_changed(IntPtr core, IntPtr proxyConfig, int state, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string message)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_registration_state_changed_public?.Invoke(thiz, fromNativePtr<Linphone.ProxyConfig>(proxyConfig), (Linphone.RegistrationState)state, message);
			}
		}

		public OnRegistrationStateChangedDelegate OnRegistrationStateChanged
		{
			get
			{
				return on_registration_state_changed_public;
			}
			set
			{
				on_registration_state_changed_public = value;
				on_registration_state_changed_private = on_registration_state_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_registration_state_changed_private);
				linphone_core_cbs_set_registration_state_changed(nativePtr, cb);
#else
				linphone_core_cbs_set_registration_state_changed(nativePtr, on_registration_state_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_conference_info_received(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_conference_info_received(IntPtr thiz, OnConferenceInfoReceivedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnConferenceInfoReceivedDelegatePrivate(IntPtr core, IntPtr conferenceInfo);

		/// <summary>
		/// Callback prototype for notifying the application about a received conference
		/// info. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="conferenceInfo">the <see cref="Linphone.ConferenceInfo">
		/// Linphone.ConferenceInfo</see> received    </param>
		public delegate void OnConferenceInfoReceivedDelegate(Linphone.Core core, Linphone.ConferenceInfo conferenceInfo);
		private OnConferenceInfoReceivedDelegatePrivate on_conference_info_received_private;
		private OnConferenceInfoReceivedDelegate on_conference_info_received_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnConferenceInfoReceivedDelegatePrivate))]
#endif
		private static void on_conference_info_received(IntPtr core, IntPtr conferenceInfo)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_conference_info_received_public?.Invoke(thiz, fromNativePtr<Linphone.ConferenceInfo>(conferenceInfo));
			}
		}

		public OnConferenceInfoReceivedDelegate OnConferenceInfoReceived
		{
			get
			{
				return on_conference_info_received_public;
			}
			set
			{
				on_conference_info_received_public = value;
				on_conference_info_received_private = on_conference_info_received;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_conference_info_received_private);
				linphone_core_cbs_set_conference_info_received(nativePtr, cb);
#else
				linphone_core_cbs_set_conference_info_received(nativePtr, on_conference_info_received_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_push_notification_received(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_push_notification_received(IntPtr thiz, OnPushNotificationReceivedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnPushNotificationReceivedDelegatePrivate(IntPtr core, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string payload);

		/// <summary>
		/// Callback prototype for notifying the application a push notification was
		/// received. 
		/// <para>
		/// On iOS it only works with pushkit (VoIP) pushes. 
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="payload">the body of the push notification, if any    </param>
		public delegate void OnPushNotificationReceivedDelegate(Linphone.Core core, string payload);
		private OnPushNotificationReceivedDelegatePrivate on_push_notification_received_private;
		private OnPushNotificationReceivedDelegate on_push_notification_received_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnPushNotificationReceivedDelegatePrivate))]
#endif
		private static void on_push_notification_received(IntPtr core, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string payload)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_push_notification_received_public?.Invoke(thiz, payload);
			}
		}

		public OnPushNotificationReceivedDelegate OnPushNotificationReceived
		{
			get
			{
				return on_push_notification_received_public;
			}
			set
			{
				on_push_notification_received_public = value;
				on_push_notification_received_private = on_push_notification_received;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_push_notification_received_private);
				linphone_core_cbs_set_push_notification_received(nativePtr, cb);
#else
				linphone_core_cbs_set_push_notification_received(nativePtr, on_push_notification_received_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_preview_display_error_occurred(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_preview_display_error_occurred(IntPtr thiz, OnPreviewDisplayErrorOccurredDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnPreviewDisplayErrorOccurredDelegatePrivate(IntPtr core, int errorCode);

		/// <summary>
		/// Callback to notify that there are errors from the video rendering. 
		/// <para>
		/// Check LinphoneCallCbsVideoDisplayErrorOccurredCb for more details.
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="errorCode">The error code. It depends of the display filter
		/// (available for OpenGL) </param>
		public delegate void OnPreviewDisplayErrorOccurredDelegate(Linphone.Core core, int errorCode);
		private OnPreviewDisplayErrorOccurredDelegatePrivate on_preview_display_error_occurred_private;
		private OnPreviewDisplayErrorOccurredDelegate on_preview_display_error_occurred_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnPreviewDisplayErrorOccurredDelegatePrivate))]
#endif
		private static void on_preview_display_error_occurred(IntPtr core, int errorCode)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_preview_display_error_occurred_public?.Invoke(thiz, errorCode);
			}
		}

		public OnPreviewDisplayErrorOccurredDelegate OnPreviewDisplayErrorOccurred
		{
			get
			{
				return on_preview_display_error_occurred_public;
			}
			set
			{
				on_preview_display_error_occurred_public = value;
				on_preview_display_error_occurred_private = on_preview_display_error_occurred;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_preview_display_error_occurred_private);
				linphone_core_cbs_set_preview_display_error_occurred(nativePtr, cb);
#else
				linphone_core_cbs_set_preview_display_error_occurred(nativePtr, on_preview_display_error_occurred_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_call_state_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_call_state_changed(IntPtr thiz, OnCallStateChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnCallStateChangedDelegatePrivate(IntPtr core, IntPtr call, int state, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string message);

		/// <summary>
		/// Call state notification callback. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core">the <see cref="Linphone.Core">Linphone.Core</see>    </param>
		/// <param name="call">the <see cref="Linphone.Call">Linphone.Call</see> object
		/// whose state is changed.    </param>
		/// <param name="state">the new <see cref="Linphone.CallState">
		/// Linphone.CallState</see> of the call </param>
		/// <param name="message">a non null informational message about the state.   
		/// </param>
		public delegate void OnCallStateChangedDelegate(Linphone.Core core, Linphone.Call call, Linphone.CallState state, string message);
		private OnCallStateChangedDelegatePrivate on_call_state_changed_private;
		private OnCallStateChangedDelegate on_call_state_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnCallStateChangedDelegatePrivate))]
#endif
		private static void on_call_state_changed(IntPtr core, IntPtr call, int state, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string message)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_call_state_changed_public?.Invoke(thiz, fromNativePtr<Linphone.Call>(call), (Linphone.CallState)state, message);
			}
		}

		public OnCallStateChangedDelegate OnCallStateChanged
		{
			get
			{
				return on_call_state_changed_public;
			}
			set
			{
				on_call_state_changed_public = value;
				on_call_state_changed_private = on_call_state_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_call_state_changed_private);
				linphone_core_cbs_set_call_state_changed(nativePtr, cb);
#else
				linphone_core_cbs_set_call_state_changed(nativePtr, on_call_state_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_notify_presence_received(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_notify_presence_received(IntPtr thiz, OnNotifyPresenceReceivedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnNotifyPresenceReceivedDelegatePrivate(IntPtr core, IntPtr linphoneFriend);

		/// <summary>
		/// Report status change for a friend previously added to the <see cref="Linphone.Core">
		/// Linphone.Core</see> with linphone_core_add_friend(). 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="linphoneFriend">Updated <see cref="Linphone.Friend">
		/// Linphone.Friend</see>    </param>
		public delegate void OnNotifyPresenceReceivedDelegate(Linphone.Core core, Linphone.Friend linphoneFriend);
		private OnNotifyPresenceReceivedDelegatePrivate on_notify_presence_received_private;
		private OnNotifyPresenceReceivedDelegate on_notify_presence_received_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnNotifyPresenceReceivedDelegatePrivate))]
#endif
		private static void on_notify_presence_received(IntPtr core, IntPtr linphoneFriend)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_notify_presence_received_public?.Invoke(thiz, fromNativePtr<Linphone.Friend>(linphoneFriend));
			}
		}

		public OnNotifyPresenceReceivedDelegate OnNotifyPresenceReceived
		{
			get
			{
				return on_notify_presence_received_public;
			}
			set
			{
				on_notify_presence_received_public = value;
				on_notify_presence_received_private = on_notify_presence_received;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_notify_presence_received_private);
				linphone_core_cbs_set_notify_presence_received(nativePtr, cb);
#else
				linphone_core_cbs_set_notify_presence_received(nativePtr, on_notify_presence_received_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_notify_presence_received_for_uri_or_tel(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_notify_presence_received_for_uri_or_tel(IntPtr thiz, OnNotifyPresenceReceivedForUriOrTelDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnNotifyPresenceReceivedForUriOrTelDelegatePrivate(IntPtr core, IntPtr linphoneFriend, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string uriOrTel, IntPtr presenceModel);

		/// <summary>
		/// Reports presence model change for a specific URI or phone number of a friend. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="linphoneFriend"><see cref="Linphone.Friend">Linphone.Friend</see>
		/// object    </param>
		/// <param name="uriOrTel">The URI or phone number for which the presence model has
		/// changed    </param>
		/// <param name="presenceModel">The new <see cref="Linphone.PresenceModel">
		/// Linphone.PresenceModel</see>    </param>
		public delegate void OnNotifyPresenceReceivedForUriOrTelDelegate(Linphone.Core core, Linphone.Friend linphoneFriend, string uriOrTel, Linphone.PresenceModel presenceModel);
		private OnNotifyPresenceReceivedForUriOrTelDelegatePrivate on_notify_presence_received_for_uri_or_tel_private;
		private OnNotifyPresenceReceivedForUriOrTelDelegate on_notify_presence_received_for_uri_or_tel_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnNotifyPresenceReceivedForUriOrTelDelegatePrivate))]
#endif
		private static void on_notify_presence_received_for_uri_or_tel(IntPtr core, IntPtr linphoneFriend, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string uriOrTel, IntPtr presenceModel)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_notify_presence_received_for_uri_or_tel_public?.Invoke(thiz, fromNativePtr<Linphone.Friend>(linphoneFriend), uriOrTel, fromNativePtr<Linphone.PresenceModel>(presenceModel));
			}
		}

		public OnNotifyPresenceReceivedForUriOrTelDelegate OnNotifyPresenceReceivedForUriOrTel
		{
			get
			{
				return on_notify_presence_received_for_uri_or_tel_public;
			}
			set
			{
				on_notify_presence_received_for_uri_or_tel_public = value;
				on_notify_presence_received_for_uri_or_tel_private = on_notify_presence_received_for_uri_or_tel;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_notify_presence_received_for_uri_or_tel_private);
				linphone_core_cbs_set_notify_presence_received_for_uri_or_tel(nativePtr, cb);
#else
				linphone_core_cbs_set_notify_presence_received_for_uri_or_tel(nativePtr, on_notify_presence_received_for_uri_or_tel_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_new_subscription_requested(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_new_subscription_requested(IntPtr thiz, OnNewSubscriptionRequestedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnNewSubscriptionRequestedDelegatePrivate(IntPtr core, IntPtr linphoneFriend, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string url);

		/// <summary>
		/// Reports that a new subscription request has been received and wait for a
		/// decision. 
		/// <para>
		/// </para>
		/// </summary>
		/// <remarks>Note : A subscription request is notified by this function only if the
		/// <see cref="Linphone.SubscribePolicy">Linphone.SubscribePolicy</see> for the
		/// given <see cref="Linphone.Friend">Linphone.Friend</see> has been set to <see cref="Linphone.SubscribePolicy.SPWait">
		/// Linphone.SubscribePolicy.SPWait</see>. See <see cref="Linphone.Friend.IncSubscribePolicy">
		/// Linphone.Friend.IncSubscribePolicy</see>. </remarks>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="linphoneFriend">The <see cref="Linphone.Friend">
		/// Linphone.Friend</see> aimed by the subscription.    </param>
		/// <param name="url">URI of the subscriber    </param>
		public delegate void OnNewSubscriptionRequestedDelegate(Linphone.Core core, Linphone.Friend linphoneFriend, string url);
		private OnNewSubscriptionRequestedDelegatePrivate on_new_subscription_requested_private;
		private OnNewSubscriptionRequestedDelegate on_new_subscription_requested_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnNewSubscriptionRequestedDelegatePrivate))]
#endif
		private static void on_new_subscription_requested(IntPtr core, IntPtr linphoneFriend, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string url)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_new_subscription_requested_public?.Invoke(thiz, fromNativePtr<Linphone.Friend>(linphoneFriend), url);
			}
		}

		public OnNewSubscriptionRequestedDelegate OnNewSubscriptionRequested
		{
			get
			{
				return on_new_subscription_requested_public;
			}
			set
			{
				on_new_subscription_requested_public = value;
				on_new_subscription_requested_private = on_new_subscription_requested;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_new_subscription_requested_private);
				linphone_core_cbs_set_new_subscription_requested(nativePtr, cb);
#else
				linphone_core_cbs_set_new_subscription_requested(nativePtr, on_new_subscription_requested_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_authentication_requested(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_authentication_requested(IntPtr thiz, OnAuthenticationRequestedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnAuthenticationRequestedDelegatePrivate(IntPtr core, IntPtr authInfo, int method);

		/// <summary>
		/// Callback for requesting authentication information to application or user. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core">the <see cref="Linphone.Core">Linphone.Core</see>    </param>
		/// <param name="authInfo">a <see cref="Linphone.AuthInfo">Linphone.AuthInfo</see>
		/// pre-filled with username, realm and domain values as much as possible   
		/// </param>
		/// <param name="method">the type of authentication requested as <see cref="Linphone.AuthMethod">
		/// Linphone.AuthMethod</see> enum    Application shall reply to this callback
		/// using <see cref="Linphone.Core.AddAuthInfo()">
		/// Linphone.Core.AddAuthInfo()</see>. </param>
		public delegate void OnAuthenticationRequestedDelegate(Linphone.Core core, Linphone.AuthInfo authInfo, Linphone.AuthMethod method);
		private OnAuthenticationRequestedDelegatePrivate on_authentication_requested_private;
		private OnAuthenticationRequestedDelegate on_authentication_requested_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnAuthenticationRequestedDelegatePrivate))]
#endif
		private static void on_authentication_requested(IntPtr core, IntPtr authInfo, int method)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_authentication_requested_public?.Invoke(thiz, fromNativePtr<Linphone.AuthInfo>(authInfo), (Linphone.AuthMethod)method);
			}
		}

		public OnAuthenticationRequestedDelegate OnAuthenticationRequested
		{
			get
			{
				return on_authentication_requested_public;
			}
			set
			{
				on_authentication_requested_public = value;
				on_authentication_requested_private = on_authentication_requested;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_authentication_requested_private);
				linphone_core_cbs_set_authentication_requested(nativePtr, cb);
#else
				linphone_core_cbs_set_authentication_requested(nativePtr, on_authentication_requested_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_call_log_updated(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_call_log_updated(IntPtr thiz, OnCallLogUpdatedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnCallLogUpdatedDelegatePrivate(IntPtr core, IntPtr callLog);

		/// <summary>
		/// Callback to notify a new call-log entry has been added. 
		/// <para>
		/// This is done typically when a call terminates. 
		/// </para>
		/// </summary>
		/// <param name="core">the <see cref="Linphone.Core">Linphone.Core</see>    </param>
		/// <param name="callLog">the new <see cref="Linphone.CallLog">
		/// Linphone.CallLog</see> entry added.    </param>
		public delegate void OnCallLogUpdatedDelegate(Linphone.Core core, Linphone.CallLog callLog);
		private OnCallLogUpdatedDelegatePrivate on_call_log_updated_private;
		private OnCallLogUpdatedDelegate on_call_log_updated_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnCallLogUpdatedDelegatePrivate))]
#endif
		private static void on_call_log_updated(IntPtr core, IntPtr callLog)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_call_log_updated_public?.Invoke(thiz, fromNativePtr<Linphone.CallLog>(callLog));
			}
		}

		public OnCallLogUpdatedDelegate OnCallLogUpdated
		{
			get
			{
				return on_call_log_updated_public;
			}
			set
			{
				on_call_log_updated_public = value;
				on_call_log_updated_private = on_call_log_updated;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_call_log_updated_private);
				linphone_core_cbs_set_call_log_updated(nativePtr, cb);
#else
				linphone_core_cbs_set_call_log_updated(nativePtr, on_call_log_updated_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_call_id_updated(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_call_id_updated(IntPtr thiz, OnCallIdUpdatedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnCallIdUpdatedDelegatePrivate(IntPtr core, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string previousCallId, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string currentCallId);

		/// <summary>
		/// Callback to notify the callid of a call has been updated. 
		/// <para>
		/// This is done typically when a call retry. 
		/// </para>
		/// </summary>
		/// <param name="core">the <see cref="Linphone.Core">Linphone.Core</see>    </param>
		/// <param name="previousCallId">the previous callid.    </param>
		/// <param name="currentCallId">the new callid.    </param>
		public delegate void OnCallIdUpdatedDelegate(Linphone.Core core, string previousCallId, string currentCallId);
		private OnCallIdUpdatedDelegatePrivate on_call_id_updated_private;
		private OnCallIdUpdatedDelegate on_call_id_updated_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnCallIdUpdatedDelegatePrivate))]
#endif
		private static void on_call_id_updated(IntPtr core, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string previousCallId, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string currentCallId)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_call_id_updated_public?.Invoke(thiz, previousCallId, currentCallId);
			}
		}

		public OnCallIdUpdatedDelegate OnCallIdUpdated
		{
			get
			{
				return on_call_id_updated_public;
			}
			set
			{
				on_call_id_updated_public = value;
				on_call_id_updated_private = on_call_id_updated;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_call_id_updated_private);
				linphone_core_cbs_set_call_id_updated(nativePtr, cb);
#else
				linphone_core_cbs_set_call_id_updated(nativePtr, on_call_id_updated_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_remaining_number_of_file_transfer_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_remaining_number_of_file_transfer_changed(IntPtr thiz, OnRemainingNumberOfFileTransferChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnRemainingNumberOfFileTransferChangedDelegatePrivate(IntPtr core, uint downloadCount, uint uploadCount);

		/// <summary>
		/// Called after a download is terminated or a new attachement is to be downloaded. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="downloadCount">outstanding number of files to be downloaded
		/// including the one that is currently downloading </param>
		/// <param name="uploadCount">outstanding number of files to be upload including
		/// the one that is currently uploading </param>
		public delegate void OnRemainingNumberOfFileTransferChangedDelegate(Linphone.Core core, uint downloadCount, uint uploadCount);
		private OnRemainingNumberOfFileTransferChangedDelegatePrivate on_remaining_number_of_file_transfer_changed_private;
		private OnRemainingNumberOfFileTransferChangedDelegate on_remaining_number_of_file_transfer_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnRemainingNumberOfFileTransferChangedDelegatePrivate))]
#endif
		private static void on_remaining_number_of_file_transfer_changed(IntPtr core, uint downloadCount, uint uploadCount)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_remaining_number_of_file_transfer_changed_public?.Invoke(thiz, downloadCount, uploadCount);
			}
		}

		public OnRemainingNumberOfFileTransferChangedDelegate OnRemainingNumberOfFileTransferChanged
		{
			get
			{
				return on_remaining_number_of_file_transfer_changed_public;
			}
			set
			{
				on_remaining_number_of_file_transfer_changed_public = value;
				on_remaining_number_of_file_transfer_changed_private = on_remaining_number_of_file_transfer_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_remaining_number_of_file_transfer_changed_private);
				linphone_core_cbs_set_remaining_number_of_file_transfer_changed(nativePtr, cb);
#else
				linphone_core_cbs_set_remaining_number_of_file_transfer_changed(nativePtr, on_remaining_number_of_file_transfer_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_message_received(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_message_received(IntPtr thiz, OnMessageReceivedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnMessageReceivedDelegatePrivate(IntPtr core, IntPtr chatRoom, IntPtr message);

		/// <summary>
		/// Chat message callback prototype. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// involved in this conversation. Can be created by the framework in case the
		/// From-URI is not present in any chat room.    </param>
		/// <param name="message"><see cref="Linphone.ChatMessage">
		/// Linphone.ChatMessage</see> incoming message    </param>
		public delegate void OnMessageReceivedDelegate(Linphone.Core core, Linphone.ChatRoom chatRoom, Linphone.ChatMessage message);
		private OnMessageReceivedDelegatePrivate on_message_received_private;
		private OnMessageReceivedDelegate on_message_received_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnMessageReceivedDelegatePrivate))]
#endif
		private static void on_message_received(IntPtr core, IntPtr chatRoom, IntPtr message)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_message_received_public?.Invoke(thiz, fromNativePtr<Linphone.ChatRoom>(chatRoom), fromNativePtr<Linphone.ChatMessage>(message));
			}
		}

		public OnMessageReceivedDelegate OnMessageReceived
		{
			get
			{
				return on_message_received_public;
			}
			set
			{
				on_message_received_public = value;
				on_message_received_private = on_message_received;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_message_received_private);
				linphone_core_cbs_set_message_received(nativePtr, cb);
#else
				linphone_core_cbs_set_message_received(nativePtr, on_message_received_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_new_message_reaction(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_new_message_reaction(IntPtr thiz, OnNewMessageReactionDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnNewMessageReactionDelegatePrivate(IntPtr core, IntPtr chatRoom, IntPtr message, IntPtr reaction);

		/// <summary>
		/// Chat message new reaction callback prototype. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// involved in this conversation. Can be created by the framework in case the
		/// From-URI is not present in any chat room.    </param>
		/// <param name="message">the <see cref="Linphone.ChatMessage">
		/// Linphone.ChatMessage</see> to which the reaction applies to    </param>
		/// <param name="reaction">the <see cref="Linphone.ChatMessageReaction">
		/// Linphone.ChatMessageReaction</see> that has been sent or received    </param>
		public delegate void OnNewMessageReactionDelegate(Linphone.Core core, Linphone.ChatRoom chatRoom, Linphone.ChatMessage message, Linphone.ChatMessageReaction reaction);
		private OnNewMessageReactionDelegatePrivate on_new_message_reaction_private;
		private OnNewMessageReactionDelegate on_new_message_reaction_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnNewMessageReactionDelegatePrivate))]
#endif
		private static void on_new_message_reaction(IntPtr core, IntPtr chatRoom, IntPtr message, IntPtr reaction)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_new_message_reaction_public?.Invoke(thiz, fromNativePtr<Linphone.ChatRoom>(chatRoom), fromNativePtr<Linphone.ChatMessage>(message), fromNativePtr<Linphone.ChatMessageReaction>(reaction));
			}
		}

		public OnNewMessageReactionDelegate OnNewMessageReaction
		{
			get
			{
				return on_new_message_reaction_public;
			}
			set
			{
				on_new_message_reaction_public = value;
				on_new_message_reaction_private = on_new_message_reaction;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_new_message_reaction_private);
				linphone_core_cbs_set_new_message_reaction(nativePtr, cb);
#else
				linphone_core_cbs_set_new_message_reaction(nativePtr, on_new_message_reaction_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_reaction_removed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_reaction_removed(IntPtr thiz, OnReactionRemovedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnReactionRemovedDelegatePrivate(IntPtr core, IntPtr chatRoom, IntPtr message, IntPtr address);

		/// <summary>
		/// Chat message removed reaction callback prototype. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// involved in this conversation. Can be created by the framework in case the
		/// From-URI is not present in any chat room.    </param>
		/// <param name="message">the <see cref="Linphone.ChatMessage">
		/// Linphone.ChatMessage</see> to which a reaction has been removed from    </param>
		/// <param name="address">the <see cref="Linphone.Address">Linphone.Address</see>
		/// of the person that removed it&apos;s reaction    </param>
		public delegate void OnReactionRemovedDelegate(Linphone.Core core, Linphone.ChatRoom chatRoom, Linphone.ChatMessage message, Linphone.Address address);
		private OnReactionRemovedDelegatePrivate on_reaction_removed_private;
		private OnReactionRemovedDelegate on_reaction_removed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnReactionRemovedDelegatePrivate))]
#endif
		private static void on_reaction_removed(IntPtr core, IntPtr chatRoom, IntPtr message, IntPtr address)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_reaction_removed_public?.Invoke(thiz, fromNativePtr<Linphone.ChatRoom>(chatRoom), fromNativePtr<Linphone.ChatMessage>(message), fromNativePtr<Linphone.Address>(address));
			}
		}

		public OnReactionRemovedDelegate OnReactionRemoved
		{
			get
			{
				return on_reaction_removed_public;
			}
			set
			{
				on_reaction_removed_public = value;
				on_reaction_removed_private = on_reaction_removed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_reaction_removed_private);
				linphone_core_cbs_set_reaction_removed(nativePtr, cb);
#else
				linphone_core_cbs_set_reaction_removed(nativePtr, on_reaction_removed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_messages_received(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_messages_received(IntPtr thiz, OnMessagesReceivedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnMessagesReceivedDelegatePrivate(IntPtr core, IntPtr chatRoom, IntPtr messages);

		/// <summary>
		/// Chat messages callback prototype. 
		/// <para>
		/// Only called when aggregation is enabled (aka [sip] chat_messages_aggregation ==
		/// 1 or using <see cref="Linphone.Core.ChatMessagesAggregationEnabled">
		/// Linphone.Core.ChatMessagesAggregationEnabled</see>), it replaces the single
		/// message received callback. 
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// involved in this conversation. Can be created by the framework in case the
		/// From-URI is not present in any chat room.    </param>
		/// <param name="messages">The   of incoming messages    </param>
		public delegate void OnMessagesReceivedDelegate(Linphone.Core core, Linphone.ChatRoom chatRoom, IEnumerable<Linphone.ChatMessage> messages);
		private OnMessagesReceivedDelegatePrivate on_messages_received_private;
		private OnMessagesReceivedDelegate on_messages_received_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnMessagesReceivedDelegatePrivate))]
#endif
		private static void on_messages_received(IntPtr core, IntPtr chatRoom, IntPtr messages)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_messages_received_public?.Invoke(thiz, fromNativePtr<Linphone.ChatRoom>(chatRoom), MarshalBctbxList<Linphone.ChatMessage>(messages));
			}
		}

		public OnMessagesReceivedDelegate OnMessagesReceived
		{
			get
			{
				return on_messages_received_public;
			}
			set
			{
				on_messages_received_public = value;
				on_messages_received_private = on_messages_received;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_messages_received_private);
				linphone_core_cbs_set_messages_received(nativePtr, cb);
#else
				linphone_core_cbs_set_messages_received(nativePtr, on_messages_received_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_message_sent(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_message_sent(IntPtr thiz, OnMessageSentDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnMessageSentDelegatePrivate(IntPtr core, IntPtr chatRoom, IntPtr message);

		/// <summary>
		/// Called after the <see cref="Linphone.ChatMessage.Send()">
		/// Linphone.ChatMessage.Send()</see> was called. 
		/// <para>
		/// The message will be in state InProgress. In case of resend this callback
		/// won&apos;t be called. 
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// involved in this conversation. Can be be created by the framework in case the
		/// From-URI is not present in any chat room.    </param>
		/// <param name="message"><see cref="Linphone.ChatMessage">
		/// Linphone.ChatMessage</see> outgoing message    </param>
		public delegate void OnMessageSentDelegate(Linphone.Core core, Linphone.ChatRoom chatRoom, Linphone.ChatMessage message);
		private OnMessageSentDelegatePrivate on_message_sent_private;
		private OnMessageSentDelegate on_message_sent_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnMessageSentDelegatePrivate))]
#endif
		private static void on_message_sent(IntPtr core, IntPtr chatRoom, IntPtr message)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_message_sent_public?.Invoke(thiz, fromNativePtr<Linphone.ChatRoom>(chatRoom), fromNativePtr<Linphone.ChatMessage>(message));
			}
		}

		public OnMessageSentDelegate OnMessageSent
		{
			get
			{
				return on_message_sent_public;
			}
			set
			{
				on_message_sent_public = value;
				on_message_sent_private = on_message_sent;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_message_sent_private);
				linphone_core_cbs_set_message_sent(nativePtr, cb);
#else
				linphone_core_cbs_set_message_sent(nativePtr, on_message_sent_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_chat_room_session_state_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_chat_room_session_state_changed(IntPtr thiz, OnChatRoomSessionStateChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnChatRoomSessionStateChangedDelegatePrivate(IntPtr core, IntPtr chatRoom, int state, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string message);

		/// <summary>
		/// Chat room session state changed callback. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// that has been marked as read.    </param>
		/// <param name="state">the new <see cref="Linphone.CallState">
		/// Linphone.CallState</see> of the call </param>
		/// <param name="message">a non null informational message about the state.   
		/// </param>
		public delegate void OnChatRoomSessionStateChangedDelegate(Linphone.Core core, Linphone.ChatRoom chatRoom, Linphone.CallState state, string message);
		private OnChatRoomSessionStateChangedDelegatePrivate on_chat_room_session_state_changed_private;
		private OnChatRoomSessionStateChangedDelegate on_chat_room_session_state_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnChatRoomSessionStateChangedDelegatePrivate))]
#endif
		private static void on_chat_room_session_state_changed(IntPtr core, IntPtr chatRoom, int state, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string message)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_chat_room_session_state_changed_public?.Invoke(thiz, fromNativePtr<Linphone.ChatRoom>(chatRoom), (Linphone.CallState)state, message);
			}
		}

		public OnChatRoomSessionStateChangedDelegate OnChatRoomSessionStateChanged
		{
			get
			{
				return on_chat_room_session_state_changed_public;
			}
			set
			{
				on_chat_room_session_state_changed_public = value;
				on_chat_room_session_state_changed_private = on_chat_room_session_state_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_chat_room_session_state_changed_private);
				linphone_core_cbs_set_chat_room_session_state_changed(nativePtr, cb);
#else
				linphone_core_cbs_set_chat_room_session_state_changed(nativePtr, on_chat_room_session_state_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_chat_room_read(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_chat_room_read(IntPtr thiz, OnChatRoomReadDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnChatRoomReadDelegatePrivate(IntPtr core, IntPtr chatRoom);

		/// <summary>
		/// Chat room marked as read callback. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// that has been marked as read.    </param>
		public delegate void OnChatRoomReadDelegate(Linphone.Core core, Linphone.ChatRoom chatRoom);
		private OnChatRoomReadDelegatePrivate on_chat_room_read_private;
		private OnChatRoomReadDelegate on_chat_room_read_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnChatRoomReadDelegatePrivate))]
#endif
		private static void on_chat_room_read(IntPtr core, IntPtr chatRoom)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_chat_room_read_public?.Invoke(thiz, fromNativePtr<Linphone.ChatRoom>(chatRoom));
			}
		}

		public OnChatRoomReadDelegate OnChatRoomRead
		{
			get
			{
				return on_chat_room_read_public;
			}
			set
			{
				on_chat_room_read_public = value;
				on_chat_room_read_private = on_chat_room_read;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_chat_room_read_private);
				linphone_core_cbs_set_chat_room_read(nativePtr, cb);
#else
				linphone_core_cbs_set_chat_room_read(nativePtr, on_chat_room_read_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_message_received_unable_decrypt(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_message_received_unable_decrypt(IntPtr thiz, OnMessageReceivedUnableDecryptDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnMessageReceivedUnableDecryptDelegatePrivate(IntPtr core, IntPtr chatRoom, IntPtr message);

		/// <summary>
		/// Chat message not decrypted callback prototype. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// involved in this conversation. Can be be created by the framework in case the
		/// from-URI is not present in any chat room.    </param>
		/// <param name="message"><see cref="Linphone.ChatMessage">
		/// Linphone.ChatMessage</see> incoming message    </param>
		public delegate void OnMessageReceivedUnableDecryptDelegate(Linphone.Core core, Linphone.ChatRoom chatRoom, Linphone.ChatMessage message);
		private OnMessageReceivedUnableDecryptDelegatePrivate on_message_received_unable_decrypt_private;
		private OnMessageReceivedUnableDecryptDelegate on_message_received_unable_decrypt_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnMessageReceivedUnableDecryptDelegatePrivate))]
#endif
		private static void on_message_received_unable_decrypt(IntPtr core, IntPtr chatRoom, IntPtr message)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_message_received_unable_decrypt_public?.Invoke(thiz, fromNativePtr<Linphone.ChatRoom>(chatRoom), fromNativePtr<Linphone.ChatMessage>(message));
			}
		}

		public OnMessageReceivedUnableDecryptDelegate OnMessageReceivedUnableDecrypt
		{
			get
			{
				return on_message_received_unable_decrypt_public;
			}
			set
			{
				on_message_received_unable_decrypt_public = value;
				on_message_received_unable_decrypt_private = on_message_received_unable_decrypt;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_message_received_unable_decrypt_private);
				linphone_core_cbs_set_message_received_unable_decrypt(nativePtr, cb);
#else
				linphone_core_cbs_set_message_received_unable_decrypt(nativePtr, on_message_received_unable_decrypt_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_is_composing_received(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_is_composing_received(IntPtr thiz, OnIsComposingReceivedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnIsComposingReceivedDelegatePrivate(IntPtr core, IntPtr chatRoom);

		/// <summary>
		/// Is composing notification callback prototype. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="chatRoom"><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see>
		/// involved in the conversation.    </param>
		public delegate void OnIsComposingReceivedDelegate(Linphone.Core core, Linphone.ChatRoom chatRoom);
		private OnIsComposingReceivedDelegatePrivate on_is_composing_received_private;
		private OnIsComposingReceivedDelegate on_is_composing_received_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnIsComposingReceivedDelegatePrivate))]
#endif
		private static void on_is_composing_received(IntPtr core, IntPtr chatRoom)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_is_composing_received_public?.Invoke(thiz, fromNativePtr<Linphone.ChatRoom>(chatRoom));
			}
		}

		public OnIsComposingReceivedDelegate OnIsComposingReceived
		{
			get
			{
				return on_is_composing_received_public;
			}
			set
			{
				on_is_composing_received_public = value;
				on_is_composing_received_private = on_is_composing_received;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_is_composing_received_private);
				linphone_core_cbs_set_is_composing_received(nativePtr, cb);
#else
				linphone_core_cbs_set_is_composing_received(nativePtr, on_is_composing_received_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_dtmf_received(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_dtmf_received(IntPtr thiz, OnDtmfReceivedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnDtmfReceivedDelegatePrivate(IntPtr core, IntPtr call, int dtmf);

		/// <summary>
		/// Callback for being notified of DTMFs received. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core">the <see cref="Linphone.Core">Linphone.Core</see>    </param>
		/// <param name="call">the <see cref="Linphone.Call">Linphone.Call</see> that
		/// received the dtmf    </param>
		/// <param name="dtmf">the ascii code of the dtmf </param>
		public delegate void OnDtmfReceivedDelegate(Linphone.Core core, Linphone.Call call, int dtmf);
		private OnDtmfReceivedDelegatePrivate on_dtmf_received_private;
		private OnDtmfReceivedDelegate on_dtmf_received_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnDtmfReceivedDelegatePrivate))]
#endif
		private static void on_dtmf_received(IntPtr core, IntPtr call, int dtmf)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_dtmf_received_public?.Invoke(thiz, fromNativePtr<Linphone.Call>(call), dtmf);
			}
		}

		public OnDtmfReceivedDelegate OnDtmfReceived
		{
			get
			{
				return on_dtmf_received_public;
			}
			set
			{
				on_dtmf_received_public = value;
				on_dtmf_received_private = on_dtmf_received;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_dtmf_received_private);
				linphone_core_cbs_set_dtmf_received(nativePtr, cb);
#else
				linphone_core_cbs_set_dtmf_received(nativePtr, on_dtmf_received_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_refer_received(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_refer_received(IntPtr thiz, OnReferReceivedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnReferReceivedDelegatePrivate(IntPtr core, IntPtr referToAddr, IntPtr customHeaders, IntPtr content);

		/// <summary>
		/// Callback prototype for when a refer is received. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core">the <see cref="Linphone.Core">Linphone.Core</see>    </param>
		/// <param name="referToAddr">the address of the refer    </param>
		/// <param name="customHeaders">the headers of the received REFER message   
		/// </param>
		/// <param name="content">the content of the refer    </param>
		public delegate void OnReferReceivedDelegate(Linphone.Core core, Linphone.Address referToAddr, Linphone.Headers customHeaders, Linphone.Content content);
		private OnReferReceivedDelegatePrivate on_refer_received_private;
		private OnReferReceivedDelegate on_refer_received_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnReferReceivedDelegatePrivate))]
#endif
		private static void on_refer_received(IntPtr core, IntPtr referToAddr, IntPtr customHeaders, IntPtr content)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_refer_received_public?.Invoke(thiz, fromNativePtr<Linphone.Address>(referToAddr), fromNativePtr<Linphone.Headers>(customHeaders), fromNativePtr<Linphone.Content>(content));
			}
		}

		public OnReferReceivedDelegate OnReferReceived
		{
			get
			{
				return on_refer_received_public;
			}
			set
			{
				on_refer_received_public = value;
				on_refer_received_private = on_refer_received;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_refer_received_private);
				linphone_core_cbs_set_refer_received(nativePtr, cb);
#else
				linphone_core_cbs_set_refer_received(nativePtr, on_refer_received_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_call_goclear_ack_sent(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_call_goclear_ack_sent(IntPtr thiz, OnCallGoclearAckSentDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnCallGoclearAckSentDelegatePrivate(IntPtr core, IntPtr call);

		/// <summary>
		/// GoClear ACK sent on call callback. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core">the <see cref="Linphone.Core">Linphone.Core</see>    </param>
		/// <param name="call">the <see cref="Linphone.Call">Linphone.Call</see> on which
		/// the GoClear ACK was sent.    </param>
		public delegate void OnCallGoclearAckSentDelegate(Linphone.Core core, Linphone.Call call);
		private OnCallGoclearAckSentDelegatePrivate on_call_goclear_ack_sent_private;
		private OnCallGoclearAckSentDelegate on_call_goclear_ack_sent_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnCallGoclearAckSentDelegatePrivate))]
#endif
		private static void on_call_goclear_ack_sent(IntPtr core, IntPtr call)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_call_goclear_ack_sent_public?.Invoke(thiz, fromNativePtr<Linphone.Call>(call));
			}
		}

		public OnCallGoclearAckSentDelegate OnCallGoclearAckSent
		{
			get
			{
				return on_call_goclear_ack_sent_public;
			}
			set
			{
				on_call_goclear_ack_sent_public = value;
				on_call_goclear_ack_sent_private = on_call_goclear_ack_sent;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_call_goclear_ack_sent_private);
				linphone_core_cbs_set_call_goclear_ack_sent(nativePtr, cb);
#else
				linphone_core_cbs_set_call_goclear_ack_sent(nativePtr, on_call_goclear_ack_sent_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_call_encryption_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_call_encryption_changed(IntPtr thiz, OnCallEncryptionChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnCallEncryptionChangedDelegatePrivate(IntPtr core, IntPtr call, char mediaEncryptionEnabled, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string authenticationToken);

		/// <summary>
		/// Call encryption changed callback. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core">the <see cref="Linphone.Core">Linphone.Core</see>    </param>
		/// <param name="call">the <see cref="Linphone.Call">Linphone.Call</see> on which
		/// encryption is changed.    </param>
		/// <param name="mediaEncryptionEnabled">whether encryption is activated. </param>
		/// <param name="authenticationToken">an authentication token, currently set for
		/// ZRTP kind of encryption only.    </param>
		public delegate void OnCallEncryptionChangedDelegate(Linphone.Core core, Linphone.Call call, bool mediaEncryptionEnabled, string authenticationToken);
		private OnCallEncryptionChangedDelegatePrivate on_call_encryption_changed_private;
		private OnCallEncryptionChangedDelegate on_call_encryption_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnCallEncryptionChangedDelegatePrivate))]
#endif
		private static void on_call_encryption_changed(IntPtr core, IntPtr call, char mediaEncryptionEnabled, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string authenticationToken)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_call_encryption_changed_public?.Invoke(thiz, fromNativePtr<Linphone.Call>(call), mediaEncryptionEnabled == 0, authenticationToken);
			}
		}

		public OnCallEncryptionChangedDelegate OnCallEncryptionChanged
		{
			get
			{
				return on_call_encryption_changed_public;
			}
			set
			{
				on_call_encryption_changed_public = value;
				on_call_encryption_changed_private = on_call_encryption_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_call_encryption_changed_private);
				linphone_core_cbs_set_call_encryption_changed(nativePtr, cb);
#else
				linphone_core_cbs_set_call_encryption_changed(nativePtr, on_call_encryption_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_call_send_master_key_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_call_send_master_key_changed(IntPtr thiz, OnCallSendMasterKeyChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnCallSendMasterKeyChangedDelegatePrivate(IntPtr core, IntPtr call, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string masterKey);

		/// <summary>
		/// Call send master key changed callback. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core">the <see cref="Linphone.Core">Linphone.Core</see>    </param>
		/// <param name="call">the <see cref="Linphone.Call">Linphone.Call</see> on which
		/// the GoClear ACK was sent.    </param>
		/// <param name="masterKey">new master key.    </param>
		public delegate void OnCallSendMasterKeyChangedDelegate(Linphone.Core core, Linphone.Call call, string masterKey);
		private OnCallSendMasterKeyChangedDelegatePrivate on_call_send_master_key_changed_private;
		private OnCallSendMasterKeyChangedDelegate on_call_send_master_key_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnCallSendMasterKeyChangedDelegatePrivate))]
#endif
		private static void on_call_send_master_key_changed(IntPtr core, IntPtr call, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string masterKey)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_call_send_master_key_changed_public?.Invoke(thiz, fromNativePtr<Linphone.Call>(call), masterKey);
			}
		}

		public OnCallSendMasterKeyChangedDelegate OnCallSendMasterKeyChanged
		{
			get
			{
				return on_call_send_master_key_changed_public;
			}
			set
			{
				on_call_send_master_key_changed_public = value;
				on_call_send_master_key_changed_private = on_call_send_master_key_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_call_send_master_key_changed_private);
				linphone_core_cbs_set_call_send_master_key_changed(nativePtr, cb);
#else
				linphone_core_cbs_set_call_send_master_key_changed(nativePtr, on_call_send_master_key_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_call_receive_master_key_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_call_receive_master_key_changed(IntPtr thiz, OnCallReceiveMasterKeyChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnCallReceiveMasterKeyChangedDelegatePrivate(IntPtr core, IntPtr call, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string masterKey);

		/// <summary>
		/// Call receive master key changed callback. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core">the <see cref="Linphone.Core">Linphone.Core</see>    </param>
		/// <param name="call">the <see cref="Linphone.Call">Linphone.Call</see> on which
		/// the GoClear ACK was sent.    </param>
		/// <param name="masterKey">new master key.    </param>
		public delegate void OnCallReceiveMasterKeyChangedDelegate(Linphone.Core core, Linphone.Call call, string masterKey);
		private OnCallReceiveMasterKeyChangedDelegatePrivate on_call_receive_master_key_changed_private;
		private OnCallReceiveMasterKeyChangedDelegate on_call_receive_master_key_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnCallReceiveMasterKeyChangedDelegatePrivate))]
#endif
		private static void on_call_receive_master_key_changed(IntPtr core, IntPtr call, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string masterKey)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_call_receive_master_key_changed_public?.Invoke(thiz, fromNativePtr<Linphone.Call>(call), masterKey);
			}
		}

		public OnCallReceiveMasterKeyChangedDelegate OnCallReceiveMasterKeyChanged
		{
			get
			{
				return on_call_receive_master_key_changed_public;
			}
			set
			{
				on_call_receive_master_key_changed_public = value;
				on_call_receive_master_key_changed_private = on_call_receive_master_key_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_call_receive_master_key_changed_private);
				linphone_core_cbs_set_call_receive_master_key_changed(nativePtr, cb);
#else
				linphone_core_cbs_set_call_receive_master_key_changed(nativePtr, on_call_receive_master_key_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_transfer_state_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_transfer_state_changed(IntPtr thiz, OnTransferStateChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnTransferStateChangedDelegatePrivate(IntPtr core, IntPtr transferred, int callState);

		/// <summary>
		/// Callback for notifying progresses of transfers. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core">the <see cref="Linphone.Core">Linphone.Core</see>    </param>
		/// <param name="transferred">the <see cref="Linphone.Call">Linphone.Call</see>
		/// that was transferred    </param>
		/// <param name="callState">the <see cref="Linphone.CallState">
		/// Linphone.CallState</see> of the call to transfer target at the far end. </param>
		public delegate void OnTransferStateChangedDelegate(Linphone.Core core, Linphone.Call transferred, Linphone.CallState callState);
		private OnTransferStateChangedDelegatePrivate on_transfer_state_changed_private;
		private OnTransferStateChangedDelegate on_transfer_state_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnTransferStateChangedDelegatePrivate))]
#endif
		private static void on_transfer_state_changed(IntPtr core, IntPtr transferred, int callState)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_transfer_state_changed_public?.Invoke(thiz, fromNativePtr<Linphone.Call>(transferred), (Linphone.CallState)callState);
			}
		}

		public OnTransferStateChangedDelegate OnTransferStateChanged
		{
			get
			{
				return on_transfer_state_changed_public;
			}
			set
			{
				on_transfer_state_changed_public = value;
				on_transfer_state_changed_private = on_transfer_state_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_transfer_state_changed_private);
				linphone_core_cbs_set_transfer_state_changed(nativePtr, cb);
#else
				linphone_core_cbs_set_transfer_state_changed(nativePtr, on_transfer_state_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_buddy_info_updated(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_buddy_info_updated(IntPtr thiz, OnBuddyInfoUpdatedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnBuddyInfoUpdatedDelegatePrivate(IntPtr core, IntPtr linphoneFriend);

		/// <summary>
		/// Callback prototype when using the buddy plugin. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core">the <see cref="Linphone.Core">Linphone.Core</see>    </param>
		/// <param name="linphoneFriend">the <see cref="Linphone.Friend">
		/// Linphone.Friend</see> that has been updated    </param>
		public delegate void OnBuddyInfoUpdatedDelegate(Linphone.Core core, Linphone.Friend linphoneFriend);
		private OnBuddyInfoUpdatedDelegatePrivate on_buddy_info_updated_private;
		private OnBuddyInfoUpdatedDelegate on_buddy_info_updated_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnBuddyInfoUpdatedDelegatePrivate))]
#endif
		private static void on_buddy_info_updated(IntPtr core, IntPtr linphoneFriend)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_buddy_info_updated_public?.Invoke(thiz, fromNativePtr<Linphone.Friend>(linphoneFriend));
			}
		}

		public OnBuddyInfoUpdatedDelegate OnBuddyInfoUpdated
		{
			get
			{
				return on_buddy_info_updated_public;
			}
			set
			{
				on_buddy_info_updated_public = value;
				on_buddy_info_updated_private = on_buddy_info_updated;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_buddy_info_updated_private);
				linphone_core_cbs_set_buddy_info_updated(nativePtr, cb);
#else
				linphone_core_cbs_set_buddy_info_updated(nativePtr, on_buddy_info_updated_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_call_stats_updated(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_call_stats_updated(IntPtr thiz, OnCallStatsUpdatedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnCallStatsUpdatedDelegatePrivate(IntPtr core, IntPtr call, IntPtr callStats);

		/// <summary>
		/// Callback for receiving quality statistics for calls. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core">the <see cref="Linphone.Core">Linphone.Core</see>    </param>
		/// <param name="call">the <see cref="Linphone.Call">Linphone.Call</see>    </param>
		/// <param name="callStats">the <see cref="Linphone.CallStats">
		/// Linphone.CallStats</see> statistics.    </param>
		public delegate void OnCallStatsUpdatedDelegate(Linphone.Core core, Linphone.Call call, Linphone.CallStats callStats);
		private OnCallStatsUpdatedDelegatePrivate on_call_stats_updated_private;
		private OnCallStatsUpdatedDelegate on_call_stats_updated_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnCallStatsUpdatedDelegatePrivate))]
#endif
		private static void on_call_stats_updated(IntPtr core, IntPtr call, IntPtr callStats)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_call_stats_updated_public?.Invoke(thiz, fromNativePtr<Linphone.Call>(call), fromNativePtr<Linphone.CallStats>(callStats));
			}
		}

		public OnCallStatsUpdatedDelegate OnCallStatsUpdated
		{
			get
			{
				return on_call_stats_updated_public;
			}
			set
			{
				on_call_stats_updated_public = value;
				on_call_stats_updated_private = on_call_stats_updated;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_call_stats_updated_private);
				linphone_core_cbs_set_call_stats_updated(nativePtr, cb);
#else
				linphone_core_cbs_set_call_stats_updated(nativePtr, on_call_stats_updated_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_info_received(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_info_received(IntPtr thiz, OnInfoReceivedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnInfoReceivedDelegatePrivate(IntPtr core, IntPtr call, IntPtr message);

		/// <summary>
		/// Callback prototype for receiving info messages. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core">the <see cref="Linphone.Core">Linphone.Core</see>    </param>
		/// <param name="call">the call whose info message belongs to.    </param>
		/// <param name="message">the info message.    </param>
		public delegate void OnInfoReceivedDelegate(Linphone.Core core, Linphone.Call call, Linphone.InfoMessage message);
		private OnInfoReceivedDelegatePrivate on_info_received_private;
		private OnInfoReceivedDelegate on_info_received_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnInfoReceivedDelegatePrivate))]
#endif
		private static void on_info_received(IntPtr core, IntPtr call, IntPtr message)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_info_received_public?.Invoke(thiz, fromNativePtr<Linphone.Call>(call), fromNativePtr<Linphone.InfoMessage>(message));
			}
		}

		public OnInfoReceivedDelegate OnInfoReceived
		{
			get
			{
				return on_info_received_public;
			}
			set
			{
				on_info_received_public = value;
				on_info_received_private = on_info_received;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_info_received_private);
				linphone_core_cbs_set_info_received(nativePtr, cb);
#else
				linphone_core_cbs_set_info_received(nativePtr, on_info_received_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_subscription_state_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_subscription_state_changed(IntPtr thiz, OnSubscriptionStateChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnSubscriptionStateChangedDelegatePrivate(IntPtr core, IntPtr linphoneEvent, int state);

		/// <summary>
		/// Callback prototype for notifying the application about changes of subscription
		/// states, including arrival of new subscriptions. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="linphoneEvent">the <see cref="Linphone.Event">Linphone.Event</see>
		///    </param>
		/// <param name="state">the new <see cref="Linphone.SubscriptionState">
		/// Linphone.SubscriptionState</see> </param>
		public delegate void OnSubscriptionStateChangedDelegate(Linphone.Core core, Linphone.Event linphoneEvent, Linphone.SubscriptionState state);
		private OnSubscriptionStateChangedDelegatePrivate on_subscription_state_changed_private;
		private OnSubscriptionStateChangedDelegate on_subscription_state_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnSubscriptionStateChangedDelegatePrivate))]
#endif
		private static void on_subscription_state_changed(IntPtr core, IntPtr linphoneEvent, int state)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_subscription_state_changed_public?.Invoke(thiz, fromNativePtr<Linphone.Event>(linphoneEvent), (Linphone.SubscriptionState)state);
			}
		}

		public OnSubscriptionStateChangedDelegate OnSubscriptionStateChanged
		{
			get
			{
				return on_subscription_state_changed_public;
			}
			set
			{
				on_subscription_state_changed_public = value;
				on_subscription_state_changed_private = on_subscription_state_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_subscription_state_changed_private);
				linphone_core_cbs_set_subscription_state_changed(nativePtr, cb);
#else
				linphone_core_cbs_set_subscription_state_changed(nativePtr, on_subscription_state_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_notify_sent(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_notify_sent(IntPtr thiz, OnNotifySentDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnNotifySentDelegatePrivate(IntPtr core, IntPtr linphoneEvent, IntPtr body);

		/// <summary>
		/// Callback prototype for notifying the application about notification that is
		/// being sent. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="linphoneEvent">the <see cref="Linphone.Event">Linphone.Event</see>
		/// received    </param>
		/// <param name="body">the <see cref="Linphone.Content">Linphone.Content</see> of
		/// the event    </param>
		public delegate void OnNotifySentDelegate(Linphone.Core core, Linphone.Event linphoneEvent, Linphone.Content body);
		private OnNotifySentDelegatePrivate on_notify_sent_private;
		private OnNotifySentDelegate on_notify_sent_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnNotifySentDelegatePrivate))]
#endif
		private static void on_notify_sent(IntPtr core, IntPtr linphoneEvent, IntPtr body)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_notify_sent_public?.Invoke(thiz, fromNativePtr<Linphone.Event>(linphoneEvent), fromNativePtr<Linphone.Content>(body));
			}
		}

		public OnNotifySentDelegate OnNotifySent
		{
			get
			{
				return on_notify_sent_public;
			}
			set
			{
				on_notify_sent_public = value;
				on_notify_sent_private = on_notify_sent;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_notify_sent_private);
				linphone_core_cbs_set_notify_sent(nativePtr, cb);
#else
				linphone_core_cbs_set_notify_sent(nativePtr, on_notify_sent_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_notify_received(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_notify_received(IntPtr thiz, OnNotifyReceivedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnNotifyReceivedDelegatePrivate(IntPtr core, IntPtr linphoneEvent, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string notifiedEvent, IntPtr body);

		/// <summary>
		/// Callback prototype for notifying the application about notification received
		/// from the network. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="linphoneEvent">the <see cref="Linphone.Event">Linphone.Event</see>
		/// received    </param>
		/// <param name="notifiedEvent">The event as string    </param>
		/// <param name="body">the <see cref="Linphone.Content">Linphone.Content</see> of
		/// the event    </param>
		public delegate void OnNotifyReceivedDelegate(Linphone.Core core, Linphone.Event linphoneEvent, string notifiedEvent, Linphone.Content body);
		private OnNotifyReceivedDelegatePrivate on_notify_received_private;
		private OnNotifyReceivedDelegate on_notify_received_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnNotifyReceivedDelegatePrivate))]
#endif
		private static void on_notify_received(IntPtr core, IntPtr linphoneEvent, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string notifiedEvent, IntPtr body)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_notify_received_public?.Invoke(thiz, fromNativePtr<Linphone.Event>(linphoneEvent), notifiedEvent, fromNativePtr<Linphone.Content>(body));
			}
		}

		public OnNotifyReceivedDelegate OnNotifyReceived
		{
			get
			{
				return on_notify_received_public;
			}
			set
			{
				on_notify_received_public = value;
				on_notify_received_private = on_notify_received;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_notify_received_private);
				linphone_core_cbs_set_notify_received(nativePtr, cb);
#else
				linphone_core_cbs_set_notify_received(nativePtr, on_notify_received_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_subscribe_received(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_subscribe_received(IntPtr thiz, OnSubscribeReceivedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnSubscribeReceivedDelegatePrivate(IntPtr core, IntPtr linphoneEvent, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string subscribeEvent, IntPtr body);

		/// <summary>
		/// Callback prototype for notifying the application about subscription received
		/// from the network. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="linphoneEvent">the <see cref="Linphone.Event">Linphone.Event</see>
		/// received    </param>
		/// <param name="subscribeEvent">The event as string    </param>
		/// <param name="body">the <see cref="Linphone.Content">Linphone.Content</see> of
		/// the event    </param>
		public delegate void OnSubscribeReceivedDelegate(Linphone.Core core, Linphone.Event linphoneEvent, string subscribeEvent, Linphone.Content body);
		private OnSubscribeReceivedDelegatePrivate on_subscribe_received_private;
		private OnSubscribeReceivedDelegate on_subscribe_received_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnSubscribeReceivedDelegatePrivate))]
#endif
		private static void on_subscribe_received(IntPtr core, IntPtr linphoneEvent, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string subscribeEvent, IntPtr body)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_subscribe_received_public?.Invoke(thiz, fromNativePtr<Linphone.Event>(linphoneEvent), subscribeEvent, fromNativePtr<Linphone.Content>(body));
			}
		}

		public OnSubscribeReceivedDelegate OnSubscribeReceived
		{
			get
			{
				return on_subscribe_received_public;
			}
			set
			{
				on_subscribe_received_public = value;
				on_subscribe_received_private = on_subscribe_received;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_subscribe_received_private);
				linphone_core_cbs_set_subscribe_received(nativePtr, cb);
#else
				linphone_core_cbs_set_subscribe_received(nativePtr, on_subscribe_received_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_publish_state_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_publish_state_changed(IntPtr thiz, OnPublishStateChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnPublishStateChangedDelegatePrivate(IntPtr core, IntPtr linphoneEvent, int state);

		/// <summary>
		/// Callback prototype for notifying the application about changes of publish
		/// states. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="linphoneEvent">the <see cref="Linphone.Event">Linphone.Event</see>
		///    </param>
		/// <param name="state">the new <see cref="Linphone.PublishState">
		/// Linphone.PublishState</see> </param>
		public delegate void OnPublishStateChangedDelegate(Linphone.Core core, Linphone.Event linphoneEvent, Linphone.PublishState state);
		private OnPublishStateChangedDelegatePrivate on_publish_state_changed_private;
		private OnPublishStateChangedDelegate on_publish_state_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnPublishStateChangedDelegatePrivate))]
#endif
		private static void on_publish_state_changed(IntPtr core, IntPtr linphoneEvent, int state)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_publish_state_changed_public?.Invoke(thiz, fromNativePtr<Linphone.Event>(linphoneEvent), (Linphone.PublishState)state);
			}
		}

		public OnPublishStateChangedDelegate OnPublishStateChanged
		{
			get
			{
				return on_publish_state_changed_public;
			}
			set
			{
				on_publish_state_changed_public = value;
				on_publish_state_changed_private = on_publish_state_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_publish_state_changed_private);
				linphone_core_cbs_set_publish_state_changed(nativePtr, cb);
#else
				linphone_core_cbs_set_publish_state_changed(nativePtr, on_publish_state_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_publish_received(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_publish_received(IntPtr thiz, OnPublishReceivedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnPublishReceivedDelegatePrivate(IntPtr core, IntPtr linphoneEvent, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string publishEvent, IntPtr body);

		/// <summary>
		/// Callback prototype for notifying the application about publish received from
		/// the network. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="linphoneEvent">the <see cref="Linphone.Event">Linphone.Event</see>
		/// received    </param>
		/// <param name="publishEvent">The event as string    </param>
		/// <param name="body">the <see cref="Linphone.Content">Linphone.Content</see> of
		/// the event    </param>
		public delegate void OnPublishReceivedDelegate(Linphone.Core core, Linphone.Event linphoneEvent, string publishEvent, Linphone.Content body);
		private OnPublishReceivedDelegatePrivate on_publish_received_private;
		private OnPublishReceivedDelegate on_publish_received_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnPublishReceivedDelegatePrivate))]
#endif
		private static void on_publish_received(IntPtr core, IntPtr linphoneEvent, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string publishEvent, IntPtr body)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_publish_received_public?.Invoke(thiz, fromNativePtr<Linphone.Event>(linphoneEvent), publishEvent, fromNativePtr<Linphone.Content>(body));
			}
		}

		public OnPublishReceivedDelegate OnPublishReceived
		{
			get
			{
				return on_publish_received_public;
			}
			set
			{
				on_publish_received_public = value;
				on_publish_received_private = on_publish_received;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_publish_received_private);
				linphone_core_cbs_set_publish_received(nativePtr, cb);
#else
				linphone_core_cbs_set_publish_received(nativePtr, on_publish_received_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_configuring_status(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_configuring_status(IntPtr thiz, OnConfiguringStatusDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnConfiguringStatusDelegatePrivate(IntPtr core, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string message);

		/// <summary>
		/// Callback prototype for configuring status changes notification. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core">the <see cref="Linphone.Core">Linphone.Core</see>    </param>
		/// <param name="status">the current <see cref="Linphone.ConfiguringState">
		/// Linphone.ConfiguringState</see> </param>
		/// <param name="message">informational message.    </param>
		public delegate void OnConfiguringStatusDelegate(Linphone.Core core, Linphone.ConfiguringState status, string message);
		private OnConfiguringStatusDelegatePrivate on_configuring_status_private;
		private OnConfiguringStatusDelegate on_configuring_status_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnConfiguringStatusDelegatePrivate))]
#endif
		private static void on_configuring_status(IntPtr core, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string message)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_configuring_status_public?.Invoke(thiz, (Linphone.ConfiguringState)status, message);
			}
		}

		public OnConfiguringStatusDelegate OnConfiguringStatus
		{
			get
			{
				return on_configuring_status_public;
			}
			set
			{
				on_configuring_status_public = value;
				on_configuring_status_private = on_configuring_status;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_configuring_status_private);
				linphone_core_cbs_set_configuring_status(nativePtr, cb);
#else
				linphone_core_cbs_set_configuring_status(nativePtr, on_configuring_status_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_network_reachable(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_network_reachable(IntPtr thiz, OnNetworkReachableDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnNetworkReachableDelegatePrivate(IntPtr core, char reachable);

		/// <summary>
		/// Callback prototype for reporting network change either automatically detected
		/// or notified by <see cref="Linphone.Core.NetworkReachable">
		/// Linphone.Core.NetworkReachable</see>. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core">the <see cref="Linphone.Core">Linphone.Core</see>    </param>
		/// <param name="reachable">true if network is reachable. </param>
		public delegate void OnNetworkReachableDelegate(Linphone.Core core, bool reachable);
		private OnNetworkReachableDelegatePrivate on_network_reachable_private;
		private OnNetworkReachableDelegate on_network_reachable_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnNetworkReachableDelegatePrivate))]
#endif
		private static void on_network_reachable(IntPtr core, char reachable)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_network_reachable_public?.Invoke(thiz, reachable == 0);
			}
		}

		public OnNetworkReachableDelegate OnNetworkReachable
		{
			get
			{
				return on_network_reachable_public;
			}
			set
			{
				on_network_reachable_public = value;
				on_network_reachable_private = on_network_reachable;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_network_reachable_private);
				linphone_core_cbs_set_network_reachable(nativePtr, cb);
#else
				linphone_core_cbs_set_network_reachable(nativePtr, on_network_reachable_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_log_collection_upload_state_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_log_collection_upload_state_changed(IntPtr thiz, OnLogCollectionUploadStateChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnLogCollectionUploadStateChangedDelegatePrivate(IntPtr core, int state, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string info);

		/// <summary>
		/// Callback prototype for reporting log collection upload state change. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="state">The state of the log collection upload </param>
		/// <param name="info">Additional information: error message in case of error
		/// state, URL of uploaded file in case of success.    </param>
		public delegate void OnLogCollectionUploadStateChangedDelegate(Linphone.Core core, Linphone.CoreLogCollectionUploadState state, string info);
		private OnLogCollectionUploadStateChangedDelegatePrivate on_log_collection_upload_state_changed_private;
		private OnLogCollectionUploadStateChangedDelegate on_log_collection_upload_state_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnLogCollectionUploadStateChangedDelegatePrivate))]
#endif
		private static void on_log_collection_upload_state_changed(IntPtr core, int state, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string info)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_log_collection_upload_state_changed_public?.Invoke(thiz, (Linphone.CoreLogCollectionUploadState)state, info);
			}
		}

		public OnLogCollectionUploadStateChangedDelegate OnLogCollectionUploadStateChanged
		{
			get
			{
				return on_log_collection_upload_state_changed_public;
			}
			set
			{
				on_log_collection_upload_state_changed_public = value;
				on_log_collection_upload_state_changed_private = on_log_collection_upload_state_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_log_collection_upload_state_changed_private);
				linphone_core_cbs_set_log_collection_upload_state_changed(nativePtr, cb);
#else
				linphone_core_cbs_set_log_collection_upload_state_changed(nativePtr, on_log_collection_upload_state_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_log_collection_upload_progress_indication(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_log_collection_upload_progress_indication(IntPtr thiz, OnLogCollectionUploadProgressIndicationDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnLogCollectionUploadProgressIndicationDelegatePrivate(IntPtr core, long offset, long total);

		/// <summary>
		/// Callback prototype for reporting log collection upload progress indication. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="offset">the number of bytes sent since the start of the upload
		/// </param>
		/// <param name="total">the total number of bytes to upload </param>
		public delegate void OnLogCollectionUploadProgressIndicationDelegate(Linphone.Core core, long offset, long total);
		private OnLogCollectionUploadProgressIndicationDelegatePrivate on_log_collection_upload_progress_indication_private;
		private OnLogCollectionUploadProgressIndicationDelegate on_log_collection_upload_progress_indication_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnLogCollectionUploadProgressIndicationDelegatePrivate))]
#endif
		private static void on_log_collection_upload_progress_indication(IntPtr core, long offset, long total)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_log_collection_upload_progress_indication_public?.Invoke(thiz, offset, total);
			}
		}

		public OnLogCollectionUploadProgressIndicationDelegate OnLogCollectionUploadProgressIndication
		{
			get
			{
				return on_log_collection_upload_progress_indication_public;
			}
			set
			{
				on_log_collection_upload_progress_indication_public = value;
				on_log_collection_upload_progress_indication_private = on_log_collection_upload_progress_indication;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_log_collection_upload_progress_indication_private);
				linphone_core_cbs_set_log_collection_upload_progress_indication(nativePtr, cb);
#else
				linphone_core_cbs_set_log_collection_upload_progress_indication(nativePtr, on_log_collection_upload_progress_indication_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_friend_list_created(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_friend_list_created(IntPtr thiz, OnFriendListCreatedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnFriendListCreatedDelegatePrivate(IntPtr core, IntPtr friendList);

		/// <summary>
		/// Callback prototype for reporting when a friend list has been added to the core
		/// friend lists. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="friendList"><see cref="Linphone.FriendList">
		/// Linphone.FriendList</see> object    </param>
		public delegate void OnFriendListCreatedDelegate(Linphone.Core core, Linphone.FriendList friendList);
		private OnFriendListCreatedDelegatePrivate on_friend_list_created_private;
		private OnFriendListCreatedDelegate on_friend_list_created_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnFriendListCreatedDelegatePrivate))]
#endif
		private static void on_friend_list_created(IntPtr core, IntPtr friendList)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_friend_list_created_public?.Invoke(thiz, fromNativePtr<Linphone.FriendList>(friendList));
			}
		}

		public OnFriendListCreatedDelegate OnFriendListCreated
		{
			get
			{
				return on_friend_list_created_public;
			}
			set
			{
				on_friend_list_created_public = value;
				on_friend_list_created_private = on_friend_list_created;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_friend_list_created_private);
				linphone_core_cbs_set_friend_list_created(nativePtr, cb);
#else
				linphone_core_cbs_set_friend_list_created(nativePtr, on_friend_list_created_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_friend_list_removed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_friend_list_removed(IntPtr thiz, OnFriendListRemovedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnFriendListRemovedDelegatePrivate(IntPtr core, IntPtr friendList);

		/// <summary>
		/// Callback prototype for reporting when a friend list has been removed from the
		/// core friend lists. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="friendList"><see cref="Linphone.FriendList">
		/// Linphone.FriendList</see> object    </param>
		public delegate void OnFriendListRemovedDelegate(Linphone.Core core, Linphone.FriendList friendList);
		private OnFriendListRemovedDelegatePrivate on_friend_list_removed_private;
		private OnFriendListRemovedDelegate on_friend_list_removed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnFriendListRemovedDelegatePrivate))]
#endif
		private static void on_friend_list_removed(IntPtr core, IntPtr friendList)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_friend_list_removed_public?.Invoke(thiz, fromNativePtr<Linphone.FriendList>(friendList));
			}
		}

		public OnFriendListRemovedDelegate OnFriendListRemoved
		{
			get
			{
				return on_friend_list_removed_public;
			}
			set
			{
				on_friend_list_removed_public = value;
				on_friend_list_removed_private = on_friend_list_removed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_friend_list_removed_private);
				linphone_core_cbs_set_friend_list_removed(nativePtr, cb);
#else
				linphone_core_cbs_set_friend_list_removed(nativePtr, on_friend_list_removed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_call_created(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_call_created(IntPtr thiz, OnCallCreatedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnCallCreatedDelegatePrivate(IntPtr core, IntPtr call);

		/// <summary>
		/// Callback notifying that a new <see cref="Linphone.Call">Linphone.Call</see>
		/// (either incoming or outgoing) has been created. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object that
		/// has created the call    </param>
		/// <param name="call">The newly created <see cref="Linphone.Call">
		/// Linphone.Call</see> object    </param>
		public delegate void OnCallCreatedDelegate(Linphone.Core core, Linphone.Call call);
		private OnCallCreatedDelegatePrivate on_call_created_private;
		private OnCallCreatedDelegate on_call_created_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnCallCreatedDelegatePrivate))]
#endif
		private static void on_call_created(IntPtr core, IntPtr call)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_call_created_public?.Invoke(thiz, fromNativePtr<Linphone.Call>(call));
			}
		}

		public OnCallCreatedDelegate OnCallCreated
		{
			get
			{
				return on_call_created_public;
			}
			set
			{
				on_call_created_public = value;
				on_call_created_private = on_call_created;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_call_created_private);
				linphone_core_cbs_set_call_created(nativePtr, cb);
#else
				linphone_core_cbs_set_call_created(nativePtr, on_call_created_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_version_update_check_result_received(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_version_update_check_result_received(IntPtr thiz, OnVersionUpdateCheckResultReceivedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnVersionUpdateCheckResultReceivedDelegatePrivate(IntPtr core, int result, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string version, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string url);

		/// <summary>
		/// Callback prototype for reporting the result of a version update check. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="result">The result of the version update check    </param>
		/// <param name="version">The version to update to    </param>
		/// <param name="url">The url where to download the new version if the result is
		/// #LinphoneVersionUpdateCheckNewVersionAvailable    </param>
		public delegate void OnVersionUpdateCheckResultReceivedDelegate(Linphone.Core core, Linphone.VersionUpdateCheckResult result, string version, string url);
		private OnVersionUpdateCheckResultReceivedDelegatePrivate on_version_update_check_result_received_private;
		private OnVersionUpdateCheckResultReceivedDelegate on_version_update_check_result_received_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnVersionUpdateCheckResultReceivedDelegatePrivate))]
#endif
		private static void on_version_update_check_result_received(IntPtr core, int result, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string version, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string url)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_version_update_check_result_received_public?.Invoke(thiz, (Linphone.VersionUpdateCheckResult)result, version, url);
			}
		}

		public OnVersionUpdateCheckResultReceivedDelegate OnVersionUpdateCheckResultReceived
		{
			get
			{
				return on_version_update_check_result_received_public;
			}
			set
			{
				on_version_update_check_result_received_public = value;
				on_version_update_check_result_received_private = on_version_update_check_result_received;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_version_update_check_result_received_private);
				linphone_core_cbs_set_version_update_check_result_received(nativePtr, cb);
#else
				linphone_core_cbs_set_version_update_check_result_received(nativePtr, on_version_update_check_result_received_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_conference_state_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_conference_state_changed(IntPtr thiz, OnConferenceStateChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnConferenceStateChangedDelegatePrivate(IntPtr core, IntPtr conference, int state);

		/// <summary>
		/// Callback prototype telling that a <see cref="Linphone.Conference">
		/// Linphone.Conference</see> state has changed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="conference">The <see cref="Linphone.Conference">
		/// Linphone.Conference</see> object for which the state has changed    </param>
		/// <param name="state">the current <see cref="Linphone.ChatRoomState">
		/// Linphone.ChatRoomState</see> </param>
		public delegate void OnConferenceStateChangedDelegate(Linphone.Core core, Linphone.Conference conference, Linphone.ConferenceState state);
		private OnConferenceStateChangedDelegatePrivate on_conference_state_changed_private;
		private OnConferenceStateChangedDelegate on_conference_state_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnConferenceStateChangedDelegatePrivate))]
#endif
		private static void on_conference_state_changed(IntPtr core, IntPtr conference, int state)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_conference_state_changed_public?.Invoke(thiz, fromNativePtr<Linphone.Conference>(conference), (Linphone.ConferenceState)state);
			}
		}

		public OnConferenceStateChangedDelegate OnConferenceStateChanged
		{
			get
			{
				return on_conference_state_changed_public;
			}
			set
			{
				on_conference_state_changed_public = value;
				on_conference_state_changed_private = on_conference_state_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_conference_state_changed_private);
				linphone_core_cbs_set_conference_state_changed(nativePtr, cb);
#else
				linphone_core_cbs_set_conference_state_changed(nativePtr, on_conference_state_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_chat_room_state_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_chat_room_state_changed(IntPtr thiz, OnChatRoomStateChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnChatRoomStateChangedDelegatePrivate(IntPtr core, IntPtr chatRoom, int state);

		/// <summary>
		/// Callback prototype telling that a <see cref="Linphone.ChatRoom">
		/// Linphone.ChatRoom</see> state has changed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="chatRoom">The <see cref="Linphone.ChatRoom">
		/// Linphone.ChatRoom</see> object for which the state has changed    </param>
		/// <param name="state">the current <see cref="Linphone.ChatRoomState">
		/// Linphone.ChatRoomState</see> </param>
		public delegate void OnChatRoomStateChangedDelegate(Linphone.Core core, Linphone.ChatRoom chatRoom, Linphone.ChatRoomState state);
		private OnChatRoomStateChangedDelegatePrivate on_chat_room_state_changed_private;
		private OnChatRoomStateChangedDelegate on_chat_room_state_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnChatRoomStateChangedDelegatePrivate))]
#endif
		private static void on_chat_room_state_changed(IntPtr core, IntPtr chatRoom, int state)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_chat_room_state_changed_public?.Invoke(thiz, fromNativePtr<Linphone.ChatRoom>(chatRoom), (Linphone.ChatRoomState)state);
			}
		}

		public OnChatRoomStateChangedDelegate OnChatRoomStateChanged
		{
			get
			{
				return on_chat_room_state_changed_public;
			}
			set
			{
				on_chat_room_state_changed_public = value;
				on_chat_room_state_changed_private = on_chat_room_state_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_chat_room_state_changed_private);
				linphone_core_cbs_set_chat_room_state_changed(nativePtr, cb);
#else
				linphone_core_cbs_set_chat_room_state_changed(nativePtr, on_chat_room_state_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_chat_room_subject_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_chat_room_subject_changed(IntPtr thiz, OnChatRoomSubjectChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnChatRoomSubjectChangedDelegatePrivate(IntPtr core, IntPtr chatRoom);

		/// <summary>
		/// Callback prototype telling that a <see cref="Linphone.ChatRoom">
		/// Linphone.ChatRoom</see> subject has changed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="chatRoom">The <see cref="Linphone.ChatRoom">
		/// Linphone.ChatRoom</see> object for which the subject has changed    </param>
		public delegate void OnChatRoomSubjectChangedDelegate(Linphone.Core core, Linphone.ChatRoom chatRoom);
		private OnChatRoomSubjectChangedDelegatePrivate on_chat_room_subject_changed_private;
		private OnChatRoomSubjectChangedDelegate on_chat_room_subject_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnChatRoomSubjectChangedDelegatePrivate))]
#endif
		private static void on_chat_room_subject_changed(IntPtr core, IntPtr chatRoom)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_chat_room_subject_changed_public?.Invoke(thiz, fromNativePtr<Linphone.ChatRoom>(chatRoom));
			}
		}

		public OnChatRoomSubjectChangedDelegate OnChatRoomSubjectChanged
		{
			get
			{
				return on_chat_room_subject_changed_public;
			}
			set
			{
				on_chat_room_subject_changed_public = value;
				on_chat_room_subject_changed_private = on_chat_room_subject_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_chat_room_subject_changed_private);
				linphone_core_cbs_set_chat_room_subject_changed(nativePtr, cb);
#else
				linphone_core_cbs_set_chat_room_subject_changed(nativePtr, on_chat_room_subject_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_chat_room_ephemeral_message_deleted(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_chat_room_ephemeral_message_deleted(IntPtr thiz, OnChatRoomEphemeralMessageDeletedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnChatRoomEphemeralMessageDeletedDelegatePrivate(IntPtr core, IntPtr chatRoom);

		/// <summary>
		/// Callback prototype telling that a <see cref="Linphone.ChatRoom">
		/// Linphone.ChatRoom</see> ephemeral message has expired. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="chatRoom">The <see cref="Linphone.ChatRoom">
		/// Linphone.ChatRoom</see> object for which a message has expired.    </param>
		public delegate void OnChatRoomEphemeralMessageDeletedDelegate(Linphone.Core core, Linphone.ChatRoom chatRoom);
		private OnChatRoomEphemeralMessageDeletedDelegatePrivate on_chat_room_ephemeral_message_deleted_private;
		private OnChatRoomEphemeralMessageDeletedDelegate on_chat_room_ephemeral_message_deleted_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnChatRoomEphemeralMessageDeletedDelegatePrivate))]
#endif
		private static void on_chat_room_ephemeral_message_deleted(IntPtr core, IntPtr chatRoom)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_chat_room_ephemeral_message_deleted_public?.Invoke(thiz, fromNativePtr<Linphone.ChatRoom>(chatRoom));
			}
		}

		public OnChatRoomEphemeralMessageDeletedDelegate OnChatRoomEphemeralMessageDeleted
		{
			get
			{
				return on_chat_room_ephemeral_message_deleted_public;
			}
			set
			{
				on_chat_room_ephemeral_message_deleted_public = value;
				on_chat_room_ephemeral_message_deleted_private = on_chat_room_ephemeral_message_deleted;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_chat_room_ephemeral_message_deleted_private);
				linphone_core_cbs_set_chat_room_ephemeral_message_deleted(nativePtr, cb);
#else
				linphone_core_cbs_set_chat_room_ephemeral_message_deleted(nativePtr, on_chat_room_ephemeral_message_deleted_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_imee_user_registration(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_imee_user_registration(IntPtr thiz, OnImeeUserRegistrationDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnImeeUserRegistrationDelegatePrivate(IntPtr core, char status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string userId, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string info);

		/// <summary>
		/// Callback prototype telling that an Instant Message Encryption Engine user
		/// registered on the server with or without success. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="status">the return status of the registration action. </param>
		/// <param name="userId">the userId published on the encryption engine server   
		/// </param>
		/// <param name="info">information about failure    </param>
		public delegate void OnImeeUserRegistrationDelegate(Linphone.Core core, bool status, string userId, string info);
		private OnImeeUserRegistrationDelegatePrivate on_imee_user_registration_private;
		private OnImeeUserRegistrationDelegate on_imee_user_registration_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnImeeUserRegistrationDelegatePrivate))]
#endif
		private static void on_imee_user_registration(IntPtr core, char status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string userId, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string info)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_imee_user_registration_public?.Invoke(thiz, status == 0, userId, info);
			}
		}

		public OnImeeUserRegistrationDelegate OnImeeUserRegistration
		{
			get
			{
				return on_imee_user_registration_public;
			}
			set
			{
				on_imee_user_registration_public = value;
				on_imee_user_registration_private = on_imee_user_registration;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_imee_user_registration_private);
				linphone_core_cbs_set_imee_user_registration(nativePtr, cb);
#else
				linphone_core_cbs_set_imee_user_registration(nativePtr, on_imee_user_registration_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_qrcode_found(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_qrcode_found(IntPtr thiz, OnQrcodeFoundDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnQrcodeFoundDelegatePrivate(IntPtr core, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string result);

		/// <summary>
		/// Callback prototype telling the result of decoded qrcode. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="result">The result of the decoded qrcode    </param>
		public delegate void OnQrcodeFoundDelegate(Linphone.Core core, string result);
		private OnQrcodeFoundDelegatePrivate on_qrcode_found_private;
		private OnQrcodeFoundDelegate on_qrcode_found_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnQrcodeFoundDelegatePrivate))]
#endif
		private static void on_qrcode_found(IntPtr core, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string result)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_qrcode_found_public?.Invoke(thiz, result);
			}
		}

		public OnQrcodeFoundDelegate OnQrcodeFound
		{
			get
			{
				return on_qrcode_found_public;
			}
			set
			{
				on_qrcode_found_public = value;
				on_qrcode_found_private = on_qrcode_found;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_qrcode_found_private);
				linphone_core_cbs_set_qrcode_found(nativePtr, cb);
#else
				linphone_core_cbs_set_qrcode_found(nativePtr, on_qrcode_found_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_first_call_started(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_first_call_started(IntPtr thiz, OnFirstCallStartedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnFirstCallStartedDelegatePrivate(IntPtr core);

		/// <summary>
		/// Callback prototype telling a call has started (incoming or outgoing) while
		/// there was no other call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		public delegate void OnFirstCallStartedDelegate(Linphone.Core core);
		private OnFirstCallStartedDelegatePrivate on_first_call_started_private;
		private OnFirstCallStartedDelegate on_first_call_started_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnFirstCallStartedDelegatePrivate))]
#endif
		private static void on_first_call_started(IntPtr core)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_first_call_started_public?.Invoke(thiz);
			}
		}

		public OnFirstCallStartedDelegate OnFirstCallStarted
		{
			get
			{
				return on_first_call_started_public;
			}
			set
			{
				on_first_call_started_public = value;
				on_first_call_started_private = on_first_call_started;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_first_call_started_private);
				linphone_core_cbs_set_first_call_started(nativePtr, cb);
#else
				linphone_core_cbs_set_first_call_started(nativePtr, on_first_call_started_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_last_call_ended(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_last_call_ended(IntPtr thiz, OnLastCallEndedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnLastCallEndedDelegatePrivate(IntPtr core);

		/// <summary>
		/// Callback prototype telling the last call has ended (<see cref="Linphone.Core.CallsNb">
		/// Linphone.Core.CallsNb</see> returns 0) 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		public delegate void OnLastCallEndedDelegate(Linphone.Core core);
		private OnLastCallEndedDelegatePrivate on_last_call_ended_private;
		private OnLastCallEndedDelegate on_last_call_ended_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnLastCallEndedDelegatePrivate))]
#endif
		private static void on_last_call_ended(IntPtr core)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_last_call_ended_public?.Invoke(thiz);
			}
		}

		public OnLastCallEndedDelegate OnLastCallEnded
		{
			get
			{
				return on_last_call_ended_public;
			}
			set
			{
				on_last_call_ended_public = value;
				on_last_call_ended_private = on_last_call_ended;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_last_call_ended_private);
				linphone_core_cbs_set_last_call_ended(nativePtr, cb);
#else
				linphone_core_cbs_set_last_call_ended(nativePtr, on_last_call_ended_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_audio_device_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_audio_device_changed(IntPtr thiz, OnAudioDeviceChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnAudioDeviceChangedDelegatePrivate(IntPtr core, IntPtr audioDevice);

		/// <summary>
		/// Callback prototype telling that the audio device for at least one call has
		/// changed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="audioDevice">the newly used <see cref="Linphone.AudioDevice">
		/// Linphone.AudioDevice</see> object    </param>
		public delegate void OnAudioDeviceChangedDelegate(Linphone.Core core, Linphone.AudioDevice audioDevice);
		private OnAudioDeviceChangedDelegatePrivate on_audio_device_changed_private;
		private OnAudioDeviceChangedDelegate on_audio_device_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnAudioDeviceChangedDelegatePrivate))]
#endif
		private static void on_audio_device_changed(IntPtr core, IntPtr audioDevice)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_audio_device_changed_public?.Invoke(thiz, fromNativePtr<Linphone.AudioDevice>(audioDevice));
			}
		}

		public OnAudioDeviceChangedDelegate OnAudioDeviceChanged
		{
			get
			{
				return on_audio_device_changed_public;
			}
			set
			{
				on_audio_device_changed_public = value;
				on_audio_device_changed_private = on_audio_device_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_audio_device_changed_private);
				linphone_core_cbs_set_audio_device_changed(nativePtr, cb);
#else
				linphone_core_cbs_set_audio_device_changed(nativePtr, on_audio_device_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_audio_devices_list_updated(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_audio_devices_list_updated(IntPtr thiz, OnAudioDevicesListUpdatedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnAudioDevicesListUpdatedDelegatePrivate(IntPtr core);

		/// <summary>
		/// Callback prototype telling the audio devices list has been updated. 
		/// <para>
		/// Either a new device is available or a previously available device isn&apos;t
		/// anymore. You can call <see cref="Linphone.Core.AudioDevices">
		/// Linphone.Core.AudioDevices</see> to get the new list. 
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		public delegate void OnAudioDevicesListUpdatedDelegate(Linphone.Core core);
		private OnAudioDevicesListUpdatedDelegatePrivate on_audio_devices_list_updated_private;
		private OnAudioDevicesListUpdatedDelegate on_audio_devices_list_updated_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnAudioDevicesListUpdatedDelegatePrivate))]
#endif
		private static void on_audio_devices_list_updated(IntPtr core)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_audio_devices_list_updated_public?.Invoke(thiz);
			}
		}

		public OnAudioDevicesListUpdatedDelegate OnAudioDevicesListUpdated
		{
			get
			{
				return on_audio_devices_list_updated_public;
			}
			set
			{
				on_audio_devices_list_updated_public = value;
				on_audio_devices_list_updated_private = on_audio_devices_list_updated;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_audio_devices_list_updated_private);
				linphone_core_cbs_set_audio_devices_list_updated(nativePtr, cb);
#else
				linphone_core_cbs_set_audio_devices_list_updated(nativePtr, on_audio_devices_list_updated_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_ec_calibration_result(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_ec_calibration_result(IntPtr thiz, OnEcCalibrationResultDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnEcCalibrationResultDelegatePrivate(IntPtr core, int status, int delayMs);

		/// <summary>
		/// Function prototype used by linphone_core_cbs_set_ec_calibration_result. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core">The <see cref="Linphone.Core">Linphone.Core</see>.   
		/// </param>
		/// <param name="status">The <see cref="Linphone.EcCalibratorStatus">
		/// Linphone.EcCalibratorStatus</see> of the calibrator. </param>
		/// <param name="delayMs">The measured delay if available. </param>
		public delegate void OnEcCalibrationResultDelegate(Linphone.Core core, Linphone.EcCalibratorStatus status, int delayMs);
		private OnEcCalibrationResultDelegatePrivate on_ec_calibration_result_private;
		private OnEcCalibrationResultDelegate on_ec_calibration_result_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnEcCalibrationResultDelegatePrivate))]
#endif
		private static void on_ec_calibration_result(IntPtr core, int status, int delayMs)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_ec_calibration_result_public?.Invoke(thiz, (Linphone.EcCalibratorStatus)status, delayMs);
			}
		}

		public OnEcCalibrationResultDelegate OnEcCalibrationResult
		{
			get
			{
				return on_ec_calibration_result_public;
			}
			set
			{
				on_ec_calibration_result_public = value;
				on_ec_calibration_result_private = on_ec_calibration_result;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_ec_calibration_result_private);
				linphone_core_cbs_set_ec_calibration_result(nativePtr, cb);
#else
				linphone_core_cbs_set_ec_calibration_result(nativePtr, on_ec_calibration_result_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_ec_calibration_audio_init(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_ec_calibration_audio_init(IntPtr thiz, OnEcCalibrationAudioInitDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnEcCalibrationAudioInitDelegatePrivate(IntPtr core);

		/// <summary>
		/// Function prototype used by linphone_core_cbs_set_ec_calibration_audio_init. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core">The <see cref="Linphone.Core">Linphone.Core</see>.   
		/// </param>
		public delegate void OnEcCalibrationAudioInitDelegate(Linphone.Core core);
		private OnEcCalibrationAudioInitDelegatePrivate on_ec_calibration_audio_init_private;
		private OnEcCalibrationAudioInitDelegate on_ec_calibration_audio_init_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnEcCalibrationAudioInitDelegatePrivate))]
#endif
		private static void on_ec_calibration_audio_init(IntPtr core)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_ec_calibration_audio_init_public?.Invoke(thiz);
			}
		}

		public OnEcCalibrationAudioInitDelegate OnEcCalibrationAudioInit
		{
			get
			{
				return on_ec_calibration_audio_init_public;
			}
			set
			{
				on_ec_calibration_audio_init_public = value;
				on_ec_calibration_audio_init_private = on_ec_calibration_audio_init;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_ec_calibration_audio_init_private);
				linphone_core_cbs_set_ec_calibration_audio_init(nativePtr, cb);
#else
				linphone_core_cbs_set_ec_calibration_audio_init(nativePtr, on_ec_calibration_audio_init_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_ec_calibration_audio_uninit(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_ec_calibration_audio_uninit(IntPtr thiz, OnEcCalibrationAudioUninitDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnEcCalibrationAudioUninitDelegatePrivate(IntPtr core);

		/// <summary>
		/// Function prototype used by linphone_core_cbs_set_ec_calibration_audio_uninit. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core">The <see cref="Linphone.Core">Linphone.Core</see>.   
		/// </param>
		public delegate void OnEcCalibrationAudioUninitDelegate(Linphone.Core core);
		private OnEcCalibrationAudioUninitDelegatePrivate on_ec_calibration_audio_uninit_private;
		private OnEcCalibrationAudioUninitDelegate on_ec_calibration_audio_uninit_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnEcCalibrationAudioUninitDelegatePrivate))]
#endif
		private static void on_ec_calibration_audio_uninit(IntPtr core)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_ec_calibration_audio_uninit_public?.Invoke(thiz);
			}
		}

		public OnEcCalibrationAudioUninitDelegate OnEcCalibrationAudioUninit
		{
			get
			{
				return on_ec_calibration_audio_uninit_public;
			}
			set
			{
				on_ec_calibration_audio_uninit_public = value;
				on_ec_calibration_audio_uninit_private = on_ec_calibration_audio_uninit;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_ec_calibration_audio_uninit_private);
				linphone_core_cbs_set_ec_calibration_audio_uninit(nativePtr, cb);
#else
				linphone_core_cbs_set_ec_calibration_audio_uninit(nativePtr, on_ec_calibration_audio_uninit_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_account_registration_state_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_account_registration_state_changed(IntPtr thiz, OnAccountRegistrationStateChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnAccountRegistrationStateChangedDelegatePrivate(IntPtr core, IntPtr account, int state, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string message);

		/// <summary>
		/// Callback notifying that a <see cref="Linphone.Account">Linphone.Account</see>
		/// has its registration state changed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core">The <see cref="Linphone.Core">Linphone.Core</see> object.   
		/// </param>
		/// <param name="account">The <see cref="Linphone.Account">Linphone.Account</see>
		/// object which has its registration changed.    </param>
		/// <param name="state">The new <see cref="Linphone.RegistrationState">
		/// Linphone.RegistrationState</see> for this account. </param>
		/// <param name="message">a non null informational message about the state   
		/// </param>
		public delegate void OnAccountRegistrationStateChangedDelegate(Linphone.Core core, Linphone.Account account, Linphone.RegistrationState state, string message);
		private OnAccountRegistrationStateChangedDelegatePrivate on_account_registration_state_changed_private;
		private OnAccountRegistrationStateChangedDelegate on_account_registration_state_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnAccountRegistrationStateChangedDelegatePrivate))]
#endif
		private static void on_account_registration_state_changed(IntPtr core, IntPtr account, int state, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string message)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_account_registration_state_changed_public?.Invoke(thiz, fromNativePtr<Linphone.Account>(account), (Linphone.RegistrationState)state, message);
			}
		}

		public OnAccountRegistrationStateChangedDelegate OnAccountRegistrationStateChanged
		{
			get
			{
				return on_account_registration_state_changed_public;
			}
			set
			{
				on_account_registration_state_changed_public = value;
				on_account_registration_state_changed_private = on_account_registration_state_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_account_registration_state_changed_private);
				linphone_core_cbs_set_account_registration_state_changed(nativePtr, cb);
#else
				linphone_core_cbs_set_account_registration_state_changed(nativePtr, on_account_registration_state_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_default_account_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_default_account_changed(IntPtr thiz, OnDefaultAccountChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnDefaultAccountChangedDelegatePrivate(IntPtr core, IntPtr account);

		/// <summary>
		/// Default account changed callback prototype. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="account"><see cref="Linphone.Account">Linphone.Account</see>
		/// object that has been set as the default account, probably by calling <see cref="Linphone.Core.DefaultAccount">
		/// Linphone.Core.DefaultAccount</see>, or null if the default account was removed.
		///    </param>
		public delegate void OnDefaultAccountChangedDelegate(Linphone.Core core, Linphone.Account account);
		private OnDefaultAccountChangedDelegatePrivate on_default_account_changed_private;
		private OnDefaultAccountChangedDelegate on_default_account_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnDefaultAccountChangedDelegatePrivate))]
#endif
		private static void on_default_account_changed(IntPtr core, IntPtr account)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_default_account_changed_public?.Invoke(thiz, fromNativePtr<Linphone.Account>(account));
			}
		}

		public OnDefaultAccountChangedDelegate OnDefaultAccountChanged
		{
			get
			{
				return on_default_account_changed_public;
			}
			set
			{
				on_default_account_changed_public = value;
				on_default_account_changed_private = on_default_account_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_default_account_changed_private);
				linphone_core_cbs_set_default_account_changed(nativePtr, cb);
#else
				linphone_core_cbs_set_default_account_changed(nativePtr, on_default_account_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_account_added(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_account_added(IntPtr thiz, OnAccountAddedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnAccountAddedDelegatePrivate(IntPtr core, IntPtr account);

		/// <summary>
		/// Account added callback prototype. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="account"><see cref="Linphone.Account">Linphone.Account</see>
		/// object that has been added to the Core using <see cref="Linphone.Core.AddAccount()">
		/// Linphone.Core.AddAccount()</see> for example.    </param>
		public delegate void OnAccountAddedDelegate(Linphone.Core core, Linphone.Account account);
		private OnAccountAddedDelegatePrivate on_account_added_private;
		private OnAccountAddedDelegate on_account_added_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnAccountAddedDelegatePrivate))]
#endif
		private static void on_account_added(IntPtr core, IntPtr account)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_account_added_public?.Invoke(thiz, fromNativePtr<Linphone.Account>(account));
			}
		}

		public OnAccountAddedDelegate OnAccountAdded
		{
			get
			{
				return on_account_added_public;
			}
			set
			{
				on_account_added_public = value;
				on_account_added_private = on_account_added;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_account_added_private);
				linphone_core_cbs_set_account_added(nativePtr, cb);
#else
				linphone_core_cbs_set_account_added(nativePtr, on_account_added_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_account_removed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_account_removed(IntPtr thiz, OnAccountRemovedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnAccountRemovedDelegatePrivate(IntPtr core, IntPtr account);

		/// <summary>
		/// Account removed callback prototype. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core"><see cref="Linphone.Core">Linphone.Core</see> object   
		/// </param>
		/// <param name="account"><see cref="Linphone.Account">Linphone.Account</see>
		/// object that has been added to the Core using <see cref="Linphone.Core.RemoveAccount()">
		/// Linphone.Core.RemoveAccount()</see> for example.    </param>
		public delegate void OnAccountRemovedDelegate(Linphone.Core core, Linphone.Account account);
		private OnAccountRemovedDelegatePrivate on_account_removed_private;
		private OnAccountRemovedDelegate on_account_removed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnAccountRemovedDelegatePrivate))]
#endif
		private static void on_account_removed(IntPtr core, IntPtr account)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_account_removed_public?.Invoke(thiz, fromNativePtr<Linphone.Account>(account));
			}
		}

		public OnAccountRemovedDelegate OnAccountRemoved
		{
			get
			{
				return on_account_removed_public;
			}
			set
			{
				on_account_removed_public = value;
				on_account_removed_private = on_account_removed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_account_removed_private);
				linphone_core_cbs_set_account_removed(nativePtr, cb);
#else
				linphone_core_cbs_set_account_removed(nativePtr, on_account_removed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_message_waiting_indication_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_message_waiting_indication_changed(IntPtr thiz, OnMessageWaitingIndicationChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnMessageWaitingIndicationChangedDelegatePrivate(IntPtr core, IntPtr lev, IntPtr mwi);

		/// <summary>
		/// Callback notifying that a Message Waiting Indication state has changed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core">The <see cref="Linphone.Core">Linphone.Core</see> object.   
		/// </param>
		/// <param name="lev">The <see cref="Linphone.Event">Linphone.Event</see> object
		/// notifying the MWI.    </param>
		/// <param name="mwi">The <see cref="Linphone.MessageWaitingIndication">
		/// Linphone.MessageWaitingIndication</see> object that is notified.    </param>
		public delegate void OnMessageWaitingIndicationChangedDelegate(Linphone.Core core, Linphone.Event lev, Linphone.MessageWaitingIndication mwi);
		private OnMessageWaitingIndicationChangedDelegatePrivate on_message_waiting_indication_changed_private;
		private OnMessageWaitingIndicationChangedDelegate on_message_waiting_indication_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnMessageWaitingIndicationChangedDelegatePrivate))]
#endif
		private static void on_message_waiting_indication_changed(IntPtr core, IntPtr lev, IntPtr mwi)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_message_waiting_indication_changed_public?.Invoke(thiz, fromNativePtr<Linphone.Event>(lev), fromNativePtr<Linphone.MessageWaitingIndication>(mwi));
			}
		}

		public OnMessageWaitingIndicationChangedDelegate OnMessageWaitingIndicationChanged
		{
			get
			{
				return on_message_waiting_indication_changed_public;
			}
			set
			{
				on_message_waiting_indication_changed_public = value;
				on_message_waiting_indication_changed_private = on_message_waiting_indication_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_message_waiting_indication_changed_private);
				linphone_core_cbs_set_message_waiting_indication_changed(nativePtr, cb);
#else
				linphone_core_cbs_set_message_waiting_indication_changed(nativePtr, on_message_waiting_indication_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_core_cbs_set_snapshot_taken(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_core_cbs_set_snapshot_taken(IntPtr thiz, OnSnapshotTakenDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnSnapshotTakenDelegatePrivate(IntPtr core, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string filePath);

		/// <summary>
		/// Callback notifying a snapshot has been taken. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core">LinphoneCore object.    </param>
		/// <param name="filePath">the name of the saved file.    </param>
		public delegate void OnSnapshotTakenDelegate(Linphone.Core core, string filePath);
		private OnSnapshotTakenDelegatePrivate on_snapshot_taken_private;
		private OnSnapshotTakenDelegate on_snapshot_taken_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnSnapshotTakenDelegatePrivate))]
#endif
		private static void on_snapshot_taken(IntPtr core, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string filePath)
		{
			Core thiz = fromNativePtr<Core>(core);

			IntPtr ptr = linphone_core_get_current_callbacks(thiz.nativePtr);
			CoreListener current_listener = fromNativePtr<CoreListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_snapshot_taken_public?.Invoke(thiz, filePath);
			}
		}

		public OnSnapshotTakenDelegate OnSnapshotTaken
		{
			get
			{
				return on_snapshot_taken_public;
			}
			set
			{
				on_snapshot_taken_public = value;
				on_snapshot_taken_private = on_snapshot_taken;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_snapshot_taken_private);
				linphone_core_cbs_set_snapshot_taken(nativePtr, cb);
#else
				linphone_core_cbs_set_snapshot_taken(nativePtr, on_snapshot_taken_private);
#endif
			}
		}
	}

	/// <summary>
	/// An object to handle the callbacks for handling the LinphoneEvent operations. 
	/// <para>
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class EventListener : LinphoneObject
	{
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_event_get_current_callbacks(IntPtr thiz);

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_event_cbs_set_notify_response(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_event_cbs_set_notify_response(IntPtr thiz, OnNotifyResponseDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnNotifyResponseDelegatePrivate(IntPtr ev);

		/// <summary>
		/// Callback used to notify the response to a sent NOTIFY. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="ev">The <see cref="Linphone.Event">Linphone.Event</see> object
		/// that has sent the NOTIFY and for which we received a response    </param>
		public delegate void OnNotifyResponseDelegate(Linphone.Event ev);
		private OnNotifyResponseDelegatePrivate on_notify_response_private;
		private OnNotifyResponseDelegate on_notify_response_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnNotifyResponseDelegatePrivate))]
#endif
		private static void on_notify_response(IntPtr ev)
		{
			Event thiz = fromNativePtr<Event>(ev);

			IntPtr ptr = linphone_event_get_current_callbacks(thiz.nativePtr);
			EventListener current_listener = fromNativePtr<EventListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_notify_response_public?.Invoke(thiz);
			}
		}

		public OnNotifyResponseDelegate OnNotifyResponse
		{
			get
			{
				return on_notify_response_public;
			}
			set
			{
				on_notify_response_public = value;
				on_notify_response_private = on_notify_response;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_notify_response_private);
				linphone_event_cbs_set_notify_response(nativePtr, cb);
#else
				linphone_event_cbs_set_notify_response(nativePtr, on_notify_response_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_event_cbs_set_notify_received(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_event_cbs_set_notify_received(IntPtr thiz, OnNotifyReceivedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnNotifyReceivedDelegatePrivate(IntPtr ev, IntPtr content);

		/// <summary>
		/// Callback used to notify the received to a NOTIFY. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="ev">The LinphoneEvent object that receive the NOTIFY    </param>
		/// <param name="content">The LinphoneContent object that containe the body of the
		/// event    </param>
		public delegate void OnNotifyReceivedDelegate(Linphone.Event ev, Linphone.Content content);
		private OnNotifyReceivedDelegatePrivate on_notify_received_private;
		private OnNotifyReceivedDelegate on_notify_received_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnNotifyReceivedDelegatePrivate))]
#endif
		private static void on_notify_received(IntPtr ev, IntPtr content)
		{
			Event thiz = fromNativePtr<Event>(ev);

			IntPtr ptr = linphone_event_get_current_callbacks(thiz.nativePtr);
			EventListener current_listener = fromNativePtr<EventListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_notify_received_public?.Invoke(thiz, fromNativePtr<Linphone.Content>(content));
			}
		}

		public OnNotifyReceivedDelegate OnNotifyReceived
		{
			get
			{
				return on_notify_received_public;
			}
			set
			{
				on_notify_received_public = value;
				on_notify_received_private = on_notify_received;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_notify_received_private);
				linphone_event_cbs_set_notify_received(nativePtr, cb);
#else
				linphone_event_cbs_set_notify_received(nativePtr, on_notify_received_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_event_cbs_set_subscribe_received(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_event_cbs_set_subscribe_received(IntPtr thiz, OnSubscribeReceivedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnSubscribeReceivedDelegatePrivate(IntPtr ev);

		/// <summary>
		/// Callback used to notify the received to a SUBSCRIBE. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="ev">The LinphoneEvent object that receive the SUBSCRIBE    </param>
		public delegate void OnSubscribeReceivedDelegate(Linphone.Event ev);
		private OnSubscribeReceivedDelegatePrivate on_subscribe_received_private;
		private OnSubscribeReceivedDelegate on_subscribe_received_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnSubscribeReceivedDelegatePrivate))]
#endif
		private static void on_subscribe_received(IntPtr ev)
		{
			Event thiz = fromNativePtr<Event>(ev);

			IntPtr ptr = linphone_event_get_current_callbacks(thiz.nativePtr);
			EventListener current_listener = fromNativePtr<EventListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_subscribe_received_public?.Invoke(thiz);
			}
		}

		public OnSubscribeReceivedDelegate OnSubscribeReceived
		{
			get
			{
				return on_subscribe_received_public;
			}
			set
			{
				on_subscribe_received_public = value;
				on_subscribe_received_private = on_subscribe_received;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_subscribe_received_private);
				linphone_event_cbs_set_subscribe_received(nativePtr, cb);
#else
				linphone_event_cbs_set_subscribe_received(nativePtr, on_subscribe_received_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_event_cbs_set_subscribe_state_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_event_cbs_set_subscribe_state_changed(IntPtr thiz, OnSubscribeStateChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnSubscribeStateChangedDelegatePrivate(IntPtr ev, int state);

		/// <summary>
		/// SUBSCRIBE state changed callback. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="ev">The <see cref="Linphone.Event">Linphone.Event</see> object
		/// that state changed    </param>
		/// <param name="state">The <see cref="Linphone.SubscriptionState">
		/// Linphone.SubscriptionState</see> </param>
		public delegate void OnSubscribeStateChangedDelegate(Linphone.Event ev, Linphone.SubscriptionState state);
		private OnSubscribeStateChangedDelegatePrivate on_subscribe_state_changed_private;
		private OnSubscribeStateChangedDelegate on_subscribe_state_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnSubscribeStateChangedDelegatePrivate))]
#endif
		private static void on_subscribe_state_changed(IntPtr ev, int state)
		{
			Event thiz = fromNativePtr<Event>(ev);

			IntPtr ptr = linphone_event_get_current_callbacks(thiz.nativePtr);
			EventListener current_listener = fromNativePtr<EventListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_subscribe_state_changed_public?.Invoke(thiz, (Linphone.SubscriptionState)state);
			}
		}

		public OnSubscribeStateChangedDelegate OnSubscribeStateChanged
		{
			get
			{
				return on_subscribe_state_changed_public;
			}
			set
			{
				on_subscribe_state_changed_public = value;
				on_subscribe_state_changed_private = on_subscribe_state_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_subscribe_state_changed_private);
				linphone_event_cbs_set_subscribe_state_changed(nativePtr, cb);
#else
				linphone_event_cbs_set_subscribe_state_changed(nativePtr, on_subscribe_state_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_event_cbs_set_publish_received(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_event_cbs_set_publish_received(IntPtr thiz, OnPublishReceivedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnPublishReceivedDelegatePrivate(IntPtr ev, IntPtr content);

		/// <summary>
		/// Callback used to notify the received to a PUBLISH. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="ev">The LinphoneEvent object that receive the PUBLISH    </param>
		/// <param name="content">The LinphoneContent object that containe the body of the
		/// event    </param>
		public delegate void OnPublishReceivedDelegate(Linphone.Event ev, Linphone.Content content);
		private OnPublishReceivedDelegatePrivate on_publish_received_private;
		private OnPublishReceivedDelegate on_publish_received_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnPublishReceivedDelegatePrivate))]
#endif
		private static void on_publish_received(IntPtr ev, IntPtr content)
		{
			Event thiz = fromNativePtr<Event>(ev);

			IntPtr ptr = linphone_event_get_current_callbacks(thiz.nativePtr);
			EventListener current_listener = fromNativePtr<EventListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_publish_received_public?.Invoke(thiz, fromNativePtr<Linphone.Content>(content));
			}
		}

		public OnPublishReceivedDelegate OnPublishReceived
		{
			get
			{
				return on_publish_received_public;
			}
			set
			{
				on_publish_received_public = value;
				on_publish_received_private = on_publish_received;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_publish_received_private);
				linphone_event_cbs_set_publish_received(nativePtr, cb);
#else
				linphone_event_cbs_set_publish_received(nativePtr, on_publish_received_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_event_cbs_set_publish_state_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_event_cbs_set_publish_state_changed(IntPtr thiz, OnPublishStateChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnPublishStateChangedDelegatePrivate(IntPtr ev, int state);

		/// <summary>
		/// PUBLISH state changed callback. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="ev">The LinphoneEvent object that state changed    </param>
		/// <param name="state">The LinphonePublishState </param>
		public delegate void OnPublishStateChangedDelegate(Linphone.Event ev, Linphone.PublishState state);
		private OnPublishStateChangedDelegatePrivate on_publish_state_changed_private;
		private OnPublishStateChangedDelegate on_publish_state_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnPublishStateChangedDelegatePrivate))]
#endif
		private static void on_publish_state_changed(IntPtr ev, int state)
		{
			Event thiz = fromNativePtr<Event>(ev);

			IntPtr ptr = linphone_event_get_current_callbacks(thiz.nativePtr);
			EventListener current_listener = fromNativePtr<EventListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_publish_state_changed_public?.Invoke(thiz, (Linphone.PublishState)state);
			}
		}

		public OnPublishStateChangedDelegate OnPublishStateChanged
		{
			get
			{
				return on_publish_state_changed_public;
			}
			set
			{
				on_publish_state_changed_public = value;
				on_publish_state_changed_private = on_publish_state_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_publish_state_changed_private);
				linphone_event_cbs_set_publish_state_changed(nativePtr, cb);
#else
				linphone_event_cbs_set_publish_state_changed(nativePtr, on_publish_state_changed_private);
#endif
			}
		}
	}

	/// <summary>
	/// An object to handle the callbacks for <see cref="Linphone.Friend">
	/// Linphone.Friend</see>. 
	/// <para>
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class FriendListener : LinphoneObject
	{
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_friend_get_current_callbacks(IntPtr thiz);

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_friend_cbs_set_presence_received(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_friend_cbs_set_presence_received(IntPtr thiz, OnPresenceReceivedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnPresenceReceivedDelegatePrivate(IntPtr linphoneFriend);

		/// <summary>
		/// Callback used to notify a friend that it has received presence information. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="linphoneFriend">The <see cref="Linphone.Friend">
		/// Linphone.Friend</see> object for which the status has changed    </param>
		public delegate void OnPresenceReceivedDelegate(Linphone.Friend linphoneFriend);
		private OnPresenceReceivedDelegatePrivate on_presence_received_private;
		private OnPresenceReceivedDelegate on_presence_received_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnPresenceReceivedDelegatePrivate))]
#endif
		private static void on_presence_received(IntPtr linphoneFriend)
		{
			Friend thiz = fromNativePtr<Friend>(linphoneFriend);

			IntPtr ptr = linphone_friend_get_current_callbacks(thiz.nativePtr);
			FriendListener current_listener = fromNativePtr<FriendListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_presence_received_public?.Invoke(thiz);
			}
		}

		public OnPresenceReceivedDelegate OnPresenceReceived
		{
			get
			{
				return on_presence_received_public;
			}
			set
			{
				on_presence_received_public = value;
				on_presence_received_private = on_presence_received;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_presence_received_private);
				linphone_friend_cbs_set_presence_received(nativePtr, cb);
#else
				linphone_friend_cbs_set_presence_received(nativePtr, on_presence_received_private);
#endif
			}
		}
	}

	/// <summary>
	/// An object to handle the callbacks for <see cref="Linphone.Friend">
	/// Linphone.Friend</see> synchronization. 
	/// <para>
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class FriendListListener : LinphoneObject
	{
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_friend_list_get_current_callbacks(IntPtr thiz);

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_friend_list_cbs_set_contact_created(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_friend_list_cbs_set_contact_created(IntPtr thiz, OnContactCreatedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnContactCreatedDelegatePrivate(IntPtr friendList, IntPtr linphoneFriend);

		/// <summary>
		/// Callback used to notify a new contact has been created on the CardDAV server
		/// and downloaded locally. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="friendList">The <see cref="Linphone.FriendList">
		/// Linphone.FriendList</see> object the new contact is added to    </param>
		/// <param name="linphoneFriend">The <see cref="Linphone.Friend">
		/// Linphone.Friend</see> object that has been created    </param>
		public delegate void OnContactCreatedDelegate(Linphone.FriendList friendList, Linphone.Friend linphoneFriend);
		private OnContactCreatedDelegatePrivate on_contact_created_private;
		private OnContactCreatedDelegate on_contact_created_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnContactCreatedDelegatePrivate))]
#endif
		private static void on_contact_created(IntPtr friendList, IntPtr linphoneFriend)
		{
			FriendList thiz = fromNativePtr<FriendList>(friendList);

			IntPtr ptr = linphone_friend_list_get_current_callbacks(thiz.nativePtr);
			FriendListListener current_listener = fromNativePtr<FriendListListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_contact_created_public?.Invoke(thiz, fromNativePtr<Linphone.Friend>(linphoneFriend));
			}
		}

		public OnContactCreatedDelegate OnContactCreated
		{
			get
			{
				return on_contact_created_public;
			}
			set
			{
				on_contact_created_public = value;
				on_contact_created_private = on_contact_created;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_contact_created_private);
				linphone_friend_list_cbs_set_contact_created(nativePtr, cb);
#else
				linphone_friend_list_cbs_set_contact_created(nativePtr, on_contact_created_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_friend_list_cbs_set_contact_deleted(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_friend_list_cbs_set_contact_deleted(IntPtr thiz, OnContactDeletedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnContactDeletedDelegatePrivate(IntPtr friendList, IntPtr linphoneFriend);

		/// <summary>
		/// Callback used to notify a contact has been deleted on the CardDAV server. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="friendList">The <see cref="Linphone.FriendList">
		/// Linphone.FriendList</see> object a contact has been removed from    </param>
		/// <param name="linphoneFriend">The <see cref="Linphone.Friend">
		/// Linphone.Friend</see> object that has been deleted    </param>
		public delegate void OnContactDeletedDelegate(Linphone.FriendList friendList, Linphone.Friend linphoneFriend);
		private OnContactDeletedDelegatePrivate on_contact_deleted_private;
		private OnContactDeletedDelegate on_contact_deleted_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnContactDeletedDelegatePrivate))]
#endif
		private static void on_contact_deleted(IntPtr friendList, IntPtr linphoneFriend)
		{
			FriendList thiz = fromNativePtr<FriendList>(friendList);

			IntPtr ptr = linphone_friend_list_get_current_callbacks(thiz.nativePtr);
			FriendListListener current_listener = fromNativePtr<FriendListListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_contact_deleted_public?.Invoke(thiz, fromNativePtr<Linphone.Friend>(linphoneFriend));
			}
		}

		public OnContactDeletedDelegate OnContactDeleted
		{
			get
			{
				return on_contact_deleted_public;
			}
			set
			{
				on_contact_deleted_public = value;
				on_contact_deleted_private = on_contact_deleted;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_contact_deleted_private);
				linphone_friend_list_cbs_set_contact_deleted(nativePtr, cb);
#else
				linphone_friend_list_cbs_set_contact_deleted(nativePtr, on_contact_deleted_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_friend_list_cbs_set_contact_updated(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_friend_list_cbs_set_contact_updated(IntPtr thiz, OnContactUpdatedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnContactUpdatedDelegatePrivate(IntPtr friendList, IntPtr newFriend, IntPtr oldFriend);

		/// <summary>
		/// Callback used to notify a contact has been updated on the CardDAV server. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="friendList">The <see cref="Linphone.FriendList">
		/// Linphone.FriendList</see> object in which a contact has been updated    </param>
		/// <param name="newFriend">The new <see cref="Linphone.Friend">
		/// Linphone.Friend</see> object corresponding to the updated contact    </param>
		/// <param name="oldFriend">The old <see cref="Linphone.Friend">
		/// Linphone.Friend</see> object before update    </param>
		public delegate void OnContactUpdatedDelegate(Linphone.FriendList friendList, Linphone.Friend newFriend, Linphone.Friend oldFriend);
		private OnContactUpdatedDelegatePrivate on_contact_updated_private;
		private OnContactUpdatedDelegate on_contact_updated_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnContactUpdatedDelegatePrivate))]
#endif
		private static void on_contact_updated(IntPtr friendList, IntPtr newFriend, IntPtr oldFriend)
		{
			FriendList thiz = fromNativePtr<FriendList>(friendList);

			IntPtr ptr = linphone_friend_list_get_current_callbacks(thiz.nativePtr);
			FriendListListener current_listener = fromNativePtr<FriendListListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_contact_updated_public?.Invoke(thiz, fromNativePtr<Linphone.Friend>(newFriend), fromNativePtr<Linphone.Friend>(oldFriend));
			}
		}

		public OnContactUpdatedDelegate OnContactUpdated
		{
			get
			{
				return on_contact_updated_public;
			}
			set
			{
				on_contact_updated_public = value;
				on_contact_updated_private = on_contact_updated;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_contact_updated_private);
				linphone_friend_list_cbs_set_contact_updated(nativePtr, cb);
#else
				linphone_friend_list_cbs_set_contact_updated(nativePtr, on_contact_updated_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_friend_list_cbs_set_sync_status_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_friend_list_cbs_set_sync_status_changed(IntPtr thiz, OnSyncStatusChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnSyncStatusChangedDelegatePrivate(IntPtr friendList, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string message);

		/// <summary>
		/// Callback used to notify the status of the synchronization has changed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="friendList">The <see cref="Linphone.FriendList">
		/// Linphone.FriendList</see> object for which the status has changed    </param>
		/// <param name="status">The new <see cref="Linphone.FriendListSyncStatus">
		/// Linphone.FriendListSyncStatus</see> </param>
		/// <param name="message">An additional information on the status update    </param>
		public delegate void OnSyncStatusChangedDelegate(Linphone.FriendList friendList, Linphone.FriendListSyncStatus status, string message);
		private OnSyncStatusChangedDelegatePrivate on_sync_status_changed_private;
		private OnSyncStatusChangedDelegate on_sync_status_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnSyncStatusChangedDelegatePrivate))]
#endif
		private static void on_sync_status_changed(IntPtr friendList, int status, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string message)
		{
			FriendList thiz = fromNativePtr<FriendList>(friendList);

			IntPtr ptr = linphone_friend_list_get_current_callbacks(thiz.nativePtr);
			FriendListListener current_listener = fromNativePtr<FriendListListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_sync_status_changed_public?.Invoke(thiz, (Linphone.FriendListSyncStatus)status, message);
			}
		}

		public OnSyncStatusChangedDelegate OnSyncStatusChanged
		{
			get
			{
				return on_sync_status_changed_public;
			}
			set
			{
				on_sync_status_changed_public = value;
				on_sync_status_changed_private = on_sync_status_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_sync_status_changed_private);
				linphone_friend_list_cbs_set_sync_status_changed(nativePtr, cb);
#else
				linphone_friend_list_cbs_set_sync_status_changed(nativePtr, on_sync_status_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_friend_list_cbs_set_presence_received(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_friend_list_cbs_set_presence_received(IntPtr thiz, OnPresenceReceivedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnPresenceReceivedDelegatePrivate(IntPtr friendList, IntPtr friends);

		/// <summary>
		/// Callback used to notify a list with all friends that have received presence
		/// information. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="friendList">The LinphoneFriendList object for which the status has
		/// changed    </param>
		/// <param name="friends">A   of the relevant friends    </param>
		public delegate void OnPresenceReceivedDelegate(Linphone.FriendList friendList, IEnumerable<Linphone.Friend> friends);
		private OnPresenceReceivedDelegatePrivate on_presence_received_private;
		private OnPresenceReceivedDelegate on_presence_received_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnPresenceReceivedDelegatePrivate))]
#endif
		private static void on_presence_received(IntPtr friendList, IntPtr friends)
		{
			FriendList thiz = fromNativePtr<FriendList>(friendList);

			IntPtr ptr = linphone_friend_list_get_current_callbacks(thiz.nativePtr);
			FriendListListener current_listener = fromNativePtr<FriendListListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_presence_received_public?.Invoke(thiz, MarshalBctbxList<Linphone.Friend>(friends));
			}
		}

		public OnPresenceReceivedDelegate OnPresenceReceived
		{
			get
			{
				return on_presence_received_public;
			}
			set
			{
				on_presence_received_public = value;
				on_presence_received_private = on_presence_received;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_presence_received_private);
				linphone_friend_list_cbs_set_presence_received(nativePtr, cb);
#else
				linphone_friend_list_cbs_set_presence_received(nativePtr, on_presence_received_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_friend_list_cbs_set_new_sip_address_discovered(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_friend_list_cbs_set_new_sip_address_discovered(IntPtr thiz, OnNewSipAddressDiscoveredDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnNewSipAddressDiscoveredDelegatePrivate(IntPtr friendList, IntPtr linphoneFriend, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string sipUri);

		/// <summary>
		/// Callback used to notify a list that a new SIP address was discovered through
		/// long term presence mechanism. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="friendList">The <see cref="Linphone.FriendList">
		/// Linphone.FriendList</see> object for which the status has changed    </param>
		/// <param name="linphoneFriend">The <see cref="Linphone.Friend">
		/// Linphone.Friend</see> for which the SIP address was discovered    </param>
		/// <param name="sipUri">The newly discovered SIP URI    </param>
		public delegate void OnNewSipAddressDiscoveredDelegate(Linphone.FriendList friendList, Linphone.Friend linphoneFriend, string sipUri);
		private OnNewSipAddressDiscoveredDelegatePrivate on_new_sip_address_discovered_private;
		private OnNewSipAddressDiscoveredDelegate on_new_sip_address_discovered_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnNewSipAddressDiscoveredDelegatePrivate))]
#endif
		private static void on_new_sip_address_discovered(IntPtr friendList, IntPtr linphoneFriend, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string sipUri)
		{
			FriendList thiz = fromNativePtr<FriendList>(friendList);

			IntPtr ptr = linphone_friend_list_get_current_callbacks(thiz.nativePtr);
			FriendListListener current_listener = fromNativePtr<FriendListListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_new_sip_address_discovered_public?.Invoke(thiz, fromNativePtr<Linphone.Friend>(linphoneFriend), sipUri);
			}
		}

		public OnNewSipAddressDiscoveredDelegate OnNewSipAddressDiscovered
		{
			get
			{
				return on_new_sip_address_discovered_public;
			}
			set
			{
				on_new_sip_address_discovered_public = value;
				on_new_sip_address_discovered_private = on_new_sip_address_discovered;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_new_sip_address_discovered_private);
				linphone_friend_list_cbs_set_new_sip_address_discovered(nativePtr, cb);
#else
				linphone_friend_list_cbs_set_new_sip_address_discovered(nativePtr, on_new_sip_address_discovered_private);
#endif
			}
		}
	}

	/// <summary>
	/// Listener for <see cref="Linphone.LoggingService">Linphone.LoggingService</see>. 
	/// <para>
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class LoggingServiceListener : LinphoneObject
	{
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_logging_service_get_current_callbacks(IntPtr thiz);

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_logging_service_cbs_set_log_message_written(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_logging_service_cbs_set_log_message_written(IntPtr thiz, OnLogMessageWrittenDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnLogMessageWrittenDelegatePrivate(IntPtr logService, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string domain, int level, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string message);

		/// <summary>
		/// Type of callbacks called each time liblinphone write a log message. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="logService">A pointer on the logging service singleton.    </param>
		/// <param name="domain">A string describing which sub-library of liblinphone the
		/// message is coming from.    </param>
		/// <param name="level">Verbosity <see cref="Linphone.LogLevel">
		/// Linphone.LogLevel</see> of the message. </param>
		/// <param name="message">Content of the message.    </param>
		public delegate void OnLogMessageWrittenDelegate(Linphone.LoggingService logService, string domain, Linphone.LogLevel level, string message);
		private OnLogMessageWrittenDelegatePrivate on_log_message_written_private;
		private OnLogMessageWrittenDelegate on_log_message_written_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnLogMessageWrittenDelegatePrivate))]
#endif
		private static void on_log_message_written(IntPtr logService, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string domain, int level, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string message)
		{
			LoggingService thiz = fromNativePtr<LoggingService>(logService);

			IntPtr ptr = linphone_logging_service_get_current_callbacks(thiz.nativePtr);
			LoggingServiceListener current_listener = fromNativePtr<LoggingServiceListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_log_message_written_public?.Invoke(thiz, domain, (Linphone.LogLevel)level, message);
			}
		}

		public OnLogMessageWrittenDelegate OnLogMessageWritten
		{
			get
			{
				return on_log_message_written_public;
			}
			set
			{
				on_log_message_written_public = value;
				on_log_message_written_private = on_log_message_written;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_log_message_written_private);
				linphone_logging_service_cbs_set_log_message_written(nativePtr, cb);
#else
				linphone_logging_service_cbs_set_log_message_written(nativePtr, on_log_message_written_private);
#endif
			}
		}
	}

	/// <summary>
	/// <see cref="Linphone.MagicSearchListener">Linphone.MagicSearchListener</see> is
	/// an interface to be notified of results of contact searches initiated from the
	/// <see cref="Linphone.MagicSearch">Linphone.MagicSearch</see>. 
	/// <para>
	/// </para>
	/// </summary>
	/// <remarks>See : <see cref="Linphone.MagicSearch.AddListener()">
	/// Linphone.MagicSearch.AddListener()</see> </remarks>
	[StructLayout(LayoutKind.Sequential)]
	public class MagicSearchListener : LinphoneObject
	{
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_magic_search_get_current_callbacks(IntPtr thiz);

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_magic_search_cbs_set_search_results_received(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_magic_search_cbs_set_search_results_received(IntPtr thiz, OnSearchResultsReceivedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnSearchResultsReceivedDelegatePrivate(IntPtr magicSearch);

		/// <summary>
		/// Callback used to notify when results are received. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="magicSearch"><see cref="Linphone.MagicSearch">
		/// Linphone.MagicSearch</see> object    </param>
		public delegate void OnSearchResultsReceivedDelegate(Linphone.MagicSearch magicSearch);
		private OnSearchResultsReceivedDelegatePrivate on_search_results_received_private;
		private OnSearchResultsReceivedDelegate on_search_results_received_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnSearchResultsReceivedDelegatePrivate))]
#endif
		private static void on_search_results_received(IntPtr magicSearch)
		{
			MagicSearch thiz = fromNativePtr<MagicSearch>(magicSearch);

			IntPtr ptr = linphone_magic_search_get_current_callbacks(thiz.nativePtr);
			MagicSearchListener current_listener = fromNativePtr<MagicSearchListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_search_results_received_public?.Invoke(thiz);
			}
		}

		public OnSearchResultsReceivedDelegate OnSearchResultsReceived
		{
			get
			{
				return on_search_results_received_public;
			}
			set
			{
				on_search_results_received_public = value;
				on_search_results_received_private = on_search_results_received;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_search_results_received_private);
				linphone_magic_search_cbs_set_search_results_received(nativePtr, cb);
#else
				linphone_magic_search_cbs_set_search_results_received(nativePtr, on_search_results_received_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_magic_search_cbs_set_ldap_have_more_results(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_magic_search_cbs_set_ldap_have_more_results(IntPtr thiz, OnLdapHaveMoreResultsDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnLdapHaveMoreResultsDelegatePrivate(IntPtr magicSearch, IntPtr ldap);

		/// <summary>
		/// Callback used to notify when LDAP have more results available. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="magicSearch"><see cref="Linphone.MagicSearch">
		/// Linphone.MagicSearch</see> object    </param>
		/// <param name="ldap"><see cref="Linphone.Ldap">Linphone.Ldap</see> object   
		/// </param>
		/// deprecated : 18/11/2024 use LinphoneMagicSearchCbsMoreResultsAvailableCb
		/// instead. 
		public delegate void OnLdapHaveMoreResultsDelegate(Linphone.MagicSearch magicSearch, Linphone.Ldap ldap);
		private OnLdapHaveMoreResultsDelegatePrivate on_ldap_have_more_results_private;
		private OnLdapHaveMoreResultsDelegate on_ldap_have_more_results_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnLdapHaveMoreResultsDelegatePrivate))]
#endif
		private static void on_ldap_have_more_results(IntPtr magicSearch, IntPtr ldap)
		{
			MagicSearch thiz = fromNativePtr<MagicSearch>(magicSearch);

			IntPtr ptr = linphone_magic_search_get_current_callbacks(thiz.nativePtr);
			MagicSearchListener current_listener = fromNativePtr<MagicSearchListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_ldap_have_more_results_public?.Invoke(thiz, fromNativePtr<Linphone.Ldap>(ldap));
			}
		}

		public OnLdapHaveMoreResultsDelegate OnLdapHaveMoreResults
		{
			get
			{
				return on_ldap_have_more_results_public;
			}
			set
			{
				on_ldap_have_more_results_public = value;
				on_ldap_have_more_results_private = on_ldap_have_more_results;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_ldap_have_more_results_private);
				linphone_magic_search_cbs_set_ldap_have_more_results(nativePtr, cb);
#else
				linphone_magic_search_cbs_set_ldap_have_more_results(nativePtr, on_ldap_have_more_results_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_magic_search_cbs_set_more_results_available(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_magic_search_cbs_set_more_results_available(IntPtr thiz, OnMoreResultsAvailableDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnMoreResultsAvailableDelegatePrivate(IntPtr magicSearch, int source);

		/// <summary>
		/// Callback used to notify when more results are available for a given <see cref="Linphone.MagicSearchSource">
		/// Linphone.MagicSearchSource</see> flag. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="magicSearch"><see cref="Linphone.MagicSearch">
		/// Linphone.MagicSearch</see> object    </param>
		/// <param name="source">The source flag indicating for which type of result there
		/// is more results available. </param>
		public delegate void OnMoreResultsAvailableDelegate(Linphone.MagicSearch magicSearch, Linphone.MagicSearchSource source);
		private OnMoreResultsAvailableDelegatePrivate on_more_results_available_private;
		private OnMoreResultsAvailableDelegate on_more_results_available_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnMoreResultsAvailableDelegatePrivate))]
#endif
		private static void on_more_results_available(IntPtr magicSearch, int source)
		{
			MagicSearch thiz = fromNativePtr<MagicSearch>(magicSearch);

			IntPtr ptr = linphone_magic_search_get_current_callbacks(thiz.nativePtr);
			MagicSearchListener current_listener = fromNativePtr<MagicSearchListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_more_results_available_public?.Invoke(thiz, (Linphone.MagicSearchSource)source);
			}
		}

		public OnMoreResultsAvailableDelegate OnMoreResultsAvailable
		{
			get
			{
				return on_more_results_available_public;
			}
			set
			{
				on_more_results_available_public = value;
				on_more_results_available_private = on_more_results_available;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_more_results_available_private);
				linphone_magic_search_cbs_set_more_results_available(nativePtr, cb);
#else
				linphone_magic_search_cbs_set_more_results_available(nativePtr, on_more_results_available_private);
#endif
			}
		}
	}

	/// <summary>
	/// An object to handle the callbacks for the handling a <see cref="Linphone.ParticipantDevice">
	/// Linphone.ParticipantDevice</see> objects. 
	/// <para>
	/// Use <see cref="Linphone.Factory.CreateParticipantDeviceCbs()">
	/// Linphone.Factory.CreateParticipantDeviceCbs()</see> to create an instance. Then
	/// pass the object to a <see cref="Linphone.ParticipantDevice">
	/// Linphone.ParticipantDevice</see> instance through <see cref="Linphone.ParticipantDevice.AddListener()">
	/// Linphone.ParticipantDevice.AddListener()</see>. 
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class ParticipantDeviceListener : LinphoneObject
	{
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_participant_device_get_current_callbacks(IntPtr thiz);

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_participant_device_cbs_set_is_speaking_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_participant_device_cbs_set_is_speaking_changed(IntPtr thiz, OnIsSpeakingChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnIsSpeakingChangedDelegatePrivate(IntPtr participantDevice, char isSpeaking);

		/// <summary>
		/// Callback used to notify that is this participant device speaking has changed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="participantDevice"><see cref="Linphone.ParticipantDevice">
		/// Linphone.ParticipantDevice</see> object    </param>
		/// <param name="isSpeaking">is this participant device speaking </param>
		public delegate void OnIsSpeakingChangedDelegate(Linphone.ParticipantDevice participantDevice, bool isSpeaking);
		private OnIsSpeakingChangedDelegatePrivate on_is_speaking_changed_private;
		private OnIsSpeakingChangedDelegate on_is_speaking_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnIsSpeakingChangedDelegatePrivate))]
#endif
		private static void on_is_speaking_changed(IntPtr participantDevice, char isSpeaking)
		{
			ParticipantDevice thiz = fromNativePtr<ParticipantDevice>(participantDevice);

			IntPtr ptr = linphone_participant_device_get_current_callbacks(thiz.nativePtr);
			ParticipantDeviceListener current_listener = fromNativePtr<ParticipantDeviceListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_is_speaking_changed_public?.Invoke(thiz, isSpeaking == 0);
			}
		}

		public OnIsSpeakingChangedDelegate OnIsSpeakingChanged
		{
			get
			{
				return on_is_speaking_changed_public;
			}
			set
			{
				on_is_speaking_changed_public = value;
				on_is_speaking_changed_private = on_is_speaking_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_is_speaking_changed_private);
				linphone_participant_device_cbs_set_is_speaking_changed(nativePtr, cb);
#else
				linphone_participant_device_cbs_set_is_speaking_changed(nativePtr, on_is_speaking_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_participant_device_cbs_set_is_muted(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_participant_device_cbs_set_is_muted(IntPtr thiz, OnIsMutedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnIsMutedDelegatePrivate(IntPtr participantDevice, char isMuted);

		/// <summary>
		/// Callback used to notify that this participant device is muted or is no longer
		/// muted. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="participantDevice"><see cref="Linphone.ParticipantDevice">
		/// Linphone.ParticipantDevice</see> object    </param>
		/// <param name="isMuted">is this participant device muted </param>
		public delegate void OnIsMutedDelegate(Linphone.ParticipantDevice participantDevice, bool isMuted);
		private OnIsMutedDelegatePrivate on_is_muted_private;
		private OnIsMutedDelegate on_is_muted_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnIsMutedDelegatePrivate))]
#endif
		private static void on_is_muted(IntPtr participantDevice, char isMuted)
		{
			ParticipantDevice thiz = fromNativePtr<ParticipantDevice>(participantDevice);

			IntPtr ptr = linphone_participant_device_get_current_callbacks(thiz.nativePtr);
			ParticipantDeviceListener current_listener = fromNativePtr<ParticipantDeviceListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_is_muted_public?.Invoke(thiz, isMuted == 0);
			}
		}

		public OnIsMutedDelegate OnIsMuted
		{
			get
			{
				return on_is_muted_public;
			}
			set
			{
				on_is_muted_public = value;
				on_is_muted_private = on_is_muted;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_is_muted_private);
				linphone_participant_device_cbs_set_is_muted(nativePtr, cb);
#else
				linphone_participant_device_cbs_set_is_muted(nativePtr, on_is_muted_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_participant_device_cbs_set_screen_sharing_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_participant_device_cbs_set_screen_sharing_changed(IntPtr thiz, OnScreenSharingChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnScreenSharingChangedDelegatePrivate(IntPtr participantDevice, char isScreenSharing);

		/// <summary>
		/// Callback used to notify that this participant device is screen sharing or is no
		/// longer screen sharing. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="participantDevice"><see cref="Linphone.ParticipantDevice">
		/// Linphone.ParticipantDevice</see> object    </param>
		/// <param name="isScreenSharing">is this participant device screen sharing </param>
		public delegate void OnScreenSharingChangedDelegate(Linphone.ParticipantDevice participantDevice, bool isScreenSharing);
		private OnScreenSharingChangedDelegatePrivate on_screen_sharing_changed_private;
		private OnScreenSharingChangedDelegate on_screen_sharing_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnScreenSharingChangedDelegatePrivate))]
#endif
		private static void on_screen_sharing_changed(IntPtr participantDevice, char isScreenSharing)
		{
			ParticipantDevice thiz = fromNativePtr<ParticipantDevice>(participantDevice);

			IntPtr ptr = linphone_participant_device_get_current_callbacks(thiz.nativePtr);
			ParticipantDeviceListener current_listener = fromNativePtr<ParticipantDeviceListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_screen_sharing_changed_public?.Invoke(thiz, isScreenSharing == 0);
			}
		}

		public OnScreenSharingChangedDelegate OnScreenSharingChanged
		{
			get
			{
				return on_screen_sharing_changed_public;
			}
			set
			{
				on_screen_sharing_changed_public = value;
				on_screen_sharing_changed_private = on_screen_sharing_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_screen_sharing_changed_private);
				linphone_participant_device_cbs_set_screen_sharing_changed(nativePtr, cb);
#else
				linphone_participant_device_cbs_set_screen_sharing_changed(nativePtr, on_screen_sharing_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_participant_device_cbs_set_state_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_participant_device_cbs_set_state_changed(IntPtr thiz, OnStateChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnStateChangedDelegatePrivate(IntPtr participantDevice, int state);

		/// <summary>
		/// Callback used to notify that participant device changed state. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="participantDevice">LinphoneParticipantDevice object    </param>
		/// <param name="state">new participant device state </param>
		public delegate void OnStateChangedDelegate(Linphone.ParticipantDevice participantDevice, Linphone.ParticipantDeviceState state);
		private OnStateChangedDelegatePrivate on_state_changed_private;
		private OnStateChangedDelegate on_state_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnStateChangedDelegatePrivate))]
#endif
		private static void on_state_changed(IntPtr participantDevice, int state)
		{
			ParticipantDevice thiz = fromNativePtr<ParticipantDevice>(participantDevice);

			IntPtr ptr = linphone_participant_device_get_current_callbacks(thiz.nativePtr);
			ParticipantDeviceListener current_listener = fromNativePtr<ParticipantDeviceListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_state_changed_public?.Invoke(thiz, (Linphone.ParticipantDeviceState)state);
			}
		}

		public OnStateChangedDelegate OnStateChanged
		{
			get
			{
				return on_state_changed_public;
			}
			set
			{
				on_state_changed_public = value;
				on_state_changed_private = on_state_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_state_changed_private);
				linphone_participant_device_cbs_set_state_changed(nativePtr, cb);
#else
				linphone_participant_device_cbs_set_state_changed(nativePtr, on_state_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_participant_device_cbs_set_stream_capability_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_participant_device_cbs_set_stream_capability_changed(IntPtr thiz, OnStreamCapabilityChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnStreamCapabilityChangedDelegatePrivate(IntPtr participantDevice, int direction, int streamType);

		/// <summary>
		/// Callback used to notify that participant device stream capability has changed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="participantDevice"><see cref="Linphone.ParticipantDevice">
		/// Linphone.ParticipantDevice</see> object    </param>
		/// <param name="direction">participant device&apos;s stream direction </param>
		/// <param name="streamType">type of stream: audio, video or text </param>
		public delegate void OnStreamCapabilityChangedDelegate(Linphone.ParticipantDevice participantDevice, Linphone.MediaDirection direction, Linphone.StreamType streamType);
		private OnStreamCapabilityChangedDelegatePrivate on_stream_capability_changed_private;
		private OnStreamCapabilityChangedDelegate on_stream_capability_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnStreamCapabilityChangedDelegatePrivate))]
#endif
		private static void on_stream_capability_changed(IntPtr participantDevice, int direction, int streamType)
		{
			ParticipantDevice thiz = fromNativePtr<ParticipantDevice>(participantDevice);

			IntPtr ptr = linphone_participant_device_get_current_callbacks(thiz.nativePtr);
			ParticipantDeviceListener current_listener = fromNativePtr<ParticipantDeviceListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_stream_capability_changed_public?.Invoke(thiz, (Linphone.MediaDirection)direction, (Linphone.StreamType)streamType);
			}
		}

		public OnStreamCapabilityChangedDelegate OnStreamCapabilityChanged
		{
			get
			{
				return on_stream_capability_changed_public;
			}
			set
			{
				on_stream_capability_changed_public = value;
				on_stream_capability_changed_private = on_stream_capability_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_stream_capability_changed_private);
				linphone_participant_device_cbs_set_stream_capability_changed(nativePtr, cb);
#else
				linphone_participant_device_cbs_set_stream_capability_changed(nativePtr, on_stream_capability_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_participant_device_cbs_set_thumbnail_stream_capability_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_participant_device_cbs_set_thumbnail_stream_capability_changed(IntPtr thiz, OnThumbnailStreamCapabilityChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnThumbnailStreamCapabilityChangedDelegatePrivate(IntPtr participantDevice, int direction);

		/// <summary>
		/// Callback used to notify that participant device thumbnail stream capability has
		/// changed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="participantDevice"><see cref="Linphone.ParticipantDevice">
		/// Linphone.ParticipantDevice</see> object    </param>
		/// <param name="direction">participant device&apos;s thumbnail direction </param>
		public delegate void OnThumbnailStreamCapabilityChangedDelegate(Linphone.ParticipantDevice participantDevice, Linphone.MediaDirection direction);
		private OnThumbnailStreamCapabilityChangedDelegatePrivate on_thumbnail_stream_capability_changed_private;
		private OnThumbnailStreamCapabilityChangedDelegate on_thumbnail_stream_capability_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnThumbnailStreamCapabilityChangedDelegatePrivate))]
#endif
		private static void on_thumbnail_stream_capability_changed(IntPtr participantDevice, int direction)
		{
			ParticipantDevice thiz = fromNativePtr<ParticipantDevice>(participantDevice);

			IntPtr ptr = linphone_participant_device_get_current_callbacks(thiz.nativePtr);
			ParticipantDeviceListener current_listener = fromNativePtr<ParticipantDeviceListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_thumbnail_stream_capability_changed_public?.Invoke(thiz, (Linphone.MediaDirection)direction);
			}
		}

		public OnThumbnailStreamCapabilityChangedDelegate OnThumbnailStreamCapabilityChanged
		{
			get
			{
				return on_thumbnail_stream_capability_changed_public;
			}
			set
			{
				on_thumbnail_stream_capability_changed_public = value;
				on_thumbnail_stream_capability_changed_private = on_thumbnail_stream_capability_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_thumbnail_stream_capability_changed_private);
				linphone_participant_device_cbs_set_thumbnail_stream_capability_changed(nativePtr, cb);
#else
				linphone_participant_device_cbs_set_thumbnail_stream_capability_changed(nativePtr, on_thumbnail_stream_capability_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_participant_device_cbs_set_stream_availability_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_participant_device_cbs_set_stream_availability_changed(IntPtr thiz, OnStreamAvailabilityChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnStreamAvailabilityChangedDelegatePrivate(IntPtr participantDevice, char available, int streamType);

		/// <summary>
		/// Callback used to notify that participant device stream availability has
		/// changed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="participantDevice"><see cref="Linphone.ParticipantDevice">
		/// Linphone.ParticipantDevice</see> object    </param>
		/// <param name="available">true if the stream is available on our side </param>
		/// <param name="streamType">type of stream: audio, video or text </param>
		public delegate void OnStreamAvailabilityChangedDelegate(Linphone.ParticipantDevice participantDevice, bool available, Linphone.StreamType streamType);
		private OnStreamAvailabilityChangedDelegatePrivate on_stream_availability_changed_private;
		private OnStreamAvailabilityChangedDelegate on_stream_availability_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnStreamAvailabilityChangedDelegatePrivate))]
#endif
		private static void on_stream_availability_changed(IntPtr participantDevice, char available, int streamType)
		{
			ParticipantDevice thiz = fromNativePtr<ParticipantDevice>(participantDevice);

			IntPtr ptr = linphone_participant_device_get_current_callbacks(thiz.nativePtr);
			ParticipantDeviceListener current_listener = fromNativePtr<ParticipantDeviceListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_stream_availability_changed_public?.Invoke(thiz, available == 0, (Linphone.StreamType)streamType);
			}
		}

		public OnStreamAvailabilityChangedDelegate OnStreamAvailabilityChanged
		{
			get
			{
				return on_stream_availability_changed_public;
			}
			set
			{
				on_stream_availability_changed_public = value;
				on_stream_availability_changed_private = on_stream_availability_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_stream_availability_changed_private);
				linphone_participant_device_cbs_set_stream_availability_changed(nativePtr, cb);
#else
				linphone_participant_device_cbs_set_stream_availability_changed(nativePtr, on_stream_availability_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_participant_device_cbs_set_thumbnail_stream_availability_changed(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_participant_device_cbs_set_thumbnail_stream_availability_changed(IntPtr thiz, OnThumbnailStreamAvailabilityChangedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnThumbnailStreamAvailabilityChangedDelegatePrivate(IntPtr participantDevice, char available);

		/// <summary>
		/// Callback used to notify that participant device thumbnail stream availability
		/// has changed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="participantDevice"><see cref="Linphone.ParticipantDevice">
		/// Linphone.ParticipantDevice</see> object    </param>
		/// <param name="available">participant device&apos;s thumbnail stream availability
		/// </param>
		public delegate void OnThumbnailStreamAvailabilityChangedDelegate(Linphone.ParticipantDevice participantDevice, bool available);
		private OnThumbnailStreamAvailabilityChangedDelegatePrivate on_thumbnail_stream_availability_changed_private;
		private OnThumbnailStreamAvailabilityChangedDelegate on_thumbnail_stream_availability_changed_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnThumbnailStreamAvailabilityChangedDelegatePrivate))]
#endif
		private static void on_thumbnail_stream_availability_changed(IntPtr participantDevice, char available)
		{
			ParticipantDevice thiz = fromNativePtr<ParticipantDevice>(participantDevice);

			IntPtr ptr = linphone_participant_device_get_current_callbacks(thiz.nativePtr);
			ParticipantDeviceListener current_listener = fromNativePtr<ParticipantDeviceListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_thumbnail_stream_availability_changed_public?.Invoke(thiz, available == 0);
			}
		}

		public OnThumbnailStreamAvailabilityChangedDelegate OnThumbnailStreamAvailabilityChanged
		{
			get
			{
				return on_thumbnail_stream_availability_changed_public;
			}
			set
			{
				on_thumbnail_stream_availability_changed_public = value;
				on_thumbnail_stream_availability_changed_private = on_thumbnail_stream_availability_changed;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_thumbnail_stream_availability_changed_private);
				linphone_participant_device_cbs_set_thumbnail_stream_availability_changed(nativePtr, cb);
#else
				linphone_participant_device_cbs_set_thumbnail_stream_availability_changed(nativePtr, on_thumbnail_stream_availability_changed_private);
#endif
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_participant_device_cbs_set_video_display_error_occurred(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_participant_device_cbs_set_video_display_error_occurred(IntPtr thiz, OnVideoDisplayErrorOccurredDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnVideoDisplayErrorOccurredDelegatePrivate(IntPtr participantDevice, int errorCode);

		/// <summary>
		/// Callback to notify that there are errors from the video rendering of the
		/// participant device. 
		/// <para>
		/// Check LinphoneCallCbsVideoDisplayErrorOccurredCb for more details.
		/// </para>
		/// </summary>
		/// <param name="participantDevice">LinphoneParticipantDevice object    </param>
		/// <param name="errorCode">the error code coming from the display render. </param>
		public delegate void OnVideoDisplayErrorOccurredDelegate(Linphone.ParticipantDevice participantDevice, int errorCode);
		private OnVideoDisplayErrorOccurredDelegatePrivate on_video_display_error_occurred_private;
		private OnVideoDisplayErrorOccurredDelegate on_video_display_error_occurred_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnVideoDisplayErrorOccurredDelegatePrivate))]
#endif
		private static void on_video_display_error_occurred(IntPtr participantDevice, int errorCode)
		{
			ParticipantDevice thiz = fromNativePtr<ParticipantDevice>(participantDevice);

			IntPtr ptr = linphone_participant_device_get_current_callbacks(thiz.nativePtr);
			ParticipantDeviceListener current_listener = fromNativePtr<ParticipantDeviceListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_video_display_error_occurred_public?.Invoke(thiz, errorCode);
			}
		}

		public OnVideoDisplayErrorOccurredDelegate OnVideoDisplayErrorOccurred
		{
			get
			{
				return on_video_display_error_occurred_public;
			}
			set
			{
				on_video_display_error_occurred_public = value;
				on_video_display_error_occurred_private = on_video_display_error_occurred;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_video_display_error_occurred_private);
				linphone_participant_device_cbs_set_video_display_error_occurred(nativePtr, cb);
#else
				linphone_participant_device_cbs_set_video_display_error_occurred(nativePtr, on_video_display_error_occurred_private);
#endif
			}
		}
	}

	/// <summary>
	/// An object to handle the callbacks for the handling a <see cref="Linphone.Player">
	/// Linphone.Player</see> objects. 
	/// <para>
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class PlayerListener : LinphoneObject
	{
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_player_get_current_callbacks(IntPtr thiz);

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_player_cbs_set_eof_reached(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_player_cbs_set_eof_reached(IntPtr thiz, OnEofReachedDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnEofReachedDelegatePrivate(IntPtr player);

		/// <summary>
		/// Callback for notifying end of play (file). 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="player">The <see cref="Linphone.Player">Linphone.Player</see>
		/// object    </param>
		public delegate void OnEofReachedDelegate(Linphone.Player player);
		private OnEofReachedDelegatePrivate on_eof_reached_private;
		private OnEofReachedDelegate on_eof_reached_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnEofReachedDelegatePrivate))]
#endif
		private static void on_eof_reached(IntPtr player)
		{
			Player thiz = fromNativePtr<Player>(player);

			IntPtr ptr = linphone_player_get_current_callbacks(thiz.nativePtr);
			PlayerListener current_listener = fromNativePtr<PlayerListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_eof_reached_public?.Invoke(thiz);
			}
		}

		public OnEofReachedDelegate OnEofReached
		{
			get
			{
				return on_eof_reached_public;
			}
			set
			{
				on_eof_reached_public = value;
				on_eof_reached_private = on_eof_reached;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_eof_reached_private);
				linphone_player_cbs_set_eof_reached(nativePtr, cb);
#else
				linphone_player_cbs_set_eof_reached(nativePtr, on_eof_reached_private);
#endif
			}
		}
	}

	/// <summary>
	/// An object to handle the callbacks for handling the <see cref="Linphone.XmlRpcRequest">
	/// Linphone.XmlRpcRequest</see> operations. 
	/// <para>
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class XmlRpcRequestListener : LinphoneObject
	{
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_xml_rpc_request_get_current_callbacks(IntPtr thiz);

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
#if !(ANDROID || IOS)
		static extern void linphone_xml_rpc_request_cbs_set_response(IntPtr thiz, IntPtr cb);
#else
		static extern void linphone_xml_rpc_request_cbs_set_response(IntPtr thiz, OnResponseDelegatePrivate cb);
#endif
		
		[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
		private delegate void OnResponseDelegatePrivate(IntPtr request);

		/// <summary>
		/// Callback used to notify the response to an XML-RPC request. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="request"><see cref="Linphone.XmlRpcRequest">
		/// Linphone.XmlRpcRequest</see> object    </param>
		public delegate void OnResponseDelegate(Linphone.XmlRpcRequest request);
		private OnResponseDelegatePrivate on_response_private;
		private OnResponseDelegate on_response_public;

#if IOS
		[MonoPInvokeCallback(typeof(OnResponseDelegatePrivate))]
#endif
		private static void on_response(IntPtr request)
		{
			XmlRpcRequest thiz = fromNativePtr<XmlRpcRequest>(request);

			IntPtr ptr = linphone_xml_rpc_request_get_current_callbacks(thiz.nativePtr);
			XmlRpcRequestListener current_listener = fromNativePtr<XmlRpcRequestListener>(ptr, false, true);

			if (current_listener != null)
			{
				current_listener.on_response_public?.Invoke(thiz);
			}
		}

		public OnResponseDelegate OnResponse
		{
			get
			{
				return on_response_public;
			}
			set
			{
				on_response_public = value;
				on_response_private = on_response;
#if !(IOS || ANDROID)
				IntPtr cb = Marshal.GetFunctionPointerForDelegate(on_response_private);
				linphone_xml_rpc_request_cbs_set_response(nativePtr, cb);
#else
				linphone_xml_rpc_request_cbs_set_response(nativePtr, on_response_private);
#endif
			}
		}
	}

#endregion

#region Classes
	/// <summary>
	/// Object that represents a Linphone Account. 
	/// <para>
	/// This object replaces the deprecated <see cref="Linphone.ProxyConfig">
	/// Linphone.ProxyConfig</see>. Use a <see cref="Linphone.AccountParams">
	/// Linphone.AccountParams</see> object to configure it. 
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class Account : LinphoneObject
	{

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_factory_get();

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_factory_create_account_cbs(IntPtr factory);

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_add_callbacks(IntPtr thiz, IntPtr cbs);

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_remove_callbacks(IntPtr thiz, IntPtr cbs);

		private AccountListener listener;

		public AccountListener Listener
		{
			get {
				if (listener == null)
				{
					IntPtr nativeListener = linphone_factory_create_account_cbs(linphone_factory_get());
					listener = fromNativePtr<AccountListener>(nativeListener, false, true);
					linphone_account_add_callbacks(nativePtr, nativeListener);
					belle_sip_object_unref(nativeListener);
				}
				return listener;
			}
			set {
				if (listener != null && value == null) {
					linphone_account_remove_callbacks(nativePtr, listener.nativePtr);
				}
				listener = value;
			}
		}

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_new_with_config(IntPtr lc, IntPtr parameters, IntPtr config);

		/// <summary>
		/// Create a new <see cref="Linphone.Account">Linphone.Account</see> with a Proxy
		/// config backpointer. 
		/// <para>
		/// This is only intended to be used while keeping a backward compatibility with
		/// proxy config. 
		/// </para>
		/// </summary>
		/// <param name="lc">The <see cref="Linphone.Core">Linphone.Core</see> object.   
		/// </param>
		/// <param name="parameters">The <see cref="Linphone.AccountParams">
		/// Linphone.AccountParams</see> object.    </param>
		/// <param name="config">The <see cref="Linphone.ProxyConfig">
		/// Linphone.ProxyConfig</see> object.    </param>
		/// <returns>The newly created <see cref="Linphone.Account">Linphone.Account</see>
		/// object.    </returns>
		public static Linphone.Account NewWithConfig(Linphone.Core lc, Linphone.AccountParams parameters, Linphone.ProxyConfig config)
		{
			
			IntPtr ptr = linphone_account_new_with_config(lc != null ? lc.nativePtr : IntPtr.Zero, parameters != null ? parameters.nativePtr : IntPtr.Zero, config != null ? config.nativePtr : IntPtr.Zero);
			Linphone.Account returnVal = fromNativePtr<Linphone.Account>(ptr, false);
			
			GC.KeepAlive(lc);
GC.KeepAlive(parameters);
GC.KeepAlive(config);
			return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_account_avpf_enabled(IntPtr thiz);

		/// <summary>
		/// Indicates whether AVPF/SAVPF is being used for calls using this account. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if AVPF/SAVPF is enabled, false otherwise. </returns>
		public bool AvpfEnabled
		{
			get
			{
				return linphone_account_avpf_enabled(nativePtr) != 0;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_get_call_logs(IntPtr thiz);

		/// <summary>
		/// Returns the list of call logs for a given account. 
		/// <para>
		/// This list must be freed after use. 
		/// </para>
		/// </summary>
		/// <returns>The list of call logs  .       </returns>
		public IEnumerable<Linphone.CallLog> CallLogs
		{
			get
			{
				return MarshalBctbxList<Linphone.CallLog>(linphone_account_get_call_logs(nativePtr), true);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_get_chat_rooms(IntPtr thiz);

		/// <summary>
		/// Returns the list of chat rooms for a given account. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The list of chat rooms  .    </returns>
		public IEnumerable<Linphone.ChatRoom> ChatRooms
		{
			get
			{
				return MarshalBctbxList<Linphone.ChatRoom>(linphone_account_get_chat_rooms(nativePtr), true);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_get_conference_information_list(IntPtr thiz);

		/// <summary>
		/// Returns the list of conference information stored locally for a given account. 
		/// <para>
		/// This list must be freed after use. 
		/// </para>
		/// </summary>
		/// <returns>The list of call logs  .    </returns>
		/// <remarks>Warning : this method also start the synchronization with the CCMP
		/// server, should it be defined in the #AccountParams. The application may want to
		/// wait for the callback conference_information_updated to get a up-to-date list
		/// of conferences </remarks> 
		public IEnumerable<Linphone.ConferenceInfo> ConferenceInformationList
		{
			get
			{
				return MarshalBctbxList<Linphone.ConferenceInfo>(linphone_account_get_conference_information_list(nativePtr), true);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_get_contact_address(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_set_contact_address(IntPtr thiz, IntPtr addr);

		/// <summary>
		/// Return the contact address of the account. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>a <see cref="Linphone.Address">Linphone.Address</see> correspong to
		/// the contact address of the account.    </returns>
		public Linphone.Address ContactAddress
		{
			get
			{
				IntPtr ptr = linphone_account_get_contact_address(nativePtr);
				Linphone.Address obj = fromNativePtr<Linphone.Address>(ptr, true);
				return obj;
			}
			set
			{
				linphone_account_set_contact_address(nativePtr, value.nativePtr);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_get_core(IntPtr thiz);

		/// <summary>
		/// Get the <see cref="Linphone.Core">Linphone.Core</see> object to which is
		/// associated the <see cref="Linphone.Account">Linphone.Account</see>. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The <see cref="Linphone.Core">Linphone.Core</see> object to which is
		/// associated the <see cref="Linphone.Account">Linphone.Account</see>.   
		/// </returns>
		public Linphone.Core Core
		{
			get
			{
				IntPtr ptr = linphone_account_get_core(nativePtr);
				Linphone.Core obj = fromNativePtr<Linphone.Core>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_get_dependency(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_set_dependency(IntPtr thiz, IntPtr dependsOn);

		/// <summary>
		/// Get the dependency of a <see cref="Linphone.Account">Linphone.Account</see>. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The account this one is dependent upon, or null if not marked
		/// dependent.    </returns>
		public Linphone.Account Dependency
		{
			get
			{
				IntPtr ptr = linphone_account_get_dependency(nativePtr);
				Linphone.Account obj = fromNativePtr<Linphone.Account>(ptr, true);
				return obj;
			}
			set
			{
				linphone_account_set_dependency(nativePtr, value.nativePtr);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.Reason linphone_account_get_error(IntPtr thiz);

		/// <summary>
		/// Get the reason why registration failed when the account state is
		/// LinphoneRegistrationFailed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The <see cref="Linphone.Reason">Linphone.Reason</see> why registration
		/// failed for this account. </returns>
		public Linphone.Reason Error
		{
			get
			{
				return linphone_account_get_error(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_get_error_info(IntPtr thiz);

		/// <summary>
		/// Get detailed information why registration failed when the account state is
		/// LinphoneRegistrationFailed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The <see cref="Linphone.ErrorInfo">Linphone.ErrorInfo</see> explaining
		/// why registration failed for this account.    </returns>
		public Linphone.ErrorInfo ErrorInfo
		{
			get
			{
				IntPtr ptr = linphone_account_get_error_info(nativePtr);
				Linphone.ErrorInfo obj = fromNativePtr<Linphone.ErrorInfo>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_account_is_avpf_enabled(IntPtr thiz);

		/// <summary>
		/// Indicates whether AVPF/SAVPF is being used for calls using this account. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if AVPF/SAVPF is enabled, false otherwise. </returns>
		/// deprecated : 16/12/2021 Use <see cref="Linphone.Account.AvpfEnabled()">
		/// Linphone.Account.AvpfEnabled()</see> instead. 
		public bool IsAvpfEnabled
		{
			get
			{
				return linphone_account_is_avpf_enabled(nativePtr) != 0;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_account_get_missed_calls_count(IntPtr thiz);

		/// <summary>
		/// Returns the missed calls count for a given account. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The missed calls count. </returns>
		public int MissedCallsCount
		{
			get
			{
				return linphone_account_get_missed_calls_count(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_get_params(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_account_set_params(IntPtr thiz, IntPtr parameters);

		/// <summary>
		/// Get the <see cref="Linphone.AccountParams">Linphone.AccountParams</see> as
		/// read-only object. 
		/// <para>
		/// To make changes, clone the returned object using <see cref="Linphone.AccountParams.Clone()">
		/// Linphone.AccountParams.Clone()</see> method, make your changes on it and apply
		/// them using with <see cref="Linphone.Account.Params">
		/// Linphone.Account.Params</see>. 
		/// </para>
		/// </summary>
		/// <returns>The <see cref="Linphone.AccountParams">Linphone.AccountParams</see>
		/// attached to this account.    </returns>
		public Linphone.AccountParams Params
		{
			get
			{
				IntPtr ptr = linphone_account_get_params(nativePtr);
				Linphone.AccountParams obj = fromNativePtr<Linphone.AccountParams>(ptr, true);
				return obj;
			}
			set
			{
				linphone_account_set_params(nativePtr, value.nativePtr);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.RegistrationState linphone_account_get_state(IntPtr thiz);

		/// <summary>
		/// Get the registration state of the given account. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The <see cref="Linphone.RegistrationState">
		/// Linphone.RegistrationState</see> of the account. </returns>
		public Linphone.RegistrationState State
		{
			get
			{
				return linphone_account_get_state(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.TransportType linphone_account_get_transport(IntPtr thiz);

		/// <summary>
		/// Get the transport from either service route, route or addr. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The transport as a string (I.E udp, tcp, tls, dtls). </returns>
		/// deprecated : 01/03/2021 Use Linphone_account_params_get_transport() instead. 
		public Linphone.TransportType Transport
		{
			get
			{
				return linphone_account_get_transport(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_account_get_unread_chat_message_count(IntPtr thiz);

		/// <summary>
		/// Returns the unread chat message count for a given account. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The unread chat message count. </returns>
		public int UnreadChatMessageCount
		{
			get
			{
				return linphone_account_get_unread_chat_message_count(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_add_custom_param(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string key, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string val);

		/// <summary>
		/// Set one custom parameter to this <see cref="Linphone.Account">
		/// Linphone.Account</see>. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="key">key of the searched parameter.    </param>
		/// <param name="val">value of the searched parameter.    </param>
		public void AddCustomParam(string key, string val)
		{
			
			linphone_account_add_custom_param(nativePtr, key, val);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_clear_call_logs(IntPtr thiz);

		/// <summary>
		/// Deletes all the call logs related to this account from the database. 
		/// <para>
		/// </para>
		/// </summary>
		public void ClearCallLogs()
		{
			
			linphone_account_clear_call_logs(nativePtr);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_clone(IntPtr thiz);

		/// <summary>
		/// Instantiate a new account with values from source. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The newly created <see cref="Linphone.Account">Linphone.Account</see>
		/// object.    </returns>
		public Linphone.Account Clone()
		{
			
			IntPtr ptr = linphone_account_clone(nativePtr);
			Linphone.Account returnVal = fromNativePtr<Linphone.Account>(ptr, false);
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_filter_chat_rooms(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string filter);

		/// <summary>
		/// Returns a filtered list of chat rooms for a given account. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="filter">the criteria a chat room must meet somehow (subject,
		/// participant address, friend&apos;s name).    </param>
		/// <returns>The list of chat rooms  .       </returns>
		public IEnumerable<Linphone.ChatRoom> FilterChatRooms(string filter)
		{
			
			IEnumerable<Linphone.ChatRoom> returnVal = MarshalBctbxList<Linphone.ChatRoom>(linphone_account_filter_chat_rooms(nativePtr, filter), false);
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_find_auth_info(IntPtr thiz);

		/// <summary>
		/// Find authentication info matching account, if any, similarly to
		/// linphone_core_find_auth_info. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>a <see cref="Linphone.AuthInfo">Linphone.AuthInfo</see> matching
		/// account criteria if possible, null if nothing can be found.    </returns>
		public Linphone.AuthInfo FindAuthInfo()
		{
			
			IntPtr ptr = linphone_account_find_auth_info(nativePtr);
			Linphone.AuthInfo returnVal = fromNativePtr<Linphone.AuthInfo>(ptr, true);
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_get_call_logs_for_address(IntPtr thiz, IntPtr remoteAddress);

		/// <summary>
		/// Returns the list of call logs for a given account. 
		/// <para>
		/// This list must be freed after use. 
		/// </para>
		/// </summary>
		/// <param name="remoteAddress">the <see cref="Linphone.Address">
		/// Linphone.Address</see> object to filter call logs.    </param>
		/// <returns>The list of filtered call logs  .       </returns>
		public IEnumerable<Linphone.CallLog> GetCallLogsForAddress(Linphone.Address remoteAddress)
		{
			
			IEnumerable<Linphone.CallLog> returnVal = MarshalBctbxList<Linphone.CallLog>(linphone_account_get_call_logs_for_address(nativePtr, remoteAddress != null ? remoteAddress.nativePtr : IntPtr.Zero), false);
			
			GC.KeepAlive(remoteAddress);
			return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_get_custom_header(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string headerName);

		/// <summary>
		/// Obtain the value of a header sent by the server in last answer to REGISTER. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="headerName">The header name for which to fetch corresponding
		/// value.    </param>
		/// <returns>The value of the queried header.    </returns>
		public string GetCustomHeader(string headerName)
		{
			
			string returnVal = linphone_pointer_to_string(linphone_account_get_custom_header(nativePtr, headerName));
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_get_custom_param(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string key);

		/// <summary>
		/// Get the custom parameter with key to this <see cref="Linphone.Account">
		/// Linphone.Account</see>. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="key">key of the searched parameter.    </param>
		/// <returns>The value of the parameter with key if found or an empty string
		/// otherwise.    </returns>
		public string GetCustomParam(string key)
		{
			
			string returnVal = linphone_pointer_to_string(linphone_account_get_custom_param(nativePtr, key));
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_account_is_phone_number(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string username);

		/// <summary>
		/// Detect if the given input is a phone number or not. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="username">The string to parse.    </param>
		/// <returns>true if input is a phone number, false otherwise. </returns>
		public bool IsPhoneNumber(string username)
		{
			
			bool returnVal = linphone_account_is_phone_number(nativePtr, username) == (char)0 ? false : true;
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_normalize_phone_number(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string username);

		/// <summary>
		/// Normalize a human readable phone number into a basic string. 
		/// <para>
		/// 888-444-222 becomes 888444222 or +33888444222 depending on the <see cref="Linphone.Account">
		/// Linphone.Account</see> object. This function will always generate a normalized
		/// username if input is a phone number. 
		/// </para>
		/// </summary>
		/// <param name="username">The string to parse.    </param>
		/// <returns>null if input is an invalid phone number, normalized phone number from
		/// username input otherwise.       </returns>
		public string NormalizePhoneNumber(string username)
		{
			
			string returnVal = linphone_pointer_to_string(linphone_account_normalize_phone_number(nativePtr, username));
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_normalize_sip_uri(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string username);

		/// <summary>
		/// Normalize a human readable sip uri into a fully qualified LinphoneAddress. 
		/// <para>
		/// A sip address should look like DisplayName &lt;sip:username@domain:port&gt; .
		/// Basically this function performs the following tasks
		/// The result is a syntactically correct SIP address. 
		/// </para>
		/// </summary>
		/// <param name="username">The string to parse.    </param>
		/// <returns>null if invalid input, normalized sip address otherwise.      
		/// </returns>
		public Linphone.Address NormalizeSipUri(string username)
		{
			
			IntPtr ptr = linphone_account_normalize_sip_uri(nativePtr, username);
			Linphone.Address returnVal = fromNativePtr<Linphone.Address>(ptr, false);
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_pause_register(IntPtr thiz);

		/// <summary>
		/// Prevent an account from refreshing its registration. 
		/// <para>
		/// This is useful to let registrations to expire naturally (or) when the
		/// application wants to keep control on when refreshes are sent. However,
		/// linphone_core_set_network_reachable(lc,true) will always request the accounts
		/// to refresh their registrations. The refreshing operations can be resumed with
		/// <see cref="Linphone.Account.RefreshRegister()">
		/// Linphone.Account.RefreshRegister()</see>. 
		/// </para>
		/// </summary>
		public void PauseRegister()
		{
			
			linphone_account_pause_register(nativePtr);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_refresh_register(IntPtr thiz);

		/// <summary>
		/// Refresh a proxy registration. 
		/// <para>
		/// This is useful if for example you resuming from suspend, thus IP address may
		/// have changed. 
		/// </para>
		/// </summary>
		public void RefreshRegister()
		{
			
			linphone_account_refresh_register(nativePtr);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_reset_missed_calls_count(IntPtr thiz);

		/// <summary>
		/// Re-sets the number of missed calls for this account to 0. 
		/// <para>
		/// </para>
		/// </summary>
		public void ResetMissedCallsCount()
		{
			
			linphone_account_reset_missed_calls_count(nativePtr);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_set_custom_header(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string headerName, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string headerValue);

		/// <summary>
		/// Set the value of a custom header sent to the server in REGISTERs request. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="headerName">The header name.    </param>
		/// <param name="headerValue">The header value.    </param>
		public void SetCustomHeader(string headerName, string headerValue)
		{
			
			linphone_account_set_custom_header(nativePtr, headerName, headerValue);
			
			
						
		}
	}
	/// <summary>
	/// The object used to configure an account on a server via XML-RPC, see
	/// https://wiki.linphone.org/xwiki/wiki/public/view/Lib/Features/Override%20account%20creator%20request/.
	/// <para>
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class AccountCreator : LinphoneObject
	{

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_factory_get();

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_factory_create_account_creator_cbs(IntPtr factory);

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_creator_add_callbacks(IntPtr thiz, IntPtr cbs);

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_creator_remove_callbacks(IntPtr thiz, IntPtr cbs);

		private AccountCreatorListener listener;

		public AccountCreatorListener Listener
		{
			get {
				if (listener == null)
				{
					IntPtr nativeListener = linphone_factory_create_account_creator_cbs(linphone_factory_get());
					listener = fromNativePtr<AccountCreatorListener>(nativeListener, false, true);
					linphone_account_creator_add_callbacks(nativePtr, nativeListener);
					belle_sip_object_unref(nativeListener);
				}
				return listener;
			}
			set {
				if (listener != null && value == null) {
					linphone_account_creator_remove_callbacks(nativePtr, listener.nativePtr);
				}
				listener = value;
			}
		}

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_creator_create(IntPtr core);

		/// <summary>
		/// Create a <see cref="Linphone.AccountCreator">Linphone.AccountCreator</see> and
		/// set Linphone Request callbacks. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="core">The <see cref="Linphone.Core">Linphone.Core</see> used for
		/// the XML-RPC communication    </param>
		/// <returns>The new <see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see> object.    </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public static Linphone.AccountCreator Create(Linphone.Core core)
		{
			
			IntPtr ptr = linphone_account_creator_create(core != null ? core.nativePtr : IntPtr.Zero);
			Linphone.AccountCreator returnVal = fromNativePtr<Linphone.AccountCreator>(ptr, false);
			
			GC.KeepAlive(core);
			return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_creator_set_account(IntPtr thiz, IntPtr account);

		/// <summary>
		/// Assign a proxy config pointer to the LinphoneAccountCreator. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="account">The LinphoneAccount to associate with the
		/// LinphoneAccountCreator.    </param>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public Linphone.Account Account
		{
			set
			{
				linphone_account_creator_set_account(nativePtr, value.nativePtr);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_creator_get_account_creation_request_token(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_creator_set_account_creation_request_token(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string token);

		/// <summary>
		/// Get the account creation request token received to be used to check user
		/// validation. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The token set, if any    </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public string AccountCreationRequestToken
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_creator_get_account_creation_request_token(nativePtr) );
			}
			set
			{
				linphone_account_creator_set_account_creation_request_token(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_creator_get_activation_code(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AccountCreatorActivationCodeStatus linphone_account_creator_set_activation_code(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string activationCode);

		/// <summary>
		/// Get the activation code. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The activation code of the <see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see>.    </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public string ActivationCode
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_creator_get_activation_code(nativePtr) );
			}
			set
			{
				linphone_account_creator_set_activation_code(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_creator_get_algorithm(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AccountCreatorAlgoStatus linphone_account_creator_set_algorithm(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string algorithm);

		/// <summary>
		/// Get the algorithm configured in the account creator. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The algorithm of the <see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see>.    </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public string Algorithm
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_creator_get_algorithm(nativePtr) );
			}
			set
			{
				linphone_account_creator_set_algorithm(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AccountCreatorStatus linphone_account_creator_set_as_default(IntPtr thiz, char setAsDefault);

		/// <summary>
		/// Set the set_as_default property. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="setAsDefault">true for the created proxy config to be set as
		/// default in <see cref="Linphone.Core">Linphone.Core</see>, false otherwise
		/// </param>
		/// <returns><see cref="Linphone.AccountCreatorStatus.RequestOk">
		/// Linphone.AccountCreatorStatus.RequestOk</see> if everything is OK, or a
		/// specific error otherwise. </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public bool AsDefault
		{
			set
			{
				linphone_account_creator_set_as_default(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_creator_get_display_name(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AccountCreatorUsernameStatus linphone_account_creator_set_display_name(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string displayName);

		/// <summary>
		/// Get the display name. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The display name of the <see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see>.    </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public string DisplayName
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_creator_get_display_name(nativePtr) );
			}
			set
			{
				linphone_account_creator_set_display_name(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_creator_get_domain(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AccountCreatorDomainStatus linphone_account_creator_set_domain(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string domain);

		/// <summary>
		/// Get the domain. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The domain of the <see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see>.    </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public string Domain
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_creator_get_domain(nativePtr) );
			}
			set
			{
				linphone_account_creator_set_domain(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_creator_get_email(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AccountCreatorEmailStatus linphone_account_creator_set_email(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string email);

		/// <summary>
		/// Get the email. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The email of the <see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see>.    </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public string Email
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_creator_get_email(nativePtr) );
			}
			set
			{
				linphone_account_creator_set_email(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_creator_get_ha1(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AccountCreatorPasswordStatus linphone_account_creator_set_ha1(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string ha1);

		/// <summary>
		/// Get the ha1. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The ha1 of the <see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see>.    </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public string Ha1
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_creator_get_ha1(nativePtr) );
			}
			set
			{
				linphone_account_creator_set_ha1(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_creator_get_language(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AccountCreatorLanguageStatus linphone_account_creator_set_language(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string lang);

		/// <summary>
		/// Get the language use in email of SMS. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The language of the <see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see>.    </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public string Language
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_creator_get_language(nativePtr) );
			}
			set
			{
				linphone_account_creator_set_language(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_creator_get_password(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AccountCreatorPasswordStatus linphone_account_creator_set_password(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string password);

		/// <summary>
		/// Get the password. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The password of the <see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see>.    </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public string Password
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_creator_get_password(nativePtr) );
			}
			set
			{
				linphone_account_creator_set_password(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_creator_get_phone_country_code(IntPtr thiz);

		/// <summary>
		/// Get the international prefix. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The international prefix (or phone country code) of the <see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see>.    </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public string PhoneCountryCode
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_creator_get_phone_country_code(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_creator_get_phone_number(IntPtr thiz);

		/// <summary>
		/// Get the RFC 3966 normalized phone number. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The phone number of the <see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see>.    </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public string PhoneNumber
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_creator_get_phone_number(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_creator_get_pn_param(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_creator_set_pn_param(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string pnParam);

		/// <summary>
		/// Get the param to be used by the backend to send the push notification to the
		/// device asking for an auth token. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The pn_param set, if any    </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public string PnParam
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_creator_get_pn_param(nativePtr) );
			}
			set
			{
				linphone_account_creator_set_pn_param(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_creator_get_pn_prid(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_creator_set_pn_prid(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string pnPrid);

		/// <summary>
		/// Get the prid to be used by the backend to send the push notification to the
		/// device asking for an auth token. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The pn_prid set, if any    </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public string PnPrid
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_creator_get_pn_prid(nativePtr) );
			}
			set
			{
				linphone_account_creator_set_pn_prid(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_creator_get_pn_provider(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_creator_set_pn_provider(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string pnProvider);

		/// <summary>
		/// Get the provider to be used by the backend to send the push notification to the
		/// device asking for an auth token. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The pn_provider set, if any    </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public string PnProvider
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_creator_get_pn_provider(nativePtr) );
			}
			set
			{
				linphone_account_creator_set_pn_provider(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_creator_set_proxy_config(IntPtr thiz, IntPtr cfg);

		/// <summary>
		/// Assign a proxy config pointer to the LinphoneAccountCreator. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="cfg">The LinphoneProxyConfig to associate with the
		/// LinphoneAccountCreator.    </param>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public Linphone.ProxyConfig ProxyConfig
		{
			set
			{
				linphone_account_creator_set_proxy_config(nativePtr, value.nativePtr);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_creator_get_route(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AccountCreatorStatus linphone_account_creator_set_route(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string route);

		/// <summary>
		/// Get the route. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The route of the <see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see>.    </returns>
		public string Route
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_creator_get_route(nativePtr) );
			}
			set
			{
				linphone_account_creator_set_route(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_account_creator_get_set_as_default(IntPtr thiz);

		/// <summary>
		/// Get the set_as_default property. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if account will be set as default, false otherwise. </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public bool SetAsDefault
		{
			get
			{
				return linphone_account_creator_get_set_as_default(nativePtr) != 0;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_creator_get_token(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_creator_set_token(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string token);

		/// <summary>
		/// Get the authentication token set (if any) to be used to authenticate next
		/// queries, if required. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The token set, if any    </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public string Token
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_creator_get_token(nativePtr) );
			}
			set
			{
				linphone_account_creator_set_token(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.TransportType linphone_account_creator_get_transport(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AccountCreatorTransportStatus linphone_account_creator_set_transport(IntPtr thiz, int transport);

		/// <summary>
		/// Get Transport. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The <see cref="Linphone.TransportType">Linphone.TransportType</see> of
		/// the creator. </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public Linphone.TransportType Transport
		{
			get
			{
				return linphone_account_creator_get_transport(nativePtr);
			}
			set
			{
				linphone_account_creator_set_transport(nativePtr, (int)value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_creator_get_username(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AccountCreatorUsernameStatus linphone_account_creator_set_username(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string username);

		/// <summary>
		/// Get the username. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The username of the <see cref="Linphone.AccountCreator">
		/// Linphone.AccountCreator</see>.    </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public string Username
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_creator_get_username(nativePtr) );
			}
			set
			{
				linphone_account_creator_set_username(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AccountCreatorStatus linphone_account_creator_activate_account(IntPtr thiz);

		/// <summary>
		/// Send a request to activate an account on server. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns><see cref="Linphone.AccountCreatorStatus.RequestOk">
		/// Linphone.AccountCreatorStatus.RequestOk</see> if the request has been sent,
		/// <see cref="Linphone.AccountCreatorStatus.RequestFailed">
		/// Linphone.AccountCreatorStatus.RequestFailed</see> otherwise </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public Linphone.AccountCreatorStatus ActivateAccount()
		{
			
			Linphone.AccountCreatorStatus returnVal = linphone_account_creator_activate_account(nativePtr);
			
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AccountCreatorStatus linphone_account_creator_activate_alias(IntPtr thiz);

		/// <summary>
		/// Send a request to activate an alias. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns><see cref="Linphone.AccountCreatorStatus.RequestOk">
		/// Linphone.AccountCreatorStatus.RequestOk</see> if the request has been sent,
		/// <see cref="Linphone.AccountCreatorStatus.RequestFailed">
		/// Linphone.AccountCreatorStatus.RequestFailed</see> otherwise </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public Linphone.AccountCreatorStatus ActivateAlias()
		{
			
			Linphone.AccountCreatorStatus returnVal = linphone_account_creator_activate_alias(nativePtr);
			
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AccountCreatorStatus linphone_account_creator_create_account(IntPtr thiz);

		/// <summary>
		/// Send a request to create an account on server. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns><see cref="Linphone.AccountCreatorStatus.RequestOk">
		/// Linphone.AccountCreatorStatus.RequestOk</see> if the request has been sent,
		/// <see cref="Linphone.AccountCreatorStatus.RequestFailed">
		/// Linphone.AccountCreatorStatus.RequestFailed</see> otherwise </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public Linphone.AccountCreatorStatus CreateAccount()
		{
			
			Linphone.AccountCreatorStatus returnVal = linphone_account_creator_create_account(nativePtr);
			
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_creator_create_account_in_core(IntPtr thiz);

		/// <summary>
		/// Create and configure a <see cref="Linphone.Account">Linphone.Account</see> and
		/// a <see cref="Linphone.AuthInfo">Linphone.AuthInfo</see> from informations set
		/// in the <see cref="Linphone.AccountCreator">Linphone.AccountCreator</see>. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>A <see cref="Linphone.Account">Linphone.Account</see> object if
		/// successful, null otherwise.       </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public Linphone.Account CreateAccountInCore()
		{
			
			IntPtr ptr = linphone_account_creator_create_account_in_core(nativePtr);
			Linphone.Account returnVal = fromNativePtr<Linphone.Account>(ptr, false);
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_creator_create_proxy_config(IntPtr thiz);

		/// <summary>
		/// Create and configure a proxy config and a authentication info for an account
		/// creator. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>A <see cref="Linphone.ProxyConfig">Linphone.ProxyConfig</see> object
		/// if successful, null otherwise.    </returns>
		/// deprecated : 05/05/2023 Use <see cref="Linphone.AccountCreator.CreateAccountInCore()">
		/// Linphone.AccountCreator.CreateAccountInCore()</see> instead. 
		public Linphone.ProxyConfig CreateProxyConfig()
		{
			
			IntPtr ptr = linphone_account_creator_create_proxy_config(nativePtr);
			Linphone.ProxyConfig returnVal = fromNativePtr<Linphone.ProxyConfig>(ptr, false);
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AccountCreatorStatus linphone_account_creator_create_push_account(IntPtr thiz);

		/// <summary>
		/// Send a request to create a push account on server. 
		/// <para>
		/// Push accounts are used in account dependent situation when account cannot send
		/// push notifications. A username and password are automatically generated, an
		/// account is automatically activated. 
		/// </para>
		/// </summary>
		/// <returns><see cref="Linphone.AccountCreatorStatus.RequestOk">
		/// Linphone.AccountCreatorStatus.RequestOk</see> if the request has been sent,
		/// <see cref="Linphone.AccountCreatorStatus.RequestFailed">
		/// Linphone.AccountCreatorStatus.RequestFailed</see> otherwise </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public Linphone.AccountCreatorStatus CreatePushAccount()
		{
			
			Linphone.AccountCreatorStatus returnVal = linphone_account_creator_create_push_account(nativePtr);
			
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AccountCreatorStatus linphone_account_creator_is_account_activated(IntPtr thiz);

		/// <summary>
		/// Send a request to know if an account is activated on server. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns><see cref="Linphone.AccountCreatorStatus.RequestOk">
		/// Linphone.AccountCreatorStatus.RequestOk</see> if the request has been sent,
		/// <see cref="Linphone.AccountCreatorStatus.RequestFailed">
		/// Linphone.AccountCreatorStatus.RequestFailed</see> otherwise </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public Linphone.AccountCreatorStatus IsAccountActivated()
		{
			
			Linphone.AccountCreatorStatus returnVal = linphone_account_creator_is_account_activated(nativePtr);
			
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AccountCreatorStatus linphone_account_creator_is_account_exist(IntPtr thiz);

		/// <summary>
		/// Send a request to know the existence of account on server. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns><see cref="Linphone.AccountCreatorStatus.RequestOk">
		/// Linphone.AccountCreatorStatus.RequestOk</see> if the request has been sent,
		/// <see cref="Linphone.AccountCreatorStatus.RequestFailed">
		/// Linphone.AccountCreatorStatus.RequestFailed</see> otherwise </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public Linphone.AccountCreatorStatus IsAccountExist()
		{
			
			Linphone.AccountCreatorStatus returnVal = linphone_account_creator_is_account_exist(nativePtr);
			
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AccountCreatorStatus linphone_account_creator_is_account_linked(IntPtr thiz);

		/// <summary>
		/// Send a request to know if an account is linked. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns><see cref="Linphone.AccountCreatorStatus.RequestOk">
		/// Linphone.AccountCreatorStatus.RequestOk</see> if the request has been sent,
		/// <see cref="Linphone.AccountCreatorStatus.RequestFailed">
		/// Linphone.AccountCreatorStatus.RequestFailed</see> otherwise </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public Linphone.AccountCreatorStatus IsAccountLinked()
		{
			
			Linphone.AccountCreatorStatus returnVal = linphone_account_creator_is_account_linked(nativePtr);
			
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AccountCreatorStatus linphone_account_creator_is_alias_used(IntPtr thiz);

		/// <summary>
		/// Send a request to know if an alias is used. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns><see cref="Linphone.AccountCreatorStatus.RequestOk">
		/// Linphone.AccountCreatorStatus.RequestOk</see> if the request has been sent,
		/// <see cref="Linphone.AccountCreatorStatus.RequestFailed">
		/// Linphone.AccountCreatorStatus.RequestFailed</see> otherwise </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public Linphone.AccountCreatorStatus IsAliasUsed()
		{
			
			Linphone.AccountCreatorStatus returnVal = linphone_account_creator_is_alias_used(nativePtr);
			
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AccountCreatorStatus linphone_account_creator_link_account(IntPtr thiz);

		/// <summary>
		/// Send a request to link an account to an alias. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns><see cref="Linphone.AccountCreatorStatus.RequestOk">
		/// Linphone.AccountCreatorStatus.RequestOk</see> if the request has been sent,
		/// <see cref="Linphone.AccountCreatorStatus.RequestFailed">
		/// Linphone.AccountCreatorStatus.RequestFailed</see> otherwise </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public Linphone.AccountCreatorStatus LinkAccount()
		{
			
			Linphone.AccountCreatorStatus returnVal = linphone_account_creator_link_account(nativePtr);
			
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AccountCreatorStatus linphone_account_creator_login_linphone_account(IntPtr thiz);

		/// <summary>
		/// Send a request to get the password &amp; algorithm of an account using the
		/// confirmation key. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns><see cref="Linphone.AccountCreatorStatus.RequestOk">
		/// Linphone.AccountCreatorStatus.RequestOk</see> if everything is OK, or a
		/// specific error otherwise. </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public Linphone.AccountCreatorStatus LoginLinphoneAccount()
		{
			
			Linphone.AccountCreatorStatus returnVal = linphone_account_creator_login_linphone_account(nativePtr);
			
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AccountCreatorStatus linphone_account_creator_recover_account(IntPtr thiz);

		/// <summary>
		/// Send a request to recover an account. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns><see cref="Linphone.AccountCreatorStatus.RequestOk">
		/// Linphone.AccountCreatorStatus.RequestOk</see> if the request has been sent,
		/// <see cref="Linphone.AccountCreatorStatus.RequestFailed">
		/// Linphone.AccountCreatorStatus.RequestFailed</see> otherwise </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public Linphone.AccountCreatorStatus RecoverAccount()
		{
			
			Linphone.AccountCreatorStatus returnVal = linphone_account_creator_recover_account(nativePtr);
			
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AccountCreatorStatus linphone_account_creator_request_account_creation_request_token(IntPtr thiz);

		/// <summary>
		/// Request an account creation &quot;request_token&quot; to be used on account
		/// creations. 
		/// <para>
		/// The request_token is retrieved from the callback
		/// linphone_account_creator_cbs_get_account_creation_request_token 
		/// </para>
		/// </summary>
		/// <returns><see cref="Linphone.AccountCreatorStatus.RequestOk">
		/// Linphone.AccountCreatorStatus.RequestOk</see> if everything is OK, or a
		/// specific error otherwise. </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public Linphone.AccountCreatorStatus RequestAccountCreationRequestToken()
		{
			
			Linphone.AccountCreatorStatus returnVal = linphone_account_creator_request_account_creation_request_token(nativePtr);
			
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AccountCreatorStatus linphone_account_creator_request_account_creation_token_using_request_token(IntPtr thiz);

		/// <summary>
		/// Send a request to get a token to be used for account creation from a
		/// request_token. 
		/// <para>
		/// The token is retrieved from the callback
		/// linphone_account_creator_cbs_get_account_creation_token_using_request_token 
		/// </para>
		/// </summary>
		/// <returns><see cref="Linphone.AccountCreatorStatus.RequestOk">
		/// Linphone.AccountCreatorStatus.RequestOk</see> if the request has been sent,
		/// <see cref="Linphone.AccountCreatorStatus.RequestFailed">
		/// Linphone.AccountCreatorStatus.RequestFailed</see> otherwise </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public Linphone.AccountCreatorStatus RequestAccountCreationTokenUsingRequestToken()
		{
			
			Linphone.AccountCreatorStatus returnVal = linphone_account_creator_request_account_creation_token_using_request_token(nativePtr);
			
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AccountCreatorStatus linphone_account_creator_request_auth_token(IntPtr thiz);

		/// <summary>
		/// Request an auth token to be send by the backend by push notification. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns><see cref="Linphone.AccountCreatorStatus.RequestOk">
		/// Linphone.AccountCreatorStatus.RequestOk</see> if everything is OK, or a
		/// specific error otherwise. </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public Linphone.AccountCreatorStatus RequestAuthToken()
		{
			
			Linphone.AccountCreatorStatus returnVal = linphone_account_creator_request_auth_token(nativePtr);
			
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_creator_reset(IntPtr thiz);

		/// <summary>
		/// Reset the account creator entries like username, password, phone number... 
		/// <para>
		/// </para>
		/// </summary>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public void Reset()
		{
			
			linphone_account_creator_reset(nativePtr);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern uint linphone_account_creator_set_phone_number(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string phoneNumber, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string countryCode);

		/// <summary>
		/// Set the phone number normalized. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="phoneNumber">The phone number to set    </param>
		/// <param name="countryCode">Country code to associate phone number with   
		/// </param>
		/// <returns><see cref="Linphone.AccountCreatorPhoneNumberStatus.Ok">
		/// Linphone.AccountCreatorPhoneNumberStatus.Ok</see> if everything is OK, or
		/// specific(s) error(s) otherwise. </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public uint SetPhoneNumber(string phoneNumber, string countryCode)
		{
			
			uint returnVal = linphone_account_creator_set_phone_number(nativePtr, phoneNumber, countryCode);
			
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AccountCreatorStatus linphone_account_creator_update_account(IntPtr thiz);

		/// <summary>
		/// Send a request to update an account. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns><see cref="Linphone.AccountCreatorStatus.RequestOk">
		/// Linphone.AccountCreatorStatus.RequestOk</see> if the request has been sent,
		/// <see cref="Linphone.AccountCreatorStatus.RequestFailed">
		/// Linphone.AccountCreatorStatus.RequestFailed</see> otherwise </returns>
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public Linphone.AccountCreatorStatus UpdateAccount()
		{
			
			Linphone.AccountCreatorStatus returnVal = linphone_account_creator_update_account(nativePtr);
			
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_creator_use_test_admin_account(IntPtr thiz);

		/// <summary>
		/// Require the account creator to use special &quot;test admin account&quot;. 
		/// <para>
		/// </para>
		/// </summary>
		/// <remarks>Warning : The &quot;test admin account&quot; is a special feature
		/// required for automated test, and requires the APP_EVERYONE_IS_ADMIN property to
		/// be enabled on the remote Flexisip Account Manager (FlexiAPI). This feature must
		/// never be turned on for a production-stage app. </remarks> 
		/// deprecated : 11/06/2024 use <see cref="Linphone.AccountManagerServices">
		/// Linphone.AccountManagerServices</see> instead 
		public void UseTestAdminAccount()
		{
			
			linphone_account_creator_use_test_admin_account(nativePtr);
			
			
						
		}
	}
	/// <summary>
	/// Object that represents a device that at least once was connected to a given
	/// account. 
	/// <para>
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class AccountDevice : LinphoneObject
	{


		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_device_get_last_update_time(IntPtr thiz);

		/// <summary>
		/// Gets the timestamp at which this devices was updated for the last time. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the time under ISO 8601 format at which the device was updated.   
		/// </returns>
		public string LastUpdateTime
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_device_get_last_update_time(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern long linphone_account_device_get_last_update_timestamp(IntPtr thiz);

		/// <summary>
		/// Gets the timestamp at which this devices was updated for the last time. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the timestamp (time_t) at which the device was updated.    </returns>
		public long LastUpdateTimestamp
		{
			get
			{
				return linphone_account_device_get_last_update_timestamp(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_device_get_name(IntPtr thiz);

		/// <summary>
		/// Gets the name of the device. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the name of the device.    </returns>
		public string Name
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_device_get_name(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_device_get_user_agent(IntPtr thiz);

		/// <summary>
		/// Gets the user-agent of the device. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the user-agent of the device.    </returns>
		public string UserAgent
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_device_get_user_agent(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_device_get_uuid(IntPtr thiz);

		/// <summary>
		/// Gets the UUID of the device. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the UUID of the device.    </returns>
		public string Uuid
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_device_get_uuid(nativePtr) );
			}
		}
	}
	/// <summary>
	/// Creates and manage SIP user accounts remotely, using the REST API of the
	/// Flexisip Account Manager. 
	/// <para>
	/// </para>
	/// </summary>
	/// <remarks>See : https://gitlab.linphone.org/BC/public/flexisip-account-manager
	/// </remarks>
	[StructLayout(LayoutKind.Sequential)]
	public class AccountManagerServices : LinphoneObject
	{


		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_manager_services_get_language(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_manager_services_set_language(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string language);

		/// <summary>
		/// Gets the configurer preferred language, if set. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the language previously set, if any (otherwise null).    </returns>
		public string Language
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_manager_services_get_language(nativePtr) );
			}
			set
			{
				linphone_account_manager_services_set_language(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_manager_services_create_delete_device_request(IntPtr thiz, IntPtr sipIdentity, IntPtr device);

		/// <summary>
		/// Requests to delete a device from the list of currently known devices. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="sipIdentity">the SIP identity URI that identifies your account for
		/// which you want the devices list.    </param>
		/// <param name="device">the <see cref="Linphone.AccountDevice">
		/// Linphone.AccountDevice</see> to delete.    </param>
		/// <returns>the <see cref="Linphone.AccountManagerServicesRequest">
		/// Linphone.AccountManagerServicesRequest</see> request object.    </returns>
		public Linphone.AccountManagerServicesRequest CreateDeleteDeviceRequest(Linphone.Address sipIdentity, Linphone.AccountDevice device)
		{
			
			IntPtr ptr = linphone_account_manager_services_create_delete_device_request(nativePtr, sipIdentity != null ? sipIdentity.nativePtr : IntPtr.Zero, device != null ? device.nativePtr : IntPtr.Zero);
			Linphone.AccountManagerServicesRequest returnVal = fromNativePtr<Linphone.AccountManagerServicesRequest>(ptr, false);
			
			GC.KeepAlive(sipIdentity);
GC.KeepAlive(device);
			return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_manager_services_create_get_account_creation_request_token_request(IntPtr thiz);

		/// <summary>
		/// Requests a an account creation request token that once validated using the URL
		/// returned by this method upon success, will allow you to call <see cref="Linphone.AccountManagerServices.CreateGetAccountCreationTokenFromRequestTokenRequest()">
		/// Linphone.AccountManagerServices.CreateGetAccountCreationTokenFromRequestTokenRequest()</see> to obtain a valid account creation token. 
		/// <para>
		/// Once the account creation token is obtained, you can call <see cref="Linphone.AccountManagerServices.CreateNewAccountUsingTokenRequest()">
		/// Linphone.AccountManagerServices.CreateNewAccountUsingTokenRequest()</see>. 
		/// </para>
		/// </summary>
		/// <returns>the <see cref="Linphone.AccountManagerServicesRequest">
		/// Linphone.AccountManagerServicesRequest</see> request object.    </returns>
		public Linphone.AccountManagerServicesRequest CreateGetAccountCreationRequestTokenRequest()
		{
			
			IntPtr ptr = linphone_account_manager_services_create_get_account_creation_request_token_request(nativePtr);
			Linphone.AccountManagerServicesRequest returnVal = fromNativePtr<Linphone.AccountManagerServicesRequest>(ptr, false);
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_manager_services_create_get_account_creation_token_from_request_token_request(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string requestToken);

		/// <summary>
		/// Converts an account creation request token obtained by
		/// linphone_account_manager_services_request_account_creation_request_token to an
		/// account creation token. 
		/// <para>
		/// The obtained token can be used to call <see cref="Linphone.AccountManagerServices.CreateNewAccountUsingTokenRequest()">
		/// Linphone.AccountManagerServices.CreateNewAccountUsingTokenRequest()</see>. 
		/// </para>
		/// </summary>
		/// <param name="requestToken">the token obtained previously and validated using
		/// your web browser.    </param>
		/// <returns>the <see cref="Linphone.AccountManagerServicesRequest">
		/// Linphone.AccountManagerServicesRequest</see> request object.    </returns>
		public Linphone.AccountManagerServicesRequest CreateGetAccountCreationTokenFromRequestTokenRequest(string requestToken)
		{
			
			IntPtr ptr = linphone_account_manager_services_create_get_account_creation_token_from_request_token_request(nativePtr, requestToken);
			Linphone.AccountManagerServicesRequest returnVal = fromNativePtr<Linphone.AccountManagerServicesRequest>(ptr, false);
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_manager_services_create_get_devices_list_request(IntPtr thiz, IntPtr sipIdentity);

		/// <summary>
		/// Requests the list of devices currently known. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="sipIdentity">the SIP identity URI that identifies your account for
		/// which you want the devices list.    </param>
		/// <returns>the <see cref="Linphone.AccountManagerServicesRequest">
		/// Linphone.AccountManagerServicesRequest</see> request object.    </returns>
		public Linphone.AccountManagerServicesRequest CreateGetDevicesListRequest(Linphone.Address sipIdentity)
		{
			
			IntPtr ptr = linphone_account_manager_services_create_get_devices_list_request(nativePtr, sipIdentity != null ? sipIdentity.nativePtr : IntPtr.Zero);
			Linphone.AccountManagerServicesRequest returnVal = fromNativePtr<Linphone.AccountManagerServicesRequest>(ptr, false);
			
			GC.KeepAlive(sipIdentity);
			return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_manager_services_create_link_email_to_account_using_code_request(IntPtr thiz, IntPtr sipIdentity, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string code);

		/// <summary>
		/// Validates the link between an email address and an account using a code
		/// received by email after calling <see cref="Linphone.AccountManagerServices.CreateSendEmailLinkingCodeByEmailRequest()">
		/// Linphone.AccountManagerServices.CreateSendEmailLinkingCodeByEmailRequest()</see> 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="sipIdentity">the SIP identity URI that identifies the account to
		/// which you want to link the email address to.    </param>
		/// <param name="code">the code received by email.    </param>
		/// <returns>the <see cref="Linphone.AccountManagerServicesRequest">
		/// Linphone.AccountManagerServicesRequest</see> request object.    </returns>
		public Linphone.AccountManagerServicesRequest CreateLinkEmailToAccountUsingCodeRequest(Linphone.Address sipIdentity, string code)
		{
			
			IntPtr ptr = linphone_account_manager_services_create_link_email_to_account_using_code_request(nativePtr, sipIdentity != null ? sipIdentity.nativePtr : IntPtr.Zero, code);
			Linphone.AccountManagerServicesRequest returnVal = fromNativePtr<Linphone.AccountManagerServicesRequest>(ptr, false);
			
			GC.KeepAlive(sipIdentity);
			return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_manager_services_create_link_phone_number_to_account_using_code_request(IntPtr thiz, IntPtr sipIdentity, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string code);

		/// <summary>
		/// Validates the link between a phone number and an account using a code received
		/// by SMS after calling <see cref="Linphone.AccountManagerServices.CreateSendPhoneNumberLinkingCodeBySmsRequest()">
		/// Linphone.AccountManagerServices.CreateSendPhoneNumberLinkingCodeBySmsRequest()</see> 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="sipIdentity">the SIP identity URI that identifies the account to
		/// which you want to link the phone number to.    </param>
		/// <param name="code">the code received by SMS.    </param>
		/// <returns>the <see cref="Linphone.AccountManagerServicesRequest">
		/// Linphone.AccountManagerServicesRequest</see> request object.    </returns>
		public Linphone.AccountManagerServicesRequest CreateLinkPhoneNumberToAccountUsingCodeRequest(Linphone.Address sipIdentity, string code)
		{
			
			IntPtr ptr = linphone_account_manager_services_create_link_phone_number_to_account_using_code_request(nativePtr, sipIdentity != null ? sipIdentity.nativePtr : IntPtr.Zero, code);
			Linphone.AccountManagerServicesRequest returnVal = fromNativePtr<Linphone.AccountManagerServicesRequest>(ptr, false);
			
			GC.KeepAlive(sipIdentity);
			return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_manager_services_create_new_account_using_token_request(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string username, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string password, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string algorithm, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string token);

		/// <summary>
		/// Creates an account using an account creation token, for example obtained using
		/// <see cref="Linphone.AccountManagerServices.CreateSendAccountCreationTokenByPushRequest()">
		/// Linphone.AccountManagerServices.CreateSendAccountCreationTokenByPushRequest()</see>. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="username">the username of the newly created account.    </param>
		/// <param name="password">the password to use for the newly created account.   
		/// </param>
		/// <param name="algorithm">the algorithm to use to hash the password (must be
		/// either MD5 or SHA-256).    </param>
		/// <param name="token">the account creation token obtained previously.    </param>
		/// <returns>the <see cref="Linphone.AccountManagerServicesRequest">
		/// Linphone.AccountManagerServicesRequest</see> request object.    </returns>
		public Linphone.AccountManagerServicesRequest CreateNewAccountUsingTokenRequest(string username, string password, string algorithm, string token)
		{
			
			IntPtr ptr = linphone_account_manager_services_create_new_account_using_token_request(nativePtr, username, password, algorithm, token);
			Linphone.AccountManagerServicesRequest returnVal = fromNativePtr<Linphone.AccountManagerServicesRequest>(ptr, false);
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_manager_services_create_send_account_creation_token_by_push_request(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string pnProvider, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string pnParam, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string pnPrid);

		/// <summary>
		/// Requests a push notification to be sent to device, containing a valid account
		/// creation token. 
		/// <para>
		/// Provider, param &amp; prid can be obtained from <see cref="Linphone.Core.PushNotificationConfig">
		/// Linphone.Core.PushNotificationConfig</see>, but on iOS may need some
		/// modifications (depending on debug mode for example). Once the token is
		/// obtained, you can call <see cref="Linphone.AccountManagerServices.CreateNewAccountUsingTokenRequest()">
		/// Linphone.AccountManagerServices.CreateNewAccountUsingTokenRequest()</see>. 
		/// </para>
		/// </summary>
		/// <param name="pnProvider">The provider, for example &quot;apns.dev&quot;.   
		/// </param>
		/// <param name="pnParam">The parameters, for example
		/// &quot;ABCD1234.org.linphone.phone.remote&quot;.    </param>
		/// <param name="pnPrid">The prid, also known as push token.    </param>
		/// <returns>the <see cref="Linphone.AccountManagerServicesRequest">
		/// Linphone.AccountManagerServicesRequest</see> request object.    </returns>
		public Linphone.AccountManagerServicesRequest CreateSendAccountCreationTokenByPushRequest(string pnProvider, string pnParam, string pnPrid)
		{
			
			IntPtr ptr = linphone_account_manager_services_create_send_account_creation_token_by_push_request(nativePtr, pnProvider, pnParam, pnPrid);
			Linphone.AccountManagerServicesRequest returnVal = fromNativePtr<Linphone.AccountManagerServicesRequest>(ptr, false);
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_manager_services_create_send_email_linking_code_by_email_request(IntPtr thiz, IntPtr sipIdentity, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string emailAddress);

		/// <summary>
		/// Requests a code to be sent to a given email address, that can be used later to
		/// associate said email to an account using <see cref="Linphone.AccountManagerServices.CreateLinkEmailToAccountUsingCodeRequest()">
		/// Linphone.AccountManagerServices.CreateLinkEmailToAccountUsingCodeRequest()</see>. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="sipIdentity">the SIP identity URI that identifies the account to
		/// which you want to link the email address to.    </param>
		/// <param name="emailAddress">the email address to which to send the code to.   
		/// </param>
		/// <returns>the <see cref="Linphone.AccountManagerServicesRequest">
		/// Linphone.AccountManagerServicesRequest</see> request object.    </returns>
		public Linphone.AccountManagerServicesRequest CreateSendEmailLinkingCodeByEmailRequest(Linphone.Address sipIdentity, string emailAddress)
		{
			
			IntPtr ptr = linphone_account_manager_services_create_send_email_linking_code_by_email_request(nativePtr, sipIdentity != null ? sipIdentity.nativePtr : IntPtr.Zero, emailAddress);
			Linphone.AccountManagerServicesRequest returnVal = fromNativePtr<Linphone.AccountManagerServicesRequest>(ptr, false);
			
			GC.KeepAlive(sipIdentity);
			return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_manager_services_create_send_phone_number_linking_code_by_sms_request(IntPtr thiz, IntPtr sipIdentity, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string phoneNumber);

		/// <summary>
		/// Requests a code to be sent to a given phone number, that can be used later to
		/// associate said phone number to an account using <see cref="Linphone.AccountManagerServices.CreateLinkPhoneNumberToAccountUsingCodeRequest()">
		/// Linphone.AccountManagerServices.CreateLinkPhoneNumberToAccountUsingCodeRequest()</see>. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="sipIdentity">the SIP identity URI that identifies the account to
		/// which you want to link the phone number to.    </param>
		/// <param name="phoneNumber">the phone number to which to send the code by SMS.   
		/// </param>
		/// <returns>the <see cref="Linphone.AccountManagerServicesRequest">
		/// Linphone.AccountManagerServicesRequest</see> request object.    </returns>
		public Linphone.AccountManagerServicesRequest CreateSendPhoneNumberLinkingCodeBySmsRequest(Linphone.Address sipIdentity, string phoneNumber)
		{
			
			IntPtr ptr = linphone_account_manager_services_create_send_phone_number_linking_code_by_sms_request(nativePtr, sipIdentity != null ? sipIdentity.nativePtr : IntPtr.Zero, phoneNumber);
			Linphone.AccountManagerServicesRequest returnVal = fromNativePtr<Linphone.AccountManagerServicesRequest>(ptr, false);
			
			GC.KeepAlive(sipIdentity);
			return returnVal;
		}
	}
	/// <summary>
	/// Request object created by <see cref="Linphone.AccountManagerServices">
	/// Linphone.AccountManagerServices</see>. 
	/// <para>
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class AccountManagerServicesRequest : LinphoneObject
	{

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_factory_get();

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_factory_create_account_manager_services_request_cbs(IntPtr factory);

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_manager_services_request_add_callbacks(IntPtr thiz, IntPtr cbs);

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_manager_services_request_remove_callbacks(IntPtr thiz, IntPtr cbs);

		private AccountManagerServicesRequestListener listener;

		public AccountManagerServicesRequestListener Listener
		{
			get {
				if (listener == null)
				{
					IntPtr nativeListener = linphone_factory_create_account_manager_services_request_cbs(linphone_factory_get());
					listener = fromNativePtr<AccountManagerServicesRequestListener>(nativeListener, false, true);
					linphone_account_manager_services_request_add_callbacks(nativePtr, nativeListener);
					belle_sip_object_unref(nativeListener);
				}
				return listener;
			}
			set {
				if (listener != null && value == null) {
					linphone_account_manager_services_request_remove_callbacks(nativePtr, listener.nativePtr);
				}
				listener = value;
			}
		}

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AccountManagerServicesRequestType linphone_account_manager_services_request_get_type(IntPtr thiz);

		/// <summary>
		/// Gets the <see cref="Linphone.AccountManagerServicesRequestType">
		/// Linphone.AccountManagerServicesRequestType</see> representing this request. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the <see cref="Linphone.AccountManagerServicesRequestType">
		/// Linphone.AccountManagerServicesRequestType</see>. </returns>
		public Linphone.AccountManagerServicesRequestType Type
		{
			get
			{
				return linphone_account_manager_services_request_get_type(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_manager_services_request_submit(IntPtr thiz);

		/// <summary>
		/// Executes the request represented by <see cref="Linphone.AccountManagerServicesRequest">
		/// Linphone.AccountManagerServicesRequest</see>. 
		/// <para>
		/// </para>
		/// </summary>
		public void Submit()
		{
			
			linphone_account_manager_services_request_submit(nativePtr);
			
			
						
		}
	}
	/// <summary>
	/// Object that is used to set the different parameters of a <see cref="Linphone.Account">
	/// Linphone.Account</see>. 
	/// <para>
	/// Note that authenticated accounts should have a corresponding <see cref="Linphone.AuthInfo">
	/// Linphone.AuthInfo</see> added to the <see cref="Linphone.Core">
	/// Linphone.Core</see> to register properly. 
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class AccountParams : LinphoneObject
	{


		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_new_with_config(IntPtr lc, int index);

		/// <summary>
		/// Create a new <see cref="Linphone.AccountParams">Linphone.AccountParams</see>
		/// object from a configuration. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="lc">The <see cref="Linphone.Core">Linphone.Core</see> object.   
		/// </param>
		/// <param name="index">The index of the configuration. </param>
		/// <returns>The newly created <see cref="Linphone.AccountParams">
		/// Linphone.AccountParams</see> object.    </returns>
		public static Linphone.AccountParams NewWithConfig(Linphone.Core lc, int index)
		{
			
			IntPtr ptr = linphone_account_params_new_with_config(lc != null ? lc.nativePtr : IntPtr.Zero, index);
			Linphone.AccountParams returnVal = fromNativePtr<Linphone.AccountParams>(ptr, false);
			
			GC.KeepAlive(lc);
			return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_get_audio_video_conference_factory_address(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_audio_video_conference_factory_address(IntPtr thiz, IntPtr address);

		/// <summary>
		/// Get the audio video conference factory uri. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The <see cref="Linphone.Address">Linphone.Address</see> of the audio
		/// video conference factory.    </returns>
		public Linphone.Address AudioVideoConferenceFactoryAddress
		{
			get
			{
				IntPtr ptr = linphone_account_params_get_audio_video_conference_factory_address(nativePtr);
				Linphone.Address obj = fromNativePtr<Linphone.Address>(ptr, true);
				return obj;
			}
			set
			{
				linphone_account_params_set_audio_video_conference_factory_address(nativePtr, value.nativePtr);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AVPFMode linphone_account_params_get_avpf_mode(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_avpf_mode(IntPtr thiz, int mode);

		/// <summary>
		/// Get enablement status of RTCP feedback (also known as AVPF profile). 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the enablement mode, which can be <see cref="Linphone.AVPFMode.Default">
		/// Linphone.AVPFMode.Default</see> (use LinphoneCore&apos;s mode), <see cref="Linphone.AVPFMode.Enabled">
		/// Linphone.AVPFMode.Enabled</see> (avpf is enabled), or <see cref="Linphone.AVPFMode.Disabled">
		/// Linphone.AVPFMode.Disabled</see> (disabled). </returns>
		public Linphone.AVPFMode AvpfMode
		{
			get
			{
				return linphone_account_params_get_avpf_mode(nativePtr);
			}
			set
			{
				linphone_account_params_set_avpf_mode(nativePtr, (int)value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern uint linphone_account_params_get_avpf_rr_interval(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_avpf_rr_interval(IntPtr thiz, uint interval);

		/// <summary>
		/// Get the interval between regular RTCP reports when using AVPF/SAVPF. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The interval in seconds. </returns>
		public uint AvpfRrInterval
		{
			get
			{
				return linphone_account_params_get_avpf_rr_interval(nativePtr);
			}
			set
			{
				linphone_account_params_set_avpf_rr_interval(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_get_ccmp_server_url(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_ccmp_server_url(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string url);

		/// <summary>
		/// Get the CCMP server address. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The URL of the CCMP server.    </returns>
		public string CcmpServerUrl
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_params_get_ccmp_server_url(nativePtr) );
			}
			set
			{
				linphone_account_params_set_ccmp_server_url(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_get_ccmp_user_id(IntPtr thiz);

		/// <summary>
		/// Get the CCMP user ID. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The ID of the CCMP user.    </returns>
		public string CcmpUserId
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_params_get_ccmp_user_id(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_get_conference_factory_address(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_conference_factory_address(IntPtr thiz, IntPtr address);

		/// <summary>
		/// Get the conference factory uri. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The <see cref="Linphone.Address">Linphone.Address</see> of the
		/// conference factory.    </returns>
		public Linphone.Address ConferenceFactoryAddress
		{
			get
			{
				IntPtr ptr = linphone_account_params_get_conference_factory_address(nativePtr);
				Linphone.Address obj = fromNativePtr<Linphone.Address>(ptr, true);
				return obj;
			}
			set
			{
				linphone_account_params_set_conference_factory_address(nativePtr, value.nativePtr);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_get_conference_factory_uri(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_conference_factory_uri(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string uri);

		/// <summary>
		/// Get the conference factory uri. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The uri of the conference factory.    </returns>
		public string ConferenceFactoryUri
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_params_get_conference_factory_uri(nativePtr) );
			}
			set
			{
				linphone_account_params_set_conference_factory_uri(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_get_contact_parameters(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_contact_parameters(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string contactParams);

		/// <summary>
		/// Returns the contact parameters. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The previously set contact parameters.    </returns>
		public string ContactParameters
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_params_get_contact_parameters(nativePtr) );
			}
			set
			{
				linphone_account_params_set_contact_parameters(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_get_contact_uri_parameters(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_contact_uri_parameters(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string contactUriParams);

		/// <summary>
		/// Return the contact URI parameters. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The previously set contact URI parameters.    </returns>
		public string ContactUriParameters
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_params_get_contact_uri_parameters(nativePtr) );
			}
			set
			{
				linphone_account_params_set_contact_uri_parameters(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_account_params_cpim_in_basic_chat_room_enabled(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_enable_cpim_in_basic_chat_room(IntPtr thiz, char enable);

		/// <summary>
		/// Indicates whether chat messages sent by this account in a <see cref="Linphone.ChatRoomBackend.Basic">
		/// Linphone.ChatRoomBackend.Basic</see> chat room will be using CPIM format or
		/// not. 
		/// <para>
		/// By default SIP SIMPLE format is used for &quot;basic&quot; chat rooms, CPIM is
		/// only used for <see cref="Linphone.ChatRoomBackend.FlexisipChat">
		/// Linphone.ChatRoomBackend.FlexisipChat</see> chat rooms. </para>
		/// </summary>
		/// <remarks>See :
		/// https://wiki.linphone.org/xwiki/wiki/public/view/Lib/Features/Instant%20Messaging/Reply%20to%20a%20specific%20message/ </remarks>
		/// <returns>true if chat messages will be sent out in CPIM format, false if chat
		/// messages will be sent out as SIP SIMPLE. </returns>
		public bool CpimInBasicChatRoomEnabled
		{
			get
			{
				return linphone_account_params_cpim_in_basic_chat_room_enabled(nativePtr) != 0;
			}
			set
			{
				linphone_account_params_enable_cpim_in_basic_chat_room(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_get_custom_contact(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_custom_contact(IntPtr thiz, IntPtr contact);

		/// <summary>
		/// Get the custom contact address previously used when registering to the SIP
		/// server. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>a <see cref="Linphone.Address">Linphone.Address</see>    </returns>
		public Linphone.Address CustomContact
		{
			get
			{
				IntPtr ptr = linphone_account_params_get_custom_contact(nativePtr);
				Linphone.Address obj = fromNativePtr<Linphone.Address>(ptr, true);
				return obj;
			}
			set
			{
				linphone_account_params_set_custom_contact(nativePtr, value.nativePtr);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_account_params_dial_escape_plus_enabled(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_enable_dial_escape_plus(IntPtr thiz, char enable);

		/// <summary>
		/// Return whether or not the + should be replaced by the International Call
		/// Prefix. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>Whether liblinphone should replace &quot;+&quot; by the International
		/// Call Prefix. in dialed numbers (passed to <see cref="Linphone.Core.Invite()">
		/// Linphone.Core.Invite()</see>). </returns>
		public bool DialEscapePlusEnabled
		{
			get
			{
				return linphone_account_params_dial_escape_plus_enabled(nativePtr) != 0;
			}
			set
			{
				linphone_account_params_enable_dial_escape_plus(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_get_domain(IntPtr thiz);

		/// <summary>
		/// Get the domain name of the given account params. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The domain name of the account params.    </returns>
		public string Domain
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_params_get_domain(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_account_params_get_expires(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_expires(IntPtr thiz, int expires);

		/// <summary>
		/// Get the account params expires. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The duration of registration. </returns>
		public int Expires
		{
			get
			{
				return linphone_account_params_get_expires(nativePtr);
			}
			set
			{
				linphone_account_params_set_expires(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_get_identity(IntPtr thiz);

		/// <summary>
		/// Get the identity of the account params. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The SIP identity that belongs to this account params.    </returns>
		/// deprecated : 01/03/2021 Use <see cref="Linphone.AccountParams.IdentityAddress">
		/// Linphone.AccountParams.IdentityAddress</see> instead. 
		public string Identity
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_params_get_identity(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_get_identity_address(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_account_params_set_identity_address(IntPtr thiz, IntPtr identity);

		/// <summary>
		/// Get the identity address of the account params. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The SIP identity that belongs to this account params.    </returns>
		public Linphone.Address IdentityAddress
		{
			get
			{
				IntPtr ptr = linphone_account_params_get_identity_address(nativePtr);
				Linphone.Address obj = fromNativePtr<Linphone.Address>(ptr, true);
				return obj;
			}
			set
			{
				int exception_result = linphone_account_params_set_identity_address(nativePtr, value.nativePtr);
				if (exception_result != 0) throw new LinphoneException("IdentityAddress setter returned value " + exception_result);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_get_idkey(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_idkey(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string idkey);

		/// <summary>
		/// Get the idkey property of a <see cref="Linphone.AccountParams">
		/// Linphone.AccountParams</see>. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The idkey string, or null.    </returns>
		public string Idkey
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_params_get_idkey(nativePtr) );
			}
			set
			{
				linphone_account_params_set_idkey(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_account_params_get_instant_messaging_encryption_mandatory(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_instant_messaging_encryption_mandatory(IntPtr thiz, char mandatory);

		/// <summary>
		/// Checks if encryption is mandatory for instant messages or not. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if encryption is mandatory; false otherwise. </returns>
		public bool InstantMessagingEncryptionMandatory
		{
			get
			{
				return linphone_account_params_get_instant_messaging_encryption_mandatory(nativePtr) != 0;
			}
			set
			{
				linphone_account_params_set_instant_messaging_encryption_mandatory(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_get_international_prefix(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_international_prefix(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string prefix);

		/// <summary>
		/// Gets the prefix set for this account params. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The international prefix if set, null otherwise.    </returns>
		public string InternationalPrefix
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_params_get_international_prefix(nativePtr) );
			}
			set
			{
				linphone_account_params_set_international_prefix(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_get_international_prefix_iso_country_code(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_international_prefix_iso_country_code(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string prefixIsoCountryCode);

		/// <summary>
		/// Gets the ISO country code set for the international prefix in this account
		/// params. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The international prefix ISO country code if set, null otherwise.   
		/// </returns>
		public string InternationalPrefixIsoCountryCode
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_params_get_international_prefix_iso_country_code(nativePtr) );
			}
			set
			{
				linphone_account_params_set_international_prefix_iso_country_code(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_account_params_is_push_notification_available(IntPtr thiz);

		/// <summary>
		/// Gets whether push notifications are available or not (Android &amp; iOS only). 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if push notifications are available, false otherwise </returns>
		public bool IsPushNotificationAvailable
		{
			get
			{
				return linphone_account_params_is_push_notification_available(nativePtr) != 0;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_get_lime_algo(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_lime_algo(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string algo);

		/// <summary>
		/// Get the base x3dh algorithm. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The x3dh base algorithm.    </returns>
		public string LimeAlgo
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_params_get_lime_algo(nativePtr) );
			}
			set
			{
				linphone_account_params_set_lime_algo(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_get_lime_server_url(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_lime_server_url(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string url);

		/// <summary>
		/// Get the x3dh server url. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The x3dh server url.    </returns>
		public string LimeServerUrl
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_params_get_lime_server_url(nativePtr) );
			}
			set
			{
				linphone_account_params_set_lime_server_url(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_get_mwi_server_address(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_mwi_server_address(IntPtr thiz, IntPtr address);

		/// <summary>
		/// Gets the Message Waiting Indication server address. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The Message Waiting Indication server address.    </returns>
		public Linphone.Address MwiServerAddress
		{
			get
			{
				IntPtr ptr = linphone_account_params_get_mwi_server_address(nativePtr);
				Linphone.Address obj = fromNativePtr<Linphone.Address>(ptr, true);
				return obj;
			}
			set
			{
				linphone_account_params_set_mwi_server_address(nativePtr, value.nativePtr);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_get_nat_policy(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_nat_policy(IntPtr thiz, IntPtr policy);

		/// <summary>
		/// Get The policy that is used to pass through NATs/firewalls when using this
		/// account params. 
		/// <para>
		/// If it is set to null, the default NAT policy from the core will be used
		/// instead. 
		/// </para>
		/// </summary>
		/// <returns>The <see cref="Linphone.NatPolicy">Linphone.NatPolicy</see> object in
		/// use.    </returns>
		/// <remarks>See : <see cref="Linphone.Core.NatPolicy">
		/// Linphone.Core.NatPolicy</see> </remarks>
		public Linphone.NatPolicy NatPolicy
		{
			get
			{
				IntPtr ptr = linphone_account_params_get_nat_policy(nativePtr);
				Linphone.NatPolicy obj = fromNativePtr<Linphone.NatPolicy>(ptr, true);
				return obj;
			}
			set
			{
				linphone_account_params_set_nat_policy(nativePtr, value.nativePtr);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_account_params_outbound_proxy_enabled(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_enable_outbound_proxy(IntPtr thiz, char enable);

		/// <summary>
		/// Tell if the proxy is used as the only route. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>enable true if enabled, false otherwise. </returns>
		public bool OutboundProxyEnabled
		{
			get
			{
				return linphone_account_params_outbound_proxy_enabled(nativePtr) != 0;
			}
			set
			{
				linphone_account_params_enable_outbound_proxy(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_get_picture_uri(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_picture_uri(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string uri);

		/// <summary>
		/// Gets the account picture URI if set, null otherwise. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The account picture URI.    </returns>
		public string PictureUri
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_params_get_picture_uri(nativePtr) );
			}
			set
			{
				linphone_account_params_set_picture_uri(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern uint linphone_account_params_get_privacy(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_privacy(IntPtr thiz, uint privacy);

		/// <summary>
		/// Get default privacy policy for all calls routed through this proxy. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>Privacy mode as LinphonePrivacyMask </returns>
		public uint Privacy
		{
			get
			{
				return linphone_account_params_get_privacy(nativePtr);
			}
			set
			{
				linphone_account_params_set_privacy(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_account_params_publish_enabled(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_enable_publish(IntPtr thiz, char enable);

		/// <summary>
		/// Tell if the PUBLISH is enabled. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if PUBLISH request is enabled for this proxy. </returns>
		public bool PublishEnabled
		{
			get
			{
				return linphone_account_params_publish_enabled(nativePtr) != 0;
			}
			set
			{
				linphone_account_params_enable_publish(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_account_params_get_publish_expires(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_publish_expires(IntPtr thiz, int expires);

		/// <summary>
		/// Get the publish expiration time in second. 
		/// <para>
		/// Default value is the registration expiration value. 
		/// </para>
		/// </summary>
		/// <returns>The expire time in seconds. </returns>
		public int PublishExpires
		{
			get
			{
				return linphone_account_params_get_publish_expires(nativePtr);
			}
			set
			{
				linphone_account_params_set_publish_expires(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_account_params_get_push_notification_allowed(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_push_notification_allowed(IntPtr thiz, char allow);

		/// <summary>
		/// Indicates whether to add to the contact parameters the push notification
		/// information. 
		/// <para>
		/// For IOS, it indicates for VOIP push notification. 
		/// </para>
		/// </summary>
		/// <returns>true if push notification informations should be added, false
		/// otherwise. </returns>
		public bool PushNotificationAllowed
		{
			get
			{
				return linphone_account_params_get_push_notification_allowed(nativePtr) != 0;
			}
			set
			{
				linphone_account_params_set_push_notification_allowed(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_get_push_notification_config(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_push_notification_config(IntPtr thiz, IntPtr config);

		/// <summary>
		/// Returns the push notification configuration. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The <see cref="Linphone.PushNotificationConfig">
		/// Linphone.PushNotificationConfig</see> object.    </returns>
		public Linphone.PushNotificationConfig PushNotificationConfig
		{
			get
			{
				IntPtr ptr = linphone_account_params_get_push_notification_config(nativePtr);
				Linphone.PushNotificationConfig obj = fromNativePtr<Linphone.PushNotificationConfig>(ptr, true);
				return obj;
			}
			set
			{
				linphone_account_params_set_push_notification_config(nativePtr, value.nativePtr);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_get_quality_reporting_collector(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_quality_reporting_collector(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string collector);

		/// <summary>
		/// Get the route of the collector end-point when using quality reporting. 
		/// <para>
		/// This SIP address should be used on server-side to process packets directly
		/// before discarding packets. Collector address should be a non existing account
		/// and will not receive any messages. If null, reports will be send to the proxy
		/// domain. 
		/// </para>
		/// </summary>
		/// <returns>The SIP address of the collector end-point.    </returns>
		public string QualityReportingCollector
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_params_get_quality_reporting_collector(nativePtr) );
			}
			set
			{
				linphone_account_params_set_quality_reporting_collector(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_account_params_quality_reporting_enabled(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_enable_quality_reporting(IntPtr thiz, char enable);

		/// <summary>
		/// Indicates whether quality statistics during call should be stored and sent to a
		/// collector according to RFC 6035. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if quality repotring is enabled, false otherwise. </returns>
		public bool QualityReportingEnabled
		{
			get
			{
				return linphone_account_params_quality_reporting_enabled(nativePtr) != 0;
			}
			set
			{
				linphone_account_params_enable_quality_reporting(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_account_params_get_quality_reporting_interval(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_quality_reporting_interval(IntPtr thiz, int interval);

		/// <summary>
		/// Get the interval between interval reports when using quality reporting. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The interval in seconds, 0 means interval reports are disabled.
		/// </returns>
		public int QualityReportingInterval
		{
			get
			{
				return linphone_account_params_get_quality_reporting_interval(nativePtr);
			}
			set
			{
				linphone_account_params_set_quality_reporting_interval(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_get_realm(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_realm(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string realm);

		/// <summary>
		/// Get the realm of the given account params. 
		/// <para>
		/// This is optional, but recommended as it allows digest authentication context to
		/// be re-used accross subsequent SIP requests, which reduces by almost half the
		/// number of SIP rmessages exchanged between a client and a server. The server is
		/// required to support the qop=auth digest authentication mode to benefit from
		/// this feature. </para>
		/// </summary>
		/// <remarks>See : rfc7616 https://datatracker.ietf.org/doc/html/rfc7616 </remarks>
		/// <returns>The realm of the account params.    </returns>
		public string Realm
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_params_get_realm(nativePtr) );
			}
			set
			{
				linphone_account_params_set_realm(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_get_ref_key(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_ref_key(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string refkey);

		/// <summary>
		/// Get the persistent reference key associated to the account params. 
		/// <para>
		/// The reference key can be for example an id to an external database. It is
		/// stored in the config file, thus can survive to process exits/restarts.
		/// </para>
		/// </summary>
		/// <returns>The reference key string that has been associated to the account
		/// params, or null if none has been associated.    </returns>
		public string RefKey
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_params_get_ref_key(nativePtr) );
			}
			set
			{
				linphone_account_params_set_ref_key(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_account_params_register_enabled(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_enable_register(IntPtr thiz, char enable);

		/// <summary>
		/// Returns whether the account params is enabled or not. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if registration to the proxy is enabled. </returns>
		public bool RegisterEnabled
		{
			get
			{
				return linphone_account_params_register_enabled(nativePtr) != 0;
			}
			set
			{
				linphone_account_params_enable_register(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_account_params_get_remote_push_notification_allowed(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_remote_push_notification_allowed(IntPtr thiz, char allow);

		/// <summary>
		/// Indicates whether to add to the contact parameters the push notification
		/// information. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if remote push notification informations should be added, false
		/// otherwise. </returns>
		public bool RemotePushNotificationAllowed
		{
			get
			{
				return linphone_account_params_get_remote_push_notification_allowed(nativePtr) != 0;
			}
			set
			{
				linphone_account_params_set_remote_push_notification_allowed(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_get_routes_addresses(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_account_params_set_routes_addresses(IntPtr thiz, IntPtr routes);

		/// <summary>
		/// Gets the list of the routes set for this account params. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The list of routes.      </returns>
		public IEnumerable<Linphone.Address> RoutesAddresses
		{
			get
			{
				return MarshalBctbxList<Linphone.Address>(linphone_account_params_get_routes_addresses(nativePtr), true);
			}
			set
			{
				int exception_result = linphone_account_params_set_routes_addresses(nativePtr, ObjectArrayToBctbxList<Linphone.Address>(value));
				if (exception_result != 0) throw new LinphoneException("RoutesAddresses setter returned value " + exception_result);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_account_params_rtp_bundle_assumption_enabled(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_enable_rtp_bundle_assumption(IntPtr thiz, char val);

		/// <summary>
		/// Returns whether RTP bundle mode is assumed. 
		/// <para>
		/// See https://datatracker.ietf.org/doc/html/rfc8843 for more information. 
		/// </para>
		/// </summary>
		/// <returns>a boolean indicating when rtp bundle support is assumed. </returns>
		public bool RtpBundleAssumptionEnabled
		{
			get
			{
				return linphone_account_params_rtp_bundle_assumption_enabled(nativePtr) != 0;
			}
			set
			{
				linphone_account_params_enable_rtp_bundle_assumption(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_account_params_rtp_bundle_enabled(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_enable_rtp_bundle(IntPtr thiz, char val);

		/// <summary>
		/// Returns whether RTP bundle mode (also known as Media Multiplexing) is enabled. 
		/// <para>
		/// See https://datatracker.ietf.org/doc/html/rfc8843 for more information. 
		/// </para>
		/// </summary>
		/// <returns>a boolean indicating the enablement of rtp bundle mode. </returns>
		public bool RtpBundleEnabled
		{
			get
			{
				return linphone_account_params_rtp_bundle_enabled(nativePtr) != 0;
			}
			set
			{
				linphone_account_params_enable_rtp_bundle(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_get_server_addr(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_account_params_set_server_addr(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string serverAddress);

		/// <summary>
		/// Get the account params proxy address. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The proxy&apos;s SIP address.    </returns>
		/// deprecated : 01/03/2021 Use <see cref="Linphone.AccountParams.ServerAddress">
		/// Linphone.AccountParams.ServerAddress</see> instead. 
		public string ServerAddr
		{
			get
			{
				return linphone_pointer_to_string( linphone_account_params_get_server_addr(nativePtr) );
			}
			set
			{
				int exception_result = linphone_account_params_set_server_addr(nativePtr, value);
				if (exception_result != 0) throw new LinphoneException("ServerAddr setter returned value " + exception_result);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_get_server_address(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_account_params_set_server_address(IntPtr thiz, IntPtr serverAddress);

		/// <summary>
		/// Get the account params SIP proxy or registrar address. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The proxy&apos;s SIP <see cref="Linphone.Address">
		/// Linphone.Address</see>.    </returns>
		public Linphone.Address ServerAddress
		{
			get
			{
				IntPtr ptr = linphone_account_params_get_server_address(nativePtr);
				Linphone.Address obj = fromNativePtr<Linphone.Address>(ptr, true);
				return obj;
			}
			set
			{
				int exception_result = linphone_account_params_set_server_address(nativePtr, value.nativePtr);
				if (exception_result != 0) throw new LinphoneException("ServerAddress setter returned value " + exception_result);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_get_supported_tags_list(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_supported_tags_list(IntPtr thiz, IntPtr supportedTags);

		/// <summary>
		/// Gets the list of supported tags. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The list of supported tags  .    </returns>
		public IEnumerable<string> SupportedTagsList
		{
			get
			{
				return MarshalStringArray(linphone_account_params_get_supported_tags_list(nativePtr));
			}
			set
			{
				IntPtr bctbxList = StringArrayToBctbxList(value);
				linphone_account_params_set_supported_tags_list(nativePtr, bctbxList);
				CleanStringArrayPtrs(bctbxList);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.TransportType linphone_account_params_get_transport(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_transport(IntPtr thiz, int transport);

		/// <summary>
		/// Returns the transport type of the server address. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The <see cref="Linphone.TransportType">Linphone.TransportType</see> of
		/// the server address. </returns>
		public Linphone.TransportType Transport
		{
			get
			{
				return linphone_account_params_get_transport(nativePtr);
			}
			set
			{
				linphone_account_params_set_transport(nativePtr, (int)value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_account_params_unregister_at_stop_enabled(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_enable_unregister_at_stop(IntPtr thiz, char unregister);

		/// <summary>
		/// Gets whether the account will unREGISTER when the core stops but only if the
		/// push notifications are not allowed for the account. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if the account will unREGISTER at stop, false otherwise.
		/// </returns>
		public bool UnregisterAtStopEnabled
		{
			get
			{
				return linphone_account_params_unregister_at_stop_enabled(nativePtr) != 0;
			}
			set
			{
				linphone_account_params_enable_unregister_at_stop(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_account_params_get_use_international_prefix_for_calls_and_chats(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_use_international_prefix_for_calls_and_chats(IntPtr thiz, char enable);

		/// <summary>
		/// Return whether or not the international prefix will automaticaly be used for
		/// calls and chats. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>Whether we should use international prefix automatically for calls.
		/// </returns>
		public bool UseInternationalPrefixForCallsAndChats
		{
			get
			{
				return linphone_account_params_get_use_international_prefix_for_calls_and_chats(nativePtr) != 0;
			}
			set
			{
				linphone_account_params_set_use_international_prefix_for_calls_and_chats(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_get_voicemail_address(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_set_voicemail_address(IntPtr thiz, IntPtr address);

		/// <summary>
		/// Gets the Voicemail address. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The Voicemail address.    </returns>
		public Linphone.Address VoicemailAddress
		{
			get
			{
				IntPtr ptr = linphone_account_params_get_voicemail_address(nativePtr);
				Linphone.Address obj = fromNativePtr<Linphone.Address>(ptr, true);
				return obj;
			}
			set
			{
				linphone_account_params_set_voicemail_address(nativePtr, value.nativePtr);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_account_params_add_custom_param(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string key, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string val);

		/// <summary>
		/// Set one custom parameter to this <see cref="Linphone.AccountParams">
		/// Linphone.AccountParams</see>. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="key">key of the searched parameter.    </param>
		/// <param name="val">value of the searched parameter.    </param>
		public void AddCustomParam(string key, string val)
		{
			
			linphone_account_params_add_custom_param(nativePtr, key, val);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_clone(IntPtr thiz);

		/// <summary>
		/// Instantiate a new account params with values from source. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The newly created <see cref="Linphone.AccountParams">
		/// Linphone.AccountParams</see> object.    </returns>
		public Linphone.AccountParams Clone()
		{
			
			IntPtr ptr = linphone_account_params_clone(nativePtr);
			Linphone.AccountParams returnVal = fromNativePtr<Linphone.AccountParams>(ptr, false);
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_account_params_get_custom_param(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string key);

		/// <summary>
		/// Get the custom parameter with key to this <see cref="Linphone.AccountParams">
		/// Linphone.AccountParams</see>. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="key">key of the searched parameter.    </param>
		/// <returns>The value of the parameter with key if found or an empty string
		/// otherwise.    </returns>
		public string GetCustomParam(string key)
		{
			
			string returnVal = linphone_pointer_to_string(linphone_account_params_get_custom_param(nativePtr, key));
			
						return returnVal;
		}
	}
	/// <summary>
	/// Object that represents a parsed SIP address. 
	/// <para>
	/// A SIP address is made of display name, username, domain name, port, and various
	/// uri headers (such as tags). It looks like &apos;Alice
	/// &lt;sip:alice@example.net&gt;&apos;.
	/// You can create an address using <see cref="Linphone.Factory.CreateAddress()">
	/// Linphone.Factory.CreateAddress()</see> or <see cref="Linphone.Core.InterpretUrl()">
	/// Linphone.Core.InterpretUrl()</see> and both will return a null object if it
	/// doesn&apos;t match the grammar defined by the standard.
	/// This object is used in almost every other major objects to identity people
	/// (including yourself) &amp; servers.
	/// The <see cref="Linphone.Address">Linphone.Address</see> has methods to extract
	/// and manipulate all parts of the address. 
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class Address : LinphoneObject
	{


		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_address_get_display_name(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_address_set_display_name(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string displayName);

		/// <summary>
		/// Returns the display name. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the display name if any, null otherwise.    </returns>
		public string DisplayName
		{
			get
			{
				return linphone_pointer_to_string( linphone_address_get_display_name(nativePtr) );
			}
			set
			{
				int exception_result = linphone_address_set_display_name(nativePtr, value);
				if (exception_result != 0) throw new LinphoneException("DisplayName setter returned value " + exception_result);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_address_get_domain(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_address_set_domain(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string domain);

		/// <summary>
		/// Returns the domain name. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the domain name if any, null otherwise.    </returns>
		public string Domain
		{
			get
			{
				return linphone_pointer_to_string( linphone_address_get_domain(nativePtr) );
			}
			set
			{
				int exception_result = linphone_address_set_domain(nativePtr, value);
				if (exception_result != 0) throw new LinphoneException("Domain setter returned value " + exception_result);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_address_is_sip(IntPtr thiz);

		/// <summary>
		/// returns whether the address is a routable SIP address or not 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if it is a routable SIP address, false otherwise </returns>
		public bool IsSip
		{
			get
			{
				return linphone_address_is_sip(nativePtr) != 0;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_address_is_valid(IntPtr thiz);

		/// <summary>
		/// Returns if address is valid. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true is the address is valid, false otherwise </returns>
		public bool IsValid
		{
			get
			{
				return linphone_address_is_valid(nativePtr) != 0;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_address_get_method_param(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_address_set_method_param(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string methodParam);

		/// <summary>
		/// Get the value of the method parameter. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the value of the parameter or null.    </returns>
		public string MethodParam
		{
			get
			{
				return linphone_pointer_to_string( linphone_address_get_method_param(nativePtr) );
			}
			set
			{
				linphone_address_set_method_param(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_address_get_password(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_address_set_password(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string password);

		/// <summary>
		/// Get the password encoded in the address. 
		/// <para>
		/// It is used for basic authentication (not recommended). 
		/// </para>
		/// </summary>
		/// <returns>the password if any, null otherwise.    </returns>
		public string Password
		{
			get
			{
				return linphone_pointer_to_string( linphone_address_get_password(nativePtr) );
			}
			set
			{
				linphone_address_set_password(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_address_get_port(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_address_set_port(IntPtr thiz, int port);

		/// <summary>
		/// Get port number as an integer value, 0 if not present. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the port set in the address or 0 if not present. </returns>
		public int Port
		{
			get
			{
				return linphone_address_get_port(nativePtr);
			}
			set
			{
				int exception_result = linphone_address_set_port(nativePtr, value);
				if (exception_result != 0) throw new LinphoneException("Port setter returned value " + exception_result);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_address_get_scheme(IntPtr thiz);

		/// <summary>
		/// Returns the address scheme, normally &quot;sip&quot;. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the scheme if any, null otherwise.    </returns>
		public string Scheme
		{
			get
			{
				return linphone_pointer_to_string( linphone_address_get_scheme(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_address_get_secure(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_address_set_secure(IntPtr thiz, char enabled);

		/// <summary>
		/// Returns whether the address refers to a secure location (sips) or not. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if address refers to a secure location, false otherwise </returns>
		public bool Secure
		{
			get
			{
				return linphone_address_get_secure(nativePtr) != 0;
			}
			set
			{
				linphone_address_set_secure(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.TransportType linphone_address_get_transport(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_address_set_transport(IntPtr thiz, int transport);

		/// <summary>
		/// Get the transport. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>a <see cref="Linphone.TransportType">Linphone.TransportType</see>,
		/// default value if not set is UDP. </returns>
		public Linphone.TransportType Transport
		{
			get
			{
				return linphone_address_get_transport(nativePtr);
			}
			set
			{
				int exception_result = linphone_address_set_transport(nativePtr, (int)value);
				if (exception_result != 0) throw new LinphoneException("Transport setter returned value " + exception_result);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_address_set_uri_params(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string parameters);

		/// <summary>
		/// Set the value of the parameters of the URI of the address. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="parameters">The parameters string </param>
		public string UriParams
		{
			set
			{
				linphone_address_set_uri_params(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_address_get_username(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_address_set_username(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string username);

		/// <summary>
		/// Returns the username. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the username name if any, null otherwise.    </returns>
		public string Username
		{
			get
			{
				return linphone_pointer_to_string( linphone_address_get_username(nativePtr) );
			}
			set
			{
				int exception_result = linphone_address_set_username(nativePtr, value);
				if (exception_result != 0) throw new LinphoneException("Username setter returned value " + exception_result);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_address_as_string(IntPtr thiz);

		/// <summary>
		/// Returns the address as a string. 
		/// <para>
		/// The returned char * must be freed by the application. Use ms_free(). 
		/// </para>
		/// </summary>
		/// <returns>a string representation of the address.       </returns>
		public string AsString()
		{
			
			string returnVal = linphone_pointer_to_string(linphone_address_as_string(nativePtr));
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_address_as_string_uri_only(IntPtr thiz);

		/// <summary>
		/// Returns the SIP uri only as a string, that is display name is removed. 
		/// <para>
		/// The returned char * must be freed by the application. Use ms_free(). 
		/// </para>
		/// </summary>
		/// <returns>a string representation of the address.       </returns>
		public string AsStringUriOnly()
		{
			
			string returnVal = linphone_pointer_to_string(linphone_address_as_string_uri_only(nativePtr));
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_address_as_string_uri_only_ordered(IntPtr thiz);

		/// <summary>
		/// Returns the SIP uri only as a string, that is display name is removed. 
		/// <para>
		/// This function always returns a string whose URI parameters has a constant order
		/// allowing to easily compute hashes for example. The returned char * must be
		/// freed by the application. Use ms_free(). 
		/// </para>
		/// </summary>
		/// <returns>a string representation of the address.       </returns>
		public string AsStringUriOnlyOrdered()
		{
			
			string returnVal = linphone_pointer_to_string(linphone_address_as_string_uri_only_ordered(nativePtr));
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_address_clean(IntPtr thiz);

		/// <summary>
		/// Removes address&apos;s tags and uri headers so that it is displayable to the
		/// user. 
		/// <para>
		/// </para>
		/// </summary>
		public void Clean()
		{
			
			linphone_address_clean(nativePtr);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_address_clone(IntPtr thiz);

		/// <summary>
		/// Clones a <see cref="Linphone.Address">Linphone.Address</see> object. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>a new <see cref="Linphone.Address">Linphone.Address</see> object.   
		/// </returns>
		public Linphone.Address Clone()
		{
			
			IntPtr ptr = linphone_address_clone(nativePtr);
			Linphone.Address returnVal = fromNativePtr<Linphone.Address>(ptr, false);
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_address_equal(IntPtr thiz, IntPtr address2);

		/// <summary>
		/// Compare two <see cref="Linphone.Address">Linphone.Address</see> taking the tags
		/// and headers into account. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="address2"><see cref="Linphone.Address">Linphone.Address</see>
		/// object.    </param>
		/// <returns>Boolean value telling if the <see cref="Linphone.Address">
		/// Linphone.Address</see> objects are equal. </returns>
		/// <remarks>See : <see cref="Linphone.Address.WeakEqual()">
		/// Linphone.Address.WeakEqual()</see> </remarks>
		public bool Equal(Linphone.Address address2)
		{
			
			bool returnVal = linphone_address_equal(nativePtr, address2 != null ? address2.nativePtr : IntPtr.Zero) == (char)0 ? false : true;
			
			GC.KeepAlive(address2);
			return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_address_get_header(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string headerName);

		/// <summary>
		/// Get the header encoded in the address. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="headerName">the header name.    </param>
		/// <returns>the header value or null if it doesn&apos;t exists.    </returns>
		public string GetHeader(string headerName)
		{
			
			string returnVal = linphone_pointer_to_string(linphone_address_get_header(nativePtr, headerName));
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_address_get_param(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string paramName);

		/// <summary>
		/// Get the value of a parameter of the address. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="paramName">The name of the parameter.    </param>
		/// <returns>The value of the parameter or null if it doesn&apos;t exists.   
		/// </returns>
		public string GetParam(string paramName)
		{
			
			string returnVal = linphone_pointer_to_string(linphone_address_get_param(nativePtr, paramName));
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_address_get_uri_param(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string uriParamName);

		/// <summary>
		/// Get the value of a parameter of the URI of the address. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="uriParamName">The name of the parameter.    </param>
		/// <returns>The value of the parameter or null if it doesn&apos;t exists.   
		/// </returns>
		public string GetUriParam(string uriParamName)
		{
			
			string returnVal = linphone_pointer_to_string(linphone_address_get_uri_param(nativePtr, uriParamName));
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_address_has_param(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string paramName);

		/// <summary>
		/// Tell whether a parameter is present in the address. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="paramName">The name of the parameter.    </param>
		/// <returns>A boolean value telling whether the parameter is present in the
		/// address </returns>
		public bool HasParam(string paramName)
		{
			
			bool returnVal = linphone_address_has_param(nativePtr, paramName) == (char)0 ? false : true;
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_address_has_uri_param(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string uriParamName);

		/// <summary>
		/// Tell whether a parameter is present in the URI of the address. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="uriParamName">The name of the parameter.    </param>
		/// <returns>A boolean value telling whether the parameter is present in the URI of
		/// the address </returns>
		public bool HasUriParam(string uriParamName)
		{
			
			bool returnVal = linphone_address_has_uri_param(nativePtr, uriParamName) == (char)0 ? false : true;
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_address_lesser(IntPtr thiz, IntPtr other);

		/// <summary>
		/// Compare two addresses. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="other">an other <see cref="Linphone.Address">
		/// Linphone.Address</see> object.    </param>
		public bool Lesser(Linphone.Address other)
		{
			
			bool returnVal = linphone_address_lesser(nativePtr, other != null ? other.nativePtr : IntPtr.Zero) == (char)0 ? false : true;
			
			GC.KeepAlive(other);
			return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_address_remove_uri_param(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string uriParamName);

		/// <summary>
		/// Removes the value of a parameter of the URI of the address. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="uriParamName">The name of the parameter.    </param>
		public void RemoveUriParam(string uriParamName)
		{
			
			linphone_address_remove_uri_param(nativePtr, uriParamName);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_address_set_header(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string headerName, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string headerValue);

		/// <summary>
		/// Set a header into the address. 
		/// <para>
		/// Headers appear in the URI with &apos;?&apos;, such as
		/// &lt;sip:test@linphone.org?SomeHeader=SomeValue&gt;. 
		/// </para>
		/// </summary>
		/// <param name="headerName">the header name.    </param>
		/// <param name="headerValue">the header value.    </param>
		public void SetHeader(string headerName, string headerValue)
		{
			
			linphone_address_set_header(nativePtr, headerName, headerValue);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_address_set_param(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string paramName, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string paramValue);

		/// <summary>
		/// Set the value of a parameter of the address. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="paramName">The name of the parameter.    </param>
		/// <param name="paramValue">The new value of the parameter.    </param>
		public void SetParam(string paramName, string paramValue)
		{
			
			linphone_address_set_param(nativePtr, paramName, paramValue);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_address_set_uri_param(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string uriParamName, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string uriParamValue);

		/// <summary>
		/// Set the value of a parameter of the URI of the address. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="uriParamName">The name of the parameter.    </param>
		/// <param name="uriParamValue">The new value of the parameter.    </param>
		public void SetUriParam(string uriParamName, string uriParamValue)
		{
			
			linphone_address_set_uri_param(nativePtr, uriParamName, uriParamValue);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_address_weak_equal(IntPtr thiz, IntPtr address2);

		/// <summary>
		/// Compare two <see cref="Linphone.Address">Linphone.Address</see> ignoring tags
		/// and headers, basically just domain, username, and port. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="address2"><see cref="Linphone.Address">Linphone.Address</see>
		/// object.    </param>
		/// <returns>Boolean value telling if the <see cref="Linphone.Address">
		/// Linphone.Address</see> objects are equal. </returns>
		/// <remarks>See : <see cref="Linphone.Address.Equal()">
		/// Linphone.Address.Equal()</see> </remarks>
		public bool WeakEqual(Linphone.Address address2)
		{
			
			bool returnVal = linphone_address_weak_equal(nativePtr, address2 != null ? address2.nativePtr : IntPtr.Zero) == (char)0 ? false : true;
			
			GC.KeepAlive(address2);
			return returnVal;
		}
	}
	/// <summary>
	/// Object that represents an alert. 
	/// <para>
	/// Alerts are raised at run-time when particular conditions are met, for example
	/// bad network quality. The full list of available alert types is described by the
	/// <see cref="Linphone.AlertType">Linphone.AlertType</see> enum. An application is
	/// notified of new alerts through the <see cref="Linphone.CoreListener">
	/// Linphone.CoreListener</see> interface. Once raised, the application may use the
	/// <see cref="Linphone.AlertListener">Linphone.AlertListener</see> interface to
	/// get notified when the alert stops. For each kind of alert, a <see cref="Linphone.Dictionary">
	/// Linphone.Dictionary</see> is filled with relevant informations, returned by
	/// <see cref="Linphone.Alert.Informations">Linphone.Alert.Informations</see>. The
	/// keys available are documented per-type in <see cref="Linphone.AlertType">
	/// Linphone.AlertType</see> enum. 
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class Alert : LinphoneObject
	{

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_factory_get();

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_factory_create_alert_cbs(IntPtr factory);

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_alert_add_callbacks(IntPtr thiz, IntPtr cbs);

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_alert_remove_callbacks(IntPtr thiz, IntPtr cbs);

		private AlertListener listener;

		public AlertListener Listener
		{
			get {
				if (listener == null)
				{
					IntPtr nativeListener = linphone_factory_create_alert_cbs(linphone_factory_get());
					listener = fromNativePtr<AlertListener>(nativeListener, false, true);
					linphone_alert_add_callbacks(nativePtr, nativeListener);
					belle_sip_object_unref(nativeListener);
				}
				return listener;
			}
			set {
				if (listener != null && value == null) {
					linphone_alert_remove_callbacks(nativePtr, listener.nativePtr);
				}
				listener = value;
			}
		}

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_alert_type_to_string(int type);

		/// <summary>
		/// Provide a string describing the alert type. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="type">the <see cref="Linphone.AlertType">Linphone.AlertType</see>
		/// </param>
		/// <returns>a string </returns>
		public static string TypeToString(Linphone.AlertType type)
		{
			
			string returnVal = linphone_pointer_to_string(linphone_alert_type_to_string((int)type));
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_alert_get_call(IntPtr thiz);

		/// <summary>
		/// Return the call from the alert. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>A <see cref="Linphone.Call">Linphone.Call</see> from the alert.   
		/// </returns>
		public Linphone.Call Call
		{
			get
			{
				IntPtr ptr = linphone_alert_get_call(nativePtr);
				Linphone.Call obj = fromNativePtr<Linphone.Call>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern long linphone_alert_get_end_time(IntPtr thiz);

		/// <summary>
		/// Return the end time of the alert. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the end time of the alert. </returns>
		public long EndTime
		{
			get
			{
				return linphone_alert_get_end_time(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_alert_get_informations(IntPtr thiz);

		/// <summary>
		/// Return more informations about the alerts. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>A <see cref="Linphone.Dictionary">Linphone.Dictionary</see> containing
		/// informations about the current alert.    </returns>
		public Linphone.Dictionary Informations
		{
			get
			{
				IntPtr ptr = linphone_alert_get_informations(nativePtr);
				Linphone.Dictionary obj = fromNativePtr<Linphone.Dictionary>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern long linphone_alert_get_start_time(IntPtr thiz);

		/// <summary>
		/// Return the start time of the alert. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the start time of the alert. </returns>
		public long StartTime
		{
			get
			{
				return linphone_alert_get_start_time(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_alert_get_state(IntPtr thiz);

		/// <summary>
		/// Return the state of the alert. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if and only if the alert is active. </returns>
		public bool State
		{
			get
			{
				return linphone_alert_get_state(nativePtr) != 0;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AlertType linphone_alert_get_type(IntPtr thiz);

		/// <summary>
		/// Return the type of the alert. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>A <see cref="Linphone.AlertType">Linphone.AlertType</see>
		/// corresponding to the current alert. </returns>
		public Linphone.AlertType Type
		{
			get
			{
				return linphone_alert_get_type(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_alert_clone(IntPtr thiz);

		/// <summary>
		/// Clone the given alert. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>A new alert with exactly same informations that param.    </returns>
		public Linphone.Alert Clone()
		{
			
			IntPtr ptr = linphone_alert_clone(nativePtr);
			Linphone.Alert returnVal = fromNativePtr<Linphone.Alert>(ptr, false);
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_alert_notify_on_terminated(IntPtr thiz);

		/// <summary>
		/// Notify the alert if it is terminated. 
		/// <para>
		/// </para>
		/// </summary>
		public void NotifyOnTerminated()
		{
			
			linphone_alert_notify_on_terminated(nativePtr);
			
			
						
		}
	}
	/// <summary>
	/// Object holding audio device information. 
	/// <para>
	/// It contains the name of the device, it&apos;s type if available (Earpiece,
	/// Speaker, Bluetooth, etc..) and capabilities (input, output or both) the name of
	/// the driver that created it (filter in mediastreamer).
	/// You can use the <see cref="Linphone.AudioDevice">Linphone.AudioDevice</see>
	/// objects to configure default input/output devices or do it dynamically during a
	/// call.
	/// To get the list of available devices, use <see cref="Linphone.Core.AudioDevices">
	/// Linphone.Core.AudioDevices</see>. This list will be limited to one device of
	/// each type. Use <see cref="Linphone.Core.ExtendedAudioDevices">
	/// Linphone.Core.ExtendedAudioDevices</see> for a complete list. 
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class AudioDevice : LinphoneObject
	{


		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AudioDeviceCapabilities linphone_audio_device_get_capabilities(IntPtr thiz);

		/// <summary>
		/// Returns the capabilities of the device. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the <see cref="Linphone.AudioDeviceCapabilities">
		/// Linphone.AudioDeviceCapabilities</see> of the audio device (RECORD, PLAY or
		/// both) as a bit mask </returns>
		public Linphone.AudioDeviceCapabilities Capabilities
		{
			get
			{
				return linphone_audio_device_get_capabilities(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_audio_device_get_device_name(IntPtr thiz);

		/// <summary>
		/// Returns the name of the audio device. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the name of the audio device.    </returns>
		public string DeviceName
		{
			get
			{
				return linphone_pointer_to_string( linphone_audio_device_get_device_name(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_audio_device_get_driver_name(IntPtr thiz);

		/// <summary>
		/// Returns the driver name used by the device. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the name of the driver used by this audio device.    </returns>
		public string DriverName
		{
			get
			{
				return linphone_pointer_to_string( linphone_audio_device_get_driver_name(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_audio_device_get_follows_system_routing_policy(IntPtr thiz);

		/// <summary>
		/// Returns whether the audio device automatically follows the system&apos;s audio
		/// routing policy. 
		/// <para>
		/// This capability is available on some system (typically iOS) and might be
		/// convenient to simply specify liblinphone to let the system decide about which
		/// audio route is being used to handle a call. The actual <see cref="Linphone.AudioDeviceType">
		/// Linphone.AudioDeviceType</see> may be unknown at some point, typically when no
		/// calls are running, otherwise it is reflected to be the actual system&apos;s
		/// audio route. 
		/// </para>
		/// </summary>
		/// <returns>true if the audio device automatically follows the system audio
		/// routing policy. </returns>
		public bool FollowsSystemRoutingPolicy
		{
			get
			{
				return linphone_audio_device_get_follows_system_routing_policy(nativePtr) != 0;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_audio_device_get_id(IntPtr thiz);

		/// <summary>
		/// Returns the id of the audio device. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the id of the audio device.    </returns>
		public string Id
		{
			get
			{
				return linphone_pointer_to_string( linphone_audio_device_get_id(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AudioDeviceType linphone_audio_device_get_type(IntPtr thiz);

		/// <summary>
		/// Returns the type of the device. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the <see cref="Linphone.AudioDeviceType">
		/// Linphone.AudioDeviceType</see> of the audio device (microphone, speaker,
		/// earpiece, bluetooth, etc...) </returns>
		public Linphone.AudioDeviceType Type
		{
			get
			{
				return linphone_audio_device_get_type(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_audio_device_has_capability(IntPtr thiz, int capability);

		/// <summary>
		/// Returns whether or not the audio device has the given capability. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="capability">the <see cref="Linphone.AudioDeviceCapabilities">
		/// Linphone.AudioDeviceCapabilities</see> to check </param>
		/// <returns>true if the audio device has the capability, false otherwise </returns>
		public bool HasCapability(Linphone.AudioDeviceCapabilities capability)
		{
			
			bool returnVal = linphone_audio_device_has_capability(nativePtr, (int)capability) == (char)0 ? false : true;
			
						return returnVal;
		}
	}
	/// <summary>
	/// Object holding authentication information. 
	/// <para>
	/// In most case, authentication information consists of a username and password.
	/// If realm isn&apos;t set, it will be deduced automatically from the first
	/// authentication challenge as for the hash algorithm. Sometimes, a userid is
	/// required by the proxy and then domain can be useful to discriminate different
	/// credentials. You can also use this object if you need to use a client
	/// certificate.
	/// Once created and filled, a <see cref="Linphone.AuthInfo">
	/// Linphone.AuthInfo</see> must be added to the <see cref="Linphone.Core">
	/// Linphone.Core</see> in order to become known and used automatically when
	/// needed. Use <see cref="Linphone.Core.AddAuthInfo()">
	/// Linphone.Core.AddAuthInfo()</see> for that purpose.
	/// The <see cref="Linphone.Core">Linphone.Core</see> object can take the
	/// initiative to request authentication information when needed to the application
	/// through the authentication_requested() callback of it&apos;s <see cref="Linphone.CoreListener">
	/// Linphone.CoreListener</see>.
	/// The application can respond to this information request later using <see cref="Linphone.Core.AddAuthInfo()">
	/// Linphone.Core.AddAuthInfo()</see>. This will unblock all pending authentication
	/// transactions and retry them with authentication headers. 
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class AuthInfo : LinphoneObject
	{


		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_auth_info_get_access_token(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_auth_info_set_access_token(IntPtr thiz, IntPtr token);

		/// <summary>
		/// Return a previously set access token. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the access token, as a <see cref="Linphone.BearerToken">
		/// Linphone.BearerToken</see> object    </returns>
		public Linphone.BearerToken AccessToken
		{
			get
			{
				IntPtr ptr = linphone_auth_info_get_access_token(nativePtr);
				Linphone.BearerToken obj = fromNativePtr<Linphone.BearerToken>(ptr, true);
				return obj;
			}
			set
			{
				linphone_auth_info_set_access_token(nativePtr, value.nativePtr);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_auth_info_get_algorithm(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_auth_info_set_algorithm(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string algorithm);

		/// <summary>
		/// Gets the algorithm. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The algorithm.    </returns>
		public string Algorithm
		{
			get
			{
				return linphone_pointer_to_string( linphone_auth_info_get_algorithm(nativePtr) );
			}
			set
			{
				linphone_auth_info_set_algorithm(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_auth_info_get_authorization_server(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_auth_info_set_authorization_server(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string uri);

		/// <summary>
		/// Get the previously set authorization server uri. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the authorization server uri.    </returns>
		public string AuthorizationServer
		{
			get
			{
				return linphone_pointer_to_string( linphone_auth_info_get_authorization_server(nativePtr) );
			}
			set
			{
				linphone_auth_info_set_authorization_server(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_auth_info_get_available_algorithms(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_auth_info_set_available_algorithms(IntPtr thiz, IntPtr algorithms);

		/// <summary>
		/// Gets all available algorithms. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>A list of available algorithms.      </returns>
		public IEnumerable<string> AvailableAlgorithms
		{
			get
			{
				return MarshalStringArray(linphone_auth_info_get_available_algorithms(nativePtr));
			}
			set
			{
				IntPtr bctbxList = StringArrayToBctbxList(value);
				linphone_auth_info_set_available_algorithms(nativePtr, bctbxList);
				CleanStringArrayPtrs(bctbxList);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_auth_info_get_client_id(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_auth_info_set_client_id(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string clientId);

		/// <summary>
		/// Get the previously set OAUTH2 client_id. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the client_id.    </returns>
		public string ClientId
		{
			get
			{
				return linphone_pointer_to_string( linphone_auth_info_get_client_id(nativePtr) );
			}
			set
			{
				linphone_auth_info_set_client_id(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_auth_info_get_client_secret(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_auth_info_set_client_secret(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string clientSecret);

		/// <summary>
		/// Get the previously set OAUTH2 client_secret. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the client_secret.    </returns>
		public string ClientSecret
		{
			get
			{
				return linphone_pointer_to_string( linphone_auth_info_get_client_secret(nativePtr) );
			}
			set
			{
				linphone_auth_info_set_client_secret(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_auth_info_get_domain(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_auth_info_set_domain(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string domain);

		/// <summary>
		/// Gets the domain. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The domain.    </returns>
		public string Domain
		{
			get
			{
				return linphone_pointer_to_string( linphone_auth_info_get_domain(nativePtr) );
			}
			set
			{
				linphone_auth_info_set_domain(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern long linphone_auth_info_get_expires(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_auth_info_set_expires(IntPtr thiz, long expires);

		/// <summary>
		/// Get the expiration time for the current authentication information. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The expiration time as a number of seconds since the Epoch, 1970-01-01
		/// 00:00:00 +0000 (UTC) </returns>
		public long Expires
		{
			get
			{
				return linphone_auth_info_get_expires(nativePtr);
			}
			set
			{
				linphone_auth_info_set_expires(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_auth_info_get_ha1(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_auth_info_set_ha1(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string ha1);

		/// <summary>
		/// Gets the ha1. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The ha1.    </returns>
		public string Ha1
		{
			get
			{
				return linphone_pointer_to_string( linphone_auth_info_get_ha1(nativePtr) );
			}
			set
			{
				linphone_auth_info_set_ha1(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_auth_info_get_password(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_auth_info_set_password(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string password);

		/// <summary>
		/// Gets the password. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The password.    </returns>
		public string Password
		{
			get
			{
				return linphone_pointer_to_string( linphone_auth_info_get_password(nativePtr) );
			}
			set
			{
				linphone_auth_info_set_password(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_auth_info_get_realm(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_auth_info_set_realm(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string realm);

		/// <summary>
		/// Gets the realm. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The realm.    </returns>
		public string Realm
		{
			get
			{
				return linphone_pointer_to_string( linphone_auth_info_get_realm(nativePtr) );
			}
			set
			{
				linphone_auth_info_set_realm(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_auth_info_get_refresh_token(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_auth_info_set_refresh_token(IntPtr thiz, IntPtr token);

		/// <summary>
		/// Return a previously set refresh token. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the refresh token, as a <see cref="Linphone.BearerToken">
		/// Linphone.BearerToken</see> object.    </returns>
		public Linphone.BearerToken RefreshToken
		{
			get
			{
				IntPtr ptr = linphone_auth_info_get_refresh_token(nativePtr);
				Linphone.BearerToken obj = fromNativePtr<Linphone.BearerToken>(ptr, true);
				return obj;
			}
			set
			{
				linphone_auth_info_set_refresh_token(nativePtr, value.nativePtr);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_auth_info_get_tls_cert(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_auth_info_set_tls_cert(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string tlsCert);

		/// <summary>
		/// Gets the TLS certificate. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The TLS certificate.    </returns>
		public string TlsCert
		{
			get
			{
				return linphone_pointer_to_string( linphone_auth_info_get_tls_cert(nativePtr) );
			}
			set
			{
				linphone_auth_info_set_tls_cert(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_auth_info_get_tls_cert_path(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_auth_info_set_tls_cert_path(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string tlsCertPath);

		/// <summary>
		/// Gets the TLS certificate path. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The TLS certificate path.    </returns>
		public string TlsCertPath
		{
			get
			{
				return linphone_pointer_to_string( linphone_auth_info_get_tls_cert_path(nativePtr) );
			}
			set
			{
				linphone_auth_info_set_tls_cert_path(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_auth_info_get_tls_key(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_auth_info_set_tls_key(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string tlsKey);

		/// <summary>
		/// Gets the TLS key. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The TLS key.    </returns>
		public string TlsKey
		{
			get
			{
				return linphone_pointer_to_string( linphone_auth_info_get_tls_key(nativePtr) );
			}
			set
			{
				linphone_auth_info_set_tls_key(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_auth_info_get_tls_key_path(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_auth_info_set_tls_key_path(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string tlsKeyPath);

		/// <summary>
		/// Gets the TLS key path. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The TLS key path.    </returns>
		public string TlsKeyPath
		{
			get
			{
				return linphone_pointer_to_string( linphone_auth_info_get_tls_key_path(nativePtr) );
			}
			set
			{
				linphone_auth_info_set_tls_key_path(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_auth_info_get_token_endpoint_uri(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_auth_info_set_token_endpoint_uri(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string uri);

		/// <summary>
		/// Get the previously set token endpoint https uri (OAUTH2). 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the token endpoint uri.    </returns>
		public string TokenEndpointUri
		{
			get
			{
				return linphone_pointer_to_string( linphone_auth_info_get_token_endpoint_uri(nativePtr) );
			}
			set
			{
				linphone_auth_info_set_token_endpoint_uri(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_auth_info_get_userid(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_auth_info_set_userid(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string userId);

		/// <summary>
		/// Gets the user id. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The user id.    </returns>
		public string Userid
		{
			get
			{
				return linphone_pointer_to_string( linphone_auth_info_get_userid(nativePtr) );
			}
			set
			{
				linphone_auth_info_set_userid(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_auth_info_get_username(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_auth_info_set_username(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string username);

		/// <summary>
		/// Gets the username. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The username.    </returns>
		public string Username
		{
			get
			{
				return linphone_pointer_to_string( linphone_auth_info_get_username(nativePtr) );
			}
			set
			{
				linphone_auth_info_set_username(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_auth_info_add_available_algorithm(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string algorithm);

		/// <summary>
		/// Add an unique algorithm in the the available algorithms list : Algorithms that
		/// already exist will not be added. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="algorithm">The algorithm to add.    </param>
		public void AddAvailableAlgorithm(string algorithm)
		{
			
			linphone_auth_info_add_available_algorithm(nativePtr, algorithm);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_auth_info_clear_available_algorithms(IntPtr thiz);

		/// <summary>
		/// Remove all algorithms from the available algorithms list. 
		/// <para>
		/// </para>
		/// </summary>
		public void ClearAvailableAlgorithms()
		{
			
			linphone_auth_info_clear_available_algorithms(nativePtr);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_auth_info_clone(IntPtr thiz);

		/// <summary>
		/// Instantiates a new auth info with values from source. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The newly created <see cref="Linphone.AuthInfo">
		/// Linphone.AuthInfo</see> object.    </returns>
		public Linphone.AuthInfo Clone()
		{
			
			IntPtr ptr = linphone_auth_info_clone(nativePtr);
			Linphone.AuthInfo returnVal = fromNativePtr<Linphone.AuthInfo>(ptr, false);
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_auth_info_is_equal_but_algorithms(IntPtr thiz, IntPtr authInfo2);

		/// <summary>
		/// Check if Authinfos are the same without taking account algorithms. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="authInfo2">The second <see cref="Linphone.AuthInfo">
		/// Linphone.AuthInfo</see> object.    </param>
		/// <returns>true if all fields (Username, UserId, Realm, Domain) are the same.
		/// </returns>
		public bool IsEqualButAlgorithms(Linphone.AuthInfo authInfo2)
		{
			
			bool returnVal = linphone_auth_info_is_equal_but_algorithms(nativePtr, authInfo2 != null ? authInfo2.nativePtr : IntPtr.Zero) == (char)0 ? false : true;
			
			GC.KeepAlive(authInfo2);
			return returnVal;
		}
	}
	/// <summary>
	/// Object that represents a bearer token (eg OAUTH). 
	/// <para>
	/// SIP servers may support &quot;bearer&quot; kind of authentication, in which
	/// case an authentication token needs to be supplied in order to authenticate to
	/// the SIP service. Applications are responsible to obtain the token from an
	/// authentication server. In order to pass it to liblinphone for usage, the token
	/// needs to be encapsulated into a <see cref="Linphone.BearerToken">
	/// Linphone.BearerToken</see>, together with its expiration time and target server
	/// name for which it is intended to use, then passed into a <see cref="Linphone.AuthInfo">
	/// Linphone.AuthInfo</see> object. Both access and refresh tokens may be
	/// represented. If both are provided to the <see cref="Linphone.AuthInfo">
	/// Linphone.AuthInfo</see>, then liblinphone automatically uses the refresh token
	/// to obtain a new access token when the latter is expired. 
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class BearerToken : LinphoneObject
	{


		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern long linphone_bearer_token_get_expiration_time(IntPtr thiz);

		/// <summary>
		/// Get the token exiration time, as a number of seconds since EPOCH. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the expiration time </returns>
		public long ExpirationTime
		{
			get
			{
				return linphone_bearer_token_get_expiration_time(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_bearer_token_get_token(IntPtr thiz);

		/// <summary>
		/// Get the token as a string. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the token.    </returns>
		public string Token
		{
			get
			{
				return linphone_pointer_to_string( linphone_bearer_token_get_token(nativePtr) );
			}
		}
	}
	/// <summary>
	/// The object representing a data buffer. 
	/// <para>
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class Buffer : LinphoneObject
	{


		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_buffer_new_from_data(uint data, long size);

		/// <summary>
		/// Create a new <see cref="Linphone.Buffer">Linphone.Buffer</see> object from
		/// existing data. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="data">The initial data to store in the LinphoneBuffer.    </param>
		/// <param name="size">The size of the initial data to store in the LinphoneBuffer.
		/// </param>
		/// <returns>A new <see cref="Linphone.Buffer">Linphone.Buffer</see> object.   
		/// </returns>
		public static Linphone.Buffer NewFromData(uint data, long size)
		{
			
			IntPtr ptr = linphone_buffer_new_from_data(data, size);
			Linphone.Buffer returnVal = fromNativePtr<Linphone.Buffer>(ptr, false);
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_buffer_new_from_string([MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string data);

		/// <summary>
		/// Create a new <see cref="Linphone.Buffer">Linphone.Buffer</see> object from a
		/// string. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="data">The initial string content of the LinphoneBuffer.    </param>
		/// <returns>A new <see cref="Linphone.Buffer">Linphone.Buffer</see> object.   
		/// </returns>
		public static Linphone.Buffer NewFromString(string data)
		{
			
			IntPtr ptr = linphone_buffer_new_from_string(data);
			Linphone.Buffer returnVal = fromNativePtr<Linphone.Buffer>(ptr, false);
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern uint linphone_buffer_get_content(IntPtr thiz);

		/// <summary>
		/// Get the content of the data buffer. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The content of the data buffer.    </returns>
		public uint Content
		{
			get
			{
				return linphone_buffer_get_content(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_buffer_is_empty(IntPtr thiz);

		/// <summary>
		/// Tell whether the <see cref="Linphone.Buffer">Linphone.Buffer</see> is empty. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>A boolean value telling whether the <see cref="Linphone.Buffer">
		/// Linphone.Buffer</see> is empty or not. </returns>
		public bool IsEmpty
		{
			get
			{
				return linphone_buffer_is_empty(nativePtr) != 0;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern long linphone_buffer_get_size(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_buffer_set_size(IntPtr thiz, long size);

		/// <summary>
		/// Get the size of the content of the data buffer. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The size of the content of the data buffer. </returns>
		public long Size
		{
			get
			{
				return linphone_buffer_get_size(nativePtr);
			}
			set
			{
				linphone_buffer_set_size(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_buffer_get_string_content(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_buffer_set_string_content(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string content);

		/// <summary>
		/// Get the string content of the data buffer. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The string content of the data buffer.    </returns>
		public string StringContent
		{
			get
			{
				return linphone_pointer_to_string( linphone_buffer_get_string_content(nativePtr) );
			}
			set
			{
				linphone_buffer_set_string_content(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_buffer_set_content(IntPtr thiz, uint content, long size);

		/// <summary>
		/// Set the content of the data buffer. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="content">The content of the data buffer.    </param>
		/// <param name="size">The size of the content of the data buffer. </param>
		public void SetContent(uint content, long size)
		{
			
			linphone_buffer_set_content(nativePtr, content, size);
			
			
						
		}
	}
	/// <summary>
	/// This object represents a call issued or received by the <see cref="Linphone.Core">
	/// Linphone.Core</see>. 
	/// <para>
	/// Linphone only allows at most one active call at any given time and it will be
	/// in <see cref="Linphone.CallState.StreamsRunning">
	/// Linphone.CallState.StreamsRunning</see>. However, if the core is locally
	/// hosting a <see cref="Linphone.Conference">Linphone.Conference</see>, you may
	/// have some or all the calls in the conference in <see cref="Linphone.CallState.StreamsRunning">
	/// Linphone.CallState.StreamsRunning</see> as well as an additional active call
	/// outside of the conference in <see cref="Linphone.CallState.StreamsRunning">
	/// Linphone.CallState.StreamsRunning</see> if the local participant of the <see cref="Linphone.Conference">
	/// Linphone.Conference</see> is not part of it.
	/// You can get the <see cref="Linphone.CallState">Linphone.CallState</see> of the
	/// call using <see cref="Linphone.Call.State">Linphone.Call.State</see>, it&apos;s
	/// current <see cref="Linphone.CallParams">Linphone.CallParams</see> with <see cref="Linphone.Call.CurrentParams">
	/// Linphone.Call.CurrentParams</see> and the latest statistics by calling <see cref="Linphone.Call.AudioStats">
	/// Linphone.Call.AudioStats</see> or <see cref="Linphone.Call.VideoStats">
	/// Linphone.Call.VideoStats</see>.
	/// The application can receive the various kind of events occuring in a call
	/// through the <see cref="Linphone.CallListener">Linphone.CallListener</see>
	/// interface, see also <see cref="Linphone.Call.AddListener()">
	/// Linphone.Call.AddListener()</see>. 
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class Call : LinphoneObject
	{
		/// Get the native window handle of the video window, casted as an unsigned long.
		public string NativeVideoWindowIdString
		{
			get
			{
				return Marshal.PtrToStringUni(linphone_call_get_native_video_window_id(nativePtr));
			}
			set
			{
				IntPtr string_ptr_to_remove = linphone_call_get_native_video_window_id(nativePtr);
				IntPtr string_ptr = Marshal.StringToHGlobalUni(value);
				linphone_call_set_native_video_window_id(nativePtr, string_ptr);
				if(string_ptr_to_remove != IntPtr.Zero)
					Marshal.FreeHGlobal(string_ptr_to_remove);
			}
		}

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_factory_get();

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_factory_create_call_cbs(IntPtr factory);

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_add_callbacks(IntPtr thiz, IntPtr cbs);

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_remove_callbacks(IntPtr thiz, IntPtr cbs);

		private CallListener listener;

		public CallListener Listener
		{
			get {
				if (listener == null)
				{
					IntPtr nativeListener = linphone_factory_create_call_cbs(linphone_factory_get());
					listener = fromNativePtr<CallListener>(nativeListener, false, true);
					linphone_call_add_callbacks(nativePtr, nativeListener);
					belle_sip_object_unref(nativeListener);
				}
				return listener;
			}
			set {
				if (listener != null && value == null) {
					linphone_call_remove_callbacks(nativePtr, listener.nativePtr);
				}
				listener = value;
			}
		}

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_audio_stats(IntPtr thiz);

		/// <summary>
		/// Returns a copy of the call statistics for the audio stream. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>a <see cref="Linphone.CallStats">Linphone.CallStats</see> object for
		/// the audio stream or null if it isn&apos;t available.       </returns>
		public Linphone.CallStats AudioStats
		{
			get
			{
				IntPtr ptr = linphone_call_get_audio_stats(nativePtr);
				Linphone.CallStats obj = fromNativePtr<Linphone.CallStats>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_authentication_token(IntPtr thiz);

		/// <summary>
		/// Returns the ZRTP authentication token to verify. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the authentication token to verify or null if ZRTP isn&apos;t enabled.
		///    </returns>
		public string AuthenticationToken
		{
			get
			{
				return linphone_pointer_to_string( linphone_call_get_authentication_token(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_get_authentication_token_verified(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_set_authentication_token_verified(IntPtr thiz, char verified);

		/// <summary>
		/// Returns whether ZRTP authentication token is verified. 
		/// <para>
		/// If not, it must be verified by users as described in ZRTP procedure. Once done,
		/// the application must inform of the results with <see cref="Linphone.Call.AuthenticationTokenVerified">
		/// Linphone.Call.AuthenticationTokenVerified</see>. 
		/// </para>
		/// </summary>
		/// <returns>true if authentication token is verifed, false otherwise. </returns>
		public bool AuthenticationTokenVerified
		{
			get
			{
				return linphone_call_get_authentication_token_verified(nativePtr) != 0;
			}
			set
			{
				linphone_call_set_authentication_token_verified(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern float linphone_call_get_average_quality(IntPtr thiz);

		/// <summary>
		/// Returns call quality averaged over all the duration of the call. 
		/// <para>
		/// See <see cref="Linphone.Call.CurrentQuality">Linphone.Call.CurrentQuality</see>
		/// for more details about quality measurement. 
		/// </para>
		/// </summary>
		/// <returns>the call average quality since tbe beginning of the call. </returns>
		public float AverageQuality
		{
			get
			{
				return linphone_call_get_average_quality(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_enable_baudot_detection(IntPtr thiz, char enabled);

		/// <summary>
		/// Indicates whether received Baudot tones should be detected. 
		/// <para>
		/// The Baudot functionality is to be enabled first by calling <see cref="Linphone.Core.EnableBaudot()">
		/// Linphone.Core.EnableBaudot()</see>. 
		/// </para>
		/// </summary>
		/// <param name="enabled">wether or not to detect received Baudot tones. </param>
		public bool BaudotDetectionEnabled
		{
			set
			{
				linphone_call_enable_baudot_detection(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_set_baudot_mode(IntPtr thiz, int mode);

		/// <summary>
		/// Defines the Baudot mode for the call. 
		/// <para>
		/// The Baudot functionality is to be enabled first by calling <see cref="Linphone.Core.EnableBaudot()">
		/// Linphone.Core.EnableBaudot()</see>. 
		/// </para>
		/// </summary>
		/// <param name="mode">The Baudot mode to use for the call. </param>
		public Linphone.BaudotMode BaudotMode
		{
			set
			{
				linphone_call_set_baudot_mode(nativePtr, (int)value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_set_baudot_pause_timeout(IntPtr thiz, uint seconds);

		/// <summary>
		/// Set the Baudot significant pause timeout after which a LETTERS tone is
		/// retransmitted before resuming transmission (in seconds). 
		/// <para>
		/// Default is 5s. The Baudot functionality is to be enabled first by calling <see cref="Linphone.Core.EnableBaudot()">
		/// Linphone.Core.EnableBaudot()</see>. 
		/// </para>
		/// </summary>
		/// <param name="seconds">The significant pause timeout in seconds. </param>
		public uint BaudotPauseTimeout
		{
			set
			{
				linphone_call_set_baudot_pause_timeout(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_set_baudot_sending_standard(IntPtr thiz, int standard);

		/// <summary>
		/// Defines the Baudot standard to use for sending Baudot tones in the call. 
		/// <para>
		/// The Baudot functionality is to be enabled first by calling <see cref="Linphone.Core.EnableBaudot()">
		/// Linphone.Core.EnableBaudot()</see>. 
		/// </para>
		/// </summary>
		/// <param name="standard">The Baudot standard to use for sending Baudot tones.
		/// </param>
		public Linphone.BaudotStandard BaudotSendingStandard
		{
			set
			{
				linphone_call_set_baudot_sending_standard(nativePtr, (int)value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_call_log(IntPtr thiz);

		/// <summary>
		/// Gets the call log associated to this call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The <see cref="Linphone.CallLog">Linphone.CallLog</see> associated
		/// with the specified <see cref="Linphone.Call">Linphone.Call</see>.    </returns>
		public Linphone.CallLog CallLog
		{
			get
			{
				IntPtr ptr = linphone_call_get_call_log(nativePtr);
				Linphone.CallLog obj = fromNativePtr<Linphone.CallLog>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_camera_enabled(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_enable_camera(IntPtr thiz, char enabled);

		/// <summary>
		/// Returns if camera pictures are allowed to be sent to the remote party. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if local video stream is being sent, false otherwise. </returns>
		public bool CameraEnabled
		{
			get
			{
				return linphone_call_camera_enabled(nativePtr) != 0;
			}
			set
			{
				linphone_call_enable_camera(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_chat_room(IntPtr thiz);

		/// <summary>
		/// Obtain a chat room for real time messaging from a call if not already existing,
		/// else return existing one. 
		/// <para>
		/// No reference is given to the caller: the chat room will be deleted when the
		/// call is ended. The call must have been accepted with a real time text stream
		/// (see <see cref="Linphone.CallParams.EnableRealtimeText()">
		/// Linphone.CallParams.EnableRealtimeText()</see>). 
		/// </para>
		/// </summary>
		/// <returns><see cref="Linphone.ChatRoom">Linphone.ChatRoom</see> where real time
		/// messaging can take place or null if chat room couldn&apos;t be created.   
		/// </returns>
		public Linphone.ChatRoom ChatRoom
		{
			get
			{
				IntPtr ptr = linphone_call_get_chat_room(nativePtr);
				Linphone.ChatRoom obj = fromNativePtr<Linphone.ChatRoom>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_conference(IntPtr thiz);

		/// <summary>
		/// Returns the associated conference object if any. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>A pointer on <see cref="Linphone.Conference">Linphone.Conference</see>
		/// or null if the call is not part of any conference.    </returns>
		public Linphone.Conference Conference
		{
			get
			{
				IntPtr ptr = linphone_call_get_conference(nativePtr);
				Linphone.Conference obj = fromNativePtr<Linphone.Conference>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_core(IntPtr thiz);

		/// <summary>
		/// Gets the core that has created the specified call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The <see cref="Linphone.Core">Linphone.Core</see> object that has
		/// created the specified call.    </returns>
		public Linphone.Core Core
		{
			get
			{
				IntPtr ptr = linphone_call_get_core(nativePtr);
				Linphone.Core obj = fromNativePtr<Linphone.Core>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_current_params(IntPtr thiz);

		/// <summary>
		/// Returns current parameters associated to the call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the current <see cref="Linphone.CallParams">Linphone.CallParams</see>
		/// of this call.    </returns>
		public Linphone.CallParams CurrentParams
		{
			get
			{
				IntPtr ptr = linphone_call_get_current_params(nativePtr);
				Linphone.CallParams obj = fromNativePtr<Linphone.CallParams>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern float linphone_call_get_current_quality(IntPtr thiz);

		/// <summary>
		/// Obtains real-time quality rating of the call. 
		/// <para>
		/// Based on local RTP statistics and RTCP feedback, a quality rating is computed
		/// and updated during all the duration of the call. This function returns its
		/// value at the time of the function call. It is expected that the rating is
		/// updated at least every 5 seconds or so. The rating is a floating point number
		/// comprised between 0 and 5.
		/// 4-5 = good quality  3-4 = average quality  2-3 = poor quality  1-2 = very poor
		/// quality  0-1 = can&apos;t be worse, mostly unusable 
		/// </para>
		/// </summary>
		/// <returns>The function returns -1 if no quality measurement is available, for
		/// example if no active audio stream exist. Otherwise it returns the quality
		/// rating. </returns>
		public float CurrentQuality
		{
			get
			{
				return linphone_call_get_current_quality(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.CallDir linphone_call_get_dir(IntPtr thiz);

		/// <summary>
		/// Returns direction of the call (incoming or outgoing). 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the <see cref="Linphone.CallDir">Linphone.CallDir</see> </returns>
		public Linphone.CallDir Dir
		{
			get
			{
				return linphone_call_get_dir(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_diversion_address(IntPtr thiz);

		/// <summary>
		/// Returns the diversion address associated to this call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the diversion address as <see cref="Linphone.Address">
		/// Linphone.Address</see> or null.    </returns>
		public Linphone.Address DiversionAddress
		{
			get
			{
				IntPtr ptr = linphone_call_get_diversion_address(nativePtr);
				Linphone.Address obj = fromNativePtr<Linphone.Address>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_call_get_duration(IntPtr thiz);

		/// <summary>
		/// Returns call&apos;s duration in seconds. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the call&apos;s duration in seconds. </returns>
		public int Duration
		{
			get
			{
				return linphone_call_get_duration(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_echo_cancellation_enabled(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_enable_echo_cancellation(IntPtr thiz, char enable);

		/// <summary>
		/// Returns if echo cancellation is enabled. 
		/// <para>
		/// </para>
		/// </summary>
		/// <remarks>See : <see cref="Linphone.Core.EnableEchoCancellation()">
		/// Linphone.Core.EnableEchoCancellation()</see>. </remarks>
		/// <returns>true if echo cancellation is enabled, false otherwise. </returns>
		public bool EchoCancellationEnabled
		{
			get
			{
				return linphone_call_echo_cancellation_enabled(nativePtr) != 0;
			}
			set
			{
				linphone_call_enable_echo_cancellation(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_echo_limiter_enabled(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_enable_echo_limiter(IntPtr thiz, char enable);

		/// <summary>
		/// Returns if echo limiter is enabled. 
		/// <para>
		/// </para>
		/// </summary>
		/// <remarks>See : <see cref="Linphone.Core.EnableEchoLimiter()">
		/// Linphone.Core.EnableEchoLimiter()</see>. </remarks>
		/// <returns>true if echo limiter is enabled, false otherwise. </returns>
		public bool EchoLimiterEnabled
		{
			get
			{
				return linphone_call_echo_limiter_enabled(nativePtr) != 0;
			}
			set
			{
				linphone_call_enable_echo_limiter(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_error_info(IntPtr thiz);

		/// <summary>
		/// Returns full details about call errors or termination reasons. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns><see cref="Linphone.ErrorInfo">Linphone.ErrorInfo</see> object holding
		/// the reason error.    </returns>
		public Linphone.ErrorInfo ErrorInfo
		{
			get
			{
				IntPtr ptr = linphone_call_get_error_info(nativePtr);
				Linphone.ErrorInfo obj = fromNativePtr<Linphone.ErrorInfo>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_input_audio_device(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_set_input_audio_device(IntPtr thiz, IntPtr audioDevice);

		/// <summary>
		/// Gets the current input device for this call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the <see cref="Linphone.AudioDevice">Linphone.AudioDevice</see> used
		/// by this call as input or null if there is currently no soundcard configured
		/// (depending on the state of the call)    </returns>
		public Linphone.AudioDevice InputAudioDevice
		{
			get
			{
				IntPtr ptr = linphone_call_get_input_audio_device(nativePtr);
				Linphone.AudioDevice obj = fromNativePtr<Linphone.AudioDevice>(ptr, true);
				return obj;
			}
			set
			{
				linphone_call_set_input_audio_device(nativePtr, value.nativePtr);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_is_recording(IntPtr thiz);

		/// <summary>
		/// Returns whether or not the call is currently being recorded. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if recording is in progress, false otherwise </returns>
		/// deprecated : 15/09/2021 Use <see cref="Linphone.CallParams.IsRecording()">
		/// Linphone.CallParams.IsRecording()</see> instead. 
		public bool IsRecording
		{
			get
			{
				return linphone_call_is_recording(nativePtr) != 0;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_local_authentication_token(IntPtr thiz);

		/// <summary>
		/// Returns the local ZRTP authentication token to verify by the remote. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the local authentication token to verify or null if ZRTP isn&apos;t
		/// enabled.    </returns>
		public string LocalAuthenticationToken
		{
			get
			{
				return linphone_pointer_to_string( linphone_call_get_local_authentication_token(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_local_tag(IntPtr thiz);

		/// <summary>
		/// Returns the local tag of the <see cref="Linphone.Call">Linphone.Call</see>. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the local tag.    </returns>
		public string LocalTag
		{
			get
			{
				return linphone_pointer_to_string( linphone_call_get_local_tag(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_get_microphone_muted(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_set_microphone_muted(IntPtr thiz, char muted);

		/// <summary>
		/// Gets microphone muted state. 
		/// <para>
		/// Note that the microphone may be disabled globally if false was given to <see cref="Linphone.Core.EnableMic()">
		/// Linphone.Core.EnableMic()</see>. 
		/// </para>
		/// </summary>
		/// <returns>The microphone muted state. </returns>
		/// <remarks>Warning : This method returns state of the mute capability of the call
		/// passed as argument. If this call is part of a conference, it is strongly
		/// recommended to call <see cref="Linphone.Conference.MicrophoneMuted">
		/// Linphone.Conference.MicrophoneMuted</see> to know whether this device is muted
		/// or not. </remarks> 
		public bool MicrophoneMuted
		{
			get
			{
				return linphone_call_get_microphone_muted(nativePtr) != 0;
			}
			set
			{
				linphone_call_set_microphone_muted(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern float linphone_call_get_microphone_volume_gain(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_set_microphone_volume_gain(IntPtr thiz, float volume);

		/// <summary>
		/// Gets microphone volume gain. 
		/// <para>
		/// If the sound backend supports it, the returned gain is equal to the gain set
		/// with the system mixer. 
		/// </para>
		/// </summary>
		/// <returns>double Percentage of the max supported volume gain. Valid values are
		/// in [ 0.0 : 1.0 ]. In case of failure, a negative value is returned </returns>
		public float MicrophoneVolumeGain
		{
			get
			{
				return linphone_call_get_microphone_volume_gain(nativePtr);
			}
			set
			{
				linphone_call_set_microphone_volume_gain(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_native_video_window_id(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_set_native_video_window_id(IntPtr thiz, IntPtr windowId);

		/// <summary>
		/// Get the native window handle of the video window, casted as an unsigned long. 
		/// <para>
		/// </para>
		/// </summary>
		/// <remarks>See : linphone_core_set_native_video_window_id for a general
		/// discussion about window IDs. </remarks>
		/// <returns>the native video window id (type may vary depending on platform).   
		/// </returns>
		public IntPtr NativeVideoWindowId
		{
			get
			{
				return linphone_call_get_native_video_window_id(nativePtr);
			}
			set
			{
				linphone_call_set_native_video_window_id(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_output_audio_device(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_set_output_audio_device(IntPtr thiz, IntPtr audioDevice);

		/// <summary>
		/// Gets the current output device for this call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the <see cref="Linphone.AudioDevice">Linphone.AudioDevice</see> used
		/// by this call as output or null if there is currently no soundcard configured
		/// (depending on the state of the call)    </returns>
		public Linphone.AudioDevice OutputAudioDevice
		{
			get
			{
				IntPtr ptr = linphone_call_get_output_audio_device(nativePtr);
				Linphone.AudioDevice obj = fromNativePtr<Linphone.AudioDevice>(ptr, true);
				return obj;
			}
			set
			{
				linphone_call_set_output_audio_device(nativePtr, value.nativePtr);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_params(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_set_params(IntPtr thiz, IntPtr parameters);

		/// <summary>
		/// Returns read-only local parameters associated with the call. 
		/// <para>
		/// This is typically the parameters passed at call initiation to <see cref="Linphone.Core.InviteAddressWithParams()">
		/// Linphone.Core.InviteAddressWithParams()</see> or <see cref="Linphone.Call.AcceptWithParams()">
		/// Linphone.Call.AcceptWithParams()</see>, or some default parameters if no <see cref="Linphone.CallParams">
		/// Linphone.CallParams</see> was explicitely passed during call initiation. 
		/// </para>
		/// </summary>
		/// <returns>the call&apos;s local parameters.    </returns>
		public Linphone.CallParams Params
		{
			get
			{
				IntPtr ptr = linphone_call_get_params(nativePtr);
				Linphone.CallParams obj = fromNativePtr<Linphone.CallParams>(ptr, true);
				return obj;
			}
			set
			{
				linphone_call_set_params(nativePtr, value.nativePtr);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern float linphone_call_get_play_volume(IntPtr thiz);

		/// <summary>
		/// Gets the mesured playback volume level (received from remote) in dbm0. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>float Volume level in dbm0. </returns>
		public float PlayVolume
		{
			get
			{
				return linphone_call_get_play_volume(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_player(IntPtr thiz);

		/// <summary>
		/// Gets a player associated with the call to play a local file and stream it to
		/// the remote peer. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>A <see cref="Linphone.Player">Linphone.Player</see> object.   
		/// </returns>
		public Linphone.Player Player
		{
			get
			{
				IntPtr ptr = linphone_call_get_player(nativePtr);
				Linphone.Player obj = fromNativePtr<Linphone.Player>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.Reason linphone_call_get_reason(IntPtr thiz);

		/// <summary>
		/// Returns the reason for a call termination (either error or normal termination) 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the <see cref="Linphone.Reason">Linphone.Reason</see> of the call
		/// termination. </returns>
		public Linphone.Reason Reason
		{
			get
			{
				return linphone_call_get_reason(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern float linphone_call_get_record_volume(IntPtr thiz);

		/// <summary>
		/// Gets the mesured record volume level (sent to remote) in dbm0. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>float Volume level in dbm0. </returns>
		public float RecordVolume
		{
			get
			{
				return linphone_call_get_record_volume(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_refer_to(IntPtr thiz);

		/// <summary>
		/// Gets the refer-to uri (if the call was transferred). 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The refer-to uri of the call (if it was transferred).    </returns>
		public string ReferTo
		{
			get
			{
				return linphone_pointer_to_string( linphone_call_get_refer_to(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_refer_to_address(IntPtr thiz);

		/// <summary>
		/// Gets the refer-to uri (if the call was transferred). 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The refer-to uri of the call (if it was transferred).    </returns>
		public Linphone.Address ReferToAddress
		{
			get
			{
				IntPtr ptr = linphone_call_get_refer_to_address(nativePtr);
				Linphone.Address obj = fromNativePtr<Linphone.Address>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_referred_by_address(IntPtr thiz);

		/// <summary>
		/// Gets the referred-by address, which is set when an incoming call is received as
		/// a consequence of a call transfer operation by a third party. 
		/// <para>
		/// The referred-by address is the sip identity of the one who initiated the
		/// transfer. 
		/// </para>
		/// </summary>
		/// <returns>The referred-by address    </returns>
		public Linphone.Address ReferredByAddress
		{
			get
			{
				IntPtr ptr = linphone_call_get_referred_by_address(nativePtr);
				Linphone.Address obj = fromNativePtr<Linphone.Address>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_remote_address(IntPtr thiz);

		/// <summary>
		/// Returns the remote address associated to this call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The <see cref="Linphone.Address">Linphone.Address</see> of the remote
		/// end of the call.    </returns>
		public Linphone.Address RemoteAddress
		{
			get
			{
				IntPtr ptr = linphone_call_get_remote_address(nativePtr);
				Linphone.Address obj = fromNativePtr<Linphone.Address>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_remote_address_as_string(IntPtr thiz);

		/// <summary>
		/// Returns the remote address associated to this call as a string. 
		/// <para>
		/// The result string must be freed by user using ms_free(). 
		/// </para>
		/// </summary>
		/// <returns>the remote address as a string.       </returns>
		/// deprecated : 06/07/2020 use <see cref="Linphone.Call.RemoteAddress">
		/// Linphone.Call.RemoteAddress</see> instead. 
		public string RemoteAddressAsString
		{
			get
			{
				return linphone_pointer_to_string( linphone_call_get_remote_address_as_string(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_remote_authentication_tokens(IntPtr thiz);

		/// <summary>
		/// Returns a list of 4 remote ZRTP authentication tokens. 
		/// <para>
		/// The user needs to select one. 
		/// </para>
		/// </summary>
		/// <returns>the authentication tokens to verify or null if ZRTP isn&apos;t
		/// enabled.      </returns>
		public IEnumerable<string> RemoteAuthenticationTokens
		{
			get
			{
				return MarshalStringArray(linphone_call_get_remote_authentication_tokens(nativePtr));
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_remote_contact(IntPtr thiz);

		/// <summary>
		/// Returns the far end&apos;s sip contact as a string, if available. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the remote contact or null.    </returns>
		public string RemoteContact
		{
			get
			{
				return linphone_pointer_to_string( linphone_call_get_remote_contact(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_remote_contact_address(IntPtr thiz);

		/// <summary>
		/// Returns the far end&apos;s sip contact as an address, if available. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the remote contact as a <see cref="Linphone.Address">
		/// Linphone.Address</see> or null.    </returns>
		public Linphone.Address RemoteContactAddress
		{
			get
			{
				IntPtr ptr = linphone_call_get_remote_contact_address(nativePtr);
				Linphone.Address obj = fromNativePtr<Linphone.Address>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_remote_params(IntPtr thiz);

		/// <summary>
		/// Returns call parameters proposed by remote. 
		/// <para>
		/// This is useful when receiving an incoming call, to know whether the remote
		/// party supports video, encryption or whatever.
		/// </para>
		/// </summary>
		/// <returns>the <see cref="Linphone.CallParams">Linphone.CallParams</see>
		/// suggested by the remote or null.    </returns>
		public Linphone.CallParams RemoteParams
		{
			get
			{
				IntPtr ptr = linphone_call_get_remote_params(nativePtr);
				Linphone.CallParams obj = fromNativePtr<Linphone.CallParams>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_remote_tag(IntPtr thiz);

		/// <summary>
		/// Returns the remote tag of the <see cref="Linphone.Call">Linphone.Call</see>. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the remote tag.    </returns>
		public string RemoteTag
		{
			get
			{
				return linphone_pointer_to_string( linphone_call_get_remote_tag(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_remote_user_agent(IntPtr thiz);

		/// <summary>
		/// Returns the far end&apos;s user agent description string, if available. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the remote user agent or null.    </returns>
		public string RemoteUserAgent
		{
			get
			{
				return linphone_pointer_to_string( linphone_call_get_remote_user_agent(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_replaced_call(IntPtr thiz);

		/// <summary>
		/// Returns the call object this call is replacing, if any. 
		/// <para>
		/// Call replacement can occur during call transfers. By default, the core
		/// automatically terminates the replaced call and accept the new one. This
		/// function allows the application to know whether a new incoming call is a one
		/// that replaces another one. 
		/// </para>
		/// </summary>
		/// <returns>the <see cref="Linphone.Call">Linphone.Call</see> object this call is
		/// replacing or null.    </returns>
		public Linphone.Call ReplacedCall
		{
			get
			{
				IntPtr ptr = linphone_call_get_replaced_call(nativePtr);
				Linphone.Call obj = fromNativePtr<Linphone.Call>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_request_address(IntPtr thiz);

		/// <summary>
		/// The address to which the call has been sent, taken directly from the SIP
		/// request-URI of the INVITE. 
		/// <para>
		/// Usually equal to the To field, except when e.g. using a fallback contact
		/// address. You should probably use getToAddress() instead, unless you know what
		/// you&apos;re doing. 
		/// </para>
		/// </summary>
		/// <returns>the <see cref="Linphone.Address">Linphone.Address</see> matching the
		/// URI of the INVITE request.    </returns>
		public Linphone.Address RequestAddress
		{
			get
			{
				IntPtr ptr = linphone_call_get_request_address(nativePtr);
				Linphone.Address obj = fromNativePtr<Linphone.Address>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_get_speaker_muted(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_set_speaker_muted(IntPtr thiz, char muted);

		/// <summary>
		/// Gets speaker muted state. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The speaker muted state. </returns>
		public bool SpeakerMuted
		{
			get
			{
				return linphone_call_get_speaker_muted(nativePtr) != 0;
			}
			set
			{
				linphone_call_set_speaker_muted(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern float linphone_call_get_speaker_volume_gain(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_set_speaker_volume_gain(IntPtr thiz, float volume);

		/// <summary>
		/// Gets speaker volume gain. 
		/// <para>
		/// If the sound backend supports it, the returned gain is equal to the gain set
		/// with the system mixer. 
		/// </para>
		/// </summary>
		/// <returns>Percentage of the max supported volume gain. Valid values are in [ 0.0
		/// : 1.0 ]. In case of failure, a negative value is returned </returns>
		public float SpeakerVolumeGain
		{
			get
			{
				return linphone_call_get_speaker_volume_gain(nativePtr);
			}
			set
			{
				linphone_call_set_speaker_volume_gain(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.CallState linphone_call_get_state(IntPtr thiz);

		/// <summary>
		/// Retrieves the call&apos;s current state. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the current <see cref="Linphone.CallState">Linphone.CallState</see> of
		/// this call. </returns>
		public Linphone.CallState State
		{
			get
			{
				return linphone_call_get_state(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_call_get_stream_count(IntPtr thiz);

		/// <summary>
		/// Returns the number of stream for the given call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the amount of streams for this call. </returns>
		public int StreamCount
		{
			get
			{
				return linphone_call_get_stream_count(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_text_stats(IntPtr thiz);

		/// <summary>
		/// Returns a copy of the call statistics for the text stream. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>a <see cref="Linphone.CallStats">Linphone.CallStats</see> object for
		/// the text stream or null if it isn&apos;t available.       </returns>
		public Linphone.CallStats TextStats
		{
			get
			{
				IntPtr ptr = linphone_call_get_text_stats(nativePtr);
				Linphone.CallStats obj = fromNativePtr<Linphone.CallStats>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_to_address(IntPtr thiz);

		/// <summary>
		/// Returns the to address with its headers associated to this call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the <see cref="Linphone.Address">Linphone.Address</see> matching the
		/// TO of the call.    </returns>
		public Linphone.Address ToAddress
		{
			get
			{
				IntPtr ptr = linphone_call_get_to_address(nativePtr);
				Linphone.Address obj = fromNativePtr<Linphone.Address>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.CallState linphone_call_get_transfer_state(IntPtr thiz);

		/// <summary>
		/// Returns the current transfer state, if a transfer has been initiated from this
		/// call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <remarks>See : linphone_core_transfer_call ,
		/// linphone_core_transfer_call_to_another </remarks>
		/// <returns>the <see cref="Linphone.CallState">Linphone.CallState</see>. </returns>
		public Linphone.CallState TransferState
		{
			get
			{
				return linphone_call_get_transfer_state(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_transfer_target_call(IntPtr thiz);

		/// <summary>
		/// When this call has received a transfer request, returns the new call that was
		/// automatically created as a result of the transfer. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the transfer <see cref="Linphone.Call">Linphone.Call</see> created.   
		/// </returns>
		public Linphone.Call TransferTargetCall
		{
			get
			{
				IntPtr ptr = linphone_call_get_transfer_target_call(nativePtr);
				Linphone.Call obj = fromNativePtr<Linphone.Call>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_transferer_call(IntPtr thiz);

		/// <summary>
		/// Gets the transferer if this call was started automatically as a result of an
		/// incoming transfer request. 
		/// <para>
		/// The call in which the transfer request was received is returned in this case. 
		/// </para>
		/// </summary>
		/// <returns>The transferer <see cref="Linphone.Call">Linphone.Call</see> if the
		/// specified call was started automatically as a result of an incoming transfer
		/// request, null otherwise.    </returns>
		public Linphone.Call TransfererCall
		{
			get
			{
				IntPtr ptr = linphone_call_get_transferer_call(nativePtr);
				Linphone.Call obj = fromNativePtr<Linphone.Call>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_video_source(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_set_video_source(IntPtr thiz, IntPtr descriptor);

		/// <summary>
		/// Gets the video source of a call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The <see cref="Linphone.VideoSourceDescriptor">
		/// Linphone.VideoSourceDescriptor</see> describing the video source that is set   
		/// </returns>
		public Linphone.VideoSourceDescriptor VideoSource
		{
			get
			{
				IntPtr ptr = linphone_call_get_video_source(nativePtr);
				Linphone.VideoSourceDescriptor obj = fromNativePtr<Linphone.VideoSourceDescriptor>(ptr, true);
				return obj;
			}
			set
			{
				linphone_call_set_video_source(nativePtr, value.nativePtr);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_video_stats(IntPtr thiz);

		/// <summary>
		/// Returns a copy of the call statistics for the video stream. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>a <see cref="Linphone.CallStats">Linphone.CallStats</see> object for
		/// the video stream or null if it isn&apos;t available.       </returns>
		public Linphone.CallStats VideoStats
		{
			get
			{
				IntPtr ptr = linphone_call_get_video_stats(nativePtr);
				Linphone.CallStats obj = fromNativePtr<Linphone.CallStats>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_get_zrtp_cache_mismatch_flag(IntPtr thiz);

		/// <summary>
		/// Returns whether ZRTP cache mismatch. 
		/// <para>
		/// If mismatch, the ZRTP authentication token must be verified by users as
		/// described in ZRTP procedure. 
		/// </para>
		/// </summary>
		/// <returns>true if ZRTP cache mismatch, false otherwise. </returns>
		public bool ZrtpCacheMismatchFlag
		{
			get
			{
				return linphone_call_get_zrtp_cache_mismatch_flag(nativePtr) != 0;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_call_accept(IntPtr thiz);

		/// <summary>
		/// Accepts an incoming call. 
		/// <para>
		/// Basically the application is notified of incoming calls within the
		/// call_state_changed callback of the LinphoneCoreVTable structure, where it will
		/// receive a #LinphoneCallStateIncoming event with the associated <see cref="Linphone.Call">
		/// Linphone.Call</see> object. The application can later accept the call using
		/// this method. 
		/// </para>
		/// </summary>
		/// <returns>0 on success, -1 on failure </returns>
		public void Accept()
		{
			
			int exception_result = linphone_call_accept(nativePtr);
			if (exception_result != 0) throw new LinphoneException("Accept returned value" + exception_result);
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_call_accept_early_media(IntPtr thiz);

		/// <summary>
		/// Accepts an early media session for an incoming call. 
		/// <para>
		/// This is identical as calling <see cref="Linphone.Call.AcceptEarlyMediaWithParams()">
		/// Linphone.Call.AcceptEarlyMediaWithParams()</see> with null parameters. 
		/// </para>
		/// </summary>
		/// <returns>0 if successful, -1 otherwise </returns>
		/// <remarks>See : <see cref="Linphone.Call.AcceptEarlyMediaWithParams()">
		/// Linphone.Call.AcceptEarlyMediaWithParams()</see> </remarks>
		public void AcceptEarlyMedia()
		{
			
			int exception_result = linphone_call_accept_early_media(nativePtr);
			if (exception_result != 0) throw new LinphoneException("AcceptEarlyMedia returned value" + exception_result);
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_call_accept_early_media_with_params(IntPtr thiz, IntPtr parameters);

		/// <summary>
		/// When receiving an incoming, accepts to start a media session as early-media. 
		/// <para>
		/// This means the call is not accepted but audio &amp; video streams can be
		/// established if the remote party supports early media. However, unlike after
		/// call acceptance, mic and camera input are not sent during early-media, though
		/// received audio &amp; video are played normally. The call can then later be
		/// fully accepted using <see cref="Linphone.Call.Accept()">
		/// Linphone.Call.Accept()</see> or <see cref="Linphone.Call.AcceptWithParams()">
		/// Linphone.Call.AcceptWithParams()</see>. 
		/// </para>
		/// </summary>
		/// <param name="parameters">The call parameters to use (can be null).    </param>
		/// <returns>0 if successful, -1 otherwise </returns>
		public void AcceptEarlyMediaWithParams(Linphone.CallParams parameters)
		{
			
			int exception_result = linphone_call_accept_early_media_with_params(nativePtr, parameters != null ? parameters.nativePtr : IntPtr.Zero);
			if (exception_result != 0) throw new LinphoneException("AcceptEarlyMediaWithParams returned value" + exception_result);
			
			GC.KeepAlive(parameters);
			
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_accept_transfer(IntPtr thiz);

		/// <summary>
		/// Accepts and execute a transfer (ie an incoming REFER request), to use when
		/// [sip] &apos;auto_accept_refer&apos; property is false. 
		/// <para>
		/// Default behaviour is to accept and execute the transfer automatically. 
		/// </para>
		/// </summary>
		public void AcceptTransfer()
		{
			
			linphone_call_accept_transfer(nativePtr);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_call_accept_update(IntPtr thiz, IntPtr parameters);

		/// <summary>
		/// Accepts call modifications initiated by other end through an incoming reINVITE
		/// or UPDATE request. 
		/// <para>
		/// <see cref="Linphone.Call.AcceptUpdate()">Linphone.Call.AcceptUpdate()</see> is
		/// typically used in response to a <see cref="Linphone.CallState.UpdatedByRemote">
		/// Linphone.CallState.UpdatedByRemote</see> state notification. When such
		/// notification arrives, the application has several choices:
		/// An application may use <see cref="Linphone.Call.RemoteParams">
		/// Linphone.Call.RemoteParams</see> to get information about the call parameters
		/// proposed by the other party (for example when he wants to add a video stream),
		/// in order to decide what to do, like for example requesting the end-user
		/// approval.
		/// The params argument must be constructed with <see cref="Linphone.Core.CreateCallParams()">
		/// Linphone.Core.CreateCallParams()</see>. <see cref="Linphone.Core.CreateCallParams()">
		/// Linphone.Core.CreateCallParams()</see> will initialize the returned <see cref="Linphone.CallParams">
		/// Linphone.CallParams</see> according to the offered parameters, previously used
		/// local parameters and local policies. For example, if the remote user-agent
		/// wants to add a video stream but the <see cref="Linphone.VideoActivationPolicy">
		/// Linphone.VideoActivationPolicy</see> of the <see cref="Linphone.Core">
		/// Linphone.Core</see> is to not automatically accept video, then the <see cref="Linphone.CallParams">
		/// Linphone.CallParams</see> will get its video stream disabled. The application
		/// is free to modify the <see cref="Linphone.CallParams">Linphone.CallParams</see>
		/// before passing it to <see cref="Linphone.Call.AcceptUpdate()">
		/// Linphone.Call.AcceptUpdate()</see>. Using null as params argument is allowed
		/// and equivalent to passing a <see cref="Linphone.CallParams">
		/// Linphone.CallParams</see> created by <see cref="Linphone.Core.CreateCallParams()">
		/// Linphone.Core.CreateCallParams()</see> and left unmodified.
		/// The answer generated by <see cref="Linphone.Call.AcceptUpdate()">
		/// Linphone.Call.AcceptUpdate()</see> at first follows RFC3264 SDP offer answer
		/// model rules. The supplied <see cref="Linphone.CallParams">
		/// Linphone.CallParams</see> allows to give application instructions about the
		/// response to generate, such as accepting or not the enablement of new stream.
		/// <see cref="Linphone.Call.AcceptUpdate()">Linphone.Call.AcceptUpdate()</see>
		/// does not have the pretention to control every aspect of the generation of the
		/// answer of an SDP offer/answer procedure.
		/// </para>
		/// </summary>
		/// <param name="parameters">A <see cref="Linphone.CallParams">
		/// Linphone.CallParams</see> object describing the call parameters to accept.   
		/// </param>
		/// <returns>0 if successful, -1 otherwise (actually when this function call is
		/// performed outside ot <see cref="Linphone.CallState.UpdatedByRemote">
		/// Linphone.CallState.UpdatedByRemote</see> state) </returns>
		public void AcceptUpdate(Linphone.CallParams parameters)
		{
			
			int exception_result = linphone_call_accept_update(nativePtr, parameters != null ? parameters.nativePtr : IntPtr.Zero);
			if (exception_result != 0) throw new LinphoneException("AcceptUpdate returned value" + exception_result);
			
			GC.KeepAlive(parameters);
			
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_call_accept_with_params(IntPtr thiz, IntPtr parameters);

		/// <summary>
		/// Accepts an incoming call, with parameters. 
		/// <para>
		/// Basically the application is notified of incoming calls within the
		/// call_state_changed callback of the LinphoneCoreVTable structure, where it will
		/// receive a #LinphoneCallStateIncoming event with the associated <see cref="Linphone.Call">
		/// Linphone.Call</see> object. The application can later accept the call using
		/// this method. 
		/// </para>
		/// </summary>
		/// <param name="parameters">The specific parameters for this call, for example
		/// whether video is accepted or not. Use null to use default parameters.   
		/// </param>
		/// <returns>0 on success, -1 on failure </returns>
		public void AcceptWithParams(Linphone.CallParams parameters)
		{
			
			int exception_result = linphone_call_accept_with_params(nativePtr, parameters != null ? parameters.nativePtr : IntPtr.Zero);
			if (exception_result != 0) throw new LinphoneException("AcceptWithParams returned value" + exception_result);
			
			GC.KeepAlive(parameters);
			
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_asked_to_autoanswer(IntPtr thiz);

		/// <summary>
		/// Tells whether a call has been asked to autoanswer. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>A boolean value telling whether the call has been asked to autoanswer
		/// </returns>
		public bool AskedToAutoanswer()
		{
			
			bool returnVal = linphone_call_asked_to_autoanswer(nativePtr) == (char)0 ? false : true;
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_cancel_dtmfs(IntPtr thiz);

		/// <summary>
		/// Stops current DTMF sequence sending. 
		/// <para>
		/// Please note that some DTMF could be already sent, depending on when this
		/// function call is delayed from <see cref="Linphone.Call.SendDtmfs()">
		/// Linphone.Call.SendDtmfs()</see>. This function will be automatically called if
		/// call state change to anything but LinphoneCallStreamsRunning. 
		/// </para>
		/// </summary>
		public void CancelDtmfs()
		{
			
			linphone_call_cancel_dtmfs(nativePtr);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_check_authentication_token_selected(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string selectedValue);

		/// <summary>
		/// Verifies that the half ZRTP short authentication string (SAS) selected by the
		/// user is correct, and set the verification result accordingly. 
		/// <para>
		/// If the remote party also verifies his/her half code correctly, the ZRTP cache
		/// will be updated to prevent future verification requirements for these two
		/// users. 
		/// </para>
		/// </summary>
		/// <param name="selectedValue">The ZRTP SAS selected by the user, or an empty
		/// string if the user cannot find the SAS </param>
		public void CheckAuthenticationTokenSelected(string selectedValue)
		{
			
			linphone_call_check_authentication_token_selected(nativePtr, selectedValue);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_confirm_go_clear(IntPtr thiz);

		/// <summary>
		/// Method to be called after the user confirm that he/she is notifed of the on
		/// going Go Clear procedure. 
		/// <para>
		/// </para>
		/// </summary>
		/// <remarks>Warning : this operation must be imperatevely initiate by a user
		/// action on sending of the GoClear ACK </remarks> 
		public void ConfirmGoClear()
		{
			
			linphone_call_confirm_go_clear(nativePtr);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_create_native_video_window_id_2(IntPtr thiz, IntPtr context);

		/// <summary>
		/// Create a native video window id where the video is to be displayed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <remarks>See : linphone_core_set_native_video_window_id for a general
		/// discussion about window IDs.</remarks>
		/// A context can be used to prevent Linphone from allocating the container
		/// (#MSOglContextInfo for MSOGL). null if not used. 
		/// <param name="context">preallocated Window ID (Used only for MSOGL)    </param>
		/// <returns>the native video window id (type may vary depending on platform).   
		/// </returns>
		public IntPtr CreateNativeVideoWindowId(IntPtr context)
		{
			
			IntPtr returnVal = linphone_call_create_native_video_window_id_2(nativePtr, context);
			
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_create_native_video_window_id(IntPtr thiz);

		/// <summary>
		/// Create a native video window id where the video is to be displayed. 
		/// <para>
		/// </para>
		/// </summary>
		/// <remarks>See : linphone_core_set_native_video_window_id for a general
		/// discussion about window IDs.</remarks>
		/// <returns>the native video window id (type may vary depending on platform).   
		/// </returns>
		public IntPtr CreateNativeVideoWindowId()
		{
			
			IntPtr returnVal = linphone_call_create_native_video_window_id(nativePtr);
			
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_create_notify(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string ev);

		/// <summary>
		/// Create a <see cref="Linphone.Event">Linphone.Event</see> in order to send
		/// NOTIFY requests through the SIP dialog created by the call. 
		/// <para>
		/// The call state must have passed through <see cref="Linphone.CallState.Connected">
		/// Linphone.CallState.Connected</see>. 
		/// </para>
		/// </summary>
		/// <param name="ev">The event type to be notified. </param>
		/// <returns>a new <see cref="Linphone.Event">Linphone.Event</see>    </returns>
		public Linphone.Event CreateNotify(string ev)
		{
			
			IntPtr ptr = linphone_call_create_notify(nativePtr, ev);
			Linphone.Event returnVal = fromNativePtr<Linphone.Event>(ptr, false);
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_call_decline(IntPtr thiz, int reason);

		/// <summary>
		/// Declines a pending incoming call, with a reason. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="reason">The reason for rejecting the call: <see cref="Linphone.Reason.Declined">
		/// Linphone.Reason.Declined</see> or <see cref="Linphone.Reason.Busy">
		/// Linphone.Reason.Busy</see> </param>
		/// <returns>0 on success, -1 on failure </returns>
		public void Decline(Linphone.Reason reason)
		{
			
			int exception_result = linphone_call_decline(nativePtr, (int)reason);
			if (exception_result != 0) throw new LinphoneException("Decline returned value" + exception_result);
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_call_decline_with_error_info(IntPtr thiz, IntPtr ei);

		/// <summary>
		/// Declines a pending incoming call, with a <see cref="Linphone.ErrorInfo">
		/// Linphone.ErrorInfo</see> object. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="ei"><see cref="Linphone.ErrorInfo">Linphone.ErrorInfo</see>
		/// containing more information on the call rejection.    </param>
		/// <returns>0 on success, -1 on failure </returns>
		public int DeclineWithErrorInfo(Linphone.ErrorInfo ei)
		{
			
			int returnVal = linphone_call_decline_with_error_info(nativePtr, ei != null ? ei.nativePtr : IntPtr.Zero);
			
			
			GC.KeepAlive(ei);
			return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_call_defer_update(IntPtr thiz);

		/// <summary>
		/// When receiving a <see cref="Linphone.CallState.UpdatedByRemote">
		/// Linphone.CallState.UpdatedByRemote</see> state notification, prevent <see cref="Linphone.Core">
		/// Linphone.Core</see> from performing an automatic answer. 
		/// <para>
		/// When receiving a <see cref="Linphone.CallState.UpdatedByRemote">
		/// Linphone.CallState.UpdatedByRemote</see> state notification (ie an incoming
		/// reINVITE), the default behaviour of <see cref="Linphone.Core">
		/// Linphone.Core</see> is defined by the &quot;defer_update_default&quot; option
		/// of the &quot;sip&quot; section of the config. If this option is 0 (the default)
		/// then the <see cref="Linphone.Core">Linphone.Core</see> automatically answers
		/// the reINIVTE with call parameters unchanged. However when for example when the
		/// remote party updated the call to propose a video stream, it can be useful to
		/// prompt the user before answering. This can be achieved by calling
		/// linphone_core_defer_call_update during the call state notification, to
		/// deactivate the automatic answer that would just confirm the audio but reject
		/// the video. Then, when the user responds to dialog prompt, it becomes possible
		/// to call <see cref="Linphone.Call.AcceptUpdate()">
		/// Linphone.Call.AcceptUpdate()</see> to answer the reINVITE, with video possibly
		/// enabled in the <see cref="Linphone.CallParams">Linphone.CallParams</see>
		/// argument.
		/// The <see cref="Linphone.CallState.UpdatedByRemote">
		/// Linphone.CallState.UpdatedByRemote</see> notification can also happen when
		/// receiving an INVITE without SDP. In such case, an unchanged offer is made in
		/// the 200Ok, and when the ACK containing the SDP answer is received, <see cref="Linphone.CallState.UpdatedByRemote">
		/// Linphone.CallState.UpdatedByRemote</see> is triggered to notify the application
		/// of possible changes in the media session. However in such case defering the
		/// update has no meaning since we just generated an offer.
		/// </para>
		/// </summary>
		/// <returns>0 if successful, -1 if the <see cref="Linphone.Call.DeferUpdate()">
		/// Linphone.Call.DeferUpdate()</see> was done outside a valid
		/// #LinphoneCallUpdatedByRemote notification </returns>
		public void DeferUpdate()
		{
			
			int exception_result = linphone_call_defer_update(nativePtr);
			if (exception_result != 0) throw new LinphoneException("DeferUpdate returned value" + exception_result);
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_stats(IntPtr thiz, int type);

		/// <summary>
		/// Returns a copy of the call statistics for a particular stream type. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="type">the <see cref="Linphone.StreamType">
		/// Linphone.StreamType</see> </param>
		/// <returns>a <see cref="Linphone.CallStats">Linphone.CallStats</see> object for
		/// the given stream or null if stream isn&apos;t available.       </returns>
		public Linphone.CallStats GetStats(Linphone.StreamType type)
		{
			
			IntPtr ptr = linphone_call_get_stats(nativePtr, (int)type);
			Linphone.CallStats returnVal = fromNativePtr<Linphone.CallStats>(ptr, false);
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_get_to_header(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string headerName);

		/// <summary>
		/// Returns the value of the header name. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="headerName">the name of the header to check.    </param>
		/// <returns>the value of the header if exists.    </returns>
		/// deprecated : 27/10/2020. Use <see cref="Linphone.CallParams.CustomHeader">
		/// Linphone.CallParams.CustomHeader</see> on <see cref="Linphone.Call.RemoteParams">
		/// Linphone.Call.RemoteParams</see> instead. 
		public string GetToHeader(string headerName)
		{
			
			string returnVal = linphone_pointer_to_string(linphone_call_get_to_header(nativePtr, headerName));
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_has_transfer_pending(IntPtr thiz);

		/// <summary>
		/// Returns if this calls has received a transfer that has not been executed yet. 
		/// <para>
		/// Pending transfers are executed when this call is being paused or closed,
		/// locally or by remote endpoint. If the call is already paused while receiving
		/// the transfer request, the transfer immediately occurs. 
		/// </para>
		/// </summary>
		/// <returns>true if transfer is pending, false otherwise. </returns>
		public bool HasTransferPending()
		{
			
			bool returnVal = linphone_call_has_transfer_pending(nativePtr) == (char)0 ? false : true;
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_media_in_progress(IntPtr thiz);

		/// <summary>
		/// Indicates whether an operation is in progress at the media side. 
		/// <para>
		/// It can be a bad idea to initiate signaling operations (adding video, pausing
		/// the call, removing video, changing video parameters) while the media is busy in
		/// establishing the connection (typically ICE connectivity checks). It can result
		/// in failures generating loss of time in future operations in the call.
		/// Applications are invited to check this function after each call state change to
		/// decide whether certain operations are permitted or not. 
		/// </para>
		/// </summary>
		/// <returns>true if media is busy in establishing the connection, false otherwise.
		/// </returns>
		public bool MediaInProgress()
		{
			
			bool returnVal = linphone_call_media_in_progress(nativePtr) == (char)0 ? false : true;
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_notify_ringing(IntPtr thiz);

		/// <summary>
		/// Starts the process of replying 180 Ringing. 
		/// <para>
		/// This function is used in conjonction with <see cref="Linphone.Core.EnableAutoSendRinging()">
		/// Linphone.Core.EnableAutoSendRinging()</see>. If the automatic sending of the
		/// 180 Ringing is disabled, this function needs to be called manually before the
		/// call timeouts.
		/// </para>
		/// </summary>
		public void NotifyRinging()
		{
			
			linphone_call_notify_ringing(nativePtr);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_ogl_render(IntPtr thiz);

		/// <summary>
		/// Calls generic OpenGL render for a given call. 
		/// <para>
		/// </para>
		/// </summary>
		public void OglRender()
		{
			
			linphone_call_ogl_render(nativePtr);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_call_pause(IntPtr thiz);

		/// <summary>
		/// Pauses the call. 
		/// <para>
		/// If a music file has been setup using <see cref="Linphone.Core.PlayFile">
		/// Linphone.Core.PlayFile</see>, this file will be played to the remote user. The
		/// only way to resume a paused call is to call <see cref="Linphone.Call.Resume()">
		/// Linphone.Call.Resume()</see>. 
		/// </para>
		/// </summary>
		/// <returns>0 on success, -1 on failure </returns>
		/// <remarks>See : <see cref="Linphone.Call.Resume()">Linphone.Call.Resume()</see>
		/// </remarks>
		public void Pause()
		{
			
			int exception_result = linphone_call_pause(nativePtr);
			if (exception_result != 0) throw new LinphoneException("Pause returned value" + exception_result);
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_call_redirect(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string redirectUri);

		/// <summary>
		/// Redirect the specified call to the given redirect URI. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="redirectUri">The URI to redirect the call to    </param>
		/// <returns>0 if successful, -1 on error. </returns>
		/// deprecated : 27/10/2020. Use <see cref="Linphone.Call.RedirectTo()">
		/// Linphone.Call.RedirectTo()</see> instead. 
		public void Redirect(string redirectUri)
		{
			
			int exception_result = linphone_call_redirect(nativePtr, redirectUri);
			if (exception_result != 0) throw new LinphoneException("Redirect returned value" + exception_result);
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_call_redirect_to(IntPtr thiz, IntPtr redirectAddress);

		/// <summary>
		/// Redirects the specified call to the given redirect Address. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="redirectAddress">The <see cref="Linphone.Address">
		/// Linphone.Address</see> to redirect the call to    </param>
		/// <returns>0 if successful, -1 on error. </returns>
		public void RedirectTo(Linphone.Address redirectAddress)
		{
			
			int exception_result = linphone_call_redirect_to(nativePtr, redirectAddress != null ? redirectAddress.nativePtr : IntPtr.Zero);
			if (exception_result != 0) throw new LinphoneException("RedirectTo returned value" + exception_result);
			
			GC.KeepAlive(redirectAddress);
			
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_request_notify_next_video_frame_decoded(IntPtr thiz);

		/// <summary>
		/// Requests the callback passed to linphone_call_cbs_set_next_video_frame_decoded
		/// to be called the next time the video decoder properly decodes a video frame. 
		/// <para>
		/// </para>
		/// </summary>
		public void RequestNotifyNextVideoFrameDecoded()
		{
			
			linphone_call_request_notify_next_video_frame_decoded(nativePtr);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_call_resume(IntPtr thiz);

		/// <summary>
		/// Resumes a call. 
		/// <para>
		/// The call needs to have been paused previously with <see cref="Linphone.Call.Pause()">
		/// Linphone.Call.Pause()</see>. 
		/// </para>
		/// </summary>
		/// <returns>0 on success, -1 on failure </returns>
		/// <remarks>See : <see cref="Linphone.Call.Pause()">Linphone.Call.Pause()</see>
		/// </remarks>
		public void Resume()
		{
			
			int exception_result = linphone_call_resume(nativePtr);
			if (exception_result != 0) throw new LinphoneException("Resume returned value" + exception_result);
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_call_send_dtmf(IntPtr thiz, sbyte dtmf);

		/// <summary>
		/// Sends the specified dtmf. 
		/// <para>
		/// The dtmf is automatically played to the user. </para>
		/// </summary>
		/// <remarks>See : <see cref="Linphone.Core.UseRfc2833ForDtmf">
		/// Linphone.Core.UseRfc2833ForDtmf</see> </remarks>
		/// <param name="dtmf">The dtmf name specified as a char, such as &apos;0&apos;,
		/// &apos;#&apos; etc... </param>
		/// <returns>0 if successful, -1 on error. </returns>
		public void SendDtmf(sbyte dtmf)
		{
			
			int exception_result = linphone_call_send_dtmf(nativePtr, dtmf);
			if (exception_result != 0) throw new LinphoneException("SendDtmf returned value" + exception_result);
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_call_send_dtmfs(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string dtmfs);

		/// <summary>
		/// Sends a list of dtmf. 
		/// <para>
		/// The dtmfs are automatically sent to remote, separated by some needed
		/// customizable delay. Sending is canceled if the call state changes to something
		/// not LinphoneCallStreamsRunning. </para>
		/// </summary>
		/// <remarks>See : <see cref="Linphone.Core.UseRfc2833ForDtmf">
		/// Linphone.Core.UseRfc2833ForDtmf</see> </remarks>
		/// <param name="dtmfs">A dtmf sequence such as &apos;123#123123&apos;    </param>
		/// <returns>-2 if there is already a DTMF sequence, -1 if call is not ready, 0
		/// otherwise. </returns>
		public void SendDtmfs(string dtmfs)
		{
			
			int exception_result = linphone_call_send_dtmfs(nativePtr, dtmfs);
			if (exception_result != 0) throw new LinphoneException("SendDtmfs returned value" + exception_result);
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_call_send_info_message(IntPtr thiz, IntPtr info);

		/// <summary>
		/// Sends an info message through an established call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="info">the <see cref="Linphone.InfoMessage">
		/// Linphone.InfoMessage</see> to send    </param>
		public void SendInfoMessage(Linphone.InfoMessage info)
		{
			
			int exception_result = linphone_call_send_info_message(nativePtr, info != null ? info.nativePtr : IntPtr.Zero);
			if (exception_result != 0) throw new LinphoneException("SendInfoMessage returned value" + exception_result);
			
			GC.KeepAlive(info);
			
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_send_vfu_request(IntPtr thiz);

		/// <summary>
		/// Requests remote side to send us a Video Fast Update. 
		/// <para>
		/// </para>
		/// </summary>
		public void SendVfuRequest()
		{
			
			linphone_call_send_vfu_request(nativePtr);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_skip_zrtp_authentication(IntPtr thiz);

		/// <summary>
		/// If the user skips the ZRTP authentication check, stop the security alert. 
		/// <para>
		/// </para>
		/// </summary>
		public void SkipZrtpAuthentication()
		{
			
			linphone_call_skip_zrtp_authentication(nativePtr);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_start_recording(IntPtr thiz);

		/// <summary>
		/// Starts call recording. 
		/// <para>
		/// Video record is only available if this function is called in state
		/// StreamRunning. The output file where audio is recorded must be previously
		/// specified with <see cref="Linphone.CallParams.RecordFile">
		/// Linphone.CallParams.RecordFile</see>. 
		/// </para>
		/// </summary>
		public void StartRecording()
		{
			
			linphone_call_start_recording(nativePtr);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_stop_recording(IntPtr thiz);

		/// <summary>
		/// Stops call recording. 
		/// <para>
		/// </para>
		/// </summary>
		public void StopRecording()
		{
			
			linphone_call_stop_recording(nativePtr);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_call_take_preview_snapshot(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string filePath);

		/// <summary>
		/// Takes a photo of currently captured video and write it into a jpeg file. 
		/// <para>
		/// Note that the snapshot is asynchronous, an application shall not assume that
		/// the file is created when the function returns. 
		/// </para>
		/// </summary>
		/// <param name="filePath">a path where to write the jpeg content.    </param>
		/// <returns>0 if successful, -1 otherwise (typically if jpeg format is not
		/// supported). </returns>
		public void TakePreviewSnapshot(string filePath)
		{
			
			int exception_result = linphone_call_take_preview_snapshot(nativePtr, filePath);
			if (exception_result != 0) throw new LinphoneException("TakePreviewSnapshot returned value" + exception_result);
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_call_take_video_snapshot(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string filePath);

		/// <summary>
		/// Takes a photo of currently received video and write it into a jpeg file. 
		/// <para>
		/// Note that the snapshot is asynchronous, an application shall not assume that
		/// the file is created when the function returns. 
		/// </para>
		/// </summary>
		/// <param name="filePath">a path where to write the jpeg content.    </param>
		/// <returns>0 if successful, -1 otherwise (typically if jpeg format is not
		/// supported). </returns>
		public void TakeVideoSnapshot(string filePath)
		{
			
			int exception_result = linphone_call_take_video_snapshot(nativePtr, filePath);
			if (exception_result != 0) throw new LinphoneException("TakeVideoSnapshot returned value" + exception_result);
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_call_terminate(IntPtr thiz);

		/// <summary>
		/// Terminates (ends) a call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>0 on success, -1 on failure </returns>
		public void Terminate()
		{
			
			int exception_result = linphone_call_terminate(nativePtr);
			if (exception_result != 0) throw new LinphoneException("Terminate returned value" + exception_result);
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_call_terminate_with_error_info(IntPtr thiz, IntPtr ei);

		/// <summary>
		/// Terminates a call with additional information, serialized as a Reason header. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="ei"><see cref="Linphone.ErrorInfo">Linphone.ErrorInfo</see>   
		/// </param>
		/// <returns>0 on success, -1 on failure </returns>
		public void TerminateWithErrorInfo(Linphone.ErrorInfo ei)
		{
			
			int exception_result = linphone_call_terminate_with_error_info(nativePtr, ei != null ? ei.nativePtr : IntPtr.Zero);
			if (exception_result != 0) throw new LinphoneException("TerminateWithErrorInfo returned value" + exception_result);
			
			GC.KeepAlive(ei);
			
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_call_transfer(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string referTo);

		/// <summary>
		/// Performs a simple call transfer to the specified destination. 
		/// <para>
		/// The remote endpoint is expected to issue a new call to the specified
		/// destination. The current call remains active and thus can be later paused or
		/// terminated. It is possible to follow the progress of the transfer provided that
		/// transferee sends notification about it. In this case, the
		/// transfer_state_changed callback of the LinphoneCoreVTable is invoked to notify
		/// of the state of the new call at the other party. The notified states are
		/// #LinphoneCallOutgoingInit , #LinphoneCallOutgoingProgress,
		/// #LinphoneCallOutgoingRinging and #LinphoneCallConnected. 
		/// </para>
		/// </summary>
		/// <param name="referTo">The destination the call is to be referred to.    </param>
		/// <returns>0 on success, -1 on failure </returns>
		/// deprecated : 27/10/2020. Use <see cref="Linphone.Call.TransferTo()">
		/// Linphone.Call.TransferTo()</see> instead. 
		public void Transfer(string referTo)
		{
			
			int exception_result = linphone_call_transfer(nativePtr, referTo);
			if (exception_result != 0) throw new LinphoneException("Transfer returned value" + exception_result);
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_call_transfer_to(IntPtr thiz, IntPtr referTo);

		/// <summary>
		/// Performs a simple call transfer to the specified destination. 
		/// <para>
		/// The remote endpoint is expected to issue a new call to the specified
		/// destination. The current call remains active and thus can be later paused or
		/// terminated. It is possible to follow the progress of the transfer provided that
		/// transferee sends notification about it. In this case, the &apos;transfer state
		/// changed&apos; callback of the <see cref="Linphone.CoreListener">
		/// Linphone.CoreListener</see> is invoked to notify of the state of the new call
		/// to the other party. The notified states are <see cref="Linphone.CallState.OutgoingInit">
		/// Linphone.CallState.OutgoingInit</see> , <see cref="Linphone.CallState.OutgoingProgress">
		/// Linphone.CallState.OutgoingProgress</see>, <see cref="Linphone.CallState.OutgoingRinging">
		/// Linphone.CallState.OutgoingRinging</see> and <see cref="Linphone.CallState.Connected">
		/// Linphone.CallState.Connected</see>. 
		/// </para>
		/// </summary>
		/// <param name="referTo">The <see cref="Linphone.Address">Linphone.Address</see>
		/// the call is to be referred to.    </param>
		/// <returns>0 on success, -1 on failure </returns>
		public void TransferTo(Linphone.Address referTo)
		{
			
			int exception_result = linphone_call_transfer_to(nativePtr, referTo != null ? referTo.nativePtr : IntPtr.Zero);
			if (exception_result != 0) throw new LinphoneException("TransferTo returned value" + exception_result);
			
			GC.KeepAlive(referTo);
			
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_call_transfer_to_another(IntPtr thiz, IntPtr dest);

		/// <summary>
		/// Transfers a call to destination of another running call. 
		/// <para>
		/// This is used for &quot;attended transfer&quot; scenarios. The transferred call
		/// is supposed to be in paused state, so that it is able to accept the transfer
		/// immediately. The destination call is a call previously established to introduce
		/// the transferred person. This method will send a transfer request to the
		/// transferred person. The phone of the transferred is then expected to
		/// automatically call to the destination of the transfer. The receiver of the
		/// transfer will then automatically close the call with us (the &apos;dest&apos;
		/// call). It is possible to follow the progress of the transfer provided that
		/// transferee sends notification about it. In this case, the &quot;transfer state
		/// changed&quot; callback of the <see cref="Linphone.CoreListener">
		/// Linphone.CoreListener</see> is invoked to notify of the state of the new call
		/// to the other party. The notified states are <see cref="Linphone.CallState.OutgoingInit">
		/// Linphone.CallState.OutgoingInit</see> , <see cref="Linphone.CallState.OutgoingProgress">
		/// Linphone.CallState.OutgoingProgress</see>, <see cref="Linphone.CallState.OutgoingRinging">
		/// Linphone.CallState.OutgoingRinging</see> and <see cref="Linphone.CallState.Connected">
		/// Linphone.CallState.Connected</see>. 
		/// </para>
		/// </summary>
		/// <param name="dest">A running call whose remote person will receive the transfer
		///    </param>
		/// <returns>0 on success, -1 on failure </returns>
		public void TransferToAnother(Linphone.Call dest)
		{
			
			int exception_result = linphone_call_transfer_to_another(nativePtr, dest != null ? dest.nativePtr : IntPtr.Zero);
			if (exception_result != 0) throw new LinphoneException("TransferToAnother returned value" + exception_result);
			
			GC.KeepAlive(dest);
			
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_call_update(IntPtr thiz, IntPtr parameters);

		/// <summary>
		/// Updates a running call according to supplied call parameters or parameters
		/// changed in the LinphoneCore. 
		/// <para>
		/// It triggers a SIP reINVITE in order to perform a new offer/answer of media
		/// capabilities. Changing the size of the transmitted video after calling
		/// linphone_core_set_preferred_video_size can be used by passing null as params
		/// argument. In case no changes are requested through the <see cref="Linphone.CallParams">
		/// Linphone.CallParams</see> argument, then this argument can be omitted and set
		/// to null. WARNING: Updating a call in the <see cref="Linphone.CallState.Paused">
		/// Linphone.CallState.Paused</see> state will still result in a paused call even
		/// if the media directions set in the params are sendrecv. To resume a paused
		/// call, you need to call <see cref="Linphone.Call.Resume()">
		/// Linphone.Call.Resume()</see>.
		/// </para>
		/// </summary>
		/// <param name="parameters">The new call parameters to use (may be null).   
		/// </param>
		/// <returns>0 if successful, -1 otherwise. </returns>
		public void Update(Linphone.CallParams parameters)
		{
			
			int exception_result = linphone_call_update(nativePtr, parameters != null ? parameters.nativePtr : IntPtr.Zero);
			if (exception_result != 0) throw new LinphoneException("Update returned value" + exception_result);
			
			GC.KeepAlive(parameters);
			
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_zoom(IntPtr thiz, float zoomFactor, float cx, float cy);

		/// <summary>
		/// Performs a zoom of the video displayed during a call. 
		/// <para>
		/// The zoom ensures that all the screen is fullfilled with the video. 
		/// </para>
		/// </summary>
		/// <param name="zoomFactor">a floating point number describing the zoom factor. A
		/// value 1.0 corresponds to no zoom applied. </param>
		/// <param name="cx">a floating point number pointing the horizontal center of the
		/// zoom to be applied. This value should be between 0.0 and 1.0. </param>
		/// <param name="cy">a floating point number pointing the vertical center of the
		/// zoom to be applied. This value should be between 0.0 and 1.0. </param>
		public void Zoom(float zoomFactor, float cx, float cy)
		{
			
			linphone_call_zoom(nativePtr, zoomFactor, cx, cy);
			
			
						
		}
	}
	/// <summary>
	/// Object used to keep track of all calls initiated, received or missed. 
	/// <para>
	/// It contains the call ID, date &amp; time at which the call took place and
	/// it&apos;s duration (0 if it wasn&apos;t answered). You can also know if video
	/// was enabled or not or if it was a conference, as well as it&apos;s average
	/// quality.
	/// If needed, you can also create a fake <see cref="Linphone.CallLog">
	/// Linphone.CallLog</see> using <see cref="Linphone.Core.CreateCallLog()">
	/// Linphone.Core.CreateCallLog()</see>, otherwise use <see cref="Linphone.Core.CallLogs">
	/// Linphone.Core.CallLogs</see> or even <see cref="Linphone.Call.CallLog">
	/// Linphone.Call.CallLog</see> to get the log of an ongoing call. 
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class CallLog : LinphoneObject
	{


		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_log_get_call_id(IntPtr thiz);

		/// <summary>
		/// Gets the call ID used by the call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The call ID used by the call as a string.    </returns>
		public string CallId
		{
			get
			{
				return linphone_pointer_to_string( linphone_call_log_get_call_id(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_log_get_chat_room(IntPtr thiz);

		/// <summary>
		/// Returns the chat room associated with this call-log, if any. 
		/// <para>
		/// This method is typically useful in order to retrieve an IM conversation
		/// associated with a past conference call. 
		/// </para>
		/// </summary>
		/// <returns>The <see cref="Linphone.ChatRoom">Linphone.ChatRoom</see> associated. 
		///   </returns>
		public Linphone.ChatRoom ChatRoom
		{
			get
			{
				IntPtr ptr = linphone_call_log_get_chat_room(nativePtr);
				Linphone.ChatRoom obj = fromNativePtr<Linphone.ChatRoom>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_log_get_conference_info(IntPtr thiz);

		/// <summary>
		/// Retrieves the conference info associated to this call log in DB. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The <see cref="Linphone.ConferenceInfo">Linphone.ConferenceInfo</see>
		/// associated.    </returns>
		public Linphone.ConferenceInfo ConferenceInfo
		{
			get
			{
				IntPtr ptr = linphone_call_log_get_conference_info(nativePtr);
				Linphone.ConferenceInfo obj = fromNativePtr<Linphone.ConferenceInfo>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.CallDir linphone_call_log_get_dir(IntPtr thiz);

		/// <summary>
		/// Gets the direction of the call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The <see cref="Linphone.CallDir">Linphone.CallDir</see> of the call.
		/// </returns>
		public Linphone.CallDir Dir
		{
			get
			{
				return linphone_call_log_get_dir(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_call_log_get_duration(IntPtr thiz);

		/// <summary>
		/// Gets the duration of the call since it was connected. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The duration of the call in seconds. </returns>
		public int Duration
		{
			get
			{
				return linphone_call_log_get_duration(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_log_get_error_info(IntPtr thiz);

		/// <summary>
		/// When the call was failed, return an object describing the failure. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns><see cref="Linphone.ErrorInfo">Linphone.ErrorInfo</see> about the
		/// error encountered by the call associated with this call log or null.   
		/// </returns>
		public Linphone.ErrorInfo ErrorInfo
		{
			get
			{
				IntPtr ptr = linphone_call_log_get_error_info(nativePtr);
				Linphone.ErrorInfo obj = fromNativePtr<Linphone.ErrorInfo>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_log_get_from_address(IntPtr thiz);

		/// <summary>
		/// Gets the origin address (ie from) of the call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The origin <see cref="Linphone.Address">Linphone.Address</see> (ie
		/// from) of the call.    </returns>
		public Linphone.Address FromAddress
		{
			get
			{
				IntPtr ptr = linphone_call_log_get_from_address(nativePtr);
				Linphone.Address obj = fromNativePtr<Linphone.Address>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_log_get_local_address(IntPtr thiz);

		/// <summary>
		/// Gets the local address (that is from or to depending on call direction) 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The local <see cref="Linphone.Address">Linphone.Address</see> of the
		/// call    </returns>
		public Linphone.Address LocalAddress
		{
			get
			{
				IntPtr ptr = linphone_call_log_get_local_address(nativePtr);
				Linphone.Address obj = fromNativePtr<Linphone.Address>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern float linphone_call_log_get_quality(IntPtr thiz);

		/// <summary>
		/// Gets the overall quality indication of the call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <remarks>See : <see cref="Linphone.Call.CurrentQuality">
		/// Linphone.Call.CurrentQuality</see> </remarks>
		/// <returns>The overall quality indication of the call. </returns>
		public float Quality
		{
			get
			{
				return linphone_call_log_get_quality(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_log_get_ref_key(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_log_set_ref_key(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string refkey);

		/// <summary>
		/// Gets the persistent reference key associated to the call log. 
		/// <para>
		/// The reference key can be for example an id to an external database. It is
		/// stored in the config file, thus can survive to process exits/restarts.
		/// </para>
		/// </summary>
		/// <returns>The reference key string that has been associated to the call log, or
		/// null if none has been associated.    </returns>
		public string RefKey
		{
			get
			{
				return linphone_pointer_to_string( linphone_call_log_get_ref_key(nativePtr) );
			}
			set
			{
				linphone_call_log_set_ref_key(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_log_get_remote_address(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_log_set_remote_address(IntPtr thiz, IntPtr address);

		/// <summary>
		/// Gets the remote address (that is from or to depending on call direction). 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The remote <see cref="Linphone.Address">Linphone.Address</see> of the
		/// call.    </returns>
		public Linphone.Address RemoteAddress
		{
			get
			{
				IntPtr ptr = linphone_call_log_get_remote_address(nativePtr);
				Linphone.Address obj = fromNativePtr<Linphone.Address>(ptr, true);
				return obj;
			}
			set
			{
				linphone_call_log_set_remote_address(nativePtr, value.nativePtr);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern long linphone_call_log_get_start_date(IntPtr thiz);

		/// <summary>
		/// Gets the start date of the call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The date of the beginning of the call. </returns>
		public long StartDate
		{
			get
			{
				return linphone_call_log_get_start_date(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.CallStatus linphone_call_log_get_status(IntPtr thiz);

		/// <summary>
		/// Gets the status of the call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The <see cref="Linphone.CallStatus">Linphone.CallStatus</see> of the
		/// call. </returns>
		public Linphone.CallStatus Status
		{
			get
			{
				return linphone_call_log_get_status(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_log_get_to_address(IntPtr thiz);

		/// <summary>
		/// Gets the destination address (ie to) of the call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The destination <see cref="Linphone.Address">Linphone.Address</see>
		/// (ie to) of the call.    </returns>
		public Linphone.Address ToAddress
		{
			get
			{
				IntPtr ptr = linphone_call_log_get_to_address(nativePtr);
				Linphone.Address obj = fromNativePtr<Linphone.Address>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_log_video_enabled(IntPtr thiz);

		/// <summary>
		/// Tells whether video was enabled at the end of the call or not. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>A boolean value telling whether video was enabled at the end of the
		/// call. </returns>
		public bool VideoEnabled
		{
			get
			{
				return linphone_call_log_video_enabled(nativePtr) != 0;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_log_to_str(IntPtr thiz);

		/// <summary>
		/// Gets a human readable string describing the call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <remarks>Note : : the returned string must be freed by the application (use
		/// ms_free()). </remarks>
		/// <returns>A human readable string describing the call.       </returns>
		public string ToStr()
		{
			
			string returnVal = linphone_pointer_to_string(linphone_call_log_to_str(nativePtr));
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_log_was_conference(IntPtr thiz);

		/// <summary>
		/// Tells whether that call was part of a conference. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if the call was part of a conference, false otherwise. </returns>
		public bool WasConference()
		{
			
			bool returnVal = linphone_call_log_was_conference(nativePtr) == (char)0 ? false : true;
			
						return returnVal;
		}
	}
	/// <summary>
	/// An object containing various parameters of a <see cref="Linphone.Call">
	/// Linphone.Call</see>. 
	/// <para>
	/// You can specify your params while answering an incoming call using <see cref="Linphone.Call.AcceptWithParams()">
	/// Linphone.Call.AcceptWithParams()</see> or while initiating an outgoing call
	/// with <see cref="Linphone.Core.InviteAddressWithParams()">
	/// Linphone.Core.InviteAddressWithParams()</see>.
	/// This object can be created using <see cref="Linphone.Core.CreateCallParams()">
	/// Linphone.Core.CreateCallParams()</see>, using null for the call pointer if you
	/// plan to use it for an outgoing call.
	/// For each call, three <see cref="Linphone.CallParams">Linphone.CallParams</see>
	/// are available: yours, your correspondent&apos;s and the one that describe the
	/// current state of the call that is the result of the negociation between the
	/// previous two. For example, you might enable a certain feature in your call
	/// param but this feature can be denied in the remote&apos;s configuration, hence
	/// the difference.
	/// </para>
	/// </summary>
	/// <remarks>See : <see cref="Linphone.Call.CurrentParams">
	/// Linphone.Call.CurrentParams</see>, <see cref="Linphone.Call.RemoteParams">
	/// Linphone.Call.RemoteParams</see> and <see cref="Linphone.Call.Params">
	/// Linphone.Call.Params</see>. </remarks>
	[StructLayout(LayoutKind.Sequential)]
	public class CallParams : LinphoneObject
	{


		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_params_get_account(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_set_account(IntPtr thiz, IntPtr account);

		/// <summary>
		/// Get the <see cref="Linphone.Account">Linphone.Account</see> that is used for
		/// the call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The selected <see cref="Linphone.Account">Linphone.Account</see> for
		/// the call, or null if none has been selected.    </returns>
		public Linphone.Account Account
		{
			get
			{
				IntPtr ptr = linphone_call_params_get_account(nativePtr);
				Linphone.Account obj = fromNativePtr<Linphone.Account>(ptr, true);
				return obj;
			}
			set
			{
				linphone_call_params_set_account(nativePtr, value.nativePtr);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_set_audio_bandwidth_limit(IntPtr thiz, int bandwidth);

		/// <summary>
		/// Refine bandwidth settings for this call by setting a bandwidth limit for audio
		/// streams. 
		/// <para>
		/// As a consequence, codecs whose bitrates are not compatible with this limit
		/// won&apos;t be used. 
		/// </para>
		/// </summary>
		/// <param name="bandwidth">The audio bandwidth limit to set in kbit/s. </param>
		public int AudioBandwidthLimit
		{
			set
			{
				linphone_call_params_set_audio_bandwidth_limit(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.MediaDirection linphone_call_params_get_audio_direction(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_set_audio_direction(IntPtr thiz, int direction);

		/// <summary>
		/// Get the audio stream direction. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The audio stream <see cref="Linphone.MediaDirection">
		/// Linphone.MediaDirection</see> associated with the call params. </returns>
		public Linphone.MediaDirection AudioDirection
		{
			get
			{
				return linphone_call_params_get_audio_direction(nativePtr);
			}
			set
			{
				linphone_call_params_set_audio_direction(nativePtr, (int)value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_params_audio_enabled(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_enable_audio(IntPtr thiz, char enabled);

		/// <summary>
		/// Tell whether audio is enabled or not. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>A boolean value telling whether audio is enabled or not. </returns>
		public bool AudioEnabled
		{
			get
			{
				return linphone_call_params_audio_enabled(nativePtr) != 0;
			}
			set
			{
				linphone_call_params_enable_audio(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_params_audio_multicast_enabled(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_enable_audio_multicast(IntPtr thiz, char yesno);

		/// <summary>
		/// Use to get multicast state of audio stream. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if subsequent calls will propose multicast ip set by <see cref="Linphone.Core.AudioMulticastAddr">
		/// Linphone.Core.AudioMulticastAddr</see> </returns>
		public bool AudioMulticastEnabled
		{
			get
			{
				return linphone_call_params_audio_multicast_enabled(nativePtr) != 0;
			}
			set
			{
				linphone_call_params_enable_audio_multicast(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_params_avpf_enabled(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_enable_avpf(IntPtr thiz, char enable);

		/// <summary>
		/// Whether or not the feedback extension will be used for AVP. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if AVPF is enabled, false otherwise </returns>
		public bool AvpfEnabled
		{
			get
			{
				return linphone_call_params_avpf_enabled(nativePtr) != 0;
			}
			set
			{
				linphone_call_params_enable_avpf(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_params_camera_enabled(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_enable_camera(IntPtr thiz, char enabled);

		/// <summary>
		/// Tell whether camera is enabled or not. 
		/// <para>
		/// The value returned by this function has a different meaning whether it is from
		/// local or remote parameters. The former states the will of the user to use the
		/// camera of his/her device. On the other hand, the latter is just a guess to know
		/// whether the remote party enabled its camera or not. For example, while the call
		/// is part of a conference a core will understand that the remote party disabled
		/// its camera if the thumbnail stream&apos;s direction is inactive. 
		/// </para>
		/// </summary>
		/// <returns>A boolean value telling whether camera is enabled or not. </returns>
		public bool CameraEnabled
		{
			get
			{
				return linphone_call_params_camera_enabled(nativePtr) != 0;
			}
			set
			{
				linphone_call_params_enable_camera(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_params_capability_negotiation_reinvite_enabled(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_enable_capability_negotiation_reinvite(IntPtr thiz, char enable);

		/// <summary>
		/// Check if the capability negotiation (RFC5939) reINVITE is enabled or not. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if capability negotiation reINVITE is enabled; false otherwise.
		/// </returns>
		public bool CapabilityNegotiationReinviteEnabled
		{
			get
			{
				return linphone_call_params_capability_negotiation_reinvite_enabled(nativePtr) != 0;
			}
			set
			{
				linphone_call_params_enable_capability_negotiation_reinvite(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_params_capability_negotiations_enabled(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_enable_capability_negotiations(IntPtr thiz, char enabled);

		/// <summary>
		/// Indicates whether capability negotiations (RFC5939) is enabled. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>a boolean indicating the enablement of capability negotiations.
		/// </returns>
		public bool CapabilityNegotiationsEnabled
		{
			get
			{
				return linphone_call_params_capability_negotiations_enabled(nativePtr) != 0;
			}
			set
			{
				linphone_call_params_enable_capability_negotiations(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_enable_cfg_lines_merging(IntPtr thiz, char enabled);

		/// <summary>
		/// Enable merging of cfg lines with consecutive indexes if capability negotiations
		/// (RFC5939) is enabled. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="enabled">A boolean value telling whether to merge pcfg and acfg
		/// lines </param>
		public bool CfgLinesMergingEnabled
		{
			set
			{
				linphone_call_params_enable_cfg_lines_merging(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.ConferenceLayout linphone_call_params_get_conference_video_layout(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_set_conference_video_layout(IntPtr thiz, int layout);

		/// <summary>
		/// Set video layout for conference. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>Preferred <see cref="Linphone.ConferenceLayout">
		/// Linphone.ConferenceLayout</see> to use at the start of a conference </returns>
		public Linphone.ConferenceLayout ConferenceVideoLayout
		{
			get
			{
				return linphone_call_params_get_conference_video_layout(nativePtr);
			}
			set
			{
				linphone_call_params_set_conference_video_layout(nativePtr, (int)value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_params_get_custom_contents(IntPtr thiz);

		/// <summary>
		/// Gets a list of <see cref="Linphone.Content">Linphone.Content</see> set if
		/// exists. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>A list of <see cref="Linphone.Content">Linphone.Content</see> set if
		/// exists, null otherwise.         </returns>
		public IEnumerable<Linphone.Content> CustomContents
		{
			get
			{
				return MarshalBctbxList<Linphone.Content>(linphone_call_params_get_custom_contents(nativePtr), true);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_params_early_media_sending_enabled(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_enable_early_media_sending(IntPtr thiz, char enabled);

		/// <summary>
		/// Indicate whether sending of early media was enabled. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>A boolean value telling whether sending of early media was enabled.
		/// </returns>
		public bool EarlyMediaSendingEnabled
		{
			get
			{
				return linphone_call_params_early_media_sending_enabled(nativePtr) != 0;
			}
			set
			{
				linphone_call_params_enable_early_media_sending(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_params_fec_enabled(IntPtr thiz);

		/// <summary>
		/// Tell whether FEC is enabled or not. 
		/// <para>
		/// The FEC is enbaled whether a FEC stream is found. 
		/// </para>
		/// </summary>
		/// <returns>A boolean value telling whether FEC is enabled or not. </returns>
		public bool FecEnabled
		{
			get
			{
				return linphone_call_params_fec_enabled(nativePtr) != 0;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_params_get_from_header(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_set_from_header(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string fromValue);

		/// <summary>
		/// Get the from header in the CallParams. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The content of the from header, may be null.    </returns>
		public string FromHeader
		{
			get
			{
				return linphone_pointer_to_string( linphone_call_params_get_from_header(nativePtr) );
			}
			set
			{
				linphone_call_params_set_from_header(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_params_get_input_audio_device(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_set_input_audio_device(IntPtr thiz, IntPtr audioDevice);

		/// <summary>
		/// Gets the default input audio device for a call that will be created using this
		/// call params. 
		/// <para>
		/// </para>
		/// </summary>
		/// <remarks>Warning : This method only concerns the call creation, it doesn&apos;t
		/// reflect the currently used input audio device of the call. Instead use <see cref="Linphone.Call.InputAudioDevice">
		/// Linphone.Call.InputAudioDevice</see> when call has been created. </remarks> 
		/// <returns>the <see cref="Linphone.AudioDevice">Linphone.AudioDevice</see> that
		/// will be used by default as input when the call will be created    </returns>
		public Linphone.AudioDevice InputAudioDevice
		{
			get
			{
				IntPtr ptr = linphone_call_params_get_input_audio_device(nativePtr);
				Linphone.AudioDevice obj = fromNativePtr<Linphone.AudioDevice>(ptr, true);
				return obj;
			}
			set
			{
				linphone_call_params_set_input_audio_device(nativePtr, value.nativePtr);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_params_is_capability_negotiation_reinvite_enabled(IntPtr thiz);

		/// <summary>
		/// Check if the capability negotiation (RFC5939) reINVITE is enabled or not. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if capability negotiation reINVITE is enabled; false otherwise.
		/// </returns>
		/// deprecated : 16/12/2021 Use <see cref="Linphone.CallParams.CapabilityNegotiationReinviteEnabled()">
		/// Linphone.CallParams.CapabilityNegotiationReinviteEnabled()</see> instead.
		public bool IsCapabilityNegotiationReinviteEnabled
		{
			get
			{
				return linphone_call_params_is_capability_negotiation_reinvite_enabled(nativePtr) != 0;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_params_is_recording(IntPtr thiz);

		/// <summary>
		/// Indicates whether the call is being recorded. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if the call is being recorded, false otherwise. </returns>
		public bool IsRecording
		{
			get
			{
				return linphone_call_params_is_recording(nativePtr) != 0;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_params_is_valid(IntPtr thiz);

		/// <summary>
		/// Check if call parameters are valid. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if the parameters are valid; false otherwise. </returns>
		public bool IsValid
		{
			get
			{
				return linphone_call_params_is_valid(nativePtr) != 0;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_params_get_local_conference_mode(IntPtr thiz);

		/// <summary>
		/// Tell whether the call is part of the locally managed conference. 
		/// <para>
		/// </para>
		/// </summary>
		/// <remarks>Warning : If a conference server is used to manage conferences, that
		/// function does not return true even if the conference is running. If you want to
		/// test whether the conference is running, you should test whether
		/// linphone_core_get_conference() return a non-null pointer. </remarks> 
		/// <returns>A boolean value telling whether the call is part of the locally
		/// managed conference. </returns>
		public bool LocalConferenceMode
		{
			get
			{
				return linphone_call_params_get_local_conference_mode(nativePtr) != 0;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_params_low_bandwidth_enabled(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_enable_low_bandwidth(IntPtr thiz, char enabled);

		/// <summary>
		/// Tell whether the call has been configured in low bandwidth mode or not. 
		/// <para>
		/// This mode can be automatically discovered thanks to a stun server when
		/// activate_edge_workarounds=1 in section [net] of configuration file. An
		/// application that would have reliable way to know network capacity may not use
		/// activate_edge_workarounds=1 but instead manually configure low bandwidth mode
		/// with <see cref="Linphone.CallParams.EnableLowBandwidth()">
		/// Linphone.CallParams.EnableLowBandwidth()</see>. When enabled, this param may
		/// transform a call request with video in audio only mode. 
		/// </para>
		/// </summary>
		/// <returns>A boolean value telling whether the low bandwidth mode has been
		/// configured/detected. </returns>
		public bool LowBandwidthEnabled
		{
			get
			{
				return linphone_call_params_low_bandwidth_enabled(nativePtr) != 0;
			}
			set
			{
				linphone_call_params_enable_low_bandwidth(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.MediaEncryption linphone_call_params_get_media_encryption(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_set_media_encryption(IntPtr thiz, int encryption);

		/// <summary>
		/// Get the kind of media encryption selected for the call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The kind of <see cref="Linphone.MediaEncryption">
		/// Linphone.MediaEncryption</see> selected for the call. </returns>
		public Linphone.MediaEncryption MediaEncryption
		{
			get
			{
				return linphone_call_params_get_media_encryption(nativePtr);
			}
			set
			{
				linphone_call_params_set_media_encryption(nativePtr, (int)value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_params_mic_enabled(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_enable_mic(IntPtr thiz, char enable);

		/// <summary>
		/// Tells whether the microphone will be enabled when the call will be created. 
		/// <para>
		/// </para>
		/// </summary>
		/// <remarks>Warning : This method only concerns the call creation, it doesn&apos;t
		/// reflect the actual microphone status during a call. Instead use <see cref="Linphone.Call.MicrophoneMuted">
		/// Linphone.Call.MicrophoneMuted</see> when call has been created. </remarks> 
		/// <returns>true if the microphone will be enabled, false if disabled. </returns>
		public bool MicEnabled
		{
			get
			{
				return linphone_call_params_mic_enabled(nativePtr) != 0;
			}
			set
			{
				linphone_call_params_enable_mic(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_params_get_output_audio_device(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_set_output_audio_device(IntPtr thiz, IntPtr audioDevice);

		/// <summary>
		/// Gets the default output audio device for a call that will be created using this
		/// call params. 
		/// <para>
		/// </para>
		/// </summary>
		/// <remarks>Warning : This method only concerns the call creation, it doesn&apos;t
		/// reflect the currently used output audio device of the call. Instead use <see cref="Linphone.Call.OutputAudioDevice">
		/// Linphone.Call.OutputAudioDevice</see> when call has been created. </remarks> 
		/// <returns>the <see cref="Linphone.AudioDevice">Linphone.AudioDevice</see> that
		/// will be used by default as output when the call will be created    </returns>
		public Linphone.AudioDevice OutputAudioDevice
		{
			get
			{
				IntPtr ptr = linphone_call_params_get_output_audio_device(nativePtr);
				Linphone.AudioDevice obj = fromNativePtr<Linphone.AudioDevice>(ptr, true);
				return obj;
			}
			set
			{
				linphone_call_params_set_output_audio_device(nativePtr, value.nativePtr);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern uint linphone_call_params_get_privacy(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_set_privacy(IntPtr thiz, uint privacy);

		/// <summary>
		/// Get requested level of privacy for the call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The LinphonePrivacyMask used for the call. </returns>
		public uint Privacy
		{
			get
			{
				return linphone_call_params_get_privacy(nativePtr);
			}
			set
			{
				linphone_call_params_set_privacy(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_params_get_proxy_config(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_set_proxy_config(IntPtr thiz, IntPtr proxyConfig);

		/// <summary>
		/// Get the <see cref="Linphone.ProxyConfig">Linphone.ProxyConfig</see> that is
		/// used for the call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The selected <see cref="Linphone.ProxyConfig">
		/// Linphone.ProxyConfig</see> for the call, or null if none has been selected.   
		/// </returns>
		/// deprecated : 28/02/2021 Use <see cref="Linphone.CallParams.Account">
		/// Linphone.CallParams.Account</see> instead. 
		public Linphone.ProxyConfig ProxyConfig
		{
			get
			{
				IntPtr ptr = linphone_call_params_get_proxy_config(nativePtr);
				Linphone.ProxyConfig obj = fromNativePtr<Linphone.ProxyConfig>(ptr, true);
				return obj;
			}
			set
			{
				linphone_call_params_set_proxy_config(nativePtr, value.nativePtr);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_params_realtime_text_enabled(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_call_params_enable_realtime_text(IntPtr thiz, char yesno);

		/// <summary>
		/// Use to get real time text following rfc4103. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>returns true if call rtt is activated. </returns>
		public bool RealtimeTextEnabled
		{
			get
			{
				return linphone_call_params_realtime_text_enabled(nativePtr) != 0;
			}
			set
			{
				int exception_result = linphone_call_params_enable_realtime_text(nativePtr, value ? (char)1 : (char)0);
				if (exception_result != 0) throw new LinphoneException("RealtimeTextEnabled setter returned value " + exception_result);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern uint linphone_call_params_get_realtime_text_keepalive_interval(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_set_realtime_text_keepalive_interval(IntPtr thiz, uint interval);

		/// <summary>
		/// Use to get keep alive interval of real time text following rfc4103. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>returns keep alive interval of real time text. </returns>
		public uint RealtimeTextKeepaliveInterval
		{
			get
			{
				return linphone_call_params_get_realtime_text_keepalive_interval(nativePtr);
			}
			set
			{
				linphone_call_params_set_realtime_text_keepalive_interval(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern float linphone_call_params_get_received_framerate(IntPtr thiz);

		/// <summary>
		/// Get the framerate of the video that is received. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The actual received framerate in frames per seconds, 0 if not
		/// available. </returns>
		public float ReceivedFramerate
		{
			get
			{
				return linphone_call_params_get_received_framerate(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_params_get_received_video_definition(IntPtr thiz);

		/// <summary>
		/// Get the definition of the received video. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The received <see cref="Linphone.VideoDefinition">
		/// Linphone.VideoDefinition</see> or null.    </returns>
		public Linphone.VideoDefinition ReceivedVideoDefinition
		{
			get
			{
				IntPtr ptr = linphone_call_params_get_received_video_definition(nativePtr);
				Linphone.VideoDefinition obj = fromNativePtr<Linphone.VideoDefinition>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_params_get_record_file(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_set_record_file(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string path);

		/// <summary>
		/// Get the path for the audio recording of the call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The path to the audio recording of the call or null.    </returns>
		public string RecordFile
		{
			get
			{
				return linphone_pointer_to_string( linphone_call_params_get_record_file(nativePtr) );
			}
			set
			{
				linphone_call_params_set_record_file(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_params_rtp_bundle_enabled(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_enable_rtp_bundle(IntPtr thiz, char val);

		/// <summary>
		/// Indicates whether RTP bundle mode (also known as Media Multiplexing) is
		/// enabled. 
		/// <para>
		/// See https://datatracker.ietf.org/doc/html/rfc8843 for more information. 
		/// </para>
		/// </summary>
		/// <returns>a boolean indicating the enablement of rtp bundle mode. </returns>
		public bool RtpBundleEnabled
		{
			get
			{
				return linphone_call_params_rtp_bundle_enabled(nativePtr) != 0;
			}
			set
			{
				linphone_call_params_enable_rtp_bundle(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_params_get_rtp_profile(IntPtr thiz);

		/// <summary>
		/// Get the RTP profile being used. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The RTP profile.    </returns>
		public string RtpProfile
		{
			get
			{
				return linphone_pointer_to_string( linphone_call_params_get_rtp_profile(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_params_screen_sharing_enabled(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_enable_screen_sharing(IntPtr thiz, char enabled);

		/// <summary>
		/// Tell whether screen sharing is enabled or not. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>A boolean value telling whether screen sharing is enabled or not.
		/// </returns>
		public bool ScreenSharingEnabled
		{
			get
			{
				return linphone_call_params_screen_sharing_enabled(nativePtr) != 0;
			}
			set
			{
				linphone_call_params_enable_screen_sharing(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern float linphone_call_params_get_sent_framerate(IntPtr thiz);

		/// <summary>
		/// Get the framerate of the video that is sent. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The actual sent framerate in frames per seconds, 0 if not available.
		/// </returns>
		public float SentFramerate
		{
			get
			{
				return linphone_call_params_get_sent_framerate(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_params_get_sent_video_definition(IntPtr thiz);

		/// <summary>
		/// Get the definition of the sent video. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The sent <see cref="Linphone.VideoDefinition">
		/// Linphone.VideoDefinition</see> or null.    </returns>
		public Linphone.VideoDefinition SentVideoDefinition
		{
			get
			{
				IntPtr ptr = linphone_call_params_get_sent_video_definition(nativePtr);
				Linphone.VideoDefinition obj = fromNativePtr<Linphone.VideoDefinition>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_params_get_session_name(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_set_session_name(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string name);

		/// <summary>
		/// Get the session name of the media session (ie in SDP). 
		/// <para>
		/// Subject from the SIP message can be retrieved using <see cref="Linphone.CallParams.CustomHeader">
		/// Linphone.CallParams.CustomHeader</see> and is different. 
		/// </para>
		/// </summary>
		/// <returns>The session name of the media session or null.    </returns>
		public string SessionName
		{
			get
			{
				return linphone_pointer_to_string( linphone_call_params_get_session_name(nativePtr) );
			}
			set
			{
				linphone_call_params_set_session_name(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_enable_tcap_line_merging(IntPtr thiz, char enabled);

		/// <summary>
		/// Enable merging of tcap lines with consecutive indexes if capability
		/// negotiations (RFC5939) is enabled. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="enabled">A boolean value telling whether to merge tcap lines
		/// </param>
		public bool TcapLineMergingEnabled
		{
			set
			{
				linphone_call_params_enable_tcap_line_merging(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_params_tone_indications_enabled(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_enable_tone_indications(IntPtr thiz, char enable);

		/// <summary>
		/// Check if tone indications are enabled. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if tone indications are enabled; false otherwise. </returns>
		public bool ToneIndicationsEnabled
		{
			get
			{
				return linphone_call_params_tone_indications_enabled(nativePtr) != 0;
			}
			set
			{
				linphone_call_params_enable_tone_indications(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_params_get_used_audio_payload_type(IntPtr thiz);

		/// <summary>
		/// Get the audio payload type that has been selected by a call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The selected <see cref="Linphone.PayloadType">
		/// Linphone.PayloadType</see>. null is returned if no audio payload type has been
		/// selected by the call.    </returns>
		public Linphone.PayloadType UsedAudioPayloadType
		{
			get
			{
				IntPtr ptr = linphone_call_params_get_used_audio_payload_type(nativePtr);
				Linphone.PayloadType obj = fromNativePtr<Linphone.PayloadType>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_params_get_used_text_payload_type(IntPtr thiz);

		/// <summary>
		/// Get the text payload type that has been selected by a call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The selected <see cref="Linphone.PayloadType">
		/// Linphone.PayloadType</see>. null is returned if no text payload type has been
		/// selected by the call.    </returns>
		public Linphone.PayloadType UsedTextPayloadType
		{
			get
			{
				IntPtr ptr = linphone_call_params_get_used_text_payload_type(nativePtr);
				Linphone.PayloadType obj = fromNativePtr<Linphone.PayloadType>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_params_get_used_video_payload_type(IntPtr thiz);

		/// <summary>
		/// Get the video payload type that has been selected by a call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The selected <see cref="Linphone.PayloadType">
		/// Linphone.PayloadType</see>. null is returned if no video payload type has been
		/// selected by the call.    </returns>
		public Linphone.PayloadType UsedVideoPayloadType
		{
			get
			{
				IntPtr ptr = linphone_call_params_get_used_video_payload_type(nativePtr);
				Linphone.PayloadType obj = fromNativePtr<Linphone.PayloadType>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.MediaDirection linphone_call_params_get_video_direction(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_set_video_direction(IntPtr thiz, int direction);

		/// <summary>
		/// Get the video stream direction. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The video stream <see cref="Linphone.MediaDirection">
		/// Linphone.MediaDirection</see> associated with the call params. </returns>
		public Linphone.MediaDirection VideoDirection
		{
			get
			{
				return linphone_call_params_get_video_direction(nativePtr);
			}
			set
			{
				linphone_call_params_set_video_direction(nativePtr, (int)value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_params_video_enabled(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_enable_video(IntPtr thiz, char enabled);

		/// <summary>
		/// Tell whether video is enabled or not. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>A boolean value telling whether video is enabled or not. </returns>
		public bool VideoEnabled
		{
			get
			{
				return linphone_call_params_video_enabled(nativePtr) != 0;
			}
			set
			{
				linphone_call_params_enable_video(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_params_video_multicast_enabled(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_enable_video_multicast(IntPtr thiz, char yesno);

		/// <summary>
		/// Use to get multicast state of video stream. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if subsequent calls will propose multicast ip set by <see cref="Linphone.Core.VideoMulticastAddr">
		/// Linphone.Core.VideoMulticastAddr</see> </returns>
		public bool VideoMulticastEnabled
		{
			get
			{
				return linphone_call_params_video_multicast_enabled(nativePtr) != 0;
			}
			set
			{
				linphone_call_params_enable_video_multicast(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_add_custom_content(IntPtr thiz, IntPtr content);

		/// <summary>
		/// Adds a <see cref="Linphone.Content">Linphone.Content</see> to be added to the
		/// INVITE SDP. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="content">The <see cref="Linphone.Content">Linphone.Content</see>
		/// to be added.    </param>
		public void AddCustomContent(Linphone.Content content)
		{
			
			linphone_call_params_add_custom_content(nativePtr, content != null ? content.nativePtr : IntPtr.Zero);
			
			
			GC.KeepAlive(content);
			
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_add_custom_header(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string headerName, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string headerValue);

		/// <summary>
		/// Add a custom SIP header in the INVITE for a call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="headerName">The name of the header to add.    </param>
		/// <param name="headerValue">The content of the header to add.    </param>
		public void AddCustomHeader(string headerName, string headerValue)
		{
			
			linphone_call_params_add_custom_header(nativePtr, headerName, headerValue);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_add_custom_sdp_attribute(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string attributeName, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string attributeValue);

		/// <summary>
		/// Add a custom attribute related to all the streams in the SDP exchanged within
		/// SIP messages during a call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="attributeName">The name of the attribute to add.    </param>
		/// <param name="attributeValue">The content value of the attribute to add.   
		/// </param>
		public void AddCustomSdpAttribute(string attributeName, string attributeValue)
		{
			
			linphone_call_params_add_custom_sdp_attribute(nativePtr, attributeName, attributeValue);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_add_custom_sdp_media_attribute(IntPtr thiz, int type, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string attributeName, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string attributeValue);

		/// <summary>
		/// Add a custom attribute related to a specific stream in the SDP exchanged within
		/// SIP messages during a call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="type">The type of the stream to add a custom SDP attribute to.
		/// </param>
		/// <param name="attributeName">The name of the attribute to add.    </param>
		/// <param name="attributeValue">The content value of the attribute to add.   
		/// </param>
		public void AddCustomSdpMediaAttribute(Linphone.StreamType type, string attributeName, string attributeValue)
		{
			
			linphone_call_params_add_custom_sdp_media_attribute(nativePtr, (int)type, attributeName, attributeValue);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_params_cfg_lines_merged(IntPtr thiz);

		/// <summary>
		/// Indicates whether cfg lines with consecutive indexes are going to be merged or
		/// not if capability negotiations (RFC5939) is enabled. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>a boolean indicating the enablement of pcfg and acfg line merging
		/// </returns>
		public bool CfgLinesMerged()
		{
			
			bool returnVal = linphone_call_params_cfg_lines_merged(nativePtr) == (char)0 ? false : true;
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_clear_custom_sdp_attributes(IntPtr thiz);

		/// <summary>
		/// Clear the custom SDP attributes related to all the streams in the SDP exchanged
		/// within SIP messages during a call. 
		/// <para>
		/// </para>
		/// </summary>
		public void ClearCustomSdpAttributes()
		{
			
			linphone_call_params_clear_custom_sdp_attributes(nativePtr);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_clear_custom_sdp_media_attributes(IntPtr thiz, int type);

		/// <summary>
		/// Clear the custom SDP attributes related to a specific stream in the SDP
		/// exchanged within SIP messages during a call. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="type">The type of the stream to clear the custom SDP attributes
		/// from. </param>
		public void ClearCustomSdpMediaAttributes(Linphone.StreamType type)
		{
			
			linphone_call_params_clear_custom_sdp_media_attributes(nativePtr, (int)type);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_params_copy(IntPtr thiz);

		/// <summary>
		/// Copy an existing <see cref="Linphone.CallParams">Linphone.CallParams</see>
		/// object to a new <see cref="Linphone.CallParams">Linphone.CallParams</see>
		/// object. 
		/// <para>
		/// <see cref="Linphone.CallParams.Copy()">Linphone.CallParams.Copy()</see> is
		/// error-prone, leading to inconsistent parameters being passed to <see cref="Linphone.Core.InviteAddressWithParams()">
		/// Linphone.Core.InviteAddressWithParams()</see> or <see cref="Linphone.Call.AcceptWithParams()">
		/// Linphone.Call.AcceptWithParams()</see>. </para>
		/// </summary>
		/// deprecated : use exclusively <see cref="Linphone.Core.CreateCallParams()">
		/// Linphone.Core.CreateCallParams()</see> to create <see cref="Linphone.CallParams">
		/// Linphone.CallParams</see> object. 
		/// <returns>A copy of the <see cref="Linphone.CallParams">
		/// Linphone.CallParams</see> object.       </returns>
		public Linphone.CallParams Copy()
		{
			
			IntPtr ptr = linphone_call_params_copy(nativePtr);
			Linphone.CallParams returnVal = fromNativePtr<Linphone.CallParams>(ptr, false);
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_call_params_disable_ringing(IntPtr thiz, char disable);

		/// <summary>
		/// Define whether ringing is disabled. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="disable">true to disable ringing; false otherwise. </param>
		public void DisableRinging(bool disable)
		{
			
			linphone_call_params_disable_ringing(nativePtr, disable ? (char)1 : (char)0);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_params_get_custom_header(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string headerName);

		/// <summary>
		/// Get a custom SIP header. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="headerName">The name of the header to get.    </param>
		/// <returns>The content of the header or null if not found.    </returns>
		public string GetCustomHeader(string headerName)
		{
			
			string returnVal = linphone_pointer_to_string(linphone_call_params_get_custom_header(nativePtr, headerName));
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_params_get_custom_sdp_attribute(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string attributeName);

		/// <summary>
		/// Get a custom SDP attribute that is related to all the streams. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="attributeName">The name of the attribute to get.    </param>
		/// <returns>The content value of the attribute or null if not found.    </returns>
		public string GetCustomSdpAttribute(string attributeName)
		{
			
			string returnVal = linphone_pointer_to_string(linphone_call_params_get_custom_sdp_attribute(nativePtr, attributeName));
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_params_get_custom_sdp_media_attribute(IntPtr thiz, int type, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string attributeName);

		/// <summary>
		/// Get a custom SDP attribute that is related to a specific stream. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="type">The type of the stream to add a custom SDP attribute to.
		/// </param>
		/// <param name="attributeName">The name of the attribute to get.    </param>
		/// <returns>The content value of the attribute or null if not found.    </returns>
		public string GetCustomSdpMediaAttribute(Linphone.StreamType type, string attributeName)
		{
			
			string returnVal = linphone_pointer_to_string(linphone_call_params_get_custom_sdp_media_attribute(nativePtr, (int)type, attributeName));
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_params_has_custom_sdp_attribute(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string attributeName);

		/// <summary>
		/// Returns true if a custom SDP attribute that is related to all the streams is
		/// present. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="attributeName">The name of the attribute to get.    </param>
		/// <returns>Whether the attribute is present. </returns>
		public bool HasCustomSdpAttribute(string attributeName)
		{
			
			bool returnVal = linphone_call_params_has_custom_sdp_attribute(nativePtr, attributeName) == (char)0 ? false : true;
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_params_has_custom_sdp_media_attribute(IntPtr thiz, int type, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string attributeName);

		/// <summary>
		/// Indicates whether a custom SDP attribute that is related to a specific stream
		/// is present or not. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="type">The type of the stream to add a custom SDP attribute to.
		/// </param>
		/// <param name="attributeName">The name of the attribute to get.    </param>
		/// <returns>Whether the attribute is present. </returns>
		public bool HasCustomSdpMediaAttribute(Linphone.StreamType type, string attributeName)
		{
			
			bool returnVal = linphone_call_params_has_custom_sdp_media_attribute(nativePtr, (int)type, attributeName) == (char)0 ? false : true;
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_params_is_media_encryption_supported(IntPtr thiz, int encryption);

		/// <summary>
		/// Returns the encryption is supported. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="encryption">The <see cref="Linphone.MediaEncryption">
		/// Linphone.MediaEncryption</see> to check whether is supported </param>
		/// <returns>a boolean indicating whether the encryption is supported </returns>
		public bool IsMediaEncryptionSupported(Linphone.MediaEncryption encryption)
		{
			
			bool returnVal = linphone_call_params_is_media_encryption_supported(nativePtr, (int)encryption) == (char)0 ? false : true;
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_params_ringing_disabled(IntPtr thiz);

		/// <summary>
		/// Check if ringing is disabled. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if ringing is disabled; false otherwise. </returns>
		public bool RingingDisabled()
		{
			
			bool returnVal = linphone_call_params_ringing_disabled(nativePtr) == (char)0 ? false : true;
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_params_tcap_lines_merged(IntPtr thiz);

		/// <summary>
		/// Indicates whether tcap lines with consecutive indexes are going to be merged or
		/// not if capability negotiations (RFC5939) is enabled. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>a boolean indicating the enablement of tcap line merging </returns>
		public bool TcapLinesMerged()
		{
			
			bool returnVal = linphone_call_params_tcap_lines_merged(nativePtr) == (char)0 ? false : true;
			
						return returnVal;
		}
	}
	/// <summary>
	/// This object carry various statistic informations regarding the quality of an
	/// audio or video stream for a given <see cref="Linphone.Call">
	/// Linphone.Call</see>. 
	/// <para>
	/// To receive these informations periodically and as soon as they are computed,
	/// implement the call_stats_updated() callback inside a <see cref="Linphone.CoreListener">
	/// Linphone.CoreListener</see>.
	/// At any time, the application can access latest computed statistics using <see cref="Linphone.Call.AudioStats">
	/// Linphone.Call.AudioStats</see> and <see cref="Linphone.Call.VideoStats">
	/// Linphone.Call.VideoStats</see>. 
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class CallStats : LinphoneObject
	{


		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern float linphone_call_stats_get_download_bandwidth(IntPtr thiz);

		/// <summary>
		/// Get the bandwidth measurement of the received stream, expressed in kbit/s,
		/// including IP/UDP/RTP headers. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The bandwidth measurement of the received stream in kbit/s. </returns>
		public float DownloadBandwidth
		{
			get
			{
				return linphone_call_stats_get_download_bandwidth(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern float linphone_call_stats_get_estimated_download_bandwidth(IntPtr thiz);

		/// <summary>
		/// Get the estimated bandwidth measurement of the received stream, expressed in
		/// kbit/s, including IP/UDP/RTP headers. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The estimated bandwidth measurement of the received stream in kbit/s.
		/// </returns>
		public float EstimatedDownloadBandwidth
		{
			get
			{
				return linphone_call_stats_get_estimated_download_bandwidth(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern uint linphone_call_stats_get_fec_cumulative_lost_packets_number(IntPtr thiz);

		/// <summary>
		/// If the FEC is enabled, gets the cumulative number of lost source packets of the
		/// RTP session that have not been repaired by the current FEC stream. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The cumulative number of lost packets </returns>
		public uint FecCumulativeLostPacketsNumber
		{
			get
			{
				return linphone_call_stats_get_fec_cumulative_lost_packets_number(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern float linphone_call_stats_get_fec_download_bandwidth(IntPtr thiz);

		/// <summary>
		/// Get the bandwidth measurement of the part of the received stream dedicated to
		/// FEC, expressed in kbit/s, including IP/UDP/RTP headers. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The bandwidth measurement of the received FEC stream in kbit/s.
		/// </returns>
		public float FecDownloadBandwidth
		{
			get
			{
				return linphone_call_stats_get_fec_download_bandwidth(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern uint linphone_call_stats_get_fec_repaired_packets_number(IntPtr thiz);

		/// <summary>
		/// If the FEC is enabled, gets the cumulative number of source packets of the RTP
		/// session that have been repaired by the current FEC stream. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The cumulative number of repaired packets </returns>
		public uint FecRepairedPacketsNumber
		{
			get
			{
				return linphone_call_stats_get_fec_repaired_packets_number(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern float linphone_call_stats_get_fec_upload_bandwidth(IntPtr thiz);

		/// <summary>
		/// Get the bandwidth measurement of the part of the sent stream dedicated to FEC,
		/// expressed in kbit/s, including IP/UDP/RTP headers. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The bandwidth measurement of the sent stream in kbit/s. </returns>
		public float FecUploadBandwidth
		{
			get
			{
				return linphone_call_stats_get_fec_upload_bandwidth(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.IceState linphone_call_stats_get_ice_state(IntPtr thiz);

		/// <summary>
		/// Get the state of ICE processing. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The <see cref="Linphone.IceState">Linphone.IceState</see> of ICE
		/// processing </returns>
		public Linphone.IceState IceState
		{
			get
			{
				return linphone_call_stats_get_ice_state(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.AddressFamily linphone_call_stats_get_ip_family_of_remote(IntPtr thiz);

		/// <summary>
		/// Get the IP address family of the remote peer. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The IP address family <see cref="Linphone.AddressFamily">
		/// Linphone.AddressFamily</see> of the remote peer. </returns>
		public Linphone.AddressFamily IpFamilyOfRemote
		{
			get
			{
				return linphone_call_stats_get_ip_family_of_remote(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_call_stats_is_zrtp_key_agreement_algo_post_quantum(IntPtr thiz);

		/// <summary>
		/// Did ZRTP used a Post Quantum algorithm to perform a key exchange. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if the ZRTP key exchange was performed using a PQ algo false
		/// otherwise: ZRTP exchange not completed or not using a PQ algo </returns>
		public bool IsZrtpKeyAgreementAlgoPostQuantum
		{
			get
			{
				return linphone_call_stats_is_zrtp_key_agreement_algo_post_quantum(nativePtr) != 0;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern float linphone_call_stats_get_jitter_buffer_size_ms(IntPtr thiz);

		/// <summary>
		/// Get the jitter buffer size in ms. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The jitter buffer size in ms. </returns>
		public float JitterBufferSizeMs
		{
			get
			{
				return linphone_call_stats_get_jitter_buffer_size_ms(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern uint linphone_call_stats_get_late_packets_cumulative_number(IntPtr thiz);

		/// <summary>
		/// Gets the cumulative number of late packets. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The cumulative number of late packets </returns>
		public uint LatePacketsCumulativeNumber
		{
			get
			{
				return linphone_call_stats_get_late_packets_cumulative_number(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern float linphone_call_stats_get_local_late_rate(IntPtr thiz);

		/// <summary>
		/// Gets the local late rate since last report, expressed as a percentage. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The local late rate </returns>
		public float LocalLateRate
		{
			get
			{
				return linphone_call_stats_get_local_late_rate(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern float linphone_call_stats_get_local_loss_rate(IntPtr thiz);

		/// <summary>
		/// Get the local loss rate since last report, expressed as a percentage. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The local loss rate </returns>
		public float LocalLossRate
		{
			get
			{
				return linphone_call_stats_get_local_loss_rate(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern float linphone_call_stats_get_receiver_interarrival_jitter(IntPtr thiz);

		/// <summary>
		/// Gets the remote reported interarrival jitter, expressed in seconds. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The interarrival jitter at last received receiver report </returns>
		public float ReceiverInterarrivalJitter
		{
			get
			{
				return linphone_call_stats_get_receiver_interarrival_jitter(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern float linphone_call_stats_get_receiver_loss_rate(IntPtr thiz);

		/// <summary>
		/// Gets the remote reported loss rate since last report, expressed as a
		/// percentage. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The receiver loss rate </returns>
		public float ReceiverLossRate
		{
			get
			{
				return linphone_call_stats_get_receiver_loss_rate(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern float linphone_call_stats_get_round_trip_delay(IntPtr thiz);

		/// <summary>
		/// Get the round trip delay in s. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The round trip delay in s. </returns>
		public float RoundTripDelay
		{
			get
			{
				return linphone_call_stats_get_round_trip_delay(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern float linphone_call_stats_get_rtcp_download_bandwidth(IntPtr thiz);

		/// <summary>
		/// Get the bandwidth measurement of the received RTCP, expressed in kbit/s,
		/// including IP/UDP/RTP headers. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The bandwidth measurement of the received RTCP in kbit/s. </returns>
		public float RtcpDownloadBandwidth
		{
			get
			{
				return linphone_call_stats_get_rtcp_download_bandwidth(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern float linphone_call_stats_get_rtcp_upload_bandwidth(IntPtr thiz);

		/// <summary>
		/// Get the bandwidth measurement of the sent RTCP, expressed in kbit/s, including
		/// IP/UDP/RTP headers. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The bandwidth measurement of the sent RTCP in kbit/s. </returns>
		public float RtcpUploadBandwidth
		{
			get
			{
				return linphone_call_stats_get_rtcp_upload_bandwidth(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_call_stats_get_rtp_cum_packet_loss(IntPtr thiz);

		/// <summary>
		/// Get the RTP cumulative number of incoming packet lost. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The number of RTP cumulative number of incoming packet lost </returns>
		public int RtpCumPacketLoss
		{
			get
			{
				return linphone_call_stats_get_rtp_cum_packet_loss(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern uint linphone_call_stats_get_rtp_discarded(IntPtr thiz);

		/// <summary>
		/// Get the RTP incoming packets discarded because the queue exceeds its max size. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The RTP incoming packets discarded because the queue exceeds its max
		/// size </returns>
		public uint RtpDiscarded
		{
			get
			{
				return linphone_call_stats_get_rtp_discarded(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern uint linphone_call_stats_get_rtp_hw_recv(IntPtr thiz);

		/// <summary>
		/// Get the number of received bytes excluding IPv4/IPv6/UDP headers and including
		/// late and duplicate packets. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the number of received bytes excluding IPv4/IPv6/UDP headers and
		/// including late and duplicate packets </returns>
		public uint RtpHwRecv
		{
			get
			{
				return linphone_call_stats_get_rtp_hw_recv(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern uint linphone_call_stats_get_rtp_packet_recv(IntPtr thiz);

		/// <summary>
		/// Get the number of RTP received packets. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The number of RTP received packets </returns>
		public uint RtpPacketRecv
		{
			get
			{
				return linphone_call_stats_get_rtp_packet_recv(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern uint linphone_call_stats_get_rtp_packet_sent(IntPtr thiz);

		/// <summary>
		/// Get the number of RTP outgoing packets. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The number of RTP outgoing packets </returns>
		public uint RtpPacketSent
		{
			get
			{
				return linphone_call_stats_get_rtp_packet_sent(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern uint linphone_call_stats_get_rtp_recv(IntPtr thiz);

		/// <summary>
		/// Get the RTP incoming recv_bytes of payload and delivered in time to the
		/// application. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The number of recv_bytes of payload and delivered in time to the
		/// application </returns>
		public uint RtpRecv
		{
			get
			{
				return linphone_call_stats_get_rtp_recv(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern uint linphone_call_stats_get_rtp_sent(IntPtr thiz);

		/// <summary>
		/// Get the RTP outgoing sent_bytes (excluding IP header) 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The number of outgoing sent_bytes (excluding IP header) </returns>
		public uint RtpSent
		{
			get
			{
				return linphone_call_stats_get_rtp_sent(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern float linphone_call_stats_get_sender_interarrival_jitter(IntPtr thiz);

		/// <summary>
		/// Gets the local interarrival jitter, expressed in seconds. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The interarrival jitter at last emitted sender report </returns>
		public float SenderInterarrivalJitter
		{
			get
			{
				return linphone_call_stats_get_sender_interarrival_jitter(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern float linphone_call_stats_get_sender_loss_rate(IntPtr thiz);

		/// <summary>
		/// Get the local loss rate since last report, expressed as a percentage. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The sender loss rate </returns>
		public float SenderLossRate
		{
			get
			{
				return linphone_call_stats_get_sender_loss_rate(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.MediaEncryption linphone_call_stats_get_srtp_source(IntPtr thiz);

		/// <summary>
		/// Get the method used for SRTP key exchange. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The <see cref="Linphone.MediaEncryption">
		/// Linphone.MediaEncryption</see> method used to exchange the SRTP keys   
		/// </returns>
		public Linphone.MediaEncryption SrtpSource
		{
			get
			{
				return linphone_call_stats_get_srtp_source(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.SrtpSuite linphone_call_stats_get_srtp_suite(IntPtr thiz);

		/// <summary>
		/// Get the SRTP Cryto suite in use. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The SRTP crypto suite currently in use <see cref="Linphone.SrtpSuite">
		/// Linphone.SrtpSuite</see>    </returns>
		public Linphone.SrtpSuite SrtpSuite
		{
			get
			{
				return linphone_call_stats_get_srtp_suite(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.StreamType linphone_call_stats_get_type(IntPtr thiz);

		/// <summary>
		/// Get the type of the stream the stats refer to. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The <see cref="Linphone.StreamType">Linphone.StreamType</see> the
		/// stats refer to </returns>
		public Linphone.StreamType Type
		{
			get
			{
				return linphone_call_stats_get_type(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern float linphone_call_stats_get_upload_bandwidth(IntPtr thiz);

		/// <summary>
		/// Get the bandwidth measurement of the sent stream, expressed in kbit/s,
		/// including IP/UDP/RTP headers. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The bandwidth measurement of the sent stream in kbit/s. </returns>
		public float UploadBandwidth
		{
			get
			{
				return linphone_call_stats_get_upload_bandwidth(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.UpnpState linphone_call_stats_get_upnp_state(IntPtr thiz);

		/// <summary>
		/// Get the state of uPnP processing. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The <see cref="Linphone.UpnpState">Linphone.UpnpState</see> of uPnP
		/// processing. </returns>
		public Linphone.UpnpState UpnpState
		{
			get
			{
				return linphone_call_stats_get_upnp_state(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_stats_get_zrtp_auth_tag_algo(IntPtr thiz);

		/// <summary>
		/// Get the ZRTP algorithm statistics details (authentication method) 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The auth tag algo </returns>
		public string ZrtpAuthTagAlgo
		{
			get
			{
				return linphone_pointer_to_string( linphone_call_stats_get_zrtp_auth_tag_algo(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_stats_get_zrtp_cipher_algo(IntPtr thiz);

		/// <summary>
		/// Get the ZRTP algorithm statistics details (cipher) 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The cipher algo </returns>
		public string ZrtpCipherAlgo
		{
			get
			{
				return linphone_pointer_to_string( linphone_call_stats_get_zrtp_cipher_algo(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_stats_get_zrtp_hash_algo(IntPtr thiz);

		/// <summary>
		/// Get the ZRTP algorithm statistics details (hash function) 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The hash algo </returns>
		public string ZrtpHashAlgo
		{
			get
			{
				return linphone_pointer_to_string( linphone_call_stats_get_zrtp_hash_algo(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_stats_get_zrtp_key_agreement_algo(IntPtr thiz);

		/// <summary>
		/// Get the ZRTP algorithm statistics details (key agreeement) 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The key agreement algo </returns>
		public string ZrtpKeyAgreementAlgo
		{
			get
			{
				return linphone_pointer_to_string( linphone_call_stats_get_zrtp_key_agreement_algo(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_call_stats_get_zrtp_sas_algo(IntPtr thiz);

		/// <summary>
		/// Get the ZRTP algorithm statistics details (SAS display) 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The sas algo </returns>
		public string ZrtpSasAlgo
		{
			get
			{
				return linphone_pointer_to_string( linphone_call_stats_get_zrtp_sas_algo(nativePtr) );
			}
		}
	}
	/// <summary>
	/// The <see cref="Linphone.CardDavParams">Linphone.CardDavParams</see> object
	/// represents a remote CardDAV server used by <see cref="Linphone.MagicSearch">
	/// Linphone.MagicSearch</see> as a plugin source. 
	/// <para>
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class CardDavParams : LinphoneObject
	{


		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_card_dav_params_get_domain_fields(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_card_dav_params_set_domain_fields(IntPtr thiz, IntPtr list);

		/// <summary>
		/// Gets the list of vCard RFC fields to use to match the domain filter on. 
		/// <para>
		/// For example you can use &quot;IMPP&quot;. 
		/// </para>
		/// </summary>
		/// <returns>The list of vCard fields to make the query on using domain filter.    
		///     </returns>
		public IEnumerable<string> DomainFields
		{
			get
			{
				return MarshalStringArray(linphone_card_dav_params_get_domain_fields(nativePtr));
			}
			set
			{
				IntPtr bctbxList = StringArrayToBctbxList(value);
				linphone_card_dav_params_set_domain_fields(nativePtr, bctbxList);
				CleanStringArrayPtrs(bctbxList);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_card_dav_params_get_use_exact_match_policy(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_card_dav_params_set_use_exact_match_policy(IntPtr thiz, char exactMatch);

		/// <summary>
		/// Gets the matching policy: true if the remote vCard value must match the filter
		/// exactly, false if it only needs to contain the filter. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The matching policy: true if the remote vCard value must match the
		/// filter exactly, false if it only needs to contain the filter. </returns>
		public bool UseExactMatchPolicy
		{
			get
			{
				return linphone_card_dav_params_get_use_exact_match_policy(nativePtr) != 0;
			}
			set
			{
				linphone_card_dav_params_set_use_exact_match_policy(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_card_dav_params_get_user_input_fields(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_card_dav_params_set_user_input_fields(IntPtr thiz, IntPtr list);

		/// <summary>
		/// Gets the list of vCard RFC fields to use to match user input text on. 
		/// <para>
		/// For example you can use &quot;FN&quot;, &quot;N&quot;, &quot;IMPP&quot;,
		/// &quot;ORG&quot;, etc... 
		/// </para>
		/// </summary>
		/// <returns>The list of vCard fields to make the query on using user input.       
		///  </returns>
		public IEnumerable<string> UserInputFields
		{
			get
			{
				return MarshalStringArray(linphone_card_dav_params_get_user_input_fields(nativePtr));
			}
			set
			{
				IntPtr bctbxList = StringArrayToBctbxList(value);
				linphone_card_dav_params_set_user_input_fields(nativePtr, bctbxList);
				CleanStringArrayPtrs(bctbxList);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_card_dav_params_clone(IntPtr thiz);

		/// <summary>
		/// Instantiates a new <see cref="Linphone.CardDavParams">
		/// Linphone.CardDavParams</see> with values from source. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The newly created <see cref="Linphone.CardDavParams">
		/// Linphone.CardDavParams</see> object.    </returns>
		public Linphone.CardDavParams Clone()
		{
			
			IntPtr ptr = linphone_card_dav_params_clone(nativePtr);
			Linphone.CardDavParams returnVal = fromNativePtr<Linphone.CardDavParams>(ptr, false);
			
						return returnVal;
		}
	}
	/// <summary>
	/// A <see cref="Linphone.ChatMessage">Linphone.ChatMessage</see> represents an
	/// instant message that can be send or received through a <see cref="Linphone.ChatRoom">
	/// Linphone.ChatRoom</see>. 
	/// <para>
	/// To create a <see cref="Linphone.ChatMessage">Linphone.ChatMessage</see>, use
	/// <see cref="Linphone.ChatRoom.CreateEmptyMessage()">
	/// Linphone.ChatRoom.CreateEmptyMessage()</see>, then either add text using <see cref="Linphone.ChatMessage.AddUtf8TextContent()">
	/// Linphone.ChatMessage.AddUtf8TextContent()</see> or a <see cref="Linphone.Content">
	/// Linphone.Content</see> with file informations using <see cref="Linphone.ChatMessage.AddFileContent()">
	/// Linphone.ChatMessage.AddFileContent()</see>. A valid <see cref="Linphone.Content">
	/// Linphone.Content</see> for file transfer must contain a type and subtype, the
	/// name of the file and it&apos;s size. Finally call <see cref="Linphone.ChatMessage.Send()">
	/// Linphone.ChatMessage.Send()</see> to send it.
	/// To send files through a <see cref="Linphone.ChatMessage">
	/// Linphone.ChatMessage</see>, you need to have configured a file transfer server
	/// URL with <see cref="Linphone.Core.FileTransferServer">
	/// Linphone.Core.FileTransferServer</see>. On the receiving side, either use <see cref="Linphone.ChatMessage.DownloadContent()">
	/// Linphone.ChatMessage.DownloadContent()</see> to download received files or
	/// enable auto-download in the <see cref="Linphone.Core">Linphone.Core</see> using
	/// <see cref="Linphone.Core.MaxSizeForAutoDownloadIncomingFiles">
	/// Linphone.Core.MaxSizeForAutoDownloadIncomingFiles</see>, -1 disabling the
	/// feature and 0 always downloading files no matter it&apos;s size.
	/// Keep in mind a <see cref="Linphone.ChatMessage">Linphone.ChatMessage</see>
	/// created by a <see cref="Linphone.ChatRoomBackend.Basic">
	/// Linphone.ChatRoomBackend.Basic</see> <see cref="Linphone.ChatRoom">
	/// Linphone.ChatRoom</see> can only contain one <see cref="Linphone.Content">
	/// Linphone.Content</see>, either text or file. 
	/// </para>
	/// </summary>
	[StructLayout(LayoutKind.Sequential)]
	public class ChatMessage : LinphoneObject
	{

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_factory_get();

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_factory_create_chat_message_cbs(IntPtr factory);

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_chat_message_add_callbacks(IntPtr thiz, IntPtr cbs);

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_chat_message_remove_callbacks(IntPtr thiz, IntPtr cbs);

		private ChatMessageListener listener;

		public ChatMessageListener Listener
		{
			get {
				if (listener == null)
				{
					IntPtr nativeListener = linphone_factory_create_chat_message_cbs(linphone_factory_get());
					listener = fromNativePtr<ChatMessageListener>(nativeListener, false, true);
					linphone_chat_message_add_callbacks(nativePtr, nativeListener);
					belle_sip_object_unref(nativeListener);
				}
				return listener;
			}
			set {
				if (listener != null && value == null) {
					linphone_chat_message_remove_callbacks(nativePtr, listener.nativePtr);
				}
				listener = value;
			}
		}

		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_chat_message_get_appdata(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_chat_message_set_appdata(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string data);

		/// <summary>
		/// Linphone message has an app-specific field that can store a text. 
		/// <para>
		/// The application might want to use it for keeping data over restarts, like
		/// thumbnail image path. 
		/// </para>
		/// </summary>
		/// <returns>the application-specific data or null if none has been stored.   
		/// </returns>
		public string Appdata
		{
			get
			{
				return linphone_pointer_to_string( linphone_chat_message_get_appdata(nativePtr) );
			}
			set
			{
				linphone_chat_message_set_appdata(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_chat_message_get_call_id(IntPtr thiz);

		/// <summary>
		/// Gets the SIP call-id accociated with the message. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the call-id    </returns>
		public string CallId
		{
			get
			{
				return linphone_pointer_to_string( linphone_chat_message_get_call_id(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_chat_message_get_chat_room(IntPtr thiz);

		/// <summary>
		/// Returns the chatroom this message belongs to. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the <see cref="Linphone.ChatRoom">Linphone.ChatRoom</see> in which
		/// this message has been sent or received.    </returns>
		public Linphone.ChatRoom ChatRoom
		{
			get
			{
				IntPtr ptr = linphone_chat_message_get_chat_room(nativePtr);
				Linphone.ChatRoom obj = fromNativePtr<Linphone.ChatRoom>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_chat_message_get_content_type(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_chat_message_set_content_type(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string contentType);

		/// <summary>
		/// Gets the content type of a chat message. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The content type of the chat message    </returns>
		public string ContentType
		{
			get
			{
				return linphone_pointer_to_string( linphone_chat_message_get_content_type(nativePtr) );
			}
			set
			{
				linphone_chat_message_set_content_type(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_chat_message_get_contents(IntPtr thiz);

		/// <summary>
		/// Returns the list of contents in the message. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The list of <see cref="Linphone.Content">Linphone.Content</see>.     
		/// </returns>
		public IEnumerable<Linphone.Content> Contents
		{
			get
			{
				return MarshalBctbxList<Linphone.Content>(linphone_chat_message_get_contents(nativePtr), true);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_chat_message_get_core(IntPtr thiz);

		/// <summary>
		/// Returns back pointer to <see cref="Linphone.Core">Linphone.Core</see> object. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the <see cref="Linphone.Core">Linphone.Core</see> object associated
		/// with this message.    </returns>
		public Linphone.Core Core
		{
			get
			{
				IntPtr ptr = linphone_chat_message_get_core(nativePtr);
				Linphone.Core obj = fromNativePtr<Linphone.Core>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern long linphone_chat_message_get_ephemeral_expire_time(IntPtr thiz);

		/// <summary>
		/// Returns the real time at which an ephemeral message expires and will be
		/// deleted. 
		/// <para>
		/// </para>
		/// </summary>
		/// <remarks>See : <see cref="Linphone.ChatMessage.IsEphemeral()">
		/// Linphone.ChatMessage.IsEphemeral()</see> </remarks>
		/// <returns>the time at which an ephemeral message expires. 0 means the message
		/// has not been read. </returns>
		public long EphemeralExpireTime
		{
			get
			{
				return linphone_chat_message_get_ephemeral_expire_time(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_chat_message_get_ephemeral_lifetime(IntPtr thiz);

		/// <summary>
		/// Returns lifetime of an ephemeral message. 
		/// <para>
		/// The lifetime is the duration after which the ephemeral message will disappear
		/// once viewed. </para>
		/// </summary>
		/// <remarks>See : <see cref="Linphone.ChatMessage.IsEphemeral()">
		/// Linphone.ChatMessage.IsEphemeral()</see> </remarks>
		/// <returns>the lifetime of an ephemeral message, by default 0 (disabled).
		/// </returns>
		public int EphemeralLifetime
		{
			get
			{
				return linphone_chat_message_get_ephemeral_lifetime(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_chat_message_get_error_info(IntPtr thiz);

		/// <summary>
		/// Gets full details about delivery error of a chat message. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>a <see cref="Linphone.ErrorInfo">Linphone.ErrorInfo</see> describing
		/// the details.    </returns>
		public Linphone.ErrorInfo ErrorInfo
		{
			get
			{
				IntPtr ptr = linphone_chat_message_get_error_info(nativePtr);
				Linphone.ErrorInfo obj = fromNativePtr<Linphone.ErrorInfo>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_chat_message_get_external_body_url(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_chat_message_set_external_body_url(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string externalBodyUrl);

		/// <summary>
		/// Messages can carry external body as defined by rfc2017. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>external body url or null if not present.    </returns>
		public string ExternalBodyUrl
		{
			get
			{
				return linphone_pointer_to_string( linphone_chat_message_get_external_body_url(nativePtr) );
			}
			set
			{
				linphone_chat_message_set_external_body_url(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_chat_message_get_file_transfer_information(IntPtr thiz);

		/// <summary>
		/// Gets the file transfer information (used by callbacks to recover informations
		/// during a rcs file transfer) 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>a pointer to the <see cref="Linphone.Content">Linphone.Content</see>
		/// structure or null if not present.    </returns>
		public Linphone.Content FileTransferInformation
		{
			get
			{
				IntPtr ptr = linphone_chat_message_get_file_transfer_information(nativePtr);
				Linphone.Content obj = fromNativePtr<Linphone.Content>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_chat_message_get_forward_info(IntPtr thiz);

		/// <summary>
		/// Gets the forward info if available as a string. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the original sender of the message if it has been forwarded, null
		/// otherwise.    </returns>
		public string ForwardInfo
		{
			get
			{
				return linphone_pointer_to_string( linphone_chat_message_get_forward_info(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_chat_message_get_from_address(IntPtr thiz);

		/// <summary>
		/// Gets origin of the message. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the <see cref="Linphone.Address">Linphone.Address</see> of the sender.
		///    </returns>
		public Linphone.Address FromAddress
		{
			get
			{
				IntPtr ptr = linphone_chat_message_get_from_address(nativePtr);
				Linphone.Address obj = fromNativePtr<Linphone.Address>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_chat_message_is_ephemeral(IntPtr thiz);

		/// <summary>
		/// Returns wether the chat message is an ephemeral message or not. 
		/// <para>
		/// An ephemeral message will automatically disappear from the recipient&apos;s
		/// screen after the message has been viewed. 
		/// </para>
		/// </summary>
		/// <returns>true if it is an ephemeral message, false otherwise </returns>
		public bool IsEphemeral
		{
			get
			{
				return linphone_chat_message_is_ephemeral(nativePtr) != 0;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_chat_message_is_file_transfer(IntPtr thiz);

		/// <summary>
		/// Return whether or not a chat message is a file transfer. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>Whether or not the message is a file transfer </returns>
		/// deprecated : 06/07/2020 check if <see cref="Linphone.ChatMessage.Contents">
		/// Linphone.ChatMessage.Contents</see> contains a <see cref="Linphone.Content">
		/// Linphone.Content</see> for which <see cref="Linphone.Content.IsFileTransfer()">
		/// Linphone.Content.IsFileTransfer()</see> returns true. 
		public bool IsFileTransfer
		{
			get
			{
				return linphone_chat_message_is_file_transfer(nativePtr) != 0;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_chat_message_is_file_transfer_in_progress(IntPtr thiz);

		/// <summary>
		/// Gets whether or not a file is currently being downloaded or uploaded. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if download or upload is in progress, false otherwise </returns>
		public bool IsFileTransferInProgress
		{
			get
			{
				return linphone_chat_message_is_file_transfer_in_progress(nativePtr) != 0;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_chat_message_is_forward(IntPtr thiz);

		/// <summary>
		/// Returns wether the chat message is a forward message or not. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if it is a forward message, false otherwise </returns>
		public bool IsForward
		{
			get
			{
				return linphone_chat_message_is_forward(nativePtr) != 0;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_chat_message_is_outgoing(IntPtr thiz);

		/// <summary>
		/// Returns wehther the message has been sent or received. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if message has been sent, false if it has been received.
		/// </returns>
		public bool IsOutgoing
		{
			get
			{
				return linphone_chat_message_is_outgoing(nativePtr) != 0;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_chat_message_is_read(IntPtr thiz);

		/// <summary>
		/// Returns wether the message has been read or not. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if message has been marked as read, false otherwise. </returns>
		public bool IsRead
		{
			get
			{
				return linphone_chat_message_is_read(nativePtr) != 0;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_chat_message_is_reply(IntPtr thiz);

		/// <summary>
		/// Returns wether the chat message is a reply message or not. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if it is a reply message, false otherwise </returns>
		public bool IsReply
		{
			get
			{
				return linphone_chat_message_is_reply(nativePtr) != 0;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_chat_message_is_secured(IntPtr thiz);

		/// <summary>
		/// Gets if the message was end-to-end encrypted when transferred. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if the message was end-to-end encrypted when transferred, false
		/// otherwise. </returns>
		public bool IsSecured
		{
			get
			{
				return linphone_chat_message_is_secured(nativePtr) != 0;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_chat_message_is_text(IntPtr thiz);

		/// <summary>
		/// Return whether or not a chat message is a text. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>Whether or not the message is a text </returns>
		/// deprecated : 06/07/2020 check if <see cref="Linphone.ChatMessage.Contents">
		/// Linphone.ChatMessage.Contents</see> contains a <see cref="Linphone.Content">
		/// Linphone.Content</see> with a PlainText content type. 
		public bool IsText
		{
			get
			{
				return linphone_chat_message_is_text(nativePtr) != 0;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_chat_message_get_local_address(IntPtr thiz);

		/// <summary>
		/// Returns the local address the message was sent or received with. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the <see cref="Linphone.Address">Linphone.Address</see> of the local
		/// address used to send/receive this message.    </returns>
		public Linphone.Address LocalAddress
		{
			get
			{
				IntPtr ptr = linphone_chat_message_get_local_address(nativePtr);
				Linphone.Address obj = fromNativePtr<Linphone.Address>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_chat_message_get_message_id(IntPtr thiz);

		/// <summary>
		/// Get the message identifier. 
		/// <para>
		/// It is used to identify a message so that it can be notified as delivered and/or
		/// displayed. 
		/// </para>
		/// </summary>
		/// <returns>The message identifier.    </returns>
		public string MessageId
		{
			get
			{
				return linphone_pointer_to_string( linphone_chat_message_get_message_id(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_chat_message_get_own_reaction(IntPtr thiz);

		/// <summary>
		/// Returns our own reaction for a given chat message, if any. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>Our own <see cref="Linphone.ChatMessageReaction">
		/// Linphone.ChatMessageReaction</see> for that message if any, null otherwise.   
		/// </returns>
		public Linphone.ChatMessageReaction OwnReaction
		{
			get
			{
				IntPtr ptr = linphone_chat_message_get_own_reaction(nativePtr);
				Linphone.ChatMessageReaction obj = fromNativePtr<Linphone.ChatMessageReaction>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_chat_message_get_peer_address(IntPtr thiz);

		/// <summary>
		/// Returns the peer (remote) address of the message. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the <see cref="Linphone.Address">Linphone.Address</see> of the peer
		/// address used to send/receive this message.    </returns>
		public Linphone.Address PeerAddress
		{
			get
			{
				IntPtr ptr = linphone_chat_message_get_peer_address(nativePtr);
				Linphone.Address obj = fromNativePtr<Linphone.Address>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_chat_message_get_reactions(IntPtr thiz);

		/// <summary>
		/// Gets the list of reactions received for this chat message. 
		/// <para>
		/// Warning: list is ordered by content of reaction message, not by received
		/// timestamp! 
		/// </para>
		/// </summary>
		/// <returns>The sorted list of reaction if any.         </returns>
		public IEnumerable<Linphone.ChatMessageReaction> Reactions
		{
			get
			{
				return MarshalBctbxList<Linphone.ChatMessageReaction>(linphone_chat_message_get_reactions(nativePtr), true);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_chat_message_get_reply_message(IntPtr thiz);

		/// <summary>
		/// Returns the <see cref="Linphone.ChatMessage">Linphone.ChatMessage</see> this
		/// message is a reply to. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the original message <see cref="Linphone.ChatMessage">
		/// Linphone.ChatMessage</see>.    </returns>
		public Linphone.ChatMessage ReplyMessage
		{
			get
			{
				IntPtr ptr = linphone_chat_message_get_reply_message(nativePtr);
				Linphone.ChatMessage obj = fromNativePtr<Linphone.ChatMessage>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_chat_message_get_reply_message_id(IntPtr thiz);

		/// <summary>
		/// Returns the ID of the message this is a reply to. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the original message id.    </returns>
		public string ReplyMessageId
		{
			get
			{
				return linphone_pointer_to_string( linphone_chat_message_get_reply_message_id(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_chat_message_get_reply_message_sender_address(IntPtr thiz);

		/// <summary>
		/// Returns the address of the sender of the message this is a reply to. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the original message sender <see cref="Linphone.Address">
		/// Linphone.Address</see>.    </returns>
		public Linphone.Address ReplyMessageSenderAddress
		{
			get
			{
				IntPtr ptr = linphone_chat_message_get_reply_message_sender_address(nativePtr);
				Linphone.Address obj = fromNativePtr<Linphone.Address>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern Linphone.ChatMessageState linphone_chat_message_get_state(IntPtr thiz);

		/// <summary>
		/// Gets the state of the message. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the current <see cref="Linphone.ChatMessageState">
		/// Linphone.ChatMessageState</see> of the message. </returns>
		public Linphone.ChatMessageState State
		{
			get
			{
				return linphone_chat_message_get_state(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_chat_message_get_text_content(IntPtr thiz);

		/// <summary>
		/// Gets the text content if available as a string. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the <see cref="Linphone.Content">Linphone.Content</see> buffer if
		/// available in System Locale, null otherwise.    </returns>
		/// deprecated : 01/07/2020. Use <see cref="Linphone.ChatMessage.Utf8Text">
		/// Linphone.ChatMessage.Utf8Text</see> instead. 
		public string TextContent
		{
			get
			{
				return linphone_pointer_to_string( linphone_chat_message_get_text_content(nativePtr) );
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern long linphone_chat_message_get_time(IntPtr thiz);

		/// <summary>
		/// Gets the time the message was sent. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the timestamp of when the message was sent. </returns>
		public long Time
		{
			get
			{
				return linphone_chat_message_get_time(nativePtr);
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_chat_message_get_to_address(IntPtr thiz);

		/// <summary>
		/// Gets destination of the message. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>the <see cref="Linphone.Address">Linphone.Address</see> of the
		/// recipient.    </returns>
		public Linphone.Address ToAddress
		{
			get
			{
				IntPtr ptr = linphone_chat_message_get_to_address(nativePtr);
				Linphone.Address obj = fromNativePtr<Linphone.Address>(ptr, true);
				return obj;
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_chat_message_get_to_be_stored(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_chat_message_set_to_be_stored(IntPtr thiz, char toBeStored);

		/// <summary>
		/// Gets if a chat message is to be stored. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>Whether or not the message is to be stored </returns>
		public bool ToBeStored
		{
			get
			{
				return linphone_chat_message_get_to_be_stored(nativePtr) != 0;
			}
			set
			{
				linphone_chat_message_set_to_be_stored(nativePtr, value ? (char)1 : (char)0);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_chat_message_get_utf8_text(IntPtr thiz);
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern int linphone_chat_message_set_utf8_text(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string text);

		/// <summary>
		/// Gets text part of this message. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>The text as UTF8 characters or null if no text.    </returns>
		public string Utf8Text
		{
			get
			{
				return linphone_pointer_to_string( linphone_chat_message_get_utf8_text(nativePtr) );
			}
			set
			{
				linphone_chat_message_set_utf8_text(nativePtr, value);
				
			}
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_chat_message_add_content(IntPtr thiz, IntPtr content);

		/// <summary>
		/// Adds a content to the ChatMessage. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="content">the <see cref="Linphone.Content">Linphone.Content</see>
		/// object to add.    </param>
		public void AddContent(Linphone.Content content)
		{
			
			linphone_chat_message_add_content(nativePtr, content != null ? content.nativePtr : IntPtr.Zero);
			
			
			GC.KeepAlive(content);
			
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_chat_message_add_custom_header(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string headerName, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string headerValue);

		/// <summary>
		/// Adds custom headers to the message. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="headerName">name of the header    </param>
		/// <param name="headerValue">header value    </param>
		public void AddCustomHeader(string headerName, string headerValue)
		{
			
			linphone_chat_message_add_custom_header(nativePtr, headerName, headerValue);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_chat_message_add_file_content(IntPtr thiz, IntPtr content);

		/// <summary>
		/// Adds a file content to the ChatMessage. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="content">the <see cref="Linphone.Content">Linphone.Content</see>
		/// object to add.    </param>
		public void AddFileContent(Linphone.Content content)
		{
			
			linphone_chat_message_add_file_content(nativePtr, content != null ? content.nativePtr : IntPtr.Zero);
			
			
			GC.KeepAlive(content);
			
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_chat_message_add_text_content(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string text);

		/// <summary>
		/// Creates a <see cref="Linphone.Content">Linphone.Content</see> of type PlainText
		/// with the given text as body. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="text">The text in System Locale to add to the message.    </param>
		/// deprecated : 01/07/2020. Use <see cref="Linphone.ChatMessage.AddUtf8TextContent()">
		/// Linphone.ChatMessage.AddUtf8TextContent()</see> instead. 
		public void AddTextContent(string text)
		{
			
			linphone_chat_message_add_text_content(nativePtr, text);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_chat_message_add_utf8_text_content(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string text);

		/// <summary>
		/// Creates a <see cref="Linphone.Content">Linphone.Content</see> of type
		/// text/plain with the provided string, and attach it to the message. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="text">The text to add to the message.    </param>
		public void AddUtf8TextContent(string text)
		{
			
			linphone_chat_message_add_utf8_text_content(nativePtr, text);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern void linphone_chat_message_cancel_file_transfer(IntPtr thiz);

		/// <summary>
		/// Cancels an ongoing file transfer attached to this message (upload or download). 
		/// <para>
		/// </para>
		/// </summary>
		public void CancelFileTransfer()
		{
			
			linphone_chat_message_cancel_file_transfer(nativePtr);
			
			
						
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_chat_message_create_reaction(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string utf8Reaction);

		/// <summary>
		/// Creates a emoji reaction for the given chat mesage. 
		/// <para>
		/// To send it, use <see cref="Linphone.ChatMessageReaction.Send()">
		/// Linphone.ChatMessageReaction.Send()</see>. 
		/// </para>
		/// </summary>
		/// <param name="utf8Reaction">the emoji character(s) as UTF-8.    </param>
		/// <returns>a <see cref="Linphone.ChatMessageReaction">
		/// Linphone.ChatMessageReaction</see> object.    </returns>
		public Linphone.ChatMessageReaction CreateReaction(string utf8Reaction)
		{
			
			IntPtr ptr = linphone_chat_message_create_reaction(nativePtr, utf8Reaction);
			Linphone.ChatMessageReaction returnVal = fromNativePtr<Linphone.ChatMessageReaction>(ptr, false);
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_chat_message_download_content(IntPtr thiz, IntPtr content);

		/// <summary>
		/// Starts the download of the <see cref="Linphone.Content">Linphone.Content</see>
		/// referenced in the <see cref="Linphone.ChatMessage">Linphone.ChatMessage</see>
		/// from remote server. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="content">the <see cref="Linphone.Content">Linphone.Content</see>
		/// object to download (must have the <see cref="Linphone.Content.IsFileTransfer()">
		/// Linphone.Content.IsFileTransfer()</see> method return true).    </param>
		/// <returns>false if there is an error, true otherwise. </returns>
		public bool DownloadContent(Linphone.Content content)
		{
			
			bool returnVal = linphone_chat_message_download_content(nativePtr, content != null ? content.nativePtr : IntPtr.Zero) == (char)0 ? false : true;
			
			GC.KeepAlive(content);
			return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_chat_message_download_contents(IntPtr thiz);

		/// <summary>
		/// Starts the download of all the <see cref="Linphone.Content">
		/// Linphone.Content</see> objects representing file transfers included in the
		/// message (<see cref="Linphone.Content.IsFileTransfer()">
		/// Linphone.Content.IsFileTransfer()</see> method returns true). 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>false if there is an error, true otherwise. </returns>
		public bool DownloadContents()
		{
			
			bool returnVal = linphone_chat_message_download_contents(nativePtr) == (char)0 ? false : true;
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_chat_message_get_custom_header(IntPtr thiz, [MarshalAs(LinphoneWrapper.ByteStringMarshalling)]string headerName);

		/// <summary>
		/// Retrieves a custom header value given its name. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="headerName">header name searched    </param>
		/// <returns>the custom header value or null if not found.    </returns>
		public string GetCustomHeader(string headerName)
		{
			
			string returnVal = linphone_pointer_to_string(linphone_chat_message_get_custom_header(nativePtr, headerName));
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern IntPtr linphone_chat_message_get_participants_by_imdn_state(IntPtr thiz, int state);

		/// <summary>
		/// Gets the list of participants for which the imdn state has reached the
		/// specified state and the time at which they did. 
		/// <para>
		/// </para>
		/// </summary>
		/// <param name="state">The LinphoneChatMessageState the imdn have reached (only
		/// use LinphoneChatMessageStateDelivered, LinphoneChatMessageStateDeliveredToUser,
		/// LinphoneChatMessageStateDisplayed and LinphoneChatMessageStateNotDelivered)
		/// </param>
		/// <returns>The list of participants.         </returns>
		public IEnumerable<Linphone.ParticipantImdnState> GetParticipantsByImdnState(Linphone.ChatMessageState state)
		{
			
			IEnumerable<Linphone.ParticipantImdnState> returnVal = MarshalBctbxList<Linphone.ParticipantImdnState>(linphone_chat_message_get_participants_by_imdn_state(nativePtr, (int)state), false);
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_chat_message_has_conference_invitation_content(IntPtr thiz);

		/// <summary>
		/// Returns wether the chat message has a conference invitation content or not. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if it has one, false otherwise. </returns>
		public bool HasConferenceInvitationContent()
		{
			
			bool returnVal = linphone_chat_message_has_conference_invitation_content(nativePtr) == (char)0 ? false : true;
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingConvention.Cdecl)]
		static extern char linphone_chat_message_has_text_content(IntPtr thiz);

		/// <summary>
		/// Returns wether the chat message has a text content or not. 
		/// <para>
		/// </para>
		/// </summary>
		/// <returns>true if it has one, false otherwise. </returns>
		/// deprecated : 27/10/2020. Check if <see cref="Linphone.ChatMessage.Contents">
		/// Linphone.ChatMessage.Contents</see> contains a <see cref="Linphone.Content">
		/// Linphone.Content</see> for which it&apos;s content type is PlainText. 
		public bool HasTextContent()
		{
			
			bool returnVal = linphone_chat_message_has_text_content(nativePtr) == (char)0 ? false : true;
			
						return returnVal;
		}
		[DllImport(LinphoneWrapper.LIB_NAME, CallingConvention = CallingCo